/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#if (GS32F00xx == 0x3000)

#include "boot_uart_v3_0.h"

void boot_uart_reset(uint32_t base)
{
	/* To clear all errors a sw reset of the module is required */
	HWREGH(base + SCI_O_CTL1) &= ~SCI_CTL1_SWRESET;
	HWREGH(base + SCI_O_CTL1) |= SCI_CTL1_SWRESET;
}

void boot_uart_disable_module(uint32_t base)
{
	/* Disable the FIFO. */
	HWREGH(base + SCI_O_FFTX) &= ~(SCI_FFTX_SCIFFENA);

	/* Disable the SCI. */
	HWREGH(base + SCI_O_CTL1) &= ~(SCI_CTL1_TXENA | SCI_CTL1_RXENA);
}

void boot_uart_enable_module(uint32_t base)
{
	/* Enable RX, TX, and the SCI.*/
	HWREGH(base + SCI_O_CTL1) |= (SCI_CTL1_TXENA | SCI_CTL1_RXENA | SCI_CTL1_SWRESET);
}

void boot_uart_set_config(uint32_t base, uint32_t lspclkHz, uint32_t baud, uint32_t config)
{
	uint32_t divider;

	boot_uart_disable_module(base);

	/* Compute the baud rate divider. */
	divider = (lspclkHz*4U  / baud );
	uint32_t intDiv = divider >> 6;
	uint32_t fractionDiv = divider & 0x3F;
	uint32_t intDivbaud =  ((intDiv *2U)-1);


	/* Set the baud rate. */
	HWREGH(base + SCI_O_HBAUD) = (intDivbaud & 0xFF00U) >> 8U;
	HWREGH(base + SCI_O_LBAUD) = intDivbaud & 0x00FFU;
	
	HWREGH(base + SCI_O_DLF) = fractionDiv;  /* DW DLF */

	/* Set parity, data length, and number of stop bits. */
	HWREGH(base + SCI_O_CCR) = ((HWREGH(base + SCI_O_CCR) &
								~(BOOT_UART_CONFIG_PAR_MASK |
								BOOT_UART_CONFIG_STOP_MASK |
								BOOT_UART_CONFIG_WLEN_MASK)) | config);

	boot_uart_enable_module(base);
}

void boot_uart_reset_channel(uint32_t base)
{
	/* Reset the Tx and Rx Channels */
	HWREGH(base + SCI_O_FFTX) &= ~SCI_FFTX_SCIRST;
	HWREGH(base + SCI_O_FFTX) |= SCI_FFTX_SCIRST;
}

/**
 * @brief enable the interrupt
 *
 * @param base The base address of SCI
 * @param intFlags Enable the required interrupt flags.
 * @details intFlags:
 * 			BOOT_UART_INT_RXERR				SCI receive error interrupt.
 * 			BOOT_UART_INT_RXRDY_BRKDT		SCI receive ready or break.
 * 			BOOT_UART_INT_TXRDY				SCI TX ready.
 * 			BOOT_UART_INT_TXFF				SCI TX level interrupt.
 * 			BOOT_UART_INT_RXFF				SCI_RX level interrupt.
 */
void boot_uart_enable_interrupt(uint32_t base, uint32_t intFlags)
{
	/* Enable the specified interrupts. */
	if((intFlags & BOOT_UART_INT_RXERR) == BOOT_UART_INT_RXERR)
		HWREGH(base + SCI_O_CTL1) |= SCI_CTL1_RXERRINTENA;

	if((intFlags & BOOT_UART_INT_RXRDY_BRKDT) == BOOT_UART_INT_RXRDY_BRKDT)
		HWREGH(base + SCI_O_CTL2) |= SCI_CTL2_RXBKINTENA;

	if((intFlags & BOOT_UART_INT_TXRDY) == BOOT_UART_INT_TXRDY)
		HWREGH(base + SCI_O_CTL2) |= SCI_CTL2_TXINTENA;

	if((intFlags & BOOT_UART_INT_TXFF) == BOOT_UART_INT_TXFF)
		HWREGH(base + SCI_O_FFTX) |= SCI_FFTX_TXFFIENA;

	if((intFlags & BOOT_UART_INT_RXFF) == BOOT_UART_INT_RXFF)
		HWREGH(base + SCI_O_FFRX) |= SCI_FFRX_RXFFIENA;
}

/**
 * @brief disable the interrupt
 *
 * @param base The base address of SCI
 * @param intFlags Disable the required interrupt flags.
 * @details intFlags:
 * 			BOOT_UART_INT_RXERR				SCI receive error interrupt.
 * 			BOOT_UART_INT_RXRDY_BRKDT		SCI receive ready or break.
 * 			BOOT_UART_INT_TXRDY				SCI TX ready.
 * 			BOOT_UART_INT_TXFF				SCI TX level interrupt.
 * 			BOOT_UART_INT_RXFF				SCI_RX level interrupt.
 */
void boot_uart_disable_interrupt(uint32_t base, uint32_t intFlags)
{
	/* Disable the specified interrupts.*/
	if((intFlags & BOOT_UART_INT_RXERR) == BOOT_UART_INT_RXERR)
		HWREGH(base + SCI_O_CTL1) &= ~SCI_CTL1_RXERRINTENA;

	if((intFlags & BOOT_UART_INT_RXRDY_BRKDT) == BOOT_UART_INT_RXRDY_BRKDT)
		HWREGH(base + SCI_O_CTL2) &= ~SCI_CTL2_RXBKINTENA;

	if((intFlags & BOOT_UART_INT_TXRDY) == BOOT_UART_INT_TXRDY)
		HWREGH(base + SCI_O_CTL2) &= ~SCI_CTL2_TXINTENA;

	if((intFlags & BOOT_UART_INT_TXFF) == BOOT_UART_INT_TXFF)
		HWREGH(base + SCI_O_FFTX) &= ~SCI_FFTX_TXFFIENA;

	if((intFlags & BOOT_UART_INT_RXFF) == BOOT_UART_INT_RXFF)
		HWREGH(base + SCI_O_FFRX) &= ~SCI_FFRX_RXFFIENA;
}

/**
 * @brief get the interrupt sources
 *
 * @param [in] base     The base address of SCI
 *
 * @return uint32_t interrupt sources mask bits.
 * @details intFlags:
 * 			BOOT_UART_INT_RXERR				SCI receive error interrupt.
 * 			BOOT_UART_INT_RXRDY_BRKDT		SCI receive ready or break.
 * 			BOOT_UART_INT_TXRDY				SCI TX ready.
 * 			BOOT_UART_INT_TXFF				SCI TX level interrupt.
 * 			BOOT_UART_INT_RXFF				SCI_RX level interrupt.
 */
uint32_t boot_uart_get_interrupt_status(uint32_t base)
{
	uint32_t interruptStatus = 0;

	if ((HWREGH(base + SCI_O_CTL2) & SCI_CTL2_TXRDY) == SCI_CTL2_TXRDY)
		interruptStatus |= BOOT_UART_INT_TXRDY;

	if ((HWREGH(base + SCI_O_RXST) & SCI_RXST_RXERROR) == SCI_RXST_RXERROR)
		interruptStatus |= BOOT_UART_INT_RXERR;

	if (((HWREGH(base + SCI_O_RXST) & SCI_RXST_RXRDY) == SCI_RXST_RXRDY) ||
		((HWREGH(base + SCI_O_RXST) & SCI_RXST_BRKDT) ==  SCI_RXST_BRKDT))
		interruptStatus |= BOOT_UART_INT_RXRDY_BRKDT;

	if ((HWREGH(base + SCI_O_FFTX) & SCI_FFTX_TXFFINT) == SCI_FFTX_TXFFINT)
		interruptStatus |= BOOT_UART_INT_TXFF;

	if ((HWREGH(base + SCI_O_FFRX) & SCI_FFRX_RXFFINT) == SCI_FFRX_RXFFINT)
		interruptStatus |= BOOT_UART_INT_RXFF;

	if ((HWREGH(base + SCI_O_RXST) & SCI_RXST_FE) == SCI_RXST_FE)
		interruptStatus |= BOOT_UART_INT_FE;

	if ((HWREGH(base + SCI_O_RXST) & SCI_RXST_OE) == SCI_RXST_OE)
		interruptStatus |= BOOT_UART_INT_OE;

	if ((HWREGH(base + SCI_O_RXST) & SCI_RXST_PE) == SCI_RXST_PE)
		interruptStatus |= BOOT_UART_INT_PE;

	return interruptStatus;
}

void boot_uart_clear_interrupt_status(uint32_t base, uint32_t intFlags)
{
	/* Clear the requested interrupt sources. */
	if(((intFlags & BOOT_UART_INT_RXERR) == BOOT_UART_INT_RXERR) ||
		((intFlags & BOOT_UART_INT_RXRDY_BRKDT) == BOOT_UART_INT_RXRDY_BRKDT) ||
		((intFlags & BOOT_UART_INT_FE) == BOOT_UART_INT_FE) ||
		((intFlags & BOOT_UART_INT_OE) == BOOT_UART_INT_OE) ||
		((intFlags & BOOT_UART_INT_PE) == BOOT_UART_INT_PE)) {

		boot_uart_reset(base);
	}

	if((intFlags & BOOT_UART_INT_TXFF) == BOOT_UART_INT_TXFF)
		HWREGH(base + SCI_O_FFTX) |= SCI_FFTX_TXFFINTCLR;

	if((intFlags & BOOT_UART_INT_RXFF) == BOOT_UART_INT_RXFF)
		HWREGH(base + SCI_O_FFRX) |= SCI_FFRX_RXFFINTCLR;
}

void boot_uart_enable_fifo_mode(uint32_t base)
{
	/* Enable the FIFO. */
	HWREGH(base + SCI_O_FFTX) |= SCI_FFTX_SCIRST;
	HWREGH(base + SCI_O_FFTX) |= SCI_FFTX_SCIFFENA | SCI_FFTX_TXFIFORESET;
	HWREGH(base + SCI_O_FFRX) |= SCI_FFRX_RXFIFORESET;
}

void boot_uart_disable_fifo_mode(uint32_t base)
{
	/* Disable the FIFO. */
	HWREGH(base + SCI_O_FFTX) &= ~SCI_FFTX_SCIFFENA;
}

void boot_uart_set_fifo_level(uint32_t base, BOOT_UART_TxFIFOLevel_t txLevel, BOOT_UART_RxFIFOLevel_t rxLevel)
{
	HWREGH(base + SCI_O_FFTX) = (HWREGH(base + SCI_O_FFTX) & (~SCI_FFTX_TXFFIL_M)) | (uint16_t)txLevel;

	HWREGH(base + SCI_O_FFRX) = (HWREGH(base + SCI_O_FFRX) & (~SCI_FFRX_RXFFIL_M)) | (uint16_t)rxLevel;
}

BOOT_UART_RxFIFOLevel_t boot_uart_get_rxfifo_status(uint32_t base)
{
	/* Get the current FIFO status */
	return (HWREGH(base + SCI_O_FFRX) & SCI_FFRX_RXFFST_M) >> SCI_FFRX_RXFFST_S;
}

BOOT_UART_TxFIFOLevel_t boot_uart_get_txfifo_status(uint32_t base)
{
	/* Get the current FIFO status */
	return (HWREGH(base + SCI_O_FFTX) & SCI_FFTX_TXFFST_M) >> SCI_FFTX_TXFFST_S;
}

uint16_t boot_uart_get_tx_status(uint32_t base)
{
	/* Get the current FIFO status */
	return HWREGH(base + SCI_O_CTL2);
}

uint16_t boot_uart_get_rx_status(uint32_t base)
{
	/* Get the current FIFO status */
	return HWREGH(base + SCI_O_RXST);
}

bool boot_uart_transmit_msg_fifo(uint32_t base, uint8_t msg)
{
	BOOT_UART_TxFIFOLevel_t status;

	status = boot_uart_get_txfifo_status(base);

	/* check if the Tx fifo is full */
	if (status == BOOT_UART_FIFO_TX16)
		return false;

	/* Send a byte of data */
	HWREGH(base + SCI_O_TXBUF) = msg;

	return true;
}

bool boot_uart_transmit_msg(uint32_t base, uint8_t msg)
{
	uint16_t status;

	status = boot_uart_get_tx_status(base);

	if (!(status & SCI_CTL2_TXEMPTY))
		return false;

	HWREGH(base + SCI_O_TXBUF) = msg;

	return true;
}

bool boot_uart_receive_msg_fifo(uint32_t base, uint8_t *msg)
{
	BOOT_UART_RxFIFOLevel_t status;

	status = boot_uart_get_rxfifo_status(base);

	if (status == BOOT_UART_FIFO_RX0)
		return false;

	*msg = (uint8_t)HWREGH(base + SCI_O_RXBUF);

	return true;
}

bool boot_uart_receive_msg(uint32_t base, uint8_t *msg)
{
	uint16_t status;

	status = boot_uart_get_rx_status(base);

	if (!(status & SCI_RXST_RXRDY))
		return false;

	*msg = (uint8_t)HWREGH(base + SCI_O_RXBUF);

	return true;
}

#endif
