/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#if (GS32F00xx == 0x3000)

#if defined(FDP_RT_FW)

#include "boot_gpio.h"
#include "boot_interrupt.h"
#include "fdp_rt/fdp_rt_uart_v3_0.h"
#include "fdp_rt/fdp_rt_errno.h"
#include "fdp_rt_defconfig.h"

int fdp_rt_uart_init(fdp_rt_uart_dev_priv_t *uart_priv, uint32_t uart_clk)
{
	if (uart_priv == NULL)
		return -FDP_RT_FAIL;

	/* Configuration the pinmux of uart. */
	boot_gpio_set_pin_config(uart_priv->rx_pinmux);
	boot_gpio_set_pin_config(uart_priv->tx_pinmux);

	/* Initialize UART */
	boot_uart_reset(uart_priv->base);
	boot_uart_set_config(uart_priv->base, uart_clk,
							uart_priv->bitrate, BOOT_UART_CONFIG_WLEN_8 |
							BOOT_UART_CONFIG_STOP_ONE | BOOT_UART_CONFIG_PAR_NONE);
	boot_uart_reset_channel(uart_priv->base);
	boot_uart_enable_fifo_mode(uart_priv->base);
	boot_uart_set_fifo_level(uart_priv->base, BOOT_UART_FIFO_TX0, BOOT_UART_FIFO_RX1);
	boot_uart_clear_interrupt_status(uart_priv->base, 0xff);
	boot_uart_disable_interrupt(uart_priv->base, 0xff);
	boot_uart_enable_interrupt(uart_priv->base, BOOT_UART_INT_RXFF);
	boot_uart_enable_module(uart_priv->base);

#if (FDP_RT_CONFIG_INTERRUPT_ENABLE == 1U)

	/* register UART interrupt handler. */
	boot_interrupt_register(uart_priv->irq_number, BOOT_ECLIC_LEVEL_TRIGGER,
							0, 0, uart_priv->irq_handler);
	boot_interrupt_enable(uart_priv->irq_number);

#else

	boot_interrupt_disable(uart_priv->irq_number);

#endif

	return FDP_RT_SUCCESS;
}

int fdp_rt_uart_deinit(fdp_rt_uart_dev_priv_t *uart_priv)
{
	if (uart_priv == NULL)
		return -FDP_RT_FAIL;

	/* Disable UART */
	boot_uart_disable_module(uart_priv->base);

	boot_interrupt_disable(uart_priv->irq_number);

	return FDP_RT_SUCCESS;
}

int fdp_rt_uart_write(fdp_rt_uart_dev_priv_t *uart_priv, uint8_t *buf, uint32_t len)
{
	if (uart_priv == NULL)
		return -FDP_RT_FAIL;

	for (uint32_t i = 0; i < len; i++)
		while (!boot_uart_transmit_msg_fifo(uart_priv->base, buf[i]));

	return FDP_RT_SUCCESS;
}

int fdp_rt_uart_read(fdp_rt_uart_dev_priv_t *uart_priv, uint8_t *buf,
					uint32_t *put_index, uint32_t *get_index, uint32_t rx_buf_size)
{
	uint32_t rxfifo_status;

	if (uart_priv == NULL)
		return -FDP_RT_FAIL;

	rxfifo_status = boot_uart_get_rxfifo_status(uart_priv->base);

	if ((*put_index + rxfifo_status > *get_index) &&
		*put_index < *get_index) {

		boot_uart_reset_channel(uart_priv->base);

		return -FDP_RT_FAIL;
	}

	for (uint32_t i = 0; i < rxfifo_status; i++) {
		if (*put_index >= rx_buf_size)
			*put_index = 0;

		boot_uart_receive_msg_fifo(uart_priv->base, &buf[(*put_index)++]);
	}

	return FDP_RT_SUCCESS;
}

#endif

#endif
