/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>

#include "device.h"
#include "interrupt.h"
#include "log.h"
#include "hal_uart_transfer.h"
#include "hal_can_transfer.h"
#include "hal_spi_transfer.h"
#include "app_info.h"
#include "user_crc.h"

#define FDP_BOOT_UPDATE_FLAG		0x9E8F6C2AU
#define FDP_BOOT_UPDATE_DISABLE		0x5A5A5A5AU
#define Delay_CYCLES				(DEVICE_SYSCLK_FREQ*4U) //4S


volatile BOOTAPP_SHARE_BSS_T uint32_t updateFlag;

static volatile uint32_t core1_ticks = 0;
extern uint32_t FLASH_BOOT_ADDR;
extern uint32_t FLASH_APP2_ADDR;
extern uint32_t FLASH_ENDM2_ADDR;
extern uint32_t __app_start_addr;
extern uint32_t __image_start_offset;
extern uint32_t jump_cmd_ready;

extern __SHARED_BSS__ volatile uint32_t ipc_cmd_m2s;
extern __SHARED_BSS__ volatile uint32_t ipc_address_m2s;
extern __SHARED_BSS__ volatile uint32_t ipc_len_m2s;
extern __SHARED_BSS__ volatile uint32_t ipc_cmd_s2m;
extern __SHARED_BSS__ volatile uint32_t ipc_address_s2m;
extern __SHARED_BSS__ volatile uint32_t ipc_len_s2m;
extern __CPU1TOCPU2_BSS__ uint32_t Ipc_m2s_Data[100];

const APP_ENDM_T   uint32_t App_End_Mark[4] = {
		0xAAAAAAAA, 0xBBBBBBBB, 0xCCCCCCCC, 0xDDDDDDDD
};

FDP_APP_INFO fdp_app_image_info_t app0_info = {
	.app_start_addr = (uint32_t)&__app_start_addr,
	.offset = (uint32_t)&__image_start_offset,
	.image_check = 0xA5A5A5A5,
	.image_size = 0,
	.key = 0x9527,
	.key_check = 0xA5A5A5A5,
	.version = 1,
	.App_Start_Mark[0] = 0xAAAAAAAA,
	.App_Start_Mark[1] = 0xBBBBBBBB,
	.App_Start_Mark[2] = 0xCCCCCCCC,
	.App_Start_Mark[3] = 0xDDDDDDDD,
};

IPC ipc;

void fdp_dcache_invaild(uintptr_t addr, uint32_t size)
{
	uint32_t lines;

	if (size / FDP_CPU_CACHE_LINES_SIZE) {
		if (size % FDP_CPU_CACHE_LINES_SIZE)
			lines = size / FDP_CPU_CACHE_LINES_SIZE + 1;
		else
			lines = size / FDP_CPU_CACHE_LINES_SIZE;
	} else
		lines = 1;

	MInvalDCacheLines(addr, lines);
}

int fdp_check_jtag_app(uintptr_t info_addr)
{
	volatile uint32_t *endmark = (volatile uint32_t *)&FLASH_ENDM2_ADDR;
	fdp_app_image_info_t *info = (fdp_app_image_info_t *)info_addr;

	if (!info)
		return -FDP_CHECKERR;

	/*Check KEY CRC, image CRC and image length  initial value */
	if ((info->key_check != 0xA5A5A5A5) ||
		(info->image_check != 0xA5A5A5A5)||
		(info->image_size != 0x0))
		return -FDP_CHECKERR;

	/* Check APP_START_MARK */
	if ((info->App_Start_Mark[0] != 0xAAAAAAAA)||
		(info->App_Start_Mark[1] != 0xBBBBBBBB)||
		(info->App_Start_Mark[2] != 0xCCCCCCCC)||
		(info->App_Start_Mark[3] != 0xDDDDDDDD))
		return -FDP_CHECKERR;

	/* check APP_END_MARK */
	if ((*(endmark + 0) != 0xAAAAAAAA)||
		(*(endmark + 1) != 0xBBBBBBBB)||
		(*(endmark + 2) != 0xCCCCCCCC)||
		(*(endmark + 3) != 0xDDDDDDDD))
		return -FDP_CHECKERR;

	return FDP_SUCCESS;
}

int fdp_check_image_info(uintptr_t info_addr)
{
	uint32_t crc32_val;

	fdp_app_image_info_t *info = (fdp_app_image_info_t *)info_addr;

	if (!info)
		return -FDP_CHECKERR;

	fdp_dcache_invaild((uintptr_t)info_addr, 1);

	/* verify key value. */
	crc32_val = fdp_crc32((uint8_t *)&info->key, sizeof(info->key));
	if (crc32_val != info->key_check)
		return -FDP_CHECKERR;

	return FDP_SUCCESS;
}

int fdp_check_image_base_on_info(uintptr_t info_addr)
{
	volatile uint32_t crc32_val;
	fdp_app_image_info_t *info;

	if (fdp_check_image_info(info_addr) != FDP_SUCCESS)
		return -FDP_CHECKERR;

	info = (fdp_app_image_info_t *)info_addr;

	/* Check if the address is in the range of the flash memory. */
	if ((info->app_start_addr < BANK_MIN_ADDR) || (info->app_start_addr >= BANK_MAX_ADDR))
		return -FDP_CHECKERR;

	fdp_dcache_invaild((uintptr_t)(info->app_start_addr), info->image_size);
	crc32_val = fdp_crc32((uint8_t *)info->app_start_addr, info->image_size);

	if (crc32_val != info->image_check)
		return -FDP_CHECKERR;

	return FDP_SUCCESS;
}


void fdp_set_entry_load_mode(bool enable)
{
	if (enable)
		updateFlag = FDP_BOOT_UPDATE_FLAG;
	else
		updateFlag = FDP_BOOT_UPDATE_DISABLE;
}

static void release_core1(bool release)
{
	if (release)
		SysCtl_setDspCpu2StopOnClr();
	else
		SysCtl_setDspCpu2StopOnSet();
}

static void set_core1_reset_vector(uintptr_t addr)
{
	SysCtl_setCpu2ResetVector(addr);
}

__INTERRUPT void IPC_ISR1(void)
{
	//if (ipc.Rx_flag == 0)
	{
		ipc.Rx_cmd  = ipc_cmd_s2m;
		ipc.Rx_addr = ipc_address_s2m;
		ipc.Rx_len  = ipc_len_s2m;
		ipc.Rx_flag = 1;
	}

	CIDU_ClearInterCoreIntReq(1, 0);
	__DSB();
}

void IPC_Send(uint32_t cmd, uint32_t address, uint32_t len)
{
	ipc_cmd_m2s = cmd;
	ipc_address_m2s = address;
	ipc_len_m2s = len;

	CIDU_TriggerInterCoreInt(0, 1);
}

void fdp_jump_to_address(uintptr_t addr)
{
	asm volatile ("jr %0" :: "r"(addr));
}


void cUartPrint_init(uint32_t SCIBase, uint32_t baud)
{

	GPIO_setAnalogMode(LOG_TX_PIN_NUM, GPIO_ANALOG_DISABLED);
	GPIO_setAnalogMode(LOG_RX_PIN_NUM, GPIO_ANALOG_DISABLED);
    GPIO_setPinConfig(LOG_TX_PIN);
    GPIO_setPinConfig(LOG_RX_PIN);
    GPIO_setPadConfig(LOG_TX_PIN_NUM, GPIO_PIN_TYPE_PULLUP);
    GPIO_setPadConfig(LOG_RX_PIN_NUM, GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(LOG_TX_PIN_NUM, GPIO_QUAL_SYNC);
	GPIO_setQualificationMode(LOG_RX_PIN_NUM, GPIO_QUAL_SYNC);

	SCI_enableLoopback(SCIBase);

	SCI_reset(SCIBase);

	SCI_setConfig(SCIBase, DEVICE_APBCLK_FREQ, baud,
				 (SCI_CONFIG_WLEN_8 | SCI_CONFIG_STOP_ONE | SCI_CONFIG_PAR_NONE));

	SCI_ClearReset(SCIBase);

	SCI_clearInterruptStatus(SCIBase);
	SCI_enableFIFO(SCIBase);

	SCI_setFIFOInterruptLevel(SCIBase, SCI_FIFO_TX0, SCI_FIFO_RX1);
	SCI_disableInterrupt(SCIBase, 0xff);
	SCI_disableLoopback(SCIBase);
}

void cInit_Leds(void)
{
	GPIO_setAnalogMode(GPIO_PIN_LED1, GPIO_ANALOG_DISABLED);
	GPIO_setAnalogMode(GPIO_PIN_LED2, GPIO_ANALOG_DISABLED);
    GPIO_setPinConfig(GPIO_CFG_LED1);
    GPIO_setPinConfig(GPIO_CFG_LED2);
    GPIO_setPadConfig(GPIO_PIN_LED1, GPIO_PIN_TYPE_STD);
    GPIO_setPadConfig(GPIO_PIN_LED2, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(GPIO_PIN_LED1, GPIO_DIR_MODE_OUT);
    GPIO_setDirectionMode(GPIO_PIN_LED2, GPIO_DIR_MODE_OUT);
    GPIO_WritePin(GPIO_PIN_LED1, 0);
    GPIO_WritePin(GPIO_PIN_LED2, 1);
}

int main(void)
{
	uint32_t delay = 0U;

	__disable_irq();

	Device_init();

    /* Initial Log SCI port  */
	cUartPrint_init(LOG_SCI_BASE, 115200);
	log_set_level(LOG_INFO);

	/* Initial Led */
	cInit_Leds();

	Uart_Init();

	can_ctrl_init();

	spi_init();

	log_info("Hello DSP300 Template Project!\r\n");
	log_info("Core running @ %d MHz\r\n", DEVICE_SYSCLK_FREQ/1000/1000);

	fdp_dcache_invaild((uintptr_t)&FLASH_APP2_ADDR, sizeof(fdp_app_image_info_t));

	/* Check if the image is valid. */
	if (!fdp_check_jtag_app((uintptr_t)&FLASH_APP2_ADDR) || !fdp_check_image_base_on_info((uintptr_t)&FLASH_APP2_ADDR))
	{
#if GS32F3xx == 0x2200
		// Start CPU2 thread.
		release_core1(0);
		// Step1: setting reset vector for CPU2;
		set_core1_reset_vector(((fdp_app_image_info_t *)&FLASH_APP2_ADDR)->app_start_addr);
		// Step2: release CPU2 core, the CPU2 is running
		release_core1(1);
#else
		SysTimer_ClearHartSWIRQ(1);         /* hart sync */
#endif
	}

    IPC_init(IPC_CPU1_L_CPU2_R);
    IPC_sync(IPC_CPU1_L_CPU2_R, IPC_FLAG0);

	Interrupt_register(InterCore_IRQn, &IPC_ISR1);
	Interrupt_SetPriority(InterCore_IRQn, 3, 0);
	Interrupt_enable(InterCore_IRQn);



	__enable_irq();

	while (1) {

		if (ipc.Rx_flag == 1U)
		{
			if ((ipc.Rx_cmd == TICK_RESPONSE_CMD)&&
				(ipc.Rx_len == 4U))
			{
				core1_ticks = *(volatile uint32_t *)(ipc.Rx_addr);
				GPIO_togglePin(GPIO_PIN_LED2);
				log_info("Core 0 is running, Core 1 systick = %d!\n", core1_ticks);

				/* Request 100ms delay update.  */
				if (core1_ticks >= 0xFFFFFFFF - 1000U)
					Ipc_m2s_Data[0]= 1000U;
				else
					Ipc_m2s_Data[0]= core1_ticks + 1000U;

				IPC_Send(TICK_REQUEST_CMD, (uint32_t)&Ipc_m2s_Data[0], sizeof(uint32_t));

			}
			CPU_clearCycleCnt();

			ipc.Rx_flag = 0U;
		}



		delay = CPU_getCycleCnt();
		if (delay >= Delay_CYCLES)
		{
			CPU_clearCycleCnt();
			/* Request update immediately.  */
			IPC_Send(TICK_REQUEST_CMD_IMM, (uint32_t)&Ipc_m2s_Data[0], 0);
		}

		if (jump_cmd_ready == 1) {
			jump_cmd_ready = 0;
			__disable_irq();
			//disable the IRQ not used in boot loader
			Interrupt_disable(InterCore_IRQn);
			SCI_enableLoopback(UART_BASE_ADDRESS);
			CAN_initModule(CAN_BASE);
			/* Update soft reset flag. */
			fdp_set_entry_load_mode(true);
			/* stop the core 1. */
			release_core1(0);
			fdp_jump_to_address((uintptr_t)&FLASH_BOOT_ADDR);
			while(1);
		}
	}

	return 0;
}
