/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <time.h>
#include <stdlib.h>
#include <string.h>

#include "device.h"
#include "interrupt.h"
#include "log.h"
#include "app_info.h"

/* Global variable */
volatile BOOTAPP_SHARE_BSS_T uint32_t updateFlag;

volatile uint32_t task10msFlag = 0;
volatile uint32_t sys_tick_response = 0U;
volatile uint32_t sys_tick_request = 0U;

extern uint32_t __app_start_addr;
extern uint32_t __image_start_offset;
extern __SHARED_BSS__ volatile uint32_t ipc_cmd_m2s;
extern __SHARED_BSS__ volatile uint32_t ipc_address_m2s;
extern __SHARED_BSS__ volatile uint32_t ipc_len_m2s;
extern __SHARED_BSS__ volatile uint32_t ipc_cmd_s2m;
extern __SHARED_BSS__ volatile uint32_t ipc_address_s2m;
extern __SHARED_BSS__ volatile uint32_t ipc_len_s2m;
extern __CPU2TOCPU1_BSS__ uint32_t Ipc_s2m_Data[100];

IPC ipc;

const APP_ENDM_T   uint32_t App_End_Mark[4] = {
		0xAAAAAAAA, 0xBBBBBBBB, 0xCCCCCCCC, 0xDDDDDDDD
};
FDP_APP_INFO fdp_app_image_info_t app1_info = {
	.app_start_addr = (uint32_t)&__app_start_addr,
	.offset = (uint32_t)&__image_start_offset,
	.image_check = 0xA5A5A5A5,
	.image_size = 0,
	.key = 0x9527,
	.key_check = 0xA5A5A5A5,
	.version = 1,
	.App_Start_Mark[0] = 0xAAAAAAAA,
	.App_Start_Mark[1] = 0xBBBBBBBB,
	.App_Start_Mark[2] = 0xCCCCCCCC,
	.App_Start_Mark[3] = 0xDDDDDDDD,
};

__INTERRUPT void IPC_ISR2(void)
{
	if (ipc.Rx_flag == 0U)
	{
		ipc.Rx_cmd  = ipc_cmd_m2s;
		ipc.Rx_addr = ipc_address_m2s;
		ipc.Rx_len  = ipc_len_m2s;
		ipc.Rx_flag = 1U;
	}
	CIDU_ClearInterCoreIntReq(0, 1);
	__DSB();
}

__INTERRUPT void APB_TIMER1_IRQ_HANDLER(void)
{
	task10msFlag = 1U;
	GPIO_togglePin(GPIO_PIN_LED1);
	CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);
	__DSB();
}

int32_t Timer4_init(uint32_t period)
{
	volatile uint32_t cpuTIntPending = 0U;

	/* Initialize CPUTimer with interrupt enable */
	CPUTimer_init(CPUTIMER1_BASE, period-1);

	Interrupt_register(INT_TIMER1, &APB_TIMER1_IRQ_HANDLER);
	Interrupt_enable(INT_TIMER1);

	return 0;
}


void IPC_Send(uint32_t cmd, uint32_t address, uint32_t len)
{
	ipc_cmd_s2m = cmd;
	ipc_address_s2m = address;
	ipc_len_s2m = len;

	CIDU_TriggerInterCoreInt(1, 0);
}

int main(void)
{
	volatile uint32_t response_flag = 0;

    IPC_init(IPC_CPU2_L_CPU1_R);
    IPC_sync(IPC_CPU2_L_CPU1_R, IPC_FLAG0);

	__disable_irq();

	Interrupt_register(InterCore_IRQn, &IPC_ISR2);
	Interrupt_SetPriority(InterCore_IRQn, 3, 0);
	Interrupt_enable(InterCore_IRQn);

	Timer4_init(DEVICE_APBCLK_FREQ / 100);	//10ms

	__enable_irq();

	while (1) {
		if (task10msFlag == 1U)
		{
			task10msFlag = 0U;
			sys_tick_response += 10;
		}

		if (ipc.Rx_flag == 1U)
		{
			if ((ipc.Rx_cmd == TICK_REQUEST_CMD_IMM) && (ipc.Rx_len == 0U))
			{
				sys_tick_request = sys_tick_response;
				response_flag = 1U;
			}
			else if ((ipc.Rx_cmd == TICK_REQUEST_CMD) && (ipc.Rx_len == 4U))
			{
				sys_tick_request = *(volatile uint32_t *)(ipc.Rx_addr);
				response_flag = 1U;
			}
			ipc.Rx_flag = 0U;
		}

		if ((sys_tick_response == sys_tick_request) && (response_flag == 1U))
		{
			response_flag =0U;
			Ipc_s2m_Data[0] = sys_tick_response;
			IPC_Send(TICK_RESPONSE_CMD, (uint32_t)&Ipc_s2m_Data[0], sizeof(uint32_t));
		}
	}

	return 0;
}
