/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <string.h>

#include "fdp_err.h"
#include "fdp.h"

#include "can_ctrl_dev.h"
#include "device.h"

#define CAN_CTRL_BASE		CANA_BASE
#define CAN_CTRL_IRQ_NUM	INT_CANA

#if (GS32_PART_NUM==0x0374 || GS32_PART_NUM==0x0377 || GS32_PART_NUM==0x0379)
#define CAN_CTRL_RX_PIN			GPIO_70_CANA_RX
#define CAN_CTRL_TX_PIN			GPIO_71_CANA_TX
#elif (GS32_PART_NUM==0x0650)
#define CAN_CTRL_RX_PIN			GPIO_58_CANA_RX
#define CAN_CTRL_TX_PIN			GPIO_59_CANA_TX
#elif (GS32_PART_NUM==0x0388A)
#define CAN_CTRL_RX_PIN			GPIO_36_CANA_RX
#define CAN_CTRL_TX_PIN			GPIO_37_CANA_TX
#endif

#define CAN_CTRL_TX_ID		0x456
#define CAN_CTRL_RX_ID		0x123

#define CAN_MSG_MAX_LEN		CAN_DATA_LENGTH_8

__INTERRUPT void can_interrupt_irq_handler(void)
{
	uint32_t status;
	CAN_RxMessage_t RxMsgBuf;

	status = CAN_getIntrStatus(CAN_CTRL_BASE);

	if (status & CAN_RTIF_RIF_M) {
//		CAN_receiveMsg(CAN_CTRL_BASE, &RxMsgBuf);
		fdp_receive_data_to_buffer();
	}

	/*
	 * Check if Rx buffer is empty.If the Rx buffer is not empty,
	 * do not clear the receive interrupt.
	 */
	if (CAN_getRxBufState(CAN_CTRL_BASE) == CAN_RXBUFFER_EMPTY)
		CAN_clearIntrStatus(CAN_CTRL_BASE, CAN_RTIF_RIF_M);
	__DSB();
}

int can_ctrl_init(void)
{
	CAN_InitParams_t initParams = {0};
	CAN_FilterElement_t stdFiltelem = {0};

	initParams.fdMode = false;
	initParams.txbmode = CAN_TS_MODE_FIFO;
	initParams.timestamp = CAN_TIMESTAMP_EOF;

	/* accept all CAN ID frames. */
	stdFiltelem.acode = CAN_CTRL_RX_ID;
	stdFiltelem.amsk = 0;
	stdFiltelem.filerFrameType = CAN_IDTYPE_ONLY_STD;

	GPIO_setPinConfig(CAN_CTRL_TX_PIN);
	GPIO_setPinConfig(CAN_CTRL_RX_PIN);

	SysCtl_setCanClkSrcSel(CAN_CLK_TYPE_PLL);

	/* CAN clock source = 40MHz, PLL_CLK = 2 * SYS_CLK */
	SysCtl_setCanClkDiv((DEVICE_SYSCLK_FREQ * 2 / 40000000));

	/* Initial CAN module. */
	if (!CAN_initModule(CAN_CTRL_BASE))
		return -FDP_CHECKERR;

	/* Configuration CAN basic function. */
	if (!CAN_initConfig(CAN_CTRL_BASE, &initParams))
		return -FDP_CHECKERR;

	/* Set the arbitration segment bit rate based on sampling points. */
	if (!CAN_setBitRateSamplePoint(CAN_CTRL_BASE, 40000000, 500 * 1000, 0.8))
		return -FDP_CHECKERR;

	/* Set CAN ID acceptance filter 0. */
	if (!CAN_configMsgFilter(CAN_CTRL_BASE, 0, &stdFiltelem))
		return -FDP_CHECKERR;

	/* Enable CAN ID acceptance filter 0. */
	if (!CAN_configMsgFilterEnable(CAN_CTRL_BASE, 0))
		return -FDP_CHECKERR;

	/* Start running CAN module. */
	if (!CAN_startModule(CAN_CTRL_BASE))
		return -FDP_CHECKERR;

	/* Enable auto retransmission. */
	if (!CAN_setAutoRetransmission(CAN_CTRL_BASE, true))
		return -FDP_CHECKERR;

	if (!CAN_enableInterrupt(CAN_CTRL_BASE, CAN_RTIE_RIE_M))
		return -FDP_CHECKERR;

#if (FDP_IS_USE_INTERRUPT)
	Interrupt_SetPriority(CAN_CTRL_IRQ_NUM, 0, 0);
	Interrupt_register(CAN_CTRL_IRQ_NUM, &can_interrupt_irq_handler);
	Interrupt_enable(CAN_CTRL_IRQ_NUM);
#endif

	return FDP_SUCCESS;
}

int CAN_CTRL_Receiver(uint8_t *buf, uint32_t *index, uint32_t buf_len)
{
	CAN_RxMessage_t RxMsgBuf = {0};
	int len;

	while(CAN_receiveMsg(CAN_CTRL_BASE, &RxMsgBuf)) {
		len = CAN_getMessageLength(RxMsgBuf.dlc);
		for (uint32_t i = 0; i < len; i++) {
			buf[(*index)++] = RxMsgBuf.data[i];
			if (*index >= buf_len)
				*index = 0;
		}
	}

	return FDP_SUCCESS;
}

int CAN_CTRL_transmit(uint8_t *buf, uint32_t len)
{
	CAN_TxMessage_t TxMsgBuf = {0};
	int datalen, maxlen;
	uint32_t i = 0;

	if (len == 0)
		return FDP_SUCCESS;

	TxMsgBuf.id = CAN_CTRL_TX_ID;

	maxlen = CAN_getMessageLength(CAN_MSG_MAX_LEN);

	while (len) {
		if (len >= maxlen) {
			TxMsgBuf.dlc = CAN_MSG_MAX_LEN;
			len -= maxlen;
		} else {
			TxMsgBuf.dlc = len;
			len = 0;
		}

		datalen = CAN_getMessageLength(TxMsgBuf.dlc);
		memcpy(TxMsgBuf.data, &buf[i], datalen);
		i += datalen;

		CAN_transmitMsg(CAN_CTRL_BASE, &TxMsgBuf);
	}

	return FDP_SUCCESS;
}
