/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>

#include "fdp_err.h"
#include "fdp.h"

#include "uart.h"
#include "interrupt.h"
#include "board_cfg.h"
#include "device.h"

#define UART_BASE_ADDRESS		SCIA_BASE

#define UART_RX_IRQ_NUM			INT_SCIA
#define BITRATE					300000U

#if (GS32_PART_NUM==0x0374 || GS32_PART_NUM==0x0377 || GS32_PART_NUM==0x0379)
#define SCI_TX_PIN                 GPIO_84_SCIA_TX
#define SCI_RX_PIN                 GPIO_85_SCIA_RX
#define SCI_TX_PIN_NUM			84
#define SCI_RX_PIN_NUM			85
#elif (GS32_PART_NUM==0x0650)
#define SCI_TX_PIN                 GPIO_12_SCIA_TX
#define SCI_RX_PIN                 GPIO_13_SCIA_RX
#define SCI_TX_PIN_NUM			12
#define SCI_RX_PIN_NUM			13
#elif (GS32_PART_NUM==0x0388A)
#define SCI_TX_PIN                 GPIO_29_SCIA_TX
#define SCI_RX_PIN                 GPIO_28_SCIA_RX
#define SCI_TX_PIN_NUM			29
#define SCI_RX_PIN_NUM			28
#endif

static void Uart_SetPinmux(void)
{
	GPIO_setPinConfig(SCI_TX_PIN);
	GPIO_setPinConfig(SCI_RX_PIN);
	GPIO_setPadConfig(SCI_TX_PIN_NUM, GPIO_PIN_TYPE_PULLUP);
	GPIO_setPadConfig(SCI_RX_PIN_NUM, GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(SCI_TX_PIN_NUM, GPIO_QUAL_SYNC);
	GPIO_setQualificationMode(SCI_RX_PIN_NUM, GPIO_QUAL_SYNC);
}
__INTERRUPT void SCI_IRQ_HANDLER(void)
{
	uint32_t irq_status = SCI_getInterruptStatus(UART_BASE_ADDRESS);

	if (irq_status == SCI_RX_DATA_AVAILABLE)
		fdp_receive_data_to_buffer();

	SCI_clearInterruptStatus(UART_BASE_ADDRESS);
	__DSB();
}

int Uart_Init(void)
{
	Uart_SetPinmux();

	SCI_enableLoopback(UART_BASE_ADDRESS);

	SCI_reset(UART_BASE_ADDRESS);

	SCI_setConfig(UART_BASE_ADDRESS, DEVICE_APBCLK_FREQ, BITRATE,
				 (SCI_CONFIG_WLEN_8 | SCI_CONFIG_STOP_ONE | SCI_CONFIG_PAR_NONE));

	SCI_ClearReset(UART_BASE_ADDRESS);

	SCI_clearInterruptStatus(UART_BASE_ADDRESS);
	SCI_enableFIFO(UART_BASE_ADDRESS);
	SCI_setFIFOInterruptLevel(UART_BASE_ADDRESS, SCI_FIFO_TX0, SCI_FIFO_RX1);

	SCI_disableInterrupt(UART_BASE_ADDRESS, 0xff);
	SCI_enableInterrupt(UART_BASE_ADDRESS, SCI_RX_DATA_AVAILABLE_INT);

#if (FDP_IS_USE_INTERRUPT)

	Interrupt_register(UART_RX_IRQ_NUM, &SCI_IRQ_HANDLER);
	Interrupt_enable(UART_RX_IRQ_NUM);

#endif

	SCI_disableLoopback(UART_BASE_ADDRESS);

	return FDP_SUCCESS;
}

int Uart_Receiver(uint8_t *buf, uint32_t *index, uint32_t buf_len)
{
	if (buf == NULL || index == NULL)
		return -FDP_ERRINPARAM;

	while(SCI_rxDataReady(UART_BASE_ADDRESS)) {
		buf[(*index)++] = (uint8_t)SCI_readCharNonBlocking(UART_BASE_ADDRESS);
		if (*index >= buf_len)
			*index = 0;
	}

	return FDP_SUCCESS;
}

int Uart_transmit(uint8_t *buf, uint32_t len)
{
	uint16_t temp;
	uint32_t i;

	if (buf == NULL || len == 0)
		return -FDP_ERRINPARAM;

	for (i = 0; i < len; i++) {
		temp = buf[i];
		SCI_writeCharArray(UART_BASE_ADDRESS, &temp, 1);
	}

	return FDP_SUCCESS;
}
