/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <string.h>

#include "uart.h"
#include "hal_uart_transfer.h"

volatile uint32_t jump_cmd_ready = 0;
uint8_t app_uart_buf[8];
uint32_t uart_rx_index = 0;

static void Uart_SetPinmux(void)
{
	GPIO_setPinConfig(SCI_TX_PIN);
	GPIO_setPinConfig(SCI_RX_PIN);
	GPIO_setPadConfig(SCI_TX_PIN_NUM, GPIO_PIN_TYPE_PULLUP);
	GPIO_setPadConfig(SCI_RX_PIN_NUM, GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(SCI_TX_PIN_NUM, GPIO_QUAL_SYNC);
	GPIO_setQualificationMode(SCI_RX_PIN_NUM, GPIO_QUAL_SYNC);
}

__INTERRUPT void SCI_IRQ_HANDLER(void)
{
	uint32_t irq_status = SCI_getInterruptStatus(UART_BASE_ADDRESS);

	if (irq_status & SCI_INT_RXFF) {
		while(!(SCI_getRxFIFOStatus(UART_BASE_ADDRESS) == SCI_FIFO_RX0)) {
			app_uart_buf[(uart_rx_index)++] = (uint8_t)SCI_readCharNonBlocking(UART_BASE_ADDRESS);
			if (uart_rx_index >= 4) {
				if (memcmp(app_uart_buf, "JUMP", 4) == 0)
					jump_cmd_ready = 1;
				else
					uart_rx_index = 0;
			}
		}
	}

	SCI_clearInterruptStatus(UART_BASE_ADDRESS, SCI_INT_RXFF);
	__DSB();
}

void Uart_Init(void)
{
	Uart_SetPinmux();

	SCI_enableLoopback(UART_BASE_ADDRESS);

	SCI_disableFIFO(UART_BASE_ADDRESS);
	while (!SCI_getRxFIFOStatus(UART_BASE_ADDRESS) == SCI_FIFO_RX0) {
		SCI_readCharBlockingFIFO(UART_BASE_ADDRESS);
	}
	SCI_resetChannels(UART_BASE_ADDRESS);
	SCI_clearInterruptStatus(UART_BASE_ADDRESS, SCI_INT_RXFF | SCI_INT_TXFF | SCI_INT_TXRDY);

	SCI_setConfig(UART_BASE_ADDRESS, DEVICE_APBCLK_FREQ, BITRATE,
				 (SCI_CONFIG_WLEN_8 | SCI_CONFIG_STOP_ONE | SCI_CONFIG_PAR_NONE));
	SCI_enableFIFO(UART_BASE_ADDRESS);
	SCI_setFIFOInterruptLevel(UART_BASE_ADDRESS, SCI_FIFO_TX0, SCI_FIFO_RX1);

	SCI_disableInterrupt(UART_BASE_ADDRESS, 0xff);
	SCI_disableInterrupt(UART_BASE_ADDRESS, SCI_INT_TXFF | SCI_INT_TXRDY);
	SCI_enableInterrupt(UART_BASE_ADDRESS, SCI_INT_RXFF);

	Interrupt_register(UART_RX_IRQ_NUM, &SCI_IRQ_HANDLER);
	Interrupt_enable(UART_RX_IRQ_NUM);

	SCI_disableLoopback(UART_BASE_ADDRESS);
}
