/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>

#include "device.h"
#include "log.h"
#include "board_cfg.h"
#include "app_info.h"
#include "hal_uart_transfer.h"
#include "hal_can_transfer.h"
#include "hal_spi_transfer.h"
#include "string.h"

/* Global variable */
volatile uint32_t Timer10msCNT = 0;
volatile uint32_t task1sFlag = 0;

volatile BOOTAPP_SHARE_BSS_T uint32_t updateFlag;

extern uint32_t __app_start_addr;
extern uint32_t jump_cmd_ready;
extern uint32_t __image_start_offset;
extern uint32_t FLASH_BOOT_ADDR;

#define FDP_BOOT_UPDATE_FLAG		0x9E8F6C2AU
#define FDP_BOOT_UPDATE_DISABLE		0x5A5A5A5AU

const APP_ENDM_T   uint32_t App_End_Mark[8] = {
		0xAAAAAAAA, 0xBBBBBBBB, 0xCCCCCCCC, 0xDDDDDDDD,
		0xAAAAAAAA, 0xBBBBBBBB, 0xCCCCCCCC, 0xDDDDDDDD,
};


FDP_APP_INFO fdp_app_image_info_t app0_info = {
	.app_start_addr = (uint32_t)&__app_start_addr,
	.offset = (uint32_t)&__image_start_offset,
	.image_check = 0xA5A5A5A5,
	.image_size = 0,
	.key = 0x9527,
	.key_check = 0xA5A5A5A5,
	.version = 1,
	.App_Start_Mark[0] = 0xAAAAAAAA,
	.App_Start_Mark[1] = 0xBBBBBBBB,
	.App_Start_Mark[2] = 0xCCCCCCCC,
	.App_Start_Mark[3] = 0xDDDDDDDD,
};

__INTERRUPT void APB_TIMER1_IRQ_HANDLER(void)
{
	Timer10msCNT ++;

	HAL_GPIO_Toggle(B, 40);		//GPIO_PIN_LED1 is GPIO40

	if (Timer10msCNT >= 100)
	{
		task1sFlag = 1;
		Timer10msCNT = 0U;
	}

	CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);

	__DSB();
}

int32_t Timer1_init(uint32_t period)
{
	/* Initialize CPUTimer with interrupt enable */
	CPUTimer_init(CPUTIMER1_BASE, period -1U);

	Interrupt_register(INT_TIMER1, &APB_TIMER1_IRQ_HANDLER);
	Interrupt_enable(INT_TIMER1);

	return 0;
}

void fdp_set_entry_load_mode(bool enable)
{
	if (enable)
		updateFlag = FDP_BOOT_UPDATE_FLAG;
	else
		updateFlag = FDP_BOOT_UPDATE_DISABLE;
}

void fdp_jump_to_address(uintptr_t addr)
{
	asm volatile ("jr %0" :: "r"(addr));
}

static void lin_module_uart_mode_init(uint32_t LINBase, uint32_t baud)
{
	LIN_initParam_t LINinitParam = {0};
	LIN_LinModeInitParam_t LINModeInitParam = {0};

	LINinitParam.bit_order = LIN_TRANSFER_LSB;
	LINinitParam.char_length = LIN_CHAR_LENGTH_8BITS;
	LINinitParam.opmode = LIN_UART_MODE;
	LINinitParam.chl_mode = LIN_CHAL_NORMAL_MODE;
	LINinitParam.stopbit = LIN_STOP_BIT_1;
	LINinitParam.parity = LIN_PARITY_NONE;

	/* Initialize LIN module. */
	LIN_initModule(LINBase, &LINinitParam);

	/* Set baud rate. */
#if (IS_GS32F00xx(0x30))
	uint32_t lin_clk_div = SysCtl_getLinClkDiv();
	lin_clk_div *= 2;

	LIN_setBaudRate(LINBase,(DEVICE_PLLCLK_FREQ / lin_clk_div), baud, LIN_OP_ASYNC_MODE, LIN_OVERSAMP_16);
#else
	LIN_setBaudRate(LINBase, DEVICE_APBCLK_FREQ, baud, LIN_OP_ASYNC_MODE, LIN_OVERSAMP_16);
#endif

	/* Enable receiver. */
	LIN_enableReceiver(LINBase, true);

	/* Enable transmitter. */
	LIN_enableTransmitter(LINBase, true);
}

static void lin_clk_init(void)
{
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_LIN);

#if (IS_GS32F00xx(0x30))

	int Lin_defaultClkDiv = 2;

	SysCtl_setLinClkDiv(Lin_defaultClkDiv);

#endif

	SysCtl_resetLin();
}

void lin_init(uint32_t LINBase, uint32_t baud)
{

}
void cUartPrint_init(uint32_t SCIBase, uint32_t baud)
{
	GPIO_setPinConfig(LOG_TX_PIN);
	GPIO_setPinConfig(LOG_RX_PIN);
    GPIO_setPadConfig(LOG_TX_PIN_NUM, GPIO_PIN_TYPE_PULLUP);
    GPIO_setPadConfig(LOG_RX_PIN_NUM, GPIO_PIN_TYPE_PULLUP);
	/* Initialize LIN clock */
	lin_clk_init();

	/* Initialize LIN to UART mode. */
	lin_module_uart_mode_init(SCIBase, baud);
}

void cInit_Leds(void)
{
    GPIO_setPinConfig(GPIO_CFG_LED1);
    GPIO_setPinConfig(GPIO_CFG_LED2);
    GPIO_setPadConfig(GPIO_PIN_LED1, GPIO_PIN_TYPE_STD);
    GPIO_setPadConfig(GPIO_PIN_LED2, GPIO_PIN_TYPE_STD);
    HAL_GPIO_OutDir(B, 40);				//GPIO_PIN_LED1 is GPIO40
    HAL_GPIO_OutDir(B, 39);				//GPIO_PIN_LED2 is GPIO39
    HAL_GPIO_Clr(B, 40);					//GPIO_PIN_LED1 is GPIO40
    HAL_GPIO_Set(B, 39);					//GPIO_PIN_LED2 is GPIO39
}

void cPrint_ticks(uint32_t ticks)
{
	uint8_t log_content[256];
	char buf[16] = "\0";
    uint32_t hour, min, sec, ms;
    uint32_t i;
    uint32_t status;
    uint32_t __ms = __get_rv_cycle()/(DEVICE_SYSCLK_FREQ/1000);

	ms = __ms % 1000;
	__ms /= 1000;
	sec = __ms % 60;
	__ms /= 60;
	min = __ms % 60;
	__ms /= 60;
	hour = __ms % 24;
	sprintf(buf, "%02d:%02d:%02d.%03d", hour, min, sec, ms);
	sprintf(log_content, "[%4s] %s %-5s %s:%d: ticks: %d\r\n", "CPU1", buf, "INFO", __FILE__, __LINE__, ticks);
	for (i = 0; i <= strlen(log_content); )
	{
		LIN_getStatus(LOG_SCI_BASE, &status);
		if (status & LIN_USART_CHANNEL_TXRDY)
			LIN_UartPutCharData(LOG_SCI_BASE, log_content[i++]);

	}
}

void main(void)
{
	uint32_t tickCnt = 0;

	__disable_irq();

	Device_init();

    /* Initial Log SCI port  */
	cUartPrint_init(LOG_SCI_BASE, 115200);
	log_set_level(LOG_INFO);

	/* Initial Led */
	cInit_Leds();

	Uart_Init();

	can_ctrl_init();

	spi_init();

	Timer1_init(DEVICE_APBCLK_FREQ / 100);	//10ms

	__enable_irq();

	while (1) {

		if (jump_cmd_ready) {
			jump_cmd_ready = 0;
			__disable_irq();
			SCI_enableLoopback(UART_BASE_ADDRESS);
			CAN_initModule(CAN_BASE);
			fdp_set_entry_load_mode(true);
			fdp_jump_to_address((uintptr_t)&FLASH_BOOT_ADDR);
		}

		if (task1sFlag == 1U)
		{
			task1sFlag = 0;
			tickCnt += 100U;
			HAL_GPIO_Toggle(B, 39);			//GPIO_PIN_LED2 is GPIO39
			cPrint_ticks(tickCnt);
		}
	}
}
