#ifndef CFFT_F32_H
#define CFFT_F32_H

#include "device.h"

/**
 * @brief Instance structure for the floating-point CFFT/CIFFT function.
 */
typedef struct
{
    uint16_t fftLen;                   /**< length of the FFT. */
    const float32_t *pTwiddle;         /**< points to the Twiddle factor table. */
    const uint16_t *pBitRevTable;      /**< points to the bit reversal table. */
    uint16_t bitRevLength;             /**< bit reversal table length. */
} riscv_cfft_instance_f32;

/**
 * @brief Structure for CFFT configuration and data management
 */
typedef struct {
    uint16_t FFTSize;           /**< Size of the FFT operation */
    uint16_t Stages;            /**< Number of FFT stages */
    float32_t *InPtr;           /**< Pointer to input buffer */
    float32_t *OutPtr;          /**< Pointer to output buffer */
    float32_t *CoefPtr;         /**< Pointer to the twiddle factors */
    float32_t *CurrentInPtr;    /**< Current input buffer pointer */
    float32_t *CurrentOutPtr;   /**< Current output buffer pointer */
    const riscv_cfft_instance_f32 *rvfft; /**< NMSIS FFT instance pointer */
} CFFT_F32_STRUCT;

/** @brief Handle type for CFFT_F32_STRUCT */
typedef CFFT_F32_STRUCT* CFFT_F32_STRUCT_Handle;

/**
 * @brief Perform complex FFT operation
 * @param hndCFFT_F32 Pointer to CFFT structure
 */
extern void CFFT_f32(CFFT_F32_STRUCT_Handle hndCFFT_F32);

/**
 * @brief Perform inverse complex FFT operation
 * @param hndCFFT_F32 Pointer to CFFT structure
 */
extern void ICFFT_f32(CFFT_F32_STRUCT_Handle hndCFFT_F32);

/**
 * @brief Calculate magnitude of complex FFT results
 * @param hndCFFT_F32 Pointer to CFFT structure
 */
extern void CFFT_f32s_mag(CFFT_F32_STRUCT_Handle hndCFFT_F32);

/**
 * @brief Calculate phase of complex FFT results
 * @param hndCFFT_F32 Pointer to CFFT structure
 */
extern void CFFT_f32_phase(CFFT_F32_STRUCT_Handle hndCFFT_F32);

/**
 * @brief Generate sine and cosine tables for FFT
 * @param cfft Pointer to CFFT structure
 */
extern void CFFT_f32_sincostable(CFFT_F32_STRUCT_Handle cfft);

/**
 * @brief Initialize CFFT structure with specified parameters
 * @param cfft_struct Pointer to CFFT structure to initialize
 * @param fft_size Size of the FFT operation
 * @param fft_in_buf Pointer to input buffer
 * @param fft_mag_buf Pointer to magnitude buffer
 * @param fft_phase_buf Pointer to phase buffer
 * @return Handle to initialized CFFT structure
 */
extern CFFT_F32_STRUCT_Handle CFFT_f32_init(CFFT_F32_STRUCT *cfft_struct,
                                    uint16_t fft_size,
                                    float32_t *fft_in_buf,
                                    float32_t *fft_mag_buf,
                                    float32_t *fft_phase_buf);

/**
 * @brief Macro for TMU0 phase calculation (alias for CFFT_f32s_mag)
 */
#define CFFT_f32_phase_TMU0   CFFT_f32s_mag

/**
 * @brief Macro for TMU0 magnitude calculation (alias for CFFT_f32s_mag)
 */
#define CFFT_f32s_mag_TMU0    CFFT_f32s_mag



/**
  @brief         Pre-initialized instances of the floating-point complex FFT structure.
  @details       These instances contain pre-computed twiddle factors and bit reversal tables
                 for different FFT lengths. They can be used directly without calling riscv_cfft_init_f32().
                 Each instance is optimized for its specific FFT length.
  @note          These instances are read-only and should not be modified.
  @note          Available FFT lengths are: 16, 32, 64, 128, 256, 512, 1024, 2048, and 4096 points.
  @note          To use these instances, simply pass their address to riscv_cfft_f32() function.
  @example
                 const riscv_cfft_instance_f32 *S = &riscv_cfft_sR_f32_len256;
                 riscv_cfft_f32(S, input, 0, 1);
 */
extern const riscv_cfft_instance_f32 riscv_cfft_sR_f32_len16;
extern const riscv_cfft_instance_f32 riscv_cfft_sR_f32_len32;
extern const riscv_cfft_instance_f32 riscv_cfft_sR_f32_len64;
extern const riscv_cfft_instance_f32 riscv_cfft_sR_f32_len128;
extern const riscv_cfft_instance_f32 riscv_cfft_sR_f32_len256;
extern const riscv_cfft_instance_f32 riscv_cfft_sR_f32_len512;
extern const riscv_cfft_instance_f32 riscv_cfft_sR_f32_len1024;
extern const riscv_cfft_instance_f32 riscv_cfft_sR_f32_len2048;
extern const riscv_cfft_instance_f32 riscv_cfft_sR_f32_len4096;

/**
  @brief         Processing function for the floating-point complex FFT.
  @param[in]     S              points to an instance of the floating-point CFFT structure
  @param[in,out] p1             points to the complex data buffer of size <code>2*fftLen</code>. Processing occurs in-place
  @param[in]     ifftFlag       flag that selects transform direction
                   - value = 0: forward transform
                   - value = 1: inverse transform
  @param[in]     bitReverseFlag flag that enables / disables bit reversal of output
                   - value = 0: disables bit reversal of output
                   - value = 1: enables bit reversal of output
 */

extern void riscv_cfft_f32(
  const riscv_cfft_instance_f32 * S,
        float32_t * p1,
        uint8_t ifftFlag,
        uint8_t bitReverseFlag);

/**
  @brief         Floating-point complex magnitude.
  @param[in]     pSrc        points to input vector
  @param[out]    pDst        points to output vector
  @param[in]     numSamples  number of samples in each vector
 */
extern void riscv_cmplx_mag_f32(
  const float32_t * pSrc,
        float32_t * pDst,
        uint32_t numSamples);


#endif // CFFT_F32_H
