/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *   @file    emavg.h
 *   @brief   Contains the public interface to the Exponential Moving
 *            Average (EMAVG)
 *
 */

#ifndef EMAVG_H
#define EMAVG_H

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "inc/hw_types.h"
#include <math.h>
#include <stdint.h>

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

//! \brief          Defines the Exponential Moving Average (EMAVG) structure
//!
//! \details        The emavg can be used to perform exponential moving
//!                 average calculation
//!
typedef struct {
    float32_t out;
    float32_t multiplier;
} EMAVG;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */
/**
 * \brief      resets internal storage data
 * \param v    The EMAVG structure
 */
__STATIC_INLINE void
EMAVG_reset(EMAVG *v) {
    v->out = 0;
}

/**
 * \brief      configures EMAVG module
 * \param v    The EMAVG structure
 * \param multiplier Multiplier value
 */
__STATIC_INLINE void
EMAVG_config(EMAVG *v,
             float32_t multiplier) {
    v->multiplier = multiplier;
}

/**
 * \brief      Run EMAVG module
 * \param v    The EMAVG structure
 * \param in   Input
 */
__STATIC_INLINE void
EMAVG_run(EMAVG *v, float in) {
    v->out = ((in - v->out) * v->multiplier) + v->out;
}

#ifdef __cplusplus
}
#endif

#endif /*EMAVG_H*/
