/*
 * hpf_driver.h
 *
 *  Created on: 2025 Jul 14
 *      Author: Gejian
 */

#ifndef HPF_INCLUDE_H_
#define HPF_INCLUDE_H_

#include "device.h"

/**
 * The base address of the CLB module that implements the TXEN function
 */
#define HPF_TXEN_CLB		CLB1_BASE

/**
 * The base address of the CLB module that implements the Counter function
 */
#define HPF_COUNTER_CLB		CLB2_BASE
/**
 * Base address of the UART module
 */
#define HPF_UART_BASE		SCIB_BASE

/**
 * Encoder startup process delay time.
 * Initialization phase
 * Change of communication settings phase
 * Operational phase
 */
#define HPF_ENC_INIT_DELAY_TIME		100000 	/* us */
#define HPF_ENC_SET_UART_DELAY_TIME	800000	/* us */
#define HPF_ENC_OPER_DELAY_TIME		200000	/* us */


/*
 * Encoder paramters
 */
/**
 * CLB working clock rate
 */
#define HPF_CLB_WORK_RATE	DEVICE_AHBCLK_FREQ
/**
 * UART working clock rate
 */
#define HPF_UART_WORK_RATE	DEVICE_APBCLK_FREQ
/**
 * The current communication frequency of the HIPERFACE encoder.
 */
#define HPF_DEFAULT_DATA_RATE	9600U
#define HPF_DATA_RATE			38400U
/**
 * The encoder address
 *
 * Note:
 * Modify according to the actual configuration of the encoder.
 */
#define HPF_ENC_ADDR				(0x0)
#define HPF_ENC_ADDR_FIELD(addr)	(0x40U | addr)

/**
 * Encoder single turn bit numbers
 */
#define HPF_SINGLETURN_BITNUM		(15U)
#define HPF_SINGLETURN_BITNUM_MASK	(0xFFFFU)

/**
 * The number of bits per frame
 */
#define HPF_DATA_BYTE_MASK			(0xFFU)

/**
 * HIPERFACE command identifier
 */
/**
 * No extra data is required when transmitting instructions.
 */
typedef enum
{
	HPF_CMD_READ_POS		=	0x42U,	/* 42h */
	HPF_CMD_RD_CNT_VAL		=	0x46U,	/* 46h */
	HPF_CMD_INC_CNT_VAL		=	0x47U,	/* 47h */
	HPF_CMD_MEM_STATUS		=	0x4EU,	/* 4Eh */
	HPF_CMD_RD_ENC_STAT		=	0x50U,	/* 50h */
	HPF_CMD_RD_TYPE_LABEL 	= 	0x52U,	/* 52h */
	HPF_CMD_ENC_RST			=	0x53U,	/* 53h */
	HPF_CMD_RD_VERSION		=	0x56U	/* 56h */
}HPF_CMD_NODATA;

#define HPF_CMD_SET_POS			0x43U	/* 43h */

#define HPF_CMD_RD_ANA_VAL		0x44U	/* 44h */
#define HPF_CMD_RST_CNT_VAL		0x49U	/* 49h */
#define HPF_CMD_DATA_FLD_STAT	0x4CU	/* 4Ch */

#define HPF_CMD_SET_ENC_ADDR	0x55U	/* 55h */
#define HPF_CMD_SET_UART_PARA	0x57U 	/* 57h */

#define HPF_CMD_SET_ACCESS_CODE 0x4FU	/* 4Fh */
#define HPF_CMD_CRT_DATA_FIELD  0x4DU 	/* 4Dh */

#define HPF_CMD_RD_DATA         0x4AU 	/* 4Ah */
#define HPF_CMD_STORE_DATA      0x4BU 	/* 4Bh */

/**
 * Command sends the number of data bytes,
 * excluding the address frame, instruction frame,
 * access code frame and checksum frame.
 *
 * @note:
 * "HPF_TX_BYTE_NUM_STORE_DATA" CMD does not include
 * the number of data frames.
 */
#define HPF_TX_BYTE_NUM_READ_POS           3U
#define HPF_TX_BYTE_NUM_SET_POS            8U
#define HPF_TX_BYTE_NUM_RD_ANA_VAL         4U
#define HPF_TX_BYTE_NUM_RD_CNT_VAL         3U
#define HPF_TX_BYTE_NUM_INC_CNT_VAL        3U
#define HPF_TX_BYTE_NUM_RST_CNT_VAL        4U
#define HPF_TX_BYTE_NUM_RD_DATA            7U
#define HPF_TX_BYTE_NUM_STORE_DATA         7U
#define HPF_TX_BYTE_NUM_DATA_FLD_STAT      4U
#define HPF_TX_BYTE_NUM_CRT_DATA_FIELD     6U
#define HPF_TX_BYTE_NUM_MEM_STATUS         3U
#define HPF_TX_BYTE_NUM_SET_ACCESS_CODE    6U
#define HPF_TX_BYTE_NUM_RD_ENC_STAT        3U
#define HPF_TX_BYTE_NUM_RD_TYPE_LABEL      3U
#define HPF_TX_BYTE_NUM_ENC_RST            3U
#define HPF_TX_BYTE_NUM_SET_ENC_ADDR       5U
#define HPF_TX_BYTE_NUM_RD_VERSION         3U
#define HPF_TX_BYTE_NUM_SET_UART_PARA      5U


/**
 * Command receives the number of data bytes,
 * excluding the address frame, instruction frame,
 * access code frame and checksum frame.
 *
 * @note:
 * "HPF_RX_BYTE_NUM_RD_DATA" CMD does not include
 * the number of data frames.
 */
#define HPF_RX_BYTE_NUM_READ_POS           7U
#define HPF_RX_BYTE_NUM_SET_POS            3U
#define HPF_RX_BYTE_NUM_RD_ANA_VAL         6U
#define HPF_RX_BYTE_NUM_RD_CNT_VAL         6U
#define HPF_RX_BYTE_NUM_INC_CNT_VAL        3U
#define HPF_RX_BYTE_NUM_RST_CNT_VAL        3U
#define HPF_RX_BYTE_NUM_RD_DATA            7U
#define HPF_RX_BYTE_NUM_STORE_DATA         6U
#define HPF_RX_BYTE_NUM_DATA_FLD_STAT      5U
#define HPF_RX_BYTE_NUM_CRT_DATA_FIELD     5U
#define HPF_RX_BYTE_NUM_MEM_STATUS         5U
#define HPF_RX_BYTE_NUM_SET_ACCESS_CODE    4U
#define HPF_RX_BYTE_NUM_RD_ENC_STAT        4U
#define HPF_RX_BYTE_NUM_RD_TYPE_LABEL      7U
#define HPF_RX_BYTE_NUM_ENC_RST            0U
#define HPF_RX_BYTE_NUM_SET_ENC_ADDR       3U
#define HPF_RX_BYTE_NUM_RD_VERSION         6U
#define HPF_RX_BYTE_NUM_SET_UART_PARA      4U

/**
 * Array storing the current access code.
 */
extern uint8_t hpf_accessCodeArr[4];
/**
 * Default access code
 */
#define HPF_ACCESS_CODE0				0x55U
#define HPF_ACCESS_CODE1				0x55U
#define HPF_ACCESS_CODE2				0x55U
#define HPF_ACCESS_CODE3				0x55U

/**
 * Analog channel code
 */
#define HPF_ANACHANNEL					0x48U
#define HPF_LEGACY_ANACHANNEL			0xF0U

/**
 * calculating temperature
 */
#define get_legacyTemp(x)	(float)((x+40)/2.048)

/**
 * @brief ڴ
 * 
 * żУ
 * ʱʱ
 * ģʽ
 * Ⱥ궨Լṹ
 */
#define HPF_DATA_RATE_600HZ		0x0U
#define HPF_DATA_RATE_1200HZ	0x1U
#define HPF_DATA_RATE_2400HZ	0x2U
#define HPF_DATA_RATE_4800HZ	0x3U
#define HPF_DATA_RATE_9600HZ	0x4U /* default */
#define HPF_DATA_RATE_19200HZ	0x5U
#define HPF_DATA_RATE_38400HZ	0x6U

#define HPF_PARITY_MODE_NO		0x0U
#define HPF_PARITY_MODE_EVEN	0x2U /* default */
#define HPF_PARITY_MODE_ODD		0x3U

#define HPF_TIMEOUT_MODE_22S	0x0U
#define HPF_TIMEOUT_MODE_55S	0x1U /* default */

#define HPF_BUS_MODE_BUS_SLAVE	0x0U
#define HPF_BUS_MODE_ST_SLAVE	0x1U /* default */

typedef struct
{
	uint8_t dataRate:3;
	uint8_t rsvd:1;
	uint8_t parity:2;
	uint8_t timeout:1;
	uint8_t busMode:1;
}HPF_SERIAL_IF_STR;
typedef union
{
	HPF_SERIAL_IF_STR bit;
	uint8_t all;
}HPF_SERIAL_IF;
extern HPF_SERIAL_IF hpf_serial_if;

typedef struct
{
	HPF_SERIAL_IF serialInface;
	uint8_t encType;
	uint8_t usingEEPROMSize;
	uint8_t options;
}HPF_ENC_MSG;
extern HPF_ENC_MSG hpf_enc_typeLabel;

/**
 * @brief Ϣṹ
 */
typedef struct
{
	uint8_t addrFrame;
	uint8_t cmdID;
	uint16_t txNum;
	uint16_t rxNum;
	uint8_t errorCode;
	uint8_t checkSum;
	Bool warningBit;
    volatile Bool  dataReady;
}HPF_DATASTRUCT;

extern volatile uint8_t hpfRxDateArr[134];

/**
 * ֶνṹ
 */
#define HPF_DATAFIELD_BYTESIZE_ONEBLOCK		16U

#define HPF_DATAFIELD_SIZE_16BYTE		0
#define HPF_DATAFIELD_SIZE_32BYTE		1
#define HPF_DATAFIELD_SIZE_48BYTE		2
#define HPF_DATAFIELD_SIZE_64BYTE		3
#define HPF_DATAFIELD_SIZE_80BYTE		4
#define HPF_DATAFIELD_SIZE_96BYTE		5
#define HPF_DATAFIELD_SIZE_112BYTE		6
#define HPF_DATAFIELD_SIZE_128BYTE		7

#define HPF_DATAFIELD_CODE_EN			1
#define HPF_DATAFIELD_CODE_DIS			0

#define HPF_DATAFIELD_ACCESSCODE0		0
#define HPF_DATAFIELD_ACCESSCODE1		1
#define HPF_DATAFIELD_ACCESSCODE2		2
#define HPF_DATAFIELD_ACCESSCODE3		3

#define HPF_DATAFIELD_RDONLY			0
#define HPF_DATAFIELD_RW				1

#define HPF_DATAFIELD_DEL				0
#define HPF_DATAFIELD_CRT_MODI  		1


typedef struct
{
	uint8_t dataFieldSize:3;/* ActualSize = (dataFieldSize+1)*16 (Byte)	*/
	uint8_t accessCodeEn:1;
	uint8_t accessCodeID:2;
	uint8_t writeEn:1;
	uint8_t dataFieldEn:1;
}HPF_DATA_FIELD_STATUS_BITS;

typedef union
{
	uint8_t dataField;
	HPF_DATA_FIELD_STATUS_BITS dataFieldBit;
}HPF_DATA_FIELD_STATUS_REG;

typedef struct
{
	uint8_t dataFieldNum;
	HPF_DATA_FIELD_STATUS_REG dataFieldStatus;
	uint8_t accessCode;
	uint16_t actualSize;
}HPF_DATA_FIELD;


/**
 * ؼ󣺾ֹڼųϣ˶ڼֹͣ
 * ҪֹͣҪִָ
 * شȡκжڳ
 */
/**
 * ERROR Status codes
 */
#define HPF_NO_FAULTS	0x00

/**
 * ʼڼڲ洢ֵ
 * HPF_ERR_IntChecksumջлдIDָ}
 * ڳʼ׶γHPF_ERR_IntAngularOffʾλУʧ
 */
typedef enum
{
	/* ȦֱʺͶȦô󡢴У׼ */
	HPF_ERR_AlignmentData = 0x01,				/* ؼ */
	/* EEPROMеλƫ򴫸ͬ */
	HPF_ERR_IntAngularOff = 0x02,				/* ؼ */
	/* ֶ */
	HPF_ERR_DataFieldPartTable = 0x03,			/* ؼ */
	/* EEPROM¶Ⱦ淶ΧLED淶Χͬ龯淶ΧϷΧֵ*/
	HPF_ERR_AnaLimitVlaue = 0x04,				/* ؼ */
	/* ޷EEPROMͨŻEEPROMеıʹ */
	HPF_ERR_IntI2CBus = 0x05,					/* ؼ */
	/* ڲ𻵣ID */
	HPF_ERR_IntChecksum = 0x06,					/* ؼ */

}HPF_FAULTGROUP_INIT;

/**
 * ˴漰ָͷĴ
 * ЩϢΪָӦ͵ģ洢ڴջС
 * ǲ
 */
typedef enum
{
	/* ڲŹ */
	HPF_ERR_Reset	 = 0x07,		/* ش */
	/* żУʧ */
	HPF_ERR_Parity   = 0x09,		/* ش·ָ */
	/* Уʧ */
	HPF_ERR_Checksum = 0x0A,		/* ش·ָ */
	/* δиָ */
	HPF_ERR_UnknownCMD = 0x0B,		/* ش·ָ */
	/* յ֡쳣 */
	HPF_ERR_NumbofTrans = 0x0C,		/* ش·ָ */
	/* յָӦ쳣 */
	HPF_ERR_TransParams = 0x0D,		/* ش·ָ */
}HPF_FAULTGROUP_PROT;

/**
 * ˴ִָʼڼֵܳûݴ洢صд
 * Щ󲻻洢ڴջСǲ
 */
typedef enum
{
	/* ֶֻ */
	HPF_ERR_SelectDataField	 = 0x0E,	/* ش󣬼޸ֶ״̬ */
	/* ֶǽյķ */
	HPF_ERR_AccessCode   = 0x0F,		/* ش󣬼޸ķ */
	/* ֶǳֶδСڴ治 */
	HPF_ERR_DataFieldSize = 0x10,		/* ش󣬼ٷֶδС */
	/* ֶηƫ Χ */
	HPF_ERR_WordAddress = 0x11,			/* ش󣬼޸ƫ */
	/* ʵֶαŲ */
	HPF_ERR_NonExistDataField = 0x12,	/* ش󣬼޸ķʵֶα */
}HPF_FAULTGROUP_DATA;

/**
 * ˴ִжȡλã42hдλã43hָʱֵܳд
 * ЩϢΪָӦ͵ģ洢ڴջС
 * £λöȡֹͣ
 */
typedef enum
{
	/* ѹ׼ڲѹ׼ */
	HPF_ERR_AnalogSignals = 0x01,		/* ش */
	/* ȡλʱλô */
	HPF_ERR_Postion	= 0x02U,			/* ؼ */
	/*  */
	HPF_ERR_SpeedOver = 0x1F,			/* ش */
	/* Ȧ */
	HPF_ERR_SinglePos = 0x20,			/* ؼ */
}HPF_FAULTGROUP_POS;

/**
 * ˴¶ȡLEDڲûֵľ
 */
typedef enum
{
	/* ģźֵ쳣 */
	HPF_ERR_AnalogVlaue = 0x1C, 		/* ؼҪλ */
	/* Χ */
	HPF_ERR_CurrentorLED = 0x1D,		/* ؼ */
	/* ¶ȳΧ */
	HPF_ERR_TempOver = 0x1E,			/* ش */
	/* ڲΧ */
	HPF_ERR_CounterOver = 0x08,			/* شҪָü*/
}HPF_FAULTGROUP_OTHER;

/**
 * Get the absolute position value
 */
#define HPF_GETABSPOS	( hpfRxDateArr[5] 		| \
					     (hpfRxDateArr[4]<<8)	| \
					     (hpfRxDateArr[3]<<16)	| \
					     (hpfRxDateArr[2]<<24)	)

#define HPF_GET_PERIODNUM(pos)	(pos >> 5)

/**
 * Get the counter value
 */
#define HPF_GETCOUNTER	( hpfRxDateArr[4]		| \
		 	 	 	 	 (hpfRxDateArr[3]<<8)	| \
		 	 	 	 	 ((hpfRxDateArr[2]<<16) )

/**
 * Get Encoder Status
 */
#define HPF_GET_ENCSTATUS	(hpfRxDateArr[2])

/**
 * Get Data Field Status
 */
#define HPF_GET_DATAFIELDSTATUS	(hpfRxDateArr[3])


#define HPF_GET_EXIST_DATAFIELNUM	(hpfRxDateArr[3])

/**
 * Initialize the peripherals and global variables
 */
void hpf_setupPeriph(uint32_t baud);

void hpf_txenInit(uint32_t baud);

void hpf_counterInit(void);

void hpf_uartInit(uint32_t baud);

/**
 * Set Txen delay time
 * In terms of communication speed.
 */
void hpf_setTxRate(uint32_t dataRate);
void hpf_setTxenDelay(uint32_t dataRate , uint32_t delayClkNum);

/**
 * One of the sending functions
 * Instructions that do not include any other data frames
 * except for address frames, instruction frames and checksum frames.
 * Use as per the following instructions:
 * 		enum value					  Describe
 *  - HPF_CMD_READ_POS		---		Read Position
 *  - HPF_CMD_RD_CNT_VAL	---		Read Counter
 *  - HPF_CMD_INC_CNT_VAL	---		Increment Counter
 *  - HPF_CMD_MEM_STATUS	---		Memory Status
 *  - HPF_CMD_RD_ENC_STAT	---		Read Encoder Status
 *  - HPF_CMD_RD_TYPE_LABEL	---		Read Type Label
 *  - HPF_CMD_ENC_RST		---		Encoder Reset
 *  - HPF_CMD_RD_VERSION	---		Read Version
 *
 */
void hpf_sendCommandNoData(uint8_t addr,HPF_CMD_NODATA cmd)
														__attribute__ ((section (".IlmFunc")));

#define hpf_readPosition(addr)		hpf_sendCommandNoData(addr,HPF_CMD_READ_POS)
#define hpf_readCounter(addr)		hpf_sendCommandNoData(addr,HPF_CMD_RD_CNT_VAL)
#define hpf_incrementCounter(addr)	hpf_sendCommandNoData(addr,HPF_CMD_INC_CNT_VAL)
#define hpf_memoryStatus(addr)		hpf_sendCommandNoData(addr,HPF_CMD_MEM_STATUS)
#define hpf_readEncoderStatus(addr)	hpf_sendCommandNoData(addr,HPF_CMD_RD_ENC_STAT)
#define hpf_readTypeLabel(addr)		hpf_sendCommandNoData(addr,HPF_CMD_RD_TYPE_LABEL)
#define hpf_resetEncoder(addr)		hpf_sendCommandNoData(addr,HPF_CMD_ENC_RST)
#define hpf_readVersion(addr)		hpf_sendCommandNoData(addr,HPF_CMD_RD_VERSION)

/**
 * One of the sending functions
 * Including an instruction that contains any other data frame
 * except for address frames, instruction frames and checksum frames.
 * Use as per the following instructions:
 * 		enum value					  Describe
 *  - HPF_CMD_RD_ANA_VAL	---		Read Analog Channel Value
 *  - HPF_CMD_RST_CNT_VAL	---		Reset Counter Value
 *  - HPF_CMD_DATA_FLD_STAT	---		Get Data Field Status
 */
void hpf_sendCommandOneData(uint8_t addr,uint8_t cmd,uint8_t data)
														__attribute__ ((section (".IlmFunc")));

#define hpf_readAnalogValue(addr,ChannelID)		\
			hpf_sendCommandOneData(addr,HPF_CMD_RD_ANA_VAL,ChannelID)
#define hpf_resetCounter(addr)					\
			hpf_sendCommandOneData(addr,HPF_CMD_RST_CNT_VAL,HPF_ACCESS_CODE0);
#define hpf_getDataFieldStatus(addr,dataField)	\
			hpf_sendCommandOneData(addr,HPF_CMD_DATA_FLD_STAT,dataField);

/**
 * One of the sending functions
 * Including two any other data frames as instructions
 * except for address frames, instruction frames and checksum frames.
 * Use as per the following instructions:
 * 		enum value					  Describe
 *  - HPF_CMD_SET_ENC_ADDR	---		Set Encoder Address
 *  - HPF_CMD_SET_UART_PARA	---		Set UART Interface Parameters
 */
void hpf_sendCommandTwoDatas(uint8_t addr,uint8_t cmd,uint8_t data0,uint8_t data1)
														__attribute__ ((section (".IlmFunc")));
#define hpf_setNewAddress(addr,newAddr)		\
			hpf_sendCommandTwoDatas(addr,HPF_CMD_SET_ENC_ADDR,newAddr,HPF_ACCESS_CODE0)
#define hpf_setSerialInterface(addr,uartPara)	\
			hpf_sendCommandTwoDatas(addr,HPF_CMD_SET_UART_PARA,uartPara,HPF_ACCESS_CODE0)

/**
 * One of the sending functions
 * Including three any other data frames as instructions
 * except for address frames, instruction frames and checksum frames.
 * Use as per the following instructions:
 * 		enum value					  Describe
 *  - HPF_CMD_SET_ACCESS_CODE	---		Set Access Code
 *  - HPF_CMD_CRT_DATA_FIELD	---		Create Data Field
 */
void hpf_sendCommandThreeDatas(uint8_t addr,uint8_t cmd,uint8_t data0,uint8_t data1,uint8_t data2)
														__attribute__ ((section (".IlmFunc")));
#define hpf_setAccessCode(addr,dataID,oldCode,newCode)	\
			hpf_sendCommandThreeDatas(addr,HPF_CMD_SET_ACCESS_CODE,dataID,oldCode,newCode)
#define hpf_createDataField(addr,dataFieldNum,dataFieldStatus,accessCode)	\
			hpf_sendCommandThreeDatas(addr,HPF_CMD_CRT_DATA_FIELD,dataFieldNum,dataFieldStatus,accessCode)

/**
 * One of the sending functions
 * Set position value.
 */
void hpf_setPosition(uint8_t addr , uint32_t *pos);

/**
 * One of the sending functions
 * Read the data in the data field of the encoder.
 */
Bool hpf_readEEPROMData(uint8_t addr,HPF_DATA_FIELD* dataMsg,uint16_t addrOff,uint16_t size);

/**
 * One of the sending functions
 * Write the data into the memory of the encoder.
 */
Bool hpf_writeEEPROMData(uint8_t addr,HPF_DATA_FIELD* dataMsg,uint8_t *pData,uint16_t addrOff,uint16_t size);

void hpf_setAnaCountValue(uint32_t value);
/**
 * Get Analog channel bilateral count value
 */
uint32_t hpf_getAnaCountValue(void);

void hpf_recieveTypeLabel(HPF_ENC_MSG *enc);

void hpf_delayMcycle(uint32_t sysClk,uint32_t delayTime);

#endif /* HPF_INCLUDE_H_ */
