/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>

#include "stl_analog_io.h"
#include "stl_err.h"
#include "driverlib.h"
#include "device.h"

stl_err_t stl_analog_io_input(uint32_t base, uint32_t result_base,
							ADC_SOCNumber soc_num, ADC_Channel chl,
							ADC_ClkPrescale clk_div, ADC_ReferenceMode ref_voltage_mode,
							ADC_ReferenceVoltage level, float expectation_val,
							float reference_voltage)
{
	float volt;
	float cmp_val;
	uint16_t ADCResult;

	ADC_setVREF(base, ref_voltage_mode, level);

	ADC_setINLTrim(base);

	/* Configures the analog-to-digital converter module prescaler. */
	ADC_setPrescaler(base, clk_div);

	/* Sets the timing of the end-of-conversion pulse. */
	ADC_setInterruptPulseMode(base, ADC_PULSE_END_OF_CONV);

	/* Powers up the analog-to-digital converter core. */
	ADC_enableConverter(base);

	ADC_disableBurstMode(base);

	ADC_setSOCPriority(base, ADC_PRI_ALL_ROUND_ROBIN);

	/* Start of Conversion 0 Configuration. */
	ADC_setupSOC(base, soc_num, ADC_TRIGGER_SW_ONLY, chl, 50U);

	DELAY_US(1000);

	ADC_setInterruptSOCTrigger(base, soc_num, ADC_INT_SOC_TRIGGER_NONE);

	ADC_setInterruptSource(base, ADC_INT_NUMBER1, soc_num);

	ADC_clearInterruptStatus(base, ADC_INT_NUMBER1);

	ADC_disableContinuousMode(base, ADC_INT_NUMBER1);

	ADC_enableInterrupt(base, ADC_INT_NUMBER1);

	/* Force convert */
	ADC_forceSOC(base, soc_num);

	/* Wait for ADC to complete, then acknowledge flag. */
	while(ADC_getInterruptStatus(base, ADC_INT_NUMBER1) == false);

	ADC_clearInterruptStatus(base, ADC_INT_NUMBER1);

	/* Store results */
	ADCResult = ADC_readResult(result_base, soc_num);

	volt = reference_voltage * (float)ADCResult / 4095.0f;

	if (expectation_val >= volt)
		cmp_val = expectation_val - volt;
	else
		cmp_val = volt - expectation_val;

	if (cmp_val > 0.05)
		return STL_ERR_CHECK_FAILED;

	return STL_SUCCESS;
}

stl_err_t stl_analog_io_output(uint32_t base, DAC_ReferenceVoltage ref_voltage_mode,
							DAC_LoadMode loadmode, DAC_GainMode gainmode,
							float expectation_val, float reference_voltage,
							ADC_SOCNumber adc_soc_num, ADC_ReferenceMode adc_ref_voltage_mode,
							float adc_reference_voltage)
{
	uint16_t dac_val;
	uint32_t adc_base;
	uint32_t adc_result_base;
	ADC_Channel adc_chl;

	/* Set DAC reference voltage. */
	DAC_setReferenceVoltage(base, ref_voltage_mode);

	/* Set DAC load mode. */
	DAC_setLoadMode(base, loadmode);

	DAC_setGainMode(base, gainmode);

	/* Enable the DAC output */
	DAC_enableOutput(base);

	DAC_enable(base);

	/* Set the DAC shadow output */
	dac_val = (expectation_val / reference_voltage) * 4095.0f;
	DAC_setShadowValue(base, dac_val);

	switch (base) {
		case DACA_BASE :
			adc_base = ADCA_BASE;
			adc_result_base = ADCARESULT_BASE;
			adc_chl = ADC_CH_ADCIN13;
			HWREG(0x400CD000+0x564) = 1;
			break;

		case DACB_BASE :
			adc_base = ADCC_BASE;
			adc_result_base = ADCCRESULT_BASE;
			adc_chl = ADC_CH_ADCIN13;
			HWREG(0x400CD000+0x56C) = 1;
			break;

		default :
			return STL_ERR_INVALID_BASE;
	}

	if (stl_analog_io_input(adc_base, adc_result_base, adc_soc_num,
					adc_chl, ADC_CLK_DIV_4_0, adc_ref_voltage_mode,
					ADC_REFERENCE_3_3V, expectation_val, adc_reference_voltage)) {

		HWREG(0x400CD000+0x564) = 0;
		HWREG(0x400CD000+0x56C) = 0;

		return STL_ERR_CHECK_FAILED;
	}

	HWREG(0x400CD000+0x564) = 0;
	HWREG(0x400CD000+0x56C) = 0;

	return STL_SUCCESS;
}
