/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 * commit history
 * 20240313, LYF, verify for HAL_CAN Send/Receive
 * 20240322, LYF, translate source files to C source, ASCII text.
 * 20240324, Jason, change CAN tx API to be non-blocking; add ConfigRxMsg API.
 * 20240417, ZhaoLei, replace sysctl_xxx.h with sysctl.h; 
 * 		      replace SysCtl_setCan0ClkEn() with SysCtl_enablePeripheral()
 * 20240716, Danne, rewrite functions according to new macros.
 * 20241006, Jason, re-create most API functions and some structure defines.
 * 20241025, Jihong, recode the CAN-CTRL driver.
 * 20241202, Jihong, Change register access to 32-bit.
 * 20241227, Jihong, Modify BUS OFF register write conflict.
 */

#ifdef __cplusplus
extern "C"{
#endif

#include "can.h"

/**
 * @brief Enable/Disable the CAN-CTRL auto retransmission.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param enable true is enable auto retransmission, false is disable auto retransmission.
 */
GS32_DRIVER_CAN_FUNC_T bool
CAN_setAutoRetransmission(uint32_t base, bool enable)
{
	uint32_t reg_val;

	if (!CAN_isBaseValid(base))
		return false;

	reg_val = HWREG(base + CAN_O_CFG_STAT);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	if (enable)
		reg_val &= ~(CAN_CFG_STAT_TPSS_M | CAN_CFG_STAT_TSSS_M);
	else
		reg_val |= (CAN_CFG_STAT_TPSS_M | CAN_CFG_STAT_TSSS_M);

	HWREG(base + CAN_O_CFG_STAT) = reg_val;

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_setOpMode(uint32_t base, CAN_OperationMode_t mode)
{
	uint32_t reg_val;

	if (!CAN_isBaseValid(base))
		return false;

	reg_val = HWREG(base + CAN_O_CFG_STAT);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	/* Reset CAN-CTRL IP instance. */
	if(mode == CAN_OPERATION_MODE_SW_INIT)
		reg_val |= CAN_CFG_STAT_RESET_M;
	else if(mode == CAN_OPERATION_MODE_NORMAL)
		reg_val &= ~CAN_CFG_STAT_RESET_M;

	HWREG(base + CAN_O_CFG_STAT) = reg_val;

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_initModule(uint32_t base)
{
	uint32_t reg_val;

	if (!CAN_isBaseValid(base))
		return false;

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	reg_val |= (CAN_CFG_STAT_RESET_M | CAN_CFG_STAT_BUSOFF_M);

	HWREG(base + CAN_O_CFG_STAT) = reg_val;

	CAN_disableInterrupt(base, CAN_RTIE_ALL_M);

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_startModule(uint32_t base)
{
	uint32_t reg_val;

	if (!CAN_isBaseValid(base))
		return false;

	reg_val = HWREG(base + CAN_O_CFG_STAT);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	reg_val &= ~CAN_CFG_STAT_RESET_M;

	HWREG(base + CAN_O_CFG_STAT) = reg_val;

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_SetCanFdIsoMode(uint32_t base, CAN_FDISOMode_t mode)
{
	uint32_t  reg_val;

	if (!CAN_isBaseValid(base))
		return false;

	reg_val = HWREG(base + CAN_O_TCTRL);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	if (mode == CAN_FD_ISO_11898_1)
		reg_val |= CAN_TCTRL_FD_ISO_M;
	else
		reg_val &= ~CAN_TCTRL_FD_ISO_M;

	HWREG(base + CAN_O_TCTRL) = reg_val;

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_initConfig(uint32_t base, const CAN_InitParams_t *initParams)
{
	if (!CAN_isBaseValid(base))
		return false;

	if (initParams == NULL)
		return false;

	/* Is enable CANFD mode. */
	if (initParams->fdMode) {
		/* Enable CANFD mode. */
		CAN_setFDMode(base, true);

		/* Set CANFD frame mode. */
		CAN_SetCanFdIsoMode(base, initParams->fdFrame);
	} else
		CAN_setFDMode(base, false);

	/* Is enable listen only mode. */
	CAN_setListenOnlyMode(base, initParams->listenOnlyEnable);

	/* Enable/Disable transmit buffer secondary operation mode. */
	CAN_setTransmitBufferMode(base, initParams->txbmode);

	/* Configuration timestamp mode. */
	CAN_setTimeStampMode(base, initParams->timestamp);

	/* Disable all interrupt. */
	CAN_disableInterrupt(base, CAN_ALL_INT_EN_MASK);

	return true;
}

/**
 * @brief Configuration the CAN-CTRL bit rate.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param configParams Bit rate configuration parameters(@CAN_BitTimingParams_t).
 * @return true Configuration the bit rate is successfully.
 * @return false Configuration the bit rate is failed.
 */
static inline bool CAN_setBitTime(uint32_t base,
								const CAN_BitTimingParams_t *configParams)
{
	if (CAN_isFDOpEnable(base)) {
		if ((configParams->nomRatePrescalar > CAN_NBTP_NBRP_MAX) ||
			(configParams->nomSynchJumpWidth > CAN_NBTP_NSJW_MAX) ||
			(configParams->nomTimeSeg1 > CAN_NBTP_NTSEG1_MAX) ||
			(configParams->nomTimeSeg2 > CAN_NBTP_NTSEG2_MAX_CANFD_MODE) ||

			(configParams->dataRatePrescalar > CAN_DBTP_DBRP_MAX) ||
			(configParams->dataSynchJumpWidth > CAN_DBTP_DSJW_MAX) ||
			(configParams->dataTimeSeg1 > CAN_DBTP_DTSEG1_MAX) ||
			(configParams->dataTimeSeg2 > CAN_DBTP_DTSEG2_MAX))

			return false;
	} else {
		if ((configParams->nomRatePrescalar > CAN_NBTP_NBRP_MAX) ||
			(configParams->nomSynchJumpWidth > CAN_NBTP_NSJW_MAX) ||
			(configParams->nomTimeSeg1 > CAN_NBTP_NTSEG1_MAX) ||
			(configParams->nomTimeSeg2 > CAN_NBTP_NTSEG2_MAX_CAN_MODE))
			return false;
	}

	/* Set Clock source prescalar in nominal phase. */
	HWREG(base + CAN_O_S_PRESC) &= ~CAN_S_PRESC_S_PRESC_M;
	HWREG(base + CAN_O_S_PRESC) |= (((uint32_t)configParams->nomRatePrescalar <<
									CAN_S_PRESC_S_PRESC_S) & CAN_S_PRESC_S_PRESC_M);

	/* Set Time Segment 1 in nominal phase. */
	HWREG(base + CAN_O_S_SEG_1) &= ~CAN_S_SEG_1_S_SEG_1_M;
	HWREG(base + CAN_O_S_SEG_1) |= (((uint32_t)configParams->nomTimeSeg1 <<
									CAN_S_SEG_1_S_SEG_1_S) & CAN_S_SEG_1_S_SEG_1_M);

	/* Set Time Segment 2 in nominal phase. */
	HWREG(base + CAN_O_S_SEG_2) &= ~CAN_S_SEG_2_S_SEG_2_M;
	HWREG(base + CAN_O_S_SEG_2) |= (((uint32_t)configParams->nomTimeSeg2 <<
									CAN_S_SEG_2_S_SEG_2_S) & CAN_S_SEG_2_S_SEG_2_M);

	/* Set Synch Jump Width in nominal phase. */
	HWREG(base + CAN_O_S_SJW) &= ~CAN_S_SJW_S_SJW_M;
	HWREG(base + CAN_O_S_SJW) |= (((uint32_t)configParams->nomSynchJumpWidth <<
									CAN_S_SJW_S_SJW_S) & CAN_S_SJW_S_SJW_M);

	/* 2. Configuration Data pahse Bit Time. */
	if (!CAN_isFDOpEnable(base))
		return true;

	/* Set Clock source prescalar in data phase. */
	HWREG(base + CAN_O_F_PRESC) &= ~CAN_F_PRESC_F_PRESC_M;
	HWREG(base + CAN_O_F_PRESC) |= (((uint32_t)configParams->dataRatePrescalar <<
									CAN_F_PRESC_F_PRESC_S) & CAN_F_PRESC_F_PRESC_M);

	/* Set Time Segment 1 in data phase. */
	HWREG(base + CAN_O_F_SEG_1) &= ~CAN_F_SEG_1_F_SEG_1_M;
	HWREG(base + CAN_O_F_SEG_1) |= (((uint32_t)configParams->dataTimeSeg1 <<
									CAN_F_SEG_1_F_SEG_1_S) & CAN_F_SEG_1_F_SEG_1_M);

	/* Set Time Segment 2 in data phase. */
	HWREG(base + CAN_O_F_SEG_2) &= ~CAN_F_SEG_2_F_SEG_2_M;
	HWREG(base + CAN_O_F_SEG_2) |= (((uint32_t)configParams->dataTimeSeg2 <<
									CAN_F_SEG_2_F_SEG_2_S) & CAN_F_SEG_2_F_SEG_2_M);

	/* Set Synch Jump Width in data phase. */
	HWREG(base + CAN_O_F_SJW) &= ~CAN_F_SJW_F_SJW_M;
	HWREG(base + CAN_O_F_SJW) |= (((uint32_t)configParams->dataSynchJumpWidth <<
									CAN_F_SJW_F_SJW_S) & CAN_F_SJW_F_SJW_M);

	return true;
}

/**
 * @brief Get the CAN-CTRL bit rate parameters.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param configParams The bit rate configuration parameters.
 */
static inline void CAN_getBitTime(uint32_t base, CAN_BitTimingParams_t *configParams)
{
	configParams->nomRatePrescalar = (HWREG(base + CAN_O_S_PRESC) &
								CAN_S_PRESC_S_PRESC_M) >> CAN_S_PRESC_S_PRESC_S;
	configParams->nomTimeSeg1 = (HWREG(base + CAN_O_S_SEG_1) &
								CAN_S_SEG_1_S_SEG_1_M) >> CAN_S_SEG_1_S_SEG_1_S;
	configParams->nomTimeSeg2 = (HWREG(base + CAN_O_S_SEG_2) &
								CAN_S_SEG_2_S_SEG_2_M) >> CAN_S_SEG_2_S_SEG_2_S;
	configParams->nomSynchJumpWidth = (HWREG(base + CAN_O_S_SJW) &
								CAN_S_SJW_S_SJW_M) >> CAN_S_SJW_S_SJW_S;

	configParams->dataRatePrescalar = (HWREG(base + CAN_O_F_PRESC) &
								CAN_F_PRESC_F_PRESC_M) >> CAN_F_PRESC_F_PRESC_S;
	configParams->dataTimeSeg1 = (HWREG(base + CAN_O_F_SEG_1) &
								CAN_F_SEG_1_F_SEG_1_M) >> CAN_F_SEG_1_F_SEG_1_S;
	configParams->dataTimeSeg2 = (HWREG(base + CAN_O_F_SEG_2) &
								CAN_F_SEG_2_F_SEG_2_M) >> CAN_F_SEG_2_F_SEG_2_S;
	configParams->dataSynchJumpWidth = (HWREG(base + CAN_O_F_SJW) &
								CAN_F_SJW_F_SJW_M) >> CAN_F_SJW_F_SJW_S;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_setBitTiming(uint32_t base, uint16_t prescaler,
					uint16_t tSeg1, uint16_t tSeg2, uint16_t sjw)
{
	if (!CAN_isBaseValid(base))
		return false;

	if (CAN_isFDOpEnable(base)) {
		if ((prescaler > CAN_NBTP_NBRP_MAX) ||
			(sjw > CAN_NBTP_NSJW_MAX) ||
			(tSeg1 > CAN_NBTP_NTSEG1_MAX) ||
			(tSeg2 > CAN_NBTP_NTSEG2_MAX_CANFD_MODE))
			return false;
	} else {
		if ((prescaler > CAN_NBTP_NBRP_MAX) ||
			(sjw > CAN_NBTP_NSJW_MAX) ||
			(tSeg1 > CAN_NBTP_NTSEG1_MAX) ||
			(tSeg2 > CAN_NBTP_NTSEG2_MAX_CAN_MODE))
			return false;
	}

	/* Set Clock source prescalar in nominal phase. */
	HWREG(base + CAN_O_S_PRESC) &= ~CAN_S_PRESC_S_PRESC_M;
	HWREG(base + CAN_O_S_PRESC) |= (((uint32_t)prescaler <<
									CAN_S_PRESC_S_PRESC_S) & CAN_S_PRESC_S_PRESC_M);

	/* Set Time Segment 1 in nominal phase. */
	HWREG(base + CAN_O_S_SEG_1) &= ~CAN_S_SEG_1_S_SEG_1_M;
	HWREG(base + CAN_O_S_SEG_1) |= (((uint32_t)tSeg1 <<
									CAN_S_SEG_1_S_SEG_1_S) & CAN_S_SEG_1_S_SEG_1_M);

	/* Set Time Segment 2 in nominal phase. */
	HWREG(base + CAN_O_S_SEG_2) &= ~CAN_S_SEG_2_S_SEG_2_M;
	HWREG(base + CAN_O_S_SEG_2) |= (((uint32_t)tSeg2 <<
									CAN_S_SEG_2_S_SEG_2_S) & CAN_S_SEG_2_S_SEG_2_M);

	/* Set Synch Jump Width in nominal phase. */
	HWREG(base + CAN_O_S_SJW) &= ~CAN_S_SJW_S_SJW_M;
	HWREG(base + CAN_O_S_SJW) |= (((uint32_t)sjw <<
									CAN_S_SJW_S_SJW_S) & CAN_S_SJW_S_SJW_M);

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_setDataBitTiming(uint32_t base, uint16_t prescaler,
						uint16_t tSeg1, uint16_t tSeg2, uint16_t sjw)
{
	if (!CAN_isBaseValid(base))
		return false;

	if ((prescaler > CAN_DBTP_DBRP_MAX) ||
		(sjw > CAN_DBTP_DSJW_MAX) ||
		(tSeg1 > CAN_DBTP_DTSEG1_MAX) ||
		(tSeg2 > CAN_DBTP_DTSEG2_MAX))
		return false;

	/* Set Clock source prescalar in data phase. */
	HWREG(base + CAN_O_F_PRESC) &= ~CAN_F_PRESC_F_PRESC_M;
	HWREG(base + CAN_O_F_PRESC) |= (((uint32_t)prescaler <<
									CAN_F_PRESC_F_PRESC_S) & CAN_F_PRESC_F_PRESC_M);

	/* Set Time Segment 1 in data phase. */
	HWREG(base + CAN_O_F_SEG_1) &= ~CAN_F_SEG_1_F_SEG_1_M;
	HWREG(base + CAN_O_F_SEG_1) |= (((uint32_t)tSeg1 <<
									CAN_F_SEG_1_F_SEG_1_S) & CAN_F_SEG_1_F_SEG_1_M);

	/* Set Time Segment 2 in data phase. */
	HWREG(base + CAN_O_F_SEG_2) &= ~CAN_F_SEG_2_F_SEG_2_M;
	HWREG(base + CAN_O_F_SEG_2) |= (((uint32_t)tSeg2 <<
									CAN_F_SEG_2_F_SEG_2_S) & CAN_F_SEG_2_F_SEG_2_M);

	/* Set Synch Jump Width in data phase. */
	HWREG(base + CAN_O_F_SJW) &= ~CAN_F_SJW_F_SJW_M;
	HWREG(base + CAN_O_F_SJW) |= (((uint32_t)sjw <<
									CAN_F_SJW_F_SJW_S) & CAN_F_SJW_F_SJW_M);

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_setBitRate(uint32_t base, uint32_t clockFreq, uint32_t bitRate, uint16_t bitTime)
{
	uint32_t brp;
	uint32_t tSeg1;
	uint32_t tSeg2;
	uint32_t sjw;
	uint32_t prescaler;
	float ftseg2 = 0.0f;
	float sampRate = 0.0f;

	if (!CAN_isBaseValid(base))
		return false;

	if (bitRate > 1000000U || bitRate == 0U)
		return false;

	/* Set the sampling rate based on the requested bit rate. */
	if (bitRate > 800000U)
		sampRate = 0.750f;
	else if (bitRate > 500000U)
		sampRate = 0.800f;
	else
		sampRate = 0.875f;

	/* Calculate bit timing values. */
	brp = clockFreq / (bitRate * bitTime);
	tSeg2 = (uint32_t)(bitTime * (1.0f - sampRate) + 0.5f);
	tSeg1 = bitTime - tSeg2;

	if(tSeg2 > 4U)
		sjw = 3U;
	else
		sjw = tSeg2;

	/* Adjust the timing values to fit the CAN register. */
	prescaler = brp - 1U;
	tSeg1 -= 2U;
	tSeg2 -= 1U;
	sjw -= 1U;

	/* Set the calculated timing parameters. */
	return CAN_setBitTiming(base, prescaler, tSeg1, tSeg2, sjw);
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_setDataBitRate(uint32_t base, uint32_t clockFreq,
						uint32_t bitRate, uint16_t bitTime)
{
	uint32_t brp;
	uint32_t tSeg1;
	uint32_t tSeg2;
	uint32_t sjw;
	uint32_t prescaler;
	float ftseg2 = 0.0f;
	float sampRate = 0.0f;

	if (!CAN_isBaseValid(base))
		return false;

	if (bitRate > 8000000U || bitRate == 0U)
		return false;

	/* Set the sampling rate based on the requested bit rate. */

	if (bitRate > 800000U)
		sampRate = 0.750f;
	else if (bitRate > 500000U)
		sampRate = 0.800f;
	else
		sampRate = 0.875f;

	/* Calculate bit timing values. */
	brp = clockFreq / (bitRate * bitTime);
	tSeg2 = (uint32_t)(bitTime * (1.0f - sampRate) + 0.5f);
	tSeg1 = bitTime - tSeg2;

	if(tSeg2 > 4U)
		sjw = 3U;
	else
		sjw = tSeg2;

	/* Adjust the timing values to fit the CAN register. */
	prescaler = brp - 1U;
	tSeg1 -= 2U;
	tSeg2 -= 1U;
	sjw -= 1U;

	/* Set the calculated timing parameters. */
	return CAN_setDataBitTiming(base, prescaler, tSeg1, tSeg2, sjw);
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_setBitRateSamplePoint(uint32_t base, uint32_t clockFreq,
							uint32_t bitrate, float32 sample_point)
{
	float32_t tmp_val;
	uint32_t bit_time;
	uint32_t bit_time_seg2;
	CAN_BitTimingParams_t configParams;
	
	if (!CAN_isBaseValid(base))
		return false;

	/* CAN2.0 maximum of bitrate is 1Mbps. */
	if (bitrate > 1000000U || sample_point <= 0.0 ||
		clockFreq == 0U || bitrate == 0 || sample_point >= 1.0)
		return false;

	CAN_getBitTime(base, &configParams);

	/* tmp_val = bit time * prescaler. */
	if (CAN_isFDOpEnable(base)) {
		bit_time = CAN_NBTP_NTSEG1_MAX + CAN_NBTP_NTSEG2_MAX_CANFD_MODE + 3;
		bit_time_seg2 = CAN_NBTP_NTSEG2_MAX_CANFD_MODE + 1;
	} else {
		bit_time = CAN_NBTP_NTSEG1_MAX + CAN_NBTP_NTSEG2_MAX_CAN_MODE + 3;
		bit_time_seg2 = CAN_NBTP_NTSEG2_MAX_CAN_MODE + 1;
	}

	tmp_val = clockFreq / bitrate;
	configParams.nomRatePrescalar = 1;

	while (1) {
		if ((((tmp_val / configParams.nomRatePrescalar) * sample_point) <=
			CAN_NBTP_NTSEG1_MAX + 2U) &&
			((tmp_val / configParams.nomRatePrescalar) - ((tmp_val / configParams.nomRatePrescalar) * sample_point) <= bit_time_seg2)
		) {

			if (tmp_val / configParams.nomRatePrescalar == (uint32_t)(tmp_val / configParams.nomRatePrescalar))
				break;
			if (tmp_val / configParams.nomRatePrescalar < 1)
				return false;
		}

		configParams.nomRatePrescalar += 1;
	}

	tmp_val = tmp_val / (configParams.nomRatePrescalar);
	configParams.nomRatePrescalar -= 1;

	configParams.nomTimeSeg1 = (uint8_t)(sample_point * tmp_val);
	configParams.nomTimeSeg2 = (uint8_t)(tmp_val - configParams.nomTimeSeg1);

	if (configParams.nomTimeSeg1 < configParams.nomTimeSeg2)
		return false;

	configParams.nomTimeSeg1 -= 2U;
	configParams.nomTimeSeg2 -= 1U;

	if (configParams.nomTimeSeg2 > CAN_NBTP_NSJW_MAX)
		configParams.nomSynchJumpWidth = CAN_NBTP_NSJW_MAX;
	else
		configParams.nomSynchJumpWidth = configParams.nomTimeSeg2;

	if (configParams.nomSynchJumpWidth > 2)
		configParams.nomSynchJumpWidth = 2;

	return CAN_setBitTime(base, &configParams);
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_setDataBitRateSamplePoint(uint32_t base, uint32_t clockFreq,
								uint32_t bitrate, float32 sample_point)
{
	float32_t tmp_val;
	uint32_t bit_time;
	CAN_BitTimingParams_t configParams;
	
	if (!CAN_isBaseValid(base))
		return false;

	/* CANFD maximum of bitrate is 4Mbps. */
	if (bitrate > 4000000U || sample_point <= 0.0 ||
		clockFreq == 0U || bitrate == 0 || sample_point >= 1.0)
		return false;

	CAN_getBitTime(base, &configParams);

	/* tmp_val = bit time * prescaler. */
	bit_time = CAN_DBTP_DTSEG1_MAX + CAN_DBTP_DTSEG2_MAX + 3;

	tmp_val = clockFreq / bitrate;
	configParams.dataRatePrescalar = 1;

	while (1) {
		if ((((tmp_val / configParams.dataRatePrescalar) * sample_point) <=
			CAN_DBTP_DTSEG1_MAX + 2U) &&
			((tmp_val / configParams.dataRatePrescalar) - ((tmp_val / configParams.dataRatePrescalar) * sample_point) <= CAN_DBTP_DTSEG2_MAX + 1)
		) {
			if (tmp_val / configParams.dataRatePrescalar == (uint32_t)(tmp_val / configParams.dataRatePrescalar))
				break;
			if (tmp_val / configParams.dataRatePrescalar < 1)
				return false;
		}

		configParams.dataRatePrescalar += 1;
	}

	tmp_val = tmp_val / (configParams.dataRatePrescalar);
	configParams.dataRatePrescalar -= 1;
	configParams.dataTimeSeg1 = (uint8_t)(sample_point * tmp_val);
	configParams.dataTimeSeg2 = (uint8_t)(tmp_val - configParams.dataTimeSeg1);

	if (configParams.dataTimeSeg1 < configParams.dataTimeSeg2)
		return false;

	configParams.dataTimeSeg1 -= 2U;
	configParams.dataTimeSeg2 -= 1U;

	if (configParams.dataTimeSeg2 > CAN_DBTP_DSJW_MAX)
		configParams.dataSynchJumpWidth = CAN_DBTP_DSJW_MAX;
	else
		configParams.dataSynchJumpWidth = configParams.dataTimeSeg2;

	if (configParams.dataSynchJumpWidth > 2)
		configParams.dataSynchJumpWidth = 2;

	tmp_val = (configParams.dataTimeSeg1 + 2U);

	/* Enable TDC mode and set TDC value. */
	CAN_setTDCValue(base, tmp_val);
	CAN_enableTDC(base, false);

	return CAN_setBitTime(base, &configParams);
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_configMsgFilter(uint32_t base, uint32_t filtNum, const CAN_FilterElement_t *elem)
{
	uint32_t reg_val;

	if (!CAN_isBaseValid(base))
		return false;

	if (elem == NULL || filtNum >= NUMBER_OF_ACF)
		return false;

	reg_val = HWREG(base + CAN_O_ACFCTRL);

	/* Select ID filter. */
	reg_val &= ~CAN_ACFCTRL_ACFADR_M;
	reg_val |= ((filtNum & (CAN_ACFCTRL_ACFADR_M >> CAN_ACFCTRL_ACFADR_S)) << CAN_ACFCTRL_ACFADR_S);

	/* Choose to set CAN ID filter. */
	reg_val &= ~CAN_ACFCTRL_SELMASK_M;

	HWREG(base + CAN_O_ACFCTRL) = reg_val;

	/* Set the CAN ID filter. */
	HWREG(base + CAN_O_ACF_0) = (elem->acode & CAN_ACF_ACODE_X_AMASK_X_M);

	/* Choose to set CAN ID mask filter. */
	HWREG(base + CAN_O_ACFCTRL) |= CAN_ACFCTRL_SELMASK_M;

	/* Set the CAN ID mask filter. */
	reg_val = (elem->amsk & CAN_ACF_ACODE_X_AMASK_X_M);

	/* Set the CAN ID filter frame type. */
	/* By default, all standard frames and extended frames are accepted. */
	reg_val &= ~(CAN_ACF_3_AIDEE_M | CAN_ACF_3_AIDE_M);

	/* Only accept standard frames. */
	if (elem->filerFrameType == CAN_IDTYPE_ONLY_STD)
		reg_val |= CAN_ACF_3_AIDEE_M;
	/* Only accept extended frames. */
	else if (elem->filerFrameType == CAN_IDTYPE_ONLY_EXT) {
		reg_val |= CAN_ACF_3_AIDEE_M;
		reg_val |= CAN_ACF_3_AIDE_M;
	}

	HWREG(base + CAN_O_ACF_0) = reg_val;

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_addMsgFilter(uint32_t base, uint32_t acode,
					uint32_t mask, CAN_IDFilterType_t type)
{
	uint32_t reg_val;
	uint8_t i;
	uint8_t filter_number = 0xff;

	CAN_FilterElement_t elem;

	if (!CAN_isBaseValid(base))
		return false;

	reg_val = HWREG(base + CAN_O_ACF_EN_0) & 0xFFFF0000U;

	reg_val >>= CAN_ACF_EN_0_AE_0_S;

	for (i = 0; i < NUMBER_OF_ACF; i++) {
		if (!((reg_val >> i) & 1U)) {
			filter_number = i;
			break;
		}
	}

	/* Check if all filter are used. */
	if (filter_number == 0xff)
		return false;

	elem.acode = acode;
	elem.amsk = mask;
	elem.filerFrameType = type;

	CAN_configMsgFilter(base, filter_number, &elem);

	return CAN_configMsgFilterEnable(base, filter_number);
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_lpbkModeEnable(uint32_t base, CAN_LpbkMode_t lpbkMode, bool enable)
{
	uint32_t reg_val;

	if (!CAN_isBaseValid(base))
		return false;

	/* Disable loopback mode only in reset mode. */
	if (!enable) {
		if (CAN_isInReset(base))
			return false;
	}

	if (lpbkMode == CAN_LPBK_MODE_EXTERNAL && enable)
		/* Enable Self-ACKnowledge. */
		setSelfACKnowledge(base, true);
	else
		setSelfACKnowledge(base, false);

	reg_val = HWREG(base + CAN_O_CFG_STAT);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	if (lpbkMode == CAN_LPBK_MODE_INTERNAL) {
		if(enable) {
			reg_val &= ~CAN_CFG_STAT_LBME_M;
			reg_val |= CAN_CFG_STAT_LBMI_M;
		} else
			reg_val &= ~CAN_CFG_STAT_LBMI_M;
	} else if (lpbkMode == CAN_LPBK_MODE_EXTERNAL) {
		if(enable) {
			reg_val &= ~CAN_CFG_STAT_LBMI_M;
			reg_val |= CAN_CFG_STAT_LBME_M;
		} else
			reg_val &= ~CAN_CFG_STAT_LBME_M;
	} else
		return false;

	HWREG(base + CAN_O_CFG_STAT) = reg_val;

	return true;
}

GS32_DRIVER_CAN_FUNC_T int
CAN_getErrStatus(uint32_t base)
{
	uint32_t reg_val;
	uint32_t status = CAN_STATUS_OK;

	if (!CAN_isBaseValid(base))
		return -1;

	/* Get bus error status. */
	reg_val = HWREG(base + CAN_O_ERRINT);
	
	if (reg_val & CAN_ERRINT_EWARN_M)
		status |= CAN_STATUS_EWARN;

	if (reg_val & CAN_ERRINT_EPASS_M)
		status |= CAN_STATUS_EPASS;

	if (HWREG(base + CAN_O_CFG_STAT) & CAN_CFG_STAT_BUSOFF_M)
		status |= CAN_STATUS_BUS_OFF;

	return status;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_clearBusOff(uint32_t base)
{
	uint32_t reg_val;

	if (!CAN_isBaseValid(base))
		return false;

	reg_val = HWREG(base + CAN_O_TCMD);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	if (CAN_getErrStatus(base) & CAN_STATUS_BUS_OFF)
		reg_val |= CAN_CFG_STAT_BUSOFF_M;

	HWREG(base + CAN_O_TCMD) = reg_val;

	return true;
}

/**
 * @brief Choose the transmission buffer.
 * Selects the transmit buffer to be loaded with a message. Use the TBUF registers for
 * access. TBSEL needs to be stable all the time the TBUF registers are written and when
 * TSNEXT is set.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param mode The buffer mode(@CAN_TxBufMode_t).
 */
static inline void CAN_selectTxBufMode(uint32_t base, CAN_TxBufMode_t mode)
{
	uint32_t reg_val = HWREG(base + CAN_O_TCMD);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	if (mode == CAN_PTB_PRIORITY)
		reg_val &= ~CAN_TCMD_TBSEL_M;
	else
		reg_val |= CAN_TCMD_TBSEL_M;

	HWREG(base + CAN_O_TCMD) = reg_val;
}

/**
 * @brief Select the next secondary buffer.
 *
 * @param base The base address of CAN-CTRL IP.
 */
GS32_DRIVER_CAN_FUNC_T void
CAN_selectNextSecondaryBuf(uint32_t base)
{
	uint32_t reg_val = HWREG(base + CAN_O_TCTRL);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	reg_val |= CAN_TCTRL_TSNEXT_M;

	HWREG(base + CAN_O_TCTRL) = reg_val;
}

/**
 * @brief Write a CAN message to the secondary buffer.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param pTxFrame The pointer to the CAN message.
 * @return true Write successfully.
 * @return false Write failed.
 */
static inline bool CAN_WriteSTB(uint32_t base, CAN_TxMessage_t *pTxFrame)
{
	/* Obtain the status of the secondary buffer. */
	if (CAN_getTxSecondaryBufState(base) >= CAN_TXBUFFER_FULL)
		return false;

	/* Switch to STB mode. */
	CAN_selectTxBufMode(base, CAN_STB_SECONDARY);

	/* Copy data to the TBUF register. */
	*(CAN_TxMessage_t *)(base + CAN_O_TBUF0) = *pTxFrame;

	/* Update the secondary buffer head pointer. */
	CAN_selectNextSecondaryBuf(base);

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_WriteTxMsgToSTB(uint32_t base, CAN_TxMessage_t *pTxFrame)
{
	if (!CAN_isBaseValid(base))
		return false;

	return CAN_WriteSTB(base, pTxFrame);
}

/**
 * @brief Write a CAN message to the primary buffer.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param pTxFrame The pointer to the CAN message.
 * @return true Write successfully.
 * @return false Write failed.
 */
static inline bool CAN_WritePrimaryBuf(uint32_t base, CAN_TxMessage_t *pTxFrame)
{
	/* Obtain the status of the primary buffer. */
	/* Check if the primary buffer is empty. */
	if (CAN_getTxPrimaryBufState(base) != 0)
		return false;

	/* Switch to PBT mode. */
	CAN_selectTxBufMode(base, CAN_PTB_PRIORITY);

	/* Copy data to the TBUF register. */
	*(CAN_TxMessage_t *)(base + CAN_O_TBUF0) = *pTxFrame;

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_startTransmissionOnce(uint32_t base)
{
	uint32_t reg_val;

	if (!CAN_isBaseValid(base))
		return false;

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	reg_val |= CAN_TCMD_TSONE_M;

	HWREG(base + CAN_O_TCMD) = reg_val;

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_startTransmissionAll(uint32_t base)
{
	uint32_t reg_val;

	if (!CAN_isBaseValid(base))
		return false;

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	reg_val |= CAN_TCMD_TSALL_M;

	HWREG(base + CAN_O_TCMD) = reg_val;

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_startTransmissionHighPrimaryMsg(uint32_t base)
{
	uint32_t reg_val;

	if (!CAN_isBaseValid(base))
		return false;

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	reg_val |= CAN_TCMD_TPE_M;

	HWREG(base + CAN_O_TCMD) = reg_val;

	return true;
}

/**
 * @brief Start transmission message in Tx buffer.
 * Only uesd in secondary buffer mode.
 *
 * @param base The base address of CAN-CTRL IP.
 * @param mode Select the transmission mode(@CAN_SendMsgMode_t).
 */
GS32_DRIVER_CAN_FUNC_T void
CAN_startTransmission(uint32_t base, CAN_SendMsgMode_t mode)
{
	uint32_t reg_val = HWREG(base + CAN_O_TCMD);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	if (mode == CAN_TX_MODE_ALL_MSG)
		reg_val |= CAN_TCMD_TSALL_M;
	else
		reg_val |= CAN_TCMD_TSONE_M;

	HWREG(base + CAN_O_TCMD) = reg_val;
}

/**
 * @brief Start transmission high priority message.
 *
 * @param base The base address of CAN-CTRL IP.
 */
GS32_DRIVER_CAN_FUNC_T void
CAN_startTransmissionPrimaryMsg(uint32_t base)
{
	uint32_t reg_val = HWREG(base + CAN_O_TCMD);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	reg_val |= CAN_TCMD_TPE_M;

	HWREG(base + CAN_O_TCMD) = reg_val;
}

GS32_DRIVER_CAN_FUNC_T int
CAN_getMessageLength(CAN_MsgDataLength_t size)
{
	if (size > CAN_DATA_LENGTH_64)
		return -1;

	return CAN_DataLengthArray[size];
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_transmitMsg(uint32_t base, CAN_TxMessage_t *pTxFrame)
{
	if (!CAN_isBaseValid(base))
		return false;

	if (pTxFrame == NULL)
		return false;

	if (HWREG(base + CAN_O_TCMD) & CAN_TCMD_TSONE_M)
		return false;

	/* Check if the Tx buffer is full. */
	if (!CAN_WriteSTB(base, pTxFrame))
		return false;

	/* Start transmission. */
	CAN_startTransmission(base, CAN_TX_MODE_SINGLE_MSG);

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_transmitHighPriorityMsg(uint32_t base, CAN_TxMessage_t *pTxFrame)
{
	if (!CAN_isBaseValid(base))
		return false;

	if (pTxFrame == NULL)
		return false;

	/* Fill Tx message to primary buffer. */
	if (!CAN_WritePrimaryBuf(base, pTxFrame))
		return false;

	/* Start transmission message. */
	CAN_startTransmissionPrimaryMsg(base);

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_transmitSecondaryAbort(uint32_t base)
{
	uint32_t reg_val;

	if (!CAN_isBaseValid(base))
		return false;

	reg_val = HWREG(base + CAN_O_TCMD);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	reg_val |= CAN_TCMD_TSA_M;

	HWREG(base + CAN_O_TCMD) = reg_val;

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_transmitPrimaryAbort(uint32_t base)
{
	uint32_t reg_val;

	if (!CAN_isBaseValid(base))
		return false;

	reg_val = HWREG(base + CAN_O_TCMD);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	reg_val |= CAN_TCMD_TPA_M;

	HWREG(base + CAN_O_TCMD) = reg_val;

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_releaseRxBuf(uint32_t base)
{
	uint32_t reg_val;

	if (!CAN_isBaseValid(base))
		return false;

	reg_val = HWREG(base + CAN_O_RCTRL);

	reg_val &= ~(CAN_TCMD_TSONE_M | CAN_TCMD_TSALL_M |
				CAN_TCMD_TPE_M | CAN_TCMD_TPA_M |
				CAN_TCMD_TSA_M | CAN_TCTRL_TSNEXT_M |
				CAN_RCTRL_RBALL_M | CAN_RCTRL_RREL_M |
				CAN_CFG_STAT_BUSOFF_M);

	reg_val |= CAN_RCTRL_RREL_M;

	HWREG(base + CAN_O_RCTRL) = reg_val;

	return true;
}

GS32_DRIVER_CAN_FUNC_T bool
CAN_receiveMsg(uint32_t base, CAN_RxMessage_t *pRxFrame)
{
	if (!CAN_isBaseValid(base))
		return false;

	if (pRxFrame == NULL)
		return false;

	/* Check if the Rx buffer is empty. */
	if (CAN_getRxBufState(base) == CAN_RXBUFFER_EMPTY)
		return false;

	/* Copy data from the RBUF register. */
	*pRxFrame = *(CAN_RxMessage_t *)(base + CAN_O_RBUF0);

	CAN_releaseRxBuf(base);

	return true;
}



GS32_DRIVER_CAN_FUNC_T uint32_t
CAN_clearIntrStatus(uint32_t base, uint32_t intFlag)
{
	uint32_t reg_val;

	if (!CAN_isBaseValid(base))
		return 0;

	reg_val = HWREG(base + CAN_O_RTIF);

	reg_val &= ~CAN_INT_STATUS_MASK;

	reg_val |= (intFlag & CAN_INT_STATUS_MASK);

	HWREG(base + CAN_O_RTIF) = reg_val;

	return true;
}

#ifdef __cplusplus
}
#endif
