/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    cputimer.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "cputimer.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
/**
 *
 *  @brief Set CPU timer pre-scaler value.
 * 
 *  @param [in] base is the base address of the timer module.
 *  @param [in] prescaler is the CPU timer pre-scaler value, only support 0/15/255.
 * 
 *  This function sets the pre-scaler value for the CPU timer.
 *
 *  The prescaler parameter can be any one of the following:
 *  - CPUTIMER_CLOCK_PRESCALER_1   - Prescaler value of / 1
 *  - CPUTIMER_CLOCK_PRESCALER_16  - Prescaler value of / 16
 *  - CPUTIMER_CLOCK_PRESCALER_256 - Prescaler value of / 256
 * 
 *  @return None.
 *
 */
GS32_DRIVER_INIT_FUNC_T void
CPUTimer_setPreScaler(uint32_t base, CPUTimer_Prescaler prescaler)
{
    uint32_t timerCtrl = HWREG(base + CPUTIMER_O_TIMERXCONTROL);

    ASSERT(CPUTimer_isBaseValid(base));

    timerCtrl &= (~CPUTIMER_TIMERXCONTROL_TIMERPRE_MASK);
    prescaler &= 3;
    timerCtrl |= (prescaler << CPUTIMER_TIMERXCONTROL_TIMERPRE_SHIFT);

    //
    // Writes to TPR.TDDR and TPRH.TDDRH bits
    //
    HWREG(base + CPUTIMER_O_TIMERXCONTROL) = timerCtrl;
}

#else
/**
 *
 *  @brief Set CPU timer pre-scaler value.
 *
 *  @param [in] base is the base address of the timer module.
 *  @param [in] prescaler is the CPU timer pre-scaler value, support 1 ~ 65536.
 *
 *  This function sets the pre-scaler value for the CPU timer.
 *
 *  @return None.
 *
 */
GS32_DRIVER_INIT_FUNC_T void
CPUTimer_setPreScaler(uint32_t base, uint32_t prescaler)
{
	ASSERT(prescaler>=1 && prescaler<=65536)

	uint32_t reg = HWREG(base + CPUTIMER_O_TIMERXCONTROL);

    reg &= ~(0xFFFFU << 16);

    reg |= ((uint32_t)(prescaler - 1) & 0xFFFFU) << 16;

    HWREG(base + CPUTIMER_O_TIMERXCONTROL) = reg;
}

#endif

/**
 *
 *  @brief Set the mode of the CPU Timer
 *
 *  @param [in] base is the base address of the timer module.
 *  @param [in] Timer mode
 *
 *  This function is used to set the CPU Timer mode.
 *
 *  The selectable modes of the CPU timer:
 *  - CPUTIMER_TIMERMODE_FREERUNNING   - The counter wraps after reaching its zero value,
 *    and continues to count down from the maximum value. This is the default mode.
 *  - CPUTIMER_TIMERMODE_PERIODIC  - The counter generates an interrupt at a constant interval,
 *    reloading the original value after wrapping past zero.
 *  - CPUTIMER_TIMERMODE_ONESHOT - The counter generates an interrupt once.
 *    When the counter reaches 0, it halts until you reprogram it.
 *
 *  @return None.
 *
 */
GS32_DRIVER_INIT_FUNC_T void
CPUTimer_setTimerMode(uint32_t base, CPUTimer_TimerMode mode)
{
    uint32_t timerCtrl = HWREG(base + CPUTIMER_O_TIMERXCONTROL);

    ASSERT(CPUTimer_isBaseValid(base));
    switch (mode)
    {
        case CPUTIMER_TIMERMODE_FREERUNNING:
            timerCtrl &= (~CPUTIMER_TIMERXCONTROL_TIMERMODE);
            timerCtrl &= (~CPUTIMER_TIMERXCONTROL_ONESHOT);
            break;
        case CPUTIMER_TIMERMODE_PERIODIC:
            timerCtrl |= CPUTIMER_TIMERXCONTROL_TIMERMODE;
            timerCtrl &= (~CPUTIMER_TIMERXCONTROL_ONESHOT);
            break;
        case CPUTIMER_TIMERMODE_ONESHOT:
            timerCtrl |= CPUTIMER_TIMERXCONTROL_ONESHOT;
            break;
        default:
            ASSERT(FALSE);
            break;
    }

    HWREG(base + CPUTIMER_O_TIMERXCONTROL) = timerCtrl;
}

/**
 *
 *  @brief Set CPU timer size.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @param [in] timerSize is 16bit or 32bit timer size.
 * 
 *  This function Set CPU timer size.
 * 
 *  @return None.
 *
 */
GS32_DRIVER_INIT_FUNC_T void
CPUTimer_setTimerSize(uint32_t base, CPUTimer_TimerSize timerSize)
{
    uint32_t timerCtrl = HWREG(base + CPUTIMER_O_TIMERXCONTROL);

    ASSERT(CPUTimer_isBaseValid(base));
    switch (timerSize)
    {
        case CPUTIMER_TIMERSIZE_16BIT:
            timerCtrl &= (~CPUTIMER_TIMERXCONTROL_TIMERSIZE);
            break;
        case CPUTIMER_TIMERSIZE_32BIT:
            timerCtrl |= CPUTIMER_TIMERXCONTROL_TIMERSIZE;
            break;
        default:
            ASSERT(FALSE);
            break;
    }

    HWREG(base + CPUTIMER_O_TIMERXCONTROL) = timerCtrl;
}

/**
 * @brief   Initialize CpuTimer timer at a specified cycle, enable timer interrupt, start timer
 * @param [in]   base   Timer module base address
 * @param [in]   period   Timer cycle, unit is count, 1 timer unit clock defaults to 4 system clocks;
 * @return  None
 */
GS32_DRIVER_INIT_FUNC_T void
CPUTimer_init(uint32_t base, uint32_t period){
	CPUTimer_stopTimer(base);
	 /* Set the timer frequency division factor, fix the timing clock source as the external clock, and set the frequency to 1/4 of the system clock */
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
    CPUTimer_setPreScaler(base, CPUTIMER_CLOCK_PRESCALER_1);
#else
    CPUTimer_setPreScaler(base, 1);
#endif
	
	/* Set the timer width to 32-bit or 16 bit */
	 if(period > 0xFFFF){
		 CPUTimer_setTimerSize(base, CPUTIMER_TIMERSIZE_32BIT);
	 } else {
		 CPUTimer_setTimerSize(base, CPUTIMER_TIMERSIZE_16BIT);
	 }
	 /* Set the timer mode to cycle timing */
	 CPUTimer_setTimerMode(base, CPUTIMER_TIMERMODE_PERIODIC);
	 /* Set timer cycle */
	 CPUTimer_setPeriod(base, period);
	 /* Enable timer interrupt */
	 CPUTimer_enableInterrupt(base);
	 /* clear interrupt flag */
	 CPUTimer_clearOverflowFlag(base);
	 /* Start timer */
	 CPUTimer_startTimer(base);
	 return;
 }


#ifdef __cplusplus
}
#endif

