/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    cputimer.h
*   @brief   
*
*   commit history
*   2024/03/12, Zhao Lei, disable Timer2/3 for 3.0
*   2025/1/1, hezhiyuan,Modify comments
*/

#ifndef DEVICE_DRIVERLIB_CPUTIMER_H_
#define DEVICE_DRIVERLIB_CPUTIMER_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "gs32_version.h"
#include "inc/hw_cputimer.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "sysctl/sysctl_apb.h"
#include "debug.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */
/**
 * 
 *  Values that can be passed to CPUTimer_setEmulationMode() as the
 *  mode parameter.
 * 
 */
typedef enum
{
  // Denotes that the timer will stop after the next decrement
  CPUTIMER_EMULATIONMODE_STOPAFTERNEXTDECREMENT = 0x0000,
  // Denotes that the timer will run free
  CPUTIMER_EMULATIONMODE_RUNFREE = 0x0800
} CPUTimer_EmulationMode;

#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
/**
 * 
 *  The following are values that can be passed to
 *  CPUTimer_setPreScaler() as the prescaler parameter.
 * 
 */
typedef enum
{
    CPUTIMER_CLOCK_PRESCALER_1   = 0,      //< Prescaler value of / 1
    CPUTIMER_CLOCK_PRESCALER_16  = 1,      //< Prescaler value of / 16
    CPUTIMER_CLOCK_PRESCALER_256 = 2,      //< Prescaler value of / 256
} CPUTimer_Prescaler;
#endif

typedef enum
{
  // The counter wraps after reaching its zero value, and continues to count down from
  // the maximum value. This is the default mode.
  CPUTIMER_TIMERMODE_FREERUNNING = 1,
  // The counter generates an interrupt at a constant interval, reloading the original
  // value after wrapping past zero.
  CPUTIMER_TIMERMODE_PERIODIC = 2,
  // The counter generates an interrupt once. When the counter reaches 0, it halts until
  // you reprogram it.
  CPUTIMER_TIMERMODE_ONESHOT = 3,
} CPUTimer_TimerMode;

typedef enum
{
  CPUTIMER_TIMERSIZE_16BIT = 1,
  CPUTIMER_TIMERSIZE_32BIT = 2,
} CPUTimer_TimerSize;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */



#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
extern void CPUTimer_setPreScaler(uint32_t base, CPUTimer_Prescaler prescaler);
#else
extern void CPUTimer_setPreScaler(uint32_t base, uint32_t prescaler);
#endif

extern void CPUTimer_setTimerMode(uint32_t base,
                                      CPUTimer_TimerMode mode);
extern void CPUTimer_setTimerSize(uint32_t base,
                                      CPUTimer_TimerSize timerSize);

extern void CPUTimer_init(uint32_t base, uint32_t period);

/**
 * 
 *  @brief Checks CPU timer base address.
 * 
 *  @param [in] base specifies the Timer module base address.
 * 
 *  @return Returns true if the base address is valid and false
 *  otherwise.
 * 
 */
static inline boolean CPUTimer_isBaseValid(uint32_t base)
{
    return((base == CPUTIMER0_BASE) || (base == CPUTIMER1_BASE)
#if defined(CPUTIMER2_BASE) && defined (CPUTIMER3_BASE)
            || (base == CPUTIMER2_BASE) || (base == CPUTIMER3_BASE)
#endif

#if defined(CPUTIMER4_BASE) && defined (CPUTIMER5_BASE)
            || (base == CPUTIMER4_BASE) || (base == CPUTIMER5_BASE)
#endif
            );
}

/**
 * 
 *  @brief Clears CPU timer overflow flag.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return None.
 * 
 */
static inline void CPUTimer_clearOverflowFlag(uint32_t base)
{
    ASSERT(CPUTimer_isBaseValid(base));

    //
    // Set TIF bit of TCR register
    //
    HWREG(base + CPUTIMER_O_TIMERXINTCLR) |= CPUTIMER_TIMERXINTCLR;
}

/**
 * 
 *  @brief Disables CPU timer interrupt.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return None.
 * 
 */
static inline void CPUTimer_disableInterrupt(uint32_t base)
{
    ASSERT(CPUTimer_isBaseValid(base));

    //
    // Clear TIE bit of TCR register
    //
    HWREG(base + CPUTIMER_O_TIMERXCONTROL) &= (~CPUTIMER_TIMERXCONTROL_INTREN);
}

/**
 * 
 *  @brief Enables CPU timer interrupt.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return None.
 * 
 */
static inline void CPUTimer_enableInterrupt(uint32_t base)
{
    ASSERT(CPUTimer_isBaseValid(base));

    //
    // Set TIE bit of TCR register
    //
    HWREG(base + CPUTIMER_O_TIMERXCONTROL) |= CPUTIMER_TIMERXCONTROL_INTREN;
}

/**
 * 
 *  @brief Reloads CPU timer counter.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @remarks
 *  -This function reloads the CPU timer counter with the values contained in
 *  -the CPU timer period register.
 * 
 *  @return None.
 * 
 */
static inline void CPUTimer_reloadTimerCounter(uint32_t base)
{
    uint32_t bgloadValue = 0;
    ASSERT(CPUTimer_isBaseValid(base));

    //
    // Set TRB bit of register TCR
    //
    bgloadValue = HWREG(base + CPUTIMER_O_TIMERXBGLOAD);
    HWREG(base + CPUTIMER_O_TIMERXLOAD) = bgloadValue;
}

/**
 * 
 *  @brief Stops CPU timer.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return None.
 * 
 */
static inline void CPUTimer_stopTimer(uint32_t base)
{
    ASSERT(CPUTimer_isBaseValid(base));

    //
    // Set TSS bit of register TCR
    //
    HWREG(base + CPUTIMER_O_TIMERXCONTROL) &= (~CPUTIMER_TIMERXCONTROL_TIMEREN);
}

/**
 * 
 *  @brief Starts(restarts) CPU timer.
 * 
 *  @details This function doesn't reset the timer counter.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return None.
 * 
 */
static inline void CPUTimer_resumeTimer(uint32_t base)
{
    ASSERT(CPUTimer_isBaseValid(base));

    //
    // Clear TSS bit of register TCR
    //
    HWREG(base + CPUTIMER_O_TIMERXCONTROL) |= CPUTIMER_TIMERXCONTROL_TIMEREN;
}

/**
 * 
 *  @brief Starts(restarts) CPU timer.
 * 
 *  @details This function reloads the timer counter.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return None.
 * 
 */
static inline void CPUTimer_startTimer(uint32_t base)
{
    uint32_t bgloadValue = 0;
    ASSERT(CPUTimer_isBaseValid(base));

    //
    // Reload the timer counter
    //
    bgloadValue = HWREG(base + CPUTIMER_O_TIMERXBGLOAD);
    HWREG(base + CPUTIMER_O_TIMERXLOAD) = bgloadValue;

    //
    // Clear TSS bit of register TCR
    //
    HWREG(base + CPUTIMER_O_TIMERXCONTROL) |= CPUTIMER_TIMERXCONTROL_TIMEREN;
}

/**
 * 
 *  @brief Sets CPU timer period.
 * 
 *  @param [in] base is the base address of the timer module.
 *  @param [in] periodCount is the CPU timer period count.
 * 
 *  @return None.
 * 
 */
static inline void CPUTimer_setPeriod(uint32_t base, uint32_t periodCount)
{
    ASSERT(CPUTimer_isBaseValid(base));

    //
    // Load the MSB period Count
    //
    HWREG(base + CPUTIMER_O_TIMERXBGLOAD) = periodCount;
}

/**
 * 
 *  @brief Get CPU timer period.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return CPU timer period count.
 * 
 */
static inline uint32_t CPUTimer_getPeriod(uint32_t base)
{
    ASSERT(CPUTimer_isBaseValid(base));

    //
    // Load the MSB period Count
    //
    return HWREG(base + CPUTIMER_O_TIMERXBGLOAD);
}

/**
 * 
 *  @brief Returns the current CPU timer counter value.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return Returns the current CPU timer count value.
 * 
 */
static inline uint32_t CPUTimer_getTimerCount(uint32_t base)
{
    ASSERT(CPUTimer_isBaseValid(base));

    //
    // Get the TIMH:TIM registers value
    //
    return(HWREG(base + CPUTIMER_O_TIMERXVALUE));
}

/**
 * 
 *  @brief Return the CPU timer overflow status.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return Returns true if the CPU timer has overflowed, false if not.
 * 
 */
static inline boolean CPUTimer_getTimerOverflowStatus(uint32_t base)
{
    ASSERT(CPUTimer_isBaseValid(base));

    //
    // Check if TIF bits of register TCR are set
    //
    return((HWREG(base + CPUTIMER_O_TIMERXMIS) ==
            CPUTIMER_TIMERXMIS_INTR) ? TRUE : FALSE);
}

/**
 * 
 *  @brief Sets Emulation mode for CPU timer.
 * 
 *  @param [in] base is the base address of the timer module.
 *  @param [in] mode is the emulation mode of the timer.
 * 
 *  @remarks
 *  -This function sets the behaviour of CPU timer during emulation. Valid
 *  -values mode are: CPUTIMER_EMULATIONMODE_STOPAFTERNEXTDECREMENT,
 *  -CPUTIMER_EMULATIONMODE_STOPATZERO and CPUTIMER_EMULATIONMODE_RUNFREE.
 * 
 *  @return None.
 * 
 */
static inline void CPUTimer_setEmulationMode(uint32_t base,
                                      CPUTimer_EmulationMode mode)
{
    uint32_t timern = 0;

    ASSERT(CPUTimer_isBaseValid(base));

    switch(base)
    {
        case CPUTIMER0_BASE: timern = 0; break;
        case CPUTIMER1_BASE: timern = 1; break;
#ifdef CPUTIMER2_BASE
        case CPUTIMER2_BASE: timern = 2; break;
#endif
#ifdef CPUTIMER3_BASE
        case CPUTIMER3_BASE: timern = 3; break;
#endif
#ifdef CPUTIMER4_BASE
        case CPUTIMER4_BASE: timern = 4; break;
#endif
#ifdef CPUTIMER5_BASE
        case CPUTIMER5_BASE: timern = 5; break;
#endif
    }

    if (mode == CPUTIMER_EMULATIONMODE_RUNFREE)
    {
    	Sysctl_ApbPara_setTimerDebugStopMask(timern, 0);
    }
	else
	{
		Sysctl_ApbPara_setTimerDebugStopMask(timern, 3);
	}
}

/**
 * 
 *  @brief Disables CPU timer Integration Test.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return None.
 * 
 */
static inline void CPUTimer_disableIntegrationTest(uint32_t base)
{
    ASSERT(CPUTimer_isBaseValid(base));

    base &= (~0x20);

    HWREG(base + CPUTIMER_O_TIMERITCR) &= (~CPUTIMER_TIMERITCR);
}

/**
 * 
 *  @brief Enables CPU timer Integration Test.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return None.
 * 
 */
static inline void CPUTimer_enableIntegrationTest(uint32_t base)
{
    ASSERT(CPUTimer_isBaseValid(base));

    base &= (~0x20);

    HWREG(base + CPUTIMER_O_TIMERITCR) |= CPUTIMER_TIMERITCR;
}

/**
 * 
 *  @brief Disables CPU timer Integration Test interrupt output.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return None.
 * 
 */
static inline void CPUTimer_disableIntegrationTestInterrupt(uint32_t base)
{
    uint32_t regBase = base;

    ASSERT(CPUTimer_isBaseValid(base));

    base &= (~0x20);

    if (regBase == base)
        HWREG(base + CPUTIMER_O_TIMERITOP) &= (~CPUTIMER_TIMERITOP1);
    else
        HWREG(base + CPUTIMER_O_TIMERITOP) &= (~CPUTIMER_TIMERITOP2);
}

/**
 * 
 *  @brief Enables CPU timer Integration Test interrupt output.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return None.
 * 
 */
static inline void CPUTimer_enableIntegrationTestInterrupt(uint32_t base)
{
    uint32_t regBase = base;

    ASSERT(CPUTimer_isBaseValid(base));

    base &= (~0x20);

    if (regBase == base)
        HWREG(base + CPUTIMER_O_TIMERITOP) |= CPUTIMER_TIMERITOP1;
    else
        HWREG(base + CPUTIMER_O_TIMERITOP) |= CPUTIMER_TIMERITOP2;
}

/**
 * 
 *  @brief Return the CPU timer raw Interrupt status.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return Returns true if the CPU timer raw Interrupt occurred, false if not.
 * 
 */
static inline boolean CPUTimer_getRawInterruptStatus(uint32_t base)
{
    ASSERT(CPUTimer_isBaseValid(base));

    //
    // Check if TIF bits of register TCR are set
    //
    return((HWREG(base + CPUTIMER_O_TIMERXRIS) ==
            CPUTIMER_TIMERXRIS_INTR) ? TRUE : FALSE);
}

/**
 * 
 *  @brief Return the CPU timer Interrupt status.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return Returns true if the CPU timer Interrupt occurred, false if not.
 * 
 */
static inline boolean CPUTimer_getInterruptStatus(uint32_t base)
{
    ASSERT(CPUTimer_isBaseValid(base));

    //
    // Check if TIF bits of register TCR are set
    //
    return((HWREG(base + CPUTIMER_O_TIMERXMIS) ==
            CPUTIMER_TIMERXMIS_INTR) ? TRUE : FALSE);
}

/**
 * 
 *  @brief Clears CPU timer interrupt status.
 * 
 *  @param [in] base is the base address of the timer module.
 * 
 *  @return None.
 * 
 */
static inline void CPUTimer_ClearInterruptStatus(uint32_t base)
{
    ASSERT(CPUTimer_isBaseValid(base));

    //
    // Set TIF bit of TCR register
    //
    HWREG(base + CPUTIMER_O_TIMERXINTCLR) |= CPUTIMER_TIMERXINTCLR;
}

#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_CPUTIMER_H_ */
