/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    dma.h
*   @brief 
*
*/
/*
 * commit history
 * 20240313, LYF, verify for chip 3.0
 * 20240319, LYF, Organize DMA and DMAMUX driver version management
 * 20240322, LYF, translate source files to C source, ASCII text.
 * 20240612, LYF, fix cache issue and channel nums for chip 3.0
 * 20240612, LYF, move the HkSelect and HardInf configuration inside the driver by force 
 * 20240617, LYF, add G5 DMA feature for chip 3.0 
 * 20240621, LYF, add block offset and block cnt feature for chip 3.0
 * 20240626, LYF, update change from ip for chip3.0 
 * 20240705, LYF, Delate legacy code and use user defined isr function name. 
 * 20240711, LYF, Add APIs to get/clear block tfr status by channel number. 
 */


#ifndef DEVICE_DRIVERLIB_DMA_H_
#define DEVICE_DRIVERLIB_DMA_H_

#ifdef __cplusplus
extern "C"{
#endif

#include "inc/hw_types.h"
#include "gs32_version.h"

#if IS_GS32F00xx(0x12, 0x30)

#include "inc/hw_memmap.h"
#include "inc/hw_dma.h"
#include "regs/regs_dma.h"
#include "dmamux.h"
#include "debug.h"

#if (DMAC_VERSION == 0x30)

#elif (DMAC_VERSION == 0x12)

#if !(defined(DMA_BASE) || defined(DMA_CH0_BASE))
#if __arm__
#define DMA_BASE                        (DMA0_BASE)
#define DMA_CH0_BASE                    (DMA0_CH0_BASE)
#define DMA_CH1_BASE                    (DMA0_CH1_BASE)
#define DMA_CH2_BASE                    (DMA0_CH2_BASE)
#define DMA_CH3_BASE                    (DMA0_CH3_BASE)
#define DMA_CH4_BASE                    (DMA0_CH4_BASE)
#define DMA_CH5_BASE                    (DMA0_CH5_BASE)
#define DMA_CH6_BASE                    (DMA0_CH6_BASE)
#define DMA_CH7_BASE                    (DMA0_CH7_BASE)
#else
#define DMA_BASE                        (DMA1_BASE)
#define DMA_CH1_BASE                    (DMA1_CH1_BASE)
#define DMA_CH2_BASE                    (DMA1_CH2_BASE)
#define DMA_CH3_BASE                    (DMA1_CH3_BASE)
#define DMA_CH4_BASE                    (DMA1_CH4_BASE)
#define DMA_CH5_BASE                    (DMA1_CH5_BASE)
#define DMA_CH6_BASE                    (DMA1_CH6_BASE)
#define DMA_CH7_BASE                    (DMA1_CH7_BASE)
#define DMA_CH8_BASE                    (DMA1_CH8_BASE)
#endif
#endif /* #if __arm__ */

#endif /*#if (DMAC_VERSION == 0x11)*/

/**
 * @brief Only one burst transfer performed per trigger.
 */
#define DMA_CFG_ONESHOT_DISABLE     0U

/**
 * @brief Burst transfers occur without additional event triggers after the first.
 */
#define DMA_CFG_ONESHOT_ENABLE      0x400U

/**
 * @brief DMA channel will be disabled at the end of a transfer.
 */
#define DMA_CFG_CONTINUOUS_DISABLE  0U

/**
 * @brief DMA reinitializes when the transfer count is zero and waits for a trigger.
 */
#define DMA_CFG_CONTINUOUS_ENABLE   0x800U

/**
 * @brief DMA transfers 8 bits at a time.
 */
#define DMA_CFG_SIZE_8BIT          0U

/**
 * @brief DMA transfers 16 bits at a time.
 */
#define DMA_CFG_SIZE_16BIT         0x4000U

/**
 * @brief DMA transfers 32 bits at a time.
 */
#define DMA_CFG_SIZE_32BIT         0x8000U

/**
 * @brief DMA transfers 64 bits at a time.
 */
#define DMA_CFG_SIZE_64BIT         0x10000U

/**
 * @brief DMA transfers 128 bits at a time.
 */
#define DMA_CFG_SIZE_128BIT        0x20000U

/**
 * @brief DMA transfers 256 bits at a time.
 */
#define DMA_CFG_SIZE_256BIT        0x40000U

#define DMA_INT_TFR         0x01U
#define DMA_INT_BLOCK       0x02U
#define DMA_INT_SRCTRAN     0x04U
#define DMA_INT_DSTTRAN     0x08U
#define DMA_INT_ERR         0x10U
#if (DMAC_VERSION == 0x30)
#define DMA_INT_FDI         0x20U
#endif

#define DMAREGH(base, offset)      (HWREG(base + (offset) + 0x04))
#define DMAREGL(base, offset)      (HWREG(base + (offset)))

/**
 * @param DMA_INT_AT_BEGINNING DMA interrupt is generated at the beginning of a transfer
 * @param DMA_INT_AT_END DMA interrupt is generated at the end of a transfer
 */
typedef enum
{
    DMA_INT_AT_BEGINNING,
    DMA_INT_AT_END
} DMA_InterruptMode;

/**
 * @param DMA_EMULATION_STOP transmission stops after current read-write access is completed Transmission stops after current read-write access is completed
 * @param DMA_EMULATION_FREE_RUN Continue DMA operation regardless of emulation suspend
 */
typedef enum
{
    DMA_EMULATION_STOP,
    DMA_EMULATION_FREE_RUN
} DMA_EmulationMode;


typedef enum
{
    DMA_AHB_MASTER_1 = 0,

} DMA_MasterInfSelect;

typedef enum
{
    DMA_TT_FC_0_M2M_DMAC = 0,
    DMA_TT_FC_1_M2P_DMAC,
    DMA_TT_FC_2_P2M_DMAC,
    DMA_TT_FC_3_P2P_DMAC,
    DMA_TT_FC_4_P2M_P,
    DMA_TT_FC_5_P2P_SP,
    DMA_TT_FC_6_M2P_P,
    DMA_TT_FC_7_P2P_DP,
} DMA_TransTypeFlowCtrl;

typedef enum
{
    DMA_BTL_1 = 0,
    DMA_BTL_4,
    DMA_BTL_8,
    DMA_BTL_16,
    DMA_BTL_32,
    DMA_BTL_64,
    DMA_BTL_128,
    DMA_BTL_256,
} DMA_BurstTransLength;

typedef enum
{
    DMA_ADDR_INCRE = 0,
    DMA_ADDR_DECRE,
    DMA_ADDR_NO_CHANGE,
} DMA_AddressIncrement;

typedef enum
{
    DMA_TR_WIDTH_BYTE_1 = 0,
    DMA_TR_WIDTH_BYTE_2,
    DMA_TR_WIDTH_BYTE_4,
    DMA_TR_WIDTH_BYTE_8,
    DMA_TR_WIDTH_BYTE_16,
    DMA_TR_WIDTH_BYTE_32,
    DMA_TR_WIDTH_BYTE_64,
    DMA_TR_WIDTH_BYTE_128,
} DMA_TransferWidth;

typedef enum
{
    DMA_HKS_HARDWARE = 0,
    DMA_HKS_SOFTWARE,
} DMA_HandshakingSelect;

typedef enum
{
    DMA_HKS_HARD_INF_0 = 0,
    DMA_HKS_HARD_INF_1,
    DMA_HKS_HARD_INF_2,
    DMA_HKS_HARD_INF_3,
    DMA_HKS_HARD_INF_4,
    DMA_HKS_HARD_INF_5,
    DMA_HKS_HARD_INF_6,
    DMA_HKS_HARD_INF_7,
    DMA_HKS_HARD_INF_8,
    DMA_HKS_HARD_INF_9,
    DMA_HKS_HARD_INF_10,
    DMA_HKS_HARD_INF_11,
    DMA_HKS_HARD_INF_12,
    DMA_HKS_HARD_INF_13,
    DMA_HKS_HARD_INF_14,
    DMA_HKS_HARD_INF_15,
} DMA_HK_HARDWARE_INF;

typedef enum
{
    DMA_CH_PRIORITY_0 = 0,
    DMA_CH_PRIORITY_1,
    DMA_CH_PRIORITY_2,
    DMA_CH_PRIORITY_3,
    DMA_CH_PRIORITY_4,
    DMA_CH_PRIORITY_5,
    DMA_CH_PRIORITY_6,
    DMA_CH_PRIORITY_7,
} DMA_CHANNEL_PRIORITY;

/**
 * @param enableInterrupt Enable/Disable interrupt mode 
 * @param destAddr destination address
 * @param srcAddr source address
 * @param blockTS block transfer size
 * @param ttfc  transfer type and flow control
 * @param srcBtl source burst transfer length
 * @param destBtl dest burst transfer length
 * @param srcAddrDirect src transfer width
 * @param destAddrDirect dest address Increment
 * @param srcTrWidthBytes src transfer width
 * @param destTrWidthBytes dest transfer width
 * @param srcHkSelect source handshaking select, reconfiged by driver inside
 * @param destHkSelect dest handshaking select, reconfiged by driver inside
 * @param srcHardInf source hardware handshaking interface, reconfiged by driver inside
 * @param destHardInf dest hardware handshaking interface, reconfiged by driver inside
 * @param reloadSrc source address auto reload
 * @param reloadDst dest address auto reload
 * @param dmaSrcReqId source request trigger id
 * @param dmaDstReqId dest request trigger id
 * @param srcGatherCount source gather count
 * @param srcGatherInterval source gather interval
 * @param destScatterCount dest scatter count
 * @param destScatterInterval dest scatter interval
 * @param chPriority A priority of 7 is the highest priority, and 0 is the lowest. 
 *                   This field must be programmed within the range 0 to DMAH_NUM_CHANNELS-1.
 *                   A programmed value outside this range will cause erroneous behavior.
 * @param enableBlkOfst Enable/Disable block offset
 * @param blockOffset Offset between first addresses of target adjacent blocks, negative complement representation
 * @param enableBlkCnt Enable/Disable block offset
 * @param blockCnt Number of blocks, when applied only in multiblock mode and non-linked table mode (i. e. auto_reload- -contiguous, 
 *                  contiguous- -auto_reload) trigger tfr interrupt when all blocks are completed.
 */
typedef struct
{
    boolean              enableInterrupt;
    uint32_t             destAddr;
    uint32_t             srcAddr;
    uint16_t blockTS;
    DMA_TransTypeFlowCtrl ttfc;
    DMA_BurstTransLength srcBtl;
    DMA_BurstTransLength destBtl;
    DMA_AddressIncrement srcAddrDirect;
    DMA_AddressIncrement destAddrDirect;
    DMA_TransferWidth    srcTrWidthBytes;
    DMA_TransferWidth    destTrWidthBytes;
    //
    // These following 4 items could not configured from user level.
    // they are re-configured by the function of DMA_configChannel inside driver.
    //
    DMA_HandshakingSelect srcHkSelect;
    DMA_HandshakingSelect destHkSelect;
    DMA_HK_HARDWARE_INF   srcHardInf;
    DMA_HK_HARDWARE_INF   destHardInf;
    boolean reloadSrc;
    boolean reloadDst;
    DMAMUX_ReqId_Type dmaSrcReqId;
    DMAMUX_ReqId_Type dmaDstReqId;
    uint32_t srcGatherCount;
    uint32_t srcGatherInterval;
    uint32_t destScatterCount;
    uint32_t destScatterInterval;

    DMA_CHANNEL_PRIORITY chPriority;
#if (DMAC_VERSION == 0x30)
    boolean                enableBlkOfst;
    uint32_t               blockOffset;
    boolean                enableBlkCnt;
    uint32_t               blockCnt;
#endif
} DMA_ConfigParams;

typedef struct
{
    union CTL_L_REG ctl_l;
    union CTL_H_REG ctl_h;
} DmaCh_Ctl;

typedef struct
{
    union CFG_L_REG cfg_l;
    union CFG_H_REG cfg_h;
} DmaCh_Cfg;

typedef struct
{
    DmaCh_Ctl ctl;
    DmaCh_Cfg cfg;
    union DMAMUX_CCR_REG  srcCcr;
    union DMAMUX_CCR_REG  destCcr;
    union DMAMUX_RGCR_REG srcRgcr;
    union DMAMUX_RGCR_REG destRgcr;
} DmaCh_Parameters;

#if (DMAC_NUMS >= 2U)
/**
 * @brief Checks an DMA channel base address.
 * @param base specifies the DMA channel base address.
 * 
 * @return true if the base address is valid
 *         false the base address is invalid
 */
__STATIC_INLINE boolean
DMA_isBaseValid(uint32_t base)
{
    return( 
            (base == DMA1_BASE)
            || (base == DMA1_CH1_BASE)
            || (base == DMA1_CH2_BASE)
            || (base == DMA1_CH3_BASE)
            || (base == DMA1_CH4_BASE)
            || (base == DMA1_CH5_BASE)
            || (base == DMA1_CH6_BASE)
            || (base == DMA1_CH7_BASE)
            || (base == DMA1_CH8_BASE)
            || (base == DMA2_BASE)
            || (base == DMA2_CH1_BASE)
            || (base == DMA2_CH2_BASE)
            || (base == DMA2_CH3_BASE)
            || (base == DMA2_CH4_BASE)
            || (base == DMA2_CH5_BASE)
            || (base == DMA2_CH6_BASE)
            || (base == DMA2_CH7_BASE)
            || (base == DMA2_CH8_BASE)
#if (DMAC_NUMS >= 3U) 
            || (base == DSP_DMA1_BASE)
            || (base == DSP_DMA1_CH0_BASE)
            || (base == DSP_DMA1_CH1_BASE)
            || (base == DSP_DMA1_CH2_BASE)
            || (base == DSP_DMA1_CH3_BASE)
            || (base == DSP_DMA1_CH4_BASE)
            || (base == DSP_DMA1_CH5_BASE)
            || (base == DSP_DMA1_CH6_BASE)
            || (base == DSP_DMA1_CH7_BASE)
#endif /* #if (DMAC_NUMS == 3U)  */
          );

}

/**
 * @brief Converts the channel base to the corresponding DMA module base.
 * @param base is the channel base address of the DMA controller.
 * 
 * @return uint32_t DMA module base
 */
__STATIC_INLINE uint32_t
DMA_convertChnBase2DmaBase(uint32_t base)
{
    uint32_t curDmacBase = 0U;
    
    switch(base)
    {
        case DMA1_CH1_BASE:
        case DMA1_CH2_BASE:
        case DMA1_CH3_BASE:
        case DMA1_CH4_BASE:
        case DMA1_CH5_BASE:
        case DMA1_CH6_BASE:
        case DMA1_CH7_BASE:
        case DMA1_CH8_BASE:
            curDmacBase = DMA1_BASE;
            break;

        case DMA2_CH1_BASE:
        case DMA2_CH2_BASE:
        case DMA2_CH3_BASE:
        case DMA2_CH4_BASE:
        case DMA2_CH5_BASE:
        case DMA2_CH6_BASE:
        case DMA2_CH7_BASE:
        case DMA2_CH8_BASE:
            curDmacBase = DMA2_BASE;
            break;

#if (DMAC_NUMS >= 3U) 
        case DSP_DMA1_CH0_BASE:
        case DSP_DMA1_CH1_BASE:
        case DSP_DMA1_CH2_BASE:
        case DSP_DMA1_CH3_BASE:
        case DSP_DMA1_CH4_BASE:
        case DSP_DMA1_CH5_BASE:
        case DSP_DMA1_CH6_BASE:
        case DSP_DMA1_CH7_BASE:
            curDmacBase = DSP_DMA1_BASE;
            break;
#endif /* #if (DMAC_NUMS >= 3U)  */

        default:
            curDmacBase = 0U;
            ASSERT(0);
            break;
    }

    return curDmacBase;    
}

/**
 * @brief Converts the channel base to the corresponding DMA MUX module base.
 * @param base is the channel base address of the DMA controller.
 * 
 * @return uint32_t Returns the corresponding DMA MUX Module base.
 */
__STATIC_INLINE uint32_t
DMA_convertChnbase2DmamuxBase(uint32_t base)
{
    uint32_t curDmamuxBase = 0U;
    
    switch(base)
    {
        case DMA1_CH1_BASE:
        case DMA1_CH2_BASE:
        case DMA1_CH3_BASE:
        case DMA1_CH4_BASE:
        case DMA1_CH5_BASE:
        case DMA1_CH6_BASE:
        case DMA1_CH7_BASE:
        case DMA1_CH8_BASE:
            curDmamuxBase = DMA1_MUX_BASE;
            break;

        case DMA2_CH1_BASE:
        case DMA2_CH2_BASE:
        case DMA2_CH3_BASE:
        case DMA2_CH4_BASE:
        case DMA2_CH5_BASE:
        case DMA2_CH6_BASE:
        case DMA2_CH7_BASE:
        case DMA2_CH8_BASE:
            curDmamuxBase = DMA2_MUX_BASE;
            break;
        
#if (DMAC_NUMS >= 3U) 
        case DSP_DMA1_CH0_BASE:
        case DSP_DMA1_CH1_BASE:
        case DSP_DMA1_CH2_BASE:
        case DSP_DMA1_CH3_BASE:
        case DSP_DMA1_CH4_BASE:
        case DSP_DMA1_CH5_BASE:
        case DSP_DMA1_CH6_BASE:
        case DSP_DMA1_CH7_BASE:
            curDmamuxBase = DSP_DMA1_MUX_BASE;
            break;
#endif /* #if (DMAC_NUMS >= 3U) */

        default:
            curDmamuxBase = 0U;
            ASSERT(0);
            break;
    }

    return curDmamuxBase;    
}

/**
 * @brief Converts the channel base to the DMA channel number.
 * @param base is the channel base address of the DMA controller.
 * 
 * @return uint32_t Returns the corresponding DMA channel Number.
 */
__STATIC_INLINE uint32_t
DMA_convertChnBase2ChnNum(uint32_t base)
{
    uint32_t chnnel = 0U;
    
    switch(base)
    {
        case DMA1_CH1_BASE:
        case DMA2_CH1_BASE:
            chnnel = 0;
            break;
        
        case DMA1_CH2_BASE:
        case DMA2_CH2_BASE:
            chnnel = 1;
            break;
        
        case DMA1_CH3_BASE:
        case DMA2_CH3_BASE:
            chnnel = 2;
            break;
        
        case DMA1_CH4_BASE:
        case DMA2_CH4_BASE:
            chnnel = 3;
            break;
        
        case DMA1_CH5_BASE:
        case DMA2_CH5_BASE:
            chnnel = 4;
            break;

        
        case DMA1_CH6_BASE:
        case DMA2_CH6_BASE:
            chnnel = 5;
            break;
        
        case DMA1_CH7_BASE:
        case DMA2_CH7_BASE:
            chnnel = 6;
            break;
        
        case DMA1_CH8_BASE:
        case DMA2_CH8_BASE:
            chnnel = 7;
            break;
        
#if (DMAC_NUMS >= 3U) 
        case DSP_DMA1_CH0_BASE:
            chnnel = 0;
            break;   

        case DSP_DMA1_CH1_BASE:
            chnnel = 1;
            break;            
        case DSP_DMA1_CH2_BASE:
            chnnel = 2;
            break;

        case DSP_DMA1_CH3_BASE:
            chnnel = 3;
            break;            
        case DSP_DMA1_CH4_BASE:
            chnnel = 4;
            break;            
        case DSP_DMA1_CH5_BASE:
            chnnel = 5;
            break;            
        case DSP_DMA1_CH6_BASE:
            chnnel = 6;
            break;            
        case DSP_DMA1_CH7_BASE:
            chnnel = 7;
            break;
#endif /* #if (DMAC_NUMS >= 3U) */        

        default:
            chnnel = 0;
            break;
    }

    return chnnel;    
}

#else

/**
 * @brief Checks an DMA channel base address.
 * @param base specifies the DMA channel base address.
 * 
 * @return boolean true the base address is valid
 *                 false invalid
*/
__STATIC_INLINE boolean
DMA_isBaseValid(uint32_t base)
{
    return( 
            (base == DMA_BASE) 
            || (base == DMA_CH1_BASE) 
            || (base == DMA_CH2_BASE) 
            || (base == DMA_CH3_BASE) 
            || (base == DMA_CH4_BASE)
            || (base == DMA_CH5_BASE)
            || (base == DMA_CH6_BASE)
#if (DMAC_CHNLS >= 7U)
            || (base == DMA_CH7_BASE)
            || (base == DMA_CH8_BASE)
#endif 
          );
}

/**
 * @brief Converts the channel base to the corresponding DMA module base.
 * @param  base is the channel base address of the DMA controller.
 * 
 * @return uint32_t Returns the corresponding DMA moudle base.
 */
__STATIC_INLINE uint32_t
DMA_convertChnBase2DmaBase(uint32_t base)
{
    uint32_t curDmacBase = 0U;
    
    switch(base)
    {
        case DMA_CH1_BASE:
        case DMA_CH2_BASE:
        case DMA_CH3_BASE:
        case DMA_CH4_BASE:
        case DMA_CH5_BASE:
        case DMA_CH6_BASE:
#if (DMAC_CHNLS >= 7U)
        case DMA_CH7_BASE:
        case DMA_CH8_BASE:
#endif            
            curDmacBase = DMA_BASE;
            break;

        default:
            curDmacBase = 0U;
            ASSERT(0);
            break;
    }

    return curDmacBase;    
}

/**
 * @brief Converts the channel base to the corresponding DMA MUX module base.
 * @param base is the channel base address of the DMA controller.
 * 
 * @return uint32_t Returns the corresponding DMA MUX Module base.
 */
__STATIC_INLINE uint32_t
DMA_convertChnbase2DmamuxBase(uint32_t base)
{
    uint32_t curDmamuxBase = 0U;
    
    switch(base)
    {
        case DMA_CH1_BASE:
        case DMA_CH2_BASE:
        case DMA_CH3_BASE:
        case DMA_CH4_BASE:
        case DMA_CH5_BASE:
        case DMA_CH6_BASE:
#if (DMAC_CHNLS >= 7U)
        case DMA_CH7_BASE:
        case DMA_CH8_BASE:
#endif             
            curDmamuxBase = DMA_MUX_BASE;
            break;

        default:
            curDmamuxBase = 0U;
            ASSERT(0);
            break;
    }

    return curDmamuxBase;    
}

/**
 * @brief Converts the channel base to the DMA channel number.
 * @param base is the channel base address of the DMA controller.
 * 
 * @return uint32_t Returns the corresponding DMA channel Number.
 */
__STATIC_INLINE uint32_t
DMA_convertChnBase2ChnNum(uint32_t base)
{
    uint32_t chnnel = 0U;
    
    switch(base)
    {
        case DMA_CH1_BASE:
            chnnel = 0;
            break;
        
        case DMA_CH2_BASE:
            chnnel = 1;
            break;
        
        case DMA_CH3_BASE:
            chnnel = 2;
            break;
        
        case DMA_CH4_BASE:
            chnnel = 3;
            break;
        
        case DMA_CH5_BASE:
            chnnel = 4;
            break;
        
        case DMA_CH6_BASE:
            chnnel = 5;
            break;
        
#if (DMAC_CHNLS >= 7U)
        case DMA_CH7_BASE:
            chnnel = 6;
            break;
        
        case DMA_CH8_BASE:
            chnnel = 7;
            break;     
#endif

        default:
            chnnel = 0;
            break;
    }

    return chnnel;    
}
#endif /* #if (DMAC_NUMS >= 2U) */

/**
 * @brief Initializes the DMA controller to a known state.
 * @param base is the channel base address of the DMA controller.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_initController(uint32_t base)
{
    uint32_t dmacBase = 0U;
    uint32_t dmamuxBase = 0U;

    ASSERT(DMA_isBaseValid(base));

    /* Covert the channel base to the Module base and the dmamux base address. */
    dmacBase = DMA_convertChnBase2DmaBase(base);
    dmamuxBase = DMA_convertChnbase2DmamuxBase(base);

    /* Dmac Disabled. */
    HWREG(dmacBase + DMA_O_DMACFGREG) &= (~DMA_DMACFGREG_DMA_EN);

    /* Check Dmac Disable success. */
    while (HWREG(dmacBase + DMA_O_DMACFGREG) & DMA_DMACFGREG_DMA_EN);

    
    /* Dmac Enabled. */
    HWREG(dmacBase + DMA_O_DMACFGREG) |= DMA_DMACFGREG_DMA_EN;

    /* Clear legacy config for dmamux module. */
    DMAMUX_resetDmaMux(dmamuxBase);    
}


/**
 * @brief Enable the DMA controller.
 * @param base is the channel base address of the DMA controller.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_enableModule(uint32_t base)
{
    uint32_t dmacBase = 0U;

    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));
    
    /* Covert the channel base to Module base. */
    dmacBase = DMA_convertChnBase2DmaBase(base);

    /* Enabled Dma Controller. */
    HWREG(dmacBase + DMA_O_DMACFGREG) |= DMA_DMACFGREG_DMA_EN;
}

/**
 * @brief Disable the DMA controller.
 * @param base is the channel base address of the DMA controller.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_disableModule(uint32_t base)
{
    uint32_t dmacBase = 0U;
    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));

    /* Covert the channel base to Module base. */
    dmacBase = DMA_convertChnBase2DmaBase(base);
    
    /* Disabled Dma Controller. */
    HWREG(dmacBase + DMA_O_DMACFGREG) &= ~(DMA_DMACFGREG_DMA_EN);
}

/**
 * @brief Trigger Soft Reset Channel 
 * @param base is the channel base address of the DMA controller.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_triggerSoftReset(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;

    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));

    /* Covert the channel base to channel number and dmac base base. */
    channel = DMA_convertChnBase2ChnNum(base);    
    dmacBase = DMA_convertChnBase2DmaBase(base);

    /* Disabled Dmac channel. */
    HWREG(dmacBase + DMA_O_CHENREG) = (HWREG(dmacBase + DMA_O_CHENREG) | (1 << (8 + channel)) & (~(1 << channel)));
    
    /* Check Dmac Disable success. */
    while (HWREG(dmacBase + DMA_O_CHENREG) & (1 << channel));
    
    /* Enabled Dmac channel again. */
    HWREG(dmacBase + DMA_O_CHENREG) |= ((1 << (8 + channel)) | (1 << channel));
}

/**
 * @brief Sets DMA emulation mode.
 * @param mode is the emulation mode to be selected.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_setEmulationMode(DMA_EmulationMode mode)
{
    //
    //  Empty function for current IP not support.
    //
    //  ASSERT(FALSE);
}

/**
 * @brief Enables peripherals to trigger a DMA transfer.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @retrun None.
 */
__STATIC_INLINE void
DMA_enableTrigger(uint32_t base)
{
    /* Empty function for current IP not support. */
    ASSERT(FALSE);
}

/**
 * @brief Disables peripherals from triggering a DMA transfer.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_disableTrigger(uint32_t base)
{
    
    /* Empty function for current IP not support. */
    ASSERT(FALSE);
}

/**
 * @brief Force a peripheral trigger to a DMA channel.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_forceTrigger(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;

    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));

    /* Covert the channel base to channel number and dmac base address. */
    channel = DMA_convertChnBase2ChnNum(base);
    dmacBase = DMA_convertChnBase2DmaBase(base);

    /* Config the channel to Softeware handshaking, Both for Ssource and Destination. */
    HWREG(base + DMA_O_CFG) |= (DMA_CFG_HS_SEL_SRC | DMA_CFG_HS_SEL_DST);

    /* Enable and Active source software transaction request for the DMA controller */
    HWREG(dmacBase + DMA_O_REQSRCREG) |= ((1 << channel) | (1 << (8 + channel)));
    
    /* Enable and Active destination software transaction request for the DMA controller */
    HWREG(dmacBase + DMA_O_REQDSTREG) |= ((1 << channel) | (1 << (8 + channel)));

    /* Enable and Active source Single transaction request for the DMA controller */
    HWREG(dmacBase + DMA_O_SGLRQSRCREG) |= ((1 << channel) | (1 << (8 + channel)));

    /* Enable and Active destination single transaction request for the DMA controller */
    HWREG(dmacBase + DMA_O_SGLRQDSTREG) |= ((1 << channel) | (1 << (8 + channel)));
}

/**
 * @brief Clears a DMA channel's peripheral trigger flag.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return  None.
 */
__STATIC_INLINE void
DMA_clearTriggerFlag(uint32_t base)
{
    //
    //  Empty function for current IP not support.
    //
    //  ASSERT(FALSE);
}

/**
 * @brief Gets the status of a DMA channel's Transfer Status Flag.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return true if the Transfer Status Flag is set. false otherwise
 */
__STATIC_INLINE boolean
DMA_getTransferStatusFlag(uint32_t base)
{
    
    /* Empty function for current IP not support. */
    ASSERT(FALSE);

    return false;
}

/**
 * @brief Gets the status of a DMA channel's Burst Status Flag.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return true if the Burst Status Flag is set. false otherwise.
 */
__STATIC_INLINE boolean
DMA_getBurstStatusFlag(uint32_t base)
{
    
    /* Empty function for current IP not support. */
    ASSERT(FALSE);

    return false;
}

/**
 * @brief Gets the status of a DMA channel's Run Status Flag.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return  true if the channel is enabled. false otherwise.
 */
__STATIC_INLINE boolean
DMA_getRunStatusFlag(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;

    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));

    /* Covert the channel base to channel number and dmac base address. */
    channel = DMA_convertChnBase2ChnNum(base);
    dmacBase = DMA_convertChnBase2DmaBase(base);
    
    /* Check the current channel is enabled or disabled and return. */
    return((HWREG(dmacBase + DMA_O_CHENREG) & (1 << channel)) != 0U);
}

/**
 * @brief Gets the status of a DMA channel's Overflow Flag.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return true if the channel is enabled. false otherwise.
 */
__STATIC_INLINE boolean
DMA_getOverflowFlag(uint32_t base)
{
    
    /* Empty function for current IP not support. */
    ASSERT(FALSE);

    return false;
}

/**
 * @brief Gets the status of a DMA channel's peripheral trigger flag.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return true if a peripheral trigger event has occurred and its flag is set.
 *         false otherwise.
 */
__STATIC_INLINE boolean
DMA_getTriggerFlagStatus(uint32_t base)
{
    
    /* Empty function for current IP not support. */
    ASSERT(FALSE);

    return false;
}

/**
 * @brief Starts a DMA channel.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_startChannel(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;
    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));
    
    /* Covert the channel base to channel number and dmac base address. */
    channel = DMA_convertChnBase2ChnNum(base);    
    dmacBase = DMA_convertChnBase2DmaBase(base);
    
    /* Enable the channel. */
    HWREG(dmacBase + DMA_O_CHENREG) = ((1 << (8 + channel)) | (1 << channel));
}

/**
 * @brief Halts a DMA channel.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_stopChannel(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;
    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));

    /* Covert the channel base to channel number and dmac base address. */
    channel = DMA_convertChnBase2ChnNum(base);     
    dmacBase = DMA_convertChnBase2DmaBase(base);
    
    /* Stop the channel. */
    HWREG(dmacBase + DMA_O_CHENREG) = ((HWREG(dmacBase + DMA_O_CHENREG) | (1 << (8 + channel))) & (~(1 << channel)));
}

/**
 * @brief Enables a DMA channel interrupt source.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_enableInterrupt(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));

    /* Enable Current Channel interrupt. */
    HWREG(base + DMA_O_CTL) |= DMA_CTL_INT_EN;
}

/**
 * @brief Disables a DMA channel interrupt source.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_disableInterrupt(uint32_t base)
{
    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));
    
    /* Disable Current Channel interrupt. */
    HWREG(base + DMA_O_CTL) &= (~DMA_CTL_INT_EN);
}

/**
 * @brief Enables the DMA channel overrun interrupt.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_enableOverrunInterrupt(uint32_t base)
{
    
    /* Empty function for current IP not support. */
    
    ASSERT(FALSE);
}


/**
 * @brief Disables the DMA channel overrun interrupt.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_disableOverrunInterrupt(uint32_t base)
{
    
    /* Empty function for current IP not support. */
    ASSERT(FALSE);
}

/**
 * @brief Clears the DMA channel error flags.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_clearErrorFlag(uint32_t base)
{
    uint32_t dmacBase = 0U;
    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));
    
    /* Covert the channel base to dmac base address. */
    dmacBase = DMA_convertChnBase2DmaBase(base);

    /* clear the whole dmac of the current channel 's Err interrupts. */
    HWREG(dmacBase + DMA_O_CLEARERR) |= DMA_CLEARERR_CLEAR;
}

/**
 * @brief Sets the interrupt generation mode of a DMA channel interrupt.
 * @param base is the base address of the DMA channel control registers.
 * @param mode is a flag to indicate the channel interrupt mode.
 * 
 * @return  None.
 */
__STATIC_INLINE void
DMA_setInterruptMode(uint32_t base, DMA_InterruptMode mode)
{
    /* Empty function for current IP not support. */
    ASSERT(FALSE);
}

/**
 * @brief Sets the DMA channel priority mode.
 * @param base is the base address of the DMA control registers.
 * @param ch1IsHighPri is a flag to indicate the channel interrupt mode.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_setPriorityMode(uint32_t base, boolean ch0IsHighPri)
{
    uint32_t dmacBase = 0U;
    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));
    
    /* Covert the channel base to dmac base address.
    Ch0 base address is the dmac base address. */
    dmacBase = DMA_convertChnBase2DmaBase(base);    

    if (ch0IsHighPri)
        HWREG(dmacBase + DMA_O_CFG) |= DMA_CFG_CH_PRIOR;
    else
        HWREG(dmacBase + DMA_O_CFG) &= (~DMA_CFG_CH_PRIOR);
}

/**
 * @brief set channel priority
 * @param base the base address of the DMA channel control registers.
 * @param prior the channel priority to set .
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_setChannelPriority(uint32_t base, DMA_CHANNEL_PRIORITY prior)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CFG) = HWREG(base + DMA_O_CFG) & (~DMA_CFG_CH_PRIOR) | (prior << 5);
}

/**
 * @brief Configures the source address for the DMA channel
 * @param base is the base address of the DMA channel control registers.
 * @param srcAddr is a source address.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_configSourceAddress(uint32_t base, uint32_t srcAddr)
{
    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));

    /* Set up SOURCE address. */
    HWREG(base + DMA_O_SAR) = (uint32_t)srcAddr;
}

/**
 * @brief Configures the destination address for the DMA channel
 * @param base is the base address of the DMA channel control registers.
 * @param destAddr is the destination address.
 * 
 * @return None
 */
__STATIC_INLINE void
DMA_configDestAddress(uint32_t base, uint32_t destAddr)
{

    ASSERT(DMA_isBaseValid(base));

    /* Set up DESTINATION address. */
    HWREG(base + DMA_O_DAR) = (uint32_t)destAddr;
}

/**
 * @brief config block transfer done
 * @param base the base address of the DMA channel control registers.
 * @param done 0: DONE bit is deasserted at the end of block transfer
 *             1: SET the DONE bit at the end of block transfer
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_configDone(uint32_t base, uint32_t done)
{
    ASSERT(DMA_isBaseValid(base));
    ASSERT((done >= 0U) && (done <= 1U));

    DMAREGH(base, DMA_O_CTL) = (DMAREGH(base, DMA_O_CTL) & (~(DMA_CTL_DONE >> 32)) | (done << 12));
}

/**
 * @brief Assigns a hardware handshaking interface (0 : 15)
 *
 * @param base the base address of the DMA channel control registers.
 * @param dest_per Dest Hardware Interface
 * @param src_per Source Hardware Interface
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_configHardwareInterface(uint32_t base, uint32_t dest_per, uint32_t src_per)
{
    ASSERT(DMA_isBaseValid(base));
    ASSERT((dest_per >= 0U) && (dest_per <= 15U));
    ASSERT((src_per >= 0U) && (src_per <= 15U));

    DMAREGH(base, DMA_O_CFG) = (DMAREGH(base, DMA_O_CFG) & (~((DMA_CFG_DEST_PER | DMA_CFG_SRC_PER) >> 32)) | (dest_per << 11) | (src_per << 7));
}

/**
 * @brief "config protection control"
 *
 * @param base     the base address of the DMA channel control registers.
 * @param protctl    Protection Control bits, 3 bits total for HPROT.
 *          CFGx.PROTCTL[1] to HPROT[1] privileged
 *          CFGx.PROTCTL[2] to HPROT[2] non-buffered
 *          CFGx.PROTCTL[3] to HPROT[3] non-cached
 *          ps: HPROT[0] is tied high
 * @return  None
 */
__STATIC_INLINE void
DMA_configProtctl(uint32_t base, uint32_t protctl)
{
    ASSERT(DMA_isBaseValid(base));
    ASSERT((protctl >= 0U) && (protctl <= 7U));

    DMAREGH(base, DMA_O_CFG) = (DMAREGH(base, DMA_O_CFG) & (~(DMA_CFG_PROTCTL >> 32)) | (protctl << 2));
}

/**
 * @brief FIFO Mode Select
 * @param base the base address of the DMA channel control registers.
 * @param fifo_mode  fifo mode as bellow:
 *          0x0 (FIFO_MODE_0): Space/data available for single
 *          AHB transfer of the specified transfer width
 *          0x1 (FIFO_MODE_1): Data available is greater than or
 *          equal to half the FIFO depth for destination transfers and
 *          space available is greater than half the fifo depth for
 *          source transfers. The exceptions are at the end of a burst
 *          transaction request or at the end of a block transfer
 * 
 * @return  None
 */
__STATIC_INLINE void
DMA_configFifoMode(uint32_t base, uint32_t fifo_mode)
{
    ASSERT(DMA_isBaseValid(base));
    ASSERT((fifo_mode >= 0U) && (fifo_mode <= 1U));

    DMAREGH(base, DMA_O_CFG) = (DMAREGH(base, DMA_O_CFG) & (~(DMA_CFG_FIFOMODE >> 32)) | (fifo_mode << 1));
}

/**
 * @brief Flow Control Mode
 * @param base the base address of the DMA channel control registers.
 * @param fc_mode  fc mode as bellow:
 *          0x0 (FCMODE_0): Source transaction requests are
 *          serviced when they occur. Data pre-fetching is enabled
 *          0x1 (FCMODE_1): Source transaction requests are not
 *          serviced until a destination transaction request occurs. In
 *           this mode, the amount of data transferred from the source
 *           is limited so that it is guaranteed to be transferred to the
 *           destination prior to block termination by the destination.
 *           Data pre-fetching is disabled.
 * 
 * @return  None
 */
__STATIC_INLINE void
DMA_configFcMode(uint32_t base, uint32_t fc_mode)
{
    ASSERT(DMA_isBaseValid(base));
    ASSERT((fc_mode >= 0U) && (fc_mode <= 1U));

    DMAREGH(base, DMA_O_CFG) = (DMAREGH(base, DMA_O_CFG) & (~(DMA_CFG_FCMODE >> 32)) | fc_mode);
}

/**
 * @brief "enable Automatic Reload"
 * @param  base the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_enableReload(uint32_t base)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CFG) |= (DMA_CFG_RELOAD_SRC | DMA_CFG_RELOAD_DST);
}

/**
 * @brief enable Automatic Reload
 * @param base  the base address of the DMA channel control registers.
 *
 * @return None
 */
__STATIC_INLINE void
DMA_enableReloadDst(uint32_t base)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CFG) |= DMA_CFG_RELOAD_DST;
}

/**
 * @brief enable Automatic Reload
 * @param base the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_disableReloadDst(uint32_t base)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CFG) &= DMA_CFG_RELOAD_DST;
}



/**
 * @brief disable Automatic Reload
 * @param base the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_disableReload(uint32_t base)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CFG) &= (~(DMA_CFG_RELOAD_SRC | DMA_CFG_RELOAD_DST));
}

/**
 * @brief config Handshaking Interface Polarity
 * @param base the base address of the DMA channel control registers.
 * @param src_hs_pol Source Handshaking Interface Polarity ,0 is Active high, 1 is Active low.
 * @param dst_hs_pol Dest Handshaking Interface Polarity ,0 is Active high, 1 is Active low.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_configHandshakingInterfacePolarity(uint32_t base, uint32_t src_hs_pol, uint32_t dst_hs_pol)
{
    ASSERT(DMA_isBaseValid(base));
    ASSERT((src_hs_pol >= 0U) && (src_hs_pol <= 1U));
    ASSERT((dst_hs_pol >= 0U) && (dst_hs_pol <= 1U));

    HWREG(base + DMA_O_CFG) = (HWREG(base + DMA_O_CFG) & (~(DMA_CFG_SRC_HS_POL | DMA_CFG_DST_HS_POL)) | (src_hs_pol << 19) | (dst_hs_pol << 18));
}

/**
 * @brief "Select Software or Hardware Handshaking"
 *
 * @param base     the base address of the DMA channel control registers.
 * @param hs_sel_src   Source Handshaking Interface select, 0 is hardware, 1 is software.
 * @param hs_sel_dst   Dest Handshaking Interface select, 0 is hardware, 1 is software.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_configHandshakingInterface(uint32_t base, uint32_t hs_sel_src, uint32_t hs_sel_dst)
{
    ASSERT(DMA_isBaseValid(base));
    ASSERT((hs_sel_src >= 0U) && (hs_sel_src <= 1U));
    ASSERT((hs_sel_dst >= 0U) && (hs_sel_dst <= 1U));

    HWREG(base + DMA_O_CFG) = (HWREG(base + DMA_O_CFG) & (~(DMA_CFG_HS_SEL_SRC | DMA_CFG_HS_SEL_DST)) | (hs_sel_src << 11) | (hs_sel_dst << 10));
}

/**
 * @brief get Channel FIFO status
 * @param base the base address of the DMA channel control registers.
 *
 * @return 0 is not empty, 1 is empty
 */
__STATIC_INLINE boolean
DMA_isFifoEmpty(uint32_t base)
{
    ASSERT(DMA_isBaseValid(base));

    return ((HWREG(base + DMA_O_CFG) & DMA_CFG_FIFO_EMPTY) != 0);
}

/**
 * @brief channel suspend
 * @param base the base address of the DMA channel control registers.
 * 
 * @return  None
 */
__STATIC_INLINE void
DMA_suspendChannel(uint32_t base)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CFG) |= DMA_CFG_CH_SUSP;
}

/**
 * @brief channel resume
 * @param base the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_resumeChannel(uint32_t base)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CFG) &= (~DMA_CFG_CH_SUSP);
}

/**
 * @brief config Block Transfer Size
 * @param base the base address of the DMA channel control registers.
 * @param block_ts Block Transfer Size, 0-4095.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_configBlockTs(uint32_t base, uint32_t block_ts)
{
    ASSERT(DMA_isBaseValid(base));

    DMAREGH(base, DMA_O_CTL) = DMAREGH(base, DMA_O_CTL) & (~(DMA_CTL_BLOCK_TS >> 32)) | block_ts;
}

/**
 * @brief config Transfer Type and Flow Control
 * @param base the base address of the DMA channel control registers.
 * @param tt_fc Transfer Type and Flow Control.
 *          0x0 (TT_FC_0): Transfer type is Memory to Memory and
 *          Flow Controller is DW_ahb_dmac
 *          0x1 (TT_FC_1): Transfer type is Memory to Peripheral
 *          and Flow Controller is DW_ahb_dmac
 *          0x2 (TT_FC_2): Transfer type is Peripheral to Memory
 *          and Flow Controller is DW_ahb_dmac
 *          0x3 (TT_FC_3): Transfer type is Peripheral to Peripheral
 *          and Flow Controller is DW_ahb_dmac
 *          0x4 (TT_FC_4): Transfer type is Peripheral to Memory
 *          and Flow Controller is Peripheral
 *          0x5 (TT_FC_5): Transfer type is Peripheral to Peripheral
 *          and Flow Controller is Source Peripheral
 *          0x6 (TT_FC_6): Transfer type is Memory to Peripheral
 *          and Flow Controller is Peripheral
 *          0x7 (TT_FC_7): Transfer type is Peripheral to Peripheral
 *          and Flow Controller is Destination Peripheral
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_configTtFc(uint32_t base, uint32_t tt_fc)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CTL) = HWREG(base + DMA_O_CTL) & (~DMA_CTL_TT_FC) | (tt_fc << 20);
}

/**
 * @brief  config Burst Transaction Length
 * @param base       the base address of the DMA channel control registers.
 * @param src_msize  source Burst Transaction Length.
 * @param dst_msize  dest Burst Transaction Length.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_configMSize(uint32_t base, uint32_t src_msize, uint32_t dst_msize)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CTL) = HWREG(base + DMA_O_CTL) & (~(DMA_CTL_SRC_MSIZE | DMA_CTL_DEST_MSIZE)) | (src_msize << 14) | (dst_msize << 11);
}

/**
 * @brief config Address Increment
 *
 * @param base the base address of the DMA channel control registers.
 * @param sinc source Address Increment.
 *          0x0 (SINC_0): Increments the source address
 *          0x1 (SINC_1): Decrements the source address
 *          0x2 (SINC_2): No change in the source address
 *          0x3 (SINC_3): No change in the source address
 * @param dinc dest Address Increment.
 *          0x0 (DINC_0): Increments the destination address
 *          0x1 (DINC_1): Decrements the destination address
 *          0x2 (DINC_2): No change in the destination address
 *          0x3 (DINC_3): No change in the destination address
 * 
 * @return  None
 */
__STATIC_INLINE void
DMA_configAddrInc(uint32_t base, uint32_t sinc, uint32_t dinc)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CTL) = HWREG(base + DMA_O_CTL) & (~(DMA_CTL_SINC | DMA_CTL_DINC)) | (sinc << 9) | (dinc << 7);
}

/**
 * @brief config Transfer Width
 *
 * @param base the base address of the DMA channel control registers.
 * @param src_tr_width source Transfer Width.
 *          0x0 (SRC_TR_WIDTH_0): Source transfer width is 8 bits
 *          0x1 (SRC_TR_WIDTH_1): Source transfer width is 16 bits
 *          0x2 (SRC_TR_WIDTH_2): Source transfer width is 32 bits
 *          0x3 (SRC_TR_WIDTH_3): Source transfer width is 64 bits
 *          0x4 (SRC_TR_WIDTH_4): Source transfer width is 128 bits
 *          0x5 (SRC_TR_WIDTH_5): Source transfer width is 256 bits
 *          0x6 (SRC_TR_WIDTH_6): Source transfer width is 256 bits
 *          0x7 (SRC_TR_WIDTH_7): Source transfer width is 256 bits
 * @param dst_tr_width dest Transfer Width.
 *          0x0 (DST_TR_WIDTH_0): Destination transfer width is 8 bits
 *          0x1 (DST_TR_WIDTH_1): Destination transfer width is 16 bits
 *          0x2 (DST_TR_WIDTH_2): Destination transfer width is 32 bits
 *          0x3 (DST_TR_WIDTH_3): Destination transfer width is 64 bits
 *          0x4 (DST_TR_WIDTH_4): Destination transfer width is 128 bits
 *          0x5 (DST_TR_WIDTH_5): Destination transfer width is 256 bits
 *          0x6 (DST_TR_WIDTH_6): Destination transfer width is 256 bits
 *          0x7 (DST_TR_WIDTH_7): Destination transfer width is 256 bits
 * 
 * @return  None
 */
__STATIC_INLINE void
DMA_configTrWidth(uint32_t base, uint32_t src_tr_width, uint32_t dst_tr_width)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CTL) = HWREG(base + DMA_O_CTL) & (~(DMA_CTL_SRC_TR_WIDTH | DMA_CTL_DST_TR_WIDTH)) | (src_tr_width << 4) | (dst_tr_width << 1);
}

/**
 * @brief Source gather enable
 * @param base the base address of the DMA channel control registers
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_enableSrcGather(uint32_t base)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CTL) |= DMA_CTL_SRC_GATHER_EN;
}

/**
 * @brief Source gather disable
 * @param base the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_disableSrcGather(uint32_t base)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CTL) &= (~DMA_CTL_SRC_GATHER_EN);
}

/**
 * @brief Destination scatter enable
 * @param base the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_enableDstScatter(uint32_t base)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CTL) |= DMA_CTL_DST_SCATTER_EN;
}

/**
 * @brief Destination scatter disable
 * @param base the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_disableDstScatter(uint32_t base)
{
    ASSERT(DMA_isBaseValid(base));

    HWREG(base + DMA_O_CTL) &= (~DMA_CTL_DST_SCATTER_EN);
}

/**
 * @brief config source gather
 * @param base the base address of the DMA channel control registers.
 * @param sgi Source Gather Interval.
 * @param sgc Source Gather Count. Source contiguous transfer count between successive gather boundaries.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_configSrcGather(uint32_t base, uint32_t sgi, uint32_t sgc)
{
    ASSERT(DMA_isBaseValid(base));

#if (DMAC_VERSION == 0x30)
    DMAREGH(base, DMA_O_SGR) = (sgc & DMA_SGR_SGC_MASK);
    DMAREGL(base, DMA_O_SGR) = (sgi & DMA_SGR_SGI_MASK);
#else
    HWREG(base + DMA_O_SGR) = ((sgc << 20) & DMA_SGR_SGC_MASK) | (sgi & DMA_SGR_SGI_MASK);
#endif
}

/**
 * @brief config dest scatter
 * @param base he base address of the DMA channel control registers.
 * @param dsi Destination Scatter Interval.
 * @param dsc Destination Scatter Count.Destination contiguous transfer count between successive scatter boundaries.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_configDstScatter(uint32_t base, uint32_t dsi, uint32_t dsc)
{
    ASSERT(DMA_isBaseValid(base));
#if (DMAC_VERSION == 0x30)
    DMAREGH(base, DMA_O_DSR) = (dsc & DMA_DSR_DSC_MASK);
    DMAREGL(base, DMA_O_DSR) = (dsi & DMA_DSR_DSI_MASK);
#else
    HWREG(base + DMA_O_DSR) = ((dsc << 20) & DMA_DSR_DSC_MASK) | (dsi & DMA_DSR_DSI_MASK);
#endif
}

/**
 * @brief config channel ctrl
 * @param base the base address of the DMA channel control registers.
 * @param ctl
 *          DONE             bit    44           Done bit.
 *          BLOCK_TS         bit    43:32        Block Transfer Size.
 *          TT_FC            bit    22:20        Transfer Type and Flow Control.
 *          DST_SCATTER_EN   bit    18           Destination scatter enable.
 *          SRC_GATHER_EN    bit    17           Source gather enable.
 *          SRC_MSIZE        bit    16:14        Source Burst Transaction Length.
 *          DEST_MSIZE       bit    13:11        Destination Burst Transaction Length.
 *          SINC             bit    10:9         Source Address Increment.
 *          DINC             bit    8:7          Destination Address Increment.
 *          SRC_TR_WIDTH     bit    6:4          Source Transfer Width.
 *          DST_TR_WIDTH     bit    3:1          Destination Transfer Width.
 *          INT_EN           bit    0            Interrupt Enable Bit.
 * 
 * @return  None
 */
__STATIC_INLINE void
DMA_configChannelCTL(uint32_t base, uint64_t ctl)
{
    DMAREGH(base, DMA_O_CTL) = ((ctl >> 32) & 0xffffffff);
    DMAREGL(base, DMA_O_CTL) = (ctl & 0xffffffff);
}

__STATIC_INLINE void
DMA_configChannelBlockTS(uint32_t base, uint32_t blockTS)
{
    DMAREGH(base, DMA_O_CTL) = (DMAREGH(base, DMA_O_CTL) & 0xfffff000) | (blockTS & 0x00000fff);
}

/**
 * @brief config channel block offset
 * @param base the base address of the DMA channel control registers.
 * @param blkOfst
 *        BLOCK_OFFSET_EN  bit    16           Block Offset En bit.
 *        BLOCK_OFFSET     bit    15:0         Block Offset.
 * 
 * @return  None
 */
__STATIC_INLINE void
DMA_configChannelBLKOFST(uint32_t base, uint64_t blkOfst)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;
    
    /* Covert the channel base to the channel number and dmac base */
    channel = DMA_convertChnBase2ChnNum(base);    
    dmacBase = DMA_convertChnBase2DmaBase(base);

    DMAREGH(base, DMA_O_BOR_OFFSET + channel *0x50U) = ((blkOfst >> 32) & 0xffffffff);
    DMAREGL(base, DMA_O_BOR_OFFSET + channel *0x50U) = (blkOfst & 0xffffffff);
}

/**
 * @brief config channel block offset
 * @param base the base address of the DMA channel control registers.
 * @param blkCnt
 *          BLOCK_CNT_EN     bit    16           Block Count En bit.
 *          BLOCK_CNT        bit    15:0         Block Count.
 * 
 * @return  None
 */
__STATIC_INLINE void
DMA_configChannelBLKCNT(uint32_t base, uint64_t blkCnt)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;
    volatile uint32_t temp = 0U;
    
    /* Covert the channel base to the channel number and dmac base */
    channel = DMA_convertChnBase2ChnNum(base);    
    dmacBase = DMA_convertChnBase2DmaBase(base);

    temp = (blkCnt >> 32) & 0xFFFFFFFF;

    DMAREGH(dmacBase, DMA_O_BLKCNT_OFFSET + channel *0x50U) = temp;//((blkCnt >> 32) & 0xffffffff);
    temp = (blkCnt) & 0xFFFFFFFF;
    DMAREGL(dmacBase, DMA_O_BLKCNT_OFFSET + channel *0x50U) = temp;//(blkCnt & 0xffffffff);
}


/**
 * @brief config channel ctl
 * @param base the base address of the DMA channel control registers.
 * @param ctl  the pointer of ctl value struct
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_Ch_setCtl(uint32_t base, DmaCh_Ctl *ctl)
{
    DMAREGH(base, DMA_O_CTL) = (ctl->ctl_h.all);
    DMAREGL(base, DMA_O_CTL) = (ctl->ctl_l.all);
}

/**
 * @brief config channel cfg
 * @param base the base address of the DMA channel control registers.
 * @param cfg
 *          DEST_PER           bit    46:43        Assigns a Destination hardware interface.
 *          SRC_PER            bit    42:39        Assigns a Source Hardware Interface.
 *          PROTCTL            bit    36:34        Protection Control bits used to drive the AHB HPROT[3:1] bus.
 *          FIFO_MODE          bit    33           FIFO Mode Select.
 *          FCMODE             bit    32           Flow Control Mode.
 *          RELOAD_DST         bit    31           Automatic Destination Reload.
 *          RELOAD_SRC         bit    30           Automatic Source Reload.
 *          SRC_HS_POL         bit    19           Source Handshaking Interface Polarity
 *          DST_HS_POL         bit    18           Destination Handshaking Interface Polarity.
 *          HS_SEL_SRC         bit    11           Source Software or Hardware Handshaking Select.
 *          HS_SEL_DST         bit    10           Destination Software or Hardware Handshaking Select.
 *          CH_SUSP            bit    8            Channel Suspend.
 *          CH_PRIOR           bit    7:5          Channel Priority.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_configChannelCFG(uint32_t base, uint64_t cfg)
{
    DMAREGH(base, DMA_O_CFG) = ((cfg >> 32) & 0xffffffff);
    DMAREGL(base, DMA_O_CFG) = (cfg & 0xffffffff);
}

/**
 * @brief config channel cfg
 * @param base the base address of the DMA channel control registers.
 * @param cfg  the pointer of cfg value struct
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_Ch_setCfg(uint32_t base, DmaCh_Cfg *cfg)
{
    DMAREGH(base, DMA_O_CFG) = (cfg->cfg_h.all);
    DMAREGL(base, DMA_O_CFG) = (cfg->cfg_l.all);
}

/**
 * @brief get channel ctr
 * @param base the base address of the DMA channel control registers.
 * 
 * @return ctl value
 *          DONE             bit    44           Done bit.
 *          BLOCK_TS         bit    43:32        Block Transfer Size.
 *          TT_FC            bit    22:20        Transfer Type and Flow Control.
 *          DST_SCATTER_EN   bit    18           Destination scatter enable.
 *          SRC_GATHER_EN    bit    17           Source gather enable.
 *          SRC_MSIZE        bit    16:14        Source Burst Transaction Length.
 *          DEST_MSIZE       bit    13:11        Destination Burst Transaction Length.
 *          SINC             bit    10:9         Source Address Increment.
 *          DINC             bit    8:7          Destination Address Increment.
 *          SRC_TR_WIDTH     bit    6:4          Source Transfer Width.
 *          DST_TR_WIDTH     bit    3:1          Destination Transfer Width.
 *          INT_EN           bit    0            Interrupt Enable Bit.
 */
__STATIC_INLINE uint64_t
DMA_getChannelCTL(uint32_t base)
{
    return ((uint64_t)DMAREGH(base, DMA_O_CTL) << 32) + (uint64_t)DMAREGL(base, DMA_O_CTL);
}

/**
 * @brief get channel ctl
 *
 * @param base the base address of the DMA channel control registers.
 * @return ctl  value of channel ctl
 *
 */
__STATIC_INLINE DmaCh_Ctl
DMA_Ch_getCtl(uint32_t base)
{
    DmaCh_Ctl ctl = {0};
    ctl.ctl_h.all = DMAREGH(base, DMA_O_CTL);
    ctl.ctl_l.all = DMAREGL(base, DMA_O_CTL);

    return ctl;
}

/**
 * @brief get channel cfg
 * @param base the base address of the DMA channel control registers.
 * 
 * @return cfg value
 *          DEST_PER           bit    46:43        Assigns a Destination hardware interface.
 *          SRC_PER            bit    42:39        Assigns a Source Hardware Interface.
 *          PROTCTL            bit    36:34        Protection Control bits used to drive the AHB HPROT[3:1] bus.
 *          FIFO_MODE          bit    33           FIFO Mode Select.
 *          FCMODE             bit    32           Flow Control Mode.
 *          RELOAD_DST         bit    31           Automatic Destination Reload.
 *          RELOAD_SRC         bit    30           Automatic Source Reload.
 *          SRC_HS_POL         bit    19           Source Handshaking Interface Polarity
 *          DST_HS_POL         bit    18           Destination Handshaking Interface Polarity.
 *          HS_SEL_SRC         bit    11           Source Software or Hardware Handshaking Select.
 *          HS_SEL_DST         bit    10           Destination Software or Hardware Handshaking Select.
 *          FIFO_EMPTY         bit    9            Channel FIFO status.
 *          CH_SUSP            bit    8            Channel Suspend.
 *          CH_PRIOR           bit    7:5          Channel Priority.
 *
 */
__STATIC_INLINE uint64_t
DMA_getChannelCFG(uint32_t base)
{
    return ((uint64_t)DMAREGH(base, DMA_O_CFG) << 32) + (uint64_t)DMAREGL(base, DMA_O_CFG);
}

/**
 * @brief get channel cfg
 * @param base the base address of the DMA channel control registers.
 * 
 * @return  cfg           value of channel cfg
 *
 */
__STATIC_INLINE DmaCh_Cfg
DMA_Ch_getCfg(uint32_t base)
{
    DmaCh_Cfg cfg = {0};
    cfg.cfg_h.all = DMAREGH(base, DMA_O_CFG);
    cfg.cfg_l.all = DMAREGL(base, DMA_O_CFG);

    return cfg;
}

/**
 * @brief get Raw Interrupt Status
 * @param base the base address of the DMA channel control registers.
 *
 * @return  status like bellow
            DMA_INT_TFR
            DMA_INT_BLOCK
            DMA_INT_SRCTRAN
            DMA_INT_DSTTRAN
            DMA_INT_ERR
 */
__STATIC_INLINE uint8_t
DMA_getRawInterruptStatus(uint32_t base)
{
    uint8_t status = 0U;
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;

    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));

    /* Covert the channel base to channel number and dmac base address. */
    channel = DMA_convertChnBase2ChnNum(base);
    dmacBase = DMA_convertChnBase2DmaBase(base);

    /* Check raw Tfr interrupt status */
    if ((HWREG(dmacBase + DMA_O_RAWTFR) & (1 << channel)) != 0)
        status |= DMA_INT_TFR;
    
    /* Check raw Block interrupt status */
    if ((HWREG(dmacBase + DMA_O_RAWBLOCK) & (1 << channel)) != 0)
        status |= DMA_INT_BLOCK;
    
    /* Check raw SrcTran interrupt status */
    if ((HWREG(dmacBase + DMA_O_RAWSRCTRAN) & (1 << channel)) != 0)
        status |= DMA_INT_SRCTRAN;
    
    /* Check raw DstTran interrupt status */
    if ((HWREG(dmacBase + DMA_O_RAWDSTTRAN) & (1 << channel)) != 0)
        status |= DMA_INT_DSTTRAN;
    
    /* Check raw Err interrupt status */
    if ((HWREG(dmacBase + DMA_O_RAWERR) & (1 << channel)) != 0)
        status |= DMA_INT_ERR;

    return status;
}

/**
 * @brief get Interrupt Status
 * @param base the base address of the DMA channel control registers.
 *
 * @return  status like bellow
 *          DMA_INT_TFR
 *          DMA_INT_BLOCK
 *          DMA_INT_SRCTRAN
 *          DMA_INT_DSTTRAN
 *          DMA_INT_ERR
 *          DMA_INT_FDI
 */
__STATIC_INLINE uint8_t
DMA_getInterruptStatus(uint32_t base)
{
    uint8_t status = 0U;
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;

    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));
    
    /* Covert the channel base to channel number and dmac base address. */
    channel = DMA_convertChnBase2ChnNum(base);   
    dmacBase = DMA_convertChnBase2DmaBase(base);

    
    /* Check Tfr interrupt status */
    if ((HWREG(dmacBase + DMA_O_STATUSTFR) & (1 << channel)) != 0)
        status |= DMA_INT_TFR;
    
    /* Check Block interrupt status */
    if ((HWREG(dmacBase + DMA_O_STATUSBLOCK) & (1 << channel)) != 0)
        status |= DMA_INT_BLOCK;
    
    /* Check SrcTran interrupt status */
    if ((HWREG(dmacBase + DMA_O_STATUSSRCTRAN) & (1 << channel)) != 0)
        status |= DMA_INT_SRCTRAN;
    
    /* Check DstTran interrupt status */
    if ((HWREG(dmacBase + DMA_O_STATUSDSTTRAN) & (1 << channel)) != 0)
        status |= DMA_INT_DSTTRAN;
    
    /* Check Err interrupt status */
    if ((HWREG(dmacBase + DMA_O_STATUSERR) & (1 << channel)) != 0)
        status |= DMA_INT_ERR;

#if (DMAC_VERSION == 0x30)
    
    /* Check FDI interrupt status */
    if ((HWREG(dmacBase + DMA_O_STATUSFDI) & (1 << channel)) != 0)
        status |= DMA_INT_FDI;
#endif
    return status;

}

/**
 * @brief Gets the block interrupt Status of the current DMA channel.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return true if the channel is enabled. false otherwise.
 */
__STATIC_INLINE boolean
DMA_getBlockInterruptStatus(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;
    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));
    
    /* Covert the channel base to channel number and dmac base address. */
    channel = DMA_convertChnBase2ChnNum(base);
    dmacBase = DMA_convertChnBase2DmaBase(base);
    
    /* Get the block interrupt status of current DMA channel. */
    if (HWREG(dmacBase + DMA_O_STATUSBLOCK) & (1 << channel))
        return true;
    else
        return false;
}

/**
 * @brief Gets the block interrupt Status of the current DMA channel.
 * @param dmacBase is the base address of the DMA controler base.
 * @param nChannel is the current dma channel number.
 * 
 * @return true if the channel is block. false otherwise.
 */
__STATIC_INLINE boolean
DMA_getBlockInterruptStatusbyChns(uint32_t dmacBase, uint32_t nChannel)
{   
    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(dmacBase));
    ASSERT(nChannel <= 7U);   
    
    /* Get the block interrupt status of current DMA channel. */
    if (HWREG(dmacBase + DMA_O_STATUSBLOCK) & (1 << nChannel))
        return true;
    else
        return false;
}

/**
 * @brief Gets the block interrupt Status of the current DMA channel.
 * @param dmacBase is the base address of the DMA controler base.
 * @param nChannel is the current dma channel number.
 * 
 * @return true if the channel is enabled. false otherwise
 */
__STATIC_INLINE boolean
DMA_getBlockInterruptStatusbyChnMsk(uint32_t dmacBase, uint32_t nChannelMask)
{   
    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(dmacBase));
    ASSERT(nChannelMask <= 0xFFU);   
    
    /* Get the block interrupt status of current DMA channel. */
    if (HWREG(dmacBase + DMA_O_STATUSBLOCK) & nChannelMask)
        return true;
    else
        return false;
}

/**
 * @brief clears the block interrupt Status of the current DMA channel.
 * @param base is the base address of the DMA channel control registers.
 * 
 * @return None.
 */
__STATIC_INLINE void
DMA_clearBlockInterruptStatus(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;
    
    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));

    /* Covert the channel base to channel number and dmac base address. */
    channel = DMA_convertChnBase2ChnNum(base);
    dmacBase = DMA_convertChnBase2DmaBase(base);
    
    /* Clear the block interrupt status of current DMA channel. */
    HWREG(dmacBase + DMA_O_CLEARBLOCK) = (1 << channel);
}

/**
 * @brief clears the block interrupt Status of the current DMA channel.
 * @param dmacBase is the base address of the DMA controler base.
 * @param nChannel is the current dma channel number.
 * 
 * @return none.
 */
__STATIC_INLINE void
DMA_clearBlockInterruptStatusbyChns(uint32_t dmacBase, uint32_t nChannel)
{
    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(dmacBase));
    ASSERT(nChannel <= 7U);  
    
    /* Clear the block interrupt status of current DMA channel. */
    HWREG(dmacBase + DMA_O_CLEARBLOCK) = (1 << nChannel);
}


/**
 * @brief "mask Interrupt"
 *
 * @param base the base address of the DMA channel control registers.
 * @param mask   interrupt to mask, as bellow
 *          DMA_INT_TFR
 *          DMA_INT_BLOCK
 *          DMA_INT_SRCTRAN
 *          DMA_INT_DSTTRAN
 *          DMA_INT_ERR
 *          DMA_INT_FDI
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_maskInterrupt(uint32_t base, uint8_t mask)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;
    
    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));

    /* Covert the channel base to channel number and dmac base address. */
    channel = DMA_convertChnBase2ChnNum(base);
    dmacBase = DMA_convertChnBase2DmaBase(base);

    /* Check Tfr interrupt mask, if mask contain Tfr, set mask and enable */
    if (mask & DMA_INT_TFR)
        HWREG(dmacBase + DMA_O_MASKTFR) = HWREG(dmacBase + DMA_O_MASKTFR) | (1 << (8 + channel)) & (~(1 << channel));
    
    /* Check Block interrupt mask, if mask contain Block, set mask and enable */
    if (mask & DMA_INT_BLOCK)
        HWREG(dmacBase + DMA_O_MASKBLOCK) = HWREG(dmacBase + DMA_O_MASKBLOCK) | (1 << (8 + channel)) & (~(1 << channel));
    
    /* Check SrcTran interrupt mask, if mask contain SrcTran, set mask and enable */
    if (mask & DMA_INT_SRCTRAN)
        HWREG(dmacBase + DMA_O_MASKSRCTRAN) = HWREG(dmacBase + DMA_O_MASKSRCTRAN) | (1 << (8 + channel)) & (~(1 << channel));
    
    /* Check DstTran interrupt mask, if mask contain DstTran, set mask and enable */
    if (mask & DMA_INT_DSTTRAN)
        HWREG(dmacBase + DMA_O_MASKDSTTRAN) = HWREG(dmacBase + DMA_O_MASKDSTTRAN) | (1 << (8 + channel)) & (~(1 << channel));
    
    /* Check Err interrupt mask, if mask contain Err, set mask and enable */
    if (mask & DMA_INT_ERR)
        HWREG(dmacBase + DMA_O_MASKERR) = HWREG(dmacBase + DMA_O_MASKERR) | (1 << (8 + channel)) & (~(1 << channel));   

#if (DMAC_VERSION == 0x30)
    
    /* Check FDI interrupt mask, if mask contain Err, set mask and enable */
    if (mask & DMA_INT_FDI)
        HWREG(dmacBase + DMA_O_MASKFDI) = HWREG(dmacBase + DMA_O_MASKFDI) | (1 << (8 + channel)) & (~(1 << channel)); 
#endif
}

/**
 * @brief unmask Interrupt
 * @param base the base address of the DMA channel control registers.
 * @param mask   interrupt to unmask, as bellow
 *          DMA_INT_TFR
 *          DMA_INT_BLOCK
 *          DMA_INT_SRCTRAN
 *          DMA_INT_DSTTRAN
 *          DMA_INT_ERR
 *          DMA_INT_FDI
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_unMaskInterrupt(uint32_t base, uint8_t mask)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;

    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));

    /* Covert the channel base to channel number and dmac base address. */
    channel = DMA_convertChnBase2ChnNum(base);
    dmacBase = DMA_convertChnBase2DmaBase(base);

    /* Check Tfr interrupt mask, if mask contain Tfr, set un-mask and enable */
    if (mask & DMA_INT_TFR)
        HWREG(dmacBase + DMA_O_MASKTFR) |= ((1 << (8 + channel)) | (1 << channel));
    
    /* Check Block interrupt mask, if mask contain Block, set un-mask and enable */
    if (mask & DMA_INT_BLOCK)
        HWREG(dmacBase + DMA_O_MASKBLOCK) |= ((1 << (8 + channel)) | (1 << channel));
    
    /* Check SrcTran interrupt mask, if mask contain SrcTran, set un-mask and enable */
    if (mask & DMA_INT_SRCTRAN)
        HWREG(dmacBase + DMA_O_MASKSRCTRAN) |= ((1 << (8 + channel)) | (1 << channel));
    
    /* Check DstTran interrupt mask, if mask contain DstTran, set un-mask and enable */
    if (mask & DMA_INT_DSTTRAN)
        HWREG(dmacBase + DMA_O_MASKDSTTRAN) |= ((1 << (8 + channel)) | (1 << channel));
    
    /* Check Err interrupt mask, if mask contain Err, set un-mask and enable */
    if (mask & DMA_INT_ERR)
        HWREG(dmacBase + DMA_O_MASKERR) |= ((1 << (8 + channel)) | (1 << channel));
    
#if (DMAC_VERSION == 0x30)    
    
    /* Check FDI interrupt mask, if mask contain Err, set un-mask and enable */
    if (mask & DMA_INT_FDI)
        HWREG(dmacBase + DMA_O_MASKFDI) |= ((1 << (8 + channel)) | (1 << channel));
#endif    
}

/**
 * @brief clear Interrupt
 *
 * @param base the base address of the DMA channel control registers.
 * @param mask   interrupt to clear, as bellow
 *          DMA_INT_TFR
 *          DMA_INT_BLOCK
 *          DMA_INT_SRCTRAN
 *          DMA_INT_DSTTRAN
 *          DMA_INT_ERR
 *          DMA_INT_FDI
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_clearInterrupt(uint32_t base, uint8_t clear)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;

    ASSERT(DMA_isBaseValid(base));

    /* Covert the channel base to channel number and dmac base address. */
    channel = DMA_convertChnBase2ChnNum(base);
    dmacBase = DMA_convertChnBase2DmaBase(base);

    /* Check Tfr interrupt clear mask, if the clear mask contain Tfr, clear all TFR interrupt status. */
    if (clear & DMA_INT_TFR)
        HWREG(dmacBase + DMA_O_CLEARTFR) = (1 << channel);
    
    /* Check Block interrupt clear mask, if the clear mask contain Block, clear all Block interrupt status.*/
    if (clear & DMA_INT_BLOCK)
        HWREG(dmacBase + DMA_O_CLEARBLOCK) = (1 << channel);
    
    /* Check SrcTran interrupt clear mask, if the clear mask contain SrcTran, clear all SrcTran interrupt status.*/
    if (clear & DMA_INT_SRCTRAN)
        HWREG(dmacBase + DMA_O_CLEARSRCTRAN) = (1 << channel);
    
    /* Check DstTran interrupt clear mask, if the clear mask contain DstTran, clear all DstTran status. */
    if (clear & DMA_INT_DSTTRAN)
        HWREG(dmacBase + DMA_O_CLEARDSTTRAN) = (1 << channel);
#if (DMAC_VERSION == 0x30)       
    
    /* Check Err interrupt clear mask, if the clear mask contain Err, clear all Err status.*/
    if (clear & DMA_INT_FDI)
        HWREG(dmacBase + DMA_O_STATUSFDI) = (1 << channel);
#endif    
}

/**
 * @brief Source Software Transaction Request
 * @param base the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_ReqSrcSoftwareTransaction(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;

    ASSERT(DMA_isBaseValid(base));

    channel = DMA_convertChnBase2ChnNum(base);
    dmacBase = DMA_convertChnBase2DmaBase(base);

    /* Set source software transaction request enable and active  */
    HWREG(dmacBase + DMA_O_REQSRCREG) |= ((1 << channel) | (1 << (8 + channel)));
}

/**
 * @brief Destination Software Transaction Request
 * @param base the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_ReqDstSoftwareTransaction(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;

    /* Check the arguments. */
    ASSERT(DMA_isBaseValid(base));

    /* Covert the channel base to channel number and dmac base address. */
    channel = DMA_convertChnBase2ChnNum(base);
    dmacBase = DMA_convertChnBase2DmaBase(base);
    
    /* Set Destination software transactionrequest enable and active  */
    HWREG(dmacBase + DMA_O_REQDSTREG)|= ((1 << channel) | (1 << (8 + channel)));
}

/**
 * @brief Source Software Sigle Transaction Request
 * @param base the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_ReqSglSrcSoftwareTransaction(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;

    ASSERT(DMA_isBaseValid(base));

    /* Covert the channel base to channel number and dmac base address. */
    channel = DMA_convertChnBase2ChnNum(base);
    dmacBase = DMA_convertChnBase2DmaBase(base);

    /* Set Source single transaction request enable and active */
    HWREG(dmacBase + DMA_O_SGLRQSRCREG) |= ((1 << channel) | (1 << (8 + channel)));
}

/**
 * @brief Destination Software Sigle Transaction Request
 * @param base the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_ReqSglDstSoftwareTransaction(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;

    ASSERT(DMA_isBaseValid(base));

    channel = DMA_convertChnBase2ChnNum(base);
    dmacBase = DMA_convertChnBase2DmaBase(base);
    
    /* Set Destination single transaction request enable and active */
    HWREG(dmacBase + DMA_O_SGLRQDSTREG) |= ((1 << channel) | (1 << (8 + channel)));
}

/**
 * @brief Last Source Software Transaction Request
 * @param base the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_ReqlstSrcSoftwareTransaction(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;

    ASSERT(DMA_isBaseValid(base));

    channel = DMA_convertChnBase2ChnNum(base);
    dmacBase = DMA_convertChnBase2DmaBase(base);

    /* Set Source Last transaction request enable and active  */
    HWREG(dmacBase + DMA_O_LSTSRCREG) |= ((1 << channel) | (1 << (8 + channel)));
}

/**
 * @brief Last Destination Software Transaction Request
 * @param base the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
DMA_ReqlstDstSoftwareTransaction(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;

    ASSERT(DMA_isBaseValid(base));

    channel = DMA_convertChnBase2ChnNum(base);
    dmacBase = DMA_convertChnBase2DmaBase(base);
    
    /* Set Destination Last transaction request enable and active  */
    HWREG(dmacBase + DMA_O_LSTDSTREG) |= ((1 << channel) | (1 << (8 + channel)));
}


#if (DMAC_VERSION == 0x30)
#define ILM_RAM_BASE                    (DSP_RAM_ILM_BASE)
#define ILM_RAM_LENGTH                  (0x20000U)
#define DLM_RAM_BASE                    (DSP_RAM_DLM_BASE)
#define DLM_RAM_LENGTH                  (0x8000U)
#define ILM_HIGH16KS_RAM_BASE           (0x20010000)
#define ILM_HIGH16KS_RAM_LENGTH         (0x10000)
#define DLM_HIGH64KS_RAM_BASE           (0x20000000)
#define DLM_HIGH64KS_RAM_LENGTH         (0x4000U)

/**
 * @brief Check if the current RAM buffer is in Auto Cache memory space. 
 * @param addr is the address of the current RAM buffer.
 * @param len is the length of the current RAM buffer.
 * 
 * @return true if in Auto Cache memory space. false otherwise.
 */
__STATIC_INLINE bool 
DMA_checkRamInAutoCache(uint32_t addr, uint32_t len)
{
    if (((addr >= ILM_RAM_BASE) && (addr < (ILM_RAM_BASE + ILM_RAM_LENGTH))) 
        || ((addr >= DLM_RAM_BASE) && (addr < (DLM_RAM_BASE + DLM_RAM_LENGTH))) 
        || ((addr >= ILM_HIGH16KS_RAM_BASE) && (addr < (ILM_HIGH16KS_RAM_BASE + ILM_HIGH16KS_RAM_LENGTH)))
        || ((addr >= DLM_HIGH64KS_RAM_BASE) && (addr < (DLM_HIGH64KS_RAM_BASE + DLM_HIGH64KS_RAM_LENGTH)))        
       )
    {
        if (((addr + len) < (ILM_RAM_BASE + ILM_RAM_LENGTH))
             || ((addr + len) < (DLM_RAM_BASE + DLM_RAM_LENGTH))
             || ((addr + len) < (ILM_HIGH16KS_RAM_BASE + ILM_HIGH16KS_RAM_LENGTH)) 
             || ((addr + len) < (DLM_HIGH64KS_RAM_BASE + DLM_HIGH64KS_RAM_LENGTH))
           )
        {
            return true;
        }
        else
        {
            ASSERT(false);  // Not a valid address and length.
        }
    }
    else
    {
        return false;
    }
}

#else 

/**
 * @brief Check if the current RAM buffer is in Auto Cache memory space. 
 * @param addr is the address of the current RAM buffer.
 * @param len is the length of the current RAM buffer.
 * 
 * @return true if in Auto Cache memory space. false otherwise.
 */
__STATIC_INLINE bool 
DMA_checkRamInAutoCache(uint32_t addr, uint32_t len)
{
    return true;
}
#endif

/**
 * @brief Setup DMA to transfer data on the specified channel.
 * @param base is Base address of the DMA channel control register
 * @param transfParams configuration parameter Refer struct #DMA_ConfigParams
 * 
 * @return None
 */
extern void
DMA_configChannel(uint32_t base, const DMA_ConfigParams *transfParams);

/**
 * @brief Clear DMA setting from previous ransfer data on the specified channel.
 * @param base is Base address of the DMA channel control register
 * @param transfParams configuration parameter Refer struct #DMA_ConfigParams
 * 
 * @return None
 */
extern void 
DMA_DeConfChannel(uint32_t base, const DMA_ConfigParams *transfParams);

/**
 * @brief config channel parameters
 *
 * @param base  the base address of the DMA channel control registers.
 * @return chParams the parameters of channel config inclue:
 *                  channel ctl, channel cfg,dmamux ccr and dmamux rgcr
 */
extern void
DmaChan_config(uint32_t base, const DmaCh_Parameters *chParams);

/**
 * @brief Configures the DMA channel
 * @param base is the base address of the DMA channel control registers.
 * @param destAddr is the destination address.
 * @param srcAddr is a source address.'
 * 
 * @return None
 */
extern void
DMA_configAddresses(uint32_t base, uint32_t destAddr, uint32_t srcAddr);

/** 
 * @brief Configures the DMA channel's burst settings.
 * @param base is the base address of the DMA channel control registers.
 * @param size is the number of words transferred per burst.
 * @param srcStep is the amount to increment or decrement the source address
 * @param destStep is the amount to increment or decrement the destination address after each word of a burst.
 * 
 * @return None.
*/
extern void DMA_configBurst(uint32_t base, uint16_t size, int16_t srcStep,
                            int16_t destStep);

/**
 * @brief Configures the DMA channel's transfer settings.
 * @param base is the base address of the DMA channel control registers.
 * @param transferSize is the number of bursts per transfer.
 * @param srcStep is the amount to increment or decrement the source address
 *                  after each burst of a transfer unless a wrap occurs.
 * @param destStep is the amount to increment or decrement the destination
 *                  address after each burst of a transfer unless a wrap occurs.
 * 
 * @return None.
 */
extern void
DMA_configTransfer(uint32_t base, uint32_t transferSize, int16_t srcStep,
                   int16_t destStep);

/**
 * @brief Configures the DMA channel's wrap settings.
 * @param base is the base address of the DMA channel control registers.
 * @param srcWrapSize is the number of bursts to be transferred before a wrap
 *                      of the source address occurs.
 * @param srcStep is the amount to increment or decrement the source address
 *                  after each burst of a transfer unless a wrap occurs.
 * @param destWrapSize is the number of bursts to be transferred before a wrap
 *                      of the destination address occurs.
 * @param destStep is the amount to increment or decrement the destination
 *                  address after each burst of a transfer unless a wrap occurs.
 * 
 * @return None.
 */
extern void
DMA_configWrap(uint32_t base, uint32_t srcWrapSize, int16_t srcStep,
               uint32_t destWrapSize, int16_t destStep);


/**
 * @brief Configures the DMA channel trigger and mode.
 * @param base is the base address of the DMA channel control registers.
 * @param trigger is the interrupt source that triggers a DMA transfer.
 * @param config is a bit field of several configuration selections.
 * 
 * @return None
 */
extern void
DMA_configMode(uint32_t base, DMAMUX_TrigId_Type trigger, uint32_t config);


#endif

#ifdef __cplusplus
}
#endif

#endif /* DEVICE_DRIVERLIB_DRV_INC_DRV_DMA_H_ */
