/*
 * Copyright (c) 2019 Nuclei Limited. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/******************************************************************************
 * \file     startup_demosoc.S
 * \brief    NMSIS Nuclei N/NX Class Core based Core Device Startup File for
 *  Nuclei Demo SoC which support Nuclei N/NX class cores
 * \version  V1.00
 * \date     17. Dec 2019
 *
 * commit history:
 * 2024/04/13, Zhao Lei, add 3.0 irq table
 * 2024/05/15, Zhao Lei, add 3.0 vector copy/remap
 * 2024/05/15, Zhao Lei, remove 3.0 shbss init
 * 2024/06/17, Zhao Lei, disable shbss copy for bootloader
 * 2024/06/18, Zhao Lei, rewrite data sections copying/ bss sections clearing
 * 2024/09/13, Zhao Lei, add low_level_init_start() low_level_init_end()
 * 2024/09/18, Zhao Lei, change low_level_init_start() low_level_init_end() attribute to .globl
 * 2024/11/19, Zhao Lei, remove FLASH_TARGET
 ******************************************************************************/

#if __riscv
#include "riscv_encoding.h"
#include "board_cfg.h"

/* If BOOT_HARTID is not defined, default value is 0 */
#ifndef BOOT_HARTID
    .equ BOOT_HARTID,    0
#endif

.macro DECLARE_INT_HANDLER  INT_HDL_NAME
#if defined(__riscv_xlen) && (__riscv_xlen == 32)
    .word \INT_HDL_NAME
#else
    .dword \INT_HDL_NAME
#endif
.endm

    .equ  RISCV_ECLIC_BASE_ADDR,            0xE1020000
    .equ  RISCV_ECLIC_CTRL_BASE_ADDR,       0xE1021000

    .section .vtable

    .weak eclic_msip_handler
    .weak eclic_mtip_handler
    .weak eclic_scia_int_handler
    .weak eclic_inter_core_int_handler
    .globl low_level_init_start
    .globl low_level_init_end
    .globl vector_base
    .type vector_base, @object
vector_base:
#ifndef VECTOR_TABLE_REMAPPED
    j _start                                                /* 0: Reserved, Jump to _start when reset for vector table not remapped cases.*/
    /* j . */
    .align LOG_REGBYTES                                     /*    Need to align 4 byte for RV32, 8 Byte for RV64 */
#else
    DECLARE_INT_HANDLER     default_intexc_handler          /* 0: Reserved, default handler for vector table remapped cases */
#endif
    DECLARE_INT_HANDLER     default_intexc_handler          /* 1: Reserved */
    DECLARE_INT_HANDLER     default_intexc_handler          /* 2: Reserved */
    DECLARE_INT_HANDLER     eclic_msip_handler              /* 3: Machine software interrupt */

    DECLARE_INT_HANDLER     default_intexc_handler          /* 4: Reserved */
    DECLARE_INT_HANDLER     default_intexc_handler          /* 5: Reserved */
    DECLARE_INT_HANDLER     default_intexc_handler          /* 6: Reserved */
    DECLARE_INT_HANDLER     eclic_mtip_handler              /* 7: Machine timer interrupt */

    DECLARE_INT_HANDLER     default_intexc_handler          /* 8: Reserved */
    DECLARE_INT_HANDLER     default_intexc_handler          /* 9: Reserved */
    DECLARE_INT_HANDLER     default_intexc_handler          /* 10: Reserved */
    DECLARE_INT_HANDLER     default_intexc_handler          /* 11: Reserved */

    DECLARE_INT_HANDLER     default_intexc_handler          /* 12: Reserved */
    DECLARE_INT_HANDLER     default_intexc_handler          /* 13: Reserved */
    DECLARE_INT_HANDLER     default_intexc_handler          /* 14: Reserved */
    DECLARE_INT_HANDLER     default_intexc_handler          /* 15: Reserved */

    DECLARE_INT_HANDLER     eclic_inter_core_int_handler    /* 16: CIDU Inter Core Interrupt */
    DECLARE_INT_HANDLER     default_intexc_handler          /* 17: Reserved */
    DECLARE_INT_HANDLER     default_intexc_handler          /* 18: Reserved */
    DECLARE_INT_HANDLER     default_intexc_handler          /* 19: Interrupt 19 */

    .rept 68-20+1
    DECLARE_INT_HANDLER     default_intexc_handler          /* 20: Interrupt 20 */
    .endr

#if (GS32F00xx & 0xFF00) == 0x1200 || (GS32F3xx & 0xFF00) == 0x2200 || (GS32F00xx & 0xFF00) == 0x3000

    .rept 258-69+1
    DECLARE_INT_HANDLER     default_intexc_handler          /* 69: Interrupt 69 */
    .endr

#endif

    .section .init

    .globl _start
    .type _start, @function

/**
 * Reset Handler called on controller reset
 */
_start:
    /* ===== Startup Stage 1 ===== */
    /* Disable Global Interrupt */
    csrc CSR_MSTATUS, MSTATUS_MIE

    /*enable nice instruction*/
    lui	a5,0x18
    csrs	mstatus,a5

    /* take bit 0-7 for hart id in a local cluster */
    csrr a0, CSR_MHARTID
    andi a0, a0, 0xFF
    /* BOOT_HARTID is configurable in Makefile via BOOT_HARTID variable */
    li a1, BOOT_HARTID

    /* If SMP_CPU_CNT is not defined,
     * assume that only 1 core is allowed to run,
     * the core hartid is defined via BOOT_HARTID.
     * other harts if run to here, just do wfi in __amp_wait
     */
#ifndef SMP_CPU_CNT
    bne a0, a1, __amp_wait
#else
    bne a0, a1, __cpu2_entry
#endif

    call low_level_init_start

    /* Initialize GP and TP */
    .option push
    .option norelax
#if defined(__USE_GP_REL16)
	la gp, __gp_rel16$
#else
    la gp, __global_pointer$
#endif
    la tp, __tls_base
    .option pop

#if defined(SMP_CPU_CNT) && (SMP_CPU_CNT > 1)
    /* Set correct sp for each cpu
     * each stack size is __STACK_SIZE
     * defined in linker script */
    la t0, __STACK_SIZE
    la sp, _sp
    csrr a0, CSR_MHARTID
    andi a0, a0, 0xFF
    li a1, 0
1:
    beq a0, a1, 2f
    sub sp, sp, t0
    addi a1, a1, 1
    j 1b
2:
#else
    /* Set correct sp for current cpu */
    la sp, _sp
#endif

    /*
     * Set the the NMI base mnvec to share
     * with mtvec by setting CSR_MMISC_CTL
     * bit 9 NMI_CAUSE_FFF to 1
     */
    li t0, MMISC_CTL_NMI_CAUSE_FFF
    csrs CSR_MMISC_CTL, t0

    /*
     * Intialize ECLIC vector interrupt
     * base address mtvt to vector_base
     */

#if defined(SMP_CPU_CNT) && (SMP_CPU_CNT > 1)   /* shared SMP vector table, remap copy by bootcore only*/
    csrr a0, CSR_MHARTID
    andi a0, a0, 0xFF
    li a1, BOOT_HARTID
    bne a0, a1, 2f
#endif

    la a0, vector_base
    la a1, __vector_remap_start
    la a2, __vector_remap_end       /* TODO: */
    beq  a0, a1, 2f                 /* No remapping is required for RAM target */
    bgeu a1, a2, 2f
1:
    /* copy vector table */
    lw t0, (a0)
    sw t0, (a1)
    addi a0, a0, 4
    addi a1, a1, 4
    bltu a1, a2, 1b
2:
    la t0, __vector_remap_start
    csrw CSR_MTVT, t0

    /*
     * Set ECLIC non-vector entry to be controlled
     * by mtvt2 CSR register.
     * Intialize ECLIC non-vector interrupt
     * base address mtvt2 to irq_entry.
     */
    la t0, irq_entry
    csrw CSR_MTVT2, t0
    csrs CSR_MTVT2, 0x1

    /*
     * Set Exception Entry MTVEC to early_exc_entry
     * Due to settings above, Exception and NMI
     * will share common entry.
     * This early_exc_entry is only used during early
     * boot stage before main
     */
    la t0, early_exc_entry
    csrw CSR_MTVEC, t0

    /* Set the interrupt processing mode to ECLIC mode */
    li t0, 0x3f
    csrc CSR_MTVEC, t0
    csrs CSR_MTVEC, 0x3

    /* ===== Startup Stage 2 ===== */

    /* Enable FPU and Vector Unit if f/d/v exist in march */
#if defined(__riscv_flen) && __riscv_flen > 0
    /* Enable FPU, and set state to initial */
    li t0, MSTATUS_FS
    csrc mstatus, t0
    li t0, MSTATUS_FS_INITIAL
    csrs mstatus, t0
#endif

#if defined(__riscv_vector)
    /* Enable Vector, and set state to initial */
    li t0, MSTATUS_VS
    csrc mstatus, t0
    li t0, MSTATUS_VS_INITIAL
    csrs mstatus, t0
#endif

    /* Enable mcycle and minstret counter */
    csrci CSR_MCOUNTINHIBIT, 0x5

#if defined(SMP_CPU_CNT) && (SMP_CPU_CNT > 1)
    csrr a0, CSR_MHARTID
    andi a0, a0, 0xFF
    li a1, BOOT_HARTID
    bne a0, a1, __skip_init
#endif

__init_common:
    /* ===== Startup Stage 3 copy data/ramfun... sections ===== */
    la a4, __copy_table_start
    la a5, __copy_table_end

1:  bgeu a4, a5, 3f
    lw a0, (a4)         /* LMA */
    lw a1, 4(a4)        /* VMA */
    lw a2, 8(a4)        /* Length */
    addi a4, a4, 12

    beq a0, a1, 1b      /* LMA == VMA ? */
    beq a2, zero, 1b    /* Length == 0 ? */
2:
    lw t0, (a0)
    sw t0, (a1)
    addi a0, a0, 4
    addi a1, a1, 4
    addi a2, a2, -1
    bne a2, zero, 2b
    j 1b
3:

    /* ===== Startup Stage 4 clear bss ===== */
    la a4, __zero_table_start
    la a5, __zero_table_end

1:  bgeu a4, a5, 3f
    lw a0, (a4)         /* start */
    lw a2, 4(a4)        /* length */
    addi a4, a4, 8

    beq a2, zero, 1b
2:
    sw zero, (a0)
    addi a0, a0, 4
    addi a2, a2, -1
    bne a2, zero, 2b
    j 1b
3:

.globl _start_premain
.type _start_premain, @function
_start_premain:
    /*
     * Call vendor defined SystemInit to
     * initialize the micro-controller system
     * SystemInit will just be called by boot cpu
     */
    call SystemInit

    /* Call global constructors */
    la a0, __libc_fini_array
    call atexit
    /* Call C/C++ constructor start up code */
    call __libc_init_array

__skip_init:
    /* Sync all harts at this function */
    call __sync_harts

    /* do pre-init steps before main */
    /* _premain_init will be called by each cpu
     * please make sure the implementation of __premain_int
     * considered this
     */
    call _premain_init

    /*
     * When all initialization steps done
     * set exception entry to correct exception
     * entry and jump to main.
     * And set the interrupt processing mode to
     * ECLIC mode
     */
    la t0, exc_entry
    csrw CSR_MTVEC, t0
    li t0, 0x3f
    csrc CSR_MTVEC, t0
    csrs CSR_MTVEC, 0x3

    /* BPU cold bringup need time, so enable BPU before enter to main */
    li t0, MMISC_CTL_BPU
    csrs CSR_MMISC_CTL, t0

    call low_level_init_end

    /* ===== Call SMP Main Function  ===== */
    /* argc = argv = 0 */
    li a0, 0
    li a1, 0
#if defined(SMP_CPU_CNT) && (SMP_CPU_CNT > 1)
    /* The weak implementation of smp_main is in this file */
    call smp_main
#else
#ifdef RTOS_RTTHREAD
    // Call entry function when using RT-Thread
    call entry
#else
    call main
#endif
#endif
    /* do post-main steps after main
     * this function will be called by each cpu */
    call _postmain_fini

__amp_wait:
1:
    wfi
    j 1b

__cpu2_entry:
1:
    /* Set the interrupt processing mode to ECLIC mode */
    li t0, 0x3f
    csrc CSR_MTVEC, t0
    csrs CSR_MTVEC, 0x3

    li t0, RISCV_ECLIC_BASE_ADDR
    sb x0, 0xB(t0)           /* MTH */

    li t1, 0x6
    sb t1, 0(t0)             /* CFG */

    li t0, RISCV_ECLIC_CTRL_BASE_ADDR
    sb x0, 0x40(t0)         /* clear INTIP */

    li t1, 0x2
    sb t1, 0x42(t0)         /* INTATTR rising, none vector*/

    li t1, 1
    sb t1, 0x41(t0)         /* enable INTIE */

    wfi

    sb x0, 0x41(t0)         /* disable INTIE */
    sb x0, 0x40(t0)         /* clear INTIP */

#if DSP_CPU2_RUN_IN_FLASH != 0		/* jump to flash*/
    li a0, (0x8000000 + DSP_CPU2_BOOT_ADDR_FLASH_OFFSET)
    jalr a0
#else
    li a0, 0x10800000	/* jump to ilm*/
    jalr a0
#endif

    j 1b

#if defined(SMP_CPU_CNT) && (SMP_CPU_CNT > 1)
.weak main_cpu2
.type main_cpu2, @function
1:
    wfi
    j 1b

/*
 * You can re-implement smp_main function in your code
 * to do smp boot process and handle multi harts
 */
.weak smp_main
.type smp_main, @function
smp_main:
    addi sp, sp, -2*REGBYTES
    STORE ra, 0*REGBYTES(sp)
    /* only boot hart goto main, other harts do wfi */
    csrr t0, CSR_MHARTID
    andi t0, t0, 0xFF
    li t1, BOOT_HARTID
    beq t0, t1, 2f
1:
    call main_cpu2
2:
#ifdef RTOS_RTTHREAD
    // Call entry function when using RT-Thread
    call entry
#else
    call main
#endif
    LOAD ra, 0*REGBYTES(sp)
    addi sp, sp, 2*REGBYTES
    ret
#endif

/* Early boot exception entry before main */
.align 6
.global early_exc_entry
.type early_exc_entry, @function
early_exc_entry:
    wfi
    j early_exc_entry

#endif /* __riscv */
