/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * enc.h
 *
 *  Created on: 2024 Oct 22
 *      Author: hpec_
 */

#ifndef CLB_ENC_H_
#define CLB_ENC_H_

#ifdef __cplusplus
extern "C"{
#endif

#include "gs32_version.h"
#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
//#include "inc/hw_clb_enc.h"
#include "inc/hw_enc.h"
#include "debug.h"

#if IS_GS32F00xx(0x30)
#define TA_FORMAT_MODULE		1
#define BISS_MODULE				0
#define ENDAT_MODULE			0
#endif

typedef enum
{
	RESERVED	= 	0X0000U,
	SPIA_SI		=	0X0001U,
	SPIB_SI		=	0X0002U,
	CLBINPUTXBAR0=	0X0004U,
	CLBINPUTXBAR1=	0X0008U,
	CLBINPUTXBAR2=	0X0010U,
	CLBINPUTXBAR3=	0X0020U,
	CLBINPUTXBAR4=	0X0040U,
	CLBINPUTXBAR5=	0X0080U,
	CLBINPUTXBAR6=	0X0100U,
	CLBINPUTXBAR7=	0X0200U,
	CLBINPUTXBAR8=	0X0400U,
	CLBINPUTXBAR9=	0x0800U,
	CLBINPUTXBAR10=	0x1000U,
	CLBINPUTXBAR11=	0x2000U,
	CLBINPUTXBAR12=	0x4000U,
	CLBINPUTXBAR13=	0x8000U
}CLB_ENC_INSEL;

typedef enum
{
	HARDWARE_DETECT_TRIGGER = 0,
	SOFTWARE_DELAY_TRIGGER	= 1
}ATFORMAT_RECEIVE_MODE;

typedef enum
{
	NONE = 0U,
	SPIA_SIMO = 1U,
	SPIB_SIMO = 2U,
	CLB_INPUT_XBAR1 = 8U,
	CLB_INPUT_XBAR2 = 9U,
	CLB_INPUT_XBAR3 = 10U,
	CLB_INPUT_XBAR4 = 11U,
	CLB_INPUT_XBAR5 = 12U,
	CLB_INPUT_XBAR6 = 13U,
	CLB_INPUT_XBAR7 = 14U,
	CLB_INPUT_XBAR8 = 15U,
}QEPDIV_INPUT_SEL;

#ifdef DEBUG
__STATIC_INLINE bool
ENC_isBaseValid(uint32_t base)
{
#if TA_FORMAT_MODULE || BISS_MODULE || ENDAT_MODULE
    return(
           (base == CLB_ENC1_BASE) ||
           (base == CLB_ENC2_BASE)
          );
#else
    return false;
#endif
}
#endif

#if TA_FORMAT_MODULE
#define CLB_ENC_TA_IDLE			0U
#define CLB_ENC_TA_T2_STATE		1U
#define CLB_ENC_TA_TX_STATE 	2U
#define CLB_ENC_TA_T4_STATE 	3U
#define CLB_ENC_TA_T3_STATE		4U
#define CLB_ENC_TA_WAIT_STATE	5U
#define CLB_ENC_TA_RX_STATE		6U
#define CLB_ENC_TA_T9_STATE		7U

/*
 * @brief Encode Format select
 */
typedef enum
{
	TFORMAT = 0U,
	AFORMAT = 1U
}TA_FORMATMODE;

/*
 * @brief start communicate with encoder
 */
__STATIC_INLINE void
ENC_TA_startOperation(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));

	HWREG(base + ENC_O_TA_FORMAT_CTRL0)|= ENC_TA_FORMAT_CTRL0_EN_M;
}

/*
 * @brief stop communicate with encoder
 */
__STATIC_INLINE void
ENC_TA_stopOperation(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));

	HWREG(base + ENC_O_TA_FORMAT_CTRL0)&= (~ENC_TA_FORMAT_CTRL0_EN_M);
}

/*
 * @brief Set TX and RXs frame numbers
 */
__STATIC_INLINE void
EN_TA_setTxRxFrameNum(uint32_t base,uint16_t txnum,uint16_t rxnum)
{
	ASSERT(ENC_isBaseValid(base));
	ASSERT(((txnum<=15)&&(txnum>=1)));
	ASSERT(((rxnum<=15)&&(rxnum>=1)));

	HWREG(base + ENC_O_TA_FORMAT_CTRL1) = \
			(HWREG(base + ENC_O_TA_FORMAT_CTRL1) & (~ENC_TA_FORMAT_CTRL1_TX_FRM_NUM_M))|\
			((txnum-1)<<ENC_TA_FORMAT_CTRL1_TX_FRM_NUM_S);

	HWREG(base + ENC_O_TA_FORMAT_CTRL1) = \
			(HWREG(base + ENC_O_TA_FORMAT_CTRL1) & (~ENC_TA_FORMAT_CTRL1_RX_FRM_NUM_M))|\
			((rxnum-1)<<ENC_TA_FORMAT_CTRL1_RX_FRM_NUM_S);
}

/*
 * @brief Set the length of one TX and RX frame
 */
__STATIC_INLINE void
EN_TA_setTxRxFrameLen(uint32_t base,uint16_t txlen,uint16_t rxlen)
{
	ASSERT(ENC_isBaseValid(base));
	ASSERT(((txlen<=255)&&(txlen>=1)));
	ASSERT(((rxlen<=255)&&(rxlen>=1)));

	HWREG(base + ENC_O_TA_FORMAT_CTRL1) = \
			(HWREG(base + ENC_O_TA_FORMAT_CTRL1) & (~ENC_TA_FORMAT_CTRL1_TX_FRM_LEN_M))|\
			((txlen-1)<<ENC_TA_FORMAT_CTRL1_TX_FRM_LEN_S);

	HWREG(base + ENC_O_TA_FORMAT_CTRL1) = \
			(HWREG(base + ENC_O_TA_FORMAT_CTRL1) & (~ENC_TA_FORMAT_CTRL1_RX_FRM_LEN_M))|\
			(rxlen-1);
}

__STATIC_INLINE uint16_t
ENC_TA_getWaitRxTime(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));

	return((uint16_t)((HWREG(base + ENC_O_TA_FORMAT_STATUS)&\
			ENC_TA_FORMAT_STATUS_WAIT_TIME_RPT_M)>>\
			ENC_TA_FORMAT_STATUS_WAIT_TIME_RPT_S));
}

__STATIC_INLINE uint16_t
ENC_TA_getTAState(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));

	return((uint16_t)(HWREG(base + ENC_O_TA_FORMAT_STATUS)&\
			ENC_TA_FORMAT_STATUS_STATE_M));
}

/*
 * @brief initialize CLB_ENC
 *
 * @param base		indicates that the CLB_ENC instance base address.
 * @param format	T-format or A-format.
 * @param clbclkHz	CLB_ENC Clock frequency (DEVICE_AHBCLK_FREQ).
 * @param bitRate	Encoder communicate frequency.
 * @param input		Data entry selection.
 */
extern void ENC_TA_setConfig(uint32_t base,\
							 TA_FORMATMODE format,\
							 uint32_t clbclkHz,\
							 uint32_t bitRate,\
							 CLB_ENC_INSEL input);

/*
 * @brief Select the Hardware Recognition Falling Edge
 * 		  to start flipping the receive clock.
 *
 *@param base indicates that the CLB_ENC instance base address.
 *
 * @note Hardware is used by default.
 */
extern void ENC_TA_reviceHardwareMode(uint32_t base);


/*
 * @brief Set the software delay for n cycles and
 * 		  start flipping the receive clock
 *
 * @param	base		indicates that the CLB_ENC instance base address.
 * @param	waittime	Wait time,The time between when the command is sent
 * 						and when the data is received from the encoder.
 *
 * @note Do not use it unless necessary,
 * 	and use the function of ENC_TA_getWaitRxTime()
 * 	to obtain the exact wait time
 * 	if you do not need to use it.
 */
extern void ENC_TA_reviceSoftwareMode(uint32_t base,uint32_t waittime);

/*
 * @brief Set Tx clock delay time
 *
 * @param base			indicates that the CLB_ENC instance base address.
 * @param startdelay 	The delay time between when TXEN is enabled and
 * 						when the transmit clock starts to flip.
 * @param framedelay	When sending multiple frames,
 * 						the delay time between frames.
 * @param enddelay		The send clock stops flipping the delay time
 * 						between the TXEN incapacitation.
 * @note framedelay must be set in the A-FORMAT,
 * and must keep zero in the T-FORMAT.
 */
extern void ENC_TA_setTxClockDelay(uint32_t base,\
							 uint16_t startdelay,
							 uint16_t framedelay,
							 uint16_t enddelay);
#endif

#if BISS_MODULE

#define CLB_ENC_BISS_IDLE		0U
#define CLB_ENC_BISS_WAITACK	1U
#define CLB_ENC_BISS_WAITSTART	2U
#define CLB_ENC_BISS_RX_STATE	3U
#define CLB_ENC_BISS_STOP_STATE	4U
#define CLB_ENC_BISS_TR_STATE	5U
#define CLB_ENC_BISS_TCAL_STATE	6U
#define CLB_ENC_BISS_TDLY_STATE	7U

/*
 * @brief Encode Format select
 *
 * BISS-C or BISS
 */
typedef enum
{
	BISS_C = 0U,
	SSI = 1U
}BISS_SSI_MODE;

typedef enum
{
	WAIT_TIME = 0U,
	WAIT_RISING = 0x1000000U,
}CLB_BISS_TIMEOUT_MODE;

typedef enum
{
	CDM_LOW = 0U,
	CDM_HIGH = 0x8U
}BISS_C_CDM;

/*
 * @brief start communicate with encoder
 */
__STATIC_INLINE void
ENC_BISS_startOperation(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));

	HWREG(base + ENC_O_BISS_CTRL)|= ENC_BISS_CTRL_EN_M;
}

/*
 * @brief stop communicate with encoder
 */
__STATIC_INLINE void
ENC_BISS_stopOperation(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));

	HWREG(base + ENC_O_BISS_CTRL)&= (~ENC_BISS_CTRL_EN_M);
}

__STATIC_INLINE uint16_t
ENC_BISS_getWaitRxTime(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));

	return((uint16_t)((HWREG(base + ENC_O_BISS_STATUS)&\
			ENC_BISS_WAIT_TIME_RPT_M)>>\
			ENC_BISS_WAIT_TIME_RPT_S));
}

__STATIC_INLINE uint16_t
ENC_BISS_getTAState(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));

	return((uint16_t)(HWREG(base + ENC_O_BISS_STATUS)&\
			ENC_BISS_STATE_M));
}

__STATIC_INLINE void
ENC_BISS_setCDM(uint32_t base,BISS_C_CDM cdm)
{
	ASSERT(ENC_isBaseValid(base));

	HWREG(base + ENC_O_BISS_CTRL) = \
			(HWREG(base + ENC_O_BISS_CTRL) & (~ENC_BISS_CTRL_CDM_BIT_M))| cdm;
}

/*
 * @brief Set TX and RXs bit numbers
 */
__STATIC_INLINE void
EN_BISS_setRxandSpiBitNum(uint32_t base,uint16_t rxnum,uint16_t spinum)
{
	ASSERT(ENC_isBaseValid(base));
	ASSERT(((rxnum<=255)&&(rxnum>=1)));

	HWREG(base + ENC_O_BISS_SPI_CRG) =
			(HWREG(base + ENC_O_BISS_SPI_CRG) & (~ENC_BISS_SPI_CRG_RX_BIT_NUM_M))|\
			((rxnum-1)<<ENC_BISS_SPI_CRG_RX_BIT_NUM_S);
	HWREG(base + ENC_O_BISS_SPI_CRG) =
			(HWREG(base + ENC_O_BISS_SPI_CRG) & (~ENC_BISS_SPI_CRG_SPI_BIT_NUM_M))|\
			(spinum-1);
}

/*
 * @brief Set software as TimeoutMode
 */
__STATIC_INLINE void
ENC_BISS_setSoftTimeoutMode(uint32_t base,uint32_t time)
{
	ASSERT(ENC_isBaseValid(base));
	ASSERT((time <= 255)&&(time >= 0));

	HWREG(base + ENC_O_BISS_CTRL) &=  (~ENC_BISS_CTRL_TIMEOUT_MODE_M);
	HWREG(base + ENC_O_BISS_TIMING1) = \
			(HWREG(base + ENC_O_BISS_TIMING1) & ENC_BISS_TR_CFG_M)|(time-1);
}

__STATIC_INLINE void
ENC_BISS_setHardTimeoutMode(uint32_t base,uint32_t time)
{
	ASSERT(ENC_isBaseValid(base));
	ASSERT((time <= 255)&&(time >= 0));

	HWREG(base + ENC_O_BISS_CTRL) |=  ENC_BISS_CTRL_TIMEOUT_MODE_M;
	HWREG(base + ENC_O_BISS_TIMING1) = \
			(HWREG(base + ENC_O_BISS_TIMING1) & ENC_BISS_TR_CFG_M)|(time-1);
}

/*
 * @brief
 */
extern void ENC_BISS_setConfig(uint32_t base,\
							   BISS_SSI_MODE format,
							   uint32_t clbclkHz,\
							   uint32_t bitRate,\
							   CLB_ENC_INSEL input);

/*
 * @brief Select the Hardware Recognition Falling Edge
 * 		  to start flipping the receive clock.
 *
 *@param base indicates that the CLB_ENC instance base address.
 *
 * @note Hardware is used by default.
 */
extern void ENC_BISS_reviceHardwareMode(uint32_t base);


/*
 * @brief Set the software delay for n cycles and
 * 		  start flipping the receive clock
 *
 * @param	base		indicates that the CLB_ENC instance base address.
 * @param	waittime	Wait time,The time between when the command is sent
 * 						and when the data is received from the encoder.
 *
 * @note Do not use it unless necessary,
 * 	and use the function of ENC_BISS_getWaitRxTime()
 * 	to obtain the exact wait time
 * 	if you do not need to use it.
 */
extern void ENC_BISS_reviceSoftwareMode(uint32_t base,uint32_t waittime);



// get the current counter

extern uint32_t getPM_abs2qep_curcnt (uint32_t base, uint32_t qclk_cnt);
#endif

#if ENDAT_MODULE

//***********************************     ENDAT_START     ***********************************//
__STATIC_INLINE void ENC_ENDAT_Enable(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_CTRL0) |=	 ENC_ENDAT_CTRL0_ENDAT_EN_M;
}

__STATIC_INLINE void ENC_ENDAT_Disable(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_CTRL0) &=	(~ ENC_ENDAT_CTRL0_ENDAT_EN_M);
}

__STATIC_INLINE void ENC_ENDAT_selRcvMode(uint32_t base,ATFORMAT_RECEIVE_MODE mode)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_CTRL0) =  (HWREG(base + ENC_O_ENDAT_CTRL0) & \
			(~ ENC_ENDAT_CTRL0_RCV_MODE_M)) | (mode<< ENC_ENDAT_CTRL0_RCV_MODE_S);
}

__STATIC_INLINE void ENC_ENDAT_selRxdIn(uint32_t base,ATFORMAT_RECEIVE_MODE signal)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_CTRL0) =  (HWREG(base + ENC_O_ENDAT_CTRL0) & \
			(~ ENC_ENDAT_CTRL0_RXD_IN_SEL_M)) | (signal <<  ENC_ENDAT_CTRL0_RXD_IN_SEL_S);
}

__STATIC_INLINE void ENC_ENDAT_setBaudrate(uint32_t base,uint32_t baudrate)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_CTRL0) =  (HWREG(base + ENC_O_ENDAT_CTRL0) & \
			(~ ENC_ENDAT_CTRL0_BAUDRATE_M)) | (baudrate<< ENC_ENDAT_CTRL0_BAUDRATE_S);
}

__STATIC_INLINE void ENC_ENDAT_setTxBitNum(uint32_t base,uint32_t bitnum)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_CTRL1) =  (HWREG(base + ENC_O_ENDAT_CTRL1) & \
			(~ ENC_ENDAT_CTRL1_ENDAT_TX_BIT_NUM_M)) | (bitnum<< ENC_ENDAT_CTRL1_ENDAT_TX_BIT_NUM_S);
}

__STATIC_INLINE void ENC_ENDAT_setRxBitNum(uint32_t base,uint32_t bitnum)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_CTRL1) =  (HWREG(base + ENC_O_ENDAT_CTRL1) & \
			(~ ENC_ENDAT_CTRL1_ENDAT_RX_BIT_NUM_M)) | (bitnum<< ENC_ENDAT_CTRL1_ENDAT_RX_BIT_NUM_S);
}

__STATIC_INLINE void ENC_ENDAT_setAddBitNum(uint32_t base,uint32_t bitnum)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_CTRL1) =  (HWREG(base + ENC_O_ENDAT_CTRL1) & \
			(~ ENC_ENDAT_CTRL1_ENDAT_ADD_BIT_NUM_M)) | (bitnum<< ENC_ENDAT_CTRL1_ENDAT_ADD_BIT_NUM_S);
}

__STATIC_INLINE void ENC_ENDAT_setSpiTxBitNum(uint32_t base,uint32_t bitnum)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_SPI_CFG) =  (HWREG(base + ENC_O_ENDAT_SPI_CFG) & \
			(~ ENC_ENDAT_SPI_CFG_ENDAT_SPI_TX_BIT_NUM_M)) | (bitnum<< ENC_ENDAT_SPI_CFG_ENDAT_SPI_TX_BIT_NUM_S);
}

__STATIC_INLINE void ENC_ENDAT_setSpiRxBitNum(uint32_t base,uint32_t bitnum)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_SPI_CFG) =  (HWREG(base + ENC_O_ENDAT_SPI_CFG) & \
			(~ ENC_ENDAT_SPI_CFG_ENDAT_SPI_RX_BIT_NUM_M)) | (bitnum<< ENC_ENDAT_SPI_CFG_ENDAT_SPI_RX_BIT_NUM_S);
}

__STATIC_INLINE void ENC_ENDAT_setSpiAddBitNum(uint32_t base,uint32_t bitnum)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_SPI_CFG) =  (HWREG(base + ENC_O_ENDAT_SPI_CFG) & \
			(~ ENC_ENDAT_SPI_CFG_ENDAT_SPI_ADD_BIT_NUM_M)) | (bitnum<< ENC_ENDAT_SPI_CFG_ENDAT_SPI_ADD_BIT_NUM_S);
}

__STATIC_INLINE void ENC_ENDAT_setTDelay(uint32_t base,uint32_t tdelay)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_TIMING0) =  (HWREG(base + ENC_O_ENDAT_TIMING0) & \
			(~ ENC_ENDAT_TIMING0_TDLY_CFG_M)) | (tdelay<< ENC_ENDAT_TIMING0_TDLY_CFG_S);
}

__STATIC_INLINE void ENC_ENDAT_setTst(uint32_t base,uint32_t tst)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_TIMING0) =  (HWREG(base + ENC_O_ENDAT_TIMING0) & \
			(~ ENC_ENDAT_TIMING0_ST_CFG_M)) | (tst<< ENC_ENDAT_TIMING0_ST_CFG_S);
}

__STATIC_INLINE void ENC_ENDAT_setTr(uint32_t base,uint32_t tr)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_TIMING1) =  (HWREG(base + ENC_O_ENDAT_TIMING1) & \
			(~ ENC_ENDAT_TIMING1_TR_CFG_M)) | (tr<< ENC_ENDAT_TIMING1_TR_CFG_S);
}

__STATIC_INLINE void ENC_ENDAT_setTAdd(uint32_t base,uint32_t tadd)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENDAT_TIMING1) =  (HWREG(base + ENC_O_ENDAT_TIMING1) & \
			(~ ENC_ENDAT_TIMING1_TADD_CFG_M)) | (tadd<< ENC_ENDAT_TIMING1_TADD_CFG_S);
}

__STATIC_INLINE uint16_t ENC_ENDAT_getTimeofWaitRpt(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	return ((HWREG(base + ENC_O_ENDAT_STATUS) & \
			ENC_ENDAT_WAIT_TIME_RPT_M)>> ENC_ENDAT_WAIT_TIME_RPT_S);
}

__STATIC_INLINE uint16_t ENC_ENDAT_getBissState(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	return (HWREG(base + ENC_O_ENDAT_STATUS) &  ENC_ENDAT_STATE_M);
}
#endif
//***********************************     ENDAT_END     ***********************************//

__STATIC_INLINE void ENC_enableInterrupt(uint32_t base,uint32_t intFlags)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENC_INTR_MASK) &=  (~intFlags);
}

__STATIC_INLINE void ENC_disableInterrupt(uint32_t base,uint32_t intFlags)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENC_INTR_MASK) |=  intFlags;
}

__STATIC_INLINE uint16_t ENC_getIntrState(uint32_t base)
{
	ASSERT(ENC_isBaseValid(base));
	return (HWREG(base + ENC_O_ENC_INTR));
}

__STATIC_INLINE void ENC_clearIntrState(uint32_t base,uint32_t intFlags)
{
	ASSERT(ENC_isBaseValid(base));
	HWREG(base + ENC_O_ENC_INTR_RAW) |= intFlags;
}
//#endif //0x30



__STATIC_INLINE uint16_t __flip16(uint16_t data)
{
	uint32_t i;
	uint16_t tmp=0;
	for(i=0;i<16;i++)
	{
		tmp=((data>>i)&0x01)|tmp;
	    if(i<15)
	    	tmp=tmp<<1;
	}
	return (tmp);
}

__STATIC_INLINE uint8_t __flip8(uint8_t data)
{
	uint32_t i;
	uint8_t tmp=0;
	for(i=0;i<8;i++)
	{
		tmp=((data>>i)&0x01)|tmp;
	    if(i<7)
	    	tmp=tmp<<1;
	}
	return (tmp);
}

#if IS_GS32F00xx(0x30)
extern uint16_t PM_tformat_setupCommand(uint16_t dataID, uint16_t eepromAddr, uint16_t eepromData, uint16_t crc);
extern uint16_t PM_tformat_receiveData(uint16_t dataID);
extern void PM_tformat_setupPeriph(void);
extern void PM_tformat_setFreq(uint32_t Freq_us);
extern void PM_tformat_startOperation(void);
extern void PM_tformat_generateCRCTable(uint16_t nBits, uint16_t polynomial, uint16_t *pTable);
extern uint16_t PM_tformat_getCRC(uint16_t input_crc_accum, uint16_t nBitsData,  uint16_t nBitsPoly, uint16_t * msg, uint16_t *crc_table, uint16_t rxLen);
extern void PM_tformat_setupCommandReadoutOrReset(uint16_t commandID0_1_2_3_7_8_C,uint16_t tformatRXClocks,uint16_t tformatRXFields,uint16_t tformatTXClocks,uint16_t tformatFIFOLevel);
extern void PM_tformat_setupCommandReadEEPROM(uint16_t eepromAddr);
extern void PM_tformat_setupCommandWriteEEPROM(uint16_t eepromAddr,uint16_t eepromData);
extern void PM_tformat_receiveDataID0_1_7_8_C(void);
extern void PM_tformat_receiveDataID2(void);
extern void PM_tformat_receiveDataID3(void);
extern uint16_t PM_tformat_receiveDataIDD(void);
extern uint16_t PM_tformat_receiveDataID6(void);

#endif


#ifdef __cplusplus
}
#endif

#endif  /* CLB_ENC_H_ */
