/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    epwm_de.h
*   @brief   
*
*   commit history
*   20240722, lihao, Separate the DIODE function from the epwm.h.
*/
#ifndef DEVICE_DRIVERLIB_EPWM_DE_H_
#define DEVICE_DRIVERLIB_EPWM_DE_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include <stdbool.h>
#include <stdint.h>
#include "gs32_version.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "inc/hw_epwm.h"
#include "inc/hw_sysctl_ahb.h"
#include "cpu.h"
#include "debug.h"

#if (EPWM_VERSION  == 30)
/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

//
// Diode Emulation Logic
//
//*****************************************************************************
//
//! Values that can be passed to EPWM_setDiodeEmulationMode() as the
//! \e mode parameter.
//
//*****************************************************************************
typedef enum{
    //! Diode Emulation mode is Cycle by Cycle
    EPWM_DIODE_EMULATION_CBC = 0,
    //! Diode Emulation mode is One Shot
    EPWM_DIODE_EMULATION_OST = 1
}EPWM_DiodeEmulationMode;


//*****************************************************************************
//
//! Values that can be passed to EPWM_configureDiodeEmulationTripLowSources()
//! as the \e source parameter.
//
//*****************************************************************************
typedef enum{
    //! Trip source is CMPSSTRIPL1 signal
    EPWM_DE_TRIPL_SRC_CMPSSTRIPL1 = 0x00U,
    //! Trip source is CMPSSTRIPL2 signal
    EPWM_DE_TRIPL_SRC_CMPSSTRIPL2 = 0x01U,
    //! Trip source is CMPSSTRIPL3 signal
    EPWM_DE_TRIPL_SRC_CMPSSTRIPL3 = 0x02U,
    //! Trip source is CMPSSTRIPL4 signal
    EPWM_DE_TRIPL_SRC_CMPSSTRIPL4 = 0x03U,
    //! Trip source is CMPSSTRIPL5 signal
    EPWM_DE_TRIPL_SRC_CMPSSTRIPL5 = 0x04U,
    //! Trip source is CMPSSTRIPL6 signal
    EPWM_DE_TRIPL_SRC_CMPSSTRIPL6 = 0x05U,
    //! Trip source is CMPSSTRIPL7 signal
    EPWM_DE_TRIPL_SRC_CMPSSTRIPL7 = 0x06U,
    //! Trip source is CMPSSTRIPL8 signal
    EPWM_DE_TRIPL_SRC_CMPSSTRIPL8 = 0x07U,
    //! Trip source is CMPSSTRIPL9 signal
    EPWM_DE_TRIPL_SRC_CMPSSTRIPL9 = 0x08U,
    //! Trip source is CMPSSTRIPL10 signal
    EPWM_DE_TRIPL_SRC_CMPSSTRIPL10 = 0x09U,
    //! Trip source is CMPSSTRIPL11 signal
    EPWM_DE_TRIPL_SRC_CMPSSTRIPL11 = 0x0AU,
    //! Trip source is INPUTXBAR out1 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT1 = 0x10U,
    //! Trip source is INPUTXBAR out2 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT2 = 0x11U,
    //! Trip source is INPUTXBAR out3 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT3 = 0x12U,
    //! Trip source is INPUTXBAR out4 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT4 = 0x13U,
    //! Trip source is INPUTXBAR out5 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT5 = 0x14U,
    //! Trip source is INPUTXBAR out6 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT6 = 0x15U,
    //! Trip source is INPUTXBAR out7 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT7 = 0x16U,
    //! Trip source is INPUTXBAR out8 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT8 = 0x17U,
    //! Trip source is INPUTXBAR out9 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT9 = 0x18U,
    //! Trip source is INPUTXBAR out10 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT10 = 0x19U,
    //! Trip source is INPUTXBAR out11 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT11 = 0x1AU,
    //! Trip source is INPUTXBAR out12 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT12 = 0x1BU,
    //! Trip source is INPUTXBAR out13 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT13 = 0x1CU,
    //! Trip source is INPUTXBAR out14 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT14 = 0x1DU,
    //! Trip source is INPUTXBAR out15 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT15 = 0x1EU,
    //! Trip source is INPUTXBAR out16 signal
    EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT16 = 0x1FU,
    //! Trip source is CLBINPUTXBAR out1 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT1 = 0x20U,
    //! Trip source is CLBINPUTXBAR out2 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT2 = 0x21U,
    //! Trip source is CLBINPUTXBAR out3 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT3 = 0x22U,
    //! Trip source is CLBINPUTXBAR out4 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT4 = 0x23U,
    //! Trip source is CLBINPUTXBAR out5 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT5 = 0x24U,
    //! Trip source is CLBINPUTXBAR out6 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT6 = 0x25U,
    //! Trip source is CLBINPUTXBAR out7 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT7 = 0x26U,
    //! Trip source is CLBINPUTXBAR out8 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT8 = 0x27U,
    //! Trip source is CLBINPUTXBAR out9 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT9 = 0x28U,
    //! Trip source is CLBINPUTXBAR out10 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT10 = 0x29U,
    //! Trip source is CLBINPUTXBAR out11 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT11 = 0x2AU,
    //! Trip source is CLBINPUTXBAR out12 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT12 = 0x2BU,
    //! Trip source is CLBINPUTXBAR out13 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT13 = 0x2CU,
    //! Trip source is CLBINPUTXBAR out14 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT14 = 0x2DU,
    //! Trip source is CLBINPUTXBAR out15 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT15 = 0x2EU,
    //! Trip source is CLBINPUTXBAR out16 signal
    EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT16 = 0x2FU,
}EPWM_DiodeEmulationTripLSource;

//*****************************************************************************
//
//! Values that can be passed to EPWM_configureDiodeEmulationTripHighSources()
//! as the \e source parameter.
//
//*****************************************************************************
typedef enum{
    //! Trip source is CMPSSTRIPH1 signal
    EPWM_DE_TRIPH_SRC_CMPSSTRIPH1 = 0x00U,
    //! Trip source is CMPSSTRIPH2 signal
    EPWM_DE_TRIPH_SRC_CMPSSTRIPH2 = 0x01U,
    //! Trip source is CMPSSTRIPH3 signal
    EPWM_DE_TRIPH_SRC_CMPSSTRIPH3 = 0x02U,
    //! Trip source is CMPSSTRIPH4 signal
    EPWM_DE_TRIPH_SRC_CMPSSTRIPH4 = 0x03U,
    //! Trip source is CMPSSTRIPH5 signal
    EPWM_DE_TRIPH_SRC_CMPSSTRIPH5 = 0x04U,
    //! Trip source is CMPSSTRIPH6 signal
    EPWM_DE_TRIPH_SRC_CMPSSTRIPH6 = 0x05U,
    //! Trip source is CMPSSTRIPH7 signal
    EPWM_DE_TRIPH_SRC_CMPSSTRIPH7 = 0x06U,
    //! Trip source is CMPSSTRIPH8 signal
    EPWM_DE_TRIPH_SRC_CMPSSTRIPH8 = 0x07U,
    //! Trip source is CMPSSTRIPH9 signal
    EPWM_DE_TRIPH_SRC_CMPSSTRIPH9 = 0x08U,
    //! Trip source is CMPSSTRIPH10 signal
    EPWM_DE_TRIPH_SRC_CMPSSTRIPH10 = 0x09U,
    //! Trip source is CMPSSTRIPH11 signal
    EPWM_DE_TRIPH_SRC_CMPSSTRIPH11 = 0x0AU,
    //! Trip source is INPUTXBAR out1 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT1 = 0x10U,
    //! Trip source is INPUTXBAR out2 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT2 = 0x11U,
    //! Trip source is INPUTXBAR out3 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT3 = 0x12U,
    //! Trip source is INPUTXBAR out4 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT4 = 0x13U,
    //! Trip source is INPUTXBAR out5 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT5 = 0x14U,
    //! Trip source is INPUTXBAR out6 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT6 = 0x15U,
    //! Trip source is INPUTXBAR out7 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT7 = 0x16U,
    //! Trip source is INPUTXBAR out8 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT8 = 0x17U,
    //! Trip source is INPUTXBAR out9 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT9 = 0x18U,
    //! Trip source is INPUTXBAR out10 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT10 = 0x19U,
    //! Trip source is INPUTXBAR out11 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT11 = 0x1AU,
    //! Trip source is INPUTXBAR out12 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT12 = 0x1BU,
    //! Trip source is INPUTXBAR out13 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT13 = 0x1CU,
    //! Trip source is INPUTXBAR out14 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT14 = 0x1DU,
    //! Trip source is INPUTXBAR out15 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT15 = 0x1EU,
    //! Trip source is INPUTXBAR out16 signal
    EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT16 = 0x1FU,
    //! Trip source is CLBINPUTXBAR out1 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT1 = 0x20U,
    //! Trip source is CLBINPUTXBAR out2 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT2 = 0x21U,
    //! Trip source is CLBINPUTXBAR out3 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT3 = 0x22U,
    //! Trip source is CLBINPUTXBAR out4 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT4 = 0x23U,
    //! Trip source is CLBINPUTXBAR out5 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT5 = 0x24U,
    //! Trip source is CLBINPUTXBAR out6 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT6 = 0x25U,
    //! Trip source is CLBINPUTXBAR out7 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT7 = 0x26U,
    //! Trip source is CLBINPUTXBAR out8 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT8 = 0x27U,
    //! Trip source is CLBINPUTXBAR out9 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT9 = 0x28U,
    //! Trip source is CLBINPUTXBAR out10 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT10 = 0x29U,
    //! Trip source is CLBINPUTXBAR out11 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT11 = 0x2AU,
    //! Trip source is CLBINPUTXBAR out12 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT12 = 0x2BU,
    //! Trip source is CLBINPUTXBAR out13 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT13 = 0x2CU,
    //! Trip source is CLBINPUTXBAR out14 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT14 = 0x2DU,
    //! Trip source is CLBINPUTXBAR out15 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT15 = 0x2EU,
    //! Trip source is CLBINPUTXBAR out16 signal
    EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT16 = 0x2FU,
}EPWM_DiodeEmulationTripHSource;

//*****************************************************************************
//
//! Values that can be passed to EPWM_configureDiodeEmulationTripSources()
//!  as the \e tripLorH parameter.
//
//*****************************************************************************
//! Diode emulation Trip L
#define EPWM_DE_SRC_TRIPL    0x1U
//! Diode emulation Trip H
#define EPWM_DE_SRC_TRIPH    0x0U

//*****************************************************************************
//
//! Values that can be passed to EPWM_selectDiodeEmulationPWMsignal()
//! as the \e signal parameter.
//
//*****************************************************************************
typedef enum{
    //! Synchronized version of TRIPH or TRIPL signal
    EPWM_DE_SYNC_TRIPHorL = 0U,
    //! Synchronized and inverted version of TRIPH or TRIPL signal
    EPWM_DE_SYNC_INV_TRIPHorL = 1U,
    //! A constant low signal
    EPWM_DE_LOW = 2U,
    //! A constant high signal
    EPWM_DE_HIGH = 3U
}EPWM_DiodeEmulationSignal;

//*****************************************************************************
//
//! Values that can be passed to EPWM_selectDiodeEmulationPWMsignal()
//! and EPWM_selectDiodeEmulationTripSignal() as the \e signal parameter.
//
//*****************************************************************************
//! Diode emulation channel A
#define EPWM_DE_CHANNEL_A    0x0U
//! Diode emulation channel B
#define EPWM_DE_CHANNEL_B    0x1U

//*****************************************************************************
//
//! Values that can be passed to EPWM_setDiodeEmulationMonitorModeStep()
//! as the \e direction parameter.
//
//*****************************************************************************
//! Diode emulation count up step size
#define EPWM_DE_COUNT_UP      0x0U
//! Diode emulation count down step size
#define EPWM_DE_COUNT_DOWN    0x1U

//*****************************************************************************
//
// Prototypes for the API.
//
//*****************************************************************************

//
// Diode Emulation logic related APIs
//
/**
 *  @brief Enable ePWM diode emulation mode.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function enables the ePWM diode emulation mode.
*/
static inline void
EPWM_enableDiodeEmulationMode(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE;

    //
    // Enable DE function
    //
    HWREGH(registerOffset + EPWM_O_DECTL) |= EPWM_DECTL_ENABLE;

}

/**
 *  @brief Disable ePWM diode emulation mode.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function disables the ePWM diode emulation mode.
*/
static inline void
EPWM_disableDiodeEmulationMode(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE;

    //
    // Disable DE function
    //
    HWREGH(registerOffset + EPWM_O_DECTL) &= ~EPWM_DECTL_ENABLE;

}

/**
 *  @brief Selects the mode for the Diode emulation mode
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] mode is load mode selection for Diode emulation mode
 *  @remarks
 *  This function gives us choice to select the mode for Diode emulation mode
 *  Valid values for input variables are:
 *  - mode
 *     EPWM_DIODE_EMULATION_CBC         - Diode Emulation mode is Cycle by Cycle
 *     EPWM_DIODE_EMULATION_OST         - Diode Emulation mode is One Shot
*/
static inline void
EPWM_setDiodeEmulationMode(uint32_t base, EPWM_DiodeEmulationMode mode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE;

    if(mode == EPWM_DIODE_EMULATION_CBC)
    {
        //
        // Enable DE CBC mode
        //
        HWREGH(registerOffset + EPWM_O_DECTL) &= ~EPWM_DECTL_MODE;
    }
    else if(mode == EPWM_DIODE_EMULATION_OST)
    {
        //
        // Enable DE OST mode
        //
        HWREGH(registerOffset + EPWM_O_DECTL) |= EPWM_DECTL_MODE;
    }
    else
    {
        //
        // For correct inputs, this function is not expected to
        // execute.
        //
//        ESTOP0;
    }
}

/**
 *  @brief Set ePWM diode emulation re-entry delay.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] delay is the re-entry delay in terms of number of PWMSYNCOUT events.
 *  @remarks
 *  This function determines the blocking window after DEACTIVE flag is cleared
 *  in which setting of DEACTIVE flag is prevented from being set.
 *
 *  Valid values for input variables are:
 *  - delay: Max value of delay is 256.
*/
static inline void
EPWM_setDiodeEmulationReentryDelay(uint32_t base, uint32_t delay)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(delay <= 256U);
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE;

    HWREGH(registerOffset + EPWM_O_DECTL) =
            (HWREGH(registerOffset + EPWM_O_DECTL) & ~EPWM_DECTL_REENTRYDLY_M) |
            (delay << EPWM_DECTL_REENTRYDLY_S);
}

/**
 *  @brief Set ePWM diode emulation trip source for TripL.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] source is the trip source
 *  @remarks
 *  This function is used to configure the comparator whose
 *  trip sources will be used.
 *  Valid values for \e source are:
 *     EPWM_DE_TRIPL_SRC_INPUTXBAR_OUT[0-16]     - INPUTXBAR out[0-16] signal
 *     EPWM_DE_TRIPL_SRC_CLBINPUTXBAR_OUT[0-16]  - CLBINPUTXBAR out[0-16] signal
 *     EPWM_DE_TRIPL_SRC_CMPSSTRIPL[1-11]        - CMPSSTRIPL[1-11] signal
*/
static inline void
EPWM_configureDiodeEmulationTripLowSources(uint32_t base,
                                        EPWM_DiodeEmulationTripLSource source)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE + EPWM_O_DECOMPSEL;

    HWREGH(registerOffset) =
            (HWREGH(registerOffset) & ~EPWM_DECOMPSEL_TRIPL_M) |
            ((uint32_t)source << EPWM_DECOMPSEL_TRIPL_S);

}

/**
 *  @brief Set ePWM diode emulation trip source for TripH.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] source is the trip source
 *  @remarks
 *  This function is used to configure the comparator whose
 *  trip sources will be used.
 *  Valid values for \e source are:
 *     EPWM_DE_TRIPH_SRC_INPUTXBAR_OUT[0-16]     - INPUTXBAR out[0-16] signal
 *     EPWM_DE_TRIPH_SRC_CLBINPUTXBAR_OUT[0-16]  - CLBINPUTXBAR out[0-16] signal
 *     EPWM_DE_TRIPH_SRC_CMPSSTRIPH[1-11]        - CMPSSTRIPH[1-11] signal
*/
static inline void
EPWM_configureDiodeEmulationTripHighSources(uint32_t base,
                                        EPWM_DiodeEmulationTripHSource source)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE + EPWM_O_DECOMPSEL;

    HWREG(registerOffset) =
            (HWREG(registerOffset) & ~EPWM_DECOMPSEL_TRIPH_M) |
            ((uint32_t)source << EPWM_DECOMPSEL_TRIPH_S);
}

/**
 *  @brief Selects ePWM diode emulation pwm signal.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] channel is the epwm output channel
 *  @param [in] signal is the signal selected by the TRIPSEL[A/B]
 *  @remarks
 *  This function is used to select the pwm signal in DE mode.
 *  Valid values for channel are:
 *     EPWM_DE_CHANNEL_A     - Diode emulation channel A
 *     EPWM_DE_CHANNEL_B     - Diode emulation channel B
 *
 *  Valid values for signal are:
 *     EPWM_DE_SYNC_TRIPHorL     - synchronized version of
 *                                 TRIPH or TRIPL signal
 *     EPWM_DE_SYNC_INV_TRIPHorL - synchronized and inverted version
 *                                 of TRIPH or TRIPL signal
 *     EPWM_DE_LOW               - a constant low signal
 *     EPWM_DE_HIGH              - a constant high signal
*/
static inline void
EPWM_selectDiodeEmulationPWMsignal(uint32_t base, uint32_t channel,
                                   EPWM_DiodeEmulationSignal emulationSignal)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE + EPWM_O_DEACTCTL;

    if(channel == EPWM_DE_CHANNEL_A)
    {
        HWREGH(registerOffset) =
                (HWREGH(registerOffset) & ~EPWM_DEACTCTL_PWMA_M) |
                ((uint32_t)emulationSignal << EPWM_DEACTCTL_PWMA_S);
    }
    else
    {
        HWREGH(registerOffset) =
                (HWREGH(registerOffset) & ~(3U<<EPWM_DEACTCTL_PWMB_S)) |
                ((uint32_t)emulationSignal << EPWM_DEACTCTL_PWMB_S);
    }
}

/**
 *  @brief Selects between TRIPH and TRIPL for an epwm channel in DE mode.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] channel is the epwm output channel
 *  @param [in] signal is the signal selected by the TRIPSEL[A/B]
 *  @remarks
 *  This function is used to select between TRIPH and TRIPL
 *  for a channel in DE mode.
 *  Valid values for channel are:
 *     EPWM_DE_CHANNEL_A     - Diode emulation channel A
 *     EPWM_DE_CHANNEL_B     - Diode emulation channel B
 *
 *  Valid values for signal are:
 *      0 for TRIPH
 *      1 for TRIPL
*/
static inline void
EPWM_selectDiodeEmulationTripSignal(uint32_t base, uint32_t channel,
                                    uint32_t emulationSignal)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE + EPWM_O_DEACTCTL;

    if(channel == EPWM_DE_CHANNEL_A)
    {
        HWREGH(registerOffset) =
                (HWREGH(registerOffset) & ~EPWM_DEACTCTL_TRIPSELA) |
                (emulationSignal << 2U);
    }
    else
    {
        HWREGH(registerOffset) =
                (HWREGH(registerOffset) & ~EPWM_DEACTCTL_TRIPSELB) |
                (emulationSignal << 6U);
    }
}

/**
 *  @brief PWMTRIP does not bypass the diode emulation logic
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function is used to go through DEL.
*/
static inline void
EPWM_nobypassDiodeEmulationLogic(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE;

    HWREG(registerOffset + EPWM_O_DEACTCTL) &= ~EPWM_DEACTCTL_TRIPENABLE;

}

/**
 *  @brief PWMTRIP bypasses the diode emulation PWM generation logic
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function is used to bypass DEL.
*/
static inline void
EPWM_bypassDiodeEmulationLogic(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE;

    HWREG(registerOffset + EPWM_O_DEACTCTL) |= EPWM_DEACTCTL_TRIPENABLE;

}

/**
 *  @brief Forces DEACTIVE flag to 1
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function is used to force DEACTIVE flag to 1.
*/
static inline void
EPWM_forceDiodeEmulationActive(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE;

    HWREGH(registerOffset + EPWM_O_DEFRC) |= EPWM_DEFRC_DEACTIVE;

}

/**
 *  @brief Enables DE monitor control mode
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function is used to enable the DE monitor counter function.
*/
static inline void
EPWM_enableDiodeEmulationMonitorModeControl(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE;

    HWREGH(registerOffset + EPWM_O_DEMONCTL) |= EPWM_DEMONCTL_ENABLE;

}

/**
 *  @brief Disables DE monitor control mode
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function is used to disable the DE monitor counter function.
*/
static inline void
EPWM_disableDiodeEmulationMonitorModeControl(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE;

    HWREGH(registerOffset + EPWM_O_DEMONCTL) &= ~EPWM_DEMONCTL_ENABLE;
}

/**
 *  @brief Sets the DE monitor counter step size
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] direction is the direction of the monitor counter counting either
 *  @param [in] stepsize is the amount by which the counter increments or decrements at each step.
 *  @remarks
 *  This function is used to set the DE monitor counter step size.
 *  Valid values for \e direction are:
 *     EPWM_DE_COUNT_UP     - Diode emulation count up step size
 *     EPWM_DE_COUNT_DOWN   - Diode emulation count down step size.
 *
 *     Max value of stepsize is 256.
*/
static inline void
EPWM_setDiodeEmulationMonitorModeStep(uint32_t base, uint32_t direction,
                                   uint32_t stepsize)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(stepsize <= 256U);
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE + EPWM_O_DEMONSTEP;

    if(direction == EPWM_DE_COUNT_UP)
    {
        HWREGH(registerOffset) =
                (HWREGH(registerOffset) & ~EPWM_DEMONSTEP_INCSTEP_M) |
                (stepsize << EPWM_DEMONSTEP_INCSTEP_S);
    }
    else if(direction == EPWM_DE_COUNT_DOWN)
    {
        HWREG(registerOffset) =
                (HWREG(registerOffset) & ~EPWM_DEMONSTEP_DECSTEP_M) |
                (stepsize << EPWM_DEMONSTEP_DECSTEP_S);
    }
    else
    {
        //
        // For correct inputs, this function is not expected to
        // execute.
        //
//        ESTOP0;
    }
}

/**
 *  @brief Sets the DE monitor counter threshold value.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] threshold is the max value till which the monitor counter can count.
 *  @remarks
 *  This function is used to set the DE monitor counter threshold.
 *
 *  Max value of threshold is 2^16.
*/
static inline void
EPWM_setDiodeEmulationMonitorCounterThreshold(uint32_t base,
                                                uint32_t threshold)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(threshold <= 0xFFFFUL);
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE + EPWM_O_DEMONTHRES;

    HWREGH(registerOffset) =
            (HWREGH(registerOffset) & ~EPWM_DEMONTHRES_THRESHOLD_M) |
            (threshold << EPWM_DEMONTHRES_THRESHOLD_S);
}

/**
 *  @brief Clears DE mode active flag
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function is used to clear the DE mode active flag
*/
static inline void
EPWM_clearDiodeEmulationActiveFlag(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_DE;

    HWREGH(registerOffset + EPWM_O_DECLR) |= EPWM_DECLR_DEACTIVE;
}
#endif

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************


#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_EPWM_DE_H_ */
