/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    epwm_xcmp.h
*   @brief   
*
*   commit history
*   20240722, lihao, Separate the XCMP function from the epwm.h.
*/
#ifndef DEVICE_DRIVERLIB_EPWM_XCMP_H_
#define DEVICE_DRIVERLIB_EPWM_XCMP_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include <stdbool.h>
#include <stdint.h>
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "inc/hw_epwm.h"
#include "inc/hw_sysctl_ahb.h"
#include "epwm.h"
#include "debug.h"


#if (EPWM_VERSION  == 30)
/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
//*****************************************************************************
//
// Address offsets from EPWM_BASE to XCMP register memory maps
//
//*****************************************************************************
//#define EPWM_O_XCMP                0x0400U



//
// XCMP
//
//*****************************************************************************
//
//! Values that can be passed to EPWM_setXCMPActionQualifierAction()
//! and EPWM_setXCMPShadowRepeatBufxCount()
//
//*****************************************************************************
//!< XCMP set = Active
#define EPWM_XCMP_ACTIVE     0x0U
//!< XCMP set = Shadow 1
#define EPWM_XCMP_SHADOW1    0x1U
//!< XCMP set = Shadow 2
#define EPWM_XCMP_SHADOW2    0x2U
//!< XCMP set = Shadow 3
#define EPWM_XCMP_SHADOW3    0x3U

//*****************************************************************************
//
//! Values that can be passed to EPWM_setXCMPRegValue() as the
//! \e xcmpReg parameter.
//
//*****************************************************************************
typedef enum
{
    //! XCMP1_ACTIVE
    EPWM_XCMP1_ACTIVE = 0x0U,
    //! XCMP2_ACTIVE
    EPWM_XCMP2_ACTIVE = 0x4U,
    //! XCMP3_ACTIVE
    EPWM_XCMP3_ACTIVE = 0x8U,
    //! XCMP4_ACTIVE
    EPWM_XCMP4_ACTIVE = 0xCU,
    //! XCMP5_ACTIVE
    EPWM_XCMP5_ACTIVE = 0x10U,
    //! XCMP6_ACTIVE
    EPWM_XCMP6_ACTIVE = 0x14U,
    //! XCMP7_ACTIVE
    EPWM_XCMP7_ACTIVE = 0x18U,
    //! XCMP8_ACTIVE
    EPWM_XCMP8_ACTIVE = 0x1CU,
    //! XTBPRD_ACTIVE
    EPWM_XTBPRD_ACTIVE = 0x20U,

    //! XCMP1_SHADOW1
    EPWM_XCMP1_SHADOW1 = 0x40U,
    //! XCMP2_SHADOW1
    EPWM_XCMP2_SHADOW1 = 0x44U,
    //! XCMP3_SHADOW1
    EPWM_XCMP3_SHADOW1 = 0x48U,
    //! XCMP4_SHADOW1
    EPWM_XCMP4_SHADOW1 = 0x4CU,
    //! XCMP5_SHADOW1
    EPWM_XCMP5_SHADOW1 = 0x50U,
    //! XCMP6_SHADOW1
    EPWM_XCMP6_SHADOW1 = 0x54U,
    //! XCMP7_SHADOW1
    EPWM_XCMP7_SHADOW1 = 0x58U,
    //! XCMP8_SHADOW1
    EPWM_XCMP8_SHADOW1 = 0x5CU,
    //! XTBPRD_SHADOW1
    EPWM_XTBPRD_SHADOW1 = 0x60U,

    //! XCMP1_SHADOW2
    EPWM_XCMP1_SHADOW2 = 0x80U,
    //! XCMP2_SHADOW2
    EPWM_XCMP2_SHADOW2 = 0x84U,
    //! XCMP3_SHADOW2
    EPWM_XCMP3_SHADOW2 = 0x88U,
    //! XCMP4_SHADOW2
    EPWM_XCMP4_SHADOW2 = 0x8CU,
    //! XCMP5_SHADOW2
    EPWM_XCMP5_SHADOW2 = 0x90U,
    //! XCMP6_SHADOW2
    EPWM_XCMP6_SHADOW2 = 0x94U,
    //! XCMP7_SHADOW2
    EPWM_XCMP7_SHADOW2 = 0x98U,
    //! XCMP8_SHADOW2
    EPWM_XCMP8_SHADOW2 = 0x9CU,
    //! XTBPRD_SHADOW2
    EPWM_XTBPRD_SHADOW2 = 0xA0U,

    //! XCMP1_SHADOW3
    EPWM_XCMP1_SHADOW3 = 0xC0U,
    //! XCMP2_SHADOW3
    EPWM_XCMP2_SHADOW3 = 0xC4U,
    //! XCMP3_SHADOW3
    EPWM_XCMP3_SHADOW3 = 0xC8U,
    //! XCMP4_SHADOW3
    EPWM_XCMP4_SHADOW3 = 0xCCU,
    //! XCMP5_SHADOW3
    EPWM_XCMP5_SHADOW3 = 0xD0U,
    //! XCMP6_SHADOW3
    EPWM_XCMP6_SHADOW3 = 0xD4U,
    //! XCMP7_SHADOW3
    EPWM_XCMP7_SHADOW3 = 0xD8U,
    //! XCMP8_SHADOW3
    EPWM_XCMP8_SHADOW3 = 0xDCU,
    //! XTBPRD_SHADOW3
    EPWM_XTBPRD_SHADOW3 = 0xE0U,

}EPWM_XCMPReg;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setCMPShadowRegValue() as the
//! \e cmpReg parameter.
//
//*****************************************************************************
typedef enum
{
    //! CMPC_SHADOW1
    EPWM_CMPC_SHADOW1 = 0x0U,
    //! CMPD_SHADOW1
    EPWM_CMPD_SHADOW1 = 0x4U,
    //! CMPC_SHADOW2
    EPWM_CMPC_SHADOW2 = 0x40U,
    //! CMPD_SHADOW2
    EPWM_CMPD_SHADOW2 = 0x44U,
    //! CMPC_SHADOW3
    EPWM_CMPC_SHADOW3 = 0x80U,
    //! CMPD_SHADOW3
    EPWM_CMPD_SHADOW3 = 0x84U

}EPWM_XCompareReg;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setXMINMAXRegValue() as the
//! \e xminmaxReg parameter.
//
//*****************************************************************************
typedef enum
{
    //! XMAX_ACTIVE
    EPWM_XMAX_ACTIVE = 0x0U,
    //! XMIN_ACTIVE
    EPWM_XMIN_ACTIVE = 0x2U,
    //! XMAX_SHADOW1
    EPWM_XMAX_SHADOW1 = 0x40U,
    //! XMIN_SHADOW1
    EPWM_XMIN_SHADOW1 = 0x42U,
    //! XMAX_SHADOW2
    EPWM_XMAX_SHADOW2 = 0x80U,
    //! XMIN_SHADOW2
    EPWM_XMIN_SHADOW2 = 0x82U,
    //! XMAX_SHADOW3
    EPWM_XMAX_SHADOW3 = 0xC0U,
    //! XMIN_SHADOW3
    EPWM_XMIN_SHADOW3 = 0xC2U

}EPWM_XMinMaxReg;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setXCMPActionQualifierAction()
//! as the \e event parameter.
//
//*****************************************************************************
typedef enum
{
    //! Time base counter equals XCMP1
    EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP1 = 0,
    //! Time base counter equals XCMP2
    EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP2 = 2,
    //! Time base counter equals XCMP3
    EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP3 = 4,
    //! Time base counter equals XCMP4
    EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP4 = 6,
    //! Time base counter equals XCMP5
    EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP5 = 8,
    //! Time base counter equals XCMP6
    EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP6 = 10,
    //! Time base counter equals XCMP7
    EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP7 = 12,
    //! Time base counter equals XCMP8
    EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP8 = 14
} EPWM_XCMPActionQualifierOutputEvent;

//*****************************************************************************
//
//! Values that can be passed to EPWM_allocAXCMP()
//! as the \e alloctype parameter.
//
//*****************************************************************************
typedef enum
{
    //! Allocate 0 XCMP registers to CMPA
    EPWM_XCMP_NONE_CMPA = 0,
    //! Allocate XCMP1 register to CMPA
    EPWM_XCMP_1_CMPA = 1,
    //! Allocate XCMP1 - XCMP2 registers to CMPA
    EPWM_XCMP_2_CMPA = 2,
    //! Allocate XCMP1 - XCMP3 registers to CMPA
    EPWM_XCMP_3_CMPA = 3,
    //! Allocate XCMP1 - XCMP4 registers to CMPA
    EPWM_XCMP_4_CMPA = 4,
    //! Allocate XCMP1 - XCMP5 registers to CMPA
    EPWM_XCMP_5_CMPA = 5,
    //! Allocate XCMP1 - XCMP6 registers to CMPA
    EPWM_XCMP_6_CMPA = 6,
    //! Allocate XCMP1 - XCMP7 registers to CMPA
    EPWM_XCMP_7_CMPA = 7,
    //! Allocate XCMP1 - XCMP8 registers to CMPA
    EPWM_XCMP_8_CMPA = 8
}EPWM_XCMP_ALLOC_CMPA;

//*****************************************************************************
//
//! Values that can be passed to EPWM_allocBXCMP()
//! as the \e alloctype parameter.
//
//*****************************************************************************
typedef enum
{
    //! Allocate XCMP5 register to CMPB
    EPWM_XCMP_5_CMPB = 5,
    //! Allocate XCMP5 - XCMP6 registers to CMPB
    EPWM_XCMP_6_CMPB = 6,
    //! Allocate XCMP5 - XCMP7 registers to CMPB
    EPWM_XCMP_7_CMPB = 7,
    //! Allocate XCMP5 - XCMP8 registers to CMPB
    EPWM_XCMP_8_CMPB = 8
}EPWM_XCMP_ALLOC_CMPB;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setXCMPLoadMode() as the
//! \e mode parameter.
//
//*****************************************************************************
typedef enum
{
    //! Load mode is LOADONCE
    EPWM_XCMP_XLOADCTL_LOADMODE_LOADONCE = 0,
    //! Load mode is LOADMULTIPLE
    EPWM_XCMP_XLOADCTL_LOADMODE_LOADMULTIPLE = 1
}EPWM_XCMPXloadCtlLoadMode;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setXCMPShadowLevel() as the
//! \e level parameter.
//
//*****************************************************************************
typedef enum
{
    //!  Only Active register is available
    EPWM_XCMP_XLOADCTL_SHDWLEVEL_0 = 0,
    //! SHDW1 and Active registers are available
    EPWM_XCMP_XLOADCTL_SHDWLEVEL_1 = 1,
    //! SHDW2, SHDW1 and Active registers are available
    EPWM_XCMP_XLOADCTL_SHDWLEVEL_2 = 2,
    //! SHDW3, SHDW2, SHDW1 and Active registers are available
    EPWM_XCMP_XLOADCTL_SHDWLEVEL_3 = 3
}EPWM_XCMP_XLOADCTL_SHDWLEVEL;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setXCMPShadowBufPtrLoadOnce() as the
//! \e ptr parameter.
//
//*****************************************************************************
typedef enum
{
    //! No Shadow buffer is in use
    EPWM_XCMP_XLOADCTL_SHDWBUFPTR_NULL  = 0,
    //! Shadow buffer 1 is in use
    EPWM_XCMP_XLOADCTL_SHDWBUFPTR_ONE   = 1,
    //! Shadow buffer 2 is in use
    EPWM_XCMP_XLOADCTL_SHDWBUFPTR_TWO   = 2,
    //! Shadow buffer 3 is in use
    EPWM_XCMP_XLOADCTL_SHDWBUFPTR_THREE = 3
}EPWM_XCMP_XLOADCTL_SHDWBUFPTR;

//*****************************************************************************
//
// Prototypes for the API.
//
//*****************************************************************************

//
// XCMP related APIs
//

/**
 *  @brief Enable ePWM XCMP mode.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function enables the ePWM XCMP mode.
*/
static inline void
EPWM_enableXCMPMode(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP;

    //
    // Enable XCMP compare register operation
    //
    HWREGH(registerOffset + EPWM_O_XCMPCTL1) |= EPWM_XCMPCTL1_XCMPEN;
}

/**
 *  @brief Disable ePWM XCMP mode.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function disables the ePWM XCMP mode.
*/
static inline void
EPWM_disableXCMPMode(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP;

    //
    // Disable XCMP compare register operation
    //
    HWREGH(registerOffset + EPWM_O_XCMPCTL1) &= ~EPWM_XCMPCTL1_XCMPEN;
}

/**
 *  @brief Enable ePWM XCMP Split.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function splits 8 XCMP registers between CMPA and CMPB equally.
*/
static inline void
EPWM_enableSplitXCMP(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP;

    HWREGH(registerOffset + EPWM_O_XCMPCTL1) |= EPWM_XCMPCTL1_XCMPSPLIT;
}

/**
 *  @brief Disable ePWM XCMP Split.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function allocates 8 XCMP registers to CMPA.
*/
static inline void
EPWM_disableSplitXCMP(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP;

    HWREGH(registerOffset + EPWM_O_XCMPCTL1) &= ~EPWM_XCMPCTL1_XCMPSPLIT;
}

/**
 *  @brief Allocates XCMP registers to CMPA.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] alloctype is range of XCMP registers to be allocated to CMPA
 *  @remarks
 *  This function gives us choice to allocate range of XCMP registers to CMPA
 *  Valid values for alloctype are:
 *     EPWM_XCMP_NONE_CMPA      - Allocate 0 XCMP registers to CMPA
 *     EPWM_XCMP_1_CMPA         - Allocate XCMP1 register to CMPA
 *     EPWM_XCMP_2_CMPA         - Allocate XCMP1- XCMP2 registers to CMPA
 *     EPWM_XCMP_3_CMPA         - Allocate XCMP1- XCMP3 register to CMPA
 *     EPWM_XCMP_4_CMPA         - Allocate XCMP1- XCMP4 registers to CMPA
 *     EPWM_XCMP_5_CMPA         - Allocate XCMP1- XCMP5 register to CMPA
 *     EPWM_XCMP_6_CMPA         - Allocate XCMP1- XCMP6 registers to CMPA
 *     EPWM_XCMP_7_CMPA         - Allocate XCMP1- XCMP7 register to CMPA
 *     EPWM_XCMP_8_CMPA         - Allocate XCMP1- XCMP8 register to CMPA
*/
static inline void
EPWM_allocAXCMP(uint32_t base, EPWM_XCMP_ALLOC_CMPA alloctype)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP + EPWM_O_XCMPCTL1;

    HWREGH(registerOffset ) =
            (HWREGH(registerOffset) & ~EPWM_XCMPCTL1_XCMPA_ALLOC_M) |
            ((uint32_t)alloctype << EPWM_XCMPCTL1_XCMPA_ALLOC_S);
}

/**
 *  @brief Allocates XCMP registers to CMPB.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] alloctype is range of XCMP registers to be allocated to CMPB
 *  @remarks
 *  This function gives us choice to allocate range of XCMP registers to CMPB
 *  Valid values for alloctype are:
 *     EPWM_XCMP_1_CMPB         - Allocate XCMP5 register to CMPB
 *     EPWM_XCMP_2_CMPB         - Allocate XCMP5- XCMP6 registers to CMPB
 *     EPWM_XCMP_3_CMPB         - Allocate XCMP5- XCMP7 register to CMPB
 *     EPWM_XCMP_4_CMPB         - Allocate XCMP5- XCMP8 registers to CMPB
*/
static inline void
EPWM_allocBXCMP(uint32_t base, EPWM_XCMP_ALLOC_CMPB alloctype)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP + EPWM_O_XCMPCTL1;

    HWREGH(registerOffset) =
            (HWREGH(registerOffset) & ~EPWM_XCMPCTL1_XCMPB_ALLOC_M) |
            ((uint32_t)alloctype << EPWM_XCMPCTL1_XCMPB_ALLOC_S);
}

/**
 *  @brief Writes values to XCMP registers
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] xcmpReg is the XCMP register offset
 *  @param [in] xcmpvalue is the value to be written to XCMP register
 *  @remarks
 *  This function writes xcmpvalue to XCMP registers.
 *  Valid values for xcmpReg are:
 *     EPWM_XCMP[1-8]_[ACTIVE/SHADOW1/SHADOW2/SHADOW3]
 *                           -XCMP[1-8]_[ACTIVE/SHADOW1/SHADOW2/SHADOW3]
 *     EPWM_XTBPRD_ACTIVE                               -XTBPRD_ACTIVE
 *     EPWM_XTBPRD_SHADOW1                              -XTBPRD_SHADOW1
 *     EPWM_XTBPRD_SHADOW2                              -XTBPRD_SHADOW2
 *     EPWM_XTBPRD_SHADOW3                              -XTBPRD_SHADOW3
*/
static inline void
EPWM_setXCMPRegValue(uint32_t base, EPWM_XCMPReg xcmpReg,
                            uint32_t xcmpvalue)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP +
                     EPWM_O_XCMP1_ACTIVE + (uint32_t)xcmpReg;

    //
    // Write to the XCMP registers.
    //
    HWREGH(registerOffset + 2U) = xcmpvalue;
}

/**
 *  @brief Writes values to CMPx Shadow registers
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] cmpReg is the CMP register offset
 *  @param [in] cmpvalue is the value to be written to CMPC/D shadow registers
 *  @remarks
 *  This function writes cmpvalue to CMPC/D shadow registers.
 *  Valid values for cmpReg are:
 *     EPWM_CMPC_SHADOW1                             -CMPC_SHADOW1
 *     EPWM_CMPD_SHADOW1                             -CMPD_SHADOW1
 *     EPWM_CMPC_SHADOW2                             -CMPC_SHADOW2
 *     EPWM_CMPD_SHADOW2                             -CMPD_SHADOW2
 *     EPWM_CMPC_SHADOW3                             -CMPC_SHADOW3
 *     EPWM_CMPD_SHADOW3                             -CMPD_SHADOW3
*/
static inline void
EPWM_setCMPShadowRegValue(uint32_t base, EPWM_XCompareReg cmpReg,
                            uint32_t cmpvalue)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP + EPWM_O_CMPC_SHDW1 + (uint32_t)cmpReg;

    //
    // Write to the CMPC/D Shadow registers.
    //
    HWREGH(registerOffset) = cmpvalue;
}

/**
 *  @brief Writes values to XMINMAX registers
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] xminmaxReg is the XCMP register offset
 *  @param [in] xcmpvalue is the value to be written to XCMP registers
 *  @remarks
 *  This function writes xcmpvalue to XCMP registers.
 *  Valid values for xminmaxReg are:
 *     EPWM_XMIN_ACTIVE                              -XMIN_ACTIVE
 *     EPWM_XMAX_ACTIVE                              -XMAX_ACTIVE
 *     EPWM_XMIN_SHADOW1                             -XMIN_SHADOW1
 *     EPWM_XMAX_SHADOW1                             -XMAX_SHADOW1
 *     EPWM_XMIN_SHADOW2                             -XMIN_SHADOW2
 *     EPWM_XMAX_SHADOW2                             -XMAX_SHADOW2
 *     EPWM_XMIN_SHADOW3                             -XMIN_SHADOW3
 *     EPWM_XMAX_SHADOW3                             -XMAX_SHADOW3
*/
static inline void
EPWM_setXMINMAXRegValue(uint32_t base, EPWM_XMinMaxReg xminmaxReg,
                            uint32_t xcmpvalue)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP +
                     EPWM_O_XMINMAX_ACTIVE + (uint32_t)xminmaxReg;

    //
    // Write to the XMINMAX register.
    //
    HWREGH(registerOffset) = xcmpvalue;
}

/**
 *  @brief  Set up Action qualifier outputs based on XAQ registers
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] shadowset is the shadow set number or active set.
 *  @param [in] epwmOutput is the ePWM pin type.
 *  @param [in] output is the Action Qualifier output.
 *  @param [in] event is the event that causes a change in output.
 *  @remarks
 *  This function sets up the Action Qualifier output on ePWM A or ePWMB,
 *  depending on the value of epwmOutput, to a value specified by outPut based
 *  on the input events - specified by event.
 *  The following are valid values for the parameters.
 *    - epwmOutput
 *        - EPWM_AQ_OUTPUT_A          - ePWMxA output
 *        - EPWM_AQ_OUTPUT_B          - ePWMxB output
 *    -shadowset
 *        - EPWM_XCMP_ACTIVE          - XCMP set is Active
 *        - EPWM_XCMP_SHADOW1         - XCMP set is Shadow 1
 *        - EPWM_XCMP_SHADOW2         - XCMP set is Shadow 2
 *        - EPWM_XCMP_SHADOW3         - XCMP set is Shadow 3
 *    - output
 *        - EPWM_AQ_OUTPUT_NO_CHANGE  - No change in the output pins
 *        - EPWM_AQ_OUTPUT_LOW        - Set output pins to low
 *        - EPWM_AQ_OUTPUT_HIGH       - Set output pins to High
 *        - EPWM_AQ_OUTPUT_TOGGLE     - Toggle the output pins
 *    - event
 *     EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP1        -Time base counter equals XCMP1
 *     EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP2        -Time base counter equals XCMP2
 *     EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP3        -Time base counter equals XCMP3
 *     EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP4        -Time base counter equals XCMP4
 *     EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP5        -Time base counter equals XCMP5
 *     EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP6        -Time base counter equals XCMP6
 *     EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP7        -Time base counter equals XCMP7
 *     EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP8        -Time base counter equals XCMP8
*/
static inline void
EPWM_setXCMPActionQualifierAction(uint32_t base, uint32_t shadowset,
		                      EPWM_ActionQualifierOutputModule epwmOutput,
                              EPWM_ActionQualifierOutput output,
                              EPWM_XCMPActionQualifierOutputEvent event)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;

    //
    // Get the register offset
    //

    if(shadowset == EPWM_XCMP_ACTIVE)
    {
        registerOffset = EPWM_O_XCMP + EPWM_O_XAQCTLA_ACTIVE +
                         ((uint32_t)epwmOutput / 2U);

        HWREGH(base + registerOffset) =
                (HWREGH(base + registerOffset) & ~(3UL << (uint32_t)event)) |
                ((uint32_t)output << (uint32_t)event);
    }
    else if(shadowset == EPWM_XCMP_SHADOW1)
    {
        registerOffset = EPWM_O_XCMP + EPWM_O_XAQCTLA_SHDW1 +
                         ((uint32_t)epwmOutput / 2U);

        HWREGH(base + registerOffset) =
                (HWREGH(base + registerOffset) & ~(3UL << (uint32_t)event)) |
                ((uint32_t)output << (uint32_t)event);
    }
    else if(shadowset == EPWM_XCMP_SHADOW2)
    {
        registerOffset = EPWM_O_XCMP + EPWM_O_XAQCTLA_SHDW2 +
                         ((uint32_t)epwmOutput / 2U);

        HWREGH(base + registerOffset) =
                (HWREGH(base + registerOffset) & ~(3UL << (uint32_t)event)) |
                ((uint32_t)output << (uint32_t)event);
    }
    else if(shadowset == EPWM_XCMP_SHADOW3)
    {
        registerOffset = EPWM_O_XCMP + EPWM_O_XAQCTLA_SHDW3 +
                         ((uint32_t)epwmOutput / 2U);

        HWREGH(base + registerOffset) =
                (HWREGH(base + registerOffset) & ~(3UL << (uint32_t)event)) |
                ((uint32_t)output << (uint32_t)event);
    }
    else
    {
        //
        // For correct inputs, this function is not expected to
        // execute.
        //
//        ESTOP0;
    }
}

/**
 *  @brief Enables ePWM XCMP reload event.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function enables the ePWM XCMP reload event.
*/
static inline void
EPWM_enableXLoad(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP;

    HWREGH(registerOffset + EPWM_O_XLOAD) |= EPWM_XLOAD_STARTLD;
}

/**
 *  @brief Disables ePWM XCMP reload event.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function disables the ePWM XCMP reload event.
*/
static inline void
EPWM_disableXLoad(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP;

    HWREGH(registerOffset + EPWM_O_XLOAD) &= ~EPWM_XLOAD_STARTLD;
}

/**
 *  @brief Forces register loading from shadow buffers.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function is used for software force loading of the events in
 *  global load mode.
*/
static inline void
EPWM_forceXLoad(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP;

    HWREGH(registerOffset + EPWM_O_XLOAD) |= EPWM_XLOAD_FRCLD;
}

/**
 *  @brief Selects the mode for the XCMP Shadow registers
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] mode is load mode selection for XCMP Shadow registers
 *  @remarks
 *  This function gives us choice to select the mode for XCMP shadow registers
 *  Valid values for mode are:
 *     EPWM_XCMP_XLOADCTL_LOADMODE_LOADONCE         - Load mode is LOADONCE
 *     EPWM_XCMP_XLOADCTL_LOADMODE_LOADMULTIPLE     - Load mode is LOADMULTIPLE
*/
static inline void
EPWM_setXCMPLoadMode(uint32_t base, EPWM_XCMPXloadCtlLoadMode mode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP;

    if(mode == EPWM_XCMP_XLOADCTL_LOADMODE_LOADONCE)
    {
        HWREGH(registerOffset + EPWM_O_XLOADCTL) &= ~EPWM_XLOADCTL_LOADMODE;
    }
    else if(mode == EPWM_XCMP_XLOADCTL_LOADMODE_LOADMULTIPLE)
    {
        HWREGH(registerOffset + EPWM_O_XLOADCTL) |= EPWM_XLOADCTL_LOADMODE;
    }
    else
    {
        //
        // For correct inputs, this function is not expected to
        // execute.
        //
//        ESTOP0;
    }
}

/**
 *  @brief Selects the shadow register level allocation
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] level is shadow register level allocation options
 *  @remarks
 *  This function gives us choice to select the XCMP shadow registers level
 *  Valid values for alloctype are:
 *     EPWM_XCMP_XLOADCTL_SHDWLEVEL_0 - Only Active reg is available
 *     EPWM_XCMP_XLOADCTL_SHDWLEVEL_1 - SHDW1 and Active regs
 *     EPWM_XCMP_XLOADCTL_SHDWLEVEL_2 - SHDW2, SHDW1 and Active regs
 *     EPWM_XCMP_XLOADCTL_SHDWLEVEL_3 - SHDW3, SHDW2, SHDW1 and Active regs
*/
static inline void
EPWM_setXCMPShadowLevel(uint32_t base, EPWM_XCMP_XLOADCTL_SHDWLEVEL level)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP + EPWM_O_XLOADCTL;

    HWREGH(registerOffset) =
            (HWREGH(registerOffset) & ~EPWM_XLOADCTL_SHDWLEVEL_M) |
            ((uint32_t)level << EPWM_XLOADCTL_SHDWLEVEL_S);
}

/**
 *  @brief Selects which shadow register set is in use
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] ptr indicates current shadow buffer in use.
 *  @remarks
 *  This function gives us choice to select the shadow buffer to use currently.
 *  Valid values for alloctype are:
 *     EPWM_XCMP_XLOADCTL_SHDWBUFPTR_NULL         - No Shadow buffer is in use
 *     EPWM_XCMP_XLOADCTL_SHDWBUFPTR_ONE          - Shadow buffer 1 is in use
 *     EPWM_XCMP_XLOADCTL_SHDWBUFPTR_TWO          - Shadow buffer 2 is in use
 *     EPWM_XCMP_XLOADCTL_SHDWBUFPTR_THREE        - Shadow buffer 3 is in use
*/
static inline void
EPWM_setXCMPShadowBufPtrLoadOnce(uint32_t base,
                                 EPWM_XCMP_XLOADCTL_SHDWBUFPTR ptr)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP + EPWM_O_XLOADCTL;

    HWREGH(registerOffset) =
            (HWREGH(registerOffset) & ~EPWM_XLOADCTL_SHDWBUFPTR_LOADONCE_M) |
            ((uint32_t)ptr << EPWM_XLOADCTL_SHDWBUFPTR_LOADONCE_S);
}

/**
 *  @brief Chooses how many times Shadow buffer 2 or 3 is applied
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] bufferset is the shadow set number.
 *  @param [in] count is number of times Shadow Buffer 2 or 3 is applied
 *  @remarks
 *  This function gives us choice to select how many times shadow buffer
 *  2 or 3 is applied. Max value is 7.
 *  count = 0   - 1 time
 *  count = 1   - 2 times .....
 *  count = 7   - 8 times
 *  Valid values for bufferset are:
 *     EPWM_XCMP_SHADOW2         - XCMP set is Shadow 2
 *     EPWM_XCMP_SHADOW3         - XCMP set is Shadow 3
*/
static inline void
EPWM_setXCMPShadowRepeatBufxCount(uint32_t base, uint32_t bufferset,
                                  uint32_t count)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(count < 8U);
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP + EPWM_O_XLOADCTL;

    if(bufferset == EPWM_XCMP_SHADOW2)
    {
        HWREG(registerOffset) =
                (HWREG(registerOffset) & ~EPWM_XLOADCTL_RPTBUF2PRD_M) |
                ((uint32_t)count << EPWM_XLOADCTL_RPTBUF2PRD_S);
    }
    else if(bufferset == EPWM_XCMP_SHADOW3)
    {
        HWREG(registerOffset) =
                (HWREG(registerOffset) & ~EPWM_XLOADCTL_RPTBUF3PRD_M) |
                ((uint32_t)count << EPWM_XLOADCTL_RPTBUF3PRD_S);
    }
    else
    {
        //
        // For correct inputs, this function is not expected to
        // execute.
        //
//        ESTOP0;
    }
}


//*****************************************************************************
//
//! Sets only the high resolution XCMP value.
//!
//! \param base is the base address of the EPWM module
//! \param xcmpReg is the XCMP register offset
//! \param hrXcmpValue is the HR value to be written to XCMP registers
//!
//! This function sets the high resolution counter compare value(XCMPx_HR) for
//! XCMP registers.
//! Valid values for xcmpReg are:
//!    EPWM_XCMP[1-8]_[ACTIVE/SHADOW1/SHADOW2/SHADOW3]
//!                          -XCMP[1-8]_[ACTIVE/SHADOW1/SHADOW2/SHADOW3]
//!    EPWM_XTBPRD_ACTIVE                               -XTBPRD_ACTIVE
//!    EPWM_XTBPRD_SHADOW1                              -XTBPRD_SHADOW1
//!    EPWM_XTBPRD_SHADOW2                              -XTBPRD_SHADOW2
//!    EPWM_XTBPRD_SHADOW3                              -XTBPRD_SHADOW3
//!
//! \return None.
//
//*****************************************************************************
static inline void
HRPWM_setHiResXCMPRegValueOnly(uint32_t base, EPWM_XCMPReg xcmpReg,
                               uint32_t hrXcmpValue)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    uint32_t registerOffset;
    registerOffset = base + EPWM_O_XCMP +
                     EPWM_O_XCMP1_ACTIVE + (uint32_t)xcmpReg;

    //
    // Write to the XCMP registers.
    //
    HWREGH(registerOffset) = hrXcmpValue;
}

#endif

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************


#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_EPWM_XCMP_H_ */
