/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    flash.h
*   @brief   
*
*   commit history
*   20240312, Zhao Lei, Disable Flash Cache Check for 3.0
*	20240315, Zeng Jia, Add Fapi headinfo
*	20240410, Jason Yang, add parenthesis to remove warnings.
*	20240515, Zhao Lei, add 3.0 efuse/option byte struct.
*
*/

#ifndef DEVICE_DRIVERLIB_FLASH_H_
#define DEVICE_DRIVERLIB_FLASH_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "gs32_version.h"
#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_flash.h"
#include "flash.h"
#include "debug.h"
#include <stdbool.h>

//*****************************************************************************
//
// Values that can be returned from Flash_getInterruptFlag and
// Flash_getECCTestStatus.
//
//*****************************************************************************
#define    FLASH_NO_ERROR         0x0 //!< No error
#define    FLASH_SINGLE_ERROR     0x1 //!< Single bit error
#define    FLASH_UNC_ERROR        0x2 //!< Uncorrectable error

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

typedef enum
{
    FLASH_BANK0 = 0x0, //!< Bank 0
} Flash_BankNumber;

#if IS_GS32F00xx(0x12, 0x30)
#define COREMODE_DUAL_DSP_MASTER    0
#define COREMODE_DUAL_ARM_MASTER    1
#define COREMODE_SINGLE_DSP         2
#define COREMODE_SINGLE_ARM         3

#define RDP_LEVEL_0     0xAA
#define RDP_LEVEL_2     0xCC

#if IS_GS32F00xx(0x12)
typedef struct UserOptionByte {
    uint8_t bootPin0;
    uint8_t bootPin1;
    uint8_t reserved;
    uint8_t bootPinCfgKey;      /*It works when opt_boot_pin_cfg_key==0x5A*/
    uint8_t bootDef;            /* 32 ~ 39 */

    uint8_t sramRst:1;          /* 40 */
    uint8_t coreMode:2;         /* 41~42 */
    uint8_t dspCacheDis:1;      /* 43 */
    uint8_t dspLSRAMMode:1;     /* 44 */
    uint8_t H2HBypass:1;        /* 45 */
    uint8_t obsToGPIO16:1;        /* 46 */
    uint8_t obsToGPIO17:1;        /* 47 */

    uint8_t dllClkRate:1;       /* 48 */
    uint8_t slvTapEn:1;         /* 49 */
    uint8_t reserved3:6;        /* 50~55 */
    //uint8_t DSPFlashBootOffset;   //dsp reset vector[20:13] when boot from flash
}__attribute((__packed__)) UserOptionByteType;

#elif IS_GS32F00xx(0x30)
typedef struct UserOptionByte {
    uint32_t bootPinSel;        /* 0 ~ 31 */
    uint8_t bootDef;            /* 32 ~ 39 */

    uint8_t sramRst:1;          /* 40 */
    uint8_t ilmShadow:1;        /* 41 */
    uint8_t dualbankMode:1;     /* 42 */
    uint8_t reserved1:1;        /* 43 */
    uint8_t bootPinLatchSet:1;  /* 44 */
    uint8_t bootPinLatch:1;     /* 45 */
    uint8_t gpio16OBS:1;        /* 46 */
    uint8_t gpio18OBS:1;        /* 47 */

    uint8_t reserved2:6;        /* 48~55 */
}__attribute((__packed__)) UserOptionByteType;
#endif

typedef struct OptionBytes {
    //byte0 ~ byte7
    UserOptionByteType UserOptionByte;
    uint8_t RDP;

    //byte8 ~ byte15
    uint8_t UserOptionByteComplement[7];
    uint8_t RDPComplement;

    //byte16 ~ byte23
    uint8_t PCROP[8];       //read out protect

    //byte24 ~ byte31
    uint8_t PCROPComplement[8];

    //byte32 ~ byte39
    uint8_t WRP[8];         //write protect

    //byte40 ~ byte47
    uint8_t WRPComplement[8];

    //byte48 ~ byte 55
    uint8_t secSize1;
    uint8_t secSize2;
    uint8_t bootLock;
    uint8_t unused2;
    uint8_t PCROP_RDP;
    uint8_t unused3;
    uint8_t secDebugEn;
    uint8_t unused4;

    //byte56 ~ byte 63
    uint8_t secSizeComplement1;
    uint8_t secSizeComplement2;
    uint8_t bootLockComplement;
    uint8_t unused2Complement;
    uint8_t PCROP_RDPComplement;
    uint8_t unused3Complement;
    uint8_t secDebugEnComplement;
    uint8_t unused4Complement;
}__attribute((__packed__)) OptionBytesType;
#elif IS_GS32F3xx(0x22)
typedef struct UserOptionByte {
    uint32_t bootPinSel;		/* 0~31 */
    uint8_t bootDef;			/* 32~39 */

    uint8_t dsp1Off:1;			/* 40 */
    uint8_t dllClkMode:1;		/* 41 */
    uint8_t sramInit:1;			/* 42 */
    uint8_t reserved1:5;

    uint8_t reserved2:8;
}__attribute((__packed__)) UserOptionByteType;

typedef struct OptionBytes {
    //byte0 ~ byte7
    UserOptionByteType UserOptionByte;
    uint8_t reserved1;

    //byte8 ~ byte15
    uint8_t UserOptionByteComplement[7];
    uint8_t reserved2;

    //byte16 ~ byte63
	uint8_t reserved3[64 - 16];
}__attribute((__packed__)) OptionBytesType;
#endif


//*****************************************************************************
//
// Delay instruction that allows for register configuration to complete.
//
//*****************************************************************************
#define	FLASH_DELAY_CONFIG	__ASM volatile("nop");\
							__ASM volatile("nop");\
							__ASM volatile("nop");\
							__ASM volatile("nop");\
							__ASM volatile("nop");\
							__ASM volatile("nop");\
							__ASM volatile("nop");\
							__ASM volatile("nop");

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */

extern void Flash_programData(uint32_t base, uint8_t *data, uint8_t length);

extern void Flash_programTrimData(uint32_t base, uint32_t *trim_data, uint8_t length);

extern void Flash_clearStatus(uint32_t base, uint32_t status);

extern void FlashSetupCache(uint32_t iCacheEn, uint32_t dCacheEn, uint32_t preFetchEn);

extern void FlashInvlaidCache(void);

extern int32_t ReadInfoFlash(uint32_t StartAddr, uint8_t *buf, uint32_t bytes);

extern uint32_t Fapi_getUserOptByte(uint32_t offset);

/**
 * \brief   "Checks a flash base address." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline bool Flash_isBaseValid(uint32_t base)
{
#if IS_GS32F00xx(0x12)
	return( (base == FLASH_INTERFACE_EFC_BASE) || (base == FLASH_INTERFACE_EFC2_BASE) );
#elif IS_GS32F3xx(0x22)
	return( (base == FLASH_INTERFACE_EFC_BASE) );
#elif IS_GS32F00xx(0x30)
	return( (base == FLASH_INTERFACE_EFC_BASE) || (base == FLASH_INTERFACE_EFC2_BASE) );
#else
#error "Use the GS32F00xx or GS32F3xx macro to define the specific chip model."
#endif
}

/**
 * \brief   "Checks a flash base address." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline bool Flash_isCacheBaseValid(uint32_t cacheBase)
{
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
#if defined(FLASH_INTERFACE_DCACHE_BASE) && defined(FLASH_INTERFACE_ICACHE_BASE)
    return((cacheBase == FLASH_INTERFACE_DCACHE_BASE) ||
        (cacheBase == FLASH_INTERFACE_ICACHE_BASE));
#else
    return 0;
#endif
#elif IS_GS32F00xx(0x30)
    return TRUE;
#else
#error "Use the GS32F00xx or GS32F3xx macro to define the specific chip model."
#endif
}

/**
 * \brief   "Checks a flash base address." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline bool Flash_isCtrlBaseValid(uint32_t ctrlBase)
{
    return Flash_isBaseValid(ctrlBase);
}

/**
 * \brief   "Checks a flash base address." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline bool Flash_isECCBaseValid(uint32_t eccBase)
{
    return Flash_isBaseValid(eccBase);
}

/**
 * \brief   "get flash status." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  status as bellow:
 *          FLASH_STATUS_EFC_BUSY
 *          FLASH_STATUS_WRITE_ERROR
 *          FLASH_STATUS_DISABLE_REDUND
 *          FLASH_STATUS_DISCHARGED_STATUS
 *          FLASH_STATUS_BROWNOUT_STATUS
 *          FLASH_STATUS_WRITE_COMP_STATUS
 *          FLASH_STATUS_SEC_DETECTED
 *          FLASH_STATUS_DED_DETECTED
 *          FLASH_STATUS_SMW_ERR_STATUS
 *          FLASH_STATUS_SMW_LATEST_STATUS
 *          FLASH_STATUS_SMW_LOOP_STATUS
 */
static inline uint32_t Flash_getStatus(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    return HWREG(base + FLASH_O_STATUS);
}

/**
 * \brief   "Sets the random read wait state amount." 
 *
 * \param   base is the base address of the flash registers.
 * \param   waitstates is the wait-state value.
 *
 * \retval  None
 */
static inline void Flash_setWaitstates(uint32_t ctrlBase, uint16_t waitstates)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCtrlBaseValid(ctrlBase));
    HWREG(ctrlBase + FLASH_O_TIMING0) = (HWREG(ctrlBase + FLASH_O_TIMING0) & (~FLASH_TIMING0_TACC_CYCLES)) | waitstates;
}

/**
 * \brief   "Enables prefetch mechanism." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void
Flash_enablePrefetch(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CCR) |= FLASH_CACHE_CCR_SET_PREFETCH;
}

/**
 * \brief   "Disables prefetch mechanism." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void
Flash_disablePrefetch(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CCR) &= (~FLASH_CACHE_CCR_SET_PREFETCH);
}

/**
 * \brief   "Enables data cache." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void
Flash_enableCache(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CCR) |= FLASH_CACHE_CCR_EN;
}

/**
 * \brief   "Disables data cache." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void
Flash_disableCache(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CCR) &= (~FLASH_CACHE_CCR_EN);
}

/**
 * \brief   "Reads the bank is ready for Flash access." 
 *
 * \param   base is the base address of the flash registers.
 * \param   bank is the flash bank that is being used.
 *
 * \retval  None
 */
static inline bool
Flash_isBankReady(uint32_t ctrlBase, Flash_BankNumber bank)
{
    ASSERT(bank == FLASH_BANK0);

    return (!(Flash_getStatus(ctrlBase) & FLASH_STATUS_EFC_BUSY));
}

/**
 * \brief   "Reads the pump is ready for Flash access." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline bool
Flash_isPumpReady(uint32_t ctrlBase)
{
    return (!(Flash_getStatus(ctrlBase) & FLASH_STATUS_EFC_BUSY));
}

/**
 * \brief   "Gets the single bit error count." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  Returns the single bit error count.
 */
static inline uint32_t
Flash_getErrorCount(uint32_t eccBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isECCBaseValid(eccBase));

    return(HWREG(eccBase + FLASH_O_SEC_COUNT));
}

/**
 * \brief   "Gets the error interrupt." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  Returns the interrupt flag.
 *          FLASH_NO_ERROR
 *          FLASH_SINGLE_ERROR
 *          FLASH_UNC_ERROR
 */
static inline uint32_t
Flash_getInterruptFlag(uint32_t eccBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isECCBaseValid(eccBase));

    //
    // Read which type of error occurred.
    //
    return(((HWREG(eccBase + FLASH_O_STATUS) >> 6)  & (uint32_t)0x3U));
}

/**
 * \brief   "Clears the single error interrupt flag." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  NONE
 */
static inline void
Flash_clearSingleErrorInterruptFlag(uint32_t eccBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isECCBaseValid(eccBase));

    HWREG(eccBase + FLASH_O_STATUS) |= FLASH_STATUS_SEC_DETECTED;
}

/**
 * \brief   "Clears the uncorrectable error interrupt flag." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  NONE
 */
static inline void
Flash_clearUncorrectableInterruptFlag(uint32_t eccBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isECCBaseValid(eccBase));

    HWREG(eccBase + FLASH_O_STATUS) |= FLASH_STATUS_DED_DETECTED;
}

/**
 * \brief   "enter sleep mode" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_powerDown(uint32_t ctrlBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(ctrlBase));
#if IS_GS32F00xx(0x30)
    HWREG(ctrlBase + FLASH_O_STATIC_CFG) |= FLASH_CONTROL_SLP_MODE;
#else
    HWREG(ctrlBase + FLASH_O_CONTROL) |= FLASH_CONTROL_SLP_MODE;
#endif
}

/**
 * \brief   "wake up from sleep mode" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_wakeFromLPM(uint32_t ctrlBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(ctrlBase));
#if IS_GS32F00xx(0x30)
    HWREG(ctrlBase + FLASH_O_STATIC_CFG) &= (~FLASH_CONTROL_SLP_MODE);
#else
    HWREG(ctrlBase + FLASH_O_CONTROL) &= (~FLASH_CONTROL_SLP_MODE);
#endif
}

/**
 * \brief   "enable register write protect" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_enableRegisterWriteProtect(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_WRITE_PROTECT) |= FLASH_CONTROL_WRITE_PROTECT;
}

/**
 * \brief   "disable register write protect" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_disableRegisterWriteProtect(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_WRITE_PROTECT) &= (~FLASH_CONTROL_WRITE_PROTECT);
}

/**
 * \brief   "erase operation" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_erase(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_CONTROL) |= FLASH_CONTROL_ERASE;

}

/**
 * \brief   "erase page operation" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_erasePage(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_CONTROL) |= FLASH_CONTROL_ERASE_PAGE;
}

/**
 * \brief   "program start operation" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_programStart(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_CONTROL) |= FLASH_CONTROL_PROGRAM_START;
}

/**
 * \brief   "erase reference cell operation" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_eraseReferenceCell(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_CONTROL) |= FLASH_CONTROL_ERASE_REFERENCE_CELL;
}

/**
 * \brief   "recall trim code operation" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_recallTrimCode(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_CONTROL) |= FLASH_CONTROL_RECALL_TRIM_CODE;
}

/**
 * \brief   "program end operation" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_programEnd(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_CONTROL) |= FLASH_CONTROL_PROGRAM_END;
}

/**
 * \brief   "get error indicate" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval   TRUE:  an uncorrectable error is detected by ECC logic
             FALSE: no uncorrectable error detected by ECC logic
 */
static inline bool Flash_getErrorIndicate(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
#if IS_GS32F00xx(0x30)
    return (HWREG(base + FLASH_O_STATIC_CFG) & FLASH_CONTROL_ERROR_INDICATE);
#else
    return (HWREG(base + FLASH_O_CONTROL) & FLASH_CONTROL_ERROR_INDICATE);
#endif
}

/**
 * \brief   "clear error indicate" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_clearErrorIndicate(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

#if IS_GS32F00xx(0x30)
    HWREG(base + FLASH_O_STATIC_CFG) &= (~FLASH_CONTROL_ERROR_INDICATE);
#else
    HWREG(base + FLASH_O_CONTROL) &= (~FLASH_CONTROL_ERROR_INDICATE);
#endif
}

/**
 * \brief   "enable write complete interrupt" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_enableWriteCompleteInterrupt(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

#if IS_GS32F00xx(0x30)
    HWREG(base + FLASH_O_STATIC_CFG) |= FLASH_CONTROL_WRITE_COMP_INTR_ENABLE;
#else
    HWREG(base + FLASH_O_CONTROL) |= FLASH_CONTROL_WRITE_COMP_INTR_ENABLE;
#endif
}

/**
 * \brief   "disable write complete interrupt" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_disableWriteCompleteInterrupt(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
#if IS_GS32F00xx(0x30)
    HWREG(base + FLASH_O_STATIC_CFG) &= (~FLASH_CONTROL_WRITE_COMP_INTR_ENABLE);
#else
    HWREG(base + FLASH_O_CONTROL) &= (~FLASH_CONTROL_WRITE_COMP_INTR_ENABLE);
#endif
}

/**
 * \brief   "enable ecc interrupt" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_enableEccInterrupt(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
#if IS_GS32F00xx(0x30)
    HWREG(base + FLASH_O_STATIC_CFG) |= FLASH_CONTROL_ECC_INTR_ENABLE;
#else
    HWREG(base + FLASH_O_CONTROL) |= FLASH_CONTROL_ECC_INTR_ENABLE;
#endif
}

/**
 * \brief   "disable ecc interrupt" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_disableEccInterrupt(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
#if IS_GS32F00xx(0x30)
    HWREG(base + FLASH_O_STATIC_CFG) &= (~FLASH_CONTROL_ECC_INTR_ENABLE);
#else
    HWREG(base + FLASH_O_CONTROL) &= (~FLASH_CONTROL_ECC_INTR_ENABLE);
#endif
}

/**
 * \brief   "enable ded interrupt" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_enableDedInterrupt(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
#if IS_GS32F00xx(0x30)
    HWREG(base + FLASH_O_STATIC_CFG) |= FLASH_CONTROL_DED_INTR_ENABLE;
#else
    HWREG(base + FLASH_O_CONTROL) |= FLASH_CONTROL_DED_INTR_ENABLE;
#endif
}

/**
 * \brief   "disable ded interrupt" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_disableDedInterrupt(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
#if IS_GS32F00xx(0x30)
    HWREG(base + FLASH_O_STATIC_CFG) &= (~FLASH_CONTROL_DED_INTR_ENABLE);
#else
    HWREG(base + FLASH_O_CONTROL) &= (~FLASH_CONTROL_DED_INTR_ENABLE);
#endif
}

#if !IS_GS32F00xx(0x30)

/**
 * \brief   "start trim operation" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_startTrim(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_CONTROL) |= FLASH_CONTROL_START_TRIM;
}

/**
 * \brief   "start erase info1 operation" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_startEraseInfo1(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
#if IS_GS32F00xx(0x30)
    HWREG(base + FLASH_O_CONTROL) |= FLASH_CONTROL_START_ERASE_INFO1;
#endif
}

/**
 * \brief   "start program info1 operation" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_startProgInfo1(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
#if IS_GS32F00xx(0x30)
    HWREG(base + FLASH_O_CONTROL) |= FLASH_CONTROL_START_PROG_INFO1;
#endif
}
#endif

/**
 * \brief   "config program cycles" 
 *
 * \param   base is the base address of the flash registers.
 * \param   prog_cycles is the program cycles.
 *          When program to Info1 block, prog_cycles * clock period should meet Tprog, which is 8us~15us.
 *
 * \retval  None
 */
static inline void Flash_config_prog_cycles(uint32_t base, uint32_t prog_cycles)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_TIMING0) = (HWREG(base + FLASH_O_TIMING0) & (~FLASH_TIMING0_PROG_CYCLES)) | prog_cycles;
}

/**
 * \brief   "config eight us cycles" 
 *
 * \param   base is the base address of the flash registers.
 * \param   eight_us_cycles is the eight us cycles.
 *          eight_us_cycles * clock period should be larger than 8us.
 *
 * \retval  None
 */
static inline void Flash_config_eight_us_cycles(uint32_t base, uint32_t eight_us_cycles)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_TIMING0) = (HWREG(base + FLASH_O_TIMING0) & (~FLASH_TIMING0_EIGHT_US_CYCLES)) | eight_us_cycles;
}

/**
 * \brief   "config erc erase cycles" 
 *
 * \param   base is the base address of the flash registers.
 * \param   erc_erase_cycles is the erc erase cycles.
 *          erc_erase_cycles*clock period should meet Tre,which is 10ms~20ms.
 *
 * \retval  None
 */
static inline void Flash_config_erc_erase_cycles(uint32_t base, uint32_t erc_erase_cycles)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_TIMING1) = (HWREG(base + FLASH_O_TIMING1) & (~FLASH_TIMING1_ERC_ERASE_CYCLES)) | erc_erase_cycles;
}

/**
 * \brief   "config two us cycles" 
 *
 * \param   base is the base address of the flash registers.
 * \param   two_us_cycles is the two us cycles.
 *          two_us_cycles*clock period should be larger than 2us  to meet program info1 timing requirement.
 *
 * \retval  None
 */
static inline void Flash_config_two_us_cycles(uint32_t base, uint32_t two_us_cycles)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_TIMING1) = (HWREG(base + FLASH_O_TIMING1) & (~FLASH_TIMING1_TWO_US_CYCLES)) | two_us_cycles;
}

/**
 * \brief   "config prog erc shared cycles" 
 *
 * \param   base is the base address of the flash registers.
 * \param   erc_shared_cycles is the prog erc shared cycles.
 *          When doing ERC, prog_erc_shared_cycles*clock period should be larger than Ttmh, which is 30us
 *          When exiting from test mode to user mode,  prog_erc_shared_cycles*clock period should be larger than Ttmrs, which is 30us
 *          So before ERC, TRIM , should config this filed to a value larger than 16'd2400
 *
 * \retval  None
 */
static inline void Flash_config_prog_erc_shared_cycles(uint32_t base, uint32_t erc_shared_cycles)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_TIMING2) = (HWREG(base + FLASH_O_TIMING2) & (~FLASH_TIMING2_PROG_ERC_SHARED_CYCLES)) | erc_shared_cycles;
}

/**
 * \brief   "config hundred ns cycles" 
 *
 * \param   base is the base address of the flash registers.
 * \param   hundred_ns_cycles is the hundred ns cycles.
 *          hundred_ns_cycles*clock period should be larger than 100ns, to meet timing Ttpl in ERC
 *
 * \retval  None
 */
static inline void Flash_config_hundred_ns_cycles(uint32_t base, uint32_t hundred_ns_cycles)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_TIMING2) = (HWREG(base + FLASH_O_TIMING2) & (~FLASH_TIMING2_HUNDRED_NS_CYCLES)) | hundred_ns_cycles;
}

/**
 * \brief   "config ten ns cycles" 
 *
 * \param   base is the base address of the flash registers.
 * \param   ten_ns_cycles is the ten ns cycles.
 *          hundred_ns_cycles*clock period should be larger than 100ns, to meet timing Ttpl in ERC
 *
 * \retval  None
 */
static inline void Flash_config_ten_ns_cycles(uint32_t base, uint32_t ten_ns_cycles)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_TIMING2) = (HWREG(base + FLASH_O_TIMING2) & (~FLASH_TIMING2_TEN_NS_CYCLES)) | ten_ns_cycles;
}

/**
 * \brief   "Write a specific sequence in order to unlock access to the write protection register." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_unlockKey0(uint32_t base, uint16_t key)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_UNLOCK1) = key;
}

/**
 * \brief   "Write a specific sequence in order to unlock access to the write protection register." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  None
 */
static inline void Flash_unlockKey1(uint32_t base, uint16_t key)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_UNLOCK2) = key;
}

/**
 * \brief   "set flash address to access." 
 *
 * \param   base is the base address of the flash registers.
 * \param   address is the flash address to access.
 *
 * \retval  None
 */
static inline void Flash_setAddress(uint32_t base, uint32_t address)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_ADDRESS) = address;
}

/**
 * \brief   "get redundancy 0 address." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  [0] Indicate redundancy 0 is used, [18:12] Redundant page 0 address
 */
static inline uint32_t Flash_getRedun0Address(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    return HWREG(base + FLASH_O_REDUN0);
}

/**
 * \brief   "set redundancy 0 address." 
 *
 * \param   base is the base address of the flash registers.
 * \param   address [0] Indicate redundancy 0 is used, [18:12] Redundant page 0 address
 *
 * \retval  NONE
 */
static inline void Flash_setRedun0Address(uint32_t base, uint32_t address)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_REDUN0) = address;
}

/**
 * \brief   "get redundancy 1 address." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  [0] Indicate redundancy 1 is used, [18:12] Redundant page 1 address
 */
static inline uint32_t Flash_getRedun1Address(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    return HWREG(base + FLASH_O_REDUN1);
}

/**
 * \brief   "set redundancy 1 address." 
 *
 * \param   base is the base address of the flash registers.
 * \param   address [0] Indicate redundancy 1 is used, [18:12] Redundant page 1 address
 *
 * \retval  NONE
 */
static inline void Flash_setRedun1Address(uint32_t base, uint32_t address)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_REDUN1) = address;
}

/**
 * \brief   "get Uncorrectable Error Count." 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  Uncorrectable Error Count
 */
static inline uint32_t
Flash_getUncorrectableErrorCount(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    return(HWREG(base + FLASH_O_DED_COUNT));
}

/**
 * \brief   "config swm timer option for debug" 
 *
 * \param   base is the base address of the flash registers.
 * \param   swm_timer_option is Used to setup timer option for smw_top.
 *
 * \retval  None
 */
static inline void Flash_debug_swm_timer_option(uint32_t base, uint32_t swm_timer_option)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_SMW_SET0) = swm_timer_option;
}

/**
 * \brief   "config swm setting option10 for debug" 
 *
 * \param   base is the base address of the flash registers.
 * \param   swm_setting_option10 is Used to setup voltage and timing parameters for smart-write algorithm.
 *
 * \retval  None
 */
static inline void Flash_debug_swm_setting_option10(uint32_t base, uint32_t swm_setting_option10)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_SMW_SET10) = swm_setting_option10;
}

/**
 * \brief   "config swm setting option11 for debug" 
 *
 * \param   base is the base address of the flash registers.
 * \param   swm_setting_option11 is Used to setup voltage and timing parameters for smart-write algorithm.
 *
 * \retval  None
 */
static inline void Flash_debug_swm_setting_option11(uint32_t base, uint32_t swm_setting_option11)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_SMW_SET11) = swm_setting_option11;
}

/**
 * \brief   "config swm setting option2 for debug" 
 *
 * \param   base is the base address of the flash registers.
 * \param   swm_setting_option2 is Extra setting for SMWR algorithm, used to setup extra voltage and timing parameters for smart-write algorithm. 
 *
 * \retval  None
 */
static inline void Flash_debug_swm_setting_option2(uint32_t base, uint32_t swm_setting_option2)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_SMW_SET2) = swm_setting_option2;
}

/**
 * \brief   "config swm setting option3 for debug" 
 *
 * \param   base is the base address of the flash registers.
 * \param   swm_setting_option3 is Extra setting for SMWR algorithm, used to setup max pulse count for erase cycle at 100K endurance array area.
 *
 * \retval  None
 */
static inline void Flash_debug_swm_setting_option3(uint32_t base, uint32_t swm_setting_option3)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_SMW_SET3) = swm_setting_option3;
}

/**
 * \brief   "config swm smp whv option0 for debug" 
 *
 * \param   base is the base address of the flash registers.
 * \param   swm_smp_whv_option0 is Used to setup voltage for smart-write algorithm. 
 *
 * \retval  None
 */
static inline void Flash_debug_swm_smp_whv_option0(uint32_t base, uint32_t swm_smp_whv_option0)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_SMW_SMP0) = swm_smp_whv_option0;
}

/**
 * \brief   "config swm smp whv option1 for debug" 
 *
 * \param   base is the base address of the flash registers.
 * \param   swm_smp_whv_option1 is Used to setup voltage for smart-write algorithm. 
 *
 * \retval  None
 */
static inline void Flash_debug_swm_smp_whv_option1(uint32_t base, uint32_t swm_smp_whv_option1)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_SMW_SMP1) = swm_smp_whv_option1;
}

/**
 * \brief   "config swm sme whv option0 for debug" 
 *
 * \param   base is the base address of the flash registers.
 * \param   swm_sme_whv_option0 is setting should be finalized after silicon data analysis is completed on smart-algorithm. 
 *
 * \retval  None
 */
static inline void Flash_debug_swm_sme_whv_option0(uint32_t base, uint32_t swm_sme_whv_option0)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_SMW_SME0) = swm_sme_whv_option0;
}

/**
 * \brief   "config swm sme whv option1 for debug" 
 *
 * \param   base is the base address of the flash registers.
 * \param   swm_sme_whv_option1 is setting should be finalized after silicon data analysis is completed on smart-algorithm. 
 *
 * \retval  None
 */
static inline void Flash_debug_swm_sme_whv_option1(uint32_t base, uint32_t swm_sme_whv_option1)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_SMW_SME1) = swm_sme_whv_option1;
}

/**
 * \brief   "config flash misc cfg for debug" 
 *
 * \param   base is the base address of the flash registers.
 * \param   smw_cfg is config of whv_cfg/wmv_cfg/wipgm. 
 *
 * \retval  None
 */
static inline void Flash_debug_misc_cfg(uint32_t base, uint32_t smw_cfg)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));
    HWREG(base + FLASH_O_SMW_CFG) = smw_cfg;
}

/**
 * \brief   "enable Write protect to all pages" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  NONE
 */
static inline void Flash_enableAllPageWriteProtect(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_WP0) |= FLASH_WP0_WRITE_PROTECT_ALL;
}

/**
 * \brief   "disable Write protect to all pages" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  NONE
 */
static inline void Flash_disableAllPageWriteProtect(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_WP0) &= (~FLASH_WP0_WRITE_PROTECT_ALL);
}

/**
 * \brief   "enable Write protect to all info pages" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  NONE
 */
static inline void Flash_enableAllInfoPageWriteProtect(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_WP0) |= FLASH_WP0_INFO_WRITE_PROTECT;
}

/**
 * \brief   "disable Write protect to all info pages" 
 *
 * \param   base is the base address of the flash registers.
 *
 * \retval  NONE
 */
static inline void Flash_disableAllInfoPageWriteProtect(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_WP0) &= (~FLASH_WP0_INFO_WRITE_PROTECT);
}

/**
 * \brief   "config Write prottect to page 0~63" 
 *
 * \param   base is the base address of the flash registers.
 * \param   page is the page bit to write protect.
 *
 * \retval  NONE
 */
static inline void Flash_configPageWriteProtect(uint32_t base, uint64_t pages)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isBaseValid(base));

    HWREG(base + FLASH_O_WP1) = (uint32_t)pages;
    HWREG(base + FLASH_O_WP2) = (uint32_t)(pages >> 32);
}

/**
 * \brief   "flash cache Manual invalidate request" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  NONE
 */
static inline void Flash_Cache_invalidateReq(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CCR) |= FLASH_CACHE_CCR_INV_REQ;
}

/**
 * \brief   "flash cache Manual SRAM power request" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  NONE
 */
static inline void Flash_Cache_powerReq(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CCR) |= FLASH_CACHE_CCR_POW_REQ;
}

/**
 * \brief   "flash cache Power Control Setting" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  NONE
 */
static inline void Flash_Cache_setManualPower(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CCR) |= FLASH_CACHE_CCR_SET_MAN_POW;
}

/**
 * \brief   "flash cache Power Control Setting" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  NONE
 */
static inline void Flash_Cache_setAutomaticPower(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CCR) &= (~FLASH_CACHE_CCR_SET_MAN_POW);
}

/**
 * \brief   "flash cache Invalidate Setting" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  NONE
 */
static inline void Flash_Cache_setManualInvalidate(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CCR) |= FLASH_CACHE_CCR_SET_MAN_INV;
}

/**
 * \brief   "flash cache Invalidate Setting" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  NONE
 */
static inline void Flash_Cache_setAutomaticInvalidate(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CCR) &= (~FLASH_CACHE_CCR_SET_MAN_INV);
}

/**
 * \brief   "flash cache Invalidate Setting" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  NONE
 */
static inline void Flash_Cache_enableStatistic(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CCR) |= FLASH_CACHE_CCR_STATISTIC_EN;
}

/**
 * \brief   "flash cache Invalidate Setting" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  NONE
 */
static inline void Flash_Cache_disableStatistic(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CCR) &= (~FLASH_CACHE_CCR_STATISTIC_EN);
}

/**
 * \brief   "get flash cache status" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  status
 *          Bits     Name          Description
 *          4        POW_STAT      SRAM power acknowledges. Real-time registered value of RAMPWRUPACK port.
 *          2        INV_STAT      Invalidating Status. Indicates if invalidation process is ongoing.
 *          [1:0]    CS            Cache status:0: Cache disabled.1: Cache enabling.2: Cache enabled.3: Cache disabling.
 */
static inline uint32_t Flash_Cache_getStatus(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    return HWREG(cacheBase + FLASH_CACHE_O_SR);
}

/**
 * \brief   "Interrupt request unmask" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 * \param   irqs      is FLASH_CACHE_IRQMASK_POW_ERR or FLASH_CACHE_IRQMASK_MAN_INV_ERR.
 *
 * \retval  NONE
 */
static inline void Flash_Cache_enableInterrupt(uint32_t cacheBase, uint32_t irqs)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    if (irqs & FLASH_CACHE_IRQMASK_POW_ERR)
        HWREG(cacheBase + FLASH_CACHE_O_IRQMASK) &= (~FLASH_CACHE_IRQMASK_POW_ERR);
    if (irqs & FLASH_CACHE_IRQMASK_MAN_INV_ERR)
        HWREG(cacheBase + FLASH_CACHE_O_IRQMASK) &= (~FLASH_CACHE_IRQMASK_MAN_INV_ERR);
}

/**
 * \brief   "Interrupt request mask" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 * \param   irqs      is FLASH_CACHE_IRQMASK_POW_ERR or FLASH_CACHE_IRQMASK_MAN_INV_ERR.
 *
 * \retval  NONE
 */
static inline void Flash_Cache_disableInterrupt(uint32_t cacheBase, uint32_t irqs)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    if (irqs & FLASH_CACHE_IRQMASK_POW_ERR)
        HWREG(cacheBase + FLASH_CACHE_O_IRQMASK) |= FLASH_CACHE_IRQMASK_POW_ERR;
    if (irqs & FLASH_CACHE_IRQMASK_MAN_INV_ERR)
        HWREG(cacheBase + FLASH_CACHE_O_IRQMASK) |= FLASH_CACHE_IRQMASK_MAN_INV_ERR;
}

/**
 * \brief   "get flash cache interrupt status" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  status
 *          Bits     Name          Description
 *          1        MAN_INV_ERR   Mask interrupt request on manual invalidation error indication (IRQSTAT.MAN_INV_ERR is set).
 *          0        POW_ERR       Mask interrupt request on Power Error indication (IRQSTAT.POW_ERR is set).
 */
static inline uint32_t Flash_Cache_getInterruptStatus(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    return HWREG(cacheBase + FLASH_CACHE_O_IRQSTAT);
}

/**
 * \brief   "get flash cache hardware parameters" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  hardware parameters
 *          Bits     Name             Description
 *          13       GEN_STAT_LOGIC   Indicates GEN_STAT_LOGIC hardwareparameter value.
 *          12       RESET_ALL_REGS   Indicates RESET_ALL_REGS hardwareparameter value.
 *          [11:10]  CACHE_WAY        Implementation-defined value for number of cache ways: 1: One cache way. 2: Two cache ways
 *          [9:5]    CW               Implementation-defined value for cache way width: 8: 256B. 9: 512B. 11: 2KB. 12: 4KB.
 *          [4:0]    AW               Implementation-defined value for AHB-Lite bus width for the flash address space:
 *                                     16: 64KB. 17: 128KB. 18: 256KB. 19: 512KB. 20: 1024KB. 21: 2048KB. 22: 4096KB.
 */
static inline uint32_t Flash_Cache_getHardwareParameters(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    return HWREG(cacheBase + FLASH_CACHE_O_HWPARAMS);
}

/**
 * \brief   "get flash cache Statistic Hit" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  Counts the number of cache hits during cache look up.
 */
static inline uint32_t Flash_Cache_getStatisticHit(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    return HWREG(cacheBase + FLASH_CACHE_O_CSHR);
}

static inline void Flash_Cache_clearStatisticHit(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CSHR) = 0;
}

/**
 * \brief   "get flash cache Statistic Miss" 
 *
 * \param   cacheBase is the base address of the flash cache registers.
 *
 * \retval  Counts the number of cache hits during cache look up.
 */
static inline uint32_t Flash_Cache_getStatisticMiss(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    return HWREG(cacheBase + FLASH_CACHE_O_CSMR);
}

static inline void Flash_Cache_clearStatisticMiss(uint32_t cacheBase)
{
    //
    // Check the arguments.
    //
    ASSERT(Flash_isCacheBaseValid(cacheBase));

    HWREG(cacheBase + FLASH_CACHE_O_CSMR) = 0;
}

#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_FLASH_H_ */
