/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *   @file    gpio.c
 *   @brief   GS32 GPIO driver.
 *   @details
 *
 */

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "gpio.h"
#include "gs32_version.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)

/**
* @brief   Initialize basic GPIO functionality
*
* @param [in] None
*
* @note    Set all pins to multiplexed GPIO operating in input mode (disable analog).
*          Disable all pull-ups and input polarity inversion.
*          Set qualification to synchronous.
*
* @retval  None
*/
GS32_DRIVER_INIT_FUNC_T void InitGpio(void)
{
    volatile uint32_t *gpioBaseAddr_S;
    volatile uint32_t *gpioBaseAddr_E;
    uint32_t           regOffset;
#if IS_GS32F00xx(0x12)
    gpioBaseAddr_S = (Uint32 *)&GpioCtrlRegs;
    gpioBaseAddr_E = (Uint32 *)&(GpioCtrlRegs.GPCGMUX1);

    for (regOffset = 0; regOffset <= (gpioBaseAddr_E - gpioBaseAddr_S); regOffset++)
    {
        if ((regOffset % (GPIO_CTRL_REGS_STEP) != (GPIO_O_GPAPUD / 4U))
        		&& (regOffset != (GPIO_O_GPBMUX1 / 4U))
				&& (regOffset != (GPIO_O_GPBGMUX1 / 4U)))
        {
            gpioBaseAddr_S[regOffset] = 0x00000000;
        }
        else if ((regOffset % (GPIO_CTRL_REGS_STEP)) == (GPIO_O_GPAPUD / 4U))
        {
            gpioBaseAddr_S[regOffset] = 0xFFFFFFFF;
        }
    }

    gpioBaseAddr_S = (Uint32 *)&(GpioCtrlRegs.GPHCTRL);
    gpioBaseAddr_E = (Uint32 *)&(GpioCtrlRegs.GPHGMUX2);

    for (regOffset = 0; regOffset <= (gpioBaseAddr_E - gpioBaseAddr_S); regOffset++)
    {
        if ((regOffset != ((GPIO_O_GPHAMSEL - GPIO_O_GPHCTRL) / 4U)))
        {
            gpioBaseAddr_S[regOffset] = 0x00000000;
        }
        else
        {
            gpioBaseAddr_S[regOffset] = 0xFFFFFFFF;
        }
    }
#elif IS_GS32F3xx(0x22)

    gpioBaseAddr_S = (Uint32 *)&GpioCtrlRegs;
    gpioBaseAddr_E = (Uint32 *)&(GpioCtrlRegs.GPHGMUX2);

    for (regOffset = 0; regOffset <= (gpioBaseAddr_E - gpioBaseAddr_S); regOffset++)
    {
        if ((regOffset % (GPIO_CTRL_REGS_STEP) != (GPIO_O_GPAPUD / 4U))
        		&& (regOffset != (GPIO_O_GPGMUX2 / 4U))
				&& (regOffset != (GPIO_O_GPGGMUX2 / 4U)))
        {
            gpioBaseAddr_S[regOffset] = 0x00000000;
        }
        else if ((regOffset % (GPIO_CTRL_REGS_STEP)) == (GPIO_O_GPAPUD / 4U))
        {
            gpioBaseAddr_S[regOffset] = 0xFFFFFFFF;
        }
    }

#endif

#if defined(GPIO_PORTA_BASE) || defined(GPIO_GROUP0_BASE)
    GpioPortaRegs.DATA.all = 0x00000000;
#endif
#if defined(GPIO_PORTB_BASE) || defined(GPIO_GROUP1_BASE)
    GpioPortbRegs.DATA.all = 0x00000000;
#endif
#if defined(GPIO_PORTC_BASE) || defined(GPIO_GROUP2_BASE)
    GpioPortcRegs.DATA.all = 0x00000000;
#endif
#if defined(GPIO_PORTD_BASE) || defined(GPIO_GROUP3_BASE)
    GpioPortdRegs.DATA.all = 0x00000000;
#endif
#if defined(GPIO_PORTE_BASE) || defined(GPIO_GROUP4_BASE)
    GpioPorteRegs.DATA.all = 0x00000000;
#endif
#if defined(GPIO_PORTF_BASE) || defined(GPIO_GROUP5_BASE)
    GpioPortfRegs.DATA.all = 0x00000000;
#endif
#if defined(GPIO_PORTG_BASE) || defined(GPIO_GROUP6_BASE)
    GpioPortgRegs.DATA.all = 0x00000000;
#endif
#if defined(GPIO_PORTH_BASE) || defined(GPIO_GROUP7_BASE)
    GpioPorthRegs.DATA.all = 0x00000000;
#endif
#if defined(GPIO_PORTI_BASE) || defined(GPIO_GROUP8_BASE)
    GpioPortiRegs.DATA.all = 0x00000000;
#endif
#if defined(GPIO_PORTJ_BASE) || defined(GPIO_GROUP9_BASE)
    GpioPortjRegs.DATA.all = 0x00000000;
#endif
#if defined(GPIO_PORTK_BASE) || defined(GPIO_GROUP10_BASE)
    GpioPortkRegs.DATA.all = 0x00000000;
#endif
#if defined(GPIO_PORTL_BASE) || defined(GPIO_GROUP11_BASE)
    GpioPortlRegs.DATA.all = 0x00000000;
#endif
#if defined(GPIO_PORTM_BASE) || defined(GPIO_GROUP12_BASE)
    GpioPortmRegs.DATA.all = 0x00000000;
#endif
#if defined(GPIO_PORTN_BASE) || defined(GPIO_GROUP13_BASE)
    GpioPortnRegs.DATA.all = 0x00000000;
#endif
}


/**
* @brief   Set the multiplexing function of an IO pin
*
* @param [in] gpioNumber  IO pin number
* @param [in] cpu         Invalid (unused)
* @param [in] muxPosition Values for GMUX and MUX (GMUX: bits 4-2, MUX: bits 1-0)
*
* @note    None
*
* @retval  None
*/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_SetupPinMux(uint16_t gpioNumber, uint16_t cpu, uint16_t muxPosition)

{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           MuxIndex;
    uint32_t           GmuxIndex;
    uint32_t           shiftAmt;
    uint32_t           pinMask;

    gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE + (gpioNumber / 32) * GPIO_CTRL_REGS_STEP * 4U);
    MuxIndex     = (GPIO_O_GPAMUX1 + (gpioNumber % 32U) / 16U * 0x4U) / 4U;
    GmuxIndex    = (GPIO_O_GPAGMUX1 + (gpioNumber % 32U) / 16U * 0x4U) / 4U;
    shiftAmt     = (gpioNumber % 16U * 2U);
    pinMask      = (uint32_t)3U << shiftAmt;

    /* check muxPosition values */
    if (muxPosition > 0xF) {
        return;
    }

    /* Clear fields in MUX register first to avoid glitches */
    gpioBaseAddr[MuxIndex] &= ~pinMask;

    /* Write value into GMUX register */
    gpioBaseAddr[GmuxIndex] = (gpioBaseAddr[GmuxIndex] & ~pinMask) | (((muxPosition >> 2U) & 0x3U) << shiftAmt);
    /* Write value into MUX register */
    gpioBaseAddr[MuxIndex] |= ((muxPosition & 0x3U) << shiftAmt);
}

/*
 * GPIO_SetupPinOptions - Setup the GPIO input/output options for the
 * specified pin. The flags are a 16-bit mask produced by ORing together
 * options. For input pins, the valid flags are:
 *
 * GPIO_PULLUP  		Enable pull-up
 * GPIO_INVERT  		Enable input polarity inversion
 * GPIO_PULLDOWN		Enable pull-down resistor
 * GPIO_OUTPUT_INVERT	Enable Invert output polarity
 *
 * GPIO_SYNC  Synchronize the input latch to PLLSYSCLK
 *              (default -- you don't need to specify this)
 * GPIO_QUAL3  Use 3-sample qualification
 * GPIO_QUAL6  Use 6-sample qualification
 * GPIO_ASYNC  Do not use synchronization or qualification
 * (Note: only one of SYNC, QUAL3, QUAL6, or ASYNC is allowed)
 *
 * For output pins, the valid flags are:
 * GPIO_OPENDRAIN  Output in open drain mode
 * GPIO_PULLUP    If open drain enabled, also enable the pull-up
 * and the input qualification flags (SYNC/QUAL3/QUAL6/ASYNC) listed above.
 *
 * With no flags, the default input state is synchronous with no
 * pull-up or polarity inversion. The default output state is
 * the standard digital output.
 */

GS32_DRIVER_GPIO_FUNC_T void
GPIO_SetupPinOptions(uint16_t gpioNumber, uint16_t output, uint16_t flags)
{
    if (output == GPIO_OUTPUT) {
        GPIO_setDirectionMode(gpioNumber, GPIO_DIR_MODE_OUT);
    } else {
        GPIO_setDirectionMode(gpioNumber, GPIO_DIR_MODE_IN);
    }
    GPIO_setPadConfig(gpioNumber, (flags & 0x1F));
    GPIO_setQualificationMode(gpioNumber, (GPIO_QualificationMode)(flags >> 6));
}


/* GPIO_ReadPin */
GS32_DRIVER_GPIO_FUNC_T uint16_t
GPIO_ReadPin(uint16_t gpioNumber)
{
    return (GPIO_readPin(gpioNumber));
}

/* GPIO_WritePin */
GS32_DRIVER_GPIO_FUNC_T void
GPIO_WritePin(uint16_t gpioNumber, uint16_t outVal)
{
    GPIO_writePin(gpioNumber, outVal);
}

/* GPIO_setDirectionMode */
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setDirectionMode(uint32_t pin, GPIO_Direction pinIO)
{
    if (GPIO_DIR_MODE_OUT == pinIO)
        GPIO_enableWritePin(pin);
    else if (GPIO_DIR_MODE_IN == pinIO)
        GPIO_disableWritePin(pin);
}

/* GPIO_getDirectionMode */
GS32_DRIVER_GPIO_FUNC_T GPIO_Direction
GPIO_getDirectionMode(uint32_t pin)
{
    return (GPIO_Direction)(HWREGH(GPIODATA_BASE + GPIO_OUTENSET + (pin / 16U) * GPIO_BASE_ADDR_STEP) >> (pin % 16));
}
#endif

#ifdef __cplusplus
}
#endif
