/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    gpio.h
*   @brief   GPIO driver.
*
*/
/*
 * commit history
 * 20240307, Jason Yang, add PORTE/F/G/H in enum GPIO_Port type define.
 * 20240513, XuShicheng, add GPIO_setInterruptTypeEdge
							GPIO_setInterruptTypeLevel
							GPIO_setInterruptType
							GPIO_setInterruptPolarityRising
							GPIO_setInterruptPolarityFalling
							and modify the driver of API functions that have the same effect as it
 * 20240514, XuShicheng, add GPIO_getIRQStatusRegister()
 * 20240531, Jason, add empty function GPIO_setControllerCore.
 * 2025/1/1, hezhiyuan,Modify comments
 * */

#ifndef DEVICE_DRIVERLIB_GPIO_H_
#define DEVICE_DRIVERLIB_GPIO_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <stdbool.h>
#include <stdint.h>

#include "gs32_version.h"
#include "inc/hw_gpio.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "regs/regs_gpio.h"
#include "cpu.h"
#include "xbar.h"
#include "debug.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)

#define GPIO_BASE_ADDR_STEP 		0x1000

//
// Flags for GPIO_SetupPinOptions(). The qualification flags (SYNC, QUAL3,
// QUAL6, and ASYNC) take up two bits and must be in the order specified.
//
#define GPIO_INPUT      0               /* Input mode */
#define GPIO_OUTPUT     1               /* Output mode */

#define GPIO_PUSHPULL   	0               /* Push-pull output */
#define GPIO_PULLUP     	(1 << 0)        /* Enable pull-up resistor */
#define GPIO_INVERT     	(1 << 1)        /* Invert input polarity */
//#define GPIO_OPENDRAIN 	(1 << 2)       	/* Open-drain output (Not Supported) */
#define GPIO_PULLDOWN   	(1 << 3)        /* Enable pull-down resistor */
#define GPIO_OUTPUT_INVERT 	(1 << 4)     	/* Invert output polarity */

#define GPIO_SYNC       (0x0 << 6)      /* Synchronous sampling */
#define GPIO_QUAL3      (0x1 << 6)      /* 3-sample debouncing */
#define GPIO_QUAL6      (0x2 << 6)      /* 6-sample debouncing */
#define GPIO_ASYNC      (0x3 << 6)      /* Asynchronous processing */


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/**
 * 
 *  Values that can be passed to GPIO_setDirectionMode() as the pinIO
 *  parameter and returned from GPIO_getDirectionMode().
 * 
 */
typedef enum
{
    GPIO_DIR_MODE_IN,                   //< Pin is a GPIO input
    GPIO_DIR_MODE_OUT                   //< Pin is a GPIO output
} GPIO_Direction;

/**
 *
 * Values that can be passed to GPIO_readPortData(), GPIO_setPortPins(),
 * GPIO_clearPortPins(), and GPIO_togglePortPins() as the port parameter.
 *
 */
#if IS_GS32F00xx(0x12)
typedef enum
{
    GPIO_PORTA_L = 0,                    //< GPIO port A_L, 16 bits,GPIO0-15
    GPIO_PORTA_H = 1,                    //< GPIO port A_H, 16 bits,GPIO16-31
	GPIO_PORTB_L = 2,                    //< GPIO port B_L, 16 bits,GPIO32-47
	GPIO_PORTB_H = 3,                    //< GPIO port B_H, 16 bits,GPIO48-63
	GPIO_PORTC_L = 4,                    //< GPIO port C_L, 16 bits,GPIO64-79
//	GPIO_PORTC_H = 5,                    //< GPIO port C_H, 16 bits,GPIO80-95
//	GPIO_PORTD_L = 6,                    //< GPIO port C_L, 16 bits,GPIO96-111
//	GPIO_PORTD_H = 7,                    //< GPIO port C_H, 16 bits,GPIO112-127
//	GPIO_PORTE_L = 8,                    //< GPIO port C_L, 16 bits,GPIO128-143
//	GPIO_PORTE_H = 9,                    //< GPIO port C_H, 16 bits,GPIO144-159
//	GPIO_PORTF_L = 10,                   //< GPIO port C_L, 16 bits,GPIO160-175
//	GPIO_PORTF_H = 11,                   //< GPIO port C_H, 16 bits,GPIO176-191
//	GPIO_PORTG_L = 12,                   //< GPIO port C_L, 16 bits,GPIO192-207
//	GPIO_PORTG_H = 13,                   //< GPIO port C_H, 16 bits,GPIO208-223
	GPIO_PORTH_L = 14,                   //< GPIO port H_L, 16 bits,GPIO224-239
	GPIO_PORTH_H = 15,                   //< GPIO port H_H, 16 bits,GPIO240-255
} GPIO_Port_16BIT;
#elif IS_GS32F3xx(0x22)
typedef enum
{
    GPIO_PORTA_L = 0,                    //< GPIO port A_L, 16 bits,GPIO0-15
    GPIO_PORTA_H = 1,                    //< GPIO port A_H, 16 bits,GPIO16-31
	GPIO_PORTB_L = 2,                    //< GPIO port B_L, 16 bits,GPIO32-47
	GPIO_PORTB_H = 3,                    //< GPIO port B_H, 16 bits,GPIO48-63
	GPIO_PORTC_L = 4,                    //< GPIO port C_L, 16 bits,GPIO64-79
	GPIO_PORTC_H = 5,                    //< GPIO port C_H, 16 bits,GPIO80-95
	GPIO_PORTD_L = 6,                    //< GPIO port C_L, 16 bits,GPIO96-111
	GPIO_PORTD_H = 7,                    //< GPIO port C_H, 16 bits,GPIO112-127
	GPIO_PORTE_L = 8,                    //< GPIO port C_L, 16 bits,GPIO128-143
	GPIO_PORTE_H = 9,                    //< GPIO port C_H, 16 bits,GPIO144-159
	GPIO_PORTF_L = 10,                   //< GPIO port C_L, 16 bits,GPIO160-175
    GPIO_PORTF_H = 11,                   //< GPIO port C_H, 16 bits,GPIO176-191
	GPIO_PORTG_L = 12,                   //< GPIO port C_L, 16 bits,GPIO192-207
	GPIO_PORTG_H = 13,                   //< GPIO port C_H, 16 bits,GPIO208-223
	GPIO_PORTH_L = 14,                   //< GPIO port H_L, 16 bits,GPIO224-239
	GPIO_PORTH_H = 15,                   //< GPIO port H_H, 16 bits,GPIO240-255
} GPIO_Port_16BIT;
#endif
/**
 *  @brief GPIO interrupt trigger types
 * 
 *  Defines the possible interrupt trigger types for GPIO.
 * 
 *  These values can be:
 *      - Passed as the `intType` parameter to `GPIO_setInterruptType()`.
 *      - Returned by `GPIO_getInterruptType()`.
 */
typedef enum
{

    GPIO_INT_TYPE_FALLING_EDGE  = 0x01,   //< Interrupt on falling edge
    GPIO_INT_TYPE_RISING_EDGE   = 0x03,   //< Interrupt on rising edge
    GPIO_INT_TYPE_LOW_POLARITY  = 0x00,   //< Interrupt on low polarity
    GPIO_INT_TYPE_HIGH_POLARITY = 0x02,   //< Interrupt on high polarity

} GPIO_IntType;

typedef enum
{
    GPIO_INT_POLARITY_H = 0x00,   //< Interrupt on falling or rising edge
    GPIO_INT_POLARITY_LEVEL  = 0x01,   //< Interrupt on HIGH or LOW level
} GPIO_IntPolarity;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

extern volatile struct GPIO_CTRL_REGS GpioCtrlRegs;

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */

/**
 *  @brief Set the GPIO interrupt trigger type
 * 
 *  This function configures the interrupt trigger type for a specified GPIO pin.
 * 
 *  @param [in] pin 
 *      The GPIO pin number to configure.
 * 
 *  @param [in] intType 
 *      The type of interrupt trigger to set. Possible values:
 *          - GPIO_INT_TYPE_FALLING_EDGE: Interrupt on falling edge
 *          - GPIO_INT_TYPE_RISING_EDGE: Interrupt on rising edge
 *          - GPIO_INT_TYPE_LOW_POLARITY: Interrupt on low polarity
 *          - GPIO_INT_TYPE_HIGH_POLARITY: Interrupt on high polarity
 * 
 *  @note This function must be called only after the GPIO pin has been properly initialized.
 * 
 *  Example:
 *  @code
 *      GPIO_setInterruptType(5, GPIO_INT_TYPE_FALLING_EDGE);
 *  @endcode
 */
static inline void
GPIO_setInterruptType(uint16_t pin, GPIO_IntType intType)
{
	uint16_t Edge;
	uint16_t Polarity;
	Edge = intType & 0x1; // bit0:1->TYPESET 0->TYPECLR bit1:1->POLSET 0->POLCLR
	Polarity = (intType>>1) & 0x1;
	if(Edge == 1)
	{
		HWREGH(GPIODATA_BASE + GPIO_INTTYPESET + (pin / 16U) * GPIO_BASE_ADDR_STEP) |= (1 << (pin % 16));
	}
	else
	{
		HWREGH(GPIODATA_BASE + GPIO_INTTYPECLR + (pin / 16U) * GPIO_BASE_ADDR_STEP) = (1 << (pin % 16));
	}

	if(Polarity == 1)
	{
		HWREGH(GPIODATA_BASE + GPIO_INTPOLSET + (pin / 16U) * GPIO_BASE_ADDR_STEP) |= (1 << (pin % 16));
	}
	else
	{
		HWREGH(GPIODATA_BASE + GPIO_INTPOLCLR + (pin / 16U) * GPIO_BASE_ADDR_STEP) = (1 << (pin % 16));
	}
}

/**
 *  @brief Get the interrupt trigger type of a GPIO pin.
 * 
 *  This function retrieves the interrupt trigger type for the specified GPIO pin.
 * 
 *  @param [in] pin 
 *      The GPIO pin number whose interrupt trigger type is to be retrieved.
 * 
 *  @return 
 *      Returns one of the following interrupt trigger types:
 *          - GPIO_INT_TYPE_FALLING_EDGE: Interrupt triggered on falling edge
 *          - GPIO_INT_TYPE_RISING_EDGE: Interrupt triggered on rising edge
 *          - GPIO_INT_TYPE_LOW_POLARITY: Interrupt triggered when signal is low
 *          - GPIO_INT_TYPE_HIGH_POLARITY: Interrupt triggered when signal is high
 * 
 *  @note 
 *      - Ensure that the GPIO pin is properly initialized before calling this function.
 *      - The returned value is a combination of edge and polarity information.
 */
static inline GPIO_IntType
GPIO_getInterruptType(uint16_t pin)
{
	uint16_t Edge;
	uint16_t Polarity;

	Edge=((HWREGH(GPIODATA_BASE + GPIO_INTTYPESET + (pin / 16U) * GPIO_BASE_ADDR_STEP))>> (pin % 16))&0x01;
	Polarity = ((HWREGH(GPIODATA_BASE + GPIO_INTPOLSET + (pin / 16U) * GPIO_BASE_ADDR_STEP))>> (pin % 16))&0x01;

    return ((GPIO_IntType)(Edge | (Polarity << 1)));
}

/**
 *  @brief Enable GPIO interrupt for a specific pin.
 * 
 *  This function enables the interrupt for the specified GPIO pin.
 * 
 *  @param [in] pin 
 *      The GPIO pin number for which the interrupt is to be enabled.
 * 
 *  @note 
 *      - Ensure that the GPIO pin has been configured properly before enabling the interrupt.
 *      - This function only sets the enable bit for the interrupt; the trigger type must be set separately using `GPIO_setInterruptType()`.
 */
static inline void
GPIO_enableInterrupt(uint16_t pin){

		HWREGH(GPIODATA_BASE + GPIO_INTENSET + (pin / 16U) * GPIO_BASE_ADDR_STEP) |= (1 << (pin % 16));
}
/**
 *  @brief Disable GPIO interrupt for a specific pin.
 * 
 *  This function disables the interrupt for the specified GPIO pin.
 * 
 *  @param [in] pin 
 *      The GPIO pin number for which the interrupt is to be disabled.
 * 
 *  @note 
 *      - This function clears the enable bit for the interrupt but does not clear any pending interrupt flags.
 *      - To clear pending flags, use the appropriate function or register manipulation.
 */
static inline void
GPIO_disableInterrupt(uint16_t pin){

		HWREGH(GPIODATA_BASE + GPIO_INTENCLR + (pin / 16U) * GPIO_BASE_ADDR_STEP) = (1 << (pin % 16));
}
/**
 *  @brief Check if the interrupt is enabled for a specific GPIO pin.
 * 
 *  This function checks whether the interrupt is currently enabled for the specified GPIO pin.
 * 
 *  @param [in] pin 
 *      The GPIO pin number to check.
 * 
 *  @return 
 *      - Returns `1` (true) if the interrupt is enabled for the specified pin.
 *      - Returns `0` (false) if the interrupt is disabled for the specified pin.
 * 
 *  @note 
 *      - This function reads the current state of the interrupt enable register for the specified pin.
 */
static inline uint16_t
GPIO_readInterruptEnabled(uint16_t pin){

		return ((HWREGH(GPIODATA_BASE + GPIO_INTENSET + (pin / 16U) * GPIO_BASE_ADDR_STEP) & (1 << (pin % 16))) != 0);
}


/**
 * @brief   Read back [15:0] IRQ status Register.
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  0		For LOW level or falling edge.
 * 			1		For HIGH level or rising edge.
 */
static inline uint16_t
GPIO_getPinInterruptStatus(uint16_t pin){

		return ((HWREGH(GPIODATA_BASE + GPIO_INTSTATUS + (pin / 16U) * GPIO_BASE_ADDR_STEP) & (1 << (pin % 16))) != 0);
}

static inline void
GPIO_ClearPinInterruptStatus(uint16_t pin){

		HWREGH(GPIODATA_BASE + GPIO_INTCLEAR + (pin / 16U) * GPIO_BASE_ADDR_STEP) = (1 << (pin % 16));
}


/**
 * @brief   Read back [15:0] IRQ status Register.
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  0		For LOW level or falling edge.
 * 			1		For HIGH level or rising edge.
 */
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
static inline uint16_t
GPIO_getPortInterruptStatus(GPIO_Port_16BIT port){

		return HWREGH(GPIODATA_BASE + GPIO_INTSTATUS+ port * GPIO_BASE_ADDR_STEP);
}
#endif


/* It's the same as GPIO_getPinData */
#if IS_GS32F00xx(0x12)
static inline uint16_t
GPIO_readPin(uint16_t pin)
{
	uint16_t RegValue;
	if(pin>=224)
	{
		pin = pin - 155;
	}
	RegValue = HWREGH(GPIODATA_BASE + GPIO_DATA + (pin / 16U) * GPIO_BASE_ADDR_STEP);

	return (((RegValue & (1 << (pin % 16))) > 0)? 1 : 0 );
}
#elif IS_GS32F3xx(0x22)
static inline uint16_t
GPIO_readPin(uint16_t pin)
{
	uint16_t RegValue;
	if(pin>=198)
	{
		pin = pin - 29;
	}
	RegValue = HWREGH(GPIODATA_BASE + GPIO_DATA + (pin / 16U) * GPIO_BASE_ADDR_STEP);

	return (((RegValue & (1 << (pin % 16))) > 0)? 1 : 0 );
}
#endif
/**
 * @brief   Writes a value to the specified pin.
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 * @param [in]   outVal is the value to write to the pin.
 *
 * @details
 * - Writes the corresponding bit values to the output pin specified by pin.  
 * - Writing to a pin configured as an input pin has no effect.
 *
 * @remarks
 * - The pin is specified by its numerical value. For example, 
 * - GPIO34 is specified by passing 34 as pin.

 * @return  None
 */
static inline void
GPIO_writePin(uint16_t pin, uint16_t outVal)
{
	if(pin%16 <= 7)
	{
		if(outVal != 0U)
		{
			HWREGH(GPIODATA_BASE + GPIO_MASKLOWBYTE + (pin / 16U) * GPIO_BASE_ADDR_STEP + 4*(0x01<<(pin%16))) = 0x01<<(pin%16);
		}
		else
		{
			HWREGH(GPIODATA_BASE + GPIO_MASKLOWBYTE + (pin / 16U) * GPIO_BASE_ADDR_STEP + 4*(0x01<<(pin%16))) = 0x00;
		}
	}
	else
	{
		if(outVal != 0U)
		{
			HWREGH(GPIODATA_BASE + GPIO_MASKHIGHBYTE + (pin / 16U) * GPIO_BASE_ADDR_STEP + 4*(0x01<<(pin%(16-8)))) = 0x0100<<(pin%(16-8));
		}
		else
		{
			HWREGH(GPIODATA_BASE + GPIO_MASKHIGHBYTE + (pin / 16U) * GPIO_BASE_ADDR_STEP + 4*(0x01<<(pin%(16-8)))) = 0x00;
		}

	}
}

/**
 * @brief     write 0 or 1 to specified GPIO pin
 * 
 * @param [in]   pin is the identifying GPIO number of the pin.
 * @param [in]   outVal is the value to write to the pin.
 * 
 * @details   for example:
 *            *(uint32_t *)(0x40045400 + 0x0004) = 0x0001; => GPIO0=1;
 *            *(uint32_t *)(0x40045408 + 0x0008) = 0x0002; => GPIO1=1;
 *            *(uint32_t *)(0x40045410 + 0x0010) = 0x0004; => GPIO2=1;
 *            *(uint32_t *)(0x40045420 + 0x0020) = 0x0008; => GPIO3=1;
 *            *(uint32_t *)(0x40045440 + 0x0040) = 0x0010; => GPIO4=1;
 *            *(uint32_t *)(0x40045480 + 0x0080) = 0x0020; => GPIO5=1;
 *            *(uint32_t *)(0x40045500 + 0x0100) = 0x0040; => GPIO6=1;
 *            *(uint32_t *)(0x40045600 + 0x0200) = 0x0080; => GPIO7=1;
 *
 *            *(uint32_t *)(0x40045800 + 0x0004) = 0x0100; => GPIO8=1;
 *            *(uint32_t *)(0x40045800 + 0x0008) = 0x0200; => GPIO9=1;
 *            *(uint32_t *)(0x40045800 + 0x0010) = 0x0400; => GPIO10=1;
 *            *(uint32_t *)(0x40045800 + 0x0020) = 0x0800; => GPIO11=1;
 *            *(uint32_t *)(0x40045800 + 0x0040) = 0x1000; => GPIO12=1;
 *            *(uint32_t *)(0x40045800 + 0x0080) = 0x2000; => GPIO13=1;
 *            *(uint32_t *)(0x40045800 + 0x0100) = 0x4000; => GPIO14=1;
 *            *(uint32_t *)(0x40045800 + 0x0200) = 0x8000; => GPIO15=1;
 */
static inline void
GPIO_writePinFast(uint16_t pin, uint16_t outVal)
{
	HWREGH(GPIODATA_BASE + GPIO_MASKLOWBYTE + 0x400*((pin>>3)&0x01) + (pin >> 4U) * GPIO_BASE_ADDR_STEP + 4*(0x01<<(pin&0x07))) = (outVal&0x01)<<(pin&0x0F);
}

/**
 * @brief    write GPIO0~7
 */
static inline void
GPIO_writePin_0_7(uint16_t pin, uint16_t outVal)
{
	HWREGH(GPIODATA_BASE + GPIO_MASKLOWBYTE + 4*(0x01<<(pin&0x07))) = (outVal&0x01)<<(pin&0x0F);
}

/**
 * @brief    write GPIO0~7
 */
static inline void
GPIO_writePin_8_15(uint16_t pin, uint16_t outVal)
{
	HWREGH(GPIODATA_BASE + GPIO_MASKHIGHBYTE + 4*(0x01<<((pin-8)&0x07))) = (outVal&0x01)<<(pin&0x0F);
}

static inline void
GPIO_togglePin(uint16_t pin)
{
	//HWREGH(GPIODATA_BASE + GPIO_DATAOUT + (pin / 16U) * GPIO_BASE_ADDR_STEP) ^= (1 << (pin % 16));

    uint16_t data = (HWREGH(GPIODATA_BASE + GPIO_DATAOUT + (pin / 16U) * GPIO_BASE_ADDR_STEP) >> (pin % 16)) & 0x01;

    GPIO_writePin(pin, data ^ 1);
}

/**
 * 
 *  @brief  Reads the data on the specified port.
 * 
 *  @param [in] port is the GPIO port being accessed in the form of GPIO_PORTX
 *  where X is the port letter.
 * 
 *  @return Returns the value available on pin for the specified port. Each
 *  bit of the the return value represents a pin on the port, where bit 0
 *  represents GPIO port pin 0, bit 1 represents GPIO port pin 1, and so on.
 * 
 */
#if IS_GS32F00xx(0x12)
static inline uint16_t
GPIO_readPortData(GPIO_Port_16BIT port)
{
	uint16_t RegValue;
	if(port<=3)
	{
		RegValue = HWREGH(GPIO_PORTA_BASE + GPIO_DATA + port * GPIO_BASE_ADDR_STEP);
	}
	else if (port == 4)
	{
		RegValue = (HWREGH(GPIO_PORTA_BASE + GPIO_DATA + port * GPIO_BASE_ADDR_STEP))&0x1F;
	}
	else
	{
		RegValue = (((HWREGH(GPIO_PORTA_BASE + GPIO_DATA + (port-10) * GPIO_BASE_ADDR_STEP))&0xFFE0)>>5)
				|(((HWREGH(GPIO_PORTA_BASE + GPIO_DATA + (port-9) * GPIO_BASE_ADDR_STEP))&0x1F)<<11);
	}


    return(RegValue);
}
#elif IS_GS32F3xx(0x22)
static inline uint16_t
GPIO_readPortData(GPIO_Port_16BIT port)
{
	uint16_t RegValue;
	if(port<=9)
	{
		RegValue = HWREGH(GPIO_PORTA_BASE + GPIO_DATA + port * GPIO_BASE_ADDR_STEP);
	}
	else if (port == 10)
	{
		RegValue = (HWREGH(GPIO_PORTA_BASE + GPIO_DATA + port * GPIO_BASE_ADDR_STEP))&0x1FF;
	}
	else if (port == 12)
	{
		RegValue = (((HWREGH(GPIO_PORTA_BASE + GPIO_DATA + (port-2) * GPIO_BASE_ADDR_STEP))&0xFE00)>>3)
				|(((HWREGH(GPIO_PORTA_BASE + GPIO_DATA + (port-1) * GPIO_BASE_ADDR_STEP))&0x7)<<13);
	}
	else if (port == 13 || port == 14)
	{
		RegValue = (((HWREGH(GPIO_PORTA_BASE + GPIO_DATA + (port-2) * GPIO_BASE_ADDR_STEP))&0xFFF8)>>3)
				|(((HWREGH(GPIO_PORTA_BASE + GPIO_DATA + (port-1) * GPIO_BASE_ADDR_STEP))&0x7)<<13);
	}
	else if (port == 15)
	{
		RegValue = ((HWREGH(GPIO_PORTA_BASE + GPIO_DATA + (port-2) * GPIO_BASE_ADDR_STEP))&0x38)>>3;
	}

    return(RegValue);
}
#endif
/**
 *
 *  @brief Writes the data into the specified port.
 * 
 *  @param [in] port is the GPIO port being accessed in the form of GPIO_PORTX
 *  where X is the port letter.
 * 
 *  @param [in] data is the value on pin for the specified port. Each
 *  bit of the the value represents a pin on the port, where bit 0
 *  represents GPIO port pin 0, bit 1 represents GPIO port pin 1, and so on.

 *  @return none
 *
 */
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
static inline void
GPIO_writePortData(GPIO_Port_16BIT port, uint32_t outVal)
{
	HWREGH(GPIO_PORTA_BASE + GPIO_DATAOUT + port * GPIO_BASE_ADDR_STEP) = outVal&0xFFFFU;
    return;
}

static inline uint16_t
GPIO_readPortOut(GPIO_Port_16BIT port)
{
	return HWREGH(GPIO_PORTA_BASE + GPIO_DATAOUT + port * GPIO_BASE_ADDR_STEP);
}

//*****************************************************************************
static inline void
GPIO_setPortPins(GPIO_Port_16BIT port, uint32_t pinMask)
{

     HWREGH(GPIODATA_BASE + GPIO_DATAOUT + port * GPIO_BASE_ADDR_STEP) |= pinMask;

}

//*****************************************************************************
static inline void
GPIO_clearPortPins(GPIO_Port_16BIT port, uint32_t pinMask)
{

	HWREGH(GPIODATA_BASE + GPIO_DATAOUT + port * GPIO_BASE_ADDR_STEP) &= ~pinMask;

}

//*****************************************************************************
static inline void
GPIO_togglePortPins(GPIO_Port_16BIT port, uint32_t pinMask)
{

	HWREGH(GPIODATA_BASE + GPIO_DATAOUT + port * GPIO_BASE_ADDR_STEP) ^= pinMask;

}
#endif
/*****************************************************************************************************************************************************

/**
 * @brief   Enable write on pins
 *
 * @param [in]   GpioGroupBase can be GPIO_GROUP0_BASE, GPIO_GROUP1_BASE, GPIO_GROUP2_BASE, GPIO_GROUP3_BASE
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  None
 */
static inline void
GPIO_enableWrite(uint32_t GpioGroupBase, uint16_t pins){
	
		HWREGH(GpioGroupBase + GPIO_OUTENSET) |= pins;
}

/**
 * @brief   Enable write on one pin
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  None
 */
static inline void
GPIO_enableWritePin(uint16_t pin){
	
		HWREGH(GPIODATA_BASE + GPIO_OUTENSET + (pin / 16U) * GPIO_BASE_ADDR_STEP) |= (1 << (pin % 16));
}

/**
 * @brief   Disable write on pins
 *
 * @param [in]   GpioGroupBase can be GPIO_GROUP0_BASE, GPIO_GROUP1_BASE, GPIO_GROUP2_BASE, GPIO_GROUP3_BASE
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  None
 */
static inline void
GPIO_disableWrite(uint32_t GpioGroupBase, uint16_t pins){
	
		HWREGH(GpioGroupBase + GPIO_OUTENCLR) = pins;
}

/**
 * @brief   Disable write on one pin
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  None
 */
static inline void
GPIO_disableWritePin(uint16_t pin){
	
		HWREGH(GPIODATA_BASE + GPIO_OUTENCLR + (pin / 16U) * GPIO_BASE_ADDR_STEP) = (1 << (pin % 16));
}

/* Read GPIO DataOut register. */
static inline uint16_t
GPIO_getPinDataOut(uint16_t pin)
{
	uint16_t RegValue;
	RegValue = HWREGH(GPIODATA_BASE + GPIO_DATAOUT + (pin / 16U) * GPIO_BASE_ADDR_STEP);
	
	return (((RegValue & (1 << (pin % 16))) > 0)? 1 : 0 );
}


#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
static inline uint16_t
GPIO_readPortDataOut(GPIO_Port_16BIT port)
{
	uint16_t RegValue;
	RegValue = HWREGH(GPIO_PORTA_BASE + GPIO_DATAOUT + port * GPIO_BASE_ADDR_STEP);

    return(RegValue);
}
#endif


/**
 * @brief   Sampled at pin.
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 * 
 * @remarks
 * - Read back value goes through double flip-flop 
 * - synchronization logic with a delay of two cycles.
 *
 * @return  None
 */
static inline uint16_t
GPIO_getPinData(uint16_t pin)
{
	uint16_t RegValue;
	RegValue = HWREGH(GPIODATA_BASE + GPIO_DATA + (pin / 16U) * GPIO_BASE_ADDR_STEP);

	return (((RegValue & (1 << (pin % 16))) > 0)? 1 : 0 );
}




/**
 * @brief   Enable interrupt on pin
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  None
 */
static inline void
GPIO_InterruptEnable(uint16_t pin){
		
		HWREGH(GPIODATA_BASE + GPIO_INTENSET + (pin / 16U) * GPIO_BASE_ADDR_STEP) |= (1 << (pin % 16));
}

/**
 * @brief   Disable interrupt on pin
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  None
 */
static inline void
GPIO_InterruptDisable(uint16_t pin){
		
		HWREGH(GPIODATA_BASE + GPIO_INTENCLR + (pin / 16U) * GPIO_BASE_ADDR_STEP) = (1 << (pin % 16));
}

/**
 * @brief   Read back interrupt status on pin
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  0		Interrupt disabled.
 * 			1		Interrupt enabled.
 */
static inline uint16_t
GPIO_ifInterruptEnabled(uint16_t pin){

		return ((HWREGH(GPIODATA_BASE + GPIO_INTENSET + (pin / 16U) * GPIO_BASE_ADDR_STEP) & (1 << (pin % 16))) != 0);
}

/**
 * @brief   Set interrupt type on pin. (Set interrupt type to falling edge or rising edge.)
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  None
 */
static inline void
GPIO_InterruptTypeSet(uint16_t pin){
		
		HWREGH(GPIODATA_BASE + GPIO_INTTYPESET + (pin / 16U) * GPIO_BASE_ADDR_STEP) |= (1 << (pin % 16));
}
//Same functionality as GPIOInterruptTypeSet()
static inline void
GPIO_setIntTypeEdge(uint16_t pin){

		HWREGH(GPIODATA_BASE + GPIO_INTTYPESET + (pin / 16U) * GPIO_BASE_ADDR_STEP) |= (1 << (pin % 16));
}



/**
 * @brief   Clear interrupt type on pin.(Set to LOW or HIGH level.)
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  None
 */
static inline void
GPIO_InterruptTypeClear(uint16_t pin){
		
		HWREGH(GPIODATA_BASE + GPIO_INTTYPECLR + (pin / 16U) * GPIO_BASE_ADDR_STEP) = (1 << (pin % 16));
}
//Same functionality as GPIO_InterruptTypeClear()
static inline void
GPIO_setIntTypeLevel(uint16_t pin){

		HWREGH(GPIODATA_BASE + GPIO_INTTYPECLR + (pin / 16U) * GPIO_BASE_ADDR_STEP) = (1 << (pin % 16));
}


/**
 * @brief   Set Polarity-level, edge IRQ configuration. (Set to HIGH level or rising edge.)
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  None
 */
static inline void
GPIO_InterruptPolaritySet(uint16_t pin){
		
		HWREGH(GPIODATA_BASE + GPIO_INTPOLSET + (pin / 16U) * GPIO_BASE_ADDR_STEP) |= (1 << (pin % 16));
}
//Same functionality as GPIO_InterruptPolaritySet()
static inline void
GPIO_setIntPolarityRising(uint16_t pin){

		HWREGH(GPIODATA_BASE + GPIO_INTPOLSET + (pin / 16U) * GPIO_BASE_ADDR_STEP) |= (1 << (pin % 16));
}

/**
 * @brief   Clear Polarity-level, edge IRQ configuration. (Set to LOW level or falling edge.) 
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  None
 */
static inline void
GPIO_InterruptPolarityClear(uint16_t pin){
		
		HWREGH(GPIODATA_BASE + GPIO_INTPOLCLR + (pin / 16U) * GPIO_BASE_ADDR_STEP) = (1 << (pin % 16));
}
//Same functionality as GPIO_InterruptPolarityClear
static inline void
GPIO_setIntPolarityFalling(uint16_t pin){

		HWREGH(GPIODATA_BASE + GPIO_INTPOLCLR + (pin / 16U) * GPIO_BASE_ADDR_STEP) = (1 << (pin % 16));
}
/**
 * @brief   Read back Polarity-level, edge IRQ configuration on pin
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  0		For LOW level or falling edge.
 * 			1		For HIGH level or rising edge.
 */
static inline uint16_t
GPIO_getInterruptPolarity(uint16_t pin){
		
		return ((HWREGH(GPIODATA_BASE + GPIO_INTPOLSET + (pin / 16U) * GPIO_BASE_ADDR_STEP) & (1 << (pin % 16))) != 0);
}


/**
 * @brief   Write one to clear interrupt request
 *
 * @param [in]   pin is the identifying GPIO number of the pin.
 *
 * @return  None
 */
static inline void
GPIO_InterruptClear(uint16_t pin){
		
		HWREGH(GPIODATA_BASE + GPIO_INTCLEAR + (pin / 16U) * GPIO_BASE_ADDR_STEP) = (1 << (pin % 16));
}


extern void InitGpio(void);
extern void GPIO_SetupPinMux(Uint16 pin, Uint16 cpu, Uint16 peripheral);
extern void GPIO_SetupPinOptions(uint16_t gpioNumber, uint16_t output, uint16_t flags);
extern uint16_t GPIO_ReadPin(uint16_t gpioNumber);
extern void GPIO_WritePin(uint16_t gpioNumber, uint16_t outVal);
extern void GPIO_setDirectionMode(uint32_t pin, GPIO_Direction pinIO);
extern GPIO_Direction GPIO_getDirectionMode(uint32_t pin);

#endif

#ifdef __cplusplus
}
#endif

#endif /* DEVICE_DRIVERLIB_GPIO_H_ */
