/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    hal_can.c
*   @brief   
*
*/
/*
 * commit history
 * 20240313, LYF, verify for HAL_CAN Send/Receive
 * 20240316, LYF, verify for HAL_CAN Bus error/Bus off/Fifo depth MailBox/Can Nums
 * 20240716, Danne, rewrite functions according to new macros.
 * 20241006, Jason, re-create some API functions and some structure defines.
 */

#ifdef __cplusplus
extern "C"
{
#endif

#include "hal_can.h"

/* ========================================================================== */
/*                            local Variables                                */
/* ========================================================================== */

/* for CAN_A ~ CAN_D global variables. */
//struct CAN_REGS_ONLY CAN_CTRL_R[HAL_CAN_NUMS] = {0};
CAN_MSGx CAN_CTRL_TXBUF[HAL_CAN_NUMS] = {0};
CAN_MSGx CAN_CTRL_RXBUF[HAL_CAN_NUMS] = {0};
/* HAL_CAN virtual registers structure */
HAL_CAN_REGS HAL_CAN_R[HAL_CAN_NUMS] = {0};
/* HAL_CAN Mailbox array for CANA~CAND */
HAL_CAN_MAIL_BOX HAL_CAN_MB[HAL_CAN_NUMS][HAL_TOTAL_MB_NUMS] = {0};

uint8_t HAL_CAN_MsgObjMapToFilterNum[32];
uint32_t HAL_CAN_UsedFilterCnt;

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */
uint32_t HAL_CAN_updateRxMB(uint32_t base);
bool HAL_CAN_updateTxMB(uint32_t base);

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

/*
 * @brief    Check and Handle BUS OFF Status
 *           Poll the BUS_OFF status bit until it's cleared automatically by HW
 */
GS32_DRIVER_HAL_CAN_FUNC_T void
CAN_handleBusOff(uint32_t base)
{
    struct CAN_REGS_ONLY* linkHw = NULL;

    /* Check the arguments. */
    ASSERT(HAL_CAN_isBaseValid(base));
    /* Point to correct ctrol domain reg base. */
    linkHw = PTR_TO_CAN_REGS(base);

    /* clear BUS off status. */
    if(linkHw->CTRL.bit.BUSOFF == CAN_ACTIVED)
    {
        linkHw->CTRL.bit.BUSOFF = CAN_CLEAR;
        while (linkHw->CTRL.bit.BUSOFF == CAN_ACTIVED)
        {
        }
    }
}

GS32_DRIVER_HAL_CAN_FUNC_T void
HAL_CAN_IRQHandler(uint32_t can_base)
{
    uint32_t intFlag;

    /* read interrupt status register. */
    intFlag = CAN_getIntrStatus(can_base);

    /* clear TxOK RxOK bit */
    HAL_CAN_clearTxOKRxOK(can_base);

    /*
     * handle Receive Interrupt Flag
     * If at least one new message received in the RxFIFO, this flag bit is set.
     */
    if (intFlag & CAN_RTIF_RIF_M)
    {
    	/* read RxFIFO and save into RxMB and update related status bits and release RxFIFO slot */
    	HAL_CAN_updateRxMB(can_base);
    }

    /* handle RB Overrun Interrupt Flag, RxFIFO full flag, Rx FIFO almost full flag */
    if (intFlag & (CAN_RTIF_ROIF_M | CAN_RTIF_RFIF_M | CAN_RTIF_RAFIF_M))
    {
        CAN_clearIntrStatus(can_base, CAN_RTIF_ROIF_M | CAN_RTIF_RFIF_M | CAN_RTIF_RAFIF_M);
    }

    /*
     * handle Transmission Primary Interrupt Flag.
     * If one message is sent out via the Primary buffer, this flag bit is set.
     * Normally we don't use the Primary Tx Buffer.
     */
    if (intFlag & CAN_RTIF_TPIF_M)
    {
        CAN_clearIntrStatus(can_base, CAN_RTIF_TPIF_M);
    }

    /*
     * handle Transmission Secondary Interrupt Flag.
     * If at least one message is sent out through TxFIFO, this flag bit is set.
     */
    if (intFlag & CAN_RTIF_TSIF_M)
    {
        HAL_CAN_updateTxMB(can_base);
        CAN_clearIntrStatus(can_base, CAN_RTIF_TSIF_M);
    }

    /*
     * handle Error Interrupt Flag
     */
    if (intFlag & CAN_RTIF_EIF_M)
    {
        /* to check bus off status */
        CAN_handleBusOff(can_base);

        CAN_clearIntrStatus(can_base, CAN_RTIF_EIF_M);
    }

    /* Bus Error Interrupt Flag */
    if (intFlag & CAN_ERRINT_BEIF_M)
    {
        /* to handle bus error status */
        // HAL_CAN_handleBusErr(can_base);
        CAN_clearIntrStatus(can_base, CAN_ERRINT_BEIF_M);
    }
}

GS32_DRIVER_HAL_CAN_FUNC_T void
HAL_CANA_IRQHandler(void)
{
	HAL_CAN_IRQHandler(CANA_BASE);
}

GS32_DRIVER_HAL_CAN_FUNC_T void
HAL_CANB_IRQHandler(void)
{
	HAL_CAN_IRQHandler(CANB_BASE);
}


#if (HAL_CAN_NUMS >= 3U)
GS32_DRIVER_HAL_CAN_FUNC_T void
HAL_CANC_IRQHandler(void)
{
	HAL_CAN_IRQHandler(CANC_BASE);
}
#endif


#if (HAL_CAN_NUMS >= 4U)
GS32_DRIVER_HAL_CAN_FUNC_T void
HAL_CAND_IRQHandler(void)
{
	HAL_CAN_IRQHandler(CAND_BASE);
}
#endif

/*
 * @brief     read CAN_REGS and update virtual registers and mailboxes
 * @para[in]  can_base, CAN base address
 * @details   Read RxFIFO and update HAL_CAN Rx Mailboxes;
 *            If TxFIFO is empty, update HA_CAN Tx related flags (meaning Tx is completed).
 *            This function need be called in CAN ISR or in 1mS task or in background task,
 *            otherwise the virtual mailboxes don't have the correct status and data.
 */
GS32_DRIVER_HAL_CAN_FUNC_T uint32_t
HAL_CAN_updateRegsAndMB(uint32_t can_base)
{
    uint32_t intFlag;

    /* read interrupt status register. */
    intFlag = CAN_getIntrStatus(can_base);

    /* clear virtual ES register TxOK and RxOK bit */
    HAL_CAN_clearTxOKRxOK(can_base);

    /*
     * handle Receive Interrupt Flag
     * If at least one new message received in the RxFIFO, this flag bit is set.
     */
    if (intFlag & CAN_RTIF_RIF_M)
    {
    	HAL_CAN_updateRxMB(can_base);
    }

    /*
     * handle Transmission Primary Interrupt Flag.
     * If one message is sent out via the Primary buffer, this flag bit is set.
     * Normally we don't use the Primary Tx Buffer.
     */
    if (intFlag & CAN_RTIF_TPIF_M)
    {
        CAN_clearIntrStatus(can_base, CAN_RTIF_TPIF_M);
    }

    /*
     * handle Transmission Secondary Interrupt Flag.
     * If at least one message is sent out through TxFIFO, this flag bit is set.
     */
    if (intFlag & CAN_RTIF_TSIF_M)
    {
    	HAL_CAN_updateTxMB(can_base);
        CAN_clearIntrStatus(can_base, CAN_RTIF_TSIF_M);
    }

    /* handle Error Interrupt Flag */
    if (intFlag & CAN_RTIF_EIF_M)
    {
        /* to check bus off status */
        CAN_handleBusOff(can_base);

        CAN_clearIntrStatus(can_base, CAN_RTIF_EIF_M);
    }

    /* Bus Error Interrupt Flag */
    if (intFlag & CAN_ERRINT_BEIF_M)
    {
        // to handle bus error status
        // HAL_CAN_handleBusErr(can_base);
        CAN_clearIntrStatus(can_base, CAN_ERRINT_BEIF_M);
    }

    /* handle RB Overrun Interrupt Flag, RxFIFO full flag, Rx FIFO almost full flag */
    if (intFlag & (CAN_RTIF_ROIF_M | CAN_RTIF_RFIF_M | CAN_RTIF_RAFIF_M))
    {
        CAN_clearIntrStatus(can_base, CAN_RTIF_ROIF_M | CAN_RTIF_RFIF_M | CAN_RTIF_RAFIF_M);
    }
    return 0;
}

/*
 * @brief Setup a mail box
 * @param[in] base
 * @param[in] objID, mail box ID, 0~31 or 0~63
 * @param[in] msgID, CAN message ID, 0~0x7FF or 0~0x1FFFFFFF
 * @param[in] idType, standard ID (11bit length) or extended ID (29bit length)
 * @param[in] msgType, mailbox type Transmit or Receive
 * @param[in] msgIDMask, the filter mask for this mail box,
 *            bit10~bit0 are used if it's STD ID
 *            bit28~bit0 are used if it's XTD ID
 * @param[in] flags, Filter with standard ID, extended ID, direction filter
 *            It's combination of the following flags:
 *            CAN_MSG_OBJ_TX_INT_ENABLE - enable Tx Interrupt for Tx mailbox;
 *            CAN_MSG_OBJ_RX_INT_ENABLE - enable Rx Interrupt for Rx mailbox;
 *            CAN_MSG_OBJ_USE_ID_FILTER - enable ID filter and mask;
 *                                        if no this flag, ID must be exactly matched instead of using ID mask;
 *            CAN_MSG_OBJ_USE_EXT_FILTER - enable EXT filter, it's no use in HAL_CAN;
 *            CAN_MSG_OBJ_USE_DIR_FILTER - enable DIR filter, it's no use in HAL_CAN;
 * @param[in] msgLen, message length in bytes, for non-FD message;
 *                    DLC value for FD message, CAN_DATA_LENGTH_0 ~ CAN_DATA_LENGTH_64;
 */
GS32_DRIVER_HAL_CAN_FUNC_T void
HAL_CAN_setupMessageObject(uint32_t base,
		               uint32_t objID, uint32_t msgID,
					   CAN_MsgFrameType idType, HAL_CAN_MsgObjType msgType,
                       uint32_t msgIDMask, uint32_t flags, uint16_t msgLen)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
    HAL_CAN_MAIL_BOX *pHalCanMB = pHalCanRegs->pMailBox;

    uint32_t maskReg = 0U;
    uint32_t msgCtrl = 0U;

    pHalCanMB += objID;

    /* Check the arguments. */
    ASSERT(HAL_CAN_isBaseValid(base));
    ASSERT((objID >= 0U) && (objID < HAL_TOTAL_MB_NUMS));

    /* Step1: configure the ID register (arbitration register) */
    /* Configuration for the Mai;Box. */
    pHalCanMB->ID.bit.xtd = idType;
    pHalCanMB->ID.bit.ID = msgID;
    pHalCanMB->ID.bit.MsgVal = 1;

    switch (msgType)
    {
        case HAL_CAN_MSG_OBJ_TYPE_TX:
        	pHalCanMB->ID.bit.Dir = HAL_CAN_MSG_TYPE_TRANSMIT;
            break;

        case HAL_CAN_MSG_OBJ_TYPE_TX_REMOTE:
        	pHalCanMB->ID.bit.Dir = HAL_CAN_MSG_TYPE_TRANSMIT;
        	msgCtrl |= HAL_CAN_IF1MCTL_RMTEN;
            break;

        case HAL_CAN_MSG_OBJ_TYPE_RX:
        	pHalCanMB->ID.bit.Dir = HAL_CAN_MSG_TYPE_RECEIVE;
            break;

            /* this type is not supported in the driver. */
        case HAL_CAN_MSG_OBJ_TYPE_RXTX_REMOTE:
        	pHalCanMB->ID.bit.Dir = HAL_CAN_MSG_TYPE_RECEIVE;
        	msgCtrl |= HAL_CAN_IF1MCTL_RMTEN;
            break;

        default:
            break;
    }

    /* Step2: configure the mask register */
    /* ID mask registers value */
    maskReg = msgIDMask;
    /* If the caller wants to filter on the extended ID bit then set it. */
	maskReg |= (flags & CAN_MSG_OBJ_USE_EXT_FILTER);
	/* The caller wants to filter on the message direction field. */
	maskReg |= (flags & CAN_MSG_OBJ_USE_DIR_FILTER);

    pHalCanMB->Mask.all = maskReg;

    /* Step3: configure the ctrl register */
    /* If any filtering is requested, set the UMASK bit to use mask register */
    if(((flags & CAN_MSG_OBJ_USE_ID_FILTER) |
        (flags & CAN_MSG_OBJ_USE_DIR_FILTER) |
        (flags & CAN_MSG_OBJ_USE_EXT_FILTER)) != 0U)
    {
        msgCtrl |= HAL_CAN_IF1MCTL_UMASK;
    }

    /*
     * Set the data length for the transfers. This is applicable only for
     * Tx mailboxes. For Rx mailboxes, dlc is updated on receving a frame.
     */
    if((msgType == CAN_MSG_OBJ_TYPE_TX) ||
        (msgType == CAN_MSG_OBJ_TYPE_RXTX_REMOTE))
    {
        msgCtrl |= ((uint32_t)msgLen & HAL_CAN_IF1MCTL_DLC_M);
    }

    /* Enable transmit interrupts if they should be enabled. pHalCanMB->Ctrl.bit.TxIE=1; */
    msgCtrl |= (flags & CAN_MSG_OBJ_TX_INT_ENABLE);

    /* Enable receive interrupts if they should be enabled. pHalCanMB->Ctrl.bit.RxIE=1; */
    msgCtrl |= (flags & CAN_MSG_OBJ_RX_INT_ENABLE);

    pHalCanMB->Ctrl.all = msgCtrl;

    if(flags & HAL_CAN_MSG_ENABLE_FDF)
    	pHalCanMB->Ctrl.bit.FDF = 1;
    else
    	pHalCanMB->Ctrl.bit.FDF = 0;

    if(flags & HAL_CAN_MSG_ENABLE_BRS)
    	pHalCanMB->Ctrl.bit.BRS = 1;
    else
    	pHalCanMB->Ctrl.bit.BRS = 0;

    /*
     * Configure message ID filter for the mailbox:
     * If it's Tx type mailbox, no need to configure the filter and need disable the
     * corresponding filter if it's ever used as Rx mailbox.
     * If it's Rx type mailbox, need configure the filter and set the pMB->filterNum.
     */
    uint32_t id_filter_type;
    if((HAL_CAN_MSG_OBJ_TYPE_TX == msgType) || (HAL_CAN_MSG_OBJ_TYPE_TX_REMOTE == msgType))
    {
        HAL_CAN_disableMsgFilter(base, objID);
    }
    else if(HAL_CAN_MSG_OBJ_TYPE_RX == msgType)
    {
    	/* EXT bit is used for filter, only one type of STD or EXT id can be accepted by this MB */
    	if(flags & CAN_MSG_OBJ_USE_EXT_FILTER)
    	{
        	if(CAN_MSG_FRAME_STD == idType)
        		id_filter_type = CAN_IDTYPE_STD;
        	else if(CAN_MSG_FRAME_EXT == idType)
        		id_filter_type = CAN_IDTYPE_EXT;
    	}
    	/* Both STD and EXT id can be accepted by this MB */
    	else
    		id_filter_type = CAN_IDTYPE_STD_OR_EXT;

    	/*
    	 * IDE bit is not used for filter (Both STD and EXT ID can be accepted).
    	 * Disable this feature at this moment: only one type of STD or EXT is accepted by a mailbox.
    	 * It's not supported for one mailbox to accept both STD and EXT ID.
    	 */
    	//if((flags & CAN_MSG_OBJ_USE_EXT_FILTER) == 0)
    	//	id_filter_type = CAN_IDTYPE_STD_OR_EXT;

    	HAL_CAN_setupMsgFilter(base, objID, id_filter_type);
    }

    /* Clear NewDat/IntPnd bit in MB and clear NDAT_21/IPEN_21 in HAL_CAN_R. */
    HAL_CAN_releaseMB(base, objID);
}

/*
 * @brief      Read CAN message from RBUF and save into pRxMsgBuf and mailbox
 * @param[in]  base, CAN base address.
 *
 * @return     true, new message received and updated to mail box
 *             false, no new message
 * @details    "HAL_CAN_lookupObjIDbyMsgID" takes about 360~420 CPU cycles.
 *             Newly received messages are stored into matching mailbox according to ID filter rule.
 *             Mailbox related status bits are updated
 */
GS32_DRIVER_HAL_CAN_FUNC_T uint32_t
HAL_CAN_updateRxMB(uint32_t base)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
	uint32_t recevied_msg_cnt = 0;
	uint32_t read_time;

    CAN_MSGx *pRxMsgBuf = pHalCanRegs->pCanRxBuf;
    HAL_CAN_MAIL_BOX *pCanMBBase = pHalCanRegs->pMailBox;
    HAL_CAN_MAIL_BOX *pCanMB = pCanMBBase;
    uint32_t objID = 0U;
    uint32_t msgID = 0U;

    /* Check the arguments. */
    ASSERT(HAL_CAN_isBaseValid(base));

    for(read_time=0; read_time<HAL_CAN_MAX_READ_TIMES; read_time++)
    {
		//if(!(CAN_getIntrStatus(base) & CAN_RTIF_RIF_M))
    	if(CAN_getRxFifoStatus(base) == CAN_FIFO_EMPTY)
			break;

		/* Read CAN message from RBUF and save into pRxMsgBuf */
		HAL_CAN_readRxBuf(base, pRxMsgBuf);
		CAN_releaseRB(base);

		/* Read CAN ID */
		msgID = HAL_CAN_getMsgIDFromBuf(pRxMsgBuf);

		/* Find the objID by message CAN ID. */
		objID = HAL_CAN_lookupObjIDbyMsgID(pHalCanRegs, msgID);

		/* If the CAN ID can match a Rx Mailbox, save the message into corresponding mailbox. */
		if (objID >= 0U && objID < HAL_TOTAL_MB_NUMS)
		{

			pCanMB = pCanMBBase + objID;

			/*
			 * If ID type filtering is enabled and ID type doesn't match, drop this message.
			 * If received STD ID but acceptance is EXT ID, drop this message.
			 * If received EXT ID but acceptance is STD ID, drop this message.
			 */
			if((pCanMB->Mask.bit.Mxtd) && (pRxMsgBuf->CTRL.bit.IDE != pCanMB->ID.bit.xtd))
				continue;

			/*
			 * NewData bit for the mailbox is set, meaning previous received data is not processed yet.
			 * Increment the lost frame count and set message lost flag.
			 */
			if(pCanMB->Ctrl.bit.NewDat)
			{
				pCanMB->lostFrameCnt++;
				pCanMB->Ctrl.bit.MsgLst = 1;
			}

			/* Update data and id and other bits from CAN RBUF to specified MailBox */
			pCanMB->Data[0].all = pRxMsgBuf->Data[0].all;
			pCanMB->Data[1].all = pRxMsgBuf->Data[1].all;

			pCanMB->ID.bit.ID = msgID;
			pCanMB->Ctrl.bit.DLC = pRxMsgBuf->CTRL.bit.DLC;
			pCanMB->Ctrl.bit.RmtEn = pRxMsgBuf->CTRL.bit.RTR;
			pCanMB->ID.bit.xtd = pRxMsgBuf->CTRL.bit.IDE;

#if HAL_CAN_ENABLE_FD
			pCanMB->Ctrl.bit.BRS = pRxMsgBuf->CTRL.bit.BRS;
			pCanMB->Ctrl.bit.FDF = pRxMsgBuf->CTRL.bit.FDF;
			if(pCanMB->Ctrl.bit.FDF)
			{
				uint32_t nSize = CAN_getMessageLength(pRxMsgBuf->CTRL.bit.DLC);
				for(uint32_t i=2; i<nSize/4; i++)
					pCanMB->Data[i].all = pRxMsgBuf->Data[i].all;
			}
#endif

			/* update control and status register */
			pCanMB->Ctrl.bit.NewDat = 1;
#if HAL_CAN_ENABLE_64_MB
			if(objID < 32)
				pHalCanRegs->CAN_NDAT_21.all |= (1<<objID);
			else if(objID < 64)
				pHalCanRegs->CAN_NDAT_32_63 |= (1<<(objID-32));
#else
			pHalCanRegs->CAN_NDAT_21.all |= (1<<objID);
#endif

			/* received frame count incremented */
			pCanMB->rxFrameCnt++;

			/* set status flag bit, a combination of NewDat bit and IntPnd bit */
			HAL_CAN_setMailBoxStatus(pHalCanRegs, objID);
			recevied_msg_cnt++;

			/* RxOK bit means at least one message is correctly received */
			pHalCanRegs->CAN_ES.bit.RxOK = 1;

			/* update interrupts related registers */
			if(pCanMB->Ctrl.bit.RxIE)
				HAL_CAN_setIntPnd(pHalCanRegs, objID);
		}
		else  /* received message doesn't match any Rx ID */
		{
			/* increment the discarded messages count - messages not accepted by any Rx MailBox. */
			pHalCanRegs->RxDiscardMsgNum ++;
		}
    }
    CAN_clearIntrStatus(base, CAN_RTIF_RIF_M);
    return recevied_msg_cnt;
}

/*
 * @brief      Read CAN TIF and TxFIFO empty status, and update Tx Mail Box status (tx completed)
 * @param[in]  base
 *
 * @return     true, Tx FIFO empty, Tx Mail Box status is updated
 *             false, Tx FIFO is not empty, Tx Mail Box cannot be updated yet
 * @details    this function takes about 275~306 CPU cycles with O1 and ILM
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_updateTxMB(uint32_t base)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];

    HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox;
    uint32_t objID = 0U;

    /*
     * must wait till TxFIFO is empty,
     * the software can update Tx Mailbox status (mark them as transmission completed)
     */
    //if(!CAN_checkTbufEmpty(base))
    if(CAN_getTxFifoStatus(base) != CAN_FIFO_EMPTY)
    	return false;

    /* if TxFIFO is empty, we assume all the Tx mailbox transmission has been completed. */
	for(objID=0; objID<HAL_TOTAL_MB_NUMS; objID++)
	{
        if((pMB->ID.bit.MsgVal) && (pMB->ID.bit.Dir == HAL_CAN_MSG_TYPE_TRANSMIT))
        {
        	if(pMB->Ctrl.bit.TxRqst)
        	{
        	    pMB->Ctrl.bit.TxRqst = 0;    //mark the Tx has completed
#if HAL_CAN_ENABLE_64_MB
        	    if(objID < 32)
        	    {
        	    	pHalCanRegs->CAN_TXRQ_21.all &= (~(1<<objID));
					pHalCanRegs->HAL_CAN_MB_Status[0] |= (1<<objID);
        	    }
        	    else if(objID < 64)
        	    {
        	    	pHalCanRegs->CAN_TXRQ_32_63 &= (~(1<<(objID-32)));
					pHalCanRegs->HAL_CAN_MB_Status[1] |= (1<<(objID-32));
        	    }
#else
        	    pHalCanRegs->CAN_TXRQ_21.all &= (~(1<<objID));
        	    pHalCanRegs->HAL_CAN_MB_Status[0] |= (1<<objID);
#endif
        	    pMB->txDoneFrameCnt++;
        	    pHalCanRegs->CAN_ES.bit.TxOK = 1; //at least one frame Tx completed

        	    if(pMB->Ctrl.bit.TxIE)
        	    	HAL_CAN_setIntPnd(pHalCanRegs, objID);
        	}
        }
        pMB++;
	}
	return TRUE;
}

/*
 * @brief      Sends a Message Object
 *
 * @param[in]  base, CAN base address
 * @param[in]  objID, mailbox ID, 0~31 or 0~63
 * @param[in]  msgLen, the message length in bytes.
 *             this length value is updated Ctrl.bit.DLC of the selected MB.
 * @param[in]  msgData, the pointer to the message data.
 *             it's uint16_t type pointer, but for each data in the array, only lower 8bit is used for transmission.
 *
 * @note       The message object requested by the objID must first be setup
 *             using the HAL_CAN_setupMessageObject() function.
 *
 * @return     true, if Message was sent success;
 *             false, to indicate the Message was not written into Tx Buffer or Tx FIFO.
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_sendMessage(uint32_t base, uint32_t objID, uint16_t msgLen, const uint16_t *msgData)
{
    bool ret = FALSE;

    ret = HAL_CAN_loadMailBoxDataOnly(base, objID, msgData, msgLen);

    if(false == ret)
    	return ret;

    ret = HAL_CAN_sendMailBox(base, objID);

    return ret;
}

/*
 * @brief    it's same as HAL_CAN_sendMessage
 *           The only difference is that the input parameter msgData, each data in the array is 8bit.
 *           So msgLen is 8 max for CAN2.0 and is 64 max for CANFD.
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_sendMessage_8bit(uint32_t base, uint32_t objID, uint16_t msgLen, const uint8_t *msgData)
{
    bool ret = FALSE;

    ret = HAL_CAN_loadMailBoxDataOnly_8bit(base, objID, msgData, msgLen);

    if(false == ret)
    	return ret;

    ret = HAL_CAN_sendMailBox(base, objID);

    return ret;
}

/*
 * @brief    it's same as HAL_CAN_sendMessage
 *           The only difference is that the input parameter msgData, each data in the array is 16bit.
 *           All 16bit are transmitted. So msgLen is 4 max for CAN2.0 and is 32 max for CANFD.
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_sendMessage_16bit(uint32_t base, uint32_t objID, uint16_t msgLen, const uint16_t *msgData)
{
    bool ret = FALSE;

    ret = HAL_CAN_loadMailBoxDataOnly_16bit(base, objID, msgData, msgLen);

    if(false == ret)
    	return ret;

    ret = HAL_CAN_sendMailBox(base, objID);

    return ret;
}

/*
 * @brief    it's same as HAL_CAN_sendMessage
 *           The only difference is that the input parameter msgData, each data in the array is 32bit.
 *           All 32bit are transmitted. So msgLen is 2 max for CAN2.0 and is 16 max for CANFD.
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_sendMessage_32bit(uint32_t base, uint32_t objID, uint16_t msgLen, const uint32_t *msgData)
{
    bool ret = FALSE;

    ret = HAL_CAN_loadMailBoxDataOnly_32bit(base, objID, msgData, msgLen);

    if(false == ret)
    	return ret;

    ret = HAL_CAN_sendMailBox(base, objID);

    return ret;
}

/*
 * @brief    Load MsgID and data into a transmit mailbox
 * @para[in] base, CAN base address.
 * @para[in] objID, the mailbox number, 0~63
 * @para[in] msgID, the message ID, 0~0x7FF or 0~0x1FFFFFFF;
 * @para[in] pSrcData, pointer to the source data to be sent.
 * @para[in] nDLC, this is the Data Length Code for CAN message.
 * @note     source data is 16bit type but only lower 8bit of each data is used.
 *           so length is max 8 for CAN2.0 and max 64 for CANFD.
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_loadMailBox(uint32_t base, uint32_t objID, uint32_t msgID, uint16_t *pMsgData, uint32_t nDLC)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
	HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox;
	bool ret;
	uint32_t nLength; /* number of data bytes */

	pMB += objID;

	ASSERT((nDLC >= 0U) && (nDLC < 16U));
#if HAL_CAN_ENABLE_FD
	nLength = CAN_getMessageLength(nDLC);
#else
	if (nDLC > 8U)
		nDLC = 8U;
	nLength = nDLC;
#endif

	if((pMB->ID.bit.MsgVal) && (pMB->ID.bit.Dir == HAL_CAN_MSG_TYPE_TRANSMIT))
	{
		uint8_t *msgData = (uint8_t *)&(pMB->Data[0]);
		uint32_t idx;
		for(idx=0; idx < nLength; ++idx)
		{
			*msgData = *pMsgData;
			msgData++;
			pMsgData++;
		}
		pMB->Ctrl.bit.DLC = nDLC;
	    pMB->ID.bit.ID = msgID;
	    pMB->Ctrl.bit.NewDat = 1;
	    HAL_CAN_clearMailBoxStatus(pHalCanRegs, objID);
	    ret = true;
	}
	else
		ret = false;
	return ret;
}

/*
 * @brief    Load data and DLC into a transmit mailbox.
 * @para[in] base, CAN base address.
 * @para[in] objID, the mailbox number.
 * @para[in] pSrcData, pointer to the source data to be sent.
 * @para[in] nDLC, this is the Data Length Code for CAN message.
 * @note     source data is 16bit type but only lower 8bit of each data is used.
 *           so length is max 8 for CAN2.0 and max 64 for CANFD.
 *           nDLC = CAN_DATA_LENGTH_0 ~ CAN_DATA_LENGTH_64.
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_loadMailBoxDataOnly(uint32_t base, uint32_t objID, const uint16_t *pSrcData, uint32_t nDLC)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
	HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox;
	bool ret;
	uint32_t nLength; /* number of data bytes */

	pMB += objID;

	ASSERT((nDLC >= 0U) && (nDLC < 16U));
#if HAL_CAN_ENABLE_FD
	nLength = CAN_getMessageLength(nDLC);
#else
	if (nDLC > 8U)
		nDLC = 8U;
	nLength = nDLC;
#endif

	if((pMB->ID.bit.MsgVal) && (pMB->ID.bit.Dir == HAL_CAN_MSG_TYPE_TRANSMIT))
	{
		uint8_t *msgData = (uint8_t *)&(pMB->Data[0]);
		uint32_t idx;
		for(idx=0; idx < nLength; ++idx)
		{
			*msgData = (uint8_t)(*pSrcData);
			msgData++;
			pSrcData++;
		}
		pMB->Ctrl.bit.DLC = nDLC;
	    pMB->Ctrl.bit.NewDat = 1;
	    HAL_CAN_clearMailBoxStatus(pHalCanRegs, objID);
	    ret = true;
	}
	else
		ret = false;

	return ret;
}

/*
 * @brief   Load data and data length into a transmit mailbox
 * @note    source data is u8 type, so length is max 8 for CAN2.0 and max 64 for CANFD.
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_loadMailBoxDataOnly_8bit(uint32_t base, uint32_t objID, const uint8_t *pSrcData,uint32_t nDLC)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
	HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox;
	bool ret;
	uint32_t nSize; /* number of data bytes */

	pMB += objID;

	ASSERT((nDLC >= 0U) && (nDLC < 16U));
#if HAL_CAN_ENABLE_FD
	nSize = CAN_getMessageLength(nDLC);
#else
	if (nDLC > 8U)
		nDLC = 8U;
	nSize = nDLC;
#endif

	if((pMB->ID.bit.MsgVal) && (pMB->ID.bit.Dir == HAL_CAN_MSG_TYPE_TRANSMIT))
	{
		/*
		uint8_t *msgData = (uint8_t *)&(pMB->Data[0]);
		uint32_t idx;
		for(idx=0; idx < nSize; ++idx)
		{
			*msgData = (uint8_t)(*pSrcData);
			msgData++;
			pSrcData++;
		}*/
		memcpy((void*)&(pMB->Data[0]), (void*)pSrcData, nSize);

		pMB->Ctrl.bit.DLC = nDLC;
	    pMB->Ctrl.bit.NewDat = 1;
	    HAL_CAN_clearMailBoxStatus(pHalCanRegs, objID);
	    ret = true;
	}
	else
		ret = false;

	return ret;
}

#if 1
/*
 * These two functions are removed because they can be replaced by HAL_CAN_loadMailBoxDataOnly_8bit
 */
/*
 * @brief   Load data and data length into a transmit mailbox
 * @note    source data is u16 type, so length is max 4 for CAN2.0 and max 32 for CANFD
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_loadMailBoxDataOnly_16bit(uint32_t base, uint32_t objID, const uint16_t *pSrcData,uint32_t nDLC)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
	HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox;
	bool ret;
	uint32_t nSize; /* number of data bytes */

	pMB += objID;

	ASSERT((nDLC >= 0U) && (nDLC < 16U));
#if HAL_CAN_ENABLE_FD
	nSize = CAN_getMessageLength(nDLC);
#else
	if (nDLC > 8U)
		nDLC = 8U;
	nSize = nDLC;
#endif

	if((pMB->ID.bit.MsgVal) && (pMB->ID.bit.Dir == HAL_CAN_MSG_TYPE_TRANSMIT))
	{
		/*
		uint16_t *msgData = (uint16_t *)&(pMB->Data[0]);
		uint32_t idx;
		for(idx=0; idx < nLength; ++idx)
		{
			*msgData = (uint16_t)(*pSrcData);
			msgData++;
			pSrcData++;
		}*/

		memcpy((void*)&(pMB->Data[0]), (void*)pSrcData, nSize);
		pMB->Ctrl.bit.DLC = nDLC;
	    pMB->Ctrl.bit.NewDat = 1;
	    HAL_CAN_clearMailBoxStatus(pHalCanRegs, objID);
	    ret = true;
	}
	else
		ret = false;

	return ret;
}

/*
 * @brief     Load data and data length into a transmit mailbox
 * @para[in]  base, CAN base address.
 * @para[in]  objID, 0~31 or 0~63
 * @para[in]  pSrcData, each data is u32 type.
 * @para[in]  nLength, for CAN2.0 max 2 data and for CANFD max 16 data.
 * @return
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_loadMailBoxDataOnly_32bit(uint32_t base, uint32_t objID, const uint32_t *pSrcData,uint32_t nDLC)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
	HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox;
	bool ret;
	uint32_t nSize; /* number of data bytes */

	pMB += objID;

	ASSERT((nDLC >= 0U) && (nDLC < 16U));
#if HAL_CAN_ENABLE_FD
	nSize = CAN_getMessageLength(nDLC);
#else
	if (nDLC > 8U)
		nDLC = 8U;
	nSize = nDLC;
#endif

	/* if MB is valid, MB type is Tx, MB is not in transmission state. */
	if((pMB->ID.bit.MsgVal) && (pMB->ID.bit.Dir == HAL_CAN_MSG_TYPE_TRANSMIT)) /*  && (pMB->Ctrl.bit.TxRqst==0) */
	{
		/*
		uint32_t *msgData = (uint32_t *)&(pMB->Data[0]);
		uint32_t idx;
		for(idx=0; idx < nLength; ++idx)
		{
			*msgData = (uint32_t)(*pSrcData);
			msgData++;
			pSrcData++;
		}*/

		memcpy((void*)&(pMB->Data[0]), (void*)pSrcData, nSize);
		pMB->Ctrl.bit.DLC = nDLC;
	    pMB->Ctrl.bit.NewDat = 1;
	    HAL_CAN_clearMailBoxStatus(pHalCanRegs, objID);
	    ret = true;
	}
	else
		ret = false;

	return ret;
}
#endif

/*
 * @brief     load the specified MB into TBUF and send out this message
 * @para[in]  base, CAN base address
 * @para[in]  objID, 0~31 or 0~63.
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_sendMailBox(uint32_t base, uint32_t objID)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
	HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox;

    /* Check the arguments. */
    ASSERT(HAL_CAN_isBaseValid(base));
    ASSERT((objID >= 0U) && (objID < HAL_TOTAL_MB_NUMS));

    pMB += objID;

    /* The mailbox should be Tx type */
    if(pMB->ID.bit.Dir != CAN_MSG_TYPE_TRANSMIT)
    	return false;

    /* If no new data is loaded into this MB, quit and return false. */
    if(pMB->Ctrl.bit.NewDat == 0)
    	return false;

    /* If STB(TxFIFO) is full, quit and return false */
    if(CAN_getTxFifoStatus(base) == CAN_FIFO_FULL)
    	return false;

    /* Select STB(TxFIFO) instead of PTB */
    CAN_selectSTB(base);

    /* Copy data/ID/DLC from MailBox to CAN TBUF. */
    HAL_CAN_writeMBtoTBUF(base, pMB);

    /* mark the current Tx slot as filled and switch to next slot */
    CAN_switchNextSTBSlot(base);

    /* let all Messages in TxFIFO be sent out */
    CAN_setTransmitAll(base);

    /* Update mailbox' status bits */
    pMB->txLaunchFrameCnt++;
	pMB->Ctrl.bit.NewDat = 0;
	pMB->Ctrl.bit.TxRqst = 1;
	if(objID < 32)
	{
		pHalCanRegs->CAN_TXRQ_21.all |= (1<<objID);
		pHalCanRegs->HAL_CAN_MB_Status[0] &= (~(1<<objID));
	}
#if HAL_CAN_ENABLE_64_MB
	else if(objID < 64)
	{
		pHalCanRegs->CAN_TXRQ_32_63 |= (1<<(objID-32));
		pHalCanRegs->HAL_CAN_MB_Status[1] &= (~(1<<(objID-32)));
	}
#endif

	//clear Tx interrupt pending register
	if(pMB->Ctrl.bit.TxIE)
		HAL_CAN_clearIntPnd(pHalCanRegs, objID);

    return true;
}

/*
 * @brief      check the objID MailBox NewDat bit, and copy data from MB to pointed buffer
 *
 * @param[in]  base is the base address of the HAL_CAN controller.
 * @param[in]  objID is the MB number to read, 0~31 or 0~63.
 * @param[out] msgData is a pointer to the array to store the message data
 *
 * @note
 * -# The message object requested by the \e objID must first be setup
 *    using the HAL_CAN_setupMessageObject() function.
 * -# If the DLC of the received message is larger than the \e msgData
 *    buffer provided, then it is possible for a buffer overflow to occur.
 *
 * @return     true if new data retrieved; false if no data read.
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_readMessage(uint32_t base, uint32_t objID, uint16_t *msgData)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
	HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox;
    bool ret = false;
    uint32_t idx = 0U;
    uint32_t dlc = 8U;
    uint8_t *ptrSrc;
    uint16_t *ptrDst;

    /* Check the arguments. */
    ASSERT(HAL_CAN_isBaseValid(base));
    ASSERT(msgData != NULL);
    ASSERT((objID >= 0U) && (objID < HAL_TOTAL_MB_NUMS));

    /* Get the specified MailBox pointer */
    pMB += objID;

    /* If MB is not valid, or MB is not Rx type, return false. */
    if((!pMB->ID.bit.MsgVal) || (pMB->ID.bit.Dir != HAL_CAN_MSG_TYPE_RECEIVE))
    	return false;

    /* if new data bit is set */
    if (pMB->Ctrl.bit.NewDat == TRUE)
    {
    	dlc = pMB->Ctrl.bit.DLC;
    	ptrSrc = (uint8_t *)(&pMB->Data[0].all);
    	ptrDst = (uint16_t *)(&msgData[0]);
        /* Copy data from MB to Msg data buffer. */
        for (idx = 0; idx < dlc; idx++)
        {
        	*ptrDst = (*ptrSrc);
        	ptrSrc++;
        	ptrDst++;
        }

        /*
         * release this MailBox,
         * same as HAL_CAN_releaseMB clear NewDat bit, IntPnd bit, NDAT_21, IPEN_21;
         */
        pMB->Ctrl.bit.NewDat = FALSE;
        if(objID < 32)
        {
            pHalCanRegs->CAN_NDAT_21.all &= (~(1<<objID));
            pHalCanRegs->HAL_CAN_MB_Status[0] &= (~(1<<objID));
        }
        else if(objID < 64)
        {
            pHalCanRegs->CAN_NDAT_32_63 &= (~(1<<(objID-32)));
            pHalCanRegs->HAL_CAN_MB_Status[0] &= (~(1<<(objID-32)));
        }
        if(pMB->Ctrl.bit.RxIE)
        {
        	pMB->Ctrl.bit.IntPnd = FALSE;
        	if(objID < 32)
        		pHalCanRegs->CAN_IPEN_21.all &= (~(1<<objID));
        	else if(objID <64)
        		pHalCanRegs->CAN_IPEN_32_63 &= (~(1<<(objID-32)));
        }

        ret = true;
    }
    else
    {
        ret = false;
    }

    return(ret);
}

/*
 * @brief      check the objID MailBox NewDat bit, and copy data from MB to pointed buffer
 *
 * @param[in]  base is the base address of the HAL_CAN controller.
 * @param[in]  objID is the MB number to read, 0~31 or 0~63.
 * @param[out] msgData is a pointer to the array to store the message data, uint8_t type array
 *
 * @note
 * -# The message object requested by the \e objID must first be setup
 *    using the HAL_CAN_setupMessageObject() function.
 * -# If the DLC of the received message is larger than the \e msgData
 *    buffer provided, then it is possible for a buffer overflow to occur.
 *
 * @return     true if new data retrieved; false if no data read.
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_readMessage_8bit(uint32_t base, uint32_t objID, uint8_t *msgData)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
	HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox;
    bool ret = false;
    uint32_t idx = 0U;
    uint32_t dlc = 8U;
    uint32_t *ptrSrc;
    uint32_t *ptrDst;

    /* Check the arguments. */
    ASSERT(HAL_CAN_isBaseValid(base));
    ASSERT(msgData != NULL);
    ASSERT((objID >= 0U) && (objID < HAL_TOTAL_MB_NUMS));

    /* Get the specified MailBox pointer */
    pMB += objID;

    /* If MB is not valid, or MB is not Rx type, return false. */
    if((!pMB->ID.bit.MsgVal) || (pMB->ID.bit.Dir != HAL_CAN_MSG_TYPE_RECEIVE))
    	return false;

    /* if new data bit is set */
    if (pMB->Ctrl.bit.NewDat == TRUE)
    {
    	dlc = pMB->Ctrl.bit.DLC;
    	ptrSrc = (uint32_t *)(&pMB->Data[0].all);
    	ptrDst = (uint32_t *)msgData;
    	/* Copy data from MB to Msg data buffer. */
#if HAL_CAN_ENABLE_FD
    	uint32_t nSize = CAN_getMessageLength(dlc)/4;
    	for(idx=0; idx<nSize; idx++)
    	{
    		*ptrDst = *ptrSrc;
    		ptrDst++;
    		ptrSrc++;
    	}
#else
    	*ptrDst = pMB->Data[0].all;
    	ptrDst++;
    	*ptrDst = pMB->Data[1].all;
#endif

        /*
         * release this MailBox,
         * same as HAL_CAN_releaseMB clear NewDat bit, IntPnd bit, NDAT_21, IPEN_21;
         */
        pMB->Ctrl.bit.NewDat = FALSE;
        if(objID < 32)
        {
            pHalCanRegs->CAN_NDAT_21.all &= (~(1<<objID));
            pHalCanRegs->HAL_CAN_MB_Status[0] &= (~(1<<objID));
        }
        else if(objID < 64)
        {
            pHalCanRegs->CAN_NDAT_32_63 &= (~(1<<(objID-32)));
            pHalCanRegs->HAL_CAN_MB_Status[0] &= (~(1<<(objID-32)));
        }
        if(pMB->Ctrl.bit.RxIE)
        {
        	pMB->Ctrl.bit.IntPnd = FALSE;
        	if(objID < 32)
        		pHalCanRegs->CAN_IPEN_21.all &= (~(1<<objID));
        	else if(objID <64)
        		pHalCanRegs->CAN_IPEN_32_63 &= (~(1<<(objID-32)));
        }

        ret = true;
    }
    else
    {
        ret = false;
    }

    return(ret);
}

/*
 * @brief     Read CAN message with ID
 *
 * @para[in]  base, CAN base address
 * @para[in]  objID, specify the mailbox number from which read a CAN message, 0~31 or 0~63
 * @para[out] frameType, the received message ID type, ext or std
 * @para[out] msgID, the received message ID.
 * @para[out] msgData, the received message data array.
 * @return    true, if read a new message.
 *            false, if no new message read.
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_readMessageWithID(uint32_t base,
                           uint32_t objID,
                           CAN_MsgFrameType *frameType,
                           uint32_t *msgID,
                           uint16_t *msgData)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
	HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox;

    bool status;

    /* Check the pointers. */
    ASSERT(msgID != 0U);
    ASSERT(frameType != 0U);

    /*
     * Read the message first this fills the IF2 registers
     * with received message for that mailbox
     */
    status = HAL_CAN_readMessage(base, objID, msgData);
    /* See if there is new data available. */
    if(status)
    {
    	pMB += objID;
        /* if it's extended ID */
        if(pMB->ID.bit.xtd)
        {
            *frameType = CAN_MSG_FRAME_EXT;
            *msgID = pMB->ID.bit.ID & 0x1FFFFFFF;
        }
        /* if it's standard ID */
        else
        {
            *frameType = CAN_MSG_FRAME_STD;
            // *msgID = (pMB->ID.bit.ID >> 18) & 0x7FF;
            //note, this is different from the original DCAN, std ID in bit10~bit0 vs in bit28~bit18;
            *msgID = (pMB->ID.bit.ID & 0x7FF);
        }
    }

    return(status);
}

/*
 * @brief    Copy CAN message from CAN RBUF to MailBox
 */
GS32_DRIVER_HAL_CAN_FUNC_T void
HAL_CAN_updateMsgBuftoMB(HAL_CAN_MAIL_BOX* pMB, CAN_MSGx *pCanBuf)
{
	if(pMB->ID.bit.Dir == HAL_CAN_MSG_TYPE_RECEIVE)
	{
		pMB->Ctrl.bit.DLC = pCanBuf->CTRL.bit.DLC;
		pMB->Ctrl.bit.NewDat = 1;
		pMB->ID.bit.ID = pCanBuf->ID.bit.ID;
		pMB->ID.bit.xtd = pCanBuf->CTRL.bit.IDE;
		pMB->Data[0].all = pCanBuf->Data[0].all;
		pMB->Data[1].all = pCanBuf->Data[1].all;
	}
}

/*
 * @brief    Copy CAN message from MailBox to CAN TBUF
 */
GS32_DRIVER_HAL_CAN_FUNC_T void
HAL_CAN_updateMBtoMsgBuf(CAN_MSGx *pCanBuf, HAL_CAN_MAIL_BOX* pMB)
{
	if(pMB->ID.bit.Dir == HAL_CAN_MSG_TYPE_TRANSMIT)
	{
		pCanBuf->CTRL.bit.DLC = pMB->Ctrl.bit.DLC;
		pMB->Ctrl.bit.NewDat = 0;
		pCanBuf->ID.bit.ID = pMB->ID.bit.ID;
		pCanBuf->CTRL.bit.IDE = pMB->ID.bit.xtd;
		pCanBuf->Data[0].all = pMB->Data[0].all;
		pCanBuf->Data[1].all = pMB->Data[1].all;
	}
}

/*
 * @brief     Go through all 32 mailbox to match the specified CAN_ID (msgID)
 *
 * @para[in]  pHalCanRegs, pointer to a HalCan structure.
 * @para[in]  msgID, the 11bit or 28bit ID to be accepted by a mailbox
 *
 * @return    objID, the first MailBox ID which matches the specified CAN_ID
 */
GS32_DRIVER_HAL_CAN_FUNC_T uint32_t
HAL_CAN_lookupObjIDbyMsgID(HAL_CAN_REGS *pHalCanRegs, uint32_t msgID)
{
    uint32_t i;
    uint32_t objID;
    HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox;

    //go through all mail box to find out the Rx Mail Box with matched ID
    for (objID = 0U; objID < HAL_TOTAL_MB_NUMS; objID++)
    {
    	if((pMB->ID.bit.MsgVal) && (pMB->ID.bit.Dir == HAL_CAN_MSG_TYPE_RECEIVE))  //check only valid mail box, and only Rx type mail box
    	{
			if(pMB->Ctrl.bit.UMask) //if ID mask is used
			{
				if ((pMB->ID.bit.ID & pMB->Mask.all) == (msgID & pMB->Mask.all))
				{
					break;
				}
			}
			else  //mask is not used, msg ID must be exactly matched
			{
				if (pMB->ID.bit.ID == msgID)
					break;
			}
    	}
        pMB++;
    }
    return objID;
}

/*
 * @brief  Initialize HAL_CAN global variables.
 *         It must be called during HAL_CAN initialization.
 */
GS32_DRIVER_HAL_CAN_FUNC_T void
HAL_CAN_initCfg(uint32_t base)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];

    HAL_CAN_R[can_index].pCanTxBuf = &CAN_CTRL_TXBUF[can_index];
    HAL_CAN_R[can_index].pCanRxBuf = &CAN_CTRL_RXBUF[can_index];
    HAL_CAN_R[can_index].pMailBox = &HAL_CAN_MB[can_index][0];
    HAL_CAN_R[can_index].HAL_CAN_UserFunc_BusOff_ISR = NULL;
    HAL_CAN_R[can_index].HAL_CAN_UserFunc_RX_ISR = NULL;
    HAL_CAN_R[can_index].HAL_CAN_UserFunc_TX_ISR = NULL;
    HAL_CAN_R[can_index].HAL_CAN_UserFunc_Error_ISR = NULL;
}

/*
 * @brief  setup the message filter for specified Mailbox
 * @param[in]  base,  CAN base address
 * @param[in]  objID, mailbox ID, 0~31
 * @param[in]  id_type, CAN_IDTYPE_STD_OR_EXT = 0, both STD or EXT ID is accepted
 *                      CAN_IDTYPE_STD = 2, accept STD ID only
 *                      CAN_IDTYPE_EXT = 3, accept EXT ID only
 * @details    There are only 16 filters hardware.
 *             Filter_0 is enabled by default after reset and accept all IDs.
 *             pMB->filterNum is used to link to a filter number 1~15.
 *             If all filters are occupied,
 *             the remaining mailboxes need additional function calls to combine the IDs into Filter_0
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_setupMsgFilter(uint32_t base, uint32_t objID, uint32_t id_type)
{
	uint32_t filter_num;
	uint32_t can_id, id_mask;

    ASSERT(HAL_CAN_isBaseValid(base));
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
	HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox + objID;

	can_id = pMB->ID.bit.ID;
	/* Note:
	 * Mask register bits usage:
	 * GS32_CAN 1=ID filter disabled, 0=ID filter enabled
	 * D-CAN    1=ID filter enabled,  1=ID filter disabled
	 * UMask bit usage:
	 * UMASK = 1, use MASK bits for filtering;
	 * UMASK = 0, each bit of ID is checked, and Mask is not used.
	 */
	if(pMB->Ctrl.bit.UMask)
	    id_mask = (~(pMB->Mask.bit.Msk))&0x1FFFFFFF;
	else
		id_mask = (~0x1FFFFFFF);


	filter_num = pMB->filterNum;
	/* if the Mailbox is not linked to any filter. */
	if((filter_num == 0) || (filter_num > 15))
		filter_num = CAN_getAvailableMsgFilterNum(base);

	if((filter_num > 0) && (filter_num < 16))
	{
		CAN_setMsgFilter(base, filter_num, can_id, id_mask, id_type);
		pMB->filterNum = filter_num;
		return TRUE;
	}
	else
	{
		/* Enable Filter_0 which accepts all IDs */
		CAN_setMsgFilter(base, 0, 0, 0x1FFFFFFF, CAN_IDTYPE_STD_OR_EXT);
		CAN_enableMsgFilter(base, 0);
	}
	return FALSE;
}

/*
 * @brief  Set ID filter and mask for the specified filter number
 * @para[in]  filter_num filter number
 * @para[in]  can_id   CAN ID to be accepted
 * @para[in]  mask     CAN ID mask, 0=filter enable, 1=filter disable
 * @para[in]  id_type  CAN ID type, 2=STD, 3=EXT, 0=STD or EXT
 * @note      Filter and Mask can be written only when CAN is in RESET state
 */
GS32_DRIVER_HAL_CAN_FUNC_T void
CAN_setMsgFilter(uint32_t baseAddr, uint32_t filter_num, uint32_t can_id, uint32_t id_mask, CAN_IDType id_type)
{
    uint32_t can_index;
    can_index = (baseAddr-CANA_BASE)/(CANB_BASE-CANA_BASE);

    if(filter_num >= ACF_NUMBER)
    	filter_num = ACF_NUMBER-1;

    uint32_t acfCtrl;
    uint32_t acode;

    /* Set ACFCTRL register to select filter */
    acfCtrl = HWREG(baseAddr + CAN_O_ACFCTRL) & 0xFFFFFF00;
    acfCtrl |= (filter_num & CAN_ACFCTRL_ACFADR_M);
    HWREG(baseAddr + CAN_O_ACFCTRL) = acfCtrl;

    /* Set filter CAN ID and ID type*/
    //acode = (can_id & CAN_ACF_ACODE_X_AMASK_X_M) | (id_type << CAN_ACF_3_AIDE_S);
    acode = (can_id & CAN_ACF_ACODE_X_AMASK_X_M);
    HWREG(baseAddr + CAN_O_ACF_0) = acode;

    /* Set ACFCTRL register to select filter mask */
    acfCtrl |= (CAN_ACFCTRL_SELMASK_M);
    HWREG(baseAddr + CAN_O_ACFCTRL) = acfCtrl;

    /* Configure Filterx ID Mask*/
    acode = (id_mask & CAN_ACF_ACODE_X_AMASK_X_M) | (id_type << CAN_ACF_3_AIDE_S);
    HWREG(baseAddr + CAN_O_ACF_0) = acode;

    /* enable the newly added filter */
    acfCtrl |= (1 << (filter_num + CAN_ACF_EN_0_AE_0_S));
    HWREG(baseAddr + CAN_O_ACFCTRL) = acfCtrl;
}


/*
 * @brief   Disable the message filter for specified Mailbox
 */
GS32_DRIVER_HAL_CAN_FUNC_T void
HAL_CAN_disableMsgFilter(uint32_t base, uint32_t objID)
{
	uint32_t filter_num;
	uint32_t can_id, id_mask;

    ASSERT(HAL_CAN_isBaseValid(base));
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
	HAL_CAN_REGS *pHalCanRegs = &HAL_CAN_R[can_index];
	HAL_CAN_MAIL_BOX *pMB = pHalCanRegs->pMailBox + objID;

	filter_num = pMB->filterNum;
	if((filter_num > 0) && (filter_num < 16))
	{
		CAN_disableMsgFilter(base, filter_num);
		pMB->filterNum = 0;
	}
}

/*
 * @brief  Manually set the HAL_CAN controller bit timing.
 *
 * @param  base is the base address of the HAL_CAN controller.
 * @param  prescaler is the baud rate prescaler
 * @param  tSeg1 is the time segment 1
 * @param  tSeg2 is the time segment 2
 * @param  sjw is the synchronization jump width
 *
 * This function sets the various timing parameters for the HAL_CAN bus bit
 * timing: baud rate prescaler, prescaler extension, time segment 1,
 * time segment 2, and the Synchronization Jump Width.
 *
 * @return  None.
 */
GS32_DRIVER_HAL_CAN_FUNC_T void
HAL_CAN_setBitTiming(uint32_t base, uint16_t prescaler,
                 uint16_t tSeg1, uint16_t tSeg2, uint16_t sjw)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
    HAL_CAN_REGS *hal_link = &HAL_CAN_R[can_index];

    uint16_t savedInit;
    uint32_t bitReg;

    /* Check the arguments. */
    ASSERT(HAL_CAN_isBaseValid(base));
    ASSERT((prescaler >= 0U) && (prescaler < 0xFFU));
    ASSERT((tSeg1 > 0U) && (tSeg1 < 0xFFU));
    ASSERT((tSeg2 > 0U) && (tSeg2 < 0x7FU));
    ASSERT((sjw > 0U) && (sjw < 0x7FU));

    /*
     * To set the bit timing register, the controller must be placed in init
     * mode (if not already), and also configuration change bit enabled.
     * State of the init bit should be saved so it can be restored at the end.
     */
    savedInit = hal_link->CAN_CTL.all;
    hal_link->CAN_CTL.all = savedInit | HAL_CAN_CTL_INIT | HAL_CAN_CTL_CCE;

    /* Set the bit fields of the bit timing register */
    hal_link->CAN_BTR.bit.SEG1 = tSeg1;
    hal_link->CAN_BTR.bit.SEG2 = tSeg2;
    hal_link->CAN_BTR.bit.SJW = sjw;
    hal_link->CAN_BTR.bit.PRESC = prescaler;

    /* Clear the config change bit, and restore the init bit. */
    savedInit &= ~((uint16_t)HAL_CAN_CTL_CCE);

    hal_link->CAN_CTL.all = savedInit;
}

/*
 * @brief Sets the HAL_CAN Bit Timing based on the requested Bit Rate.
 *
 * @param base is the base address of the HAL_CAN controller.
 * @param clockFreq is the HAL_CAN module clock frequency before the bit rate
 *        prescaler (Hertz).
 * @param bitRate is the desired bit rate (bits/sec).
 * @param bitTime is the number of time quanta per bit required for the desired
 *        bit time (Tq) and must be in the range from 8 to 255.
 *
 * This function sets the HAL_CAN bit timing values for the bit rate passed in
 * the \e bitRate and \e bitTime parameters based on the \e clockFreq
 * parameter. The HAL_CAN bit clock is calculated to be an average timing value
 * that should work for most systems. If tighter timing requirements are needed,
 * then the HAL_CAN_setBitTiming() function is available for full customization
 * of all HAL_CAN bit timing values.
 *
 * @note Not all bit-rate and bit-time combinations are valid.
 *       For combinations that would yield the correct bit-rate,
 *       refer to the DCAN_HAL_CANBTR_values.xlsx file in the "docs" directory.
 *       The HAL_CANBTR register values calculated by the function HAL_CAN_setBitRate
 *       may not be suitable for your network parameters. If this is the case
 *       and you have computed the correct values for your network, you could
 *       directly write those parameters in HAL_CANBTR register using the
 *       function HAL_CAN_setBitTiming.
 *
 * @return None.
 */
GS32_DRIVER_HAL_CAN_FUNC_T void
HAL_CAN_setBitRate(uint32_t base, uint32_t clockFreq, uint32_t bitRate,
               uint16_t bitTime)
{
    uint32_t brp;
    uint32_t tSeg1;
    uint32_t tSeg2;
    uint32_t sjw;
    uint32_t prescaler;
    float ftseg2 = 0.0f;
    float sampRate = 0.0f;

    /* Check the arguments. */
    ASSERT(HAL_CAN_isBaseValid(base));
    ASSERT(bitRate <= 1000000U);

    /* Set the sampling rate based on the requested bit rate. */
    if (bitRate > 800000U)
        sampRate = 0.750f;
    else if (bitRate > 500000U)
        sampRate = 0.800f;
    else
        sampRate = 0.875f;

    /* Calculate bit timing values. */
    brp = clockFreq / (bitRate * bitTime);
    tSeg2 = (uint32_t)(bitTime * (1.0f - sampRate) + 0.5f);
    tSeg1 = bitTime - tSeg2;

    if(tSeg2 > 4U)
    {
        sjw = 3U;
    }
    else
    {
        sjw = tSeg2;
    }

    /* Adjust the timing values to fit the CAN register. */
    prescaler = brp - 1U;
    tSeg1 -= 2U;
    tSeg2 -= 1U;
    sjw -= 1U;

    /* Set the calculated timing parameters. */
    HAL_CAN_setBitTiming(base, prescaler, tSeg1, tSeg2, sjw);
}


/*
 * @brief Manually set the HAL_CAN Data bit timing.
 *
 * @param base is the base address of the HAL_CAN controller.
 * @param prescaler is the baud rate prescaler
 * @param tSeg1 is the time segment 1
 * @param tSeg2 is the time segment 2
 * @param sjw is the synchronization jump width
 *
 * This function sets the various timing parameters for the HAL_CAN bus bit
 * timing: baud rate prescaler, prescaler extension, time segment 1,
 * time segment 2, and the Synchronization Jump Width.
 *
 * @return None.
 */
GS32_DRIVER_HAL_CAN_FUNC_T void
HAL_CAN_setDataBitTiming(uint32_t base, uint16_t prescaler,
                 uint16_t tSeg1, uint16_t tSeg2, uint16_t sjw)
{
	uint32_t can_index = (base-CANA_BASE)/(CANB_BASE-CANA_BASE);
    HAL_CAN_REGS *hal_link = &HAL_CAN_R[can_index];

    uint16_t savedInit;
    uint32_t bitReg;

    /* Check the arguments. */
    ASSERT(HAL_CAN_isBaseValid(base));
    ASSERT((prescaler >= 0U) && (prescaler < 0xFFU));
    ASSERT((tSeg1 > 0U) && (tSeg1 < 0x1FU));
    ASSERT((tSeg2 > 0U) && (tSeg2 < 0xFU));
    ASSERT((sjw > 0U) && (sjw < 0xFU));

    /* Set the bit fields of the bit timing register */
    hal_link->CAN_DTR.bit.SEG1 = tSeg1;
    hal_link->CAN_DTR.bit.SEG2 = tSeg2;
    hal_link->CAN_DTR.bit.SJW = sjw;
    hal_link->CAN_DTR.bit.PRSEC = prescaler;

}


/*
 * @brief Sets the HAL_CAN Bit Data(CANFD) Timing based on the requested Bit Rate.
 *
 * @param base is the base address of the HAL_CAN controller.
 * @param clockFreq is the HAL_CAN module clock frequency before the bit rate
 *        prescaler (Hertz).
 * @param bitRate is the desired bit rate (bits/sec).
 * @param bitTime is the number of time quanta per bit required for the desired
 *        bit time (Tq) and must be in the range from 8 to 255.
 *
 * This function sets the HAL_CAN bit timing values for the bit rate passed in
 * the \e bitRate and \e bitTime parameters based on the \e clockFreq
 * parameter. The HAL_CAN bit clock is calculated to be an average timing value
 * that should work for most systems. If tighter timing requirements are needed,
 * then the HAL_CAN_setBitTiming() function is available for full customization
 * of all HAL_CAN bit timing values.
 *
 * @note Not all bit-rate and bit-time combinations are valid.
 *       For combinations that would yield the correct bit-rate,
 *       refer to the DCAN_HAL_CANBTR_values.xlsx file in the "docs" directory.
 *       The HAL_CANBTR register values calculated by the function HAL_CAN_setBitRate
 *       may not be suitable for your network parameters. If this is the case
 *       and you have computed the correct values for your network, you could
 *       directly write those parameters in HAL_CANBTR register using the
 *       function HAL_CAN_setBitTiming.
 *
 * @return None.
 */
GS32_DRIVER_HAL_CAN_FUNC_T void
HAL_CAN_setDataBitRate(uint32_t base, uint32_t clockFreq, uint32_t bitRate,
		                    uint16_t bitTime)
{
    uint32_t brp;
    uint32_t tSeg1;
    uint32_t tSeg2;
    uint32_t sjw;
    uint32_t prescaler;
    float ftseg2 = 0.0f;
    float sampRate = 0.0f;

    /* Check the arguments. */
    ASSERT(HAL_CAN_isBaseValid(base));
    ASSERT(bitRate <= 1000000U);

    /* Set the sampling rate based on the requested bit rate. */
    if (bitRate > 800000U)
        sampRate = 0.750f;
    else if (bitRate > 500000U)
        sampRate = 0.800f;
    else
        sampRate = 0.875f;

    /* Calculate bit timing values. */
    brp = clockFreq / (bitRate * bitTime);
    tSeg2 = (uint32_t)(bitTime * (1.0f - sampRate) + 0.5f);
    tSeg1 = bitTime - tSeg2;

    if(tSeg2 > 4U)
    {
        sjw = 3U;
    }
    else
    {
        sjw = tSeg2;
    }

    /* Adjust the timing values to fit the CAN register. */
    prescaler = brp - 1U;
    tSeg1 -= 2U;
    tSeg2 -= 1U;
    sjw -= 1U;

    /* Set the calculated timing parameters. */
    HAL_CAN_setDataBitTiming(base, prescaler, tSeg1, tSeg2, sjw);
}

/*
 * @brief   Read the message CAN ID at the top of RxFIFO,
 *          Check whether it matches the specified CAN ID,
 *          and NOT release this Rx Slot, leaving it to be read by API afterwards
 * @param   base, CAN BASE address
 *          can_id, reference can id to compare with.
 */
GS32_DRIVER_HAL_CAN_FUNC_T bool
HAL_CAN_checkRxMessageId(uint32_t base, uint32_t can_id)
{
	if(CAN_getRxFifoStatus(base) == CAN_FIFO_EMPTY)
		return 0;

	CAN_MSGx* pCanRbuf = NULL;

	/* Point to rx buf domain reg base. */
	pCanRbuf = PTR_TO_CAN_RXBUF(base);

	/* Get the RX ID, compare it with the reference CAN ID. */
	if(pCanRbuf->ID.bit.ID == can_id)
		return TRUE;
	else
		return FALSE;
}

#ifdef __cplusplus
}
#endif
