/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "hrpwm.h"

#if IS_GS32F00xx(0x30)
//*****************************************************************************
//
//Up-Down Mode High-precision duty cycle Enable
//
//*****************************************************************************
HRPWM_UpDownDutyModeEnable_Extend HRPWM_UpDownDutyModeEnable_Num = HRPWM_Disabled_ALL;
#endif

//*****************************************************************************
//
//HRPWM Mstep default value
//
//*****************************************************************************
#ifdef DEVICE_SYSCLK_FREQ
	#if IS_GS32F00xx(0x30)
		#ifdef DEVICE_AHBCLK_DIV
		const uint16_t GS32_HRPWM_DEFAULT_HRMSTEP = 1.0*1000*1000*1000/(DEVICE_SYSCLK_FREQ/DEVICE_AHBCLK_DIV)*1000/150;
		#else
		const uint16_t GS32_HRPWM_DEFAULT_HRMSTEP = 1.0*1000*1000*1000/(DEVICE_SYSCLK_FREQ/2)*1000/150;
		#endif
	#elif IS_GS32F3xx(0x22)
		#ifdef DEVICE_AHBCLK_DIV
		const uint16_t GS32_HRPWM_DEFAULT_HRMSTEP = 1.0*1000*1000*1000/(DEVICE_SYSCLK_FREQ/DEVICE_AHBCLK_DIV)*1000/110;
		#else
		const uint16_t GS32_HRPWM_DEFAULT_HRMSTEP = 1.0*1000*1000*1000/(DEVICE_SYSCLK_FREQ/2)*1000/110;
		#endif
	#endif
#else
        const uint16_t GS32_HRPWM_DEFAULT_HRMSTEP = 0;
#endif



#if IS_GS32F00xx(0x30)

//*****************************************************************************
//
// HRPWM_enableUpDownInit_Compatible
//
//*****************************************************************************
GS32_DRIVER_HRPWM_FUNC_T void HRPWM_enableUpDownInit_Compatible(HRPWM_UpDownDutyModeEnable_Extend HRPWM_StmEnabledNum)
{
	HRPWM_UpDownDutyModeEnable_Num |= HRPWM_StmEnabledNum;

	for(uint32_t i = 0,HRPWM_StmEnable_idx = HRPWM_StmEnabledNum;
		HRPWM_StmEnable_idx != 0 && i < sizeof(uint32_t)*8;
		i++,HRPWM_StmEnable_idx >>= 1
	)
	{
		if(HRPWM_StmEnable_idx&0x01)
		{
			HWREGH( EPWM1_BASE+0x800*i + EPWM_O_CMPSEL) |= 0x5;
			HWREGH( EPWM1_BASE+0x800*i + EPWM_O_CMPSEL) |= 0x5;
		}

	}

}

//*****************************************************************************
//
// HRPWM_disableUpDownInit_Compatible
//
//*****************************************************************************
GS32_DRIVER_HRPWM_FUNC_T void HRPWM_disableUpDownInit_Compatible(HRPWM_UpDownDutyModeEnable_Extend HRPWM_StmEnabledNum)
{
	HRPWM_UpDownDutyModeEnable_Num &= (~HRPWM_StmEnabledNum);

	for(uint32_t i = 0,HRPWM_StmEnable_idx = HRPWM_StmEnabledNum;
		HRPWM_StmEnable_idx != 0 && i < sizeof(uint32_t)*8;
		i++,HRPWM_StmEnable_idx >>= 1
	)
	{
		if(HRPWM_StmEnable_idx&0x01)
		{
			HWREGH( EPWM1_BASE+0x800*i + EPWM_O_CMPSEL) &= (~0x5);
			HWREGH( EPWM1_BASE+0x800*i + EPWM_O_CMPSEL) &= (~0x5);
		}

	}
}


//*****************************************************************************
//
// HRPWM_setCounterCompareValueInt_Compatible
//
//*****************************************************************************
GS32_DRIVER_HRPWM_FUNC_T void HRPWM_setCounterCompareValueInt_Compatible(uint32_t base, EPWM_CounterCompareModule compModule,uint16_t compCount)
{
		 uint32_t registerOffset;

		 //
		 // Check the arguments
		 //
		 ASSERT(EPWM_isBaseValid(base));

		 //
		 // Get the register offset for the Counter compare
		 //
		 registerOffset = EPWM_O_CMPA + (uint32_t)compModule;

		//Check whether the macro is active

		if( 1<<((base - EPWM1_BASE)/(EPWM2_BASE - EPWM1_BASE)) & HRPWM_UpDownDutyModeEnable_Num )
		{

			 if(compModule == EPWM_COUNTER_COMPARE_A)
			 {

				 HWREGH(base + EPWM_O_CMPA + EPWM_COUNTER_COMPARE_A + 0x2U) = compCount;

				 HWREGH(base + EPWM_O_CMPA + EPWM_COUNTER_COMPARE_B + 0x2U) = compCount+1;

			 }else if(compModule == EPWM_COUNTER_COMPARE_B)
			 {
				 HWREGH(base + EPWM_O_CMPA1 + EPWM_COUNTER_COMPARE_A1 + 0x2U) = compCount;

				 HWREGH(base + EPWM_O_CMPA1 + EPWM_COUNTER_COMPARE_B1 + 0x2U) = compCount+1;
			 }

		}
}

//*****************************************************************************
//
// HRPWM_setHiResCounterCompareValueOnly_Compatible
//
//*****************************************************************************
GS32_DRIVER_HRPWM_FUNC_T void HRPWM_setHiResCounterCompareValueOnly_Compatible(uint32_t base,
								   HRPWM_CounterCompareModule compModule,
								   uint16_t hrCompCount)
{
		ASSERT(HRPWM_isBaseValid(base));
		ASSERT(hrCompCount < 256U);

		uint16_t  HR_mstep = 256;
		uint16_t  SetCMPHR_RED, SetCMPHR_FED;
	    uint8_t   hr_int_FED = 0,hr_int_RED = 0;
	    uint8_t Clock_Ratio = HWREGH(0x400CF720)? HWREGH(0x400CF720)*4 : 2;

		//Check whether the macro is active

		if( 1<<((base - EPWM1_BASE)/(EPWM2_BASE - EPWM1_BASE)) & HRPWM_UpDownDutyModeEnable_Num )
		{

			//Auto Mode
			if( !(HWREGH(base + HRPWM_O_HRCNFG) & HRPWM_HRCNFG_AUTOCONV) )
			{
				if( HWREGB(base + HRPWM_O_HRMSTEP) )
				{
					HR_mstep = HWREGB(base + HRPWM_O_HRMSTEP);
				}else
				{
					HR_mstep = GS32_HRPWM_DEFAULT_HRMSTEP;
				}

				SetCMPHR_RED = (float)hrCompCount/256.0*HR_mstep;
				SetCMPHR_FED = HR_mstep - hrCompCount;

			}else
			{
				SetCMPHR_RED = hrCompCount;
				SetCMPHR_FED = HR_mstep - hrCompCount;
			}

			hr_int_RED = (float)SetCMPHR_RED / (float)HR_mstep * Clock_Ratio;
			hr_int_FED = (float)SetCMPHR_FED / (float)HR_mstep * Clock_Ratio;
		    if(hr_int_RED == Clock_Ratio){
		    	hr_int_RED -= 1;
		    }
		    if(hr_int_FED == Clock_Ratio){
		    	hr_int_FED -= 1;
		    }

			//critical condition
			if(!(SetCMPHR_RED%HR_mstep) && SetCMPHR_RED)
			{
				SetCMPHR_RED -= 1;

			}else if( !(SetCMPHR_FED%HR_mstep) && SetCMPHR_FED)
			{
				SetCMPHR_FED -= 1;
			}


			//zoom factor
			if( !(HWREGH(base + HRPWM_O_HRCNFG) & HRPWM_HRCNFG_AUTOCONV) )
			{
				SetCMPHR_RED  *= (8/Clock_Ratio);
				SetCMPHR_FED  *= (8/Clock_Ratio);
			}

			 if(compModule == HRPWM_COUNTER_COMPARE_A)
			 {
				if( HWREGH(base + EPWM_O_AQCTLA) == 0x0420 )
				{
					HWREGH(base + HRPWM_O_CMPA) = SetCMPHR_RED << 8U;
					HWREGH(base + HRPWM_O_CMPB) = SetCMPHR_FED << 8U;

				}else if( HWREGH(base + EPWM_O_AQCTLA) == 0x0810 )
				{
					HWREGH(base + HRPWM_O_CMPA) = SetCMPHR_FED << 8U;
					HWREGH(base + HRPWM_O_CMPB) = SetCMPHR_RED << 8U;
				}

			 }else if(compModule == HRPWM_COUNTER_COMPARE_B)
			 {
				if( HWREGH(base + EPWM_O_AQCTLB) == 0x0420 )
				{
					HWREGH(base + HRPWM_O_CMPA1) = SetCMPHR_RED << 8U;
					HWREGH(base + HRPWM_O_CMPB1) = SetCMPHR_FED << 8U;

				}else if( HWREGH(base + EPWM_O_AQCTLB) == 0x0810 )
				{
					HWREGH(base + HRPWM_O_CMPA1) = SetCMPHR_FED << 8U;
					HWREGH(base + HRPWM_O_CMPB1) = SetCMPHR_RED << 8U;
				}

			 }

			// Not Auto
			if( !(HWREGH(base + HRPWM_O_HRCNFG) & HRPWM_HRCNFG_AUTOCONV) )
			{
				 if(compModule == HRPWM_COUNTER_COMPARE_A)
				 {
					HWREG(base + EPWM_O_HRPWM_INT) = (HWREG(base + EPWM_O_HRPWM_INT)&0xFFFFFFF0)| hr_int_RED;

					HWREG(base + EPWM_O_HRPWM_INT) = (HWREG(base + EPWM_O_HRPWM_INT)&0xFFFFFF0F)| (hr_int_FED<<4);

				 }else if(compModule == HRPWM_COUNTER_COMPARE_B)
				 {
					HWREG(base + EPWM_O_HRPWM_INT) = (HWREG(base + EPWM_O_HRPWM_INT)&0xFFFF0FFF)| (hr_int_RED<<12);

					HWREG(base + EPWM_O_HRPWM_INT) = (HWREG(base + EPWM_O_HRPWM_INT)&0xFFF0FFFF)| (hr_int_FED<<16);
				 }


			}

		}

}


//*****************************************************************************
//
// HRPWM_setCounterCompareValue_Compatible
//
//*****************************************************************************
GS32_DRIVER_HRPWM_FUNC_T void HRPWM_setCounterCompareValue_Compatible(uint32_t base,
						  HRPWM_CounterCompareModule compModule,
						  uint32_t compCount)
{
		uint16_t	temp_f,temp_i;
		uint32_t	compCount_RED,compCount_FED,compCount_Temp;
		uint16_t	HR_mstep = 256;
	    uint8_t		hr_int_FED = 0,hr_int_RED = 0,hr_int_Temp;
	    uint8_t		Clock_Ratio = HWREGH(0x400CF720)? HWREGH(0x400CF720)*4 : 2;
	    uint8_t		AQCTL_Base = (compModule == HRPWM_COUNTER_COMPARE_A)?EPWM_O_AQCTLA:EPWM_O_AQCTLB;

		//Check whether the macro is active

		if( 1<<((base - EPWM1_BASE)/(EPWM2_BASE - EPWM1_BASE)) & HRPWM_UpDownDutyModeEnable_Num )
		{
			temp_i = compCount & 0xFFFFFF00;
			temp_f = compCount & 0x000000FF;

			if( !(HWREGH(base + HRPWM_O_HRCNFG) & HRPWM_HRCNFG_AUTOCONV) )
			{
				if( HWREGB(base + HRPWM_O_HRMSTEP) )
				{
					HR_mstep = HWREGB(base + HRPWM_O_HRMSTEP);
				}else
				{
					HR_mstep = GS32_HRPWM_DEFAULT_HRMSTEP;
				}

				compCount_RED = ((float32_t)temp_f/256.0f)*(float32_t)HR_mstep;
				compCount_FED = HR_mstep - compCount_RED;

				hr_int_RED = (float)compCount_RED / (float)HR_mstep * Clock_Ratio;
				hr_int_FED = (float)compCount_FED / (float)HR_mstep * Clock_Ratio;

			    if(hr_int_RED == Clock_Ratio){
			    	hr_int_RED -= 1;
			    }
			    if(hr_int_FED == Clock_Ratio){
			    	hr_int_FED -= 1;
			    }

			}else
			{
				compCount_RED = temp_f;
				compCount_FED = HR_mstep - temp_f;
			}


			if(!(compCount_RED%HR_mstep) && compCount_RED)
			{
				compCount_RED -= 1;

			}else if( !(compCount_FED%HR_mstep) && compCount_FED)
			{
				compCount_FED -= 1;
			}

			if( !(HWREGH(base + HRPWM_O_HRCNFG) & HRPWM_HRCNFG_AUTOCONV) )
			{
				compCount_RED  *= (8/Clock_Ratio);
				compCount_FED  *= (8/Clock_Ratio);
			}

			if( HWREGH(base + AQCTL_Base) == 0x0420 )//UP-H Down-L
			{
				compCount_RED |= temp_i ;
				compCount_FED |= ((temp_i>>8)+1)<<8 ;

			}else if( HWREGH(base + AQCTL_Base) == 0x0810 )//UP-L Down-H
			{
				compCount_RED |= ((temp_i>>8)+1)<<8 ;
				compCount_FED |=  temp_i;
				compCount_Temp = compCount_RED;
				compCount_RED  = compCount_FED;
				compCount_FED  = compCount_Temp;

			}

			if( compModule == HRPWM_COUNTER_COMPARE_A )
			{


				HWREG(base + HRPWM_O_CMPA)  = compCount_RED << 8U;
				HWREG(base + HRPWM_O_CMPB)  = compCount_FED << 8U;

			}else if( compModule == HRPWM_COUNTER_COMPARE_B )
			{

				HWREG(base + HRPWM_O_CMPA1)  = compCount_RED << 8U;
				HWREG(base + HRPWM_O_CMPB1)  = compCount_FED << 8U;
			}

			// Not Auto
			if( !(HWREGH(base + HRPWM_O_HRCNFG) & HRPWM_HRCNFG_AUTOCONV) )
			{
				 if(compModule == HRPWM_COUNTER_COMPARE_A)
				 {
					HWREG(base + EPWM_O_HRPWM_INT) = (HWREG(base + EPWM_O_HRPWM_INT)&0xFFFFFFF0)| hr_int_RED;

					HWREG(base + EPWM_O_HRPWM_INT) = (HWREG(base + EPWM_O_HRPWM_INT)&0xFFFFFF0F)| (hr_int_FED<<4);

				 }else if(compModule == HRPWM_COUNTER_COMPARE_B)
				 {
					HWREG(base + EPWM_O_HRPWM_INT) = (HWREG(base + EPWM_O_HRPWM_INT)&0xFFFF0FFF)| (hr_int_RED<<12);

					HWREG(base + EPWM_O_HRPWM_INT) = (HWREG(base + EPWM_O_HRPWM_INT)&0xFFF0FFFF)| (hr_int_FED<<16);
				 }


			}

		}


}


//*****************************************************************************
//
// HRPWM_setActionQualifierAction_Compatible
//
//*****************************************************************************
GS32_DRIVER_HRPWM_FUNC_T void HRPWM_setActionQualifierAction_Compatible(uint32_t base,
							EPWM_ActionQualifierOutputModule epwmOutput,
							EPWM_ActionQualifierOutput output,
							EPWM_ActionQualifierOutputEvent event)
{

    uint32_t registerOffset;
    uint32_t registerTOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    registerOffset = EPWM_O_AQCTLA + (uint32_t)epwmOutput;
    registerTOffset = EPWM_O_AQCTLA2 + (uint32_t)epwmOutput;

	//Check whether the macro is active

	if( 1<<((base - EPWM1_BASE)/(EPWM2_BASE - EPWM1_BASE)) & HRPWM_UpDownDutyModeEnable_Num )
	{
	    if( event == EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA || event == EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA ||
	    	event == EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB || event == EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB)
	    {
	    	if( event == EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB )
	    	{
	    		event -= 0x4;

	    	}else if( event == EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA )
	    	{
	    		event += 0x4;
	    	}

	        HWREGH(base + registerOffset) =
	                ((HWREGH(base + registerOffset) & ~(3U << (uint16_t)event)) |
	                 ((uint16_t)output << (uint16_t)event));

	    }
	}

}

//
// All the API functions are in-lined in hrpwm.h
//
#endif








