/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    i2c.h
*   @brief   
*
*/
/*
 * commit history
 * 20240313, Jason, add API: I2C_pollTxEmpty, I2C_isTxEmpty, I2C_pollRxFull, I2C_isRxFull, I2C_sendCmdRestartAndReadAndStop
 * 20240317, Jason, add API I2C_isTxNOACK,
 * 20240408, Jason, add comment for "I2C_setConfig"
 * 20240515, Jason, change I2C_RxFIFOLevel enum define
 * 20240624, Jason, increase RETRY_LIMIT from 10000 to 40000
 * 20240624, LYF, Add workaround for rx fifo issue in chip 1.2
 */

#ifndef DEVICE_DRIVERLIB_I2C_H_
#define DEVICE_DRIVERLIB_I2C_H_

#ifdef __cplusplus
extern "C"{
#endif

#include <stdbool.h>

#include "gs32_version.h"
#include "inc/hw_i2c.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "debug.h"

#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)

#define I2C_RETRY_LIMIT    40000

/**
 * @brief master mode.
 *
 */
#define I2C_MASTER_MODE             (I2C_IC_CON_MASTER_MODE | I2C_IC_CON_IC_SLAVE_DISABLE) 

/**
 * @brief Controller-transmitter mode.
 *
 */
#define I2C_MASTER_SEND_MODE        I2C_MASTER_MODE

/**
 * @brief Controller-receiver mode.
 *
 */
#define I2C_MASTER_RECEIVE_MODE     I2C_MASTER_MODE

/**
 * @brief slave mode.
 *
 */
#define I2C_SLAVE_MODE              0

/* #define I2C_TARGET_SEND_MODE        I2C_SLAVE_MODE //!< Target-transmitter mode */
/* #define I2C_TARGET_RECEIVE_MODE     I2C_SLAVE_MODE //!< Target-receiver mode */

/**
 * @brief fast speed mode.
 *
 */
#define I2C_FAST_SPEED_MODE         I2C_IC_CON_SPEEDH

/**
 * @brief standard speed mode.
 *
 */
#define I2C_STANDARD_SPEED_MODE     I2C_IC_CON_SPEEDL

/**
 * @brief STOP_DET interrrupt only when it is addressed.
 *
 */
#define I2C_STOP_DET_IFADDRESSED    I2C_IC_CON_STOP_DET_IFADDRESSED

/**
 * @brief TX_EMPTY interrupt when tx empty and shift register transmission completed.
 *
 */
#define I2C_TX_EMPTY_CTRL           I2C_IC_CON_TX_EMPTY_CTRL

/**
 * @brief HOLD BUS when Rx FIFO full to BUFFER_DEPTH.
 *
 */
#define I2C_RX_FIFO_FULL_HLD_CTRL   I2C_IC_CON_RX_FIFO_FULL_HLD_CTRL

/**
 * @brief STOP_DET interrupt only when master is active.
 *
 */
#define I2C_STOP_DET_IF_MASTER_ACTIVE     I2C_IC_CON_STOP_DET_IF_MASTER_ACTIVE

/**
 * @brief enable restart condition.
 *
 */
#define I2C_RESTART_EN              I2C_IC_CON_IC_RESTART_EN

#define I2C_10BIT_ADDRESS           (I2C_IC_CON_IC_10BITADDR_MASTER | I2C_IC_CON_IC_10BITADDR_SLAVE)

#define I2C_CONTROLLER_MODE         (I2C_MASTER_MODE | I2C_RESTART_EN)
#define I2C_CONTROLLER_SEND_MODE    I2C_CONTROLLER_MODE
#define I2C_CONTROLLER_RECEIVE_MODE I2C_CONTROLLER_MODE
#define I2C_TARGET_MODE             (I2C_SLAVE_MODE | I2C_RESTART_EN)
#define I2C_TARGET_SEND_MODE        I2C_TARGET_MODE
#define I2C_TARGET_RECEIVE_MODE     I2C_TARGET_MODE
#define I2C_REPEAT_MODE             0

/**
 * @brief Arbitration-lost interrupt.
 *
 */
#define I2C_INT_ARB_LOST        I2C_IC_INTR_STAT_TX_ABRT

/**
 * @brief  NACK interrupt.
 *
 */
#define I2C_INT_NO_ACK          I2C_IC_INTR_STAT_TX_ABRT

/**
 * @brief Receive-data-ready interrupt.
 *
 */
#define I2C_INT_RX_DATA_RDY     I2C_IC_INTR_STAT_RX_FULL

/**
 * @brief Transmit-data-ready interrupt.
 *
 */
#define I2C_INT_TX_DATA_RDY     I2C_IC_INTR_STAT_TX_EMPTY

/**
 * @brief Stop condition detected.
 *
 */
#define I2C_INT_STOP_CONDITION  I2C_IC_INTR_STAT_STOP_DET

/**
 * @brief Addressed GENERAL CALL interrupt.
 *
 */
#define I2C_INT_ADDR_ZERO       I2C_IC_INTR_STAT_GEN_CALL

/**
 * @brief Addressed as target interrupt.
 *
 */
#define I2C_INT_ADDR_TARGET     I2C_IC_INTR_STAT_SLV_ADDR1_TAG

/**
 * @brief interrupt of I2C master is attempting to read data from this i2c slave.
 *
 */
#define I2C_INT_RD_REQ          I2C_IC_INTR_STAT_RD_REQ

/**
 * @brief RX FIFO level interrupt.
 *
 */
#define I2C_INT_RXFF            I2C_IC_INTR_STAT_RX_FULL

/**
 * @brief TX FIFO level interrupt.
 *
 */
#define I2C_INT_TXFF            I2C_IC_INTR_STAT_TX_EMPTY

/**
 * @brief interrupt of I2C master is attempting to write data from this i2c slave.
 *
 */
#define I2C_INT_WR_REQ          I2C_IC_INTR_STAT_WR_REQ

/**
 * @brief I2C master is attempting to.
 *
 */
#define I2C_INT_MASTER_ON_HOLD  I2C_IC_INTR_STAT_MASTER_ON_HOLD

/**
 * @brief ReStart condition detected.
 *
 */
#define I2C_INT_RESTART_DET     I2C_IC_INTR_STAT_RESTART_DET

/**
 * @brief Start condition detected.
 *
 */
#define I2C_INT_START_DET       I2C_IC_INTR_STAT_START_DET

/**
 * @brief Stop condition detected.
 *
 */
#define I2C_INT_STOP_DET        I2C_IC_INTR_STAT_STOP_DET

/**
 * @brief bus activity interrupt.
 *
 */
#define I2C_INT_ACTIVITY        I2C_IC_INTR_STAT_ACTIVITY

/**
 * @brief interrupt of when act as slave-transmitter, the master does not acknowledge a transmitted byte.
 *
 */
#define I2C_INT_RX_DONE         I2C_IC_INTR_STAT_RX_DONE

/**
 * @brief interrupt of transmit buffer is filled to IC_TX_BUFFER_DEPTH.
 *
 */
#define I2C_INT_TX_OVER         I2C_IC_INTR_STAT_TX_OVER

/**
 * @brief interrupt of receive buffer is completely filled to IC_RX_BUFFER_DEPTH.
 *
 */
#define I2C_INT_RX_OVER         I2C_IC_INTR_STAT_RX_OVER

/**
 * @brief interrupt of no receive buffer to read.
 *
 */
#define I2C_INT_RX_UNDER        I2C_IC_INTR_STAT_RX_UNDER

/**
 * @brief interrupt of unable to complete the intended actions on the contents of the transmit FIFO.
 *
 */
#define I2C_INT_TX_ABRT         I2C_IC_INTR_STAT_TX_ABRT

#define I2C_INT_REG_ACCESS_RDY  0

/**
 * @brief status flag
 *
 */
#define I2C_STS_ARB_LOST        I2C_INT_ARB_LOST
#define I2C_STS_NO_ACK          I2C_INT_NO_ACK
#define I2C_STS_RX_DATA_RDY     I2C_INT_RX_DATA_RDY
#define I2C_STS_TX_DATA_RDY     I2C_INT_TX_DATA_RDY
#define I2C_STS_STOP_CONDITION  I2C_INT_STOP_CONDITION
#define I2C_STS_ADDR_ZERO       I2C_INT_ADDR_ZERO
#define I2C_STS_ADDR_TARGET     I2C_INT_ADDR_TARGET
#define I2C_STS_TX_EMPTY        I2C_INT_TXFF
#define I2C_STS_RX_FULL         I2C_INT_RXFF
#define I2C_STS_BUS_BUSY        I2C_ICSTS_ACTIVITY
#define I2C_STS_TARGET_DIR      I2C_INT_RD_REQ
#define I2C_STS_REG_ACCESS_RDY  I2C_INT_REG_ACCESS_RDY

/**
 * @brief Helpful define to mask out the bits in the I2CSTR register that aren't associated with interrupts.
 *
 */
#define I2C_STR_INTMASK     ((uint16_t)I2C_INT_ARB_LOST |                      \
                             (uint16_t)I2C_INT_NO_ACK |                        \
                             (uint16_t)I2C_INT_RX_DATA_RDY |                   \
                             (uint16_t)I2C_INT_TX_DATA_RDY |                   \
                             (uint16_t)I2C_INT_STOP_CONDITION |                \
                             (uint16_t)I2C_INT_ADDR_TARGET)

/**
 * @brief I2C MASTER MODE.
 *
 */
#define I2C_ICMODE_MASTER 0x41U

/**
 * @brief I2C SLAVE MODE.
 *
 */
#define I2C_ICMODE_SLAVE 0x00U

/**
 * @brief I2C is not idle, the OR of MST_ACTIVITY and SLV_ACTIVITY.
 *
 */
#define I2C_ICSTS_ACTIVITY 0x01U

/**
 * @brief I2C Transmit FIFO Not Full.
 *
 */
#define I2C_ICSTS_TFNF 0x02U

/**
 * @brief I2C Transmit FIFO Completely Empty.
 *
 */
#define I2C_ICSTS_TFE 0x04U

/**
 * @brief I2C Receive FIFO Not Empty.
 *
 */
#define I2C_ICSTS_RFNE 0x08U

/**
 * @brief I2C Receive FIFO Completely Full.
 *
 */
#define I2C_ICSTS_RFF 0x10U

/**
 * @brief The Master Finite State Machine (FSM) is not in the IDLE state
 *
 */
#define I2C_ICSTS_MST_ACTIVITY 0x20U

/**
 * @brief The Slave Finite State Machine (FSM) is not in the IDLE state.
 *
 */
#define I2C_ICSTS_SLV_ACTIVITY 0x40U

/**
 * @brief NOACK for 7-bit address.
 *
 */
#define I2C_TXABRT_7B_ADDR_NOACK 0x01U

/**
 * @brief Byte 1 of 10Bit Address not ACKed by any slave.
 *
 */
#define I2C_TXABRT_10ADDR1_NOACK 0x02U

/**
 * @brief Byte 2 of 10Bit Address not ACKed by any slave.
 *
 */
#define I2C_TXABRT_10ADDR2_NOACK 0x04U

/**
 * @brief Transmitted data not ACKed by addressed slave.
 *
 */
#define I2C_TXABRT_TXDATA_NOACK 0x08U

/**
 * @brief GCALL not ACKed by any slave.
 *
 */
#define I2C_TXABRT_GCALL_NOACK 0x10U

/**
 * @brief GCALL is followed by read from bus.
 *
 */
#define I2C_TXABRT_GCALL_READ 0x20U

/**
 * @brief ACK detected for START byte.
 *
 */
#define I2C_TXABRT_SBYTE_ACKDET 0x80U

/**
 * @brief User initiating master operation when MASTER disabled.
 *
 */
#define I2C_TXABRT_MASTER_DIS 0x800U

/**
 * @brief Master or Slave-Transmitter lost arbitration.
 *
 */
#define I2C_TXARB_LOST 0x1000U

/**
 * @brief Slave flushes existing data in TX-FIFO upon getting read command.
 *
 */
#define I2C_TXABRT_SLVFLUSH_TXFIFO 0x2000U

/**
 * @brief Slave lost arbitration to remote master.
 *
 */
#define I2C_TXABRT_SLV_ARBLOST 0x4000U

/**
 * @brief Slave trying to transmit to remote master in read mode.
 *
 */
#define I2C_TXABRT_SLVRD_INTX 0x8000U

/**
 * @brief Transfer abort detected by master.
 *
 */
#define I2C_TXABRT_USER_ABRT 0x10000U

#define I2C_TXABRT_NOACK 0x1FU

/**
 * @brief I2C Enable Status.
 *
 */
#define I2C_ICENSTS_IC_EN                    0x01U

/**
 * @brief Slave Disabled While Busy (Transmit, Receive).
 *
 */
#define I2C_ICENSTS_SLV_DISABLED_WHILE_BUSY  0x02U

/**
 * @brief Slave Received Data Lost.
 *
 */
#define I2C_ICENSTS_SLV_RX_DATA_LOST         0X04U

/**
 * @brief I2C interrupts to be returned by I2C_getInterruptSource().
 *
 * @param I2C_INTSRC_NONE No interrupt pending.
 * @param I2C_INTSRC_ARB_LOST Arbitration-lost interrupt.
 * @param I2C_INTSRC_NO_ACK NACK interrupt.
 * @param I2C_INTSRC_REG_ACCESS_RDY Register-access-ready interrupt.
 * @param I2C_INTSRC_RX_DATA_RDY Receive-data-ready interrupt.
 *
 * @param I2C_INTSRC_RX_FULL Rx fifo full interrupt.
 * @param I2C_INTSRC_TX_DATA_RDY Transmit-data-ready interrupt.
 * @param I2C_INTSRC_TX_EMPTY Tx fifo empty interrupt.
 * @param I2C_INTSRC_STOP_CONDITION Stop condition detected.
 *
 * @param I2C_INTSRC_ADDR_TARGET Addressed as target interrupt.
 * @param I2C_INTSRC_ADDR_ZERO Addressed general call interrupt.
 * @param I2C_INTSRC_RD_REQ Addressed as slave transmitter interrupt.
 * @param I2C_INTSRC_RD_REQ Addressed as slave reciever interrupt.
 *
 * @param I2C_INTSRC_RESTART_DET Restart condition detected.
 * @param I2C_INTSRC_START_DET Start condition detected.
 * @param I2C_INTSRC_ACTIVITY i2c bus active interrupt.
 * @param I2C_INTSRC_RX_DONE transmission done interrupt when act as slave-transmitter.
 *
 * @param I2C_INTSRC_TX_OVER tx fifo filled to BUFFER_DEPTH interrupt.
 * @param I2C_INTSRC_RX_OVER rx fifo filled to BUFFER_DEPTH interrupt.
 * @param I2C_INTSRC_RX_UNDER no receive buffer, rx fifo empty interrupt.
 * @param I2C_INTSRC_RX_UNDER transmit abort interrupt.
 */
typedef enum
{
    I2C_INTSRC_NONE,
    I2C_INTSRC_ARB_LOST,
    I2C_INTSRC_NO_ACK,
    I2C_INTSRC_REG_ACCESS_RDY,
    I2C_INTSRC_RX_DATA_RDY,
    I2C_INTSRC_RX_FULL = I2C_INTSRC_RX_DATA_RDY,
    I2C_INTSRC_TX_DATA_RDY,
    I2C_INTSRC_TX_EMPTY = I2C_INTSRC_TX_DATA_RDY,
    I2C_INTSRC_STOP_CONDITION,
    I2C_INTSRC_ADDR_TARGET,
    I2C_INTSRC_ADDR_ZERO,
    I2C_INTSRC_RD_REQ,
    I2C_INTSRC_WR_REQ,
    I2C_INTSRC_RESTART_DET,
    I2C_INTSRC_START_DET,
    I2C_INTSRC_ACTIVITY,
    I2C_INTSRC_RX_DONE,
    I2C_INTSRC_TX_OVER,
    I2C_INTSRC_RX_OVER,
    I2C_INTSRC_RX_UNDER,
    I2C_INTSRC_TX_ABRT
} I2C_InterruptSource;

/**
 * @brief Values that can be passed to I2C_setFIFOInterruptLevel() as the \e txLevel
 * 		  parameter, returned by I2C_getFIFOInterruptLevel() in the \e txLevel
 * 		  parameter, and returned by I2C_getTxFIFOStatus().
 *
 * @param I2C_FIFO_TXEMPTY Transmit FIFO empty
 * @param I2C_FIFO_TX0 Transmit FIFO empty
 * @param I2C_FIFO_TX1 Transmit FIFO 1/16 full.
 * @param I2C_FIFO_TX2 Transmit FIFO 2/16 full.
 * @param I2C_FIFO_TX3 Transmit FIFO 3/16 full.
 * @param I2C_FIFO_TX4 Transmit FIFO 4/16 full.
 * @param I2C_FIFO_TX5 Transmit FIFO 5/16 full.
 * @param I2C_FIFO_TX6 Transmit FIFO 6/16 full.
 * @param I2C_FIFO_TX7 Transmit FIFO 7/16 full.
 * @param I2C_FIFO_TX8 Transmit FIFO 8/16 full.
 * @param I2C_FIFO_TX9 Transmit FIFO 9/16 full.
 * @param I2C_FIFO_TX10 Transmit FIFO 10/16 full.
 * @param I2C_FIFO_TX11 Transmit FIFO 11/16 full.
 * @param I2C_FIFO_TX12 Transmit FIFO 12/16 full.
 * @param I2C_FIFO_TX13 Transmit FIFO 13/16 full.
 * @param I2C_FIFO_TX14 Transmit FIFO 14/16 full.
 * @param I2C_FIFO_TX15 Transmit FIFO 15/16 full.
 * @param I2C_FIFO_TX16 Transmit FIFO full.
 * @param I2C_FIFO_TXFULL Transmit FIFO full.
 */
typedef enum
{
    I2C_FIFO_TXEMPTY    = 0x0000U,      
    I2C_FIFO_TX0        = 0x0000U,     
    I2C_FIFO_TX1        = 0x0001U,     
    I2C_FIFO_TX2        = 0x0002U,    
    I2C_FIFO_TX3        = 0x0003U,    
    I2C_FIFO_TX4        = 0x0004U,    
    I2C_FIFO_TX5        = 0x0005U,  
    I2C_FIFO_TX6        = 0x0006U,  
    I2C_FIFO_TX7        = 0x0007U,  
    I2C_FIFO_TX8        = 0x0008U,  
    I2C_FIFO_TX9        = 0x0009U,   
    I2C_FIFO_TX10       = 0x000AU,  
    I2C_FIFO_TX11       = 0x000BU, 
    I2C_FIFO_TX12       = 0x000CU,  
    I2C_FIFO_TX13       = 0x000DU,  
    I2C_FIFO_TX14       = 0x000EU, 
    I2C_FIFO_TX15       = 0x000FU, 
    I2C_FIFO_TX16       = 0x0010U, 
    I2C_FIFO_TXFULL     = 0x0010U   
} I2C_TxFIFOLevel;

/**
 * @brief Values that can be passed to I2C_setFIFOInterruptLevel() as the \e rxLevel
 * 		  parameter, returned by I2C_getFIFOInterruptLevel() in the \e rxLevel
 * 		  parameter, and returned by I2C_getRxFIFOStatus().
 *
 * @param I2C_FIFO_RXEMPTY Receive FIFO empty
 * @param I2C_FIFO_RX1 Receive FIFO 1/16 full
 * @param I2C_FIFO_RX2 Receive FIFO 2/16 full.
 * @param I2C_FIFO_RX3 Receive FIFO 3/16 full.
 * @param I2C_FIFO_RX4 Receive FIFO 4/16 full.
 * @param I2C_FIFO_RX5 Receive FIFO 5/16 full.
 * @param I2C_FIFO_RX6 Receive FIFO 6/16 full.
 * @param I2C_FIFO_RX7 Receive FIFO 7/16 full.
 * @param I2C_FIFO_RX8 Receive FIFO 8/16 full.
 * @param I2C_FIFO_RX9 Receive FIFO 9/16 full.
 * @param I2C_FIFO_RX10 Receive FIFO 10/16 full.
 * @param I2C_FIFO_RX11 Receive FIFO 11/16 full.
 * @param I2C_FIFO_RX12 Receive FIFO 12/16 full.
 * @param I2C_FIFO_RX13 Receive FIFO 13/16 full.
 * @param I2C_FIFO_RX14 Receive FIFO 14/16 full.
 * @param I2C_FIFO_RX15 Receive FIFO 15/16 full.
 * @param I2C_FIFO_RX16 Receive FIFO full.
 * @param I2C_FIFO_RXFULL Receive FIFO full.
 */
typedef enum
{
    I2C_FIFO_RXEMPTY    = 0x0000U,     
    I2C_FIFO_RX1        = 0x0001U,    
    I2C_FIFO_RX2        = 0x0002U,  
    I2C_FIFO_RX3        = 0x0003U,  
    I2C_FIFO_RX4        = 0x0004U,  
    I2C_FIFO_RX5        = 0x0005U,  
    I2C_FIFO_RX6        = 0x0006U,  
    I2C_FIFO_RX7        = 0x0007U,  
    I2C_FIFO_RX8        = 0x0008U,  
    I2C_FIFO_RX9        = 0x0009U, 
    I2C_FIFO_RX10       = 0x000AU,  
    I2C_FIFO_RX11       = 0x000BU,  
    I2C_FIFO_RX12       = 0x000CU, 
    I2C_FIFO_RX13       = 0x000DU, 
    I2C_FIFO_RX14       = 0x000EU,  
    I2C_FIFO_RX15       = 0x000FU,   
    I2C_FIFO_RX16       = 0x0010U,    
    I2C_FIFO_RXFULL     = 0x0010U   
} I2C_RxFIFOLevel;


/**
 * @brief Values that can be passed to I2C_setBitCount() as the \e size parameter.
 *
 * @param I2C_BITCOUNT_1 1 bit per data byte
 * @param I2C_BITCOUNT_2 2 bits per data byte
 * @param I2C_BITCOUNT_3 3 bits per data byte
 * @param I2C_BITCOUNT_4 4 bits per data byte
 * @param I2C_BITCOUNT_5 5 bits per data byte
 * @param I2C_BITCOUNT_6 6 bits per data byte
 * @param I2C_BITCOUNT_7 7 bits per data byte
 * @param I2C_BITCOUNT_8 8 bits per data byte
 * 
 */
typedef enum
{
    I2C_BITCOUNT_1  = 1U,  
    I2C_BITCOUNT_2  = 2U,  
    I2C_BITCOUNT_3  = 3U,
    I2C_BITCOUNT_4  = 4U, 
    I2C_BITCOUNT_5  = 5U, 
    I2C_BITCOUNT_6  = 6U, 
    I2C_BITCOUNT_7  = 7U,  
    I2C_BITCOUNT_8  = 0U  
} I2C_BitCount;

/**
 * @brief Values that can be passed to I2C_setAddressMode() as the \e mode parameter.
 *
 * @param I2C_ADDR_MODE_7BITS   7-bit address
 * @param I2C_ADDR_MODE_10BITS  10-bit address
 * 
 */
typedef enum
{
    I2C_ADDR_MODE_7BITS        = 0x0000U, 
    I2C_ADDR_MODE_10BITS       = 0x0018U   
} I2C_AddressMode;

/**
 * @brief Values that can be passed to I2C_setExtendedMode() as the \e mode parameter.
 *
 * @param I2C_EADDR_MODE_BC Backward compatibility
 * @param I2C_EADDR_MODE_FWD Forward compatibility
 * 
 */
typedef enum
{
    I2C_EADDR_MODE_BC  = 1U, 
    I2C_EADDR_MODE_FWD = 2U   
} I2C_ExtendedMode;

/**
 * @brief Values that can be passed to I2C_setEmulationMode() as the \e mode parameter.
 *
 * @param I2C_EMULATION_STOP_SCL_LOW If SCL is low, keep it low. If high, stop when it goes low again.
 * @param I2C_EMULATION_FREE_RUN Continue I2C operation regardless
 * 
 */
typedef enum
{
    I2C_EMULATION_STOP_SCL_LOW = 0x0000U,
    I2C_EMULATION_FREE_RUN     = 0x4000U
} I2C_EmulationMode;

/**
 * @brief Values that can be passed to I2C_initController() as the \e dutyCycle parameter.
 *
 * @param I2C_DUTYCYCLE_33 Clock duty cycle is 33%
 * @param I2C_DUTYCYCLE_50 Clock duty cycle is 50%
 * 
 */
typedef enum
{
    I2C_DUTYCYCLE_33,       //!< Clock duty cycle is 33%
    I2C_DUTYCYCLE_50        //!< Clock duty cycle is 50%
} I2C_DutyCycle;

/**
 * @brief I2C module initialization parameters.
 *
 * @param master_mode I2C operation mode.
 * @param slave_addr I2C slave addr.
 * @param speed_mode I2C speed mode.
 * @param baudrate I2C baudrate.
 * @param apbclk_freq APB bus clock frequency.
 * 
 */
typedef struct
{
	uint32_t master_mode;
	uint32_t slave_addr;
	uint32_t speed_mode;
	uint32_t baudrate;
	uint32_t apbclk_freq;
} I2C_INIT_PARAM;

/**
 * @brief Checks an I2C base address.
 * 
 * @param base is the base address of the I2C instance used.
 * 
 * @return Returns \b true if the base address is valid and \b false
 * otherwise.
 */
#ifdef DEBUG
__STATIC_INLINE boolean I2C_isBaseValid(uint32_t base)
{
    return(
           (base == I2CA_BASE) ||
           (base == I2CB_BASE) ||
           (base == I2CC_BASE)
          );
}
#endif

/**
 * @brief Enables the I2C module.
 * 
 * @param base is the base address of the I2C instance used.
 * @return None.
 */
__STATIC_INLINE void
I2C_enableModule(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    HWREG(base + I2C_O_IC_ENABLE) |= I2C_IC_ENABLE_ENABLE;
}

/**
 * @brief Disables the I2C module.
 * 
 * @param base is the base address of the I2C instance used.
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_disableModule(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    HWREG(base + I2C_O_IC_ENABLE) &= ~(I2C_IC_ENABLE_ENABLE);
}

/**
 * @brief Enable the sending command block of the I2C module
 * 
 * @param base is the base address of the I2C instance used.
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_enableTxCmdBlock(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    HWREG(base + I2C_O_IC_ENABLE) |= I2C_IC_ENABLE_TX_CMD_BLOCK;
}

/**
 * @brief Disable the sending command block of the I2C module
 * 
 * @param base is the base address of the I2C instance used.
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_disableTxCmdBlock(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));
    HWREG(base + I2C_O_IC_ENABLE) &= ~(I2C_IC_ENABLE_TX_CMD_BLOCK);
}

/**
 * @brief Enables the transmit and receive FIFOs.
 * 
 * @param base is the base address of the I2C instance used.
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_enableFIFO(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    /*config FIFO Threshold Level to default */
    HWREG(base + I2C_O_IC_TX_TL) = (HWREG(base + I2C_O_IC_TX_TL) &
                                 (~I2C_IC_TX_TL_TX_TL)) | I2C_FIFO_TXEMPTY;
    HWREG(base + I2C_O_IC_RX_TL) = (HWREG(base + I2C_O_IC_RX_TL) &
                                 (~I2C_IC_RX_TL_RX_TL)) | I2C_FIFO_RXEMPTY;
}

/**
 * @brief Disables the transmit and receive FIFOs.
 * 
 * @param base is the base address of the I2C instance used.
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_disableFIFO(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    /* config FIFO Threshold Level to 1 to disableFIFO */
    HWREG(base + I2C_O_IC_TX_TL) = (HWREG(base + I2C_O_IC_TX_TL) &
                                 (~I2C_IC_TX_TL_TX_TL)) | I2C_FIFO_TX1;
    HWREG(base + I2C_O_IC_RX_TL) = (HWREG(base + I2C_O_IC_RX_TL) &
                                 (~I2C_IC_RX_TL_RX_TL)) | I2C_FIFO_RX1;
}

/**
 * @brief Sets the FIFO level at which interrupts are generated.
 * 
 * @param base is the base address of the I2C instance used.
 * @param txLevel is the transmit FIFO interrupt level, specified as
 * \b I2C_FIFO_TX0, \b I2C_FIFO_TX1, \b I2C_FIFO_TX2, . . . or
 * \b I2C_FIFO_TX16.
 * @param rxLevel is the receive FIFO interrupt level, specified as
 * \b I2C_FIFO_RX0, \b I2C_FIFO_RX1, \b I2C_FIFO_RX2, . . . or
 * \b I2C_FIFO_RX16.
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_setFIFOInterruptLevel(uint32_t base, I2C_TxFIFOLevel txLevel,
                          I2C_RxFIFOLevel rxLevel)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    if(rxLevel > I2C_FIFO_RXEMPTY)
    	rxLevel = (I2C_RxFIFOLevel)((uint32_t)rxLevel - 1);

    /* Set the FIFO interrupt levels. */
    HWREG(base + I2C_O_IC_TX_TL) = (HWREG(base + I2C_O_IC_TX_TL) &
                                 (~I2C_IC_TX_TL_TX_TL)) | (uint32_t)txLevel;
    HWREG(base + I2C_O_IC_RX_TL) = (HWREG(base + I2C_O_IC_RX_TL) &
                                 (~I2C_IC_RX_TL_RX_TL)) | (uint32_t)rxLevel;
}

/**
 * @brief Gets the RX FIFO level at which interrupts are generated.
 * 
 * @param base is the base address of the I2C instance used.
 * 
 * @return Returns the current number of words in the receive FIFO specified
 * as one of the following:
 * \b I2C_FIFO_RX0, \b I2C_FIFO_RX1, \b I2C_FIFO_RX2, \b I2C_FIFO_RX3,
 * ..., or \b I2C_FIFO_RX16
 * 
 */
__STATIC_INLINE uint32_t
I2C_getReceiveFIFOLevel(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    return HWREG(base + I2C_O_IC_RXFLR);
}

/**
 * @brief Gets the FIFO level at which interrupts are generated.
 * 
 * @param base is the base address of the I2C instance used.
 * @param txLevel is a pointer to storage for the transmit FIFO level,
 * returned as one of \b I2C_FIFO_TX0, \b I2C_FIFO_TX1,
 * \b I2C_FIFO_TX2, . . . or \b I2C_FIFO_TX16.
 * @param rxLevel is a pointer to storage for the receive FIFO level,
 * returned as one of \b I2C_FIFO_RX0, \b I2C_FIFO_RX1,
 * \b I2C_FIFO_RX2, . . . or \b I2C_FIFO_RX16.
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_getFIFOInterruptLevel(uint32_t base, I2C_TxFIFOLevel *txLevel,
                          I2C_RxFIFOLevel *rxLevel)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    /* Extract the transmit and receive FIFO levels. */
    *txLevel = (I2C_TxFIFOLevel)HWREG(base + I2C_O_IC_TX_TL);
    *rxLevel = (I2C_RxFIFOLevel)HWREG(base + I2C_O_IC_RX_TL);
}

/**
 * @brief Get the transmit FIFO status
 * 
 * @param base is the base address of the I2C instance used.
 * 
 * @return Returns the current number of words in the transmit FIFO specified
 * as one of the following:
 * \b I2C_FIFO_TX0, \b I2C_FIFO_TX1, \b I2C_FIFO_TX2, \b I2C_FIFO_TX3,
 * ..., or \b I2C_FIFO_TX16
 * 
 */
__STATIC_INLINE I2C_TxFIFOLevel
I2C_getTxFIFOStatus(uint32_t base)
{
    uint16_t level;

    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    /* Get the current FIFO status*/
    level = (uint16_t)HWREG(base + I2C_O_IC_TXFLR);

    return((I2C_TxFIFOLevel)level);
}

/**
 * @brief Get the receive FIFO status
 * 
 * @param base is the base address of the I2C instance used.
 * 
 * @return Returns the current number of words in the receive FIFO specified
 * as one of the following:
 * \b I2C_FIFO_RX0, \b I2C_FIFO_RX1, \b I2C_FIFO_RX2, \b I2C_FIFO_RX3,
 * ..., or \b I2C_FIFO_RX16
 * 
 */
__STATIC_INLINE I2C_RxFIFOLevel
I2C_getRxFIFOStatus(uint32_t base)
{
    uint16_t level;

    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    /* Get the current FIFO status */
    level = (uint16_t)HWREG(base + I2C_O_IC_RXFLR);

    return((I2C_RxFIFOLevel)level);
}

/**
 * @brief Sets the own address for this I2C module.
 * 
 * @param base is the base address of the I2C instance used.
 * @param Addr is the 7-bit or 10-bit address.
 * 
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_setOwnAddress(uint32_t base, uint16_t Addr)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));
    ASSERT(Addr <= I2C_IC_SAR_IC_SAR);

    I2C_disableModule(base);
    HWREG(base + I2C_O_IC_SAR) = (HWREG(base + I2C_O_IC_SAR) & (~I2C_IC_SAR_IC_SAR)) | (uint32_t)Addr;
    I2C_enableModule(base);
}

/**
 * @brief Indicates whether or not the I2C bus is busy.
 * 
 * @param base is the base address of the I2C instance used.
 * 
 * @return Returns \b true if the I2C bus is busy; otherwise, returns
 * \b false.
 * 
 */
__STATIC_INLINE boolean
I2C_isBusBusy(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    return (HWREG(base + I2C_O_IC_STATUS) & I2C_IC_STATUS_ACTIVITY != 0);
}

/**
 * @brief Indicates whether or not the I2C master is busy.
 * 
 * @param base is the base address of the I2C instance used.
 * 
 * @return Returns \b true if the I2C bus is busy; otherwise, returns
 * \b false.
 * 
 */
__STATIC_INLINE boolean
I2C_isMasterBusy(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    return (HWREG(base + I2C_O_IC_STATUS) & I2C_IC_STATUS_MST_ACTIVITY != 0);
}

/**
 * @brief Issues an read/write command with START/STOP/RESTART CONDITION.
 * 
 * @param base is the base address of the I2C instance used.
 * @param ic_data_cmd is the data and command like data | write/read | stop/restart
 * 
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_sendIcDataCmd(uint32_t base, uint32_t ic_data_cmd)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    /* Set the CMD bit. Issues a read/write command and causes an START/RESTART/STOP condition. */
    HWREG(base + I2C_O_IC_DATA_CMD) = ic_data_cmd;
}

/**
 * @brief Issues an write command.
 * 
 * @param base is the base address of the I2C instance used.
 * @param data is the data and command like data
 * 
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_sendCmdWrite(uint32_t base,uint8_t data)
{
    /* Check the arguments. */
	ASSERT(I2C_isBaseValid(base));
	/* Set the CMD bit. Issues a write command. */
    HWREG(base + I2C_O_IC_DATA_CMD) = (data&0x00FFU);

}

/**
 * @brief Issues an write command with STOP CONDITION.
 * 
 * @param base is the base address of the I2C instance used.
 * @param data is the data and command like data | stop
 * 
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_sendCmdWriteAndStop(uint32_t base,uint8_t data)
{
	/* Check the arguments. */
	ASSERT(I2C_isBaseValid(base));
	/* Set the CMD bit. Issues a write command and causes an STOP condition. */
    HWREG(base + I2C_O_IC_DATA_CMD) = ((data&0x00FFU)|I2C_IC_DATA_CMD_STOP);
	
}

/**
 * @brief Issues an write command with RESTART CONDITION.
 * 
 * @param base is the base address of the I2C instance used.
 * @param data is the data and command like data | restart
 * 
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_sendCmdRestartAndWrite(uint32_t base,uint8_t data)
{
	/* Check the arguments. */
	ASSERT(I2C_isBaseValid(base));
	/* Set the CMD bit. Causes a RESTART condition and issues a write command. */
    HWREG(base + I2C_O_IC_DATA_CMD) =((data&0x00FFU)|I2C_IC_DATA_CMD_RESTART);
}

/**
 * @brief Issues an read command with RESTART CONDITION.
 * 
 * @param base is the base address of the I2C instance used.
 * 
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_sendCmdRestartAndRead(uint32_t base)
{
	/* Check the arguments. */
	ASSERT(I2C_isBaseValid(base));
	/* Set the CMD bit. Causes a RESTART condition and issues a read command. */
	HWREG(base + I2C_O_IC_DATA_CMD) =I2C_IC_DATA_CMD_CMD | I2C_IC_DATA_CMD_RESTART;
}

/**
 * @brief Issues an read command with STOP CONDITION.
 * 
 * @param base is the base address of the I2C instance used.
 * 
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_sendCmdReadAndStop(uint32_t base)
{
	/* Check the arguments. */
	ASSERT(I2C_isBaseValid(base));
	/* Set the CMD bit. Issues a read command and causes an STOP condition. */
	HWREG(base + I2C_O_IC_DATA_CMD) = I2C_IC_DATA_CMD_CMD | I2C_IC_DATA_CMD_STOP;
}

/**
 * @brief Issues an read command with STOP/RESTART CONDITION.
 * 
 * @param base is the base address of the I2C instance used.
 * 
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_sendCmdRestartAndReadAndStop(uint32_t base)
{
	/* Check the arguments. */
	ASSERT(I2C_isBaseValid(base));
	/* Set the CMD bit. Causes a RESTART condition and issues a read command and generate STOP. */
	HWREG(base + I2C_O_IC_DATA_CMD) =I2C_IC_DATA_CMD_CMD | I2C_IC_DATA_CMD_RESTART | I2C_IC_DATA_CMD_STOP;
}

/**
 * @brief Issues an read command.
 * 
 * @param base is the base address of the I2C instance used.
 * 
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_sendCmdRead(uint32_t base)
{
	/* Check the arguments. */
	ASSERT(I2C_isBaseValid(base));
	/* Set the CMD bit. Issues a read command. */
	HWREG(base + I2C_O_IC_DATA_CMD) =I2C_IC_DATA_CMD_READ;
}

/**
 * @brief Receives a byte that has been sent to the I2C.
 * 
 * @param base is the base address of the I2C instance used.
 * 
 * @return Returns the byte received from by the I2C cast as an uint8_t.
 * 
 */
__STATIC_INLINE uint8_t
I2C_getData(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
    /* must make sure Rx FIFO is not empty before reading RxFIFO */
    if(I2C_FIFO_RXEMPTY != (0xFF & I2C_getReceiveFIFOLevel(base)))
    {
        /* Return the contents of the receive register. */
        return(HWREG(base + I2C_O_IC_DATA_CMD) & I2C_IC_DATA_CMD_DAT);
    }
    else
    {
    	return 0;  /*return 0 means invalid data, Rx FIFO is empty */
    }
#else
    /* Return the contents of the receive register. */
    return(HWREG(base + I2C_O_IC_DATA_CMD) & I2C_IC_DATA_CMD_DAT);
#endif
}

/**
 * @brief Receives a byte that has been sent to the I2C.
 * 
 * @param base is the base address of the I2C instance used.
 * @param data is the data to be transmitted from the I2C Controller.
 * 
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_putData(uint32_t base, uint32_t data)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    /* Place the data into the transmit register. */
    HWREG(base + I2C_O_IC_DATA_CMD) = (data & 0xFF);
}

/**
 * @brief Get stop condition status.
 * 
 * @param base is the base address of the I2C instance used.
 * 
 * @return Returns \b true if the STP bit has been set by the device to
 * generate a stop condition when the internal data counter of the I2C module
 * has reached 0. Returns \b false when the STP bit is zero. This bit is
 * automatically cleared after the stop condition has been generated.
 * 
 */
__STATIC_INLINE boolean
I2C_getStopConditionStatus(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    /* Check the stop condition bit and return appropriately. */
    return((HWREG(base + I2C_O_IC_RAW_INTR_STAT) & I2C_IC_INTR_STAT_STOP_DET) != 0U);
}

/**
 * @brief Sets the addressing mode to either 7-bit or 10-bit.
 * 
 * @param base is the base address of the I2C instance used.
 * @param mode is the address mode, 7-bit or 10-bit.
 * 
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_setAddressMode(uint32_t base, I2C_AddressMode mode)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    /* Write the appropriate value to the address expansion bit. */
    HWREG(base + I2C_O_IC_CON) = (HWREG(base + I2C_O_IC_CON) &
                               ~(I2C_IC_CON_IC_10BITADDR_MASTER | I2C_IC_CON_IC_10BITADDR_SLAVE)) |
                               (uint16_t)mode;
}

/**
 * @brief Enables I2C loopback mode.
 * 
 * @param base is the base address of the I2C instance used.
 * 
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_enableLoopback(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    /* loopback exists when tar == sar */ 
    ASSERT((HWREG(base + I2C_O_IC_TAR) & I2C_IC_TAR_IC_TAR) == (HWREG(base + I2C_O_IC_SAR) & I2C_IC_SAR_IC_SAR));

    I2C_disableModule(base);
    HWREG(base + I2C_O_IC_CON) |= I2C_IC_CON_MASTER_MODE;
    HWREG(base + I2C_O_IC_CON) &= (~I2C_IC_CON_IC_SLAVE_DISABLE);
    I2C_enableModule(base);
}

/**
 * @brief Disables I2C loopback mode.
 * 
 * @param base is the base address of the I2C instance used.
 * 
 * @return None.
 * 
 */
__STATIC_INLINE void
I2C_disableLoopback(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(I2C_isBaseValid(base));

    HWREG(base + I2C_O_IC_CON) |= I2C_IC_CON_MASTER_MODE;
    HWREG(base + I2C_O_IC_CON) |= I2C_IC_CON_IC_SLAVE_DISABLE;
}

/**
 * @brief   i2c config.
 *
 * @param   base   the base address of the I2C instance used.
 * @param   config the I2C_IC_CON register value,
 *          it should be the combination of the following macro:
 *          I2C_MASTER_MODE, or I2C_SLAVE_MODE, (must include one of them)
 *          I2C_FAST_SPEED_MODE, or I2C_STANDARD_SPEED_MODE, (must include one of them)
 *          I2C_STOP_DET_IFADDRESSED,
 *          I2C_TX_EMPTY_CTRL, (must include this bit)
 *          I2C_RX_FIFO_FULL_HLD_CTRL,
 *          I2C_STOP_DET_IF_MASTER_ACTIVE,
 *          I2C_RESTART_EN,
 *          I2C_10BIT_ADDRESS,
 *
 * @return  None
 */

__STATIC_INLINE void
I2C_setConfig(uint32_t base, uint32_t config)
{
    HWREG(base + I2C_O_IC_CON) = config;
}

/**
 * @brief   enable general call.
 *
 * @param   base   the base address of the I2C instance used.
 *
 * @return  None
 */
__STATIC_INLINE void
I2C_enableGeneralCall(uint32_t base)
{
    HWREG(base + I2C_O_IC_TAR) = (HWREG(base + I2C_O_IC_TAR) | I2C_IC_TAR_SPECIAL) & (~I2C_IC_TAR_GC_OR_START);
}

/**
 * @brief   Enable start byte.
 *
 * @param   base   the base address of the I2C instance used.
 *
 * @return  None
 */
__STATIC_INLINE void
I2C_enableStartByte(uint32_t base)
{
    HWREG(base + I2C_O_IC_CON) |= (I2C_IC_TAR_SPECIAL | I2C_IC_TAR_GC_OR_START);
}


/**
 * @brief   abort the I2C transfer in master mode
 *
 * @param   base   the base address of the I2C instance used.
 *
 * @return  None
 */
__STATIC_INLINE void
I2C_transmitAbort(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));
    HWREG(base + I2C_O_IC_ENABLE) |= I2C_IC_ENABLE_ABORT;
}

/**
 * @brief   get I2C ic mode
 *
 * @param   base   the base address of the I2C instance used.
 *
 * @return  i2c ic mode, refer:
 *          I2C_ICMODE_MASTER
 *          I2C_ICMODE_SLAVE
 */
__STATIC_INLINE uint32_t
I2C_getIcMode(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));
    return HWREG(base + I2C_O_IC_CON) & (I2C_IC_CON_MASTER_MODE | I2C_IC_CON_IC_SLAVE_DISABLE);
}


/**
 * @brief   get I2C speed
 *
 * @param   base   the base address of the I2C instance used.
 *
 * @return  i2c speed value
 *
 *
 */
__STATIC_INLINE uint32_t
I2C_getSpeedMode(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));
    return HWREG(base + I2C_O_IC_CON) & (0x03 << 1);
}

/**
 * @brief    Enables I2C interrupt sources.
 *
 * @param   base        the base address of the I2C instance used.
 * @param   intFlags    is the bit mask of the interrupt sources to be enabled.
 *!  This function enables the indicated I2C Controller interrupt sources.   Only the
 *!  sources that are enabled can be reflected to the processor interrupt.
 *!  Disabled sources have no effect on the processor.
 *!  The \e intFlags parameter is the logical OR of any of the following:
 * #define I2C_IC_INTR_STAT_RX_UNDER	0x01U
 * #define I2C_IC_INTR_STAT_RX_OVER	0x02U
 * #define I2C_IC_INTR_STAT_RX_FULL	0x04U
 * #define I2C_IC_INTR_STAT_TX_OVER	0x08U
 * #define I2C_IC_INTR_STAT_TX_EMPTY	0x10U
 * #define I2C_IC_INTR_STAT_RD_REQ	0x20U
 * #define I2C_IC_INTR_STAT_TX_ABRT	0x40U
 * #define I2C_IC_INTR_STAT_RX_DONE	0x80U
 * #define I2C_IC_INTR_STAT_ACTIVITY	0x100U
 * #define I2C_IC_INTR_STAT_STOP_DET	0x200U
 * #define I2C_IC_INTR_STAT_START_DET	0x400U
 * #define I2C_IC_INTR_STAT_GEN_CALL	0x800U
 * #define I2C_IC_INTR_STAT_RESTART_DET	0x1000U
 * #define I2C_IC_INTR_STAT_MASTER_ON_HOLD 0x2000U
 * #define I2C_IC_INTR_STAT_WR_REQ	0x8000U
 * #define I2C_IC_INTR_STAT_SLV_ADDR1_TAG	0x10000U
 * @return  None
 */
__STATIC_INLINE void
I2C_enableInterrupt(uint32_t base, uint32_t intFlags)
{
    HWREG(base + I2C_O_IC_INTR_MASK) |= intFlags;
}

/**
 * @brief   disable interrupt.
 *
 * @param   base       the base address of the I2C instance used.
 * @param   intFlags   the interrupt to disable.
 *
 * @return  None
 */
__STATIC_INLINE void
I2C_disableInterrupt(uint32_t base, uint32_t intFlags)
{
    HWREG(base + I2C_O_IC_INTR_MASK) &= (~intFlags);
}

/**
 * @brief  Gets the current I2C interrupt status.
 *
 * @param  base     the base address of the I2C instance used.
 * This function returns the interrupt status for the I2C module.
 * return The current interrupt status, Macro definition:
 * #define I2C_IC_INTR_STAT_RX_UNDER	0x01U
 * #define I2C_IC_INTR_STAT_RX_OVER	0x02U
 * #define I2C_IC_INTR_STAT_RX_FULL	0x04U
 * #define I2C_IC_INTR_STAT_TX_OVER	0x08U
 * #define I2C_IC_INTR_STAT_TX_EMPTY	0x10U
 * #define I2C_IC_INTR_STAT_RD_REQ	0x20U
 * #define I2C_IC_INTR_STAT_TX_ABRT	0x40U
 * #define I2C_IC_INTR_STAT_RX_DONE	0x80U
 * #define I2C_IC_INTR_STAT_ACTIVITY	0x100U
 * #define I2C_IC_INTR_STAT_STOP_DET	0x200U
 * #define I2C_IC_INTR_STAT_START_DET	0x400U
 * #define I2C_IC_INTR_STAT_GEN_CALL	0x800U
 * #define I2C_IC_INTR_STAT_RESTART_DET	0x1000U
 * #define I2C_IC_INTR_STAT_MASTER_ON_HOLD 0x2000U
 * #define I2C_IC_INTR_STAT_WR_REQ	0x8000U
 * #define I2C_IC_INTR_STAT_SLV_ADDR1_TAG	0x10000U
 * !note This function will only return the status flags associated with
 * !The corresponding interrupt source can be set only after it is enabled
 * @return  the status of interrupt
 */
__STATIC_INLINE uint32_t I2C_getInterruptStatus(uint32_t base)
{
    return HWREG(base + I2C_O_IC_INTR_STAT);
}

#define I2C_getStatus I2C_getInterruptStatus

/**
 * @brief   get raw interrupt status.
 *
 * @param   base       the base address of the I2C instance used.
 * This function returns the status of raw interrupt
 * return The current interrupt status, Macro definition:
 * #define I2C_IC_INTR_STAT_RX_UNDER	0x01U
 * #define I2C_IC_INTR_STAT_RX_OVER	0x02U
 * #define I2C_IC_INTR_STAT_RX_FULL	0x04U
 * #define I2C_IC_INTR_STAT_TX_OVER	0x08U
 * #define I2C_IC_INTR_STAT_TX_EMPTY	0x10U
 * #define I2C_IC_INTR_STAT_RD_REQ	0x20U
 * #define I2C_IC_INTR_STAT_TX_ABRT	0x40U
 * #define I2C_IC_INTR_STAT_RX_DONE	0x80U
 * #define I2C_IC_INTR_STAT_ACTIVITY	0x100U
 * #define I2C_IC_INTR_STAT_STOP_DET	0x200U
 * #define I2C_IC_INTR_STAT_START_DET	0x400U
 * #define I2C_IC_INTR_STAT_GEN_CALL	0x800U
 * #define I2C_IC_INTR_STAT_RESTART_DET	0x1000U
 * #define I2C_IC_INTR_STAT_MASTER_ON_HOLD 0x2000U
 * #define I2C_IC_INTR_STAT_WR_REQ	0x8000U
 * #define I2C_IC_INTR_STAT_SLV_ADDR1_TAG	0x10000U
 * !note This function will only return the status flags associated with
 * !The corresponding interrupt source can be set only after it is enabled
 * @return  the status of raw interrupt
 */
__STATIC_INLINE uint32_t I2C_getInterruptRawStatus(uint32_t base)
{
    return HWREG(base + I2C_O_IC_RAW_INTR_STAT);
}

/**
 * @brief   clear all interrupt status.
 *
 * @param   base       the base address of the I2C instance used.
 *
 * @return  none
 * \details I2C Interrupt Status Register IC_INTR_STAT is cleared by reading IC_CLR_INTR register
 */
__STATIC_INLINE void I2C_clearAllInterruptStatus(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    (void)HWREG(base + I2C_O_IC_CLR_INTR);
}

/**
 * @brief   clear rx under.
 *
 * @param   base       the base address of the I2C instance used.
 *
 * @return  none
 */
__STATIC_INLINE void I2C_clearInterruptRxUnder(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    (void)HWREG(base + I2C_O_IC_CLR_RX_UNDER);
}

/**
 * @brief   clear rx over.
 *
 * @param   base       the base address of the I2C instance used.
 *
 * @return  none
 */
__STATIC_INLINE void I2C_clearInterruptRxOver(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    (void)HWREG(base + I2C_O_IC_CLR_RX_OVER);
}

/**
 * @brief   clear tx over.
 *
 * @param   base       the base address of the I2C instance used.
 *
 * @return  none
 */
__STATIC_INLINE void I2C_clearInterruptTxOver(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    (void)HWREG(base + I2C_O_IC_CLR_TX_OVER);
}

/**
 * @brief   clear rd req.
 *
 * @param   base       the base address of the I2C instance used.
 *
 * @return  none
 */
__STATIC_INLINE void I2C_clearInterruptRdReq(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    (void)HWREG(base + I2C_O_IC_CLR_RD_REQ);
}

/**
 * @brief   clear tx abrt.
 *
 * @param   base       the base address of the I2C instance used.
 *
 * @return  none
 */
__STATIC_INLINE void I2C_clearInterruptTxAbrt(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    (void)HWREG(base + I2C_O_IC_CLR_TX_ABRT);
}

/**
 * @brief   clear rx done.
 *
 * @param   base       the base address of the I2C instance used.
 *
 * @return  none
 */\
__STATIC_INLINE void I2C_clearInterruptRxDone(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    (void)HWREG(base + I2C_O_IC_CLR_RX_DONE);
}

/**
 * @brief   clear activity.
 *
 * @param   base       the base address of the I2C instance used.
 *
 * @return  none
 */
__STATIC_INLINE void I2C_clearInterruptActivity(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    (void)HWREG(base + I2C_O_IC_CLR_ACTIVITY);
}

/**
 * @brief   clear stop detect.
 *
 * @param   base       the base address of the I2C instance used.
 *
 * @return  none
 */
__STATIC_INLINE void I2C_clearInterruptStopDet(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));


    (void)HWREG(base + I2C_O_IC_CLR_STOP_DET);
}

/**
 * @brief   clear start detect.
 *
 * @param   base       the base address of the I2C instance used.
 *
 * @return  none
 */
__STATIC_INLINE void I2C_clearInterruptStartDet(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    (void)HWREG(base + I2C_O_IC_CLR_START_DET);
}

/**
 * @brief   clear general call.
 *
 * @param   base       the base address of the I2C instance used.
 *
 * @return  none
 */
__STATIC_INLINE void I2C_clearInterruptGenCall(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    (void)HWREG(base + I2C_O_IC_CLR_GEN_CALL);
}

/**
 * @brief   clear restart detect.
 *
 * @param   base the base address of the I2C instance used.
 *
 * @return  none
 */
__STATIC_INLINE void I2C_clearInterruptRestartDet(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    (void)HWREG(base + I2C_O_IC_CLR_RESTART_DET);
}

__STATIC_INLINE void I2C_clearInterruptWrReq(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    (void)HWREG(base + I2C_O_IC_CLR_WR_REQ);
}

__STATIC_INLINE void I2C_clearInterruptSlvAddrTag(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    (void)HWREG(base + I2C_O_IC_CLR_SLV_ADDR_TAG);
}

/**
 * @brief   get the current transfer status and FIFO status
 *
 * @param   base   the base address of the I2C instance used.
 *
 * @return  ic status, refer:
 *          I2C_TXABRT_7B_ADDR_NOACK
 *          I2C_TXABRT_10ADDR1_NOACK
 *          I2C_TXABRT_10ADDR2_NOACK
 *          I2C_TXABRT_TXDATA_NOACK
 *          I2C_TXABRT_GCALL_NOACK
 *          I2C_TXABRT_GCALL_READ
 *          I2C_TXABRT_SBYTE_ACKDET
 *          I2C_TXABRT_MASTER_DIS
 *          I2C_TXARB_LOST
 *          I2C_TXABRT_SLVFLUSH_TXFIFO
 *          I2C_TXABRT_SLV_ARBLOST
 *          I2C_TXABRT_SLVRD_INTX
 *          I2C_TXABRT_USER_ABRT
 */
__STATIC_INLINE uint32_t
I2C_getIcTxAbrtSource(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));
    return HWREG(base + I2C_O_IC_TX_ABRT_SOURCE);
}

/**
 * @brief   get the current transfer status and FIFO status
 *
 * @param   base   the base address of the I2C instance used.
 *
 * @return  ic status, refer:
 *          I2C_ICSTS_ACTIVITY,        bus is busy
 *          I2C_ICSTS_TFNF,            TX FIFO is not full
 *          I2C_ICSTS_TFE,             TX FIFO empty
 *          I2C_ICSTS_RFNE,            RX FIFO not empty
 *          I2C_ICSTS_RFF,             RX FIFO is full
 *          I2C_ICSTS_MST_ACTIVITY,    Master is in TX/RX activity
 *          I2C_ICSTS_SLV_ACTIVITY,    Slave is in TX/RX activity
 */
__STATIC_INLINE uint32_t
I2C_getIcStatus(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));
    return HWREG(base + I2C_O_IC_STATUS);
}

/**
 * @brief   poll the I2C tx status for nRetryCnt times
 *
 * @param   base, the base address of the I2C instance used.
 *          nRetryCnt, max re-try count of reading the status register before return.
 *              assuming 240MHz sysclk, nRetryCnt=30 is around 1uS
 *
 * @return  true, the Tx shift register is empty
 *          false, Tx shift register is not empty
 */
__STATIC_INLINE boolean
I2C_pollTxStatus(uint32_t base, uint32_t nRetryCnt)
{
	uint32_t retry_cnt = 0;
    ASSERT(I2C_isBaseValid(base));

    // wait if tx shift register is not empty
    while (!(I2C_getInterruptRawStatus(base) & I2C_IC_INTR_STAT_TX_EMPTY))
	{
        if (retry_cnt == nRetryCnt)
		    return false;
        retry_cnt++;
	}
    return true;
}

/**
 * @brief   poll the I2C rx status for nRetryCnt times
 *
 * @param   base, the base address of the I2C instance used.
 *          nRetryCnt, max re-try count of reading the status register before return.
 *              assuming 240MHz sysclk, nRetryCnt=30 is around 1uS
 *
 * @return  true, there is data received in rx buffer.
 *          false, rx buffer is empty, no data received.
 */
__STATIC_INLINE boolean
I2C_pollRxStatus(uint32_t base, uint32_t nRetryCnt)
{
	uint32_t retry_cnt = 0;
    ASSERT(I2C_isBaseValid(base));

    /* wait if Rx FIFO is empty */
    while (!(I2C_getIcStatus(base) & I2C_ICSTS_RFNE))
    {
        if (retry_cnt == nRetryCnt)
            return false;
        retry_cnt++;
    }
    return true;
}

/**
 * @brief   poll the I2C master status for nRetryCnt times
 *
 * @param   base, the base address of the I2C instance used.
 *          nRetryCnt, max re-try count of reading the status register before return.
 *                assuming 240MHz sysclk, nRetryCnt=30 is around 1uS
 *
 * @return  true, master status is inactive, master has completed read/write operation.
 *          false, master is active, master has not yet completed read/write operation.
 */
__STATIC_INLINE boolean
I2C_pollMasterStatus(uint32_t base, uint32_t nRetryCnt)
{
	uint32_t retry_cnt = 0;
    ASSERT(I2C_isBaseValid(base));

    /* wait if master status is active */
    while (I2C_getIcStatus(base) & I2C_ICSTS_MST_ACTIVITY)
    {
        if (retry_cnt == nRetryCnt)
            return false;
        retry_cnt++;
    }
    return true;
}

/**
 * @brief   poll the I2C bus busy status for nRetryCnt times
 *
 * @param   base, the base address of the I2C instance used.
 *          nRetryCnt, max re-try count of reading the status register before return.
 *                assuming 240MHz sysclk, nRetryCnt=30 is around 1uS
 *
 * @return  true, bus is idle.
 *          false, bus is busy.
 */
__STATIC_INLINE boolean
I2C_pollBusyStatus(uint32_t base, uint32_t nRetryCnt)
{
	uint32_t retry_cnt = 0;
    ASSERT(I2C_isBaseValid(base));

    /* wait if master status is active */
    while (I2C_getIcStatus(base) & I2C_IC_STATUS_ACTIVITY)
    {
        if (retry_cnt == nRetryCnt)
            return false;
        retry_cnt++;
    }
    return true;
}

/**
 * @brief   report the i2c hardware status
 *
 * @param   base   the base address of the I2C instance used.
 *
 * @return  ic enable status, refer:
 *          I2C_ICENSTS_IC_EN
 *          I2C_ICENSTS_SLV_DISABLED_WHILE_BUSY
 *          I2C_ICENSTS_SLV_RX_DATA_LOST
 */
__STATIC_INLINE uint32_t
I2C_getIcEnableStatus(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));
    return HWREG(base + I2C_O_IC_ENABLE_STATUS);
}

/**
 * @brief   controls the amount of time delay (in terms of number of ic_clk clock periods)
 *          introduced in the rising edge of SCL - relative to SDA changing
 *          when i2c services a read request in a slave-transmitter operation. 
 *
 * @param   base   the base address of the I2C instance used.
 *
 * @param   setupTime   SDA setup time (tSU:DAT).
 *
 * @return  None
 */
__STATIC_INLINE void
I2C_setSDASetupTime(uint32_t base, uint32_t setupTime)
{
    ASSERT(I2C_isBaseValid(base));
    ASSERT(setupTime > 0);
    HWREG(base + I2C_O_IC_SDA_SETUP) = setupTime;
}

/**
 * @brief   store the duration, measured in ic_clk cycles, 
 *          of the longest spike that is filtered out by the spike suppression logic.
 *
 * @param   base   the base address of the I2C instance used.
 *
 * @param   spklen   the duration, measured in ic_clk cycles, of the longest spike.
 *
 * @return  None
 */
__STATIC_INLINE void
I2C_setSpkLen(uint32_t base, uint32_t spklen)
{
    ASSERT(I2C_isBaseValid(base));
    HWREG(base + I2C_O_IC_SDA_SETUP) = spklen;
}

/**
 * @brief   poll Tx Empty flag for I2C_RETRY_TIMES
 * @param   base   the base address of the I2C instance used.
 * @return  TRUE if TxBuf is empty
 *          FALSE if TxBuf is not empty
 * \note    I2C need be initialized with EMPTY_CTRL=1
 */
__STATIC_INLINE uint32_t I2C_pollTxEmpty(uint32_t base)
{
    uint32_t retry_cnt = 0;

    while (!(I2C_getInterruptRawStatus(base) & I2C_IC_INTR_STAT_TX_EMPTY))
    {
        if (retry_cnt++ == I2C_RETRY_LIMIT)
            return FALSE;
    }
    return TRUE;
}

/**
 * @brief   Waiting for TX to be empty 
 * 
 * @param   base   the base address of the I2C instance used.
 * @return  None
 */
__STATIC_INLINE void I2C_waitTxEmpty(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));
    while((I2C_getInterruptRawStatus(base) & I2C_IC_INTR_STAT_TX_EMPTY) == 0);
}
/**
 * @brief   Determine if TX is empty
 * 
 * @param   base   the base address of the I2C instance used.
 * @return  TRUE if Tx is empty
 *          FALSE if Tx is not empty
 * 
 */
__STATIC_INLINE uint32_t I2C_isTxEmpty(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));
    if(I2C_getInterruptRawStatus(base) & I2C_IC_INTR_STAT_TX_EMPTY)
    	return 1;
    else
    	return 0;
}

/**
 * @brief   Determine whether TX is NOACK
 * 
 * @param   base   the base address of the I2C instance used.
 * @return  TRUE if Tx is NOACK
 *          FALSE if Tx is not NOACK
 * 
 */
__STATIC_INLINE uint32_t I2C_isTxNOACK(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));
    if((I2C_getInterruptRawStatus(base) & I2C_IC_INTR_STAT_TX_ABRT) && (I2C_getIcTxAbrtSource(base) & (I2C_TXABRT_NOACK)))
    	return 1;
    else
    	return 0;
}

/**
 * @brief   poll Tx Empty flag for I2C_RETRY_TIMES
 * @param   base   the base address of the I2C instance used.
 * @return  TRUE if RxBuf is full
 *          FALSE if RxBuf is not full
 * \note    I2C need be initialized with EMPTY_CTRL=1
 */
__STATIC_INLINE uint32_t I2C_pollRxFull(uint32_t base)
{
    uint32_t retry_cnt = 0;

    /* wait if Rx FIFO is empty */
    while (!(I2C_getInterruptRawStatus(base) & I2C_IC_INTR_STAT_RX_FULL))
    {
        if (retry_cnt++ == I2C_RETRY_LIMIT)
            return FALSE;
    }
    return TRUE;
}

/**
 * @brief   Waiting for RX to be full 
 * 
 * @param   base   the base address of the I2C instance used.
 * @return  None
 * 
 */
__STATIC_INLINE void I2C_waitRxFull(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));
    while((I2C_getInterruptRawStatus(base) & I2C_IC_INTR_STAT_RX_FULL) == 0);
}

/**
 * @brief   Determine if RX is full
 * 
 * @param   base   the base address of the I2C instance used.
 * @return  TRUE if Rx is full
 *          FALSE if Rx is not full
 * 
 */
__STATIC_INLINE uint32_t I2C_isRxFull(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));
    if(I2C_getInterruptRawStatus(base) & I2C_IC_INTR_STAT_RX_FULL)
    	return 1;
    else
    	return 0;
}


/**
 * @brief   controls whether i2c responds with a ACK or NACK when it receives an I2C General Call address.
 *
 * @param   base   the base address of the I2C instance used.
 *
 * @param   flag   respond or not respond GENERAL CALL
 *
 * @return  None
 */
__STATIC_INLINE void
I2C_setAckGeneralCall(uint32_t base, boolean flag)
{
    ASSERT(I2C_isBaseValid(base));
    if (flag)
        HWREG(base + I2C_O_IC_ACK_GENERAL_CALL) |= I2C_IC_ACK_GENERAL_CALL;
    else
        HWREG(base + I2C_O_IC_ACK_GENERAL_CALL) &= (~I2C_IC_ACK_GENERAL_CALL);
}

/**
 * @brief   enable i2c dma.
 *
 * @param   base   the base address of the I2C instance used.
 *
 * @return  None
 */
__STATIC_INLINE void
I2C_enableDma(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    HWREG(base + I2C_O_IC_DMA_CR) |= (I2C_IC_DMA_CR_RDMAE | I2C_IC_DMA_CR_TDMAE);
}

/**
 * @brief   disable i2c dma.
 *
 * @param   base   the base address of the I2C instance used.
 *
 * @return  None
 */
__STATIC_INLINE void
I2C_disableDma(uint32_t base)
{
    ASSERT(I2C_isBaseValid(base));

    HWREG(base + I2C_O_IC_DMA_CR) &= (~(I2C_IC_DMA_CR_RDMAE | I2C_IC_DMA_CR_TDMAE));
}

/**
 * @brief   set i2c DMA Transmit Data Level.
 *
 * @param   base   the base address of the I2C instance used.
 * @param   dmatdl   Transmit Data Level.range:0-7
 *
 * @return  None
 */
__STATIC_INLINE void
I2C_setDmaTdLr(uint32_t base, uint8_t dmatdl)
{
    ASSERT(I2C_isBaseValid(base));
    ASSERT(dmatdl >> 0 && dmatdl << 7);

    HWREG(base + I2C_O_IC_DMA_TDLR) = ((HWREG(base + I2C_O_IC_DMA_TDLR) & (~I2C_IC_DMA_TDLR_DMATDL)) | dmatdl);
}

/**
 * @brief   set i2c DMA Receive Data Level.
 *
 * @param   base   the base address of the I2C instance used.
 * @param   dmardl   Receive Data Level.range:0-7
 *
 * @return  None
 */
__STATIC_INLINE void
I2C_setDmaRdLr(uint32_t base, uint8_t dmardl)
{
    ASSERT(I2C_isBaseValid(base));

    ASSERT(dmardl >> 0 && dmardl << 7);

    HWREG(base + I2C_O_IC_DMA_RDLR) = ((HWREG(base + I2C_O_IC_DMA_RDLR) & (~I2C_IC_DMA_RDLR_DMARDL)) | dmardl);
}

__STATIC_INLINE void
I2C_setFsSpklen(uint32_t base, uint32_t spike_len)
{
    HWREGH(base + I2C_O_IC_FS_SPKLEN) = spike_len;
}

/**
 * @brief Sets the address that the I2C Controller places on the bus.
 *        This function configures the address that the I2C Controller places on the bus
 *        when initiating a transaction.
 *
 * @param base is the base address of the I2C instance used.
 * @param targetAddr 7-bit or 10-bit target address
 *
 * @return None.
 */
extern void
I2C_setTargetAddress(uint32_t base, uint16_t targetAddr);

/**
 * @brief This is same as "I2C_setTargetAddress"
 *
 * @param base is the base address of the I2C instance used.
 * @param targetAddr 7-bit or 10-bit target address
 *
 * @return None.
 */
extern void
I2C_setSlaveAddress(uint32_t base, uint16_t targetAddr);

/**
 * @brief Initializes the I2C Controller.
 *
 * @param base is the base address of the I2C instance used.
 * @param sysclkHz is the rate of the clock supplied to the I2C module (SYSCLK) in Hz.
 * @param bitRate is the rate of the controller clock signal, SCL.
 * @param dutyCycle is duty cycle of the SCL signal.
 *
 * This function initializes operation of the I2C Controller by configuring the
 * bus speed for the controller. Note that the I2C module \b must be put into
 * reset before calling this function. You can do this with the function
 * I2C_disableModule().
 *
 * A programmable prescaler in the I2C module divides down the input clock
 * (rate specified by \e sysclkHz) to produce the module clock (calculated to
 * be around 10 MHz in this function). That clock is then divided down further
 * to configure the SCL signal to run at the rate specified by \e bitRate. The
 * \e dutyCycle parameter determines the percentage of time high and time low
 * on the clock signal. The valid values are \b I2C_DUTYCYCLE_33 for 33% and
 * \b I2C_DUTYCYCLE_50 for 50%.
 *
 * The peripheral clock is the system clock.  This value is returned by
 * SysCtl_getClock(), or it can be explicitly hard coded if it is
 * constant and known (to save the code/execution overhead of a call to
 * SysCtl_getClock()).
 *
 * @return None.
 */
extern void
I2C_initController(uint32_t base, uint32_t sysclkHz, uint32_t bitRate,
               I2C_DutyCycle dutyCycle);

/**
 * @brief Control the hold time of SDA in both slave and master mode during transmit or in the receiver.
 *
 * @param base is the base address of the I2C instance used.
 * @param rxHoldTime the required SDA hold time in units of ic_clk period, when i2c acts as a receiver.
 * @param txHoldTime the required SDA hold time in units of ic_clk period, when i2c acts as a transmitter.
 *
 * @return None
 */              
extern void
I2C_setSDAHoldTime(uint32_t base, uint8_t rxHoldTime, uint16_t txHoldTime);

/**
 * @brief Clear interrupt status.
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return None
 */
extern void
I2C_clearInterruptStatus(uint32_t base, uint32_t intFlags);

#define I2C_clearStatus I2C_clearInterruptStatus

/**
 * @brief Obtain the interrupt source that is currently activated by the I2C controller
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return I2C interrupts
 */
extern I2C_InterruptSource
I2C_getInterruptSource(uint32_t base);

/**
 * @brief Initialize and configure the I2C controller
 *
 * @param base is the base address of the I2C instance used.
 * @param init_param Initialize the parameter structure pointer, including the following key configurations:
 *			-apbclk_freq: APB bus clock Frequency (Unit: Hz)
 *			-master_mode: Master/Slave mode selection (I2C_MASTER_MODE/I2C_SLAVE_MODE)
 *			-speed_mode: Speed mode (standard mode/fast mode, etc.)
 *			-slave_addr: Slave device address (Target address in master mode and self-address in slave mode)
 *			-baudrate: Communication baudrate (Unit: bps)
 *
 * @return None
 */
void I2C_initConfig(uint32_t i2c_base, I2C_INIT_PARAM *init_param);

/**
 * @brief Wait if Master is busy
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return 0 if idle
 * @return 1 if busy
 **/
extern uint32_t I2C_MasterStatusCheck(uint32_t base);

/**
 * @brief Wait until Tx FIFO is empty
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return 0 if TXFIFO empty
 * @return 1 if TXFIFO not empty
 */
extern uint32_t I2C_SendStatusCheck(uint32_t base);

/**
 * @brief Wait for the RX FIFO not to be empty
 *
 * @param base is the base address of the I2C instance used.
 *
 * @return 0 if RXFIFO not empty
 * @return 1 if RXFIFO empty
 */
uint32_t I2C_ReceiveStatusCheck(uint32_t base);

#endif


#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_I2C_H_ */
