/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    hw_types.h
*   @brief
*
*  commit history
*  2024/03/11, Zhao Lei, add/update GS32 version check macros
*  2024/03/13, Jason Yang, change type float64_t from long double to double because of type conflict in ARM fast math library.
*  2024/03/14, Jason Yang, remove duplicated Uint8 define.
*  2024/04/07, ralf, add macro definition for ERTM and interrupt.
*  2024/06/15, Jason, update the define for uint8_t, uint16_t, uint32_t, uint64_t.
*  2024/6/27, ralf, add the definition for uint8, uint16, uint32, uint64, int8, int16, int32, int64.
*/

#ifndef DEVICE_DRIVERLIB_HW_TYPES_H_
#define DEVICE_DRIVERLIB_HW_TYPES_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <stdint.h>
#include <stdbool.h>

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* The following compiler definitions are also included in cmsis_compiler.h */


/*
 * Arm Compiler 4/5
 */
#if   defined ( __CC_ARM )
	#ifndef   __ASM
	  #define __ASM                                  __asm
	#endif
	#ifndef   __INLINE
	  #define __INLINE                               __inline
	#endif
	#ifndef   __STATIC_INLINE
	  #define __STATIC_INLINE                        static __inline
	#endif
	#ifndef   __STATIC_FORCEINLINE
	  #define __STATIC_FORCEINLINE                   static __forceinline
	#endif
	#ifndef   __NO_RETURN
	  #define __NO_RETURN                            __declspec(noreturn)
	#endif
	#ifndef   __USED
	  #define __USED                                 __attribute__((used))
	#endif
	#ifndef   __WEAK
	  #define __WEAK                                 __attribute__((weak))
	#endif
	#ifndef   __ALIGNED
	  #define __ALIGNED(x)                           __attribute__((aligned(x)))
	#endif

/*
 * Arm Compiler 6.6 LTM (armclang)
 */
#elif defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050) && (__ARMCC_VERSION < 6100100)
	#ifndef   __ASM
	  #define __ASM                                  __asm
	#endif
	#ifndef   __INLINE
	  #define __INLINE                               __inline
	#endif
	#ifndef   __STATIC_INLINE
	  #define __STATIC_INLINE                        static __inline
	#endif
	#ifndef   __STATIC_FORCEINLINE
	  #define __STATIC_FORCEINLINE                   __attribute__((always_inline)) static __inline
	#endif
	#ifndef   __NO_RETURN
	  #define __NO_RETURN                            __attribute__((__noreturn__))
	#endif
	#ifndef   __USED
	  #define __USED                                 __attribute__((used))
	#endif
	#ifndef   __WEAK
	  #define __WEAK                                 __attribute__((weak))
	#endif
	#ifndef   __ALIGNED
	  #define __ALIGNED(x)                           __attribute__((aligned(x)))
	#endif

  /*
 * Arm Compiler above 6.10.1 (armclang)
 */
#elif defined (__ARMCC_VERSION) && (__ARMCC_VERSION >= 6100100)
	#ifndef   __ASM
	  #define __ASM                                  __asm
	#endif
	#ifndef   __INLINE
	  #define __INLINE                               __inline
	#endif
	#ifndef   __STATIC_INLINE
	  #define __STATIC_INLINE                        static __inline
	#endif
	#ifndef   __STATIC_FORCEINLINE
	  #define __STATIC_FORCEINLINE                   __attribute__((always_inline)) static __inline
	#endif
	#ifndef   __NO_RETURN
	  #define __NO_RETURN                            __attribute__((__noreturn__))
	#endif
	#ifndef   __USED
	  #define __USED                                 __attribute__((used))
	#endif
	#ifndef   __WEAK
	  #define __WEAK                                 __attribute__((weak))
	#endif
	#ifndef   __ALIGNED
	  #define __ALIGNED(x)                           __attribute__((aligned(x)))
	#endif

/*
 * GNU Compiler
 */
#elif defined ( __GNUC__ )
    #undef __ASM
    #define __ASM                                  __asm

    #undef __INLINE
    #define __INLINE                               inline

    #undef __STATIC_INLINE
    #define __STATIC_INLINE                        static inline

    #undef __STATIC_FORCEINLINE
    #define __STATIC_FORCEINLINE                   __attribute__((always_inline)) static inline

    #undef __NO_RETURN
    #define __NO_RETURN                            __attribute__((__noreturn__))

    #undef __USED
    #define __USED                                 __attribute__((used))

    #undef __WEAK
    #define __WEAK                                 __attribute__((weak))

    #undef __ALIGNED
    #define __ALIGNED(x)                           __attribute__((aligned(x)))

#ifndef __EXTERN_INLINE_MANUAL
#define __EXTERN_INLINE_MANUAL                     __STATIC_INLINE
#endif

#else

#ifndef   __ASM
  #define __ASM                                  __asm
#endif
#ifndef   __INLINE
  #define __INLINE                               inline
#endif
#ifndef   __STATIC_INLINE
  #define __STATIC_INLINE                        static inline
#endif
#ifndef   __STATIC_FORCEINLINE
  #define __STATIC_FORCEINLINE                   __attribute__((always_inline)) static inline
#endif
#ifndef   __NO_RETURN
  #define __NO_RETURN                            __attribute__((__noreturn__))
#endif
#ifndef   __USED
  #define __USED                                 __attribute__((used))
#endif
#ifndef   __WEAK
  #define __WEAK                                 __attribute__((weak))
#endif
#ifndef   __ALIGNED
  #define __ALIGNED(x)                           __attribute__((aligned(x)))
#endif

#endif

/* API function returs E_OK or E_NOT_OK. */
#define E_OK 		(0U)
#define E_NOT_OK 	(1U)

/* Define NULL pointer value */
#ifndef NULL
#ifdef __cplusplus
#define NULL 0
#else /* __cplusplus */
#define NULL ((void *)0)
#endif /* __cplusplus */
#endif /* NULL */

#define NULL_PTR	NULL


/* Registers must be defined with __I/__O/__IO/__IM/__OM/__IOM  */
/** \brief Defines 'read only' permissions */
#ifdef __cplusplus
  #define   __I     volatile
#else
  #define   __I     volatile const
#endif
/** \brief Defines 'write only' permissions */
#define     __O     volatile
/** \brief Defines 'read / write' permissions */
#define     __IO    volatile

/* following defines should be used for structure members */
/** \brief Defines 'read only' structure member permissions */
#define     __IM     volatile const
/** \brief Defines 'write only' structure member permissions */
#define     __OM     volatile
/** \brief Defines 'read/write' structure member permissions */
#define     __IOM    volatile

//
typedef __attribute__((mode(TI))) unsigned int    uint128_t;
typedef __attribute__((mode(DI))) unsigned int    uint64_t;
#if !(__int32_t_defined)
typedef __attribute__((mode(SI))) unsigned int    uint32_t;
#endif
typedef __attribute__((mode(HI))) unsigned int    uint16_t;
typedef __attribute__((mode(QI))) unsigned int    uint8_t;
//
typedef __attribute__((mode(TI))) signed int    int128_t;
typedef __attribute__((mode(DI))) signed int    int64_t;
#if !(__int32_t_defined)
typedef __attribute__((mode(SI))) signed int    int32_t;
#endif
typedef __attribute__((mode(HI))) signed int    int16_t;
typedef __attribute__((mode(QI))) signed int    int8_t;

/* additional type name of 8bit/16bit/32bit integer */
typedef uint8_t Uint8;
typedef uint16_t Uint16;
typedef uint32_t Uint32;
typedef uint64_t Uint64;

typedef int8_t CHAR;
typedef int16_t INT16;
typedef int32_t INT32;
typedef int64_t INT64;

typedef uint8_t uint8;
typedef uint16_t uint16;
typedef __attribute__((mode(SI))) unsigned int    uint32;
typedef uint64_t uint64;

typedef int8_t int8;
typedef int16_t int16;
typedef __attribute__((mode(SI))) signed int    int32;
typedef int64_t int64;

//typedef __int128_t int128;

typedef uint8_t UCHAR;
typedef uint8_t UINT8;
typedef uint16_t UINT16;
typedef uint32_t UINT32;
typedef uint64_t UINT64;

/* Registers can be defined with vuint32_t */
typedef volatile uint8_t 	vuint8_t;
typedef volatile uint16_t 	vuint16_t;
typedef volatile uint32_t 	vuint32_t;
typedef volatile uint64_t 	vuint64_t;
typedef uint32_t 			boolean;
typedef uint32_t 			Bool;

typedef float         		float32_t;
typedef float         		float32;
typedef float         		FLOAT32;
typedef float         		Float32;
typedef float               DOUBLE32;
typedef double              DOUBLE64;
typedef double   		    float64_t;
typedef long double   		float64;

/* write registers operation */
#define WRITE_REG8(ADDR, VAL)		((*(vuint8_t *)ADDR) = VAL)
#define WRITE_REG16(ADDR, VAL)		((*(vuint16_t *)ADDR) = VAL)
#define WRITE_REG32(ADDR, VAL)		((*(vuint32_t *)ADDR) = VAL)

/* read registers operation */
#define READ_REG8(ADDR, VAL)		(*(vuint8_t *)ADDR)
#define READ_REG16(ADDR, VAL)		(*(vuint16_t *)ADDR)
#define READ_REG32(ADDR, VAL)		(*(vuint32_t *)ADDR)

#define TRUE	(1U)
#define FALSE	(0U)

#define  EALLOW
#define  EDIS
#define  ERTM

#define STD_ON		(1U)
#define STD_OFF		(0U)

/* define bit offset */
#define BIT(ofs)                            (0x1UL << (ofs))

/* Macros for memory access operations */
#define _REG8P(p, i)                        ((volatile uint8_t *) ((uintptr_t)((p) + (i))))
#define _REG16P(p, i)                       ((volatile uint16_t *) ((uintptr_t)((p) + (i))))
#define _REG32P(p, i)                       ((volatile uint32_t *) ((uintptr_t)((p) + (i))))
#define _REG64P(p, i)                       ((volatile uint64_t *) ((uintptr_t)((p) + (i))))
#define _REG8(p, i)                         (*(_REG8P(p, i)))
#define _REG16(p, i)                        (*(_REG16P(p, i)))
#define _REG32(p, i)                        (*(_REG32P(p, i)))
#define _REG64(p, i)                        (*(_REG64P(p, i)))
#define REG8(addr)                          _REG8((addr), 0)
#define REG16(addr)                         _REG16((addr), 0)
#define REG32(addr)                         _REG32((addr), 0)
#define REG64(addr)                         _REG64((addr), 0)

/* Macros for address type convert and access operations */
#define ADDR16(addr)                        ((uint16_t)(uintptr_t)(addr))
#define ADDR32(addr)                        ((uint32_t)(uintptr_t)(addr))
#define ADDR64(addr)                        ((uint64_t)(uintptr_t)(addr))
#define ADDR8P(addr)                        ((uint8_t *)(uintptr_t)(addr))
#define ADDR16P(addr)                       ((uint16_t *)(uintptr_t)(addr))
#define ADDR32P(addr)                       ((uint32_t *)(uintptr_t)(addr))
#define ADDR64P(addr)                       ((uint64_t *)(uintptr_t)(addr))

/* Macros for Bit Operations */
#define BITMASK_MAX                         0xFFFFFFFFUL
#define BITOFS_MAX                          31

// BIT/BITS only support bit mask for __riscv_xlen
// For RISC-V 32 bit, it support mask 32 bit wide
// For RISC-V 64 bit, it support mask 64 bit wide
#define BIT(ofs)                            (0x1UL << (ofs))
#define BITS(start, end)                    ((BITMASK_MAX) << (start) & (BITMASK_MAX) >> (BITOFS_MAX - (end)))
#define GET_BIT(regval, bitofs)             (((regval) >> (bitofs)) & 0x1)
#define __SET_BIT(regval, bitofs)             ((regval) |= BIT(bitofs))
#define __CLR_BIT(regval, bitofs)             ((regval) &= (~BIT(bitofs)))
#define FLIP_BIT(regval, bitofs)            ((regval) ^= BIT(bitofs))
#define __WRITE_BIT(regval, bitofs, val)      __CLR_BIT(regval, bitofs); ((regval) |= ((val) << bitofs) & BIT(bitofs))
#define CHECK_BIT(regval, bitofs)           (!!((regval) & (0x1UL<<(bitofs))))
#define GET_BITS(regval, start, end)        (((regval) & BITS((start), (end))) >> (start))
//#define SET_BITS(regval, start, end)        ((regval) |= BITS((start), (end)))
#define CLR_BITS(regval, start, end)        ((regval) &= (~BITS((start), (end))))
#define FLIP_BITS(regval, start, end)       ((regval) ^= BITS((start), (end)))
#define WRITE_BITS(regval, start, end, val) CLR_BITS(regval, start, end); ((regval) |= ((val) << start) & BITS((start), (end)))
#define CHECK_BITS_ALL(regval, start, end)  (!((~(regval)) & BITS((start), (end))))
#define CHECK_BITS_ANY(regval, start, end)  ((regval) & BITS((start), (end)))

#define BITMASK_SET(regval, mask)           ((regval) |= (mask))
#define BITMASK_CLR(regval, mask)           ((regval) &= (~(mask)))
#define BITMASK_FLIP(regval, mask)          ((regval) ^= (mask))
#define BITMASK_CHECK_ALL(regval, mask)     (!((~(regval)) & (mask)))
#define BITMASK_CHECK_ANY(regval, mask)     ((regval) & (mask))

/* Register access definition */
#if defined(__riscv) && (__riscv_xlen==64)
#define HWREG(x)		(*((volatile uint32_t *)((unsigned long)x)))
#define HWREGH(x)		(*((volatile uint16_t *)((unsigned long)x)))
#define HWREGB(x)		(*((volatile uint8_t *)((unsigned long)x)))
#else
#define HWREG(x)		(*((volatile uint32_t *)(x)))
#define HWREGH(x)		(*((volatile uint16_t *)(x)))
#define HWREGB(x)		(*((volatile uint8_t *)(x)))
#endif

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* init by cpu1 */
#define __MSGRAM_HEADER__               __attribute__((section(".shared_header")))
#define __MSGRAM_DATA__                 __attribute__((section(".shared_data")))
#define __MSGRAM__                      __attribute__((section(".shared_data")))
#define __MSGRAM_BSS__                  __attribute__((section(".shared_bss")))

/* init by cpu1 */
#define __SHARED_HEADER__               __attribute__((section(".shared_header")))
#define __SHARED_DATA__                 __attribute__((section(".shared_data")))
#define __SHARED__                      __attribute__((section(".shared_data")))
#define __SHARED_BSS__                  __attribute__((section(".shared_bss")))

/* init by cpu1 */
#define __CPU1TOCPU2_DATA__             __attribute__((section(".shared_cpu1tocpu2_data")))
#define __CPU1TOCPU2__                  __attribute__((section(".shared_cpu1tocpu2_data")))
#define __CPU1TOCPU2_BSS__              __attribute__((section(".shared_cpu1tocpu2_bss")))

/* load by cpu1 when call bringup_cpu2() */
#define __CPU2TOCPU1_DATA__             __attribute__((section(".shared_cpu2tocpu1_data")))
#define __CPU2TOCPU1__                  __attribute__((section(".shared_cpu2tocpu1_data")))
/* init by cpu2 */
#define __CPU2TOCPU1_BSS__              __attribute__((section(".shared_cpu2tocpu1_bss")))

#define __IPC_DATA__                    __SHARED_HEADER__

#ifdef FLASH_TARGET
#define RAMFUNC_T                       __attribute__ ((section (".RamFunc")))
#define RAM_FUNC_T                      __attribute__ ((section (".RamFunc")))
#define ROMFUNC_T                       __attribute__ ((section (".RomFunc")))
#define ROM_FUNC_T                      __attribute__ ((section (".RomFunc")))
#define ROM_DATA_T                      __attribute__ ((section (".rom_data")))
#define ROM_DATA_NOT_KEEP_T             __attribute__ ((section (".rom_data_not_keep")))
#else
#define RAMFUNC_T
#define RAM_FUNC_T
#define ROMFUNC_T
#define ROM_FUNC_T
#define ROM_DATA_T
#define ROM_DATA_NOT_KEEP_T
#endif

#define GS32_DRIVER_ROM_DATA_T          ROM_DATA_NOT_KEEP_T
#define GS32_DRIVER_FUNC_T              //RAM_FUNC_T

#define GS32_DRIVER_INIT_FUNC_T         //ROM_FUNC_T
#define GS32_DRIVER_AES_FUNC_T          //ROM_FUNC_T

#define GS32_DRIVER_ASYSCTL_FUNC_T      //RAM_FUNC_T
#define GS32_DRIVER_ADC_FUNC_T          //RAM_FUNC_T
#define GS32_DRIVER_CMPSS_FUNC_T        //RAM_FUNC_T
#define GS32_DRIVER_CRC_FUNC_T          //RAM_FUNC_T
#define GS32_DRIVER_DMA_FUNC_T          //RAM_FUNC_T
#define GS32_DRIVER_GPIO_FUNC_T
#define GS32_DRIVER_IPC_FUNC_T          //RAM_FUNC_T
#define GS32_DRIVER_MCAN_FUNC_T			//RAM_FUNC_T
#define GS32_DRIVER_HAL_CAN_FUNC_T		//RAM_FUNC_T
#define GS32_DRIVER_CAN_FUNC_T			//RAM_FUNC_T
#define GS32_DRIVER_HRPWM_FUNC_T        //RAM_FUNC_T
#define GS32_DRIVER_EPWM_FUNC_T         //RAM_FUNC_T
#define GS32_DRIVER_SDFM_FUNC_T
#define GS32_DRIVER_EPG_FUNC_T
#define GS32_DRIVER_LIN_FUNC_T
#define GS32_DRIVER_PMBUS_FUNC_T
#define GS32_DRIVER_IIC_FUNC_T



#define __INIT_FUNC_T                   __attribute__ ((section(".init")))

#ifdef __cplusplus
}
#endif

#endif /* DEVICE_DRIVERLIB_HW_TYPES_H_ */
