/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */



#include <stdint.h>
#include "lin.h"

#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22) || IS_GS32F00xx(0x30)

/**
 * @brief Sets the operating mode.
 *
 * @param base LIN module base address.
 * @param opmode The operating mode.
 * @return int If successful, return 0; otherwise, return an error.
 */
GS32_DRIVER_LIN_FUNC_T static int LIN_setOpmode(uintptr_t base, LIN_opMode_t opmode)
{
	uint32_t reg_val;

	switch (opmode) {
		case LIN_UART_MODE :
		case LIN_LIN_MASTER_MODE :
		case LIN_LIN_SLAVE_MODE :
			reg_val = LIN_readRegister(base + LIN_US_MR);
			reg_val &= ~(LIN_US_MR_USART_MODE_M);
			reg_val |= (opmode & LIN_US_MR_USART_MODE_M);
			LIN_writeRegister(base + LIN_US_MR, reg_val);
			break;

		default:
			return -LIN_INVALID_PARAMTER;
	}

	return LIN_SUCCESS;
}

/**
 * @brief Sets the clock source.
 *
 * @param base LIN module base address.
 * @param clk_src The clock source.
 * @return int If successful, return 0; otherwise, return an error.
 */
GS32_DRIVER_LIN_FUNC_T static int LIN_setClockSource(uintptr_t base, LIN_clkSource_t clk_src)
{
	uint32_t reg_val;

	switch (clk_src) {
		case LIN_CLK_SRC_MCK :
		case LIN_CLK_SRC_MCK_DIV :
		case LIN_CLK_SRC_LIN_SCK :
			reg_val = LIN_readRegister(base + LIN_US_MR);
			reg_val &= ~(LIN_US_MR_USCLKS_M);
			reg_val |= ((clk_src & (LIN_US_MR_USCLKS_M >> LIN_US_MR_USCLKS_S)) <<
														LIN_US_MR_USCLKS_S);
			LIN_writeRegister(base + LIN_US_MR, reg_val);
			break;

		default :
			return -LIN_INVALID_PARAMTER;
	}

	return LIN_SUCCESS;
}

/**
 * @brief Sets the character length.
 *
 * @param base LIN module base address.
 * @param char_length The character length.
 * @return int If successful, return 0; otherwise, return an error.
 */
GS32_DRIVER_LIN_FUNC_T static int LIN_setCharLength(uintptr_t base, LIN_charLength_t char_length)
{
	uint32_t reg_val;

	reg_val = LIN_readRegister(base + LIN_US_MR);

	switch (char_length) {
		case LIN_CHAR_LENGTH_5BITS :
		case LIN_CHAR_LENGTH_6BITS :
		case LIN_CHAR_LENGTH_7BITS :
		case LIN_CHAR_LENGTH_8BITS :
			reg_val &= ~(LIN_US_MR_CHRL_M | LIN_US_MR_MODE9_M);
			reg_val |= ((char_length & (LIN_US_MR_CHRL_M >> LIN_US_MR_CHRL_S)) <<
															LIN_US_MR_CHRL_S);
			break;

		case LIN_CHAR_LENGTH_9BITS :
			reg_val |= LIN_US_MR_MODE9_M;
			break;

		default :
			return -LIN_INVALID_PARAMTER;
	}

	LIN_writeRegister(base + LIN_US_MR, reg_val);

	return LIN_SUCCESS;
}



/**
 * @brief Sets the parity type.
 *
 * @param base LIN module base address.
 * @param parity_type The parity type.
 * @return int If successful, return 0; otherwise, return an error.
 */
GS32_DRIVER_LIN_FUNC_T static int LIN_setParityType(uintptr_t base, LIN_parityType_t parity_type)
{
	uint32_t reg_val;

	switch (parity_type) {
		case LIN_PARITY_EVEN :
		case LIN_PARITY_ODD :
		case LIN_PARITY_SPACE :
		case LIN_PARITY_MARK :
		case LIN_PARITY_NONE :
		case LIN_PARITY_MULTIDROP :
			reg_val = LIN_readRegister(base + LIN_US_MR);
			reg_val &= ~(LIN_US_MR_PAR_M);
			reg_val |= ((parity_type & (LIN_US_MR_PAR_M >> LIN_US_MR_PAR_S)) <<
														LIN_US_MR_PAR_S);
			LIN_writeRegister(base + LIN_US_MR, reg_val);
			break;

		default :
			return -LIN_INVALID_PARAMTER;
	}

	return LIN_SUCCESS;
}

/**
 * @brief Sets the stop bit mode.
 *
 * @param base LIN module base address.
 * @param stop_bits Stop bit mode.
 * @return int If successful, return 0; otherwise, return an error.
 */
GS32_DRIVER_LIN_FUNC_T static int LIN_setStopBits(uintptr_t base, LIN_stopBitMode_t stop_bits)
{
	uint32_t reg_val;

	switch (stop_bits) {
		case LIN_STOP_BIT_1 :
		case LIN_STOP_BIT_1_5 :
		case LIN_STOP_BIT_2 :
			reg_val = LIN_readRegister(base + LIN_US_MR);
			reg_val &= ~(LIN_US_MR_NBSTOP_M);
			reg_val |= ((stop_bits & (LIN_US_MR_NBSTOP_M >> LIN_US_MR_NBSTOP_S)) <<
															LIN_US_MR_NBSTOP_S);
			LIN_writeRegister(base + LIN_US_MR, reg_val);
			break;

		default :
			return -LIN_INVALID_PARAMTER;
	}

	return LIN_SUCCESS;
}

/**
 * @brief Sets the channel mode.
 *
 * @param base LIN module base address.
 * @param channel_mode The channel mode.
 * @return int If successful, return 0; otherwise, return an error.
 */
GS32_DRIVER_LIN_FUNC_T static int LIN_setChannelMode(uintptr_t base, LIN_channelMode_t channel_mode)
{
	uint32_t reg_val;

	switch (channel_mode) {
		case LIN_CHAL_NORMAL_MODE :
		case LIN_CHAL_AUTO_ECHO_MODE :
		case LIN_CHAL_LOCAL_LOOPBACK_MODE :
		case LIN_CHAL_REMOTE_LOOPBACK_MODE :
			reg_val = LIN_readRegister(base + LIN_US_MR);
			reg_val &= ~(LIN_US_MR_CHMODE_M);
			reg_val |= ((channel_mode & (LIN_US_MR_CHMODE_M >> LIN_US_MR_CHMODE_S)) <<
																LIN_US_MR_CHMODE_S);
			LIN_writeRegister(base + LIN_US_MR, reg_val);
			break;

		default :
			return -LIN_INVALID_PARAMTER;
	}

	return LIN_SUCCESS;
}

GS32_DRIVER_LIN_FUNC_T int LIN_initModule(uintptr_t base, LIN_initParam_t *initParam)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	if (!initParam)
		return -LIN_INVALID_PARAMTER;

	/* Reset the module. */
	LIN_resetModule(base);

	/* Set the operating mode. */
	if (LIN_setOpmode(base, initParam->opmode))
		return -LIN_INVALID_PARAMTER;

	/* Set the character length. */
	if (LIN_setCharLength(base, initParam->char_length))
		return -LIN_INVALID_PARAMTER;

	/* USART synchronous mode. */
	LIN_setSyncMode(base, initParam->sync_mode);

	/* Set the parity type. */
	if (LIN_setParityType(base, initParam->parity))
		return -LIN_INVALID_PARAMTER;

	/* Set the stop bits. */
	if (LIN_setStopBits(base, initParam->stopbit))
		return -LIN_INVALID_PARAMTER;

	/* Set channel mode. */
	if (LIN_setChannelMode(base, initParam->chl_mode))
		return -LIN_INVALID_PARAMTER;

	/* Set the bit order is MSB or LSB. */
	LIN_setBitOrder(base, initParam->bit_order);

	/* Enable/Disable the SCK output pin. */
	LIN_setSCKOutput(base, initParam->sck_clk_out);

	/* Inverted Data. */
	LIN_setInvertedData(base, initParam->inverted_data_en);

	/*
	 * The USART filters the receive line using a
	 * three-sample filter (1/16-bit clock) (2 over 3 majority).
	 */
	LIN_ReceiveLineFilterEnable(base, initParam->rx_filter_en);

	/* Start Frame Delimiter Selector. */
	LIN_startFrameDeliniter(base, initParam->onebit);

	return LIN_SUCCESS;
}

GS32_DRIVER_LIN_FUNC_T int LIN_setBaudRate(uintptr_t base, uint32_t clock_frq,
					uint32_t baudrate, LIN_syncMode_t sync_mode,
					LIN_overSampMode_t over_sampling)
{
	uint32_t reg_val = 0;
	uint32_t over_sampl;
	uint16_t cd;
	uint8_t fp;

	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	if (baudrate == 0 || clock_frq == 0)
		return -LIN_INVALID_PARAMTER;

	/* Set LIN module clock source to main clock. */
	LIN_setClockSource(base, LIN_CLK_SRC_MCK);

	/* Set LIN module to synchronous/asynchronous mode. */
	LIN_setSyncMode(base, sync_mode);

	if (sync_mode == LIN_OP_ASYNC_MODE) {
		LIN_setOverSampling(base, over_sampling);
		over_sampl = (over_sampling == LIN_OVERSAMP_16) ? 16U : 8U;
	} else
		over_sampl = 1U;

	cd = (clock_frq / (baudrate * over_sampl));
	fp = ((8 * clock_frq) / (baudrate * over_sampl)) - cd;

	if (sync_mode == LIN_OP_ASYNC_MODE)
		reg_val |= ((fp & (LIN_US_BRGR_FP_M >> LIN_US_BRGR_FP_S)) << LIN_US_BRGR_FP_S);

	reg_val |= (cd & LIN_US_BRGR_CD_M);

	LIN_writeRegister(base + LIN_US_BRGR, reg_val);

	return LIN_SUCCESS;
}

/**
 * @brief Get the data length in LIN mode.
 *
 * @param base LIN module base address.
 * @param length data length.
 * @return int If successful, return 0; otherwise, return an error.
 */
static int LIN_LinGetDataLength(uintptr_t base, uint16_t *length)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	*length = (uint16_t)((LIN_readRegister(base + LIN_US_LINMR) &
												  LIN_US_LINMR_DLC_M) >>
												  LIN_US_LINMR_DLC_S);

	*length += 1;

	return LIN_SUCCESS;
}

GS32_DRIVER_LIN_FUNC_T int LIN_LinSetDataLength(uintptr_t base, uint16_t length)
{
	uint32_t reg_val;

	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	if (length == 0 || length > 256)
		return -LIN_INVALID_PARAMTER;

	length = length - 1;

	reg_val = LIN_readRegister(base + LIN_US_LINMR);

	reg_val &= ~(LIN_US_LINMR_DLC_M);

	reg_val |= ((length & (LIN_US_LINMR_DLC_M >> LIN_US_LINMR_DLC_S)) <<
												LIN_US_LINMR_DLC_S);

	LIN_writeRegister(base + LIN_US_LINMR, reg_val);

	return LIN_SUCCESS;
}

GS32_DRIVER_LIN_FUNC_T int LIN_LinModeInit(uintptr_t base, LIN_LinModeInitParam_t *initParam)
{
	LIN_opMode_t mode;
	uint32_t reg_val;

	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	if (!initParam)
		return -LIN_INVALID_PARAMTER;

	mode = LIN_getCurrentMode(base);
	if (mode != LIN_LIN_MASTER_MODE && mode != LIN_LIN_SLAVE_MODE)
		return -LIN_INVALID_PARAMTER;

	reg_val = LIN_readRegister(base + LIN_US_LINMR);

	/* Enable/Disable the Identifier Parity is computed and sent automatically. */
	if (initParam->parity_en)
		reg_val &= ~LIN_US_LINMR_PARDIS_M;
	else
		reg_val |= LIN_US_LINMR_PARDIS_M;

	/* Enable/Disable the Checksum is computed and sent automatically. */
	if (initParam->checksum_en)
		reg_val &= ~LIN_US_LINMR_CHKDIS_M;
	else
		reg_val |= LIN_US_LINMR_CHKDIS_M;

	/*
	 * Select the Checksum Type.
	 * Setting the bit LINWKUP in the control register sends a LIN 2.0/1.3 wakeup signal.
	 */
	if (initParam->iso_mode == LIN_LIN_ISO_MODE_1_3)
		reg_val |= (LIN_US_LINMR_CHKTYP_M | LIN_US_LINMR_WKUPTYP_M);
	else
		reg_val &= ~(LIN_US_LINMR_CHKTYP_M | LIN_US_LINMR_WKUPTYP_M);

	/* Select response data length is defined by the DLC register or IDCHR register. */
	if (initParam->data_length_mode == LIN_LIN_DLM_BY_IDCHR)
		reg_val |= LIN_US_LINMR_DLM_M;
	else
		reg_val &= ~LIN_US_LINMR_DLM_M;

	/* Enable/Disable frame slot mode only in LIN Master mode. */
	if (initParam->frame_slot_en)
		reg_val &= ~LIN_US_LINMR_FSDIS_M;
	else
		reg_val |= LIN_US_LINMR_FSDIS_M;

	/* Enable/Disable DMA mode. */
	if (initParam->dma_en)
		reg_val |= LIN_US_LINMR_PDCM_M;
	else
		reg_val &= ~LIN_US_LINMR_PDCM_M;

	/* The Synchronization procedure is performed in LIN Slave node configuration. */
	if (initParam->slave_sync_en)
		reg_val &= ~LIN_US_LINMR_SYNCDIS_M;
	else
		reg_val |= LIN_US_LINMR_SYNCDIS_M;

	LIN_writeRegister(base + LIN_US_LINMR, reg_val);

	return LIN_SUCCESS;
}

GS32_DRIVER_LIN_FUNC_T int LIN_LinSlaveReceiveFrameBlock(uintptr_t base, LIN_LinMsg_t *msg)
{
	uint32_t lin_status;
	uint16_t i = 0;
	uint16_t length;

	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	/* Waitting for the LINID received. */
	LIN_getStatus(base, &lin_status);

	if (lin_status & LIN_LIN_CHANNEL_LINID)
		msg->id = LIN_LinGetID(base);
	else
		return -LIN_RXEMPTY;

	/* Get the length of the frame. */
	LIN_LinGetDataLength(base, &length);

	if (length > LIN_FRAME_MAX_LENGTH)
			return -LIN_INVALID_PARAMTER;

	for (i = 0; i < length;) {
		LIN_getStatus(base, &lin_status);

		/*  If the data has been received, read the data. */
		if (lin_status & LIN_LIN_CHANNEL_RXRDY)
			msg->data[i++] = (uint8_t)LIN_readRegister(base + LIN_US_RHR);

		/*
		 * If all data has not been received after transmission,
		 * an error occurs during the receiving process.
		 */
		if ((lin_status & LIN_LIN_CHANNEL_LINTC) && (i <= length) ) {
			LIN_resetTransferStatus(base);
			return -LIN_RXERR;
		}
	}

	/* Waitting for the transmission to complete. */
	while (1) {
		LIN_getStatus(base, &lin_status);
		if (lin_status & LIN_LIN_CHANNEL_LINTC)
			break;
	}

	/* Reset the transfer status. */
	LIN_resetTransferStatus(base);

	return LIN_SUCCESS;
}

GS32_DRIVER_LIN_FUNC_T int LIN_LinSlaveSendFrameBlock(uintptr_t base, LIN_LinMsg_t *msg)
{
	uint32_t lin_status;
	uint16_t i = 0;
	uint16_t length;

	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	/* Waitting for the LINID received. */
	LIN_getStatus(base, &lin_status);

	if (lin_status & LIN_LIN_CHANNEL_LINID)
		msg->id = LIN_LinGetID(base);
	else
		return -LIN_RXEMPTY;

	/* Get the length of the frame. */
	LIN_LinGetDataLength(base, &length);

	if (length > LIN_FRAME_MAX_LENGTH)
		return -LIN_INVALID_PARAMTER;

	for (i = 0; i < length;) {
		LIN_getStatus(base, &lin_status);

		/* If Tx FIFO is empty, send data. */
		if (lin_status & LIN_LIN_CHANNEL_TXRDY)
			LIN_writeRegister(base + LIN_US_THR, msg->data[i++]);

		/*
		 * If all data has not been sent after transmission,
		 * an error occurs during the sending process.
		 */
		if ((lin_status & LIN_LIN_CHANNEL_LINTC) && (i <= length) ) {
			LIN_resetTransferStatus(base);
			return -LIN_TXERR;
		}
	}

	/* Waitting for the transmission to complete. */
	while (1) {
		LIN_getStatus(base, &lin_status);
		if (lin_status & LIN_LIN_CHANNEL_LINTC)
			break;
	}

	/* Reset the transfer status. */
	LIN_resetTransferStatus(base);

	return LIN_SUCCESS;
}

GS32_DRIVER_LIN_FUNC_T int LIN_LinMasterSendFrameBlock(uintptr_t base, LIN_LinMsg_t *msg)
{
	uint32_t lin_status;
	uint16_t length;

	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	/* Set the LIN ID. */
	LIN_getStatus(base, &lin_status);

	if (lin_status & LIN_LIN_CHANNEL_TXRDY)
		LIN_LinSetID(base, msg->id);
	else
		return -LIN_TXFULL;

	/* Get the length of the frame.*/
	LIN_LinGetDataLength(base, &length);

	for (uint16_t i = 0; i < length;) {
		LIN_getStatus(base, &lin_status);

		/* If Tx FIFO is empty, send data. */
		if (lin_status & LIN_LIN_CHANNEL_TXRDY)
			LIN_writeRegister(base + LIN_US_THR, msg->data[i++]);

		/*
		 * If all data has not been sent after transmission,
		 * an error occurs during the sending process.
		 */
		if ((lin_status & LIN_LIN_CHANNEL_LINTC) && (i <= length) ) {
			LIN_resetTransferStatus(base);
			return -LIN_TXERR;
		}
	}

	/* Waitting for the transmission to complete. */
	while (1) {
		LIN_getStatus(base, &lin_status);
		if (lin_status & LIN_LIN_CHANNEL_LINTC)
			break;
	}

	/* Reset the transfer status. */
	LIN_resetTransferStatus(base);

	return LIN_SUCCESS;
}

GS32_DRIVER_LIN_FUNC_T int LIN_LinMasterReceiveFrameBlock(uintptr_t base, LIN_LinMsg_t *msg)
{
	uint32_t lin_status;
	uint16_t length;

	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	/* Set the LIN ID. */
	LIN_getStatus(base, &lin_status);

	if (lin_status & LIN_LIN_CHANNEL_TXRDY)
		LIN_LinSetID(base, msg->id);
	else
		return -LIN_TXFULL;

	/* Get the length of the frame.*/
	LIN_LinGetDataLength(base, &length);

	for (uint16_t i = 0; i < length;) {
		LIN_getStatus(base, &lin_status);

		/* If the data has been received, read the data. */
		if (lin_status & LIN_LIN_CHANNEL_RXRDY)
			msg->data[i++] = (uint8_t)LIN_readRegister(base + LIN_US_RHR);

		/*
		 * If all data has not been received after transmission,
		 * an error occurs during the receiving process.
		 */
		if ((lin_status & LIN_LIN_CHANNEL_LINTC) && (i <= length) ) {
			LIN_resetTransferStatus(base);
			return -LIN_RXERR;
		}
	}

	/* Waitting for the transmission to complete. */
	while (1) {
		LIN_getStatus(base, &lin_status);
		if (lin_status & LIN_LIN_CHANNEL_LINTC)
			break;
	}

	/* Reset the transfer status. */
	LIN_resetTransferStatus(base);

	return LIN_SUCCESS;
}

#endif
