/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "mcan.h"

/**
 * @brief Set the MCAN enter internal or external lookback mode.
 * NOTE: Loopback mode cannot be used in conjunction with listening only mode.
 *
 * @param base indicates that the MCAN instance base address.
 * @param mode Select internal/external loopback mode.
 */
GS32_DRIVER_MCAN_FUNC_T bool
MCAN_lpbkModeEnable(uint32_t base, MCAN_LpbkMode_t mode, bool enable)
{
	if (!MCAN_isBaseValid(base))
		return false;

	/* Unlock protected registers. */
	if (!MCAN_writeProtectedRegAccessUnlock(base))
		return false;

	if (enable) {
		if (HWREG(base + M_CAN_CCCR) & M_CAN_CCCR_ASM_M) {
			MCAN_writeProtectedRegAccessLock(base);
			return false;
		}
		HWREG(base + M_CAN_CCCR) |= M_CAN_CCCR_TEST_M;
		HWREG(base + M_CAN_TEST) |= M_CAN_TEST_LBCK_M;
	} else {
		/* Disable test mode. */
		HWREG(base + M_CAN_CCCR) &= ~M_CAN_CCCR_TEST_M;
		HWREG(base + M_CAN_TEST) &= ~M_CAN_TEST_LBCK_M;
		if (!MCAN_writeProtectedRegAccessLock(base))
				return false;
		return true;
	}

	switch (mode) {
		case MCAN_LPBK_MODE_INTERNAL :
			/* Internal loopback mode need to enable MON mode. */
			MCAN_setListenOnlyMode(base, MCAN_LISTEN_ONLY_MON_MODE);
			break;

		case MCAN_LPBK_MODE_EXTERNAL :
			MCAN_setListenOnlyMode(base, MCAN_LISTEN_ONLY_MODE_DISABLE);
			break;

		default :
			/* Lock protected registers. */
			MCAN_writeProtectedRegAccessLock(base);
			return false;
	}

	/* Lock protected registers. */
	if (!MCAN_writeProtectedRegAccessLock(base))
		return false;

	return true;
}



/**
 * @brief Configure data Transmission bitrate in CAN2.0 or
 * arbitration phase transmission bitrate in CANFD mode.
 * NOTE: Bitrate = frequency / ((ntseg1 + ntseg2 + 3) * (nbrp + 1)).
 * ntseg1 = prop_seg + phase_seg1.
 * If frequency = 8M, nbrp = 0, ntseg1 = 10, ntseg2 = 3, Bitrate = 500kbps.
 * The nsjw valid vaule is <= phase_seg1).
 *
 * @param base indicates that the MCAN instance base address.
 * @param bitrate bitrate initialization parameters.
 */
GS32_DRIVER_MCAN_FUNC_T static bool
MCAN_setArbitrationPhaseTiming(uint32_t base, uint16_t nbrp,
							uint8_t ntseg1, uint8_t ntseg2, uint8_t nsjw)
{
	uint32_t regval, tmpval;

	/* Unlock protected registers. */
	if (!MCAN_writeProtectedRegAccessUnlock(base))
		return false;

	/* Set NSJW. */
	tmpval = (uint32_t)nsjw & (M_CAN_NBTP_NSJW_M >> M_CAN_NBTP_NSJW_S);
	regval = tmpval << M_CAN_NBTP_NSJW_S;
	/* Set NBRP. */
	tmpval = (uint32_t)nbrp & (M_CAN_NBTP_NBRP_M >> M_CAN_NBTP_NBRP_S);
	regval |= (tmpval << M_CAN_NBTP_NBRP_S);
	/* Set NTSEG1. */
	tmpval = (uint32_t)ntseg1 & (M_CAN_NBTP_NTSEG1_M >> M_CAN_NBTP_NTSEG1_S);
	regval |= (tmpval << M_CAN_NBTP_NTSEG1_S);
	/* Set NTSEG2. */
	regval |= ((uint32_t)ntseg2 & M_CAN_NBTP_NTSEG2_M);

	HWREG(base + M_CAN_NBTP) = regval;

	return MCAN_writeProtectedRegAccessLock(base);
}

/**
 * @brief Configure data phase transmission bitrate in CANFD mode.
 * NOTE: Bitrate = frequency / ((dtseg1 + dtseg2 + 3) * (dbrp + 1)).
 * dtseg1 = prop_seg + phase_seg1.
 * If frequency = 8M, nbrp = 0, dtseg1 = 10, dtseg2 = 3, Bitrate = 500kbps.
 * The dsjw valid vaule is <= phase_seg1.
 *
 * @param base indicates that the MCAN instance base address.
 * @param bitrate indicates that bitrate initialization parameters.
 */
GS32_DRIVER_MCAN_FUNC_T GS32_DRIVER_MCAN_FUNC_T static bool
MCAN_SetDataPhaseTiming(uint32_t base, uint8_t dataRatePrescalar,
						uint8_t dataTimeSeg1, uint8_t dataTimeSeg2,
						uint8_t dataSynchJumpWidth)
{
	uint32_t regval, tmpval;

	/* Unlock protected registers. */
	if (!MCAN_writeProtectedRegAccessUnlock(base))
		return false;

	regval = HWREG(base + M_CAN_DBTP);

	regval &= M_CAN_DBTP_TDC_M;
	/* Set DBRP. */
	tmpval = (uint32_t)dataRatePrescalar & (M_CAN_DBTP_DBRP_M >> M_CAN_DBTP_DBRP_S);
	regval |= (tmpval << M_CAN_DBTP_DBRP_S);
	/* Set DTSEG1. */
	tmpval = (uint32_t)dataTimeSeg1 & (M_CAN_DBTP_DTSEG1_M >> M_CAN_DBTP_DTSEG1_S);
	regval |= (tmpval << M_CAN_DBTP_DTSEG1_S);
	/* Set DTSEG2. */
	tmpval = (uint32_t)dataTimeSeg2 & (M_CAN_DBTP_DTSEG2_M >> M_CAN_DBTP_DTSEG2_S);
	regval |= (tmpval << M_CAN_DBTP_DTSEG2_S);
	/* Set DSJW. */
	tmpval = (uint32_t)dataSynchJumpWidth & M_CAN_DBTP_DSJW_M;
	regval |= tmpval;

	HWREG(base + M_CAN_DBTP) = regval;

	return MCAN_writeProtectedRegAccessLock(base);
}

/**
 * @brief Configure the Transmission Delay Compensation in CANFD mode.
 * NOTE: Offset = Phase_Seg1 + Prop_Seg + 2.
 * To avoid that a dominant glitch inside the received FDF bit ends the delay compensation
 * measurement before the falling edge of the received res bit, resulting in a to early
 * SSP position, the use of a transmitter delay compensation filter window can be
 * enabled by programming TDCR.TDCF.This defines a minimum value for the SSP position.
 * Dominant edges on m_can_rx, that would result in an earlier SSP position are ignored
 * for transmitter delay measurement. The measurement is stopped when the SSP position
 * is at least TDCR.TDCF AND m_can_rx is low.
 *
 * @param base indicates that the MCAN instance base address.
 * @param tdc indicates that TDC initialization parameters.
 */
GS32_DRIVER_MCAN_FUNC_T GS32_DRIVER_MCAN_FUNC_T static void
MCAN_SetTDC(uint32_t base, bool tdcen, MCAN_TDCConfig_t *tdc)
{
	uint32_t regval;

	/* Disable TDC. */
	if (!tdcen) {
		HWREG(base + M_CAN_DBTP) &= ~M_CAN_DBTP_TDC_M;

		return ;
	}

	/* Set TDC parameters. */
	regval = ((tdc->tdco & (M_CAN_TDCR_TDCO_M >> M_CAN_TDCR_TDCO_S)) <<
												M_CAN_TDCR_TDCO_S);
	regval |= (tdc->tdcf & M_CAN_TDCR_TDCF_M);

	HWREG(base + M_CAN_TDCR) = regval;

	/* Enable TDC. */
	HWREG(base + M_CAN_DBTP) |= M_CAN_DBTP_TDC_M;
}

/**
 * @brief Configure the MCAN global filter.
 *
 * @param base indicates that the MCAN instance base address.
 * @param global_filter indicates that global filter initialization parameters.
 */
GS32_DRIVER_MCAN_FUNC_T static void
MCAN_setGlobalFilter(uint32_t base, MCAN_GlobalFiltConfig_t *global_filter)
{
	uint32_t regval, tmpval;

	/* Reject all remote frames with 29-bit extended IDs. */
	if (global_filter->rrfe)
		regval |= M_CAN_GFC_RRFE_M;
	else
	/* Filter remote frames with 29-bit extended IDs. */
		regval &= ~M_CAN_GFC_RRFE_M;

	/* Reject all remote frames with 11-bit standard IDs. */
	if (global_filter->rrfs)
		regval |= M_CAN_GFC_RRFS_M;
	else
	/* Filter remote frames with 11-bit standard IDs. */
		regval &= ~M_CAN_GFC_RRFS_M;

	/* Reject/Accept Non-matching Frames Extended. */
	tmpval = global_filter->anfe & (M_CAN_GFC_ANFE_M >> M_CAN_GFC_ANFE_S);
	regval |= (tmpval << M_CAN_GFC_ANFE_S);

	/* Reject/Accept Non-matching Frames Standard */
	tmpval = global_filter->anfs & (M_CAN_GFC_ANFS_M >> M_CAN_GFC_ANFS_S);
	regval |= (tmpval << M_CAN_GFC_ANFS_S);

	HWREG(base + M_CAN_GFC) = regval;
}

/**
 * @brief Initialize timeout counter.
 *
 * @param base indicates that the MCAN instance base address.
 * @param timeout_counter Timeout counter configuration parameters.
 */
GS32_DRIVER_MCAN_FUNC_T static void
MCAN_setTimeoutCounter(uint32_t base, bool enable,
					MCAN_TimeOutSelect_t timeoutSelect,
					uint16_t timeoutPreload)
{
	uint32_t regval, tmpval;

	if (!enable) {
		/* Disable timeout counter. */
		HWREG(base + M_CAN_TOCC) &= ~M_CAN_TOCC_ETOC_M;
		return ;
	}

	/* Set timeout counter timeout period. */
	tmpval = timeoutPreload & (M_CAN_TOCC_TOP_M >> M_CAN_TOCC_TOP_S);
	regval = (tmpval << M_CAN_TOCC_TOP_S);

	/* Select timeout trigger event. */
	tmpval = timeoutSelect & (M_CAN_TOCC_TOS_M >> M_CAN_TOCC_TOS_S);
	regval |= (tmpval << M_CAN_TOCC_TOS_S);

	/* Enable timeout counter. */
	regval |= M_CAN_TOCC_ETOC_M;

	HWREG(base + M_CAN_TOCC) = regval;
}

/**
 * @brief Initialize the MCAN CANFD mode.
 *
 * @param base indicates that the MCAN instance base address.
 * @param canfd indicates that CANFD mode initialization parameters.
 */
GS32_DRIVER_MCAN_FUNC_T static void
MCAN_FDModeInit(uint32_t base, bool fden, MCAN_FDISOMode_t format,
				bool brs, bool tdcen, MCAN_TDCConfig_t *tdc)
{
	uint32_t regval;

	if (!fden) {
		/* Disable CANFD mode. */
		HWREG(base + M_CAN_CCCR) &= ~M_CAN_CCCR_FDOE_M;
		return ;
	}

	/* Set Transmitter Delay Compensation. */
	MCAN_SetTDC(base, tdcen, tdc);

	regval = HWREG(base + M_CAN_CCCR);
	/* Select CANFD frame format. */
	if (format == MCAN_FD_ISO_11898_1)
		regval &= ~M_CAN_CCCR_NISO_M;
	else
		regval |= M_CAN_CCCR_NISO_M;

	/* Set Bit Rate Switch Enable/Disable. */
	if (brs)
		regval |= M_CAN_CCCR_BRSE_M;
	else
		regval &= ~M_CAN_CCCR_BRSE_M;

	/* Enable CANFD mode. */
	regval |= M_CAN_CCCR_FDOE_M;
	HWREG(base + M_CAN_CCCR) = regval;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_addStdMsgIDFilter(uint32_t base,
					uint8_t filter_number,
					const MCAN_StdMsgIDFilterElement_t *std_filter)
{
	uint32_t regval;
	uint32_t filter_base_address;
	uint32_t element_address;
	uint8_t filter_size;

	if (!MCAN_isBaseValid(base))
		return false;

	if (std_filter == NULL)
		return false;

	filter_size = MCAN_getStdFilterSize(base);

	if (filter_number > filter_size)
		return false;

	/* Calculate standard ID filter element address. */
	filter_base_address = MCAN_getStdFilterBaseAddress(base);
	filter_base_address = filter_base_address + MCAN_MRAM_BASE(base);
	element_address = filter_base_address + ((uint32_t)filter_number *
					M_CAN_WORD_SIZE_OF_STD_FILTER * 4U);

	/* Configture standard filter ID 2. */
	regval = std_filter->sfid2 & MCAN_STD_SFID2_M;
	/* Configure standard sync message. */
	regval |= ((std_filter->ssync & (MCAN_STD_SSYNC_M >> MCAN_STD_SSYNC_S)) << MCAN_STD_SSYNC_S);
	/* Configture standard filter ID 1. */
	regval |= ((std_filter->sfid1 & (MCAN_STD_SFID1_M >> MCAN_STD_SFID1_S)) << MCAN_STD_SFID1_S);
	/* Standard filter element configuration. */
	regval |= ((std_filter->sfec & (MCAN_STD_SFEC_M >> MCAN_STD_SFEC_S)) << MCAN_STD_SFEC_S);
	/* Configure standard filter type. */
	regval |= ((std_filter->sft & (MCAN_STD_SFT_M >> MCAN_STD_SFT_S)) << MCAN_STD_SFT_S);

	HWREG(element_address) = regval;

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_addExtMsgIDFilter(uint32_t base,
					uint8_t filter_number,
					const MCAN_ExtMsgIDFilterElement_t *ext_filter)
{
	uint32_t regval;
	uint32_t filter_base_address, element_address;
	uint8_t filter_size;

	if (!MCAN_isBaseValid(base))
		return false;

	if (ext_filter == NULL)
		return false;

	filter_size = MCAN_getExtFilterSize(base);

	if (filter_number > filter_size)
		return false;

	/* Calculate extended ID filter element address. */
	filter_base_address = MCAN_getExtFilterBaseAddress(base);
	filter_base_address = filter_base_address + MCAN_MRAM_BASE(base);
	element_address = filter_base_address + ((uint32_t)filter_number *
					M_CAN_WORD_SIZE_OF_EXT_FILTER * 4U);

	/* Configure low 32 bit. */
	/* Configure extended filter ID 1. */
	regval = (ext_filter->efid1 & MCAN_EXT_EFID1_M);
	/* Extended filter element configuration. */
	regval |= ((ext_filter->efec & (MCAN_EXT_EFEC_M >> MCAN_EXT_EFEC_S)) <<
											MCAN_EXT_EFEC_S);
	HWREG(element_address) = regval;

	/* Configure high 32 bit. */
	element_address += 4;
	/* Configure extended filter ID 2. */
	regval = ext_filter->efid2 & MCAN_EXT_EFID2_M;
	/* Configure extended sync message. */
	regval |= ((ext_filter->esync & (MCAN_EXT_ESYNC_M >> MCAN_EXT_ESYNC_S)) <<
																MCAN_EXT_ESYNC_S);
	/* Configure extended filter type. */
	regval |= ((ext_filter->eft & (MCAN_EXT_EFT_M >> MCAN_EXT_EFT_S)) << MCAN_EXT_EFT_S);
	HWREG(element_address) = regval;

	return true;
}


/**
 * @brief Configure the maximum data format for Rx buffer, Rx fifo0,
 * Rx fifo1 and Tx buffer.
 * 
 * @param base indicates that the MCAN instance base address.
 * @param data_format indicates that the data format parameter.
 * @param fd_enable Check if CANFD mode is enabled.
 */
GS32_DRIVER_MCAN_FUNC_T static bool
MCAN_setRxBufferFifoDataFormat(uint32_t base,
							MCAN_ElemSize_t txBufElemSize,
							MCAN_ElemSize_t rxBufElemSize,
							MCAN_ElemSize_t rxFIFO0ElemSize,
							MCAN_ElemSize_t rxFIFO1ElemSize)
{
	uint32_t regval;

	/* Set Tx buffer maximum of data length. */
	HWREG(base + M_CAN_TXESC) = txBufElemSize & M_CAN_TXESC_TBDS_M;

	/* Set Rx fifo0 maximum of data length. */
	regval = (uint32_t)rxFIFO0ElemSize & M_CAN_RXESC_F0DS_M;
	/* Set Rx fifo1 maximum of data length. */
	regval |= (((uint32_t)rxFIFO1ElemSize &
							(M_CAN_RXESC_F1DS_M >> M_CAN_RXESC_F1DS_S)) <<
							 M_CAN_RXESC_F1DS_S);
	/* Set Rx buffer maximum of data length. */
	regval |= (((uint32_t)rxBufElemSize &
							(M_CAN_RXESC_RBDS_M >> M_CAN_RXESC_RBDS_S)) <<
							 M_CAN_RXESC_RBDS_S);

	HWREG(base + M_CAN_RXESC) = regval;

	return true;
}

/**
 * @brief Initialize Rx fifo 0.
 * 
 * @param base indicates that the MCAN instance base address.
 * @param rxfifo0 indicates that the Rx fifo 0 initialization parameters.
 * @return true Rx fifo 0 initialization successful.
 * @return false Rx fifo 0 initialization failed.
 */
GS32_DRIVER_MCAN_FUNC_T static bool
MCAN_setRxFifo0(uint32_t base, uint16_t rxFIFO0startAddr,
				MCAN_RxFifoOperationMode_t rxFIFO0OpMode,
				uint8_t rxFIFO0size, uint8_t rxFIFO0waterMark)
{
	uint32_t regval;

	if (rxFIFO0size > MCAN_RX_BUFFER_MAX || rxFIFO0waterMark > MCAN_RX_BUFFER_MAX)
		return false;

	/* Initialize Rx FIFO 0. */
	/* Rx FIFO 0 Start Address. */
	regval = rxFIFO0startAddr & M_CAN_RXF0C_F0SA_M;
	/* Rx FIFO 0 Size. */
	regval |= ((uint32_t)(rxFIFO0size &
			(M_CAN_RXF0C_F0S_M >> M_CAN_RXF0C_F0S_S)) <<
			M_CAN_RXF0C_F0S_S);
	/* Rx FIFO 0 Watermark. */
	regval |= ((uint32_t)(rxFIFO0waterMark &
			(M_CAN_RXF0C_F0WM_M >> M_CAN_RXF0C_F0WM_S)) <<
			M_CAN_RXF0C_F0WM_S);
	/* FIFO 0 Operation Mode. */
	regval |= ((uint32_t)(rxFIFO0OpMode &
			(M_CAN_RXF0C_F0OM_M >> M_CAN_RXF0C_F0OM_S)) <<
			M_CAN_RXF0C_F0OM_S);

	HWREG(base + M_CAN_RXF0C) = regval;

	return true;
}

/**
 * @brief Initialize Rx fifo 1.
 * 
 * @param base indicates that the MCAN instance base address.
 * @param rxfifo1 indicates that the Rx fifo 1 initialization parameters.
 * @return true Rx fifo 1 initialization successful.
 * @return false Rx fifo 1 initialization failed.
 */
GS32_DRIVER_MCAN_FUNC_T static bool
MCAN_setRxFifo1(uint32_t base, uint16_t rxFIFO1startAddr,
				MCAN_RxFifoOperationMode_t rxFIFO1OpMode,
				uint8_t rxFIFO1size, uint8_t rxFIFO1waterMark)
{
	uint32_t regval;

	if (rxFIFO1size > MCAN_RX_BUFFER_MAX || rxFIFO1waterMark > MCAN_RX_BUFFER_MAX)
		return false;

	/* Rx FIFO 1 Start Address. */
	regval = rxFIFO1startAddr & M_CAN_RXF1C_F1SA_M;
	/* Rx FIFO 1 Size. */
	regval |= ((uint32_t)(rxFIFO1size &
			(M_CAN_RXF1C_F1S_M >> M_CAN_RXF1C_F1S_S)) <<
			M_CAN_RXF1C_F1S_S);
	/* Rx FIFO 1 Watermark. */
	regval |= ((uint32_t)(rxFIFO1waterMark &
			(M_CAN_RXF1C_F1WM_M >> M_CAN_RXF1C_F1WM_S)) <<
			M_CAN_RXF1C_F1WM_S);
	/* FIFO 1 Operation Mode. */
	regval |= ((uint32_t)(rxFIFO1OpMode &
			(M_CAN_RXF1C_F1OM_M >> M_CAN_RXF1C_F1OM_S)) <<
			M_CAN_RXF1C_F1OM_S);

	HWREG(base + M_CAN_RXF1C) = regval;

	return true;
}

/**
 * @brief Initialize Tx buffer.
 * 
 * @param base indicates that the MCAN instance base address.
 * @param txbuf indicates that the Tx buffer initialization parameters.
 * @return true Tx buffer initialization successful.
 * @return false Tx buffer initialization failed.
 */
GS32_DRIVER_MCAN_FUNC_T static bool
MCAN_setTxBuffer(uint32_t base, uint16_t txStartAddr,
				uint8_t txBufNum, uint8_t txFIFOSize,
				MCAN_TxBufferOpMode_t txBufMode)
{
	uint32_t regval;

	/* Be aware that the sum of TFQS and NDTB may be not greater than 32. There is no check
	for erroneous configurations. The Tx Buffers section in the Message RAM starts with the
	dedicated Tx Buffers. */

	if ((txBufNum + txFIFOSize) > MCAN_TX_BUFFER_MAX)
		return false;

	/* Tx Buffers Start Address. */
	regval = txStartAddr & M_CAN_TXBC_TBSA_M;
	/* Number of Dedicated Transmit Buffers. */
	regval |= ((uint32_t)(txBufNum & (M_CAN_TXBC_NDTB_M >> M_CAN_TXBC_NDTB_S)) << M_CAN_TXBC_NDTB_S);
	/* Transmit FIFO/Queue Size. */
	regval |= ((uint32_t)(txFIFOSize & (M_CAN_TXBC_TFQS_M >> M_CAN_TXBC_TFQS_S)) << M_CAN_TXBC_TFQS_S);
	/* Tx FIFO/Queue Mode. */
	regval |= ((uint32_t)(txBufMode & (M_CAN_TXBC_TFQM_M >> M_CAN_TXBC_TFQM_S)) << M_CAN_TXBC_TFQM_S);

	HWREG(base + M_CAN_TXBC) = regval;

	return true;
}

/**
 * @brief Initialize the TX event fifo.
 * 
 * @param base indicates that the MCAN instance base address.
 * @param txeventfifo indicates that the Tx event initialization parameters.
 * @return true Tx event fifo initialization successful.
 * @return false Tx event fifo initialization failed.
 */
GS32_DRIVER_MCAN_FUNC_T static bool
MCAN_setTxEventFifo(uint32_t base, uint16_t txEventFIFOStartAddr,
					uint8_t txEventFIFOSize, uint8_t txEventFIFOWaterMark)
{
	uint32_t regval;

	if (txEventFIFOSize > MCAN_TX_EVENT_BUFFER_MAX ||
		txEventFIFOWaterMark > MCAN_TX_EVENT_BUFFER_MAX)
		return false;

	/* Tx Event FIFO Start Address. */
	regval = txEventFIFOStartAddr & M_CAN_TXEFC_EFSA_M;
	/* Tx Event FIFO Size. */
	regval |= ((uint32_t)(txEventFIFOSize &
			(M_CAN_TXEFC_EFS_M >> M_CAN_TXEFC_EFS_S)) <<
			M_CAN_TXEFC_EFS_S);
	/* Tx Event FIFO Watermark. */
	regval |= ((uint32_t)(txEventFIFOWaterMark &
			(M_CAN_TXEFC_EFWM_M >> M_CAN_TXEFC_EFWM_S)) <<
			M_CAN_TXEFC_EFWM_S);

	HWREG(base + M_CAN_TXEFC) = regval;

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_msgRAMConfig(uint32_t base, MCAN_MsgRAMConfigParams_t *RAMConfig)
{
	if (!MCAN_isBaseValid(base))
		return false;

	if (RAMConfig == NULL)
		return false;

	MCAN_writeProtectedRegAccessUnlock(base);

	/* Initialize STD filters */
	MCAN_setStdIDFilter(base, RAMConfig->lss, RAMConfig->flssa);

	/* Initialize EXT filters. */
	MCAN_setExtIDFilter(base, RAMConfig->lse, RAMConfig->flesa);

	/* Initialize Rx FIFO 0. */
	if (!MCAN_setRxFifo0(base, RAMConfig->rxFIFO0startAddr,
						RAMConfig->rxFIFO0OpMode, RAMConfig->rxFIFO0size,
						RAMConfig->rxFIFO0ElemSize))
		goto error_ret;

	/* Initialize Rx FIFO 1. */
	if (!MCAN_setRxFifo1(base, RAMConfig->rxFIFO1startAddr,
						RAMConfig->rxFIFO1OpMode, RAMConfig->rxFIFO1size,
						RAMConfig->rxFIFO1ElemSize))
		goto error_ret;

	/* Initialize Rx Buffer. */
	MCAN_setRxBuffer(base, RAMConfig->rxBufStartAddr);

	/* Initialize Tx Buffer. */
	if (!MCAN_setTxBuffer(base, RAMConfig->txStartAddr,
								RAMConfig->txBufNum,
								RAMConfig->txFIFOSize,
								RAMConfig->txBufMode))
		goto error_ret;

	/* Initialize Tx Event Fifo. */
	if (!MCAN_setTxEventFifo(base, RAMConfig->txEventFIFOStartAddr,
						RAMConfig->txEventFIFOSize, RAMConfig->txEventFIFOWaterMark))
		goto error_ret;

	/* Initialize Tx buffer/fifo element size. */
	if (!MCAN_setRxBufferFifoDataFormat(base, RAMConfig->txBufElemSize,
								RAMConfig->rxBufElemSize, RAMConfig->rxFIFO0ElemSize,
								RAMConfig->rxFIFO1ElemSize))
		goto error_ret;

	if (!MCAN_writeProtectedRegAccessLock(base))
		return false;

	return true;

error_ret:
	MCAN_writeProtectedRegAccessLock(base);

	return false;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_init(uint32_t base, MCAN_InitParams_t *initparams)
{
	MCAN_OperationMode_t op_mode;

	if (!MCAN_isBaseValid(base))
		return false;

	if (initparams == NULL)
		return false;

	/* Configuration MCAN wakeup and emulation. */
	MCAN_enableWakupReq(base, initparams->wkupReqEnable);
	MCAN_enableAutoWakeup(base, initparams->autoWkupEnable);
	MCAN_enableDebugFree(base, initparams->emulationEnable);

	/* Unlock protected registers. */
	if (!MCAN_writeProtectedRegAccessUnlock(base))
		return false;

	/* Enable/Disable transmit pause. */
	MCAN_setTxPause(base, initparams->txpEnable);

	/* Enable/Disable Edge Filtering during Bus Integration. */
	MCAN_setEFBI(base, initparams->efbi);

	/* Enable/Disable Protocol Exception Handling. */
	MCAN_setProtocolExceptionHandling(base, initparams->pxhddisable);

	/* Enable/Disable Automatic Retransmission. */
	MCAN_setAutomaticRetransmission(base, initparams->darEnable);

	/* Set MCAN Watchdog RAM counter. */
	MCAN_setRAMWatchdogCounter(base, initparams->wdcPreload);

	/* If CANFD is enabled, configure the parameters for CANFD mode. */
	MCAN_FDModeInit(base, initparams->fdMode,
						initparams->fdFormat,
						initparams->brsEnable,
						initparams->tdcEnable,
						&initparams->tdcConfig);

	if (!MCAN_writeProtectedRegAccessLock(base))
		return false;

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_config(uint32_t base, MCAN_ConfigParams_t *configParams)
{
	/* Unlock protected registers. */
	if (!MCAN_writeProtectedRegAccessUnlock(base))
		return false;

	MCAN_setBusMonitoringMode(base, configParams->monEnable);

	/* The Restricted Operation Mode must not be combined
	with the Loop Back Mode (internalor external). */
	if (!MCAN_setRestrictedOperationMode(base, configParams->asmEnable)) {
		/* Lock protected registers. */
		MCAN_writeProtectedRegAccessLock(base);

		return false;
	}

	/* Configture global filter. */
	MCAN_setGlobalFilter(base, &configParams->filterConfig);

	/* Configture Timestamp counter. */
	MCAN_setTimestampCounter(base, configParams->tsClock,
							configParams->tsPrescalar, configParams->tsSelect);

	/* Configture Timeout counter. */
	MCAN_setTimeoutCounter(base, configParams->timeoutCntEnable,
						configParams->timeoutSelect, configParams->timeoutPreload);

	/* Lock protected registers. */
	if (!MCAN_writeProtectedRegAccessLock(base))
		return false;

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_setBitTime(uint32_t base, const MCAN_BitTimingParams_t *configBitrate)
{
	if (!MCAN_isBaseValid(base))
		return false;

	/* Set normal phase bitrate. */
	if (!MCAN_setArbitrationPhaseTiming(base, configBitrate->nomRatePrescalar,
								configBitrate->nomTimeSeg1, configBitrate->nomTimeSeg2,
								configBitrate->nomSynchJumpWidth))
		return false;
	
	/* Set data phase bitrate(only in CANFD mode). */
	if (HWREG(base + M_CAN_CCCR) & M_CAN_CCCR_FDOE_M)
		if (!MCAN_SetDataPhaseTiming(base, configBitrate->dataRatePrescalar,
								configBitrate->dataTimeSeg1, configBitrate->dataTimeSeg2,
								configBitrate->dataSynchJumpWidth))
			return false;

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_enableIntrLine(uint32_t base, MCAN_InterruptLine_t line, bool enable)
{
	if (!MCAN_isBaseValid(base))
		return false;

	if (enable) {
		if (line == MCAN_INTERRUPT_LINE_0)
			HWREG(base + M_CAN_ILE) |= M_CAN_ILE_EINT0_M;
		else if (line == MCAN_INTERRUPT_LINE_1)
			HWREG(base + M_CAN_ILE) |= M_CAN_ILE_EINT1_M;
		else
			return false;
	} else {
		if (line == MCAN_INTERRUPT_LINE_0)
			HWREG(base + M_CAN_ILE) &= ~M_CAN_ILE_EINT0_M;
		else if (line == MCAN_INTERRUPT_LINE_1)
			HWREG(base + M_CAN_ILE) &= ~M_CAN_ILE_EINT1_M;
		else
			return false;
	}

	return true;
}



GS32_DRIVER_MCAN_FUNC_T bool
MCAN_receiveMsgFromBuffer(uint32_t base, MCAN_RxMessage_t *rxbuf, uint8_t rxbuf_num)
{
	MCAN_RxNewDataStatus_t newdat;
	MCAN_RxNewDataStatus_t newdat_clear;
	uint32_t element_base_addr, element_data_size, msg_addr;

	if (!MCAN_isBaseValid(base))
		return false;

	if (rxbuf == NULL)
		return false;

	/* The maximum Rx buffer is 64. */
	if (rxbuf_num >= MCAN_RX_BUFFER_MAX)
		return false;

	/* Read Rx buffer status. */
	MCAN_getNewDataStatus(base, &newdat);

	if (rxbuf_num <= 31) {
		newdat_clear.statusLow = 1U << rxbuf_num;
		if (!(newdat.statusLow & (1U << rxbuf_num)))
			return false;
	} else {
		newdat_clear.statusHigh = 1U << rxbuf_num;
		if (!(newdat.statusHigh & (1U << rxbuf_num)))
			return false;
	}

	/* Get message start address in SRAM. */
	element_base_addr = HWREG(base + M_CAN_RXBC);
	/* Get message data size. */
	element_data_size = ((HWREG(base + M_CAN_RXESC) &
						M_CAN_RXESC_RBDS_M) >> M_CAN_RXESC_RBDS_S);

	element_data_size = MCAN_DataPhaseSizeArray[element_data_size];
	element_data_size += 8U;
	msg_addr = element_base_addr + (uint32_t)(element_data_size * rxbuf_num) + MCAN_MRAM_BASE(base);

	/* Copy message from SRAM to user. */
	for (int i = 0; i < element_data_size / 4; i++) {
	    ((uint32_t *)rxbuf)[i] = ((uint32_t *)msg_addr)[i];
	}

	/* Update Rx buffer so that it can continue to receive messages. */
	MCAN_clearNewDataStatus(base, &newdat_clear);

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_receiveMsgFromFifo0(uint32_t base, MCAN_RxMessage_t *rxbuf)
{
	uint32_t element_base_addr, element_data_size, msg_addr;
	uint32_t regval;
	uint32_t read_idx;

	if (!MCAN_isBaseValid(base))
		return false;

	if (rxbuf == NULL)
		return false;

	/* Check if the message has been storaged into Rx fifo 0. */
	regval = HWREG(base + M_CAN_RXF0S);
	if (!(regval & M_CAN_RXF0S_F0FL_M))
		return false;

	/* Get message index. */
	read_idx = (regval & M_CAN_RXF0S_F0GI_M) >> M_CAN_RXF0S_F0GI_S;
	element_base_addr = HWREG(base + M_CAN_RXF0C) & M_CAN_RXF0C_F0SA_M;
	element_data_size = HWREG(base + M_CAN_RXESC) & M_CAN_RXESC_F0DS_M;

	element_data_size = MCAN_DataPhaseSizeArray[element_data_size];
	element_data_size += 8U;
	msg_addr = element_base_addr + (element_data_size * read_idx) + MCAN_MRAM_BASE(base);

	/* Copy message from SRAM to user. */
	for (int i = 0; i < element_data_size / 4; i++) {
	    ((uint32_t *)rxbuf)[i] = ((uint32_t *)msg_addr)[i];
	}

	/* Update Rx fifo 0 so that it can continue to receive messages. */
	HWREG(base + M_CAN_RXF0A) = read_idx;

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_receiveMsgFromFifo1(uint32_t base, MCAN_RxMessage_t *rxbuf)
{
	uint32_t element_base_addr, element_data_size, msg_addr;
	uint32_t regval;
	uint32_t read_idx;

	if (!MCAN_isBaseValid(base))
		return false;

	if (rxbuf == NULL)
		return false;

	/* Check if the message has been storaged into Rx fifo 1. */
	regval = HWREG(base + M_CAN_RXF1S);
	if (!(regval & M_CAN_RXF1S_F1FL_M))
		return false;

	/* Get message index. */
	read_idx = (regval & M_CAN_RXF1S_F1GI_M) >> M_CAN_RXF1S_F1GI_S;
	element_base_addr = HWREG(base + M_CAN_RXF1C) & M_CAN_RXF1C_F1SA_M;
	element_data_size = (HWREG(base + M_CAN_RXESC) & M_CAN_RXESC_F1DS_M) >>
													M_CAN_RXESC_F1DS_S;

	element_data_size = MCAN_DataPhaseSizeArray[element_data_size];
	element_data_size += 8U;
	msg_addr = element_base_addr + (element_data_size * read_idx) + MCAN_MRAM_BASE(base);

	/* Copy message from SRAM to user. */
	for (int i = 0; i < element_data_size / 4; i++) {
	    ((uint32_t *)rxbuf)[i] = ((uint32_t *)msg_addr)[i];
	}

	/* Update Rx fifo 1 so that it can continue to receive messages. */
	HWREG(base + M_CAN_RXF1A) = read_idx;

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_readHighPriorityMsgRam(uint32_t base, MCAN_RxMessage_t *rxbuf)
{
	MCAN_HighPriorityMsgInfo_t hpm_stat;
	uint32_t element_base_addr, element_data_size, msg_addr;

	if (!MCAN_isBaseValid(base))
		return false;

	if (rxbuf == NULL)
		return false;

	/* Get high priority message status. */
	hpm_stat.hpm_status = (uint16_t)HWREG(base + M_CAN_HPMS);

	switch (hpm_stat.msi) {
		case MCAN_HIGH_PRIORITY_STORED_FIFO0 :
			element_base_addr = HWREG(base + M_CAN_RXF0C) & M_CAN_RXF0C_F0SA_M;
			element_data_size = HWREG(base + M_CAN_RXESC) & M_CAN_RXESC_F0DS_M;

			element_data_size = MCAN_DataPhaseSizeArray[element_data_size];
			element_data_size += 8U;
			msg_addr = element_base_addr + (element_data_size * hpm_stat.bidx) +
											MCAN_MRAM_BASE(base);
			break;

		case MCAN_HIGH_PRIORITY_STORED_FIFO1 :
			element_base_addr = HWREG(base + M_CAN_RXF1C) & M_CAN_RXF1C_F1SA_M;
			element_data_size = (HWREG(base + M_CAN_RXESC) & M_CAN_RXESC_F1DS_M) >>
															M_CAN_RXESC_F1DS_S;

			element_data_size = MCAN_DataPhaseSizeArray[element_data_size];
			element_data_size += 8U;
			msg_addr = element_base_addr + (element_data_size * hpm_stat.bidx) +
											MCAN_MRAM_BASE(base);
			break;

		default :
			return false;
	}

	/* Copy message from SRAM to user. */
	for (int i = 0; i < element_data_size / 4; i++) {
	    ((uint32_t *)rxbuf)[i] = ((uint32_t *)msg_addr)[i];
	}

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_readMsgRam(uint32_t base, MCAN_MemType_t memType,
											uint32_t bufNum, MCAN_RxFIFONum_t fifoNum,
											MCAN_RxMessage_t *rxbuf)
{
	uint32_t element_base_addr, element_data_size, msg_addr;
	uint32_t read_idx;

	if (!MCAN_isBaseValid(base))
		return false;

	if (rxbuf == NULL)
		return false;

	if (memType == MCAN_MEM_TYPE_BUF) {
		/* Get message start address in SRAM. */
		element_base_addr = HWREG(base + M_CAN_RXBC);
		/* Get message data size. */
		element_data_size = ((HWREG(base + M_CAN_RXESC) &
							M_CAN_RXESC_RBDS_M) >> M_CAN_RXESC_RBDS_S);

		element_data_size = MCAN_DataPhaseSizeArray[element_data_size];
		element_data_size += 8U;
		msg_addr = element_base_addr +
				   (uint32_t)(element_data_size * bufNum) + MCAN_MRAM_BASE(base);

	} else if (memType == MCAN_MEM_TYPE_FIFO) {
		if (fifoNum == MCAN_RX_FIFO_NUM_0) {
			/* Get message index. */
			read_idx = (HWREG(base + M_CAN_RXF0S) & M_CAN_RXF0S_F0GI_M) >>
													M_CAN_RXF0S_F0GI_S;
			element_base_addr = HWREG(base + M_CAN_RXF0C) & M_CAN_RXF0C_F0SA_M;
			element_data_size = HWREG(base + M_CAN_RXESC) & M_CAN_RXESC_F0DS_M;

			element_data_size = MCAN_DataPhaseSizeArray[element_data_size];
			element_data_size += 8U;
			msg_addr = element_base_addr +
					  (element_data_size * read_idx) + MCAN_MRAM_BASE(base);
		} else if (fifoNum == MCAN_RX_FIFO_NUM_1) {
			/* Get message index. */
			read_idx = (HWREG(base + M_CAN_RXF1S) & M_CAN_RXF1S_F1GI_M) >>
													M_CAN_RXF1S_F1GI_S;
			element_base_addr = HWREG(base + M_CAN_RXF1C) & M_CAN_RXF1C_F1SA_M;
			element_data_size = (HWREG(base + M_CAN_RXESC) & M_CAN_RXESC_F1DS_M) >>
															M_CAN_RXESC_F1DS_S;

			element_data_size = MCAN_DataPhaseSizeArray[element_data_size];
			element_data_size += 8U;
			msg_addr = element_base_addr +
					  (element_data_size * read_idx) + MCAN_MRAM_BASE(base);
		} else
			return false;
	}

	/* Copy message from SRAM to user. */
	for (int i = 0; i < element_data_size / 4; i++) {
	    ((uint32_t *)rxbuf)[i] = ((uint32_t *)msg_addr)[i];
	}

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_writeMsgRam(uint32_t base, MCAN_MemType_t memType,
					  	  	  	  	  	  	  uint32_t txbuf_num, const MCAN_TxMessage_t *txbuf)
{
	uint32_t element_base_addr, element_data_size, msg_addr;
	uint32_t number;
	uint32_t regval;

	if (!MCAN_isBaseValid(base))
		return false;

	if (txbuf == NULL)
		return false;

	if (memType == MCAN_MEM_TYPE_BUF) {
		/* Get the number of Tx buffer. */
		number = (HWREG(base + M_CAN_TXBC) & M_CAN_TXBC_NDTB_M) >> M_CAN_TXBC_NDTB_S;

		/**
		* If the Tx buffer is not set or the index of txbufnum exceeds the maximum
		* Tx buffer number, return false.
		*/
		if (txbuf_num >= number || number == 0)
			return false;

		/* Get Tx buffer data field size. */
		element_data_size = HWREG(base + M_CAN_TXESC) & M_CAN_TXESC_TBDS_M;
		/* Get Tx buffer base address. */
		element_base_addr = HWREG(base + M_CAN_TXBC) & M_CAN_TXBC_TBSA_M;

		element_data_size = MCAN_DataPhaseSizeArray[element_data_size];
		element_data_size += 8U;
		msg_addr = element_base_addr +
				   (element_data_size * txbuf_num) + MCAN_MRAM_BASE(base);
	} else if (memType == MCAN_MEM_TYPE_FIFO) {
		regval = HWREG(base + M_CAN_TXFQS);

		/* Current Tx fifo is full. */
		if (regval & M_CAN_TXFQS_TFQF_M)
			return false;

		number = (regval & M_CAN_TXFQS_TFQPI_M) >> M_CAN_TXFQS_TFQPI_S;

		/* Get Tx fifo/queue data field size. */
		element_data_size = HWREG(base + M_CAN_TXESC) & M_CAN_TXESC_TBDS_M;
		/* Get Tx fifo/queue base address. */
		element_base_addr = HWREG(base + M_CAN_TXBC) & M_CAN_TXBC_TBSA_M;

		element_data_size = MCAN_DataPhaseSizeArray[element_data_size];
		element_data_size += 8U;
		msg_addr = element_base_addr +
				   (element_data_size * number) + MCAN_MRAM_BASE(base);
	} else
		return false;

	for (int i = 0; i < element_data_size / 4; i++) {
	    ((uint32_t *)msg_addr)[i] = ((uint32_t *)txbuf)[i];
	}

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_getRxFIFOStatus(uint32_t base, MCAN_RxFIFOStatus_t *fifoStatus)
{
	if (!MCAN_isBaseValid(base))
		return false;

	if (fifoStatus == NULL)
		return false;

	if (fifoStatus->num == MCAN_RX_FIFO_NUM_0)
		fifoStatus->rxfifo_stat = HWREG(base + M_CAN_RXF0S);
	else if (fifoStatus->num == MCAN_RX_FIFO_NUM_1)
		fifoStatus->rxfifo_stat = HWREG(base + M_CAN_RXF1S);
	else
		return false;

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_transmitMsgBuffer(uint32_t base, MCAN_TxMessage_t *txbuf, uint8_t txbuf_num)
{
	uint32_t element_base_addr, element_data_size, msg_addr;
	uint32_t number;

	if (!MCAN_isBaseValid(base))
		return false;

	if (txbuf == NULL)
		return false;

	/* Check if Tx buffer is pending. */
	if (HWREG(base + M_CAN_TXBRP) & (1U << txbuf_num))
		return false;

	/* Get the number of Tx buffer. */
	number = (HWREG(base + M_CAN_TXBC) & M_CAN_TXBC_NDTB_M) >> M_CAN_TXBC_NDTB_S;

	/**
	 * If the Tx buffer is not set or the index of txbufnum exceeds the maximum
	 * Tx buffer number, return false.
	 */
	if (txbuf_num >= number || number == 0)
		return false;

	/* Get Tx buffer data field size. */
	element_data_size = HWREG(base + M_CAN_TXESC) & M_CAN_TXESC_TBDS_M;
	/* Get Tx buffer base address. */
	element_base_addr = HWREG(base + M_CAN_TXBC) & M_CAN_TXBC_TBSA_M;

	element_data_size = MCAN_DataPhaseSizeArray[element_data_size];
	element_data_size += 8U;
	msg_addr = element_base_addr + (element_data_size * txbuf_num) + MCAN_MRAM_BASE(base);

	for (int i = 0; i < element_data_size / 4; i++) {
	    ((uint32_t *)msg_addr)[i] = ((uint32_t *)txbuf)[i];
	}

	/* Request to start transmission of message. */
	HWREG(base + M_CAN_TXBAR) = (1U << txbuf_num);

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_transmitMsgFifoQueue(uint32_t base, MCAN_TxMessage_t *txbuf)
{
	uint32_t element_base_addr, element_data_size, msg_addr;
	uint32_t regval, write_idx;

	if (!MCAN_isBaseValid(base))
		return false;

	if (txbuf == NULL)
		return false;

	/* Get the write and read indexes of the current Tx fifo/queue. */
	regval = HWREG(base + M_CAN_TXFQS);

	/* Current Tx fifo/queue is full. */
	if (regval & M_CAN_TXFQS_TFQF_M)
		return false;

	/* Get Put and Get index. */
	write_idx = (regval & M_CAN_TXFQS_TFQPI_M) >> M_CAN_TXFQS_TFQPI_S;

	/* Get Tx fifo/queue data field size. */
	element_data_size = HWREG(base + M_CAN_TXESC) & M_CAN_TXESC_TBDS_M;
	/* Get Tx fifo/queue base address. */
	element_base_addr = HWREG(base + M_CAN_TXBC) & M_CAN_TXBC_TBSA_M;

	element_data_size = MCAN_DataPhaseSizeArray[element_data_size];
	element_data_size += 8U;
	msg_addr = element_base_addr + (element_data_size * write_idx) + MCAN_MRAM_BASE(base);

	for (int i = 0; i < element_data_size / 4; i++) {
	    ((uint32_t *)msg_addr)[i] = ((uint32_t *)txbuf)[i];
	}

	/* Request to start transmission of message. */
	HWREG(base + M_CAN_TXBAR) = (1U << write_idx);

	return true;
}

GS32_DRIVER_MCAN_FUNC_T uint32_t
MCAN_getTxBufTransmissionFlagStatus(uint32_t base)
{
	uint32_t can_index;

	if (!MCAN_isBaseValid(base))
		return 0;

	for (uint32_t i = 0; i < M_CAN_NUMBER_OF_INSTANCES; i++) {
		if (MCAN_BaseAddressArray[i] == base) {
			can_index = i;
			break;
		}
	}

	return MCAN_TX_COMPLETED_STATUS_REG[can_index];
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_setTxBufTransmissionFlag(uint32_t base, uint8_t txbuf_num)
{
	uint32_t number;
	uint32_t can_index;

	if (!MCAN_isBaseValid(base))
		return false;

	/* Get the number of Tx buffer. */
	number = (HWREG(base + M_CAN_TXBC) & M_CAN_TXBC_NDTB_M) >> M_CAN_TXBC_NDTB_S;

	/**
	* If the Tx buffer is not set or the index of txbufnum exceeds the maximum
	* Tx buffer number, return false.
	*/
	if (txbuf_num >= number || number == 0)
		return false;

	for (uint32_t i = 0; i < M_CAN_NUMBER_OF_INSTANCES; i++) {
		if (MCAN_BaseAddressArray[i] == base) {
			can_index = i;
			break;
		}
	}

	MCAN_TX_COMPLETED_STATUS_REG[can_index] |= (1U << txbuf_num);

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_clearTxBufTransmissionFlag(uint32_t base, uint8_t txbuf_num)
{
	uint32_t number;
	uint32_t can_index;

	if (!MCAN_isBaseValid(base))
		return false;

	/* Get the number of Tx buffer. */
	number = (HWREG(base + M_CAN_TXBC) & M_CAN_TXBC_NDTB_M) >> M_CAN_TXBC_NDTB_S;

	/**
	* If the Tx buffer is not set or the index of txbufnum exceeds the maximum
	* Tx buffer number, return false.
	*/
	if (txbuf_num >= number || number == 0)
		return false;

	for (uint32_t i = 0; i < M_CAN_NUMBER_OF_INSTANCES; i++) {
		if (MCAN_BaseAddressArray[i] == base) {
			can_index = i;
			break;
		}
	}

	MCAN_TX_COMPLETED_STATUS_REG[can_index] &= ~(1U << txbuf_num);

	return true;
}

GS32_DRIVER_MCAN_FUNC_T bool
MCAN_readTxEventFIFO(uint32_t base, MCAN_TxEventFifo_t *eventbuf)
{
	uint32_t element_base_addr, fifo_addr;
	uint32_t read_idx , regval;

	if (!MCAN_isBaseValid(base))
		return false;

	if (eventbuf == NULL)
		return false;

	/* Get event fifo element information. */
	regval = HWREG(base + M_CAN_TXEFS);
	/* Check if Tx event fifo is empty. */
	if (!(regval & M_CAN_TXEFS_EFFL_M))
		return false;
	/* Get current read event fifo index. */
	read_idx = (regval & M_CAN_TXEFS_EFGI_M) >> M_CAN_TXEFS_EFGI_S;

	/* Obtain the event fifo base address. */
	regval = HWREG(base + M_CAN_TXEFC);
	element_base_addr = regval & M_CAN_TXEFC_EFSA_M;
	fifo_addr = element_base_addr +
				(read_idx * M_CAN_TX_EVENT_FIFO_SIZE) + MCAN_MRAM_BASE(base);

	for (int i = 0; i < M_CAN_TX_EVENT_FIFO_SIZE / 4; i++) {
	    ((uint32_t *)eventbuf)[i] = ((uint32_t *)fifo_addr)[i];
	}

	/* Event FIFO Acknowledge Index. */
	HWREG(base + M_CAN_TXEFA) = read_idx;

	return true;
}
