/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


#ifdef __cplusplus
extern "C"{
#endif


#include <stdbool.h>
#include <stdint.h>

#include "gs32_version.h"
#include "pmbus.h"


GS32_DRIVER_PMBUS_FUNC_T void PMBUS_setSpeedMode(uint32_t base,PMBUS_speedMode_t mode)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val,max_speed;

	max_speed = PMBUS_readRegister(base + PMBUS_O_IC_COMP_PARAM_1);

	max_speed = ((max_speed&PMBUS_COMP_PARAM_1_MAX_SPEED_MODE_M)>>PMBUS_COMP_PARAM_1_MAX_SPEED_MODE_S);

	if(mode > max_speed)
		mode = max_speed;

	reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);

	reg_val &= ~(PMBUS_CON_SPEED_M);

	reg_val |= (mode << PMBUS_CON_SPEED_S);

	PMBUS_writeRegister(base + PMBUS_O_IC_CON, reg_val);
}

GS32_DRIVER_PMBUS_FUNC_T void PMBUS_setAddrMode(uint32_t base,PMBUS_addrMode_t mode)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val;

	if(mode & PMBUS_SLAVE_7BIT){
		reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);
		reg_val &= ~PMBUS_CON_IC_10BITADDR_SLAVE;
		PMBUS_writeRegister(base + PMBUS_O_IC_CON, reg_val);
	}
	if(mode & PMBUS_SLAVE_10BIT){
		reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);
		reg_val |= PMBUS_CON_IC_10BITADDR_SLAVE;
		PMBUS_writeRegister(base + PMBUS_O_IC_CON, reg_val);
	}
	if(mode & PMBUS_MASTER_7BIT){
		reg_val = PMBUS_readRegister(base + PMBUS_O_IC_TAR);
		reg_val &= ~PMBUS_TAR_IC_10BITADDR_MASTER;
		PMBUS_writeRegister(base + PMBUS_O_IC_TAR, reg_val);
	}
	if(mode & PMBUS_MASTER_10BIT){
		reg_val = PMBUS_readRegister(base + PMBUS_O_IC_TAR);
		reg_val |= PMBUS_TAR_IC_10BITADDR_MASTER;
		PMBUS_writeRegister(base + PMBUS_O_IC_TAR, reg_val);
	}

}

GS32_DRIVER_PMBUS_FUNC_T void PMBUS_setDeviceIDMode(uint32_t base,PMBUS_deviceIDMode_t mode)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_TAR);

	reg_val &= ~(PMBUS_TAR_GC_OR_START|PMBUS_TAR_SPECIAL|PMBUS_TAR_DEVICE_ID|\
				PMBUS_TAR_SMBUS_QUICK_CMD);

	switch(mode){
		case PMBUS_SAR:
			reg_val |= PMBUS_TAR_DEVICE_ID;
			break;
		case PMBUS_GENERAIL_CALL:
			reg_val |= PMBUS_TAR_SPECIAL;
			break;
		case PMBUS_START_BYTE:
			reg_val |= (PMBUS_TAR_SPECIAL|PMBUS_TAR_GC_OR_START);
			break;
		case PMBUS_QUICK_CMD:
			reg_val |= (PMBUS_TAR_SPECIAL|PMBUS_TAR_SMBUS_QUICK_CMD);
			break;
	}

	PMBUS_writeRegister(base + PMBUS_O_IC_TAR, reg_val);
}

GS32_DRIVER_PMBUS_FUNC_T void PMBUS_setBaudRate(uint32_t base, uint32_t sysclkHz, uint32_t bitRate,
						PMBUS_DutyCycle_t dutyCycle)
{

	uint32_t lcnt,hcnt,speed_mode,spklen,divider;

	spklen = PMBUS_getFSSpkLen(base);

	speed_mode = PMBUS_getSpeedMode(base);

	divider = sysclkHz/bitRate;

	if(dutyCycle == PMBUS_DUTYCYCLE_50){
		hcnt = divider/2;
	}else if(dutyCycle == PMBUS_DUTYCYCLE_33){
		hcnt = divider/3;
	}

	lcnt = divider - hcnt;

	if(hcnt < spklen+7)
		hcnt = spklen+7;

	if(lcnt < spklen+5)
		lcnt =  spklen+5;

	if(speed_mode == PMBUS_STANDARD_MODE){
		PMBUS_writeRegister(base+PMBUS_O_IC_SS_SCL_HCNT,hcnt);
		PMBUS_writeRegister(base+PMBUS_O_IC_SS_SCL_LCNT,lcnt);
	}else if(speed_mode == PMBUS_FAST_MODE){
		PMBUS_writeRegister(base+PMBUS_O_IC_FS_SCL_HCNT,hcnt);
		PMBUS_writeRegister(base+PMBUS_O_IC_FS_SCL_LCNT,lcnt);
	}
}

GS32_DRIVER_PMBUS_FUNC_T void PMBUS_clearInterruptStatus(uint32_t base, PMBUS_InterruptRawStatus_t intFlags)
{
	ASSERT(PMBUS_isBaseValid(base));

	if(intFlags & PMBUS_RX_UNDER)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_RX_UNDER);
	if(intFlags & PMBUS_RX_OVER)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_RX_OVER);
	if(intFlags & PMBUS_TX_OVER)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_TX_OVER);
	if(intFlags & PMBUS_RD_REQ)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_RD_REQ);
	if(intFlags & PMBUS_TX_ABRT)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_TX_ABRT);
	if(intFlags & PMBUS_RX_DONE)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_RX_DONE);
	if(intFlags & PMBUS_INTR_ACTIVITY)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_ACTIVITY);
	if(intFlags & PMBUS_STOP_DET)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_STOP_DET);
	if(intFlags & PMBUS_START_DET)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_START_DET);
	if(intFlags & PMBUS_GEN_CALL)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_GEN_CALL);
	if(intFlags & PMBUS_RESTART_DET)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_RESTART_DET);
	if(intFlags & PMBUS_SCL_STUCK_AT_LOW)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_SCL_STUCK_DET);
	if(intFlags & PMBUS_WR_REQ)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_WR_REQ);
	if(intFlags & PMBUS_SLV_ADDR1_TAG)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_SLV_ADDR_TAG);
	if(intFlags & PMBUS_SLV_ADDR2_TAG)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_SLV_ADDR_TAG);
	if(intFlags & PMBUS_SLV_ADDR3_TAG)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_SLV_ADDR_TAG);
	if(intFlags & PMBUS_SLV_ADDR4_TAG)
		PMBUS_readRegister(base+PMBUS_O_IC_CLR_SLV_ADDR_TAG);

}

#ifdef __cplusplus
}
#endif
