#include "qspi.h"

#ifdef QSPIA_BASE


void QSPI_initConfig(uint32_t qspi_base, QSPI_INIT_PARAM *init_param)
{
    QSPI_CONTROL_PARAM control_param = {0};
    QSPI_SPI_CONTROL_PARAM spi_control_param = {0};

    // disable all interrupts
    QSPI_disableAllInterrupt(qspi_base);

    // clear all interrupt status
    QSPI_clearAllInterruptStatus(qspi_base);

    // set baud rate
    QSPI_setBaudRate(qspi_base, init_param->ahbclk_freq, init_param->baudrate);

    // QSPI control register configuration
    control_param.RegistValue.SSTE    = QSPI_SELECT_TOGGLE_DISABLE; // keep CS low during data transmission
    control_param.RegistValue.SPI_FRF = init_param->frame_format;   // select SPI frame format
    control_param.RegistValue.DFS_32  = init_param->data_width;     // select data width
    control_param.RegistValue.TMOD    = init_param->transfer_mode;  // select transfer mode (TX/RX)
    control_param.RegistValue.SCPOL   = init_param->scpol;          // clock polarity
    control_param.RegistValue.SCPH    = init_param->scph;           // clock phase
    control_param.RegistValue.FRF     = QSPI_MOTOROLA_SPI;          // select standard Motorola SPI
    QSPI_ControlRegisterConfig(qspi_base, control_param.param);

    // QSPI extended register configuration
    spi_control_param.RegistValue.CLK_STRETCH_EN  = init_param->clk_stretch_en;
    spi_control_param.RegistValue.WAIT_CYCLES     = init_param->wait_cycles;
    spi_control_param.RegistValue.INST_L          = init_param->inst_l;
    spi_control_param.RegistValue.ADDR_L          = init_param->addr_l;
    spi_control_param.RegistValue.TRANS_TYPE      = init_param->trans_type;
    spi_control_param.RegistValue.XIP_PREFETCH_EN = init_param->xip_prefetch_en;
    QSPI_spiControlRegisterConfig(qspi_base, spi_control_param.param);
}


#if 0
/* QSPI initialization */
uint32_t QSPI_initConfig(uint32_t base)
{
    /* Many SPI configurations registers can be written when SSIENR[SSIC_EN] is 0 */
    QSPI_disableModule(base);

    /* ****************************************************************************
     * Configure SPI master mode,
     * QSPI_FRF=Quad, CPOL=0, CPHA=0, TMOD=TxRx, FRF=SPI, DFS=32bit */
    QSPIREG(base, QSPI_CTRLR0) = ((QSPI_CTRLR0_SSI_IS_MST_MASK) + \
            (QSPI_CTRLR0_SPI_FRF(QSPI_CTRLR0_SPI_FRF_STD)) + \
            (QSPI_CTRLR0_TMOD(QSPI_CTRLR0_TMOD_TXRX)) + \
            (QSPI_CTRLR0_DFS(8U-1U)));

    /* Choose CS_0 */
    QSPI_csEnable(base);
    /* 20MHz/10 = 2MHz baudrate */
    QSPIREG(base, QSPI_BAUDR) = QSPI_BAUDR_SCKDV(10);

    /* Tx FIFO watermark for start Tx and for generate interrupt
     * Write 8 data into the TxFIFO and the transfer starts;
     * When half of the TxFIFO entries are empty, issue TxFIFO empty IRQ. */
    QSPI_setFIFOTransferStartLevel(base, 0);
    /* Rx FIFO watermark for generating Rx interrupt, if there is one entry in RxFIFO, issue Rx IRQ */
    QSPI_setFIFOInterruptLevel(base, QSPI_TX_FIFO_DEPTH/2, 0);

    /* Enable or disable all interrupts */
    QSPI_disableAllInterrupt(base);

    /* Read ICR to clear all interrupt flags*/
    QSPI_clearAllInterruptStatus(base);

    /* Configure QSPI_CTRL0 register */
    QSPIREG(base, QSPI_SPI_CTRLR0) = (QSPI_SPI_CTRLR0_CLK_STRETCH_EN_MASK) + \
            (QSPI_SPI_CTRLR0_XIP_PREFETCH_EN_MASK) + \
            (QSPI_SPI_CTRLR0_XIP_INST_EN_MASK) + \
            (QSPI_SPI_CTRLR0_XIP_DFS_HC_MASK) + \
            (QSPI_SPI_CTRLR0_WAIT_CYCLES(QSPI_DUMMY_CYCLES_LEN)) + \
            (QSPI_SPI_CTRLR0_INST_L(QSPI_SPI_CTRLR0_INST_L_8BIT)) + \
            (QSPI_SPI_CTRLR0_ADDR_L(QSPI_SPI_CTRLR0_ADDR_L_24bit)) + \
            (QSPI_SPI_CTRLR0_TRANS_TYPE(QSPI_SPI_CTRLR0_TRANS_TYPE_TT1));

    /* Enable QSPI module */
    QSPI_enableModule(base);

    return 0;
}
#endif

void QSPI_setConfig(uint32_t base, uint32_t lspclkHz, QSPI_CtrlParam *pCtrl)
{
    QSPI_disableModule(base);
    /* 20MHz/10 = 2MHz baudrate */
    QSPIREG(base, QSPI_BAUDR) = QSPI_BAUDR_SCKDV(lspclkHz / pCtrl->buad / 2);
    /* Configure CTRL0 register */
    QSPIREG(base, QSPI_CTRLR0) = pCtrl->baseCtrl;
    /* Configure SPI_CTRL0 register */
    QSPIREG(base, QSPI_SPI_CTRLR0) = pCtrl->qspiCtrl;
    /* Enable QSPI module */
    QSPI_enableModule(base);

    /* Choose CS_0 */
    QSPI_csEnable(base);
}

/*****************************************************************************
 * Send data in pTxBuf and store the received data in pRxBuf in bytes.
 * Limitations: nSize must be less than 32
 *              send and receive in SPI standard format - D0=SOUT, D1=SIN
 ******************************************************************************/
uint32_t QSPI_syncSendRecv(uint32_t base, uint8_t *pTxBuf, uint8_t *pRxBuf, uint32_t nSize)
{
    uint32_t i;
    uint32_t u32Tmp;

    /* Need configure CTRLR0 register as: QSPI_FRF=STD, TMOD=TXRX, DFS=8bit for SPI standard transfer */
    /* disable QSPI to allow cfg change */
    QSPI_disableModule(base);
    /* Configure Tx Rx mode, both Tx and Rx FIFO is working. */
    u32Tmp = QSPIREG(base, QSPI_CTRLR0);
    u32Tmp &= (~QSPI_CTRLR0_TMOD_MASK);
    u32Tmp |= QSPI_CTRLR0_TMOD(QSPI_CTRLR0_TMOD_TXRX);
    u32Tmp &= (~QSPI_CTRLR0_DFS_MASK);
    u32Tmp |= QSPI_CTRLR0_DFS(7);    //configure 8-bit frame size
    u32Tmp &= (~QSPI_CTRLR0_SPI_FRF_MASK);
    u32Tmp |= QSPI_CTRLR0_SPI_FRF(QSPI_CTRLR0_SPI_FRF_STD);    //configure 8-bit frame size
    QSPIREG(base, QSPI_CTRLR0) = u32Tmp;
    /* Set Tx Start threshold to nSize or max value 8 */
    QSPI_setFIFOTransferStartLevel(base, nSize - 1);
    /* enable QSPI */
    QSPI_enableModule(base);

    /* Write data into DRx register - TxFIFO */
    for(i=0; i<nSize; i++)
    {
        QSPIREG(base, QSPI_DR0) = pTxBuf[i];
    }

    /* wait till there is data available in RxFIFO */
    while((QSPIREG(base, QSPI_SR) & QSPI_SR_RFNE_MASK) == 0);

    /* wait till there are nSize bytes available in the Rx FIFO */
    while(QSPI_getRxFIFOStatus(base) < nSize);

    /* Read specified number of bytes from RxFIFO */
    for(i=0; i< nSize; i++)
    {
        pRxBuf[i] = QSPIREG(base, QSPI_DR0);
    }

    /* Read ICR to clear IRQ flags */
    QSPI_clearAllInterruptStatus(base);

    /* Restore CTRLR0 settings for Quad mode transmission- optional */

    return 0;
}


/* The below two functions are not used in the QSPI Flash driver */
/* Send data in pTxBuf and received data is don't care */
uint32_t QSPI_syncSend(uint32_t base, uint8_t *pTxBuf, uint32_t nSize)
{
    uint32_t i;
    uint32_t u32Tmp;

    /* disable QSPI to allow cfg change */
    QSPI_disableModule(base);

    /* Configure Tx only mode, so RxFIFO is not receiving data, only Tx FIFO is working. */
    u32Tmp = QSPIREG(base, QSPI_CTRLR0);
    u32Tmp &= (~QSPI_CTRLR0_TMOD_MASK);
    u32Tmp |= QSPI_CTRLR0_TMOD(QSPI_CTRLR0_TMOD_TXONLY);
    u32Tmp &= (~QSPI_CTRLR0_DFS_MASK);
    u32Tmp |= QSPI_CTRLR0_DFS(7);
    QSPIREG(base, QSPI_CTRLR0) = u32Tmp;

    /* Set Tx Start threshold to nSize or max value 8 */
    QSPI_setFIFOTransferStartLevel(base, nSize - 1);
    /* enable QSPI */
    QSPI_enableModule(base);

    /* Write data into DRx register - TxFIFO */
    /* Because we configure the TXFTHR register, the Tx will not start until enough frames are written into the Tx FIFO */
    for(i=0; i<nSize; i++)
    {
        QSPIREG(base, QSPI_DR0) = pTxBuf[i];
    }

    /* Write Dummy words if necessary */

    /* wait till TxFIFO is empty */
    while((QSPIREG(base, QSPI_SR) & QSPI_SR_TFE_MASK) == 0);

    return 0;
}

/* Receive data into pRxBuf and Tx data is don't care */
uint32_t QSPI_syncRecv(uint32_t base, uint8_t *pRxBuf, uint32_t nSize)
{
    uint32_t i;
    uint32_t u32Tmp;

    /* disable QSPI to allow cfg change */
    QSPI_disableModule(base);

    /* Configure Rx only mode, so TxFIFO is not sending data, only Rx FIFO is working. */
    u32Tmp = QSPIREG(base, QSPI_CTRLR0);
    u32Tmp &= (~QSPI_CTRLR0_TMOD_MASK);
    u32Tmp |= QSPI_CTRLR0_TMOD(QSPI_CTRLR0_TMOD_RXONLY);
    u32Tmp &= (~QSPI_CTRLR0_DFS_MASK);
    u32Tmp |= QSPI_CTRLR0_DFS(7);
    QSPIREG(base, QSPI_CTRLR0) = u32Tmp;

    /* Set Tx Start threshold to nSize or max value 8 */
    QSPI_setFIFOTransferStartLevel(base, nSize - 1);
    /* enable QSPI */
    QSPI_enableModule(base);

    /* write data and receive data one-byte-by-one-byte. */
    u32Tmp = 0xFFU;
    for(i=0; i<nSize; i++)
    {
        /* Write data into DRx register - TxFIFO */
        QSPIREG(base, QSPI_DR0) = u32Tmp;
        /* wait till there is data available in RxFIFO */
        while((QSPIREG(base, QSPI_SR) & QSPI_SR_RFNE_MASK) == 0);
        /* read the received data from RxFIFO */
        pRxBuf[i] = QSPIREG(base, QSPI_DR0);
    }

    /* Write Dummy words */

    /* wait till TxFIFO is empty */
    while((QSPIREG(base, QSPI_SR) & QSPI_SR_TFE_MASK) == 0);

    return 0;
}

/* Send command phase without reception.
 * Must make sure nSize is less than Max FIFO Depth. */
uint32_t QSPI_sendFlsCmd(uint32_t base, uint8_t *pTxBuf, uint32_t nSize)
{
    uint32_t i;
    uint32_t u32Tmp;

    /* disable QSPI to allow cfg change */
    QSPI_disableModule(base);

    /* Configure Tx Rx mode, both Tx and Rx FIFO is working. */
    u32Tmp = QSPIREG(base, QSPI_CTRLR0);
    u32Tmp &= (~QSPI_CTRLR0_TMOD_MASK);
    u32Tmp |= QSPI_CTRLR0_TMOD(QSPI_CTRLR0_TMOD_TXRX);
    u32Tmp &= (~QSPI_CTRLR0_DFS_MASK);
    u32Tmp |= QSPI_CTRLR0_DFS(7);
    QSPIREG(base, QSPI_CTRLR0) = u32Tmp;
    /* Set Tx Start threshold to nSize or max value 8 */
    QSPI_setFIFOTransferStartLevel(base, nSize - 1);
    /* enable QSPI */
    QSPI_enableModule(base);

    /* Write data into DRx register - TxFIFO */
    for(i=0; i<nSize; i++)
    {
        QSPIREG(base, QSPI_DR0) = pTxBuf[i];
    }
    return 0;
}


/******************************************************************************
 * Send Flash program command and address,
 * Send program data in 32bit words - max 64 words (because one page is 256 bytes)
 * The number of data frames is configured in CTRLR1.NDF register.
 * There is not Rx data.
 ******************************************************************************/
//uint32_t QSPI_SendWords(uint32_t u32CmdAddr, uint32_t *pU32Buf, uint32_t nSize){
uint32_t QSPI_programFlash(uint32_t base, uint32_t nCmd, uint32_t nAddr, uint32_t *pU32Buf, uint32_t nSize)
{
    uint32_t i;
    uint32_t u32Tmp;

    /* Re-configure CTRLR0 to set Tx-only mode, and Frame Size 32bit */
    /* disable QSPI to allow cfg change */
    QSPI_disableModule(base);

    /* Configure Tx Rx mode, both Tx and Rx FIFO is working. */
    u32Tmp = QSPIREG(base, QSPI_CTRLR0);
    u32Tmp &= (~QSPI_CTRLR0_TMOD_MASK);
    u32Tmp |= QSPI_CTRLR0_TMOD(QSPI_CTRLR0_TMOD_TXONLY);
    u32Tmp &= (~QSPI_CTRLR0_DFS_MASK);
    u32Tmp |= QSPI_CTRLR0_DFS(31);
    u32Tmp &= (~QSPI_CTRLR0_SPI_FRF_MASK);
    u32Tmp |= QSPI_CTRLR0_SPI_FRF(QSPI_CTRLR0_SPI_FRF_QUAD);
    QSPIREG(base, QSPI_CTRLR0) = u32Tmp;

    /* total Tx number of data frames will be transferred; */
    QSPIREG(base, QSPI_CTRLR1) = nSize-1;
    /* Set Tx Start threshold to 0 because it's Tx-only mode, Tx FIFO empty doesn't stop the transfer */
    QSPI_setFIFOTransferStartLevel(base, 0);

    /* Set wait cycles = 0 for Flash Program process */
    u32Tmp = QSPIREG(base, QSPI_SPI_CTRLR0);
    u32Tmp &= (~QSPI_SPI_CTRLR0_WAIT_CYCLES_MASK);
    QSPIREG(base, QSPI_SPI_CTRLR0) = u32Tmp;

    /* enable QSPI */
    QSPI_enableModule(base);

    /* wait till Tx FIFO is not full */
    //while((QSPIREG(base, QSPI_SR) & QSPI_SR_TFNF_MASK) == 0);
    /* write command into the TxFIFO */
    QSPIREG(base, QSPI_DR0) = nCmd;
    /* write address into the TxFIFO */
    QSPIREG(base, QSPI_DR0) = nAddr;

    /* Write data into DRx register - TxFIFO */
    for(i=0; i<nSize; i++)
    {
        /* if TxFIFO is not Full, continue writing Tx FIFO */
        while(!(QSPIREG(base, QSPI_SR) & QSPI_SR_TFNF_MASK));
        QSPIREG(base, QSPI_DR0) = pU32Buf[i];
        /* Because QSPI_FRF=Quad mode, RxFIFO is not working in this case */
        if((QSPIREG(base, QSPI_SR) & QSPI_SR_RFNE_MASK))
            QSPIREG(base, QSPI_DR0);
    }

    /* Wait till Tx FIFO is empty which means Tx is completed */
    while(!(QSPIREG(base, QSPI_SR) & QSPI_SR_TFE_MASK));
    /*while((QSPIREG(base, QSPI_SR) & QSPI_SR_RFNE_MASK)){
        QSPIREG(base, QSPI_DR0);
    }*/
    while((QSPIREG(base, QSPI_SR) & QSPI_SR_BUSY_MASK));

    return 0;
}

/* send 32bit word in Rx mode only - Tx data is don't care. */
uint32_t QSPI_recvWords(uint32_t base, uint32_t u32CmdAddr, uint32_t *pU32Buf, uint32_t nSize)
{
    uint32_t i = 0;
    uint32_t u32Tmp = 0;

    /* wait if Tx FIFO is full */
    while((QSPIREG(base, QSPI_SR) & QSPI_SR_TFNF_MASK) == 0);
    /* Because it's Rx-only mode, this write will trigger repeated Tx with the same data until CTRLR1.NDF completes. */
    QSPIREG(base, QSPI_DR0) = u32CmdAddr;

    /* Read data from DRx register - RxFIFO */
    for(i=0; i < nSize; i++)
    {
        /* wait if Tx FIFO is full */
        while((QSPIREG(base, QSPI_SR) & QSPI_SR_TFNF_MASK) == 0);
        QSPIREG(base, QSPI_DR0) = 0xffffffff;

        /* Wait till there is data in the Rx FIFO */
        while((QSPIREG(base, QSPI_SR) & QSPI_SR_RFNE_MASK) == 0);
        if(i != 0)
        {
            pU32Buf[i-1] = QSPIREG(base, QSPI_DR0);
        }
        else
        {
            QSPIREG(base, QSPI_DR0);
        }
    }

    /* Wait till there is data in the Rx FIFO */
    while((QSPIREG(base, QSPI_SR) & QSPI_SR_RFNE_MASK) == 0);
    pU32Buf[i-1] = QSPIREG(base, QSPI_DR0);

    /* Wait till Tx FIFO is empty which means Tx is completed */
    //while((QSPIREG(base, QSPI_SR) & QSPI_SR_TFE_MASK) == 0);

    return 0;
}

uint32_t QSPI_sendWords(uint32_t base, uint32_t u32CmdAddr, uint32_t *pU32Buf, uint32_t nSize)
{
    uint32_t i = 0;
    uint32_t u32Tmp = 0;

    /* wait if Tx FIFO is full */
    while((QSPIREG(base, QSPI_SR) & QSPI_SR_TFNF_MASK) == 0);
    /* Because it's Rx-only mode, this write will trigger repeated Tx with the same data until CTRLR1.NDF completes. */
    QSPIREG(base, QSPI_DR0) = u32CmdAddr;

    /* Read data from DRx register - RxFIFO */
    for(i=0; i < nSize; i++)
    {
        /* wait if Tx FIFO is full */
        while((QSPIREG(base, QSPI_SR) & QSPI_SR_TFNF_MASK) == 0);
        QSPIREG(base, QSPI_DR0) = pU32Buf[i];

        /* Wait till there is data in the Rx FIFO */
        while((QSPIREG(base, QSPI_SR) & QSPI_SR_RFNE_MASK) == 0);
        QSPIREG(base, QSPI_DR0);
    }

    /* Wait till there is data in the Rx FIFO */
    while((QSPIREG(base, QSPI_SR) & QSPI_SR_RFNE_MASK) == 0);
    QSPIREG(base, QSPI_DR0);

    /* Wait till Tx FIFO is empty which means Tx is completed */
    //while((QSPIREG(base, QSPI_SR) & QSPI_SR_TFE_MASK) == 0);

    return 0;
}

/*********************************************************************************
 * For how to start the transfer process, refer to QSPI data book chapter "2.16.1"
 *     An instruction takes one FIFO location and address can take more than one FIFO locations. Both the
 *     instruction and address must be programmed in the data register (DR). DWC_ssi waits until both have been
 *     programmed to start the write operation.
 * This function sends command (instruction) and address in std SPI mode,
 * and reads back Data in Quad SPI mode.
 *********************************************************************************/
uint32_t QSPI_quadFastReadWords(uint32_t base, uint32_t nCmd, uint32_t nAddr, uint32_t *pRxBuf, uint32_t nSize)
{
    uint32_t u32Tmp;
    uint32_t ret;

    /* It's assumed that before calling this function, the SPI Quad mode is properly configured already.
     * Including CTRLR0(QSPI_FRF, DFS) QSPI_CTRLR0(INST_L, ADDR_L, WAIT_CYCLES, TRANS_TYPE. CLK_STRETCH_EN, ......)  */

    /* Set the desired number of data frames */
    QSPI_disableModule(base);;
    /* Set number of data frames to read */
    QSPIREG(base, QSPI_CTRLR1) = nSize - 1;
    /* Set wait cycles = 0 for Flash Program process */
    u32Tmp = QSPIREG(base, QSPI_SPI_CTRLR0);
    u32Tmp &= (~QSPI_SPI_CTRLR0_WAIT_CYCLES_MASK);
    u32Tmp |= QSPI_SPI_CTRLR0_WAIT_CYCLES(QSPI_DUMMY_CYCLES_LEN);
    QSPIREG(base, QSPI_SPI_CTRLR0) = u32Tmp;
    QSPI_enableModule(base);

    /* Write instruction into the Tx FIFO - only LSB 8bit is effective ? */
    QSPIREG(base, QSPI_DR0) = nCmd;
    /* Write address into the Tx FIFO - only LSB 24bit is effective */
    QSPIREG(base, QSPI_DR0) = nAddr;
    /* Dummy cycles are generated automatically */

    /* SCLK and CS signal are generated automatically for receiving read data frames */
    /* wait till all desired data frames are received in RxFIFO */
    for(uint32_t i = 0; i < nSize; i++)
    {
        /* Wait till there is at least one data in the RxFIFO */
        while(QSPI_getRxFIFOStatus(base) == 0);
        pRxBuf[i] = QSPIREG(base, QSPI_DR0);
    }
    return 0;
}

uint32_t QSPI_quadFastReadWordsInit(uint32_t base, uint32_t nCmd, uint32_t nAddr, uint32_t nSize)
{
    uint32_t u32Tmp;
    uint32_t ret;

    /* It's assumed that before calling this function, the SPI Quad mode is properly configured already.
     * Including CTRLR0(QSPI_FRF, DFS) QSPI_CTRLR0(INST_L, ADDR_L, WAIT_CYCLES, TRANS_TYPE. CLK_STRETCH_EN, ......)  */

    /* Set the desired number of data frames */
    QSPI_disableModule(base);;
    /* Set number of data frames to read */
    QSPIREG(base, QSPI_CTRLR1) = nSize - 1;
    /* Set wait cycles = 0 for Flash Program process */
    u32Tmp = QSPIREG(base, QSPI_SPI_CTRLR0);
    u32Tmp &= (~QSPI_SPI_CTRLR0_WAIT_CYCLES_MASK);
    u32Tmp |= QSPI_SPI_CTRLR0_WAIT_CYCLES(QSPI_DUMMY_CYCLES_LEN);
    QSPIREG(base, QSPI_SPI_CTRLR0) = u32Tmp;
    QSPI_enableModule(base);

    /* Write instruction into the Tx FIFO - only LSB 8bit is effective ? */
    QSPIREG(base, QSPI_DR0) = nCmd;
    /* Write address into the Tx FIFO - only LSB 24bit is effective */
    QSPIREG(base, QSPI_DR0) = nAddr;
    /* Dummy cycles are generated automatically */

    return 0;
}

/* Send data immediately after cmd is sent - before TxFIFO becomes empty. */
uint32_t QSPI_sendFlsData(uint32_t base, uint8_t *pTxData, uint32_t nSize)
{
    uint32_t i;
    uint32_t u32Tmp;

    /* If TxFIFO is not Full or specified data length is not reached, write data into Tx FIFO */

    /* Write data into DRx register - TxFIFO */
    for(i=0; i<nSize; i++)
    {
        while((QSPIREG(base, QSPI_SR) & QSPI_SR_TFNF_MASK) == 0);
        QSPIREG(base, QSPI_DR0) = pTxData[i];
    }

    /* wait till Tx FIFO is empty */
    while((QSPIREG(base, QSPI_SR) & QSPI_SR_TFE_MASK) == 0);

    return 0;
}

/* Send dummy data immediately after previous read command is sent - before TxFIFO becomes empty.
 * pRxData must be length of nSize + nPreSize */
uint32_t QSPI_recvFlsData(uint32_t base, uint8_t *pRxData, uint32_t nSize)
{
    uint32_t i;
    uint32_t u32Tmp;
    uint32_t nRecvLen = 0;

    /* If TxFIFO is not Full or specified data length is not reached, write data into Tx FIFO */
    while((QSPIREG(base, QSPI_SR) & QSPI_SR_TFNF_MASK) == 0);

    /* Write data into DRx register - TxFIFO */
    u32Tmp = 0xFFU;
    for(i=0; i<nSize; i++)
    {
        /* Write unused data into DRx register - TxFIFO */
        QSPIREG(base, QSPI_DR0) = u32Tmp;
        /* wait till there is data available in RxFIFO */
        while((QSPIREG(base, QSPI_SR) & QSPI_SR_RFNE_MASK) == 0);
        /* read the received data from RxFIFO */
        pRxData[i] = QSPIREG(base, QSPI_DR0);
        nRecvLen++;
    }
    /* wait till Tx FIFO is empty */
    while((QSPIREG(base, QSPI_SR) & QSPI_SR_TFE_MASK) == 0);
    /* If Rx FIFO is not empty, read data from Rx FIFO */
    while((QSPIREG(base, QSPI_SR) & QSPI_SR_RFNE_MASK) != 0)
    {
        pRxData[i] = QSPIREG(base, QSPI_DR0);
        i++;
        nRecvLen++;
    }

    return nRecvLen;
}

/********************************************************************************
 * Set QSPI_FRF=Quad, set DFS=31
 * The following registers need be configured:
 *     CTRLR0.QSPI_FRF = 0x02, Quad mode;
 *     CTRLR0.DFS = 32bit;
 *     CTRLR1.NDF = number of data frames to Tx or Rx, (not including inst and addr)
 *     QSPI_CTRLR0.WAIT_CYCLES = 8, dummy cycles;
 *     QSPI_CTRLR0.TRANS_TYPE = 0, inst and addr are transferred in Std SPI mode, data transfered in QSPI_FRF mode;
 *     QSPI_CTRLR0.INST_L = 8bit,
 *     QSPI_CTRLR0.ADDR_L = 24bit,
 *     QSPI_CTRLR0.CLK_STRETCH_EN = 1; enable SCLK stretch when Tx FIFO is empty and CTRLR1.NDF is not reached.
 ********************************************************************************/
void QSPI_enableQuadRead(uint32_t base)
{
    uint32_t u32Tmp = 0;

    /* Re-configure CTRLR0 to set Tx-only mode, and Frame Size 32bit */
    /* disable QSPI to allow cfg change */
    QSPI_disableModule(base);;

    /* Configure CTRLR0 register. */
    u32Tmp = QSPIREG(base, QSPI_CTRLR0);
    /* QSPI_FRF = Quad SPI mode */
    u32Tmp &= (~QSPI_CTRLR0_SPI_FRF_MASK);
    u32Tmp |= QSPI_CTRLR0_SPI_FRF(QSPI_CTRLR0_SPI_FRF_QUAD);
    /* Data Frame Size = 32bit */
    u32Tmp &= (~QSPI_CTRLR0_DFS_MASK);
    u32Tmp |= QSPI_CTRLR0_DFS(31);
    /* TMOD = TXRX */
    u32Tmp &= (~QSPI_CTRLR0_TMOD_MASK);
    u32Tmp |= QSPI_CTRLR0_TMOD(QSPI_CTRLR0_TMOD_RXONLY);
    QSPIREG(base, QSPI_CTRLR0) = u32Tmp;

    /* Set CTRLR1.NDF = 15, receive 15 data frames */
    /* This is for Data Frame only, Instruction and Address are not included in this count. */
    //QSPIREG(base, QSPI_CTRLR1) = 16-1;

    /* Set Tx Start threshold to 1 ? */
    QSPI_setFIFOTransferStartLevel(base, 0);

    /* enable QSPI */
    QSPI_enableModule(base);

    return;
}


void QSPI_enableStdRead(uint32_t base)
{
    uint32_t u32Tmp;

    /* Re-configure CTRLR0 to set Tx-only mode, and Frame Size 32bit */
    /* disable QSPI to allow cfg change */
    QSPI_disableModule(base);;

    /* Configure CTRLR0 register. */
    u32Tmp = QSPIREG(base, QSPI_CTRLR0);
    /* QSPI_FRF = Quad SPI mode */
    u32Tmp &= (~QSPI_CTRLR0_SPI_FRF_MASK);
    u32Tmp |= QSPI_CTRLR0_SPI_FRF(QSPI_CTRLR0_SPI_FRF_STD);
    /* Data Frame Size = 32bit */
    u32Tmp &= (~QSPI_CTRLR0_DFS_MASK);
    u32Tmp |= QSPI_CTRLR0_DFS(31);
    /* TMOD = TXRX */
    u32Tmp &= (~QSPI_CTRLR0_TMOD_MASK);
    u32Tmp |= QSPI_CTRLR0_TMOD(QSPI_CTRLR0_TMOD_TXRX);
    QSPIREG(base, QSPI_CTRLR0) = u32Tmp;

    /* Set Tx Start threshold to 1 ? */
    QSPI_setFIFOTransferStartLevel(base, 0);
    /* enable QSPI */
    QSPI_enableModule(base);

    return;
}

/******************************************************************************
 * nTransferMode = QSPI_XIP_CTRL_TRANS_TYPE_TT0/1/2
 * nCmd =
 *****************************************************************************/
uint32_t QSPI_enableXipMode(uint32_t base, uint32_t nTransferMode, uint32_t nCmd)
{
    uint32_t u32Tmp;
    /* Re-configure CTRLR0 to set Tx-only mode, and Frame Size 32bit */
    /* disable QSPI to allow cfg change */
    QSPI_disableModule(base);

    /* Configure CTRLR0 register. */
    u32Tmp = QSPIREG(base, QSPI_CTRLR0);
    /* QSPI_FRF = Quad SPI mode */
    u32Tmp &= (~QSPI_CTRLR0_SPI_FRF_MASK);
    u32Tmp |= QSPI_CTRLR0_SPI_FRF(QSPI_CTRLR0_SPI_FRF_QUAD);
    /* Data Frame Size = 32bit */
    u32Tmp &= (~QSPI_CTRLR0_DFS_MASK);
    u32Tmp |= QSPI_CTRLR0_DFS(31);
    /* TMOD = TXRX, TX-only, or Rx-only */
    u32Tmp &= (~QSPI_CTRLR0_TMOD_MASK);
    u32Tmp |= QSPI_CTRLR0_TMOD(QSPI_CTRLR0_TMOD_RXONLY);
    QSPIREG(base, QSPI_CTRLR0) = u32Tmp;


    /* Configure QSPI_SPI_CTRLR0 register. */
    u32Tmp = QSPIREG(base, QSPI_SPI_CTRLR0);

    u32Tmp |= QSPI_SPI_CTRLR0_XIP_PREFETCH_EN_MASK;
    u32Tmp |= QSPI_SPI_CTRLR0_XIP_INST_EN_MASK;
    u32Tmp |= QSPI_SPI_CTRLR0_XIP_DFS_HC_MASK;

    u32Tmp &= (~QSPI_SPI_CTRLR0_WAIT_CYCLES_MASK);
    u32Tmp |= QSPI_SPI_CTRLR0_WAIT_CYCLES(QSPI_DUMMY_CYCLES_LEN);

    QSPIREG(base, QSPI_SPI_CTRLR0) = u32Tmp;


    /* XIP instruction */
    QSPIREG(base, QSPI_XIP_INCR_INST) = nCmd;
    QSPIREG(base, QSPI_XIP_WRAP_INST) = nCmd;

    /* XIP_CTRL register */
    u32Tmp = QSPIREG(base, QSPI_XIP_CTRL);

    //u32Tmp |= QSPI_XIP_CTRL_XIP_PREFETCH_EN_MASK;

    u32Tmp |= QSPI_XIP_CTRL_INST_EN_MASK;

    u32Tmp |= QSPI_XIP_CTRL_XIP_DFS_HC_MASK;

    u32Tmp &= (~QSPI_XIP_CTRL_WAIT_CYCLES_MASK);
    u32Tmp |= QSPI_XIP_CTRL_WAIT_CYCLES(QSPI_DUMMY_CYCLES_LEN);

    u32Tmp &= (~QSPI_XIP_CTRL_INST_L_MASK);
    u32Tmp |= QSPI_XIP_CTRL_INST_L(QSPI_XIP_CTRL_INST_L_8BIT);

    u32Tmp &= (~QSPI_XIP_CTRL_ADDR_L_MASK);
    u32Tmp |= QSPI_XIP_CTRL_ADDR_L(QSPI_XIP_CTRL_ADDR_L_24bit);

    u32Tmp &= (~QSPI_XIP_CTRL_TRANS_TYPE_MASK);
    //u32Tmp |= QSPI_XIP_CTRL_TRANS_TYPE(QSPI_XIP_CTRL_TRANS_TYPE_TT0);
    u32Tmp |= QSPI_XIP_CTRL_TRANS_TYPE(nTransferMode);

    u32Tmp &= (~QSPI_XIP_CTRL_FRF_MASK);
    u32Tmp |= QSPI_XIP_CTRL_FRF(QSPI_XIP_CTRL_FRF_QUAD);

    QSPIREG(base, QSPI_XIP_CTRL) = u32Tmp;


    QSPI_enableModule(base);

    return 0;
}

//*****************************************************************************
//
// QSPI_setBaudRate
//
//*****************************************************************************
void
QSPI_setBaudRate(uint32_t base, uint32_t lspclkHz, uint32_t bitRate)
{
    uint32_t baud;

    //
    // Check the arguments.
    //
    ASSERT(QSPI_isBaseValid(base));
    ASSERT(bitRate <= (lspclkHz / 2U));
    ASSERT((lspclkHz / bitRate) <= 65534U);

    //
    // Set the clock.
    //
    QSPI_disableModule(base);
    QSPIREG(base, QSPI_BAUDR) = (lspclkHz / bitRate); // The last bit defaults to 0, no user configuration required
    QSPI_enableModule(base);
}

void
QSPI_pollingFIFOTransaction32(uint32_t base, uint16_t dataWidth,
                           uint32_t *pTxBuffer, uint32_t *pRxBuffer,
                           uint16_t numOfWords, uint16_t dummyData)
{
    ASSERT((dataWidth >= 4U) && (dataWidth <= 32U));
    //QSPI_setcharLength(base, dataWidth);

    //
    // Reset the TX / RX FIFO buffers to default state
    //
    QSPI_resetFIFO(base);

    //
    // Determine the number of 16-level words from number of words to be
    // transmitted / received
    //
    uint16_t numOfSixteenWords = numOfWords / (uint16_t)QSPI_FIFO_DEPTH;

    //
    // Determine the number of remaining words from number of words to be
    // transmitted / received
    //
    uint16_t remainingWords = numOfWords % (uint16_t)QSPI_FIFO_DEPTH;

    uint16_t count = 0;
    uint16_t i = 0;
    uint16_t txBuffer_pos = 0;
    uint16_t rxBuffer_pos = 0;

    //
    // Number of transactions is based on numOfSixteenWords
    // Each transaction will transmit and receive 16 words.
    //
    while(count < numOfSixteenWords)
    {
        //
        // Fill-up the SPI Transmit FIFO buffers
        //
        for(i = 1; i <= (uint16_t)QSPI_FIFO_DEPTH; i++)
        {
            if(pTxBuffer == NULL)
                QSPI_writeDataBlockingFIFO(base, dummyData & 0xFFU);
            else
                QSPI_writeDataBlockingFIFO(base, pTxBuffer[txBuffer_pos]);
            txBuffer_pos++;
        }

        //
        // Wait till SPI Receive FIFO buffer is full
        //
        while(QSPI_getRxFIFOStatus(base) < QSPI_FIFO_DEPTH)
        {
        }

        //
        // Read the SPI Receive FIFO buffers
        //
        for(i = 1U; i <= (uint16_t)QSPI_FIFO_DEPTH; i++)
        {
            if(pRxBuffer == NULL)
            {
                QSPI_readDataBlockingFIFO(base);
            }
            else
            {
                pRxBuffer[rxBuffer_pos] = QSPI_readDataBlockingFIFO(base);
                rxBuffer_pos++;
            }
        }

        count++;
    }

    //
    // Number of transactions is based on remainingWords
    //
    for(i = 0U; i < remainingWords; i++)
    {
        if(pTxBuffer == NULL)
            QSPI_writeDataBlockingFIFO(base, dummyData & 0xFFU);
        else
            QSPI_writeDataBlockingFIFO(base, pTxBuffer[txBuffer_pos]);
        txBuffer_pos++;
    }

    //
    // Wait till SPI Receive FIFO buffer remaining words
    //
    while((uint16_t)QSPI_getRxFIFOStatus(base) < remainingWords)
    {
    }

    //
    // Read the SPI Receive FIFO buffers
    //
    for(i = 0; i < remainingWords; i++)
    {
        if(pRxBuffer == NULL)
        {
            QSPI_readDataBlockingFIFO(base);
        }
        else
        {
            pRxBuffer[rxBuffer_pos] = QSPI_readDataBlockingFIFO(base);
            rxBuffer_pos++;
        }
    }

}

//*****************************************************************************
//
// QSPI_transmit24Bits
//
//*****************************************************************************
void
QSPI_transmit24Bits(uint32_t base, uint32_t data, uint16_t txDelay)
{
    uint32_t rxBuffer;
    uint32_t txBuffer;

    txBuffer = data;
    QSPI_pollingFIFOTransaction32(base, 24U, &txBuffer, &rxBuffer, 1U, txDelay);
}
//*****************************************************************************
//
// QSPI_receive16Bits
//
//*****************************************************************************

uint16_t
QSPI_receive16Bits(uint32_t base, QSPI_endianess endianness, uint16_t dummyData,
                  uint16_t txDelay)
{
    uint32_t rxBuffer;
    uint32_t txBuffer;
    uint16_t rxData = 0U;
    uint8_t *pData = (uint8_t *)&rxBuffer;

    ASSERT(dummyData <= 0xFFU);
    
    //memset(&txBuffer, dummyData & 0xFFU, 4);
      txBuffer = dummyData;
    QSPI_pollingFIFOTransaction32(base, 16U, &txBuffer, &rxBuffer, 1U, txDelay);
    if(endianness == QSPI_DATA_LITTLE_ENDIAN)
    {
        //
        // LITTLE_ENDIAN
        //
        rxData = (pData[1] << 8) | pData[0];
    }
    else
    {
        //
        // BIG_ENDIAN
        //
        rxData = (pData[0] << 8) | pData[1];
    }

    return(rxData);
}
//*****************************************************************************
//
// QSPI_receive24Bits
//
//*****************************************************************************

uint32_t
QSPI_receive24Bits(uint32_t base, QSPI_endianess endianness, uint16_t dummyData,
                           uint16_t txDelay)
{
    uint32_t rxBuffer;
    uint32_t txBuffer;
    uint32_t rxData = 0U;
    uint8_t *pData = (uint8_t *)&rxBuffer;

    ASSERT(dummyData <= 0xFFU);
    
    //memset(&txBuffer, dummyData & 0xFFU, 4);
      txBuffer = dummyData;
    QSPI_pollingFIFOTransaction32(base, 24U, &txBuffer, &rxBuffer, 1U, txDelay);
    if(endianness == QSPI_DATA_LITTLE_ENDIAN)
    {
        //
        // LITTLE_ENDIAN
        //
        rxData = ((uint32_t)pData[2] << 16) |
                 ((uint32_t)pData[1] << 8)  |
                 (uint32_t)pData[0];
    }
    else
    {
        //
        // BIG_ENDIAN
        //
        rxData = ((uint32_t)pData[0] << 16) |
                 ((uint32_t)pData[1] << 8)  |
                 (uint32_t)pData[2];
    }

    return(rxData);
}
//*****************************************************************************
//
// QSPI_transmit32Bits
//
//*****************************************************************************

void
QSPI_transmit32Bits(uint32_t base, uint32_t data, uint16_t txDelay)
{
    uint32_t rxBuffer;
    uint32_t txBuffer;

    txBuffer = data;
    QSPI_pollingFIFOTransaction32(base, 32U, &txBuffer, &rxBuffer, 1U, txDelay);
}
//*****************************************************************************
//
// QSPI_receive32Bits
//
//*****************************************************************************

uint32_t
QSPI_receive32Bits(uint32_t base, QSPI_endianess endianness, uint16_t dummyData,
                  uint16_t txDelay)
{
    uint32_t rxBuffer;
    uint32_t txBuffer;
    uint32_t rxData = 0U;
    uint8_t *pData = (uint8_t *)&rxBuffer;

    ASSERT(dummyData <= 0xFFU);

    //memset(&txBuffer, dummyData & 0xFFU, 4);
      txBuffer = dummyData;
    QSPI_pollingFIFOTransaction32(base, 32U, &txBuffer, &rxBuffer, 1U, txDelay);
    if(endianness == QSPI_DATA_LITTLE_ENDIAN)
    {
        //
        // LITTLE_ENDIAN
        //
        rxData = ((uint32_t)pData[3] << 24U) |
                 ((uint32_t)pData[2] << 16U) |
                 ((uint32_t)pData[1] << 8U)  |
                 (uint32_t)pData[0];
    }
    else
    {
        //
        // BIG_ENDIAN
        //
        rxData = ((uint32_t)pData[0] << 24U) |
                 ((uint32_t)pData[1] << 16U) |
                 ((uint32_t)pData[2] << 8U)  |
                 (uint32_t)pData[3];
    }

    return(rxData);
}

#endif
