/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    spi.c
*   @brief
*   @details
*
*/

/*
 * Commit History
 *
 */

#ifdef __cplusplus
extern "C"{
#endif

/**
 * @brief include files
 */
#include "gs32_version.h"
#include "spi.h"
#include "sysctl.h"

#if (IS_GS32F00xx(0x12,0x11) || IS_GS32F3xx(0x22,0x20))

/**
 * @brief SPI_setCofig
 *
 * This function is used to initialize SPI's
 * SPI protocol, mode of operation, bit rate, and data width.
 */
void
SPI_setConfig(uint32_t base, uint32_t lspclkHz, SPI_TransferProtocol protocol,
              SPI_Mode mode, uint32_t bitRate, uint16_t dataWidth)
{
    uint32_t regValue = 0;
    uint32_t baud;

    /**
     * @brief Check the arguments.
     */
    ASSERT(SPI_isBaseValid(base));
    ASSERT(bitRate <= (lspclkHz / 2U));             /* wfmodify 4->2        */
    ASSERT((lspclkHz / bitRate) <= 65534U);         /* wfmodify 128->65534  */
    ASSERT((dataWidth >= 4U) && (dataWidth <= 32U));/* wfmodify 1->4 16->32 */
    ASSERT((HWREG(base + SPI_O_SSIENR) & SPI_SSIENR_SSIC_EN) == 0U);


    /**
     * @brief Set polarity and phase
     */
    regValue = protocol;

    /**
     * @brief Set Trandfer Mode
     */
    regValue |= (mode & SPI_CTRLR0_TMOD_M);

    /**
     * @brief Set data width.
     */
    regValue |= ((uint32_t)(dataWidth - 1) << SPI_CTRLR0_DFS_32_S);


    uint32_t cfgMasterSlaveMask = 0;
    if(base == SPIA_BASE)
    {
        cfgMasterSlaveMask = SPIA_CFG_MASTER_MASK;
    }
    else if(base == SPIB_BASE)
    {
        cfgMasterSlaveMask = SPIB_CFG_MASTER_MASK;
    }
#if IS_GS32F3xx(0x22,0x20)
    else if(base == SPIC_BASE)
    {
        cfgMasterSlaveMask = SPIC_CFG_MASTER_MASK;
    }
    else if(base == SPID_BASE)
    {
        cfgMasterSlaveMask = SPID_CFG_MASTER_MASK;
    }
    else if(base == SPIE_BASE)
    {
        cfgMasterSlaveMask = SPIE_CFG_MASTER_MASK;
    }
#endif
    /**
     * @brief Check that the master/slave mode
     * 1 is the host mode and 0 is the slave mode
     */
    if((mode&SPI_MASTER_M))
    {
        /**
         * @brief set master mode
         */
        HWREG(SPI_CONFIG_REG) |= cfgMasterSlaveMask;
        /**
         * @brief set baud
         */
        HWREG(base+SPI_O_BAUDR) = (lspclkHz / bitRate) & (~0x00000001U);

        HWREG(base+SPI_O_SER)	 |= SPI_SER_SER;
    }
    else
    {
        /**
         * @brief set slave mode
         */
        HWREG(SPI_CONFIG_REG) &= ~cfgMasterSlaveMask;
    }

    HWREG(base+SPI_O_CTRLR0) = regValue;

}

/**
 * @brief Clear interrupt flag
 *
 */
void
SPI_clearInterruptStatus(uint32_t base, uint32_t intFlags)
{
    uint32_t status;
    /**
     * @brief*Check the arguments.
     */
    ASSERT(SPI_isBaseValid(base));

    if(intFlags & SPI_INT_RXFF_OVERFLOW)
    {
        status = HWREG(base+SPI_O_RXOICR);
    }

    if(intFlags & SPI_INT_TXFF_OF)
    {
        status = HWREG(base+SPI_O_TXOICR);
    }

    if(intFlags & SPI_INT_RXFF_UNDERFLOW)
    {
        status = HWREG(base+SPI_O_RXUICR);
    }

    if(intFlags & SPI_INT_MST)
    {
        status = HWREG(base+SPI_O_MSTICR);
    }
}

/**
 * @brief SPI_setBaudRate
 */
void
SPI_setBaudRate(uint32_t base, uint32_t lspclkHz, uint32_t bitRate)
{
    uint32_t baud;

    /** @brief Check the arguments. */
    ASSERT(SPI_isBaseValid(base));
    ASSERT(bitRate <= (lspclkHz / 2U));     /** @brief wfmodify 4->2       */
    ASSERT((lspclkHz / bitRate) <= 65534U); /** @brief wfmodify 128->65534 */

    /** @brief Set the clock. */
    HWREG(base+SPI_O_BAUDR) = (lspclkHz / bitRate) & (~0x00000001U);

}

/**
 * @brief SPI_pollingTransaction
 */
uint16_t
SPI_pollingTransaction(uint32_t base, uint16_t charLength, uint16_t data)
{
    uint16_t rxData;

    ASSERT(((HWREG(base+SPI_O_CTRLR0) & ~SPI_CTRLR0_DFS_32_M) + 1) == charLength);
    ASSERT(data < ((uint32_t)1U << charLength));

    /** @brief Write to SPI Transmit buffer */
    SPI_writeDataBlockingFIFO(base, data);

    /** @brief Read SPI Receive buffer */
    rxData = SPI_readDataBlockingFIFO(base);

    return(rxData);
}

/**
 * @brief SPI_pollingFIFOTransaction
 */
void
SPI_pollingFIFOTransaction(uint32_t base, uint16_t charLength,
                           uint16_t *pTxBuffer, uint16_t *pRxBuffer,
                           uint16_t numOfWords)
{
    ASSERT((charLength >= 1U) && (charLength <= 16U));
    SPI_setcharLength(base, charLength);

    /**
     * @brief Determine the number of 32-level words from number of words to be
     * transmitted / received
     */
    uint16_t numOfSixteenWords = numOfWords / (uint16_t)SPI_FIFO_TXFULL;

    /**
     * @brief Determine the number of remaining words from number of words to be
     * transmitted / received
     */
    uint16_t remainingWords = numOfWords % (uint16_t)SPI_FIFO_TXFULL;

    uint16_t count = 0;
    uint16_t i = 0;
    uint16_t txBuffer_pos = 0;
    uint16_t rxBuffer_pos = 0;

    /**
     * @brief Number of transactions is based on numOfSixteenWords
     * Each transaction will transmit and receive 16 words.
     */
    while(count < numOfSixteenWords)
    {
        /** @brief Fill-up the SPI Transmit FIFO buffers */
        for(i = 1U; i <= (uint16_t)SPI_FIFO_TXFULL; i++)
        {
            if(pTxBuffer == NULL)
                SPI_writeDataBlockingFIFO(base, 0xFF);
            else
                SPI_writeDataBlockingFIFO(base, pTxBuffer[txBuffer_pos]);
            txBuffer_pos++;
        }

        /** @brief Wait till SPI Receive FIFO buffer is full */
        while(SPI_getRxFIFOStatus(base) < SPI_FIFO_RXFULL)
        {
        }

        /** @brief Read the SPI Receive FIFO buffers */
        for(i = 1U; i <= (uint16_t)SPI_FIFO_RXFULL; i++)
        {
            if(pRxBuffer == NULL)
            {
                SPI_readDataBlockingFIFO(base);
            }
            else
            {
                pRxBuffer[rxBuffer_pos] = SPI_readDataBlockingFIFO(base);
                rxBuffer_pos++;
            }
        }

        count++;
    }

    /** @brief Number of transactions is based on remainingWords */
    for(i = 0U; i < remainingWords; i++)
    {
        SPI_writeDataBlockingFIFO(base, pTxBuffer[txBuffer_pos]);
        txBuffer_pos++;
    }

    /** @brief Wait till SPI Receive FIFO buffer remaining words */
    while((uint16_t)SPI_getRxFIFOStatus(base) < remainingWords)
    {
    }

    /** @brief Read the SPI Receive FIFO buffers */
    for(i = 0; i < remainingWords; i++)
    {
        if(pRxBuffer == NULL)
        {
            SPI_readDataBlockingFIFO(base);
        }
        else
        {
            pRxBuffer[rxBuffer_pos] = SPI_readDataBlockingFIFO(base);
            rxBuffer_pos++;
        }
    }

}

#endif /** @brief 1.2 2.2version */

#ifdef __cplusplus
}
#endif
