/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    sysctl.c
*   @brief   
*   @details
*
*/
/*
*  commit history
*  20240324, Jason, change 2.0ES IRC trim value to 0x05B5.
*  20240429, Jason, remove all none-ASCII chars.
*  20240506, ZhaoLei, add empty api sysctl_clock_init() for 3.0.
*  20240523, Jason, add clocksel=0 for selecting internal 10MHz clock.
*  20240615, Jaosn, place Sysctl_delay into CCM to make the delay time accurate.
*  20240625, Zhaolei, update 1.2 sysctl_clock_init.
*  20240705, Zhaolei, add 1.2 XTAL type auto detect.
*  20250507, Zhaolei, clear PLL unlock history after PLL is locked.
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "gs32_version.h"
#include "sysctl.h"

#include "device.h"

/**
 * @brief delay some time in uS (Micro Second);
 *
 * @param delay_ticks is the number of while loops, 1 loop is 7~8 sysclk-cycle
 *        Assuming SysClk is 240MHz, SysCtl_delay(30) is 30*8 = 240cycles = 1uS;
 * @return none
 */

__attribute__((section(".IlmFunc")))
void SysCtl_delay(volatile uint32_t delay_ticks)
{
    while(delay_ticks--);
}

#if IS_GS32F00xx(0x12, 0x30) || IS_GS32F3xx(0x22)

__attribute__((section(".IlmFunc")))
void Auto_setupXtal(void)
{
    uint32_t clkRefPinCtrl = 0;

	do{
		SysCtl_setCmdStClr(1);
		SysCtl_setCmdStClr(0);
	    SysCtl_delay(2000);
	}while(SysCtl_getCmdCheckFailRpt());

    SysCtl_setHseCmdEn(0);		//disable clock monitor

    # if (DEVICE_OSCSRC_FREQ <= 4*1000*1000)            //1 ~ 4MHz
        clkRefPinCtrl = GPIO_ClkRef_1MHz_TO_4MHz;
    # elif (DEVICE_OSCSRC_FREQ <= 12*1000*1000)         //4.1 ~ 12MHz
        clkRefPinCtrl = GPIO_ClkRef_4MHz_TO_12MHz;
    # elif (DEVICE_OSCSRC_FREQ <= 24*1000*1000)         //12.1 ~ 24MHz
        clkRefPinCtrl = GPIO_ClkRef_12MHz_TO_24MHz;
    # else                                              //24.1 ~ 48MHz
        clkRefPinCtrl = GPIO_ClkRef_24MHz_TO_48MHz;
    # endif

        clkRefPinCtrl |= GPIO_ClkRef_R_500K;

    #if (defined(__HSE_CLOCK_TYPE) && (__HSE_CLOCK_TYPE==1))        //external oscillator
        clkRefPinCtrl |= GPIO_ClkRef_OSCILLATOR;
        GPIO_setClkRef(clkRefPinCtrl);
        SysCtl_delay(10000);
    #elif (defined(__HSE_CLOCK_TYPE) && (__HSE_CLOCK_TYPE==0))      //external crystal
        clkRefPinCtrl |= GPIO_ClkRef_CRYSTAL;
        GPIO_setClkRef(clkRefPinCtrl);
    #else       //automatic detect XTAL type
    {
        uint32_t cnt;
        uint32_t retry = 0;

        // ----- step1: try oscillator
        GPIO_setClkRef(0);
        GPIO_setClkRef(clkRefPinCtrl | GPIO_ClkRef_OSCILLATOR);

        SysCtl_delay(2000);

        do{
            SysCtl_setX1CntClr();
            cnt = SysCtl_getX1Cnt();
            SysCtl_delay(50);
            cnt = SysCtl_getX1Cnt() - cnt;
        } while(cnt < 100 && retry++ < 10);    //wait osc to startup

        if (cnt >= 100) {
            goto __Auto_setupXtal_Succeeded;
        }

        // ----- step2: try crystal
        GPIO_setClkRef(0);
        GPIO_setClkRef(clkRefPinCtrl | GPIO_ClkRef_CRYSTAL);

        SysCtl_delay(2000);

        retry = 0;
        do{
            SysCtl_setX1CntClr();
            cnt = SysCtl_getX1Cnt();
            SysCtl_delay(50);
            cnt = SysCtl_getX1Cnt() - cnt;
        } while(cnt < 100 && retry++ < 1000);    //wait crystal to startup

        if (cnt >= 100) {
            goto __Auto_setupXtal_Succeeded;
        }

        while(1);   //FIXME, XTAL is broken, using internal osc ??
    }

__Auto_setupXtal_Succeeded:
    #endif

    SysCtl_setHseCmdEn(1);		//enable clock monitor

#if IS_GS32F00xx(0x30)
    SysCtl_setExtRefClkSel(EXT_REF_CLK_SEL_XTAL);
#endif
    uint32_t XTALCount,XTALCountNum = 0;

    /* Waiting for the crystal oscillator to start oscillating normally */
	do{
		SysCtl_setX1CntClr();
		SysCtl_delay(XTALWaitDelay);
		XTALCount = SysCtl_getX1Cnt();
		XTALCountNum++;
		if(XTALCountNum >= 500){
			break;
		}
	}while(XTALCount < XTALExpectCountLower ||  XTALCount > XTALExpectCountUpper);

	SysCtl_delay(XTALWaitDelay * XTALCountNum * 5 / 4);

    SysCtl_setRefClkSel(REF_CLK_TYPE_EXTERNAL_XTAL);

    /* Waiting for the external clock to be valid */
	do{
		SysCtl_setX1CntClr();
		SysCtl_delay(10);
		XTALCount = SysCtl_getX1Cnt();
		if(XTALCount < 0x7FF && XTALCount > 10){
			SysCtl_setCmdStClr(1);
			SysCtl_setCmdStClr(0);
		}
		SysCtl_delay(2000);
	}while(SysCtl_getCmdCheckFailRpt());
}

__STATIC_FORCEINLINE int32_t Pll_divSetupInteger(uint32_t oscFreq, uint32_t pllFreq)
{
    oscFreq /= 10000;		/* xx.xx MHz */
    pllFreq /= 10000;

    uint32_t vcoFreq = pllFreq * 2;     /* pll output from 4-phase */
    uint32_t postDiv1 = 1;
    uint32_t postDiv2 = 1;
    uint32_t refDiv = 1;
    uint32_t feedbackDiv = 1;
    uint32_t postDiv;

    /* input reference clock Div */
    if (oscFreq > 10*100) {
        if (oscFreq % (5*100) == 0) {
            oscFreq /= 5;
            refDiv = 5;
        } else if (oscFreq % (3*100) == 0) {
            oscFreq /= 3;
            refDiv = 3;
        }
    }

    uint32_t a = vcoFreq;
    uint32_t b = oscFreq;

    while (a % b != 0) {
        uint32_t c = a % b;
        a = b;
        b = c;
    }

    postDiv = oscFreq/b;
    feedbackDiv = vcoFreq/b;

    if (feedbackDiv * oscFreq < 600*100) {
        postDiv = (600*100 + vcoFreq - 1)/vcoFreq;
    }

    if (postDiv > 49) {
        postDiv2 = 7;
        postDiv1 = 7;
    } else if (postDiv>7) {
        postDiv2 = (postDiv + 6) / 7;
        postDiv1 = postDiv/postDiv2;
    } else {
        postDiv1 = postDiv;
        postDiv2 = 1;
    }

    if (postDiv2 > postDiv1) {
        postDiv1 = postDiv2 ^ postDiv1;
        postDiv2 = postDiv2 ^ postDiv1;
        postDiv1 = postDiv2 ^ postDiv1;
    }

    vcoFreq *= postDiv1 * postDiv2;

    feedbackDiv = vcoFreq/oscFreq;

    if (vcoFreq < 2200*100 && vcoFreq > 375*100 && vcoFreq/postDiv1/postDiv2/2 == pllFreq) {
        SysCtl_setPllFbdivCfg(feedbackDiv);
        SysCtl_setPllDivCfg(postDiv1 | (postDiv2 << 3) | (refDiv << 6));
        SysCtl_setPllFracCfg(0);
        SysCtl_setPllOthpdCfg(2);		/* integer mode, FOUT = (FREF * FBDIV) / (REFDIV * POSTDIV1 * POSTDIV2) */
        return 0;
    }

    return -1;
}

__STATIC_FORCEINLINE int32_t Pll_divSetupFractional(uint32_t oscFreq, uint32_t pllFreq)
{
    uint32_t vcoFreq = pllFreq * 2;     /* pll output from 4-phase */
    uint32_t postDiv1 = 1;
    uint32_t postDiv2 = 1;
    uint32_t preDiv = 1;
    uint32_t feedbackDiv = 1;
    uint32_t postDiv = 1;

    if (oscFreq < 10*1000*1000) {
        return -1;
    }

    if (oscFreq >= 30*1000*1000) {
        preDiv = 2;
        oscFreq /= 2;
    }

    if (feedbackDiv * oscFreq < 600*1000*1000) {
        postDiv = (600*1000*1000 + vcoFreq - 1)/vcoFreq;
    }

    if (postDiv > 49) {
        postDiv2 = 7;
        postDiv1 = 7;
    } else if (postDiv>7) {
        postDiv2 = (postDiv + 6) / 7;
        postDiv1 = postDiv/postDiv2;
    } else {
        postDiv1 = postDiv;
        postDiv2 = 1;
    }

    if (postDiv2 > postDiv1) {
        postDiv1 = postDiv2 ^ postDiv1;
        postDiv2 = postDiv2 ^ postDiv1;
        postDiv1 = postDiv2 ^ postDiv1;
    }

    vcoFreq *= postDiv1 * postDiv2;

    feedbackDiv = ((uint64)vcoFreq << 24)/oscFreq;

    if (vcoFreq < 2200u*1000*1000 && vcoFreq > 375*1000*1000) {
        SysCtl_setPllFbdivCfg((feedbackDiv>>24) & 0xFF);
        SysCtl_setPllDivCfg(postDiv1 | (postDiv2 << 3) | (preDiv << 6));
        SysCtl_setPllFracCfg(feedbackDiv & 0xFFFFFF);
        SysCtl_setPllOthpdCfg(0);		/* Fractional mode, FOUT = (FREF / REFDIV) * ((FBDIV + FRAC/2^24) / (POSTDIV1 * POSTDIV2))*/
        return 0;
    }

    return -1;
}

void sysctl_clock_init(uint32_t nPllClkFreq, uint32_t nOscFreq, uint32_t nSysClkDiv, uint32_t nAhbClkDiv, uint32_t nApbClkDiv)
{
    uint32_t clkRefPinCtrl = 0;

    //--- Step 0 : decrease APB clock the half of the normal frequency
    SysCtl_setSysclkS2nEn(0);
    SysCtl_delay(150);

    SysCtl_setRefClkSel(REF_CLK_TYPE_INTERNAL_OSC2_10MHZ);

    SysCtl_setApbClkDiv(CRG_DIV_1);
#if IS_GS32F00xx(0x12)
    SysCtl_setSlvSysClkDiv(CRG_DIV_1);
#endif
    SysCtl_setDspSysClkDiv(CRG_DIV_1);
#if IS_GS32F00xx(0x30) || IS_GS32F3xx(0x22)
    SysCtl_setPeriClkDiv(CRG_DIV_1);
#endif
    SysCtl_setSysDivLoad();
    SysCtl_delay(150);

    //--- Step 1 : If(HSE_ENABLE) Config HSE_OSC and Clock Detector, Config Clock Detector, ELSE Read Eflash rpt, Config HSI TRIM
    #if (__HSE_ENABLE == 1)
        Auto_setupXtal();
    #endif

    //--- Step 2 : Config the PLL, Read PLL_LCK until PLL Locked
    SysCtl_setPllDisableSscg(0x01);
    SysCtl_setPllResetptr(0x01);
    SysCtl_setPllPd(0x01);
    SysCtl_delay(150);

    if (Pll_divSetupInteger(nOscFreq, nPllClkFreq) != 0) {          /* ~180 cycles */
        if (Pll_divSetupFractional(nOscFreq, nPllClkFreq) != 0) {   /* ~240 cycles */
            SysCtl_setPllFbdivCfg(2 * (nPllClkFreq) / nOscFreq);
            SysCtl_setPllDivCfg(0x1 | (0x01 << 3) | (0x01 << 6));
            SysCtl_setPllFracCfg(0);
            SysCtl_setPllOthpdCfg(2);
        }
    }

    SysCtl_setPllDivval(0x00000001UL);                        //Modulation Freq = F_clksscg / (DIVVAL * 128)
    SysCtl_setPllSpread(0);                                   //5'd1:1% ... 5'd31:31%
    SysCtl_setPllDownspread(0);                               //0:center-spread 1:downspread

    SysCtl_delay(10);

    SysCtl_setPllDisableSscg(0);
    SysCtl_delay(10);

    SysCtl_setPllPd(0);
    SysCtl_delay(150);

    SysCtl_setPllResetptr(0);

    /* Wait until PLL lock status is 1 */
    while(SysCtl_getPllLck() == 0);

    SysCtl_clearPllUnlckHis();

    //--- Step 3 : Config APBCLK/CANCLK/ANACLK Divider
#if IS_GS32F00xx(0x12)
    SysCtl_setSlvSysClkDiv(nSysClkDiv);
#endif
    SysCtl_setDspSysClkDiv(nSysClkDiv);
#if IS_GS32F3xx(0x22)
    SysCtl_setPeriClkDiv(nAhbClkDiv * nSysClkDiv);
#endif

#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
    SysCtl_setApbClkDiv(nApbClkDiv * nSysClkDiv);
#elif IS_GS32F00xx(0x30)
    SysCtl_setPeriClkDiv(nAhbClkDiv);
    SysCtl_setApbClkDiv(nApbClkDiv);
#endif
	/* After writing 1 to the register CFG_SYS_DIV_LOAD_POS,
	 cfg_div_dsp_sys, cfg_div_apb_sys registers are updated,
	 DSP/APB divider are updated at the same time */
    SysCtl_setSysDivLoad(); 

    //--- Step 4 : SYSCLK Slow2Normal, Config SYSCLK Divider to 2div and then to 1div
    SysCtl_setSysclkS2nEn(0x01);
    SysCtl_delay(150);

    SysCtl_setCanClkSrcSel(CAN_CLK_TYPE_PLL);
    SysCtl_setCanClkDiv((nPllClkFreq / 40000000)); //40MHz       pll_clk = 2* sys_clk
}

#else
void sysctl_clock_init(uint32_t nPllClkFreq, uint32_t nOscFreq, uint32_t nSysClkDiv, uint32_t nAhbClkDiv, uint32_t nApbClkDiv)
{

}
#endif

#ifdef __cplusplus
}
#endif
