/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    asysctl_v22.h
*   @brief   analog system top level configuration
*            do NOT include this file directly, include "sysctl.h" instead
*
*/
/**
 * commit history
 * 20241218,Hezhiyuan,Modify comments
*/
#ifndef DEVICE_DRIVERLIB_SYSCTL_ANA_V22_H_
#define DEVICE_DRIVERLIB_SYSCTL_ANA_V22_H_

#include "gs32_version.h"

#if IS_GS32F3xx(0x22)

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "inc/hw_types.h"
#include "debug.h"
#include "inc/hw_memmap.h"
#include "inc/hw_asysctl_v22.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define ASYSCTL_VREFHIA  0x1U //!< VREFHIA
#define ASYSCTL_VREFHIB  0x2U //!< VREFHIB
#define ASYSCTL_VREFHIC  0x4U //!< VREFHIC
#define ASYSCTL_VREFHID  0x8U //!< VREFHIC
typedef enum {
    CMP_ID_1 = 0,
    CMP_ID_2 = 1,
    CMP_ID_3 = 2,
    CMP_ID_4 = 3,
    CMP_ID_5 = 4,
    CMP_ID_6 = 5,
    CMP_ID_7 = 6,
    CMP_ID_8 = 7,
    CMP_ID_9 = 8,
    CMP_ID_10 = 9,
    CMP_ID_11 = 10,
    CMP_ID_12 = 11,
}CMPIdType_t;

/**
 * 
 * Values used for function ASysCtl_selectCMPHNMux() and
 * ASysCtl_selectCMPHNMuxValue().  These values can be OR-ed together and
 * passed to ASysCtl_selectCMPHNMux().
 * 
 */
#define ASYSCTL_CMPHNMUX_SELECT_1 0x1U //!< CMPHNMUX select 1
#define ASYSCTL_CMPHNMUX_SELECT_2 0x2U //!< CMPHNMUX select 2
#define ASYSCTL_CMPHNMUX_SELECT_3 0x4U //!< CMPHNMUX select 3
#define ASYSCTL_CMPHNMUX_SELECT_4 0x8U //!< CMPHNMUX select 4
#define ASYSCTL_CMPHNMUX_SELECT_5 0x10U //!< CMPHNMUX select 5
#define ASYSCTL_CMPHNMUX_SELECT_6 0x20U //!< CMPHNMUX select 6
#define ASYSCTL_CMPHNMUX_SELECT_7 0x40U //!< CMPHNMUX select 7
#define ASYSCTL_CMPHNMUX_SELECT_8 0x80U //!< CMPHNMUX select 8
#define ASYSCTL_CMPHNMUX_SELECT_9 0x100U //!< CMPHNMUX select 9
#define ASYSCTL_CMPHNMUX_SELECT_10 0x200U //!< CMPHNMUX select 10
#define ASYSCTL_CMPHNMUX_SELECT_11 0x400U //!< CMPHNMUX select 11
#define ASYSCTL_CMPHNMUX_SELECT_12 0x800U //!< CMPHNMUX select 12


/**
 * 
 * Values used for function ASysCtl_selectCMPLNMux() and
 * ASysCtl_selectCMPLNMuxValue().  These values can be OR-ed together and
 * passed to ASysCtl_selectCMPLNMux().
 * 
 */
#define ASYSCTL_CMPLNMUX_SELECT_1 0x1U //!< CMPLNMUX select 1
#define ASYSCTL_CMPLNMUX_SELECT_2 0x2U //!< CMPLNMUX select 2
#define ASYSCTL_CMPLNMUX_SELECT_3 0x4U //!< CMPLNMUX select 3
#define ASYSCTL_CMPLNMUX_SELECT_4 0x8U //!< CMPLNMUX select 4
#define ASYSCTL_CMPLNMUX_SELECT_5 0x10U //!< CMPLNMUX select 5
#define ASYSCTL_CMPLNMUX_SELECT_6 0x20U //!< CMPLNMUX select 6
#define ASYSCTL_CMPLNMUX_SELECT_7 0x40U //!< CMPLNMUX select 7
#define ASYSCTL_CMPLNMUX_SELECT_8 0x80U //!< CMPLNMUX select 8
#define ASYSCTL_CMPLNMUX_SELECT_9 0x100U //!< CMPLNMUX select 9
#define ASYSCTL_CMPLNMUX_SELECT_10 0x200U //!< CMPLNMUX select 10
#define ASYSCTL_CMPLNMUX_SELECT_11 0x400U //!< CMPLNMUX select 11
#define ASYSCTL_CMPLNMUX_SELECT_12 0x800U //!< CMPLNMUX select 12

/**
 * 
 *  ASysCtl_CMPHPMuxSelect used for function ASysCtl_selectCMPHPMux().
 * 
 */
typedef enum
{
    ASYSCTL_CMPHPMUX_SELECT_1  = 0U, //!< CMPHPMUX select 1
    ASYSCTL_CMPHPMUX_SELECT_2  = 3U, //!< CMPHPMUX select 2
    ASYSCTL_CMPHPMUX_SELECT_3  = 6U, //!< CMPHPMUX select 3
    ASYSCTL_CMPHPMUX_SELECT_4  = 9U, //!< CMPHPMUX select 4
	ASYSCTL_CMPHPMUX_SELECT_5  = 12U,//!< CMPHPMUX select 5
	ASYSCTL_CMPHPMUX_SELECT_6  = 15U,//!< CMPHPMUX select 6
	ASYSCTL_CMPHPMUX_SELECT_7  = 18U, //!< CMPHPMUX select 7
	ASYSCTL_CMPHPMUX_SELECT_8  = 21U, //!< CMPHPMUX select 8
	ASYSCTL_CMPHPMUX_SELECT_9  = 24U, //!< CMPHPMUX select 9
	ASYSCTL_CMPHPMUX_SELECT_10 = 27U, //!< CMPHPMUX select 10
	ASYSCTL_CMPHPMUX_SELECT_11 = 30U, //!< CMPHPMUX select 11
	ASYSCTL_CMPHPMUX_SELECT_12 = 33U //!< CMPHPMUX select 12
} ASysCtl_CMPHPMuxSelect;

/**
 * 
 *  ASysCtl_CMPLPMuxSelect used for function ASysCtl_selectCMPLPMux().
 * 
 */
typedef enum
{
    ASYSCTL_CMPLPMUX_SELECT_1  = 0U, //!< CMPLPMUX select 1
    ASYSCTL_CMPLPMUX_SELECT_2  = 3U, //!< CMPLPMUX select 2
    ASYSCTL_CMPLPMUX_SELECT_3  = 6U, //!< CMPLPMUX select 3
    ASYSCTL_CMPLPMUX_SELECT_4  = 9U, //!< CMPLPMUX select 4
	ASYSCTL_CMPLPMUX_SELECT_5  = 12U,//!< CMPLPMUX select 5
	ASYSCTL_CMPLPMUX_SELECT_6  = 15U,//!< CMPLPMUX select 6
	ASYSCTL_CMPLPMUX_SELECT_7  = 18U, //!< CMPLPMUX select 7
	ASYSCTL_CMPLPMUX_SELECT_8  = 21U, //!< CMPLPMUX select 7
	ASYSCTL_CMPLPMUX_SELECT_9  = 24U, //!< CMPLPMUX select 7
	ASYSCTL_CMPLPMUX_SELECT_10 = 27U, //!< CMPLPMUX select 7
	ASYSCTL_CMPLPMUX_SELECT_11 = 30U, //!< CMPLPMUX select 7
	ASYSCTL_CMPLPMUX_SELECT_12 = 33U //!< CMPLPMUX select 7
} ASysCtl_CMPLPMuxSelect;

typedef enum
{
    ASYSCTL_CMP_SELECT_1  = 0x484U,  //!< CMP select 1
    ASYSCTL_CMP_SELECT_2  = 0x488U,  //!< CMP select 2
    ASYSCTL_CMP_SELECT_3  = 0x48CU,  //!< CMP select 3
    ASYSCTL_CMP_SELECT_4  = 0x490U, //!< CMP select 4
	ASYSCTL_CMP_SELECT_5  = 0x494U, //!< CMP select 5
	ASYSCTL_CMP_SELECT_6  = 0x498U, //!< CMP select 6
	ASYSCTL_CMP_SELECT_7  = 0x49CU,  //!< CMP select 7
	ASYSCTL_CMP_SELECT_8  = 0x4A0U,  //!< CMP select 8
	ASYSCTL_CMP_SELECT_9  = 0x4A4U,  //!< CMP select 9
	ASYSCTL_CMP_SELECT_10 = 0x4A8U,  //!< CMP select 10
	ASYSCTL_CMP_SELECT_11 = 0x4ACU,  //!< CMP select 11
	ASYSCTL_CMP_SELECT_12 = 0x4B0U,  //!< CMP select 12
} ASysCtl_CMPSelect;


#define ANALOG_TRIM_EFUSE_VERSION_ADDR      0x1FFF8100

typedef enum AnaTrimPeriId_v0 {
    V0_TRIM_ID_ADCA_OFFSET,
    V0_TRIM_ID_ADCA_VCM_INT,
    V0_TRIM_ID_ADCA_VCM_EXT,
	V0_TRIM_ID_ADCA_OFF_DIR,

    V0_TRIM_ID_ADCB_OFFSET,
    V0_TRIM_ID_ADCB_VCM_INT,
    V0_TRIM_ID_ADCB_VCM_EXT,
	V0_TRIM_ID_ADCB_OFF_DIR,

    V0_TRIM_ID_ADCC_OFFSET,
    V0_TRIM_ID_ADCC_VCM_INT,
    V0_TRIM_ID_ADCC_VCM_EXT,
	V0_TRIM_ID_ADCC_OFF_DIR,

#if defined(ADCD_BASE)
	V0_TRIM_ID_ADCD_OFFSET,
	V0_TRIM_ID_ADCD_VCM_INT,
	V0_TRIM_ID_ADCD_VCM_EXT,
	V0_TRIM_ID_ADCD_OFF_DIR,
#endif

    V0_TRIM_ID_MAX,
}AnaTrimPeriId_v0_t;

typedef enum AnaTopTrimPeriId {
    TRIM_ID_LDO_1V2 = 0,
    TRIM_ID_TOP_BOR,
    TRIM_ID_ANA_LDO,
    TRIM_ID_OSC1,
    TRIM_ID_OSC2,

}AnaTopTrimPeriId_t;

enum {
    ADC_VREFHI_MODE_3P3V_IN = 0,
    ADC_VREFHI_MODE_2P5V_IN,
    ADC_VREFHI_MODE_3P3V_OUT,
    ADC_VREFHI_MODE_2P5V_OUT,
    ADC_VREFHI_MODE_3P0V_OUT,

    ADC_VREFHI_MODE_MAX,
};

enum {
    ADC_VCM_MODE_1P65V  = 0,
    ADC_VCM_MODE_2P475V,
    ADC_VCM_MODE_1P25V,
    ADC_VCM_MODE_1P5V,

    ADC_VCM_MODE_MAX,
};

enum {
    VREFHI_2P5V = 0,
    VREFHI_1P65V,

    VREFHI_MAX,
};

enum {
    ADCA_ID = 0,
    ADCB_ID,
    ADCC_ID,
    ADCD_ID,

    ADC_ID_MAX,
};

typedef struct __AdcModeParam_t {
    struct {
        uint32_t regAddr;
        uint32_t efuseAddr;
    }addrMap[ADC_ID_MAX];

    uint32_t regWidth : 3;  //bytes
    uint32_t endbit : 5;
    uint32_t startbit : 5;
    uint32_t defaultTrimVal : 19;

    struct {
        uint8_t efuseEndbit;
        uint8_t efuseStartbit;
    }mode[ADC_VREFHI_MODE_MAX];
}AdcModeParam_t;

typedef struct __AdcTrim_t {
    AdcModeParam_t vrefhiCoarse;
    AdcModeParam_t vrefhiFine;
    AdcModeParam_t offsetDir;
    AdcModeParam_t offset;
    AdcModeParam_t digOffset;
    AdcModeParam_t digGainErr;

    AdcModeParam_t vcmCoarse;
    AdcModeParam_t vcmFine;
}AdcTrim_t;

#define REG_WIDTH_16BITS     2
#define REG_WIDTH_32BITS     4

typedef struct AnaTopTrimTable {
    uint32_t regAddr;
    uint32_t regWidth : 3;  //bytes
    uint32_t endbit : 5;
    uint32_t startbit : 5;
    uint32_t defaultTrimVal : 19;
    uint32_t efuseAddr;
    uint16_t efuseEndbit;
    uint16_t efuseStartbit;
}AnaTopTrimTable_t;

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */



/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */

void ASysCtl_initConfig(void);

/*
 *  ANA_SYSCTRL_LOCK   Locking Register.  Offset 0x0000
 *
 *  Register Field   Field Range
 *  ana_sysctrl_lock 31 : 0
 *                
 *  Writing unlock value to this register allows writing to all registers protected by lock. Writing other values will lock all registers protected by lock.
 *  Unlock value: 0x5A5A5A5A.
 * 
 */
__STATIC_INLINE void ASysCtl_setAnaSysctrlLock(void)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CFG_LOCK) = 0x01;
}

__STATIC_INLINE void ASysCtl_setAnaSysctrlUnLock(void)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CFG_LOCK) = 0x5A5A5A5A;
}

__STATIC_INLINE uint32_t ASysCtl_Dig_getAnaSysctrlLock(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CFG_LOCK);
}

/**
 * TSNSCTL  Temperature Sensor Control Register 0x04C
 *
 *  Register Field   Field Range
 *  reserved         15:1
 *  tsns_enable      0
 *
 */
__STATIC_INLINE void ASysCtl_setTsnsEnable(uint16_t enable)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_TSNSCTL) = enable & 0x01;
}

__STATIC_INLINE uint16_t ASysCtl_getTsnsEnable(void)
{
    return HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_TSNSCTL) & 0x01;
}


/**
 * ANAREFCTL    Analog Reference Control Register   0x0D0
 *
 *  Register Field       Field Range
 *  reserved             15:12
 *  anarefd2p5sel        11 Analog reference D 2.5V source select.
 *  anarefc2p5sel        10 Analog reference C 2.5V source select.
 *  anarefb2p5sel        9  Analog reference B 2.5V source select.
 *  anarefa2p5sel        8  Analog reference A 2.5V source select. In internal reference mode,
 *                          this bit selects which voltage the internal reference buffer drives onto
 *                          the VREFHI pin. The buffer can drive either 1.65V onto the pin,
 *                          resulting in a reference range of 0 to 3.3V, or the buffer can drive
 *                          2.5V onto the pin, resulting in a reference range of 0 to 2.5V. If
 *                          switching between these two modes, the user must allow adequate
 *                          time for the external capacitor to charge to the new voltage before
 *                          using the ADC or buffered DAC. If multiple VREFHI pins are ganged
 *                          together (for lower pin-count packages), then the reference voltage
 *                          select for the ganged pins should always be configured to the same
 *                          setting.
 *                          0 Internal 1.65V reference mode (3.3V reference range)
 *                          1 Internal 2.5V reference mode (2.5V reference range)
 *
 *  reserved             7:4
 *  anarefdsel           3  Analog reference D mode select
 *  anarefcsel           2  Analog reference C mode select
 *  anarefbsel           1  Analog reference B mode select
 *  anarefasel           0  Analog reference A mode select. This bit selects whether the
 *                          VREFHIA pin uses internal reference mode (the device drives a
 *                          voltage onto the VREFHI pin) or external reference mode (the
 *                          system is expected to drive a voltage into the VREFHI pin). If
 *                          multiple VREFHI pins are ganged together (for lower pin-count
 *                          packages), then the mode select for the ganged pins should always
 *                          be configured to the same setting
 *                          0 Internal reference mode
 *                          1 External reference mode
 *
 */
/* --- Analog reference A mode select --- */
__STATIC_INLINE void ASysCtl_setAnaRefaSel(uint16_t external)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) =
            (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) & (~(1<<0))) | (external & 0x01);
}

__STATIC_INLINE uint16_t ASysCtl_getAnaRefaSel(void)
{
    return HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) & 0x01;
}

/* --- Analog reference B mode select --- */
__STATIC_INLINE void ASysCtl_setAnaRefbSel(uint16_t external)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) =
            (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) & (~(1<<1))) | ((external & 0x01) << 1);
}

__STATIC_INLINE uint16_t ASysCtl_getAnaRefbSel(void)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) >> 1) & 0x01;
}

/* --- Analog reference C mode select --- */
__STATIC_INLINE void ASysCtl_setAnaRefcSel(uint16_t external)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) =
            (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) & (~(1<<2))) | ((external & 0x01) << 2);
}

__STATIC_INLINE uint16_t ASysCtl_getAnaRefcSel(void)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) >> 2) & 0x01;
}

/* --- Analog reference D mode select --- */
__STATIC_INLINE void ASysCtl_setAnaRefdSel(uint16_t external)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) =
            (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) & (~(1<<3))) | ((external & 0x01) << 3);
}

__STATIC_INLINE uint16_t ASysCtl_getAnaRefdSel(void)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) >> 3) & 0x01;
}

/* --- Analog reference A 2.5V source select --- */
__STATIC_INLINE void ASysCtl_setAnaRefa2p5Sel(uint16_t sel2p5)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) =
            (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) & (~(1<<8))) | ((sel2p5 & 0x01) << 8);
}

__STATIC_INLINE uint16_t ASysCtl_getAnaRefa2p5Sel(void)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) >> 8) & 0x01;
}

/* --- Analog reference B 2.5V source select --- */
__STATIC_INLINE void ASysCtl_setAnaRefb2p5Sel(uint16_t sel2p5)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) =
            (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) & (~(1<<9))) | ((sel2p5 & 0x01) << 9);
}

__STATIC_INLINE uint16_t ASysCtl_getAnaRefb2p5Sel(void)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) >> 9) & 0x01;
}

/* --- Analog reference C 2.5V source select --- */
__STATIC_INLINE void ASysCtl_setAnaRefc2p5Sel(uint16_t sel2p5)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) =
            (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) & (~(1<<10))) | ((sel2p5 & 0x01) << 10);
}

__STATIC_INLINE uint16_t ASysCtl_getAnaRefc2p5Sel(void)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) >> 10) & 0x01;
}

/* --- Analog reference D 2.5V source select --- */
__STATIC_INLINE void ASysCtl_setAnaRefd2p5Sel(uint16_t sel2p5)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) =
            (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) & (~(1<<11))) | ((sel2p5 & 0x01) << 11);
}

__STATIC_INLINE uint16_t ASysCtl_getAnaRefd2p5Sel(void)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) >> 11) & 0x01;
}


/**
 * VMONCTL  Voltage Monitor Control Register    0x0E0
 *
 *  Register Field   Field Range
 *  reserved         15:9
 *  borlvmondis      8      BORL disable on VDDIO.
 *                           0 BORL is enabled on VDDIO, i.e., BOR circuit will be triggered if
 *                             VDDIO goes lower than the lower BOR threshold of VDDIO.
 *                           1 BORL is disabled on VDDIO, i.e., BOR circuit will not be triggered if
 *                             VDDIO goes lower than the lower BOR threshold of VDDIO.
 *                           Reset type: SYSRSn
 *  reserved         7:0
 *
 */
__STATIC_INLINE void SysCtl_setBorLvmonDisable(uint16_t disable)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_VMONCTL) = (disable & 0x01) << 8;
}

__STATIC_INLINE uint16_t SysCtl_getBorLvmonDisable(void)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_VMONCTL) >> 8) & 0x01;
}


/**
 * CMPHPMXSEL   Bits to select one of the many sources on CmpHP inputs  0x104
 *
 *  Register Field   Field Range
 *  reserved         31:30
 *  cmp10_hp_sel     29:27
 *  cmp9_hp_sel      26:24
 *  cmp8_hp_sel      23:21
 *  cmp7_hp_sel      20:18
 *  cmp6_hp_sel      17:15
 *  cmp5_hp_sel      14:12
 *  cmp4_hp_sel      11:9
 *  cmp3_hp_sel      8:6
 *  cmp2_hp_sel      5:3
 *  cmp1_hp_sel      2:0    CMP1HPMXSEL bits, Refer to figure 4 of Analog system control doc
 *  @remarks
 *  - Only values 0 to 4 are valid, rest are reserved
 *
 */
__STATIC_INLINE void ASysCtl_setCmpHpSel(CMPIdType_t cmpId, uint32_t sel)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL) & (~(0x07<<(cmpId*3)))) | ((sel & 0x07) << (cmpId*3));
}

__STATIC_INLINE uint32_t ASysCtl_getCmpHpSel(CMPIdType_t cmpId)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL) >> (cmpId*3)) & 0x07;
}


/**
 * CMPLPMXSEL   Bits to select one of the many sources on CmpLP inputs  0x108
 *
 *  Register Field   Field Range
 *  reserved           31:30
 *  cmp10_lp_sel       29:27
 *  cmp9_lp_sel        26:24
 *  cmp8_lp_sel        23:21
 *  cmp7_lp_sel        20:18
 *  cmp6_lp_sel        17:15
 *  cmp5_lp_sel        14:12
 *  cmp4_lp_sel        11:9
 *  cmp3_lp_sel        8:6
 *  cmp2_lp_sel        5:3
 *  cmp1_lp_sel        2:0      CMP1LPMXSEL bits, Refer to figure 4 of Analog system control doc
 *  @remarks
 *  - Only values 0 to 4 are valid, rest are reserved
 *
 */
__STATIC_INLINE void ASysCtl_setCmpLpSel(CMPIdType_t cmpId, uint32_t sel)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL) & (~(0x07<<(cmpId*3)))) | ((sel & 0x07) << (cmpId*3));
}

__STATIC_INLINE uint32_t ASysCtl_getCmpLpSel(CMPIdType_t cmpId)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL) >> (cmpId*3)) & 0x07;
}


/**
 * CMPHNMXSEL   Bits to select one of the many sources on CmpHN inputs  0x10C
 *
 *  Register Field   Field Range
 *  reserved            15:12
 *  cmp12_hn_sel        11
 *  cmp11_hn_sel        10
 *  cmp10_hn_sel        9
 *  cmp9_hn_sel         8
 *  cmp8_hn_sel         7
 *  cmp7_hn_sel         6
 *  cmp6_hn_sel         5
 *  cmp5_hn_sel         4
 *  cmp4_hn_sel         3
 *  cmp3_hn_sel         2
 *  cmp2_hn_sel         1
 *  cmp1_hn_sel         0     CMP1HNMXSEL bits, Refer to figure 4 of Analog system control doc
 *  @remarks
 *  - Only values 0 to 4 are valid, rest are reserved
 *
 */
__STATIC_INLINE void ASysCtl_setCmpHnSel(CMPIdType_t cmpId, uint16_t sel)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) =
            (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) & (~(0x01<<cmpId))) | ((sel & 0x01) << cmpId);
}

__STATIC_INLINE uint16_t ASysCtl_getCmpHnSel(CMPIdType_t cmpId)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) >> cmpId) & 0x01;
}


/**
 * CMPLNMXSEL   Bits to select one of the many sources on CmpLN inputs  0x10E
 *
 *  Register Field   Field Range
 *  cmp12_ln_sel           11
 *  cmp11_ln_sel           10
 *  cmp10_ln_sel           9
 *  cmp9_ln_sel            8
 *  cmp8_ln_sel            7
 *  cmp7_ln_sel            6
 *  cmp6_ln_sel            5
 *  cmp5_ln_sel            4
 *  cmp4_ln_sel            3
 *  cmp3_ln_sel            2
 *  cmp2_ln_sel            1
 *  cmp1_ln_sel            0    CMP1LNMXSEL bits, Refer to figure 4 of Analog system control doc
 *  @remarks
 *  - Only values 0 to 4 are valid, rest are reserved
 *
 */
__STATIC_INLINE void ASysCtl_setCmpLnSel(CMPIdType_t cmpId, uint16_t sel)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLNMXSEL) =
            (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLNMXSEL) & (~(0x01<<cmpId))) | ((sel & 0x01) << cmpId);
}

__STATIC_INLINE uint16_t ASysCtl_getCmpLnSel(CMPIdType_t cmpId)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLNMXSEL) >> cmpId) & 0x01;
}


/**
 * ADCDACLOOPBACK   Enable loopback from DAC to ADCs    0x110
 *
 *  Register Field   			Field Range
 *  reserved                    31:4
 *  adcdacloopback_enlb2adcd    3
 *  adcdacloopback_enlb2adcc    2
 *  adcdacloopback_enlb2adcb    1
 *  adcdacloopback_enlb2adca    0     1: Loops back COMPDACA output to ADCA.
 *                                    0: Loop back is broken.
 *  @remarks
 *  - Setting this bit to 1, will override the CHSEL specification for
 *  the ADC. ADC would sample COMPDACA output irrespective of the
 *  value of CHSEL.
 *
 */
__STATIC_INLINE void ASysCtl_setAdcDacLoopbackEn1b2aAdca(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) & (~(0x01<<0))) | ((enable & 0x01) << 0);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcDacLoopbackEn1b2aAdca(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) >> 0) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAdcDacLoopbackEn1b2aAdcb(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) & (~(0x01<<1))) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcDacLoopbackEn1b2aAdcb(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) >> 1) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAdcDacLoopbackEn1b2aAdcc(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) & (~(0x01<<2))) | ((enable & 0x01) << 2);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcDacLoopbackEn1b2aAdcc(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) >> 2) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAdcDacLoopbackEn1b2aAdcd(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) & (~(0x01<<3))) | ((enable & 0x01) << 3);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcDacLoopbackEn1b2aAdcd(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCDACLOOPBACK) >> 3) & 0x01;
}


/**
 *  CMPHPMXSEL1 Bits to select one of the many sources on CmpHP inputs  0x120
 *
 *  Register Field   	Field Range
 *  reserved            31:6
 *  cmp12_hp_sel        5:3
 *  cmp11_hp_sel        2:0     CMP11HPMXSEL bits, Refer to figure 4 of Analog system control doc
 *  @remarks
 *  - Only values 0 to 4 are valid, rest are reserved
 *
 */
__STATIC_INLINE void ASysCtl_setCmp11HpSel(uint32_t sel)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL1) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL1) & (~(0x07<<0))) | ((sel & 0x07) << 0);
}

__STATIC_INLINE uint32_t ASysCtl_getCmp11HpSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL1) >> 0) & 0x07;
}

__STATIC_INLINE void ASysCtl_setCmp12HpSel(uint32_t sel)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL1) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL1) & (~(0x07<<3))) | ((sel & 0x07) << 3);
}

__STATIC_INLINE uint32_t ASysCtl_getCmp12HpSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL1) >> 3) & 0x07;
}


/**
 *  CMPLPMXSEL1 Bits to select one of the many sources on CmpLP inputs  0x124
 *
 *  Register Field   	Field Range
 *  reserved            31:6
 *  cmp12_lp_sel        5:3
 *  cmp11_lp_sel        2:0     CMP11LPMXSEL bits, Refer to figure 4 of Analog system control doc
 *  @remarks
 *  - Only values 0 to 4 are valid, rest are reserved
 *
 */
__STATIC_INLINE void ASysCtl_setCmp11LpSel(uint32_t sel)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL1) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL1) & (~(0x07<<0))) | ((sel & 0x07) << 0);
}

__STATIC_INLINE uint32_t ASysCtl_getCmp11LpSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL1) >> 0) & 0x07;
}

__STATIC_INLINE void ASysCtl_setCmp12LpSel(uint32_t sel)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL1) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL1) & (~(0x07<<3))) | ((sel & 0x07) << 3);
}

__STATIC_INLINE uint32_t ASysCtl_getCmp12LpSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL1) >> 3) & 0x07;
}


/**
 *  ADCSOCFRCGB         ADC Global SOC Force    0x21C
 *
 *  Register Field   		Field Range
 *  reserved                31:16
 *  adcsocfrcgb_soc15       15
 *  adcsocfrcgb_soc14       14
 *  adcsocfrcgb_soc13       13
 *  adcsocfrcgb_soc12       12
 *  adcsocfrcgb_soc11       11
 *  adcsocfrcgb_soc10       10
 *  adcsocfrcgb_soc9        9
 *  adcsocfrcgb_soc8        8
 *  adcsocfrcgb_soc7        7
 *  adcsocfrcgb_soc6        6
 *  adcsocfrcgb_soc5        5
 *  adcsocfrcgb_soc4        4
 *  adcsocfrcgb_soc3        3
 *  adcsocfrcgb_soc2        2
 *  adcsocfrcgb_soc1        1
 *  adcsocfrcgb_soc0        0   Indicate if SOC0 selected for global SW trigger
 *                              0 : Not selected for Global SW Trigger
 *                              1 : Selected for Global SW Trigger
 *
 */
__STATIC_INLINE void ASysCtl_setAdcGlobalSocForce(uint32_t socMask, uint32_t enable)
{
    if (enable) {
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGB) |= socMask;
    } else {
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGB) &= ~socMask;
    }
}

__STATIC_INLINE uint32_t ASysCtl_getAdcGlobalSocForce(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGB) & 0xFFFF;
}


/**
 *  ADCSOCFRCGBSEL  ADC Global SOC Force Select 0x220
 *
 *  Register Field   		Field Range
 *  reserved               15:4
 *  adcsocfrcgbsel_adcd    3
 *  adcsocfrcgbsel_adcc    2
 *  adcsocfrcgbsel_adcb    1
 *  adcsocfrcgbsel_adca    0    Indicate if ADCA selected for global SW trigger
 *                              0 : Not selected for Global SW Trigger
 *                              1 : Selected for Global SW Trigger
 *
 */
__STATIC_INLINE void ASysCtl_setAdcaGlobalSocForceSel(uint16_t enable)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGBSEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGBSEL) & (~(1<<0))) | (enable & 0x01);
}

__STATIC_INLINE uint16_t ASysCtl_getAdcaGlobalSocForceSel(void)
{
    return HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGBSEL) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAdcbGlobalSocForceSel(uint16_t enable)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGBSEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGBSEL) & (~(1<<1))) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint16_t ASysCtl_getAdcGlobalSocForceSel(void)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGBSEL) >> 1) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAdccGlobalSocForceSel(uint16_t enable)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGBSEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGBSEL) & (~(1<<2))) | ((enable & 0x01) << 2);
}

__STATIC_INLINE uint16_t ASysCtl_getAdccGlobalSocForceSel(void)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGBSEL) >> 2) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAdcdGlobalSocForceSel(uint16_t enable)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGBSEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGBSEL) & (~(1<<3))) | ((enable & 0x01) << 3);
}

__STATIC_INLINE uint16_t ASysCtl_getAdcdGlobalSocForceSel(void)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCSOCFRCGBSEL) >> 3) & 0x01;
}


/**
 *  AGPIOFILTER AGPIO Filter Control Register   0x222
 *
 *  Register Field   		Field Range
 *  reserved                15:10
 *  agpiofilter_bottom      9:8     AGPIOFILTER Control for bottom side pins
 *                                  00 : Filter bypass configuration (min ACQPS = 90ns)
 *                                  01 : 333Ohm filter setting (min ACQPS = 125ns)
 *                                  10 : 500 Ohm filter setting (min ACQPS = 160ns)
 *                                  11 : 1KOhm filter setting (min ACQPS = 230ns)
 *  reserved                7:2
 *  agpiofilter_right       1:0     AGPIOFILTER Control for right side pins
 *                                  00 : Filter bypass configuration (min ACQPS = 90ns)
 *                                  01 : 333Ohm filter setting (min ACQPS = 125ns)
 *                                  10 : 500 Ohm filter setting (min ACQPS = 160ns)
 *                                  11 : 1KOhm filter setting (min ACQPS = 230ns)
 *
 */
__STATIC_INLINE void ASysCtl_setAgpioFilterRight(uint16_t filter)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOFILTER) =
            (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOFILTER) & (~(3<<0))) | (filter & 0x03);
}

__STATIC_INLINE uint16_t ASysCtl_getAgpioFilterRight(void)
{
    return HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOFILTER) & 0x03;
}

__STATIC_INLINE void ASysCtl_setAgpioFilterBottom(uint16_t filter)
{
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOFILTER) =
            (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOFILTER) & (~(3<<8))) | ((filter & 0x03) << 8);
}

__STATIC_INLINE uint16_t ASysCtl_getAgpioFilterBottom(void)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOFILTER) >> 8) & 0x03;
}

/**
 *  AGPIOCTRLG              AGPIO Control Register G    0x240
 *
 *  Register Field   			Field Range
 *  reserved                  31:28
 *  agpioctrlg_gpio219        27
 *  agpioctrlg_gpio218        26
 *  agpioctrlg_gpio217        25
 *  agpioctrlg_gpio216        24
 *  agpioctrlg_gpio215        23
 *  agpioctrlg_gpio214        22
 *  agpioctrlg_gpio213        21
 *  agpioctrlg_gpio212        20
 *  agpioctrlg_gpio211        19
 *  agpioctrlg_gpio210        18
 *  agpioctrlg_gpio209        17
 *  agpioctrlg_gpio208        16
 *  agpioctrlg_gpio207        15
 *  agpioctrlg_gpio206        14
 *  agpioctrlg_gpio205        13
 *  agpioctrlg_gpio204        12
 *  agpioctrlg_gpio203        11
 *  agpioctrlg_gpio202        10
 *  agpioctrlg_gpio201        9
 *  agpioctrlg_gpio200        8
 *  agpioctrlg_gpio199        7
 *  agpioctrlg_gpio198        6         One time configuration for GPIO198 to decide whether AGPIO
 *                                      functionality is enabled
 *                                      0 - AGPIO functionality is disabled
 *                                      1 - AGPIO functionality is enabled
 *  reserved                  5:0
 *
 *  gpioEn  198~219
 */
__STATIC_INLINE void ASysCtl_setAgpioCtrlG(uint32_t gpioNum, uint32_t en)
{
    if (gpioNum >= 198 && gpioNum <= 219) {
        gpioNum = gpioNum - 198 + 6;
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLG) =
                (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLG) & (~(1<<gpioNum))) | ((en & 0x01) << gpioNum);
    }
}

__STATIC_INLINE uint16_t ASysCtl_getAgpioCtrlG(uint32_t gpioNum)
{
    uint32_t ret = 0;

    if (gpioNum >= 198 && gpioNum <= 219) {
        gpioNum = gpioNum - 198 + 6;
        ret = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLG) >> gpioNum) & 0x01;
    }

    return ret;
}


/**
 *  AGPIOCTRLH              AGPIO Control Register H    0x244
 *
 *  Register Field   		Field Range
 *  reserved               31:19
 *  agpioctrlh_gpio242     18
 *  agpioctrlh_gpio241     17
 *  agpioctrlh_gpio240     16
 *  agpioctrlh_gpio239     15
 *  agpioctrlh_gpio238     14
 *  agpioctrlh_gpio237     13
 *  agpioctrlh_gpio236     12
 *  agpioctrlh_gpio235     11
 *  agpioctrlh_gpio234     10
 *  agpioctrlh_gpio233     9
 *  agpioctrlh_gpio232     8
 *  agpioctrlh_gpio231     7
 *  agpioctrlh_gpio230     6
 *  agpioctrlh_gpio229     5
 *  agpioctrlh_gpio228     4
 *  agpioctrlh_gpio227     3
 *  agpioctrlh_gpio226     2
 *  agpioctrlh_gpio225     1
 *  agpioctrlh_gpio224     0
 *
 *  gpioEn  224~242
 */
__STATIC_INLINE void ASysCtl_setAgpioCtrlH(uint32_t gpioNum, uint32_t en)
{
    if (gpioNum >= 224 && gpioNum <= 242) {
        gpioNum = gpioNum - 224;
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLH) =
                (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLH) & (~(1<<gpioNum))) | ((en & 0x01) << gpioNum);
    }
}

__STATIC_INLINE uint16_t ASysCtl_getAgpioCtrlH(uint32_t gpioNum)
{
    uint32_t ret = 0;

    if (gpioNum >= 224 && gpioNum <= 242) {
        gpioNum = gpioNum - 224;
        ret = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLH) >> gpioNum) & 0x01;
    }

    return ret;
}


/**
 *  GPIOINENACTRL   GPIOINENACTRL Control Register  0x268
 *
 *  Register Field   		Field Range
 *  reserved                31:6
 *  gpioinenactrl_gpio103   5           One time configuration for GPIO103 to decide whether Input buffer
 *                                      (INENA control) is enabled or disabled
 *                                      0 - Input buffer is disabled
 *                                      1 - Input buffer is enabled
 *  gpioinenactrl_gpio46    4
 *  gpioinenactrl_gpio31    3
 *  gpioinenactrl_gpio25    2
 *  gpioinenactrl_gpio23    1
 *  gpioinenactrl_gpio0     0
 *
 */
__STATIC_INLINE void ASysCtl_setGpio0InputBufferEnable(uint32_t en)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_GPIOINENACTRL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_GPIOINENACTRL) & (~(1<<0))) | ((en & 0x01) << 0);
}

__STATIC_INLINE uint16_t ASysCtl_getGpio0InputBufferEnable(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLH) >> 0) & 0x01;
}

__STATIC_INLINE void ASysCtl_setGpio23InputBufferEnable(uint32_t en)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_GPIOINENACTRL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_GPIOINENACTRL) & (~(1<<1))) | ((en & 0x01) << 1);
}

__STATIC_INLINE uint16_t ASysCtl_getGpio23InputBufferEnable(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLH) >> 1) & 0x01;
}

__STATIC_INLINE void ASysCtl_setGpio25InputBufferEnable(uint32_t en)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_GPIOINENACTRL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_GPIOINENACTRL) & (~(1<<2))) | ((en & 0x02) << 2);
}

__STATIC_INLINE uint16_t ASysCtl_getGpio25InputBufferEnable(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLH) >> 2) & 0x01;
}

__STATIC_INLINE void ASysCtl_setGpio31InputBufferEnable(uint32_t en)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_GPIOINENACTRL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_GPIOINENACTRL) & (~(1<<3))) | ((en & 0x01) << 3);
}

__STATIC_INLINE uint16_t ASysCtl_getGpio31InputBufferEnable(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLH) >> 4) & 0x01;
}

__STATIC_INLINE void ASysCtl_setGpio46InputBufferEnable(uint32_t en)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_GPIOINENACTRL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_GPIOINENACTRL) & (~(1<<4))) | ((en & 0x01) << 4);
}

__STATIC_INLINE uint16_t ASysCtl_getGpio46InputBufferEnable(void)
{
    return (HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLH) >> 4) & 0x01;
}

__STATIC_INLINE void ASysCtl_setGpio103InputBufferEnable(uint32_t en)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_GPIOINENACTRL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_GPIOINENACTRL) & (~(1<<5))) | ((en & 0x01) << 5);
}

__STATIC_INLINE uint16_t ASysCtl_getGpio103InputBufferEnable(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLH) >> 5) & 0x01;
}


/**
 * ANA_BOR_CFG  BOR Configuration       0x304
 *
 *  Register Field    Field Range
 *  reserved          31:12
 *  bor_test_en       11
 *  bor_trim          10:7
 *  bor_rising_sel    6:4
 *  bor_falling_sel   3:1
 *  reserved          0
 *
 */
__STATIC_INLINE void ASysCtl_setBorVfallDSel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) &(~(0x07 << 1))) | ((value & 0x07) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getBorVfallDSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) >> 1) & 0x07;
}

__STATIC_INLINE void ASysCtl_setBorVriseDSel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) & (~(0x07 << 4))) | ((value & 0x07) << 4);
}

__STATIC_INLINE uint32_t ASysCtl_getBorVriseDSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) >> 4) & 0x07;
}

__STATIC_INLINE void ASysCtl_setBorTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) &(~(0x0F << 7))) | ((value & 0x0F) << 7);
}

__STATIC_INLINE uint32_t ASysCtl_getBorTrim(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) >> 7) & 0x0F;
}

__STATIC_INLINE void ASysCtl_setBorTestEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG) & ~(1<<11)) | ((value & 0x01) << 11);
}

__STATIC_INLINE uint32_t ASysCtl_getBorTestEn(void)
{
    return ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_BOR_CFG)) >> 11) & 0x01;
}

/**
 * ANA_LDO_1V1_CFG  1V1LDO Configuration        0x308
 *
 *  Register Field   		Field Range
 *  ldo_1v1_test_en         11
 *  ldo_1v1_test_mux_sel    10:8
 *  ldo_1v1_chop_en         7
 *  ldo_1v1_test_mode       6:5
 *  ldo_1v1_vsel            4:0
 * *
 */
__STATIC_INLINE void ASysCtl_setLdo1v1Vsel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) &(~(0x01F << 0))) | (value & 0x1F);
}

__STATIC_INLINE uint32_t ASysCtl_getLdo1v1Vsel(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) & 0x1F;
}

__STATIC_INLINE void ASysCtl_setLdo1v1TestMode(uint32_t mode)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) &(~(0x03 << 5))) | ((mode & 0x3) << 5);
}

__STATIC_INLINE uint32_t ASysCtl_getLdo1v1TestMode(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) >> 5) & 0x3;
}

__STATIC_INLINE void ASysCtl_setLdoChopEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) &(~(0x01 << 7))) | ((enable & 0x1) << 7);
}

__STATIC_INLINE uint32_t ASysCtl_getLdoChopEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) >> 7) & 0x01;
}

__STATIC_INLINE void ASysCtl_setLdo1v1TestMuxSel(uint32_t mux)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) &(~(0x07 << 8))) | ((mux & 0x7) << 8);
}

__STATIC_INLINE uint32_t ASysCtl_getLdo1v1TestMuxSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) >> 8) & 0x07;
}

__STATIC_INLINE void ASysCtl_setLdo1v1TestEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) &(~(0x01 << 11))) | ((enable & 0x1) << 11);
}

__STATIC_INLINE uint32_t ASysCtl_getLdo1v1TestEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) >> 11) & 0x01;
}


/**
 * ANA_EFLASH_BOR_CFG   EFLASH_BOR Configuration        0x30C
 *
 *  Register Field   			Field Range
 *  reserved                    31:3
 *  eflash_bor_test_sel         2:1
 *  eflash_bor_test_en          0
 */
__STATIC_INLINE void ASysCtl_setEflashBorTestEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_EFLASH_BOR_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_EFLASH_BOR_CFG) & (~(0x01<<0))) | (enable & 0x01);
}

__STATIC_INLINE uint32_t ASysCtl_getEflashBorTestEn(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_EFLASH_BOR_CFG) & 0x01;
}

__STATIC_INLINE void ASysCtl_setEflashBorTestSel(uint32_t sel)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_EFLASH_BOR_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_EFLASH_BOR_CFG) & (~(0x03<<1))) | ((sel & 0x03) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getEflashBorTestSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_EFLASH_BOR_CFG) >> 1) & 0x03;
}


/**
 *  ANA_TOP_BG_CFG   BG Configuration        0x310
 *
 *  Register Field   			Field Range
 *  reserved                    31:5
 *  top_bg_test_mux_sel         4:2
 *  top_bg_test_mux_en          1
 *  top_bg_chop_en              0
 *
 */
__STATIC_INLINE void ASysCtl_setTopBgChopEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) & ~(0x01<<0)) | (enable & 0x01);
}

__STATIC_INLINE uint32_t ASysCtl_getTopBgChopEn(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) & 0x01;
}

__STATIC_INLINE void ASysCtl_setTopBgTestMuxEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) & ~(0x01<<1)) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getTopBgTestMuxEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) >> 1) & 0x01;
}

__STATIC_INLINE void ASysCtl_setTopBgTestMuxSel(uint32_t mux)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) & ~(0x07<<2)) | ((mux & 0x07) << 2);
}

__STATIC_INLINE uint32_t ASysCtl_getTopBgTestMuxSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_BG_CFG) >> 2) & 0x07;
}


/**
 * ANA_OSC_TRIM         OSC TRIM Configuration      0x314
 *
 *  Register Field   			Field Range
 *  osc2_freqtrim               30 : 16
 *  reserved                    15
 *  osc1_freqtrim               14:0
 */
__STATIC_INLINE void ASysCtl_setOsc1Trim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) & (~0x7FFF)) | (value & 0x7FFF);
}

__STATIC_INLINE uint32_t ASysCtl_getOsc1Trim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) & 0x7FFF;
}

__STATIC_INLINE void ASysCtl_setOsc2Trim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) & (~(0x7FFF << 16))) | ((value & 0x7FFF) << 16);
}

__STATIC_INLINE uint32_t ASysCtl_getOsc2Trim(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_OSC_TRIM) >> 16) & 0x7FFF;
}


/**
 * ANA_PVM_CFG  PVM Configuration       0x318
 *
 *  Register Field   		Field Range
 *  reserved                31:12
 *  pvm_test_en             11
 *  pvm_trim                10:7
 *  pvm_rising_sel          6:4
 *  pvm_falling_sel         3:1
 *  pvm_en                  0
 * *
 */
__STATIC_INLINE void ASysCtl_setPvmEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) & ~(1<<0)) | ((enable & 0x01) << 0);
}

__STATIC_INLINE uint32_t ASysCtl_getPvmEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG)) & 0x01;
}

__STATIC_INLINE void ASysCtl_setPvmVfallDSel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) &(~(0x07 << 1))) | ((value & 0x07) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getPvmVfallDSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) >> 1) & 0x07;
}

__STATIC_INLINE void ASysCtl_setPvmVriseDSel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) & (~(0x07 << 4))) | ((value & 0x07) << 4);
}

__STATIC_INLINE uint32_t ASysCtl_getPvmVriseDSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) >> 4) & 0x07;
}

__STATIC_INLINE void ASysCtl_setPvmTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) & ~(0x0F << 7)) | ((value & 0x0F) << 7);
}

__STATIC_INLINE uint32_t ASysCtl_getPvmTrim(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) >> 7) & 0x0F;
}

__STATIC_INLINE void ASysCtl_setPvmTestEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG) & ~(1<<11)) | ((value & 0x01) << 11);
}

__STATIC_INLINE uint32_t ASysCtl_getPvmTestEn(void)
{
    return ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_PVM_CFG)) >> 11) & 0x01;
}


/**
 * ANA_POR_1V1_CFG  POR_1V1 Configuration       0x31C
 *
 *  Register Field   		Field Range
 *  reserved                31:1
 *  por_1v1_test_en         0
 * *
 */
__STATIC_INLINE void ASysCtl_setPor1v1TestEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_POR_1V1_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_POR_1V1_CFG) & ~(1<<0)) | (enable & 0x01);
}

__STATIC_INLINE uint32_t ASysCtl_getPor1v1TestEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_POR_1V1_CFG)) & 0x01;
}


/**
 * ANA_TOP_POR_CFG      TOP_POR Configuration       0x320
 *
 *  Register Field   		Field Range
 *  reserved                31:1
 *  top_por_test_en         0
 * *
 */
__STATIC_INLINE void ASysCtl_setTopPorTestEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_POR_CFG) = \
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_POR_CFG) & ~(1<<0)) | (enable & 0x01);
}

__STATIC_INLINE uint32_t ASysCtl_getTopPorTestEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_POR_CFG)) & 0x01;
}

/**
 * ADC_RST ADC_RST Register      0x400
 *
 *  Register Field   	Field Range
 *  adcd_rst    			3
 *  adcc_rst    			2
 *  adcb_rst    			1
 *  adca_rst    			0
 *
 */
/*--- ADCA ---*/
__STATIC_INLINE void ASysCtl_setAdcReset(uint32_t rst)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) & ~(0xF<<0)) | (rst & 0xF);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcaRst(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) & 0x01;
}

__STATIC_INLINE void ASysCtl_resetAdca(void)
{
    ASysCtl_setAdcReset(1);
    ASysCtl_setAdcReset(0);
}

/*--- ADCB ---*/
__STATIC_INLINE void ASysCtl_setAdcbRst(uint32_t rst)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) & ~(1<<1)) | ((rst & 0x01) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcbRst(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) >> 1) & 0x01;
}

__STATIC_INLINE void ASysCtl_resetAdcb(void)
{
    ASysCtl_setAdcbRst(1);
    ASysCtl_setAdcbRst(0);
}

/*--- ADCC ---*/
__STATIC_INLINE void ASysCtl_setAdccRst(uint32_t rst)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) & ~(1<<2)) | ((rst & 0x01) << 2);
}

__STATIC_INLINE uint32_t ASysCtl_getAdccRst(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) >> 2) & 0x01;
}

__STATIC_INLINE void ASysCtl_resetAdcc(void)
{
    ASysCtl_setAdccRst(1);
    ASysCtl_setAdccRst(0);
}

/*--- ADCD ---*/
__STATIC_INLINE void ASysCtl_setAdcdRst(uint32_t rst)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) & ~(1<<3)) | ((rst & 0x01) << 3);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcdRst(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_RST) >> 3) & 0x01;
}
__STATIC_INLINE void ASysCtl_resetAdcd(void)
{
    ASysCtl_setAdcdRst(1);
    ASysCtl_setAdcdRst(0);
}


/*
 *  ADC_SAMP_SEL    ADC_SAMP_SEL Register     0x404
 *
 *  Register Field              Field Range
 *  reserved                    31:15
 *  adcd_samp_sel               14:12
 *  reserved                    11
 *  adcc_samp_sel               10:8
 *  reserved                    7
 *  adcb_samp_sel               6:4
 *  reserved                    3
 *  adca_samp_sel               2:0
 *
 *  adcX = 0,1,2,3
 */
__STATIC_INLINE void ASysCtl_setAdcSampSel(uint32_t adcX, uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_SAMP_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_SAMP_SEL) & ~(0x07 << (adcX*4))) | ((value& 0x07) << (adcX*4));
}

__STATIC_INLINE uint32_t ASysCtl_getAdcSampSel(uint32_t adcX)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_SAMP_SEL) >> (adcX*4)) & 0x7;
}


/*
 *  ADCA_SPARE   ADCA Reserved Register  offset  0x0408
 *
 *  Register Field              Field Range
 *  adca_spare                  31 : 0
 *                
 */
__STATIC_INLINE void ASysCtl_setAdcaSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAdcaSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_SPARE);
}


/*
 *  ADCB_SPARE   ADCB Reserved Register  offset  0x040C
 *
 *  Register Field              Field Range
 *  adcb_spare                  31 : 0
 *
 */
__STATIC_INLINE void ASysCtl_setAdcbSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAdcbSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_SPARE);
}


/*
 *  ADCC_SPARE   ADCC Reserved Register  offset  0x0410
 *
 *  Register Field              Field Range
 *  adcc_spare                  31 : 0
 *
 */
__STATIC_INLINE void ASysCtl_setAdccSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAdccSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_SPARE);
}


/*
 *  ADCD_SPARE   ADCD Reserved Register  offset  0x0414
 *
 *  Register Field              Field Range
 *  adcd_spare                  31 : 0
 *
 */
__STATIC_INLINE void ASysCtl_setAdcdSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAdcdSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_SPARE);
}

/*
 *  ADC_SPARE   ADC Spare Registers  Offset 0x408+0x4*n   n = 0~2
 *
 *  Register Field    Field Range
 *  adc_spare         31 : 0
 *
 *
 */
__STATIC_INLINE void ASysCtl_setAdcSpare(uint32_t n, uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_SPARE + 0x4*n) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAdcSpare(uint32_t n)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_SPARE + 0x4*n);
}

/*
 *  ADC_TOP_SPARE   ADC TOP Reserved Register   0x0418
 *
 *  Register Field              Field Range
 *  adc_spare                   31 : 0
 *                
 * 
 */
__STATIC_INLINE void ASysCtl_setAdcTopSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_TOP_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAdcTopSpare(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_TOP_SPARE) >> 0);
}


/*
 *  CMP_EN              CMP_EN Register       0x041C
 *
 *  Register Field              Field Range
 *  reserved                    31:1
 *  cmp_en                      0
 *
 */
__STATIC_INLINE void ASysCtl_setCmpEnable(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_EN) = enable;
}

__STATIC_INLINE uint32_t ASysCtl_getCmpEnable(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_EN) & 0x01;
}


/*
 *  CMP_RST             CMP_RST Register      0x0420
 *
 *  Register Field              Field Range
 *  reserved                    31:1
 *  cmp_rst                     0
 *
 */
__STATIC_INLINE void ASysCtl_setCmpReset(uint32_t rst)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_RST) = rst & 0x01;
}

__STATIC_INLINE uint32_t ASysCtl_getCmpReset(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_RST) & 0x01;
}

__STATIC_INLINE void ASysCtl_resetCmp(void)
{
    ASysCtl_setCmpReset(1);
    ASysCtl_setCmpReset(0);
}

/*
 *  CMP1_H_TRIM     CMPSS1.H Trim Value    0x0424
 *  CMP1_L_TRIM     CMPSS1.L Trim Value    0x0428
 *  CMP2_H_TRIM     CMPSS2.H Trim Value    0x042C
 *  CMP2_L_TRIM     CMPSS2.L Trim Value    0x0430
 *  CMP3_H_TRIM     CMPSS3.H Trim Value    0x0434
 *  CMP3_L_TRIM     CMPSS3.L Trim Value    0x0438
 *  CMP4_H_TRIM     CMPSS4.H Trim Value    0x043C
 *  CMP4_L_TRIM     CMPSS4.L Trim Value    0x0440
 *  CMP5_H_TRIM     CMPSS5.H Trim Value    0x0444
 *  CMP5_L_TRIM     CMPSS5.L Trim Value    0x0448
 *  CMP6_H_TRIM     CMPSS6.H Trim Value    0x044C
 *  CMP6_L_TRIM     CMPSS6.L Trim Value    0x0450
 *  CMP7_H_TRIM     CMPSS7.H Trim Value    0x0454
 *  CMP7_L_TRIM     CMPSS7.L Trim Value    0x0458
 *  CMP8_H_TRIM     CMPSS8.H Trim Value    0x045C
 *  CMP8_L_TRIM     CMPSS8.L Trim Value    0x0460
 *  CMP9_H_TRIM     CMPSS9.H Trim Value    0x0464
 *  CMP9_L_TRIM     CMPSS9.L Trim Value    0x0468
 *  CMP10_H_TRIM    CMPSS10.H Trim Value   0x046C
 *  CMP10_L_TRIM    CMPSS10.L Trim Value   0x0470
 *  CMP11_H_TRIM    CMPSS11.H Trim Value   0x0474
 *  CMP11_L_TRIM    CMPSS11.L Trim Value   0x0478
 *  CMP12_H_TRIM    CMPSS12.H Trim Value   0x047C
 *  CMP12_L_TRIM    CMPSS12.L Trim Value   0x0480
 *
 *  Register Field              Field Range
 *  cmp1_h_trim                 31:0
 *
 */
typedef enum {
    TRIM_CMP1_H = 0,
    TRIM_CMP1_L = 1,
    TRIM_CMP2_H = 2,
    TRIM_CMP2_L = 3,
    TRIM_CMP3_H = 4,
    TRIM_CMP3_L = 5,
    TRIM_CMP4_H = 6,
    TRIM_CMP4_L = 7,
    TRIM_CMP5_H = 8,
    TRIM_CMP5_L = 9,
    TRIM_CMP6_H = 10,
    TRIM_CMP6_L = 11,
    TRIM_CMP7_H = 12,
    TRIM_CMP7_L = 13,
    TRIM_CMP8_H = 14,
    TRIM_CMP8_L = 15,
    TRIM_CMP9_H = 16,
    TRIM_CMP9_L = 17,
    TRIM_CMP10_H = 18,
    TRIM_CMP10_L = 19,
    TRIM_CMP11_H = 20,
    TRIM_CMP11_L = 21,
    TRIM_CMP12_H = 22,
    TRIM_CMP12_L = 23,
}CMPTrimIdType_t;

__STATIC_INLINE void ASysCtl_setCmpTrim(CMPTrimIdType_t cmpTrimId, uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP1_H_TRIM + cmpTrimId * 4) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getCmpTrim(CMPTrimIdType_t cmpTrimId)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP1_H_TRIM + cmpTrimId * 4);
}

__STATIC_INLINE void ASysCtl_setCmpHTrim(CMPIdType_t cmpId, uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP1_H_TRIM + cmpId * 8) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getCmpHTrim(CMPIdType_t cmpId)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP1_H_TRIM + cmpId * 8);
}

__STATIC_INLINE void ASysCtl_setCmpLTrim(CMPIdType_t cmpId, uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP1_L_TRIM + cmpId * 8) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getCmpLTrim(CMPIdType_t cmpId)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP1_L_TRIM + cmpId * 8);
}


/*
 *  CMP1_SPARE  CMP1 Reserved Register   0x0484
 *  CMP2_SPARE  CMP2 Reserved Register   0x0488
 *  CMP3_SPARE  CMP3 Reserved Register   0x048C
 *  CMP4_SPARE  CMP4 Reserved Register   0x0490
 *  CMP5_SPARE  CMP5 Reserved Register   0x0494
 *  CMP6_SPARE  CMP6 Reserved Register   0x0498
 *  CMP7_SPARE  CMP7 Reserved Register   0x049C
 *  CMP8_SPARE  CMP8 Reserved Register   0x04A0
 *  CMP9_SPARE  CMP9 Reserved Register   0x04A4
 *  CMP10_SPARE CMP10 Reserved Register  0x04A8
 *  CMP11_SPARE CMP11 Reserved Register  0x04AC
 *  CMP12_SPARE CMP12 Reserved Register  0x04B0
 *
 *  Register Field              Field Range
 *  cmp1_spare                  31:0
 *
 */
__STATIC_INLINE void ASysCtl_setCmpSpare(CMPIdType_t cmpId, uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP1_SPARE + cmpId * 4) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getCmpSpare(CMPIdType_t cmpId)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP1_SPARE + cmpId * 4);
}


/*
 *  CMP_TOP_SPARE   CMP Top Reserved Register    0x04B4
 *
 *  Register Field              Field Range
 *  cmp1_spare                  31:0
 *
 */
__STATIC_INLINE void ASysCtl_setCmpTopSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_TOP_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getCmpTopSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_TOP_SPARE);
}

/*
 *  DAC_TOP_EN  DAC_TOP_EN Configuration Register  0x04B8
 *
 *  Register Field              Field Range
 *  reserved                    31:1
 *  dac_top_en                  0
 *
 */
__STATIC_INLINE void ASysCtl_setDacTopEnable(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DAC_TOP_EN) = value & 0x01;
}

__STATIC_INLINE uint32_t ASysCtl_getDacTopEnable(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DAC_TOP_EN) & 0x01;
}

/*
 *  DAC_RST         DAC_RST Configuration Register    0x04BC
 *
 *  Register Field      Field Range
 *  reserved            31:1
 *  dac_rst             0
 *
 */
__STATIC_INLINE void ASysCtl_setDacReset(uint32_t rst)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DAC_RST) = rst & 0x01;
}

__STATIC_INLINE uint32_t ASysCtl_getDacReset(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DAC_RST) & 0x01;
}

__STATIC_INLINE void ASysCtl_resetDac(void)
{
    ASysCtl_setDacReset(1);
    ASysCtl_setDacReset(0);
}


/*
 *  DACA_BUF_TRIM   DACA_BUF Trim Value  0x04C0
 *
 *  Register Field      Field Range
 *  reserved            31:1
 *  daca_buf_trim       0
 *
 */
__STATIC_INLINE void ASysCtl_setDacaBufTrim(uint32_t trim)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_BUF_TRIM) = trim & 0x3F;
}

__STATIC_INLINE uint32_t ASysCtl_getDacaBufTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_BUF_TRIM) & 0x3F;
}


/*
 *  DACB_BUF_TRIM   DACB_BUF Trim Value  0x04C4
 *
 *  Register Field      Field Range
 *  reserved            31:1
 *  dacb_buf_trim       0
 *
 */
__STATIC_INLINE void ASysCtl_setDacbBufTrim(uint32_t trim)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_BUF_TRIM) = trim & 0x3F;
}

__STATIC_INLINE uint32_t ASysCtl_getDacbBufTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_BUF_TRIM) & 0x3F;
}

/*
 *  DACC_BUF_TRIM   DACC_BUF Trim Value  0x04C8
 *
 *  Register Field      Field Range
 *  reserved            31:1
 *  dacc_buf_trim       0
 *
 */
__STATIC_INLINE void ASysCtl_setDaccBufTrim(uint32_t trim)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_BUF_TRIM) = trim & 0x3F;
}

__STATIC_INLINE uint32_t ASysCtl_getDaccBufTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_BUF_TRIM) & 0x3F;
}


/*
 *  DACA_SPARE      DACA Reserved Register  0x04CC
 *
 *  Register Field      Field Range
 *  daca_spare          31:0
 *
 */
__STATIC_INLINE void ASysCtl_setDacaSpare(uint32_t val)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_SPARE) = val;
}

__STATIC_INLINE uint32_t ASysCtl_getDacaSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_SPARE);
}


/*
 *  DACB_SPARE      DACB Reserved Register  0x04D0
 *
 *  Register Field      Field Range
 *  dacb_spare          31:0
 *
 */
__STATIC_INLINE void ASysCtl_setDacbSpare(uint32_t val)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_SPARE) = val;
}

__STATIC_INLINE uint32_t ASysCtl_getDacbSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_SPARE);
}


/*
 *  DACC_SPARE      DACC Reserved Register  0x04D4
 *
 *  Register Field      Field Range
 *  dacc_spare          31:0
 *
 */
__STATIC_INLINE void ASysCtl_setDaccSpare(uint32_t val)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_SPARE) = val;
}

__STATIC_INLINE uint32_t ASysCtl_getDaccSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_SPARE);
}


/*
 *  DAC_TOP_SPARE   DAC TOP Reserved Register    0x04D8
 *
 *  Register Field      Field Range
 *  dacc_spare          31:0
 *
 */
__STATIC_INLINE void ASysCtl_setDacTopSpare(uint32_t val)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DAC_TOP_SPARE) = val;
}

__STATIC_INLINE uint32_t ASysCtl_getDacTopSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DAC_TOP_SPARE);
}

/*
 *  TMUX   TMUX Configuration Register  offset  0x04DC
 *
 *  Register Field       Field Range
 *  tmux                 31:0
 *
 */
__STATIC_INLINE void ASysCtl_setTmux(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_TMUX) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getTmux(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_TMUX) >> 0);
}


/*
 *  V2I_A   V2I_A Configuration Register  offset  0x4E0
 *
 *  Register Field       Field Range
 *  v2i_a                31:0
 *
 */
__STATIC_INLINE void ASysCtl_setV2iA(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_V2I_A) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getV2iA(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_V2I_A) >> 0);
}


/*
 *  V2I_B   V2I_B Configuration Register  offset  0x4E4
 *
 *  Register Field       Field Range
 *  v2i_b                31:0
 *
 */
__STATIC_INLINE void ASysCtl_setV2iB(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_V2I_B) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getV2iB(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_V2I_B) >> 0);
}


/*
 *  BG_REG   Band Gap Configuration Register  offset  0x4E8
 *
 *  Register Field   Field Range
 *  bg_meas          3
 *  reserved         2:1
 *  bg_chop_en       0
 *
 */
__STATIC_INLINE void ASysCtl_setBgMeas(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) & (~(1 << 3))) | ((value & 0x01) << 3));
}

__STATIC_INLINE uint32_t ASysCtl_getBgMeas(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) >> 3) & 0x01;
}

__STATIC_INLINE void ASysCtl_setBgChopEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) & (~(1 << 0))) | ((value & 0x01) << 0));
}

__STATIC_INLINE uint32_t ASysCtl_getBgChopEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) >> 0) & 0x01;
}


/*
 *  ANA_CLK_TRIM   Analog Clock Trim Configuration Register  offset  0x4EC
 *
 *  Register Field   Field Range
 *  reserved         31:16
 *  ana_clk_trim     15:0
 *
 */
__STATIC_INLINE void ASysCtl_setAnaClkTrim(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CLK_TRIM) = value & 0xFFFF;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaClkTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CLK_TRIM) & 0xFFFF;
}


/*
 *  ANA_CLK_SPARE   Analog Clock Spare Configuration Register  offset  0x4F0
 *
 *  Register Field   Field Range
 *  ana_clk_spare    31:16
 *
 */
__STATIC_INLINE void ASysCtl_setAnaClkSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CLK_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaClkSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_CLK_SPARE);
}


/*
 *  ANA_TS_CHOP_EN      Analog Touchscreen Chop Enable Configuration Register     0x504
 *
 *  Register Field   Field Range
 *  reserved         31:1
 *  ana_ts_chop_en   0
 *
 */
__STATIC_INLINE void ASysCtl_setAnaTsChopEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG) = (enable & 0x01);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTsChopEn(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_BG_REG);
}


/*
 *  ANA_TS_MEAS_EN  Analog Touchscreen Measurement Enable Configuration Register  0x508
 *
 *  Register Field   Field Range
 *  reserved         31:1
 *  ana_ts_meas_en   0
 *
 */
__STATIC_INLINE void ASysCtl_setAnaTsMeasEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_MEAS_EN) = (enable & 0x01);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTsMeasEn(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_MEAS_EN);
}


/*
 *  ANA_TS_SPARE    Analog Touchscreen Spare Configuration Register   0x510
 *
 *  Register Field   Field Range
 *  reserved         31:10
 *  ana_ts_spare     9:0
 *
 */
__STATIC_INLINE void ASysCtl_setAnaTsSpare(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_SPARE) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTsSpare(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_SPARE);
}


/*
 *  ANA_TEST_MUX_SEL    Analog Test Mux Selection Configuration Register   0x524
 *
 *  Register Field    Field Range
 *  reserved          31:16
 *  ana_test_mux3_sel 15:12
 *  ana_test_mux2_sel 11:8
 *  ana_test_mux1_sel 7:4
 *  ana_test_mux0_sel 3:0
 *
 */
__STATIC_INLINE void ASysCtl_setAnaTestMux3Sel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL)) & (~(0x0F << 12))) | ((value & 0x0F) << 12);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux3Sel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) >> 12) & 0xF;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux2Sel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL)) & (~(0x0F << 8))) | ((value & 0x0F) << 8);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux2Sel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) >> 8) & 0xF;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux1Sel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL)) & (~(0x0F << 4))) | ((value & 0x0F) << 4);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux1Sel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) >> 4) & 0xF;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux0Sel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) =
            ((HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) & (~(0x0F << 0))) | ((value & 0x0F) << 0));
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux0Sel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_SEL) >> 0) & 0xF;
}


/*
 *  ANA_TEST_MUX_EN     Analog Test Mux Enable Configuration Register    0x528
 *
 *  Register Field          Field Range
 *  reserved                31:8
 *  ana_test_mux3_mode      7
 *  ana_test_mux2_mode      6
 *  ana_test_mux1_mode      5
 *  ana_test_mux0_mode      4
 *  ana_test_mux3_en        3
 *  ana_test_mux2_en        2
 *  ana_test_mux1_en        1
 *  ana_test_mux0_en        0
 *
 */
/* --- test mux en --- */
__STATIC_INLINE void ASysCtl_setAnaTestMux0En(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) & (~(1<<0))) | (enable & 0x01);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux0En(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux1En(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) & (~(1<<1))) | ((enable & 0x01) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux1En(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) >> 1) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux2En(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) & (~(1<<2))) | ((enable & 0x01) << 2);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux2En(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) >> 2) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux3En(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) & (~(1<<3))) | ((enable & 0x01) << 3);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux3En(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) >> 3) & 0x01;
}

/* --- test mux mode --- */
__STATIC_INLINE void ASysCtl_setAnaTestMux0Mode(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) & (~(1<<4))) | ((enable & 0x01) << 4);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux0Mode(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) >> 4) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux1Mode(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) & (~(1<<5))) | ((enable & 0x01) << 5);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux1Mode(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) >> 5) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux2Mode(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) & (~(1<<6))) | ((enable & 0x01) << 6);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux2Mode(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) >> 6) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAnaTestMux3Mode(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) & (~(1<<7))) | ((enable & 0x01) << 7);
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTestMux3Mode(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TEST_MUX_EN) >> 7) & 0x01;
}


/*
 *  LDO_ANA_TEST_MODE   LDO_ANA_TEST_MODE Configuration Register   offset  0x52C
 *
 *  Register Field         Field Range
 *  reserved               31:2
 *  ldo_ana_test_mode      1:0
 */

__STATIC_INLINE void ASysCtl_setLdoAnaTestMode(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_LDO_ANA_TEST_MODE) = value & 0x03;
}

__STATIC_INLINE uint32_t ASysCtl_getLdoAnaTestMode(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_LDO_ANA_TEST_MODE) & 0x3;
}


/*
 *  LDO_ANA_TEST_EN   LDO_ANA_TEST_EN Configuration Register   offset  0x530
 *
 *  Register Field         Field Range
 *  reserved               31:1
 *  ldo_ana_test_en        0
 */
__STATIC_INLINE void ASysCtl_setLdoAnaTestEn(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_LDO_ANA_TEST_EN) = value & 0x01;
}

__STATIC_INLINE uint32_t ASysCtl_getLdoAnaTestEn(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_LDO_ANA_TEST_EN) & 0x1;
}


/*
 *  LDO_ANA_VSEL   LDO_ANA_VSEL Configuration Register   offset  0x534
 *
 *  Register Field     Field Range
 *  reserved           31:6
 *  ldo_ana_vsel       5:0
 */
__STATIC_INLINE void ASysCtl_setLdoAnaVsel(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_LDO_ANA_VSEL) = value & 0x3F;
}

__STATIC_INLINE uint32_t ASysCtl_getLdoAnaVsel(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_LDO_ANA_VSEL) & 0x3F;
}


/*
 *  LDO_ANA_FASTON   LDO_ANA_FASTON Configuration Register   offset  0x538
 *
 *  Register Field     Field Range
 *  reserved           31:1
 *  ldo_ana_faston     0
 */
__STATIC_INLINE void ASysCtl_setLdoAnaFaston(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_LDO_ANA_FASTON) = enable & 0x01;
}

__STATIC_INLINE uint32_t ASysCtl_getLdoAnaFaston(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_LDO_ANA_FASTON) & 0x01;
}


/*
 *  DACA_ANA_CFG    DACA_ANA_CFG Configuration Register   offset  0x53C
 *
 *  Register Field     Field Range
 *  reserved           31:3
 *  daca_vref_en       2
 *  daca_ana_ref_2p5   1
 *  daca_ana_ref_sel   0
 */
__STATIC_INLINE void ASysCtl_setDacaAnaRefSel(uint32_t ref)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_ANA_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_ANA_CFG) & (~(1<<0))) | (ref & 0x01);
}

__STATIC_INLINE uint32_t ASysCtl_getDacaAnaRefSel(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_ANA_CFG) & 0x01;
}

__STATIC_INLINE void ASysCtl_setDacaAnaRef2p5(uint32_t ref)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_ANA_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_ANA_CFG) & (~(1<<1))) | ((ref & 0x01) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getDacaAnaRef2p5(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_ANA_CFG) >> 1) & 0x01;
}

__STATIC_INLINE void ASysCtl_setDacaVrefEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_ANA_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_ANA_CFG) & (~(1<<2))) | ((enable & 0x01) << 2);
}

__STATIC_INLINE uint32_t ASysCtl_getDacaVrefEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_ANA_CFG) >> 2) & 0x01;
}


/*
 *  DACA_INL_OFF_DLY_TRIM   DACA_INL_OFF_DLY_TRIM Configuration Register   offset  0x540
 *
 *  Register Field     Field Range
 *  daca_inl_trim      31:16
 *  reserved           15:8
 *  daca_dly_trim      7:0
 *
 */
__STATIC_INLINE void ASysCtl_setDacaDlyTrim(uint32_t dly)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_INL_OFF_DLY_TRIM) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_INL_OFF_DLY_TRIM) & (~(0xFF << 0))) | (dly & 0xFF);
}

__STATIC_INLINE uint32_t ASysCtl_getDacaDlyTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_INL_OFF_DLY_TRIM) & 0xFF;
}

__STATIC_INLINE void ASysCtl_setDacaInlTrim(uint32_t inl)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_INL_OFF_DLY_TRIM) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_INL_OFF_DLY_TRIM) & (~(0xFFFF << 16))) | ((inl & 0xFFFF) << 16);
}

__STATIC_INLINE uint32_t ASysCtl_getDacaInlTrim(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_INL_OFF_DLY_TRIM) >> 16) & 0xFFFF;
}


/*
 *  DACB_ANA_CFG    DACB_ANA_CFG Configuration Register   offset  0x544
 *
 *  Register Field     Field Range
 *  reserved           31:3
 *  dacb_vref_en       2
 *  dacb_ana_ref_2p5   1
 *  dacb_ana_ref_sel   0
 */
__STATIC_INLINE void ASysCtl_setDacbAnaRefSel(uint32_t ref)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_ANA_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_ANA_CFG) & (~(1<<0))) | (ref & 0x01);
}

__STATIC_INLINE uint32_t ASysCtl_getDacbAnaRefSel(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_ANA_CFG) & 0x01;
}

__STATIC_INLINE void ASysCtl_setDacbAnaRef2p5(uint32_t ref)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_ANA_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_ANA_CFG) & (~(1<<1))) | ((ref & 0x01) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getDacbAnaRef2p5(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_ANA_CFG) >> 1) & 0x01;
}

__STATIC_INLINE void ASysCtl_setDacbVrefEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_ANA_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_ANA_CFG) & (~(1<<2))) | ((enable & 0x01) << 2);
}

__STATIC_INLINE uint32_t ASysCtl_getDacbVrefEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_ANA_CFG) >> 2) & 0x01;
}


/*
 *  DACB_INL_OFF_DLY_TRIM   DACB_INL_OFF_DLY_TRIM Configuration Register   offset  0x548
 *
 *  Register Field     Field Range
 *  dacb_inl_trim      31:16
 *  reserved           15:8
 *  dacb_dly_trim      7:0
 */
__STATIC_INLINE void ASysCtl_setDacbDlyTrim(uint32_t dly)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_INL_OFF_DLY_TRIM) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_INL_OFF_DLY_TRIM) & (~(0xFF<<0))) | (dly & 0xFF);
}

__STATIC_INLINE uint32_t ASysCtl_getDacbDlyTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_INL_OFF_DLY_TRIM) & 0xFF;
}

__STATIC_INLINE void ASysCtl_setDacbInlTrim(uint32_t inl)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_INL_OFF_DLY_TRIM) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_INL_OFF_DLY_TRIM) & (~(0xFFFF<<16))) | ((inl & 0xFFFF) << 16);
}

__STATIC_INLINE uint32_t ASysCtl_getDacbInlTrim(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_INL_OFF_DLY_TRIM) >> 16) & 0xFFFF;
}


/*
 *  DACC_ANA_CFG    DACC_ANA_CFG Configuration Register   offset  0x54C
 *
 *  Register Field     Field Range
 *  reserved           31:3
 *  dacc_vref_en       2
 *  dacc_ana_ref_2p5   1
 *  dacc_ana_ref_sel   0
 */
__STATIC_INLINE void ASysCtl_setDaccAnaRefSel(uint32_t ref)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_ANA_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_ANA_CFG) & (~(1<<0))) | (ref & 0x01);
}

__STATIC_INLINE uint32_t ASysCtl_getDaccAnaRefSel(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_ANA_CFG) & 0x01;
}

__STATIC_INLINE void ASysCtl_setDaccAnaRef2p5(uint32_t ref)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_ANA_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_ANA_CFG) & (~(1<<1))) | ((ref & 0x01) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getDaccAnaRef2p5(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_ANA_CFG) >> 1) & 0x01;
}

__STATIC_INLINE void ASysCtl_setDaccVrefEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_ANA_CFG) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_ANA_CFG) & (~(1<<2))) | ((enable & 0x01) << 2);
}

__STATIC_INLINE uint32_t ASysCtl_getDaccVrefEn(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_ANA_CFG) >> 2) & 0x01;
}


/*
 *  DACC_INL_OFF_DLY_TRIM   DACC_INL_OFF_DLY_TRIM Configuration Register   offset  0x550
 *
 *  Register Field     Field Range
 *  dacc_inl_trim      31:16
 *  reserved           15:8
 *  dacc_dly_trim      7:0
 */
__STATIC_INLINE void ASysCtl_setDaccDlyTrim(uint32_t dly)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_INL_OFF_DLY_TRIM) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_INL_OFF_DLY_TRIM) & (~(0xFF<<0))) | (dly & 0xFF);
}

__STATIC_INLINE uint32_t ASysCtl_getDaccDlyTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_INL_OFF_DLY_TRIM) & 0xFF;
}

__STATIC_INLINE void ASysCtl_setDaccInlTrim(uint32_t inl)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_INL_OFF_DLY_TRIM) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_INL_OFF_DLY_TRIM) & (~(0xFFFF<<16))) | ((inl & 0xFFFF) << 16);
}

__STATIC_INLINE uint32_t ASysCtl_getDaccInlTrim(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_INL_OFF_DLY_TRIM) >> 16) & 0xFFFF;
}


/*
 *  ANA_TOP_SPARE1  ANA_TOP_SPARE1 Configuration Register   offset  0x554
 *
 *  Register Field     Field Range
 *  ana_top_spare1     31:0
 *
 */

__STATIC_INLINE void ASysCtl_setAnaTopSpare1(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_SPARE1) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTopSpare1(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_SPARE1);
}


/*
 *  ANA_TOP_SPARE2  ANA_TOP_SPARE2 Configuration Register   offset  0x558
 *
 *  Register Field     Field Range
 *  ana_top_spare2     31:0
 *
 */

__STATIC_INLINE void ASysCtl_setAnaTopSpare2(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_SPARE2) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTopSpare2(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_SPARE2);
}


/*
 *  ANA_TOP_SPARE3  ANA_TOP_SPARE3 Configuration Register   offset  0x55C
 *
 *  Register Field     Field Range
 *  ana_top_spare3     31:0
 *
 */

__STATIC_INLINE void ASysCtl_setAnaTopSpare3(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_SPARE3) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTopSpare3(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_SPARE3);
}


/*
 *  ANA_TOP_SPARE4  ANA_TOP_SPARE4 Configuration Register   offset  0x560
 *
 *  Register Field     Field Range
 *  ana_top_spare4     31:0
 *
 */

__STATIC_INLINE void ASysCtl_setAnaTopSpare4(uint32_t value)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_SPARE4) = value;
}

__STATIC_INLINE uint32_t ASysCtl_getAnaTopSpare4(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TOP_SPARE4);
}


/*
 *  ADCA_BUFDAC2ADC_LBEN    ADCA_BUFDAC2ADC_LBEN Configuration Register   0x564
 *
 *  Register Field         Field Range
 *  reserved               31:1
 *  adca_bufdac2adc_lben   0
 * 
 */
__STATIC_INLINE void ASysCtl_setAdcaBufDac2AdcLben(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_BUFDAC2ADC_LBEN) = enable & 0x01;
}

__STATIC_INLINE uint32_t ASysCtl_getAdcaBufDac2AdcLben(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_BUFDAC2ADC_LBEN) & 0x01;
}


/*
 *  ADCB_BUFDAC2ADC_LBEN    ADCB_BUFDAC2ADC_LBEN Configuration Register   0x568
 *
 *  Register Field         Field Range
 *  reserved               31:1
 *  adcb_bufdac2adc_lben   0
 *
 */
__STATIC_INLINE void ASysCtl_setAdcbBufDac2AdcLben(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_BUFDAC2ADC_LBEN) = enable & 0x01;
}

__STATIC_INLINE uint32_t ASysCtl_getAdcbBufDac2AdcLben(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_BUFDAC2ADC_LBEN) & 0x01;
}


/*
 *  ADCC_BUFDAC2ADC_LBEN    ADCC_BUFDAC2ADC_LBEN Configuration Register   0x56C
 *
 *  Register Field         Field Range
 *  reserved               31:1
 *  adcc_bufdac2adc_lben   0
 *
 */
__STATIC_INLINE void ASysCtl_setAdccBufDac2AdcLben(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_BUFDAC2ADC_LBEN) = enable & 0x01;
}

__STATIC_INLINE uint32_t ASysCtl_getAdccBufDac2AdcLben(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_BUFDAC2ADC_LBEN) & 0x01;
}


/*
 *  ADCD_BUFDAC2ADC_LBEN    ADCD_BUFDAC2ADC_LBEN Configuration Register   0x570
 *
 *  Register Field         Field Range
 *  reserved               31:1
 *  adcd_bufdac2adc_lben   0
 *
 */
__STATIC_INLINE void ASysCtl_setAdcdBufDac2AdcLben(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_BUFDAC2ADC_LBEN) = enable & 0x01;
}

__STATIC_INLINE uint32_t ASysCtl_getAdcdBufDac2AdcLben(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_BUFDAC2ADC_LBEN) & 0x01;
}


/*
 *  ADC_L2H_CFG         ADC_L2H_CFG Configuration Register   0x574
 *
 *  Register Field   Field Range
 *  reserved         31:1
 *  adc_l2h_en       0
 *
 */
__STATIC_INLINE void ASysCtl_setL2HCfg(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_L2H_CFG) = enable & 0x01;
}

__STATIC_INLINE uint32_t ASysCtl_getAdcL2hEn(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADC_L2H_CFG) & 0x01;
}


/*
 *  ADCA_CH_MAP_CH_SEL  ADCA_CH_MAP_CH_SEL Configuration Register   0x578
 *
 *  Register Field          Field Range
 *  reserved                31:10
 *  adca_ch19_map_ch_sel    9:5
 *  adca_ch18_map_ch_sel    4:0
 *
 */
__STATIC_INLINE void ASysCtl_setAdcaCh18MapChSel(uint32_t ch)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_CH_MAP_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_CH_MAP_CH_SEL) & (~0x1F)) | (ch & 0x1F);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcaCh18MapChSel(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_CH_MAP_CH_SEL) & 0x01F;
}

__STATIC_INLINE void ASysCtl_setAdcaCh19MapChSel(uint32_t ch)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_CH_MAP_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_CH_MAP_CH_SEL) & (~(0x1F << 5))) | ((ch & 0x1F) << 5);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcaCh19MapChSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_CH_MAP_CH_SEL) >> 5) & 0x01F;
}


/*
 *  ADCB_CH_MAP_CH_SEL  ADCB_CH_MAP_CH_SEL Configuration Register   0x57C
 *
 *  Register Field          Field Range
 *  reserved                31:10
 *  adcb_ch19_map_ch_sel    9:5
 *  adcb_ch18_map_ch_sel    4:0
 *
 */
__STATIC_INLINE void ASysCtl_setAdcbCh18MapChSel(uint32_t ch)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_CH_MAP_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_CH_MAP_CH_SEL) & (~0x1F)) | (ch & 0x1F);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcbCh18MapChSel(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_CH_MAP_CH_SEL) & 0x01F;
}

__STATIC_INLINE void ASysCtl_setAdcbCh19MapChSel(uint32_t ch)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_CH_MAP_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_CH_MAP_CH_SEL) & (~(0x1F << 5))) | ((ch & 0x1F) << 5);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcbCh19MapChSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_CH_MAP_CH_SEL) >> 5) & 0x01F;
}


/*
 *  ADCC_CH_MAP_CH_SEL  ADCC_CH_MAP_CH_SEL Configuration Register   0x580
 *
 *  Register Field          Field Range
 *  reserved                31:10
 *  adcc_ch19_map_ch_sel    9:5
 *  adcc_ch18_map_ch_sel    4:0
 *
 */
__STATIC_INLINE void ASysCtl_setAdccCh18MapChSel(uint32_t ch)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_CH_MAP_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_CH_MAP_CH_SEL) & (~0x1F)) | (ch & 0x1F);
}

__STATIC_INLINE uint32_t ASysCtl_getAdccCh18MapChSel(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_CH_MAP_CH_SEL) & 0x01F;
}

__STATIC_INLINE void ASysCtl_setAdccCh19MapChSel(uint32_t ch)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_CH_MAP_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_CH_MAP_CH_SEL) & (~(0x1F << 5))) | ((ch & 0x1F) << 5);
}

__STATIC_INLINE uint32_t ASysCtl_getAdccCh19MapChSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_CH_MAP_CH_SEL) >> 5) & 0x01F;
}

/*
 *  ADCD_CH_MAP_CH_SEL  ADCD_CH_MAP_CH_SEL Configuration Register   0x584
 *
 *  Register Field          Field Range
 *  reserved                31:10
 *  adcd_ch19_map_ch_sel    9:5
 *  adcd_ch18_map_ch_sel    4:0
 *
 */
__STATIC_INLINE void ASysCtl_setAdcdCh18MapChSel(uint32_t ch)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_CH_MAP_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_CH_MAP_CH_SEL) & (~0x1F)) | (ch & 0x1F);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcdCh18MapChSel(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_CH_MAP_CH_SEL) & 0x01F;
}

__STATIC_INLINE void ASysCtl_setAdcdCh19MapChSel(uint32_t ch)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_CH_MAP_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_CH_MAP_CH_SEL) & (~(0x1F << 5))) | ((ch & 0x1F) << 5);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcdCh19MapChSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_CH_MAP_CH_SEL) >> 5) & 0x01F;
}


/*
 *  ADCA_FRC_CH_SEL         ADCA_FRC_CH_SEL Configuration Register  0x588
 *
 *  Register Field          Field Range
 *  reserved                31:6
 *  adca_frc_ch_sel         5:1
 *  adca_frc_ch_sel_en      0
 *
 */
__STATIC_INLINE void ASysCtl_setAdcaRfcChSelEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_FRC_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_FRC_CH_SEL) & (~0x1)) | (enable & 0x1);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcaRfcChSelEn(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_FRC_CH_SEL) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAdcaRfcChSel(uint32_t ch)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_FRC_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_FRC_CH_SEL) & (~(0xF << 1))) | ((ch & 0xF) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcaRfcChSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCA_FRC_CH_SEL) >> 1) & 0x0F;
}


/*
 *  ADCB_FRC_CH_SEL         ADCB_FRC_CH_SEL Configuration Register  0x58C
 *
 *  Register Field          Field Range
 *  reserved                31:6
 *  adcb_frc_ch_sel         5:1
 *  adcb_frc_ch_sel_en      0
 *
 */
__STATIC_INLINE void ASysCtl_setAdcbRfcChSelEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_FRC_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_FRC_CH_SEL) & (~0x1)) | (enable & 0x1);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcbRfcChSelEn(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_FRC_CH_SEL) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAdcbRfcChSel(uint32_t ch)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_FRC_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_FRC_CH_SEL) & (~(0xF << 1))) | ((ch & 0xF) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcbRfcChSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCB_FRC_CH_SEL) >> 1) & 0x0F;
}


/*
 *  ADCC_FRC_CH_SEL         ADCC_FRC_CH_SEL Configuration Register  0x590
 *
 *  Register Field          Field Range
 *  reserved                31:6
 *  adcc_frc_ch_sel         5:1
 *  adcc_frc_ch_sel_en      0
 *
 */
__STATIC_INLINE void ASysCtl_setAdccRfcChSelEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_FRC_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_FRC_CH_SEL) & (~0x1)) | (enable & 0x1);
}

__STATIC_INLINE uint32_t ASysCtl_getAdccRfcChSelEn(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_FRC_CH_SEL) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAdccRfcChSel(uint32_t ch)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_FRC_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_FRC_CH_SEL) & (~(0xF << 1))) | ((ch & 0xF) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getAdccRfcChSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCC_FRC_CH_SEL) >> 1) & 0x0F;
}


/*
 *  ADCD_FRC_CH_SEL         ADCD_FRC_CH_SEL Configuration Register  0x594
 *
 *  Register Field          Field Range
 *  reserved                31:6
 *  adcd_frc_ch_sel         5:1
 *  adcd_frc_ch_sel_en      0
 *
 */
__STATIC_INLINE void ASysCtl_setAdcdRfcChSelEn(uint32_t enable)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_FRC_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_FRC_CH_SEL) & (~0x1)) | (enable & 0x1);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcdRfcChSelEn(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_FRC_CH_SEL) & 0x01;
}

__STATIC_INLINE void ASysCtl_setAdcdRfcChSel(uint32_t ch)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_FRC_CH_SEL) =
            (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_FRC_CH_SEL) & (~(0xF << 1))) | ((ch & 0xF) << 1);
}

__STATIC_INLINE uint32_t ASysCtl_getAdcdRfcChSel(void)
{
    return (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ADCD_FRC_CH_SEL) >> 1) & 0x0F;
}


/*
 *  DACA_TRIM               DACA_TRIM Configuration Register  0x598
 *
 *  Register Field          Field Range
 *  daca_trim               31:0
 *
 */
__STATIC_INLINE void ASysCtl_setDacaTrim(uint32_t trim)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_TRIM) = trim;
}

__STATIC_INLINE uint32_t ASysCtl_getDacaTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACA_TRIM);
}

/*
 *  DACB_TRIM               DACB_TRIM Configuration Register  0x59C
 *
 *  Register Field          Field Range
 *  dacb_trim               31:0
 *
 */
__STATIC_INLINE void ASysCtl_setDacbTrim(uint32_t trim)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_TRIM) = trim;
}

__STATIC_INLINE uint32_t ASysCtl_getDacbTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACB_TRIM);
}

/*
 *  DACC_TRIM               DACC_TRIM Configuration Register  0x5A0
 *
 *  Register Field          Field Range
 *  dacc_trim               31:0
 *
 */
__STATIC_INLINE void ASysCtl_setAdccTrim(uint32_t trim)
{
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_TRIM) = trim;
}

__STATIC_INLINE uint32_t ASysCtl_getAdccTrim(void)
{
    return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_DACC_TRIM);
}

/**
 * 
 *  Set the analog voltage reference selection to internal.
 * 
 *  @param [in] reference is the analog reference.
 * 
 *  The parameter  reference can be a combination of the following values:
 * 
 *  -  ASYSCTL_VREFHI
 * 
 *  @return None.
 * 
 */
static inline void
ASysCtl_setAnalogReferenceInternal(uint16_t reference)
{
    ASSERT((reference & (ASYSCTL_VREFHIA | ASYSCTL_VREFHIB | ASYSCTL_VREFHIC | ASYSCTL_VREFHID)) == reference);

    //
    // Write selection to the Analog Internal Reference Select bit.
    //

    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= reference;


}

/**
 * 
 *  Set the analog voltage reference selection to external.
 * 
 *  @param [in] reference is the analog reference.
 * 
 *  The parameter  reference can be a combination of the following values:
 * 
 *  -  ASYSCTL_VREFHI
 * 
 *  @return None.
 * 
 */
static inline void
ASysCtl_setAnalogReferenceExternal(uint16_t reference)
{
    ASSERT((reference & (ASYSCTL_VREFHIA | ASYSCTL_VREFHIB | ASYSCTL_VREFHIC | ASYSCTL_VREFHID)) == reference);

//    ASysCtl_setAdcSpare(0, 0x06625757); //bit21:24 VCM, bit10:13 VREFHI
//    ASysCtl_setAdcSpare(1, 0x06625757);
//    ASysCtl_setAdcSpare(2, 0x06625757);
    //TODO get external 3.3V top spare replace above

    //
    // Write selection to the Analog External Reference Select bit.
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &=~ (reference << 0); 					//xF:internal 0x0:external


}

/**
 * 
 *  Set the internal analog voltage reference selection to 1.65V.
 * 
 *  @param [in] reference is the analog reference.
 * 
 *  The parameter  reference can be a combination of the following values:
 * 
 *  -  ASYSCTL_VREFHI
 * 
 *  @return None.
 * 
 */
static inline void ASysCtl_setAnalogReference1P65(uint16_t reference)
{
    ASSERT((reference & (ASYSCTL_VREFHIA | ASYSCTL_VREFHIB | ASYSCTL_VREFHIC | ASYSCTL_VREFHID)) == reference);

    if(HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL))//for internal
    {
        ASysCtl_setAdcSpare(0, 0x1662D717);
        ASysCtl_setAdcSpare(1, 0x1662D717);
        ASysCtl_setAdcSpare(2, 0x1662D717);
    }


    //
    // Write selection to the Analog Voltage Reference Select bit.
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) &= ~(reference << 8U);

}

/**
 * 
 *  Set the internal analog voltage reference selection to 2.5V.
 * 
 *  @param [in] reference is the analog reference.
 * 
 *  The parameter  reference can be a combination of the following values:
 * 
 *  -  ASYSCTL_VREFHI
 * 
 *  @return None.
 * 
 */
static inline void ASysCtl_setAnalogReference2P5(uint16_t reference)
{
    ASSERT((reference & (ASYSCTL_VREFHIA | ASYSCTL_VREFHIB | ASYSCTL_VREFHIC | ASYSCTL_VREFHID)) == reference);

    if(HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL))//for internal
    {
        ASysCtl_setAdcSpare(0, 0x1AE1DF57);
        ASysCtl_setAdcSpare(1, 0x1AE1DF57);
        ASysCtl_setAdcSpare(2, 0x1AE1DF57);
    }

    //
    // Write selection to the Analog Voltage Reference Select bit.
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_ANAREFCTL) |= (reference << 8U);

}

__STATIC_INLINE void ASysCtl_setCMPEn(uint32_t enable)
{
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_EN) =(enable & 0x01);
}

__STATIC_INLINE uint32_t ASysCtl_getCMPEn(void)
{
   return HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_EN) & 0X01;
}

//__STATIC_INLINE uint32_t ASysCtl_setCMP_HysteresisMod(uint32_t enable)
//{
//   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_EN) |=0X01;
//}


__STATIC_INLINE void ASysCtl_setCMP_Mux_AllDisable()
{
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP1_SPARE)  = 0x00;
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP2_SPARE)  = 0x00;
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP3_SPARE)  = 0x00;
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP4_SPARE)  = 0x00;
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP5_SPARE)  = 0x00;
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP6_SPARE)  = 0x00;
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP7_SPARE)  = 0x00;
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP8_SPARE)  = 0x00;
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP9_SPARE)  = 0x00;
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP10_SPARE) = 0x00;
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP11_SPARE) = 0x00;
   HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP12_SPARE) = 0x00;
}


__STATIC_INLINE void ASysCtl_setCMP_MuxEn(ASysCtl_CMPSelect select,uint8_t hp_en,uint8_t hn_en,uint8_t lp_en,uint8_t ln_en)
{
	uint16_t value = (hn_en  & 0X01) | ((hp_en  & 0X01)<<1) | ((ln_en  & 0X01)<<2) | ((lp_en  & 0X01)<<3);

   HWREG(ANALOGSUBSYS_BASE + select) = (HWREG(ANALOGSUBSYS_BASE + select) &~(0xf)) | value;
}

/**
 * 
 *  Select the value for CMPHNMXSEL.
 * 
 *  @param [in] select is a combination of CMPHNMXSEL values.
 * 
 *  The parameter  select can be a bitwise OR of the below values:
 * 
 *  -  ASYSCTL_CMPHNMUX_SELECT_1
 *  -  ASYSCTL_CMPHNMUX_SELECT_2
 *  -  ASYSCTL_CMPHNMUX_SELECT_3
 *  -  ASYSCTL_CMPHNMUX_SELECT_4
 * 
 *  @return None.
 * 
 */
static inline void ASysCtl_selectCMPHNMux(uint16_t select)
{
    //
    // Write a select to the mux select bit.
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) = select;

}

/**
 * 
 *  Select the value for individual CMPxHNMXSEL.
 * 
 *  @param [in] select is the CMPxHNMXSEL to be set.
 *  @param [in] value is 0 or 1.
 * 
 *  The parameter  select can be one of the below values:
 * 
 *  -  ASYSCTL_CMPHNMUX_SELECT_1
 *  -  ASYSCTL_CMPHNMUX_SELECT_2
 *  -  ASYSCTL_CMPHNMUX_SELECT_3
 *  -  ASYSCTL_CMPHNMUX_SELECT_4
 * 
 *  @return None.
 * 
 */
static inline void ASysCtl_selectCMPHNMuxValue(uint16_t select, uint16_t value)
{
    //
    // Write a select to the mux select bit.
    //
    if(value == 0U)
    {
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) &= ~select;
    }
    else if(value == 1U)
    {
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) |= select;
    }
    else
    {
        //Do nothing. Not a valid value
    }


}

/**
 * 
 *  Select the value for individual CMPxLNMXSEL.
 * 
 *  @param [in] select is a combination of CMPLNMXSEL values.
 * 
 *  The parameter  select can be the bitwise OR of the below values:
 * 
 *  -  ASYSCTL_CMPLNMUX_SELECT_1
 *  -  ASYSCTL_CMPLNMUX_SELECT_2
 *  -  ASYSCTL_CMPLNMUX_SELECT_3
 *  -  ASYSCTL_CMPLNMUX_SELECT_4
 * 
 *  @return None.
 * 
 */
static inline void ASysCtl_selectCMPLNMux(uint16_t select)
{

    //
    // Write a select to the mux select bit.
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLNMXSEL) = select;


}

/**
 * 
 *  Select the value for CMPLNMXSEL.
 * 
 *  @param [in] select is the CMPxHNMXSEL to be set.
 *  @param [in] value is 0 or 1.
 * 
 *  The parameter  select can be one of the below values:
 * 
 *  -  ASYSCTL_CMPLNMUX_SELECT_1
 *  -  ASYSCTL_CMPLNMUX_SELECT_2
 *  -  ASYSCTL_CMPLNMUX_SELECT_3
 *  -  ASYSCTL_CMPLNMUX_SELECT_4
 * 
 *  @return None.
 * 
 */
static inline void ASysCtl_selectCMPLNMuxValue(uint32_t select, uint16_t value)
{

    select = select<<16;
    //
    // Write a select to the mux select bit.
    //
    if(value == 0U)
    {
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) &= ~select;
    }
    else if(value == 1U)
    {
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHNMXSEL) |= select;
    }
    else
    {
        //Do nothing. Not a valid value
    }

}

/**
 * 
 *  Select the value for CMPHPMXSEL.
 * 
 *  @param [in] select is of type ASysCtl_CMPHPMuxSelect.
 *  @param [in] value is 0, 1, 2, 3, 4 or 5.
 * 
 *  This function is used to write a value to one mux select at a time.
 *  The parameter  select can be one of the following values:
 * 
 *  -  ASYSCTL_CMPHPMUX_SELECT_1
 *  -  ASYSCTL_CMPHPMUX_SELECT_2
 *  -  ASYSCTL_CMPHPMUX_SELECT_3
 *  -  ASYSCTL_CMPHPMUX_SELECT_4
 *  -  ASYSCTL_CMPHPMUX_SELECT_5
 *  -  ASYSCTL_CMPHPMUX_SELECT_6
 *  -  ASYSCTL_CMPHPMUX_SELECT_7
 * 
 *  @return None.
 * 
 */

__STATIC_INLINE void ASysCtl_selectCMPHPMux(ASysCtl_CMPHPMuxSelect select, uint32_t value)
{
//    ASSERT(value <= 5U);
//    EALLOW;
    //
    // Set the value for the appropriate Mux Select.
    //
	if(select <= ASYSCTL_CMPHPMUX_SELECT_10)
	{
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL) =
        (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL) &
        ~((uint32_t)ASYSCTL_CMPHPMXSEL_CMP1HPMXSEL_M << (uint32_t)select)) |
        (value << (uint32_t)select);
	}
	else
	{
		select = (ASysCtl_CMPHPMuxSelect)((uint32_t)select - ASYSCTL_CMPHPMUX_SELECT_11);

        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL1) =
        (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPHPMXSEL1) &
        ~((uint32_t)ASYSCTL_CMPHPMXSEL_CMP1HPMXSEL_M << (uint32_t)select)) |
        (value << (uint32_t)select);

	}
//    EDIS;
}
/**
 * 
 *  Select the value for CMPLPMXSEL.
 * 
 *  @param [in] select is of type ASysCtl_CMPLPMuxSelect.
 *  @param [in] value is 0, 1, 2, 3, 4 or 5.
 * 
 *  This function is used to write a value to one mux select at a time.
 *  The parameter  select can be one of the following values:
 * 
 *  -  ASYSCTL_CMPLPMUX_SELECT_1
 *  -  ASYSCTL_CMPLPMUX_SELECT_2
 *  -  ASYSCTL_CMPLPMUX_SELECT_3
 *  -  ASYSCTL_CMPLPMUX_SELECT_4
 *  -  ASYSCTL_CMPLPMUX_SELECT_5
 *  -  ASYSCTL_CMPLPMUX_SELECT_6
 *  -  ASYSCTL_CMPLPMUX_SELECT_7
 * 
 *  @return None.
 * 
 */
__STATIC_INLINE void ASysCtl_selectCMPLPMux(ASysCtl_CMPLPMuxSelect select, uint32_t value)
{
//    ASSERT(value <= 5U);
//
//    EALLOW;

    //
    // Set the value for the appropriate Mux Select.
    //
	if(select <= ASYSCTL_CMPLPMUX_SELECT_10)
	{
        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL) =
        (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL) &
        ~((uint32_t)ASYSCTL_CMPLPMXSEL_CMP1LPMXSEL_M << (uint32_t)select)) |
        (value << (uint32_t)select);
	}
	else
	{
		select = (ASysCtl_CMPLPMuxSelect)((uint32_t)select - ASYSCTL_CMPLPMUX_SELECT_11);

        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL1) =
        (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMPLPMXSEL1) &
        ~((uint32_t)ASYSCTL_CMPLPMXSEL_CMP1LPMXSEL_M << (uint32_t)select)) |
        (value << (uint32_t)select);

	}

//    EDIS;
}



/**
 * 
 *  Enable temperature sensor.
 * 
 *  This function enables the temperature sensor output to the ADC.
 * 
 *  @return None.
 * 
 */
static inline void
ASysCtl_enableTemperatureSensor(void)
{
    EALLOW;

    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_MEAS_EN) |= (0x1<<0);//ana_ts_meas_en
    HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_TS_SPARE) |= (0x1<<0);//ana_ts_spare
    //
    // Set the temperature sensor enable bit.
    //
    HWREGH(ANALOGSUBSYS_BASE + ANA_CFG_O_TSNSCTL) |= ASYSCTL_TSNSCTL_ENABLE;

    EDIS;
}

void ASysCtl_initConfig(void);
void ASysCtl_trim(void);
void AsysCtl_trimAdc(uint32_t adcBase, uint32_t refMode, uint32_t refVoltage);

#ifdef __cplusplus
}
#endif

#endif

#endif /* DEVICE_DRIVERLIB_SYSCTL_ANA_H_ */
