/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    asysctl_v30.c
*   @brief   digital system top level configuration
*   @details
*
*/
/*
 * commit history
 * 20240306, Jason Yang, Add calling ADC_setINLTrim in ASysCtl_initConfig.
 *                       This is a must to get accurate ADC results.
 * 20240324, Jason Yang, add code to init AnaTop for CMPSS and enable CMPSS.
 */

#include "gs32_version.h"

#if IS_GS32F00xx(0x30)

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "sysctl/asysctl_v30.h"
#include "adc.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
__attribute__((section(".gnu.linkonce.t.trim"))) AnaTrimTable_t anaTrimTable_v1[] = {
    [V1_TRIM_ID_LDO_1V2                          ] = {0x400D6308, REG_WIDTH_32BITS,  4,  0,    22,   0x1FFF4200,  4,  0},     //TOP_LDO            0   31
    [V1_TRIM_ID_ANA_LDO                          ] = {0x400D6534, REG_WIDTH_32BITS,  2,  0,     0,   0x1FFF4210,  2,  0},     //LDO_ANA            0   7
    [V1_TRIM_ID_OSC1                             ] = {0x400D6314, REG_WIDTH_32BITS, 14,  0,  1397,   0x1FFF4230, 14,  0},     //OSC                0   2047
    [V1_TRIM_ID_OSC2_FINE_IN                     ] = {0x400D6314, REG_WIDTH_32BITS, 30, 16,   580,   0x1FFF4230, 30, 16},     //OSC                0   2047
	[V1_TRIM_ID_OSC2_FINE_OUT                    ] = {0x400D6314, REG_WIDTH_32BITS, 30, 16,   580,   0x1FFF4234, 30, 16},     //OSC                0   2047
	[V1_TRIM_ID_OSC2_COARSE_IN					 ] = {0x400D6328, REG_WIDTH_32BITS, 10,  9,     0,   0x1FFF4238, 10, 9},	  //OSC

    //ADCA
	/* ADC ANA Ofsset Direction*/
    [V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P3V_IN      ] = {0x400D6408, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4240,  9,  8},     //ADCA Offset direction    0  3
    [V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_2P5V_IN      ] = {0x400D6408, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4240, 25, 24},     //ADCA Offset direction    0  3
    [V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P3V_OUT     ] = {0x400D6408, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4244,  9,  8},     //ADCA Offset direction    0  3
    [V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_2P5V_OUT     ] = {0x400D6408, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4244, 25, 24},     //ADCA Offset direction    0  3
    [V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P0V_OUT     ] = {0x400D6408, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4248,  9,  8},     //ADCA Offset direction    0  3

	/* ADC VRIFHI */
    [V1_TRIM_ID_VREFHI_COARSE_2P5V               ] = {0x400D6408, REG_WIDTH_32BITS, 13, 10,     7,   0x1FFF4250, 13, 10},     //VREFHI             0   15 //coarse
    [V1_TRIM_ID_VREFHI_FINE_2P5V                 ] = {0x400D6408, REG_WIDTH_32BITS, 18, 15,     3,   0x1FFF4250, 18, 15},     //VREFHI             0   7  //fine
    [V1_TRIM_ID_VREFHI_COARSE_1P65V              ] = {0x400D6408, REG_WIDTH_32BITS, 13, 10,     7,   0x1FFF4254, 13, 10},     //VREFHI             0   15 //coarse
    [V1_TRIM_ID_VREFHI_FINE_1P65V                ] = {0x400D6408, REG_WIDTH_32BITS, 18, 15,     3,   0x1FFF4254, 18, 15},     //VREFHI             0   7  //fine

	/* ADC VCM */
    [V1_TRIM_ID_ADCA_VCM_COARSE_1P65V            ] = {0x400D6408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4260, 24, 21},
    [V1_TRIM_ID_ADCA_VCM_FINE_1P65V              ] = {0x400D6408, REG_WIDTH_32BITS, 29, 26,     3,   0x1FFF4260, 29, 26},
    [V1_TRIM_ID_ADCA_VCM_COARSE_2P475V           ] = {0x400D6408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4264, 24, 21},
    [V1_TRIM_ID_ADCA_VCM_FINE_2P475V             ] = {0x400D6408, REG_WIDTH_32BITS, 29, 26,     3,   0x1FFF4264, 29, 26},
    [V1_TRIM_ID_ADCA_VCM_COARSE_1P25V            ] = {0x400D6408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4268, 24, 21},
    [V1_TRIM_ID_ADCA_VCM_FINE_1P25V              ] = {0x400D6408, REG_WIDTH_32BITS, 29, 26,     3,   0x1FFF4268, 29, 26},
    [V1_TRIM_ID_ADCA_VCM_COARSE_1P5V             ] = {0x400D6408, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF426C, 24, 21},
    [V1_TRIM_ID_ADCA_VCM_FINE_1P5V               ] = {0x400D6408, REG_WIDTH_32BITS, 29, 26,     3,   0x1FFF426C, 29, 26},

	/* ADC ANA Offset Value */
    [V1_TRIM_ID_ADCA_OFFSET_VREFHI_3P3V_IN       ] = {0x40000076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF4270,  7,  0},     //ADCA Offset    0   255
    [V1_TRIM_ID_ADCA_OFFSET_VREFHI_2P5V_IN       ] = {0x40000076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF4272,  7,  0},     //ADCA Offset    0   255
    [V1_TRIM_ID_ADCA_OFFSET_VREFHI_3P3V_OUT      ] = {0x40000076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF4274,  7,  0},     //ADCA Offset    0   255
    [V1_TRIM_ID_ADCA_OFFSET_VREFHI_2P5V_OUT      ] = {0x40000076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF4276,  7,  0},     //ADCA Offset    0   255
    [V1_TRIM_ID_ADCA_OFFSET_VREFHI_3P0V_OUT      ] = {0x40000076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF4278,  7,  0},     //ADCA Offset    0   255

    //ADCC
    [V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P3V_IN      ] = {0x400D6410, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4280,  9,  8},     //ADCC Offset direction    0  3
    [V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_2P5V_IN      ] = {0x400D6410, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4280, 25, 24},     //ADCC Offset direction    0  3
    [V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P3V_OUT     ] = {0x400D6410, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4284,  9,  8},     //ADCC Offset direction    0  3
    [V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_2P5V_OUT     ] = {0x400D6410, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4284, 25, 24},     //ADCC Offset direction    0  3
    [V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P0V_OUT     ] = {0x400D6410, REG_WIDTH_32BITS,  9,  8,     0,   0x1FFF4288,  9,  8},     //ADCC Offset direction    0  3

    [V1_TRIM_ID_ADCC_VCM_COARSE_1P65V            ] = {0x400D6410, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4290, 24, 21},
    [V1_TRIM_ID_ADCC_VCM_FINE_1P65V              ] = {0x400D6410, REG_WIDTH_32BITS, 29, 26,     3,   0x1FFF4290, 29, 26},
    [V1_TRIM_ID_ADCC_VCM_COARSE_2P475V           ] = {0x400D6410, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4294, 24, 21},
    [V1_TRIM_ID_ADCC_VCM_FINE_2P475V             ] = {0x400D6410, REG_WIDTH_32BITS, 29, 26,     3,   0x1FFF4294, 29, 26},
    [V1_TRIM_ID_ADCC_VCM_COARSE_1P25V            ] = {0x400D6410, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF4298, 24, 21},
    [V1_TRIM_ID_ADCC_VCM_FINE_1P25V              ] = {0x400D6410, REG_WIDTH_32BITS, 29, 26,     3,   0x1FFF4298, 29, 26},
    [V1_TRIM_ID_ADCC_VCM_COARSE_1P5V             ] = {0x400D6410, REG_WIDTH_32BITS, 24, 21,     7,   0x1FFF429C, 24, 21},
    [V1_TRIM_ID_ADCC_VCM_FINE_1P5V               ] = {0x400D6410, REG_WIDTH_32BITS, 29, 26,     3,   0x1FFF429C, 29, 26},

    [V1_TRIM_ID_ADCC_OFFSET_VREFHI_3P3V_IN       ] = {0x40001076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42A0,  7,  0},     //ADCC Offset    0   255
    [V1_TRIM_ID_ADCC_OFFSET_VREFHI_2P5V_IN       ] = {0x40001076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42A2,  7,  0},     //ADCC Offset    0   255
    [V1_TRIM_ID_ADCC_OFFSET_VREFHI_3P3V_OUT      ] = {0x40001076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42A4,  7,  0},     //ADCC Offset    0   255
    [V1_TRIM_ID_ADCC_OFFSET_VREFHI_2P5V_OUT      ] = {0x40001076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42A6,  7,  0},     //ADCC Offset    0   255
    [V1_TRIM_ID_ADCC_OFFSET_VREFHI_3P0V_OUT      ] = {0x40001076, REG_WIDTH_16BITS,  7,  0,     0,   0x1FFF42A8,  7,  0},     //ADCC Offset    0   255

    //Digital gain offset
    [V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P3V_IN   ] = {0x40000354, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42B0,  9,  0},
    [V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P3V_IN ] = {0x40000344, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42B0, 27, 16},
    [V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_2P5V_IN   ] = {0x40000354, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42B4,  9,  0},
    [V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_2P5V_IN ] = {0x40000344, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42B4, 27, 16},
    [V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P3V_OUT  ] = {0x40000354, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42B8,  9,  0},
    [V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P3V_OUT] = {0x40000344, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42B8, 27, 16},
    [V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_2P5V_OUT  ] = {0x40000354, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42BC,  9,  0},
    [V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_2P5V_OUT] = {0x40000344, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42BC, 27, 16},
    [V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P0V_OUT  ] = {0x40000354, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42C0,  9,  0},
    [V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P0V_OUT] = {0x40000344, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42C0, 27, 16},

    [V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P3V_IN   ] = {0x40001354, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42D0,  9,  0},
    [V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P3V_IN ] = {0x40001344, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42D0, 27, 16},
    [V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_2P5V_IN   ] = {0x40001354, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42D4,  9,  0},
    [V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_2P5V_IN ] = {0x40001344, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42D4, 27, 16},
    [V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P3V_OUT  ] = {0x40001354, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42D8,  9,  0},
    [V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P3V_OUT] = {0x40001344, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42D8, 27, 16},
    [V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_2P5V_OUT  ] = {0x40001354, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42DC,  9,  0},
    [V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_2P5V_OUT] = {0x40001344, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42DC, 27, 16},
    [V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P0V_OUT  ] = {0x40001354, REG_WIDTH_32BITS,  9,  0,     0,   0x1FFF42E0,  9,  0},
    [V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P0V_OUT] = {0x40001344, REG_WIDTH_32BITS, 11,  0, 0x800,   0x1FFF42E0, 27, 16},
};

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */


/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

GS32_DRIVER_INIT_FUNC_T void ASysCtl_initConfig(void)
{
	ASysCtl_setAnaClkTrim(0x49);

	ASysCtl_setL2HCfg(0x01);

	ASysCtl_setAdcTopSpare(0X47791F);

	ASysCtl_resetAllADC();   //Initially reset the ADC

	// CMPSS init
	ASysCtl_enableCMPSS();   //Enable cmpss module

	ASysCtl_unresetCMPSS();  //No reset  cmpss module

	HWREG(0x4004012C) = 3;    // enable  cmpss1  High and Low DAC   only  cmpss1 need do this

	ASysCtl_disableAllCMPMUX();   //Initialization does not enable any P-terminal and N-terminal channels

	ASysCtl_enableCMPMUXIO();    // enable  cmpss  io mux
}

GS32_DRIVER_ASYSCTL_FUNC_T static void AsysCtl_trimPeri(AnaTrimTable_t *trim)
{
    uint32_t shift, mask;
    shift = trim->startbit;
    mask = (1<<(trim->endbit - trim->startbit + 1)) - 1;

    if (HWREG(trim->efuseAddr) == 0xFFFFFFFF)
    {

	}
    else
    {
        uint32_t efuse_shift, efuse_mask;
        efuse_shift = trim->efuseStartbit;
        efuse_mask = (1<<(trim->efuseEndbit - trim->efuseStartbit + 1)) - 1;

        if(trim->regWidth == REG_WIDTH_16BITS)
            HWREGH(trim->regAddr) = \
    						(HWREGH(trim->regAddr) & ~(mask << shift)) | \
    						(((HWREGH(trim->efuseAddr) >> efuse_shift) & efuse_mask) << shift);
        else
            HWREG(trim->regAddr) = \
    						(HWREG(trim->regAddr) & ~(mask << shift)) | \
    						(((HWREG(trim->efuseAddr) >> efuse_shift) & efuse_mask) << shift);
	}

}


GS32_DRIVER_ASYSCTL_FUNC_T static void AsysCtl_trimPeri_v1(AnaTrimPeriId_v1_t id)
{
    AsysCtl_trimPeri(&anaTrimTable_v1[id]);
}

GS32_DRIVER_INIT_FUNC_T void ASysCtl_trim(void)
{
    uint32_t version = *((uint32_t *)ANALOG_TRIM_EFUSE_VERSION_ADDR);
    uint32_t id;

    ASysCtl_setOsc2FreqRange(OSC2_FREQ_HIGH);       //OSC2 ~10MHz

    ASysCtl_setOsc2Trim(0x7C0);

    SysCtl_setLdoChopEn(1);     //enable internal vcore ldo chop

	AsysCtl_trimPeri_v1(V1_TRIM_ID_LDO_1V2);
	AsysCtl_trimPeri_v1(V1_TRIM_ID_ANA_LDO);
	AsysCtl_trimPeri_v1(V1_TRIM_ID_OSC1);
	AsysCtl_trimPeri_v1(V1_TRIM_ID_OSC2_FINE_IN);

#ifdef TOP_LDO_CAL_MODE
    uint16 top_ldo_val = 0;
    top_ldo_val = HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG)&0x1f;

    for(uint16_t data = top_ldo_val;data <= 0x1f;data++)
    {
    	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) = (HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_ANA_LDO_1V1_CFG) &~(0x1f)) | data;

    }
#endif
}

GS32_DRIVER_ASYSCTL_FUNC_T static void AsysCtl_trimAdcVref_v1(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
    if(refMode == ADC_REFERENCE_INTERNAL) {
        if (refVolt == ADC_REFERENCE_3_3V) {
            AsysCtl_trimPeri_v1(V1_TRIM_ID_VREFHI_COARSE_1P65V);
            AsysCtl_trimPeri_v1(V1_TRIM_ID_VREFHI_FINE_1P65V);
        } else if (refVolt == ADC_REFERENCE_2_5V) {
            AsysCtl_trimPeri_v1(V1_TRIM_ID_VREFHI_COARSE_2P5V);
            AsysCtl_trimPeri_v1(V1_TRIM_ID_VREFHI_FINE_2P5V);
        }
    }
}

GS32_DRIVER_ASYSCTL_FUNC_T static void AsysCtl_trimAdcVcm_v1(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
    if (refVolt == ADC_REFERENCE_2_5V) {     //internal external 2.5V
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_VCM_COARSE_1P25V);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_VCM_FINE_1P25V);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_VCM_COARSE_1P25V);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_VCM_FINE_1P25V);
                break;
        }
    } else if (refVolt == ADC_REFERENCE_3_0V) {  //external 3.0V
         switch (adcBase) {
             case ADCA_BASE:
                 AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_VCM_COARSE_1P5V);
                 AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_VCM_FINE_1P5V);
             break;
             case ADCC_BASE:
                 AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_VCM_COARSE_1P5V);
                 AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_VCM_FINE_1P5V);
             break;
         }
    } else {
        if(refMode == ADC_REFERENCE_INTERNAL) {  //internal 3.3V
            switch (adcBase) {
                case ADCA_BASE:
                    AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_VCM_COARSE_2P475V);
                    AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_VCM_FINE_2P475V);
                    break;
                case ADCC_BASE:
                    AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_VCM_COARSE_2P475V);
                    AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_VCM_FINE_2P475V);
                    break;
            }
        } else {
            switch (adcBase) {       //external 3.3V
                case ADCA_BASE:
                    AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_VCM_COARSE_1P65V);
                    AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_VCM_FINE_1P65V);
                    break;
                case ADCC_BASE:
                    AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_VCM_COARSE_1P65V);
                    AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_VCM_FINE_1P65V);
                    break;
            }
        }
    }
}

GS32_DRIVER_ASYSCTL_FUNC_T static void AsysCtl_trimAdcOffset_v1(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
    if(refMode == ADC_REFERENCE_INTERNAL && refVolt == ADC_REFERENCE_2_5V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_2P5V_IN);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_OFFSET_VREFHI_2P5V_IN);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_2P5V_IN);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_OFFSET_VREFHI_2P5V_IN);
                break;
        }
    } else if(refMode == ADC_REFERENCE_INTERNAL && refVolt == ADC_REFERENCE_3_3V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P3V_IN);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_OFFSET_VREFHI_3P3V_IN);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P3V_IN);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_OFFSET_VREFHI_3P3V_IN);
                break;
        }
    } if(refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_2_5V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_2P5V_OUT);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_OFFSET_VREFHI_2P5V_OUT);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_2P5V_OUT);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_OFFSET_VREFHI_2P5V_OUT);
                break;
        }
    } else if(refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_3_0V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P0V_OUT);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_OFFSET_VREFHI_3P0V_OUT);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P0V_OUT);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_OFFSET_VREFHI_3P0V_OUT);
                break;
        }
    } else if(refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_3_3V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_OFF_DIR_VREFHI_3P3V_OUT);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_OFFSET_VREFHI_3P3V_OUT);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_OFF_DIR_VREFHI_3P3V_OUT);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_OFFSET_VREFHI_3P3V_OUT);
                break;
        }
    }
}

GS32_DRIVER_ASYSCTL_FUNC_T static void AsysCtl_trimAdcDigOffsetGain_v1(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
    //enable digital trim
    HWREG(adcBase + 0x364) |= 1<<8;        //adc_pre_pro_en

    if(refMode == ADC_REFERENCE_INTERNAL && refVolt == ADC_REFERENCE_2_5V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_2P5V_IN);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_2P5V_IN);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_2P5V_IN);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_2P5V_IN);
                break;
        }
    } else if(refMode == ADC_REFERENCE_INTERNAL && refVolt == ADC_REFERENCE_3_3V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P3V_IN);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P3V_IN);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P3V_IN);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P3V_IN);
                break;
        }
    } if(refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_2_5V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_2P5V_OUT);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_2P5V_OUT);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_2P5V_OUT);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_2P5V_OUT);
                break;
        }
    } else if(refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_3_0V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P0V_OUT);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P0V_OUT);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P0V_OUT);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P0V_OUT);
                break;
        }
    } else if(refMode == ADC_REFERENCE_EXTERNAL && refVolt == ADC_REFERENCE_3_3V) {
        switch (adcBase) {
            case ADCA_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_DIG_OFFSET_VREFHI_3P3V_OUT);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCA_DIG_GAIN_ERR_VREFHI_3P3V_OUT);
                break;
            case ADCC_BASE:
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_DIG_OFFSET_VREFHI_3P3V_OUT);
                AsysCtl_trimPeri_v1(V1_TRIM_ID_ADCC_DIG_GAIN_ERR_VREFHI_3P3V_OUT);
                break;
        }
    }
}

GS32_DRIVER_ASYSCTL_FUNC_T static void AsysCtl_trimAdc_v1(uint32_t adcBase, uint32_t refMode, uint32_t refVolt)
{
    AsysCtl_trimAdcVref_v1(adcBase, refMode, refVolt);
    AsysCtl_trimAdcVcm_v1(adcBase, refMode, refVolt);
    AsysCtl_trimAdcOffset_v1(adcBase, refMode, refVolt);
    AsysCtl_trimAdcDigOffsetGain_v1(adcBase, refMode, refVolt);
}

GS32_DRIVER_ASYSCTL_FUNC_T void AsysCtl_trimAdc(uint32_t adcBase, uint32_t refMode, uint32_t refVoltage)
{
	/* ADC_TOP_SPARE<6:4> == 010*/
	HWREG(0x400D6418U) = (HWREG(0x400D6418U) & (~(7<<4))) | (2<<4);

	/* ADCA_SPARE[2] = 0 */
	HWREG(0x400D6408) = HWREG(0x400D6408) & (~(1<<2));

	/* ADC_SOC_RST_CTRL<1:0> == 11*/
	HWREG(adcBase + 0x36CU) = (HWREG(adcBase + 0x36CU) & (~0x3)) | 0x03;

	/* ADC_SOC_RST_CTRL<11:4> == 00000000*/
	HWREG(adcBase + 0x36CU) = HWREG(adcBase + 0x36CU) & (~(0xFF<<4));

	AsysCtl_trimAdc_v1(adcBase, refMode, refVoltage);
}



#endif

#ifdef __cplusplus
}
#endif
