/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    sysctl_ahb.h
*   @brief   ABH system top level configuration
*            do NOT include this file directly, include "sysctl.h" instead
*
*/
/**
 * commit history
 * 20241218,Hezhiyuan,Modify comments
*/
#ifndef DEVICE_DRIVERLIB_SYSCTL_AHB_H_
#define DEVICE_DRIVERLIB_SYSCTL_AHB_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "gs32_version.h"
#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_sysctl_ahb.h"
#include "debug.h"

#define SYSCTL_SYNCSEL_ECAP1_SYNCIN     AHB1_COMM_PARA_SYNCSELECAPSYNCIN_1      //0x84U
#define SYSCTL_SYNCSEL_ECAP2_SYNCIN     AHB1_COMM_PARA_SYNCSELECAPSYNCIN_2      //0x88U
#define SYSCTL_SYNCSEL_ECAP3_SYNCIN     AHB1_COMM_PARA_SYNCSELECAPSYNCIN_3      //0x8CU
#define SYSCTL_SYNCSEL_ECAP4_SYNCIN     AHB1_COMM_PARA_SYNCSELECAPSYNCIN_4      //0x90U
#define SYSCTL_SYNCSEL_ECAP5_SYNCIN     AHB1_COMM_PARA_SYNCSELECAPSYNCIN_5      //0x94U
#define SYSCTL_SYNCSEL_ECAP6_SYNCIN     AHB1_COMM_PARA_SYNCSELECAPSYNCIN_6      //0x98U
#define SYSCTL_SYNCSEL_ECAP7_SYNCIN     AHB1_COMM_PARA_SYNCSELECAPSYNCIN_7      //0x9CU



#if IS_GS32F00xx(0x12)

typedef enum
{
	SYSCTL_SYNC_OUT_SRC_EPWM1SYNCOUT      = 0,
	SYSCTL_SYNC_OUT_SRC_EPWM2SYNCOUT      = 1,
	SYSCTL_SYNC_OUT_SRC_EPWM3SYNCOUT      = 2,
	SYSCTL_SYNC_OUT_SRC_EPWM4SYNCOUT      = 3,
	SYSCTL_SYNC_OUT_SRC_EPWM5SYNCOUT      = 4,
	SYSCTL_SYNC_OUT_SRC_EPWM6SYNCOUT      = 5,
	SYSCTL_SYNC_OUT_SRC_EPWM7SYNCOUT      = 6,
	SYSCTL_SYNC_OUT_SRC_EPWM8SYNCOUT      = 7,
	SYSCTL_SYNC_OUT_SRC_EPWM9SYNCOUT      = 8,
	SYSCTL_SYNC_OUT_SRC_EPWM10SYNCOUT     = 9,
	SYSCTL_SYNC_OUT_SRC_EPWM11SYNCOUT     = 10,
	SYSCTL_SYNC_OUT_SRC_EPWM12SYNCOUT     = 11,
    SYSCTL_SYNC_OUT_SRC_ECAP1SYNCOUT      = 24,
    SYSCTL_SYNC_OUT_SRC_ECAP2SYNCOUT      = 25,
    SYSCTL_SYNC_OUT_SRC_ECAP3SYNCOUT      = 26,
    SYSCTL_SYNC_OUT_SRC_ECAP4SYNCOUT      = 27,
    SYSCTL_SYNC_OUT_SRC_ECAP5SYNCOUT      = 28,
    SYSCTL_SYNC_OUT_SRC_ECAP6SYNCOUT      = 29,
    SYSCTL_SYNC_OUT_SRC_ECAP7SYNCOUT      = 30,
} SysCtl_SyncOutputSource;


#elif IS_GS32F00xx(0x30)
typedef enum
{
	SYSCTL_SYNC_OUT_SRC_EPWM1SYNCOUT      = 0,
	SYSCTL_SYNC_OUT_SRC_EPWM2SYNCOUT      = 1,
	SYSCTL_SYNC_OUT_SRC_EPWM3SYNCOUT      = 2,
	SYSCTL_SYNC_OUT_SRC_EPWM4SYNCOUT      = 3,
	SYSCTL_SYNC_OUT_SRC_EPWM5SYNCOUT      = 4,
	SYSCTL_SYNC_OUT_SRC_EPWM6SYNCOUT      = 5,
	SYSCTL_SYNC_OUT_SRC_EPWM7SYNCOUT      = 6,
	SYSCTL_SYNC_OUT_SRC_EPWM8SYNCOUT      = 7,
	SYSCTL_SYNC_OUT_SRC_EPWM9SYNCOUT      = 8,
	SYSCTL_SYNC_OUT_SRC_EPWM10SYNCOUT     = 9,
	SYSCTL_SYNC_OUT_SRC_EPWM11SYNCOUT     = 10,
	SYSCTL_SYNC_OUT_SRC_EPWM12SYNCOUT     = 11,
	SYSCTL_SYNC_OUT_SRC_EPWM13SYNCOUT     = 12,
	SYSCTL_SYNC_OUT_SRC_EPWM14SYNCOUT     = 13,
	SYSCTL_SYNC_OUT_SRC_EPWM15SYNCOUT     = 14,
	SYSCTL_SYNC_OUT_SRC_EPWM16SYNCOUT     = 15,
	SYSCTL_SYNC_OUT_SRC_EPWM17SYNCOUT     = 16,
	SYSCTL_SYNC_OUT_SRC_EPWM18SYNCOUT     = 17,
    SYSCTL_SYNC_OUT_SRC_ECAP1SYNCOUT      = 24,
    SYSCTL_SYNC_OUT_SRC_ECAP2SYNCOUT      = 25,
    SYSCTL_SYNC_OUT_SRC_ECAP3SYNCOUT      = 26,
    SYSCTL_SYNC_OUT_SRC_ECAP4SYNCOUT      = 27,
    SYSCTL_SYNC_OUT_SRC_ECAP5SYNCOUT      = 28,
    SYSCTL_SYNC_OUT_SRC_ECAP6SYNCOUT      = 29,
    SYSCTL_SYNC_OUT_SRC_ECAP7SYNCOUT      = 30,
} SysCtl_SyncOutputSource;

#elif IS_GS32F3xx(0x22)

typedef enum
{
	SYSCTL_SYNC_OUT_SRC_EPWM1SYNCOUT      = 0,
	SYSCTL_SYNC_OUT_SRC_EPWM2SYNCOUT      = 1,
	SYSCTL_SYNC_OUT_SRC_EPWM3SYNCOUT      = 2,
	SYSCTL_SYNC_OUT_SRC_EPWM4SYNCOUT      = 3,
	SYSCTL_SYNC_OUT_SRC_EPWM5SYNCOUT      = 4,
	SYSCTL_SYNC_OUT_SRC_EPWM6SYNCOUT      = 5,
	SYSCTL_SYNC_OUT_SRC_EPWM7SYNCOUT      = 6,
	SYSCTL_SYNC_OUT_SRC_EPWM8SYNCOUT      = 7,
	SYSCTL_SYNC_OUT_SRC_EPWM9SYNCOUT      = 8,
	SYSCTL_SYNC_OUT_SRC_EPWM10SYNCOUT     = 9,
	SYSCTL_SYNC_OUT_SRC_EPWM11SYNCOUT     = 10,
	SYSCTL_SYNC_OUT_SRC_EPWM12SYNCOUT     = 11,
	SYSCTL_SYNC_OUT_SRC_EPWM13SYNCOUT     = 12,
	SYSCTL_SYNC_OUT_SRC_EPWM14SYNCOUT     = 13,
	SYSCTL_SYNC_OUT_SRC_EPWM15SYNCOUT     = 14,
	SYSCTL_SYNC_OUT_SRC_EPWM16SYNCOUT     = 15,
	SYSCTL_SYNC_OUT_SRC_EPWM17SYNCOUT     = 16,
	SYSCTL_SYNC_OUT_SRC_EPWM18SYNCOUT     = 17,
    SYSCTL_SYNC_OUT_SRC_ECAP1SYNCOUT      = 24,
    SYSCTL_SYNC_OUT_SRC_ECAP2SYNCOUT      = 25,
    SYSCTL_SYNC_OUT_SRC_ECAP3SYNCOUT      = 26,
    SYSCTL_SYNC_OUT_SRC_ECAP4SYNCOUT      = 27,
    SYSCTL_SYNC_OUT_SRC_ECAP5SYNCOUT      = 28,
    SYSCTL_SYNC_OUT_SRC_ECAP6SYNCOUT      = 29,
    SYSCTL_SYNC_OUT_SRC_ECAP7SYNCOUT      = 30,
} SysCtl_SyncOutputSource;

#else
#error GS32F00xx define error
#endif
/* ========================================================================== */

/*
 *   Module Name                    Base_Addr
 *
 *   AHB1_COMM_PARA_CFG             0x4000_6400
*/

/* ========================================================================== */

#if IS_GS32F00xx(0x12)


/*
 * Set the value of syncselsyncout for the output synchronization signal
 */
__STATIC_INLINE void SysCtl_setSyncOutputConfig(SysCtl_SyncOutputSource syncSrc)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SYNCSELSYNCOUT) = ((uint32_t)syncSrc<<24);
}


/**
 * 
 *  Enables ePWM SOC signals to drive an external (off-chip) ADCSOC signal.
 * 
 *  @param [in] adcsocSrc is a bit field of the selected signals to be enabled
 * 
 *  This function configures which ePWM SOC signals are enabled as a source for
 *  either ADCSOCAO or ADCSOCBO. The  adcsocSrc parameter takes a logical OR
 *  of  SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
 *  signals.
 * 
 *  @return None.
 * 
 */
static inline void
SysCtl_enableExtADCSOCSource(uint32_t adcsocSrc)
{
    //
    // Set the bits that correspond to signal to be enabled.
    //
    EALLOW;
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_ADCSOCOUTSELECT) |= adcsocSrc;
    EDIS;
}

/**
 * 
 *  Disables ePWM SOC signals from driving an external ADCSOC signal.
 * 
 *  @param [in] adcsocSrc is a bit field of the selected signals to be disabled
 * 
 *  This function configures which ePWM SOC signals are disabled as a source
 *  for either ADCSOCAO or ADCSOCBO. The  adcsocSrc parameter takes a logical
 *  OR of  SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
 *  signals.
 * 
 *  @return None.
 * 
 */
static inline void
SysCtl_disableExtADCSOCSource(uint32_t adcsocSrc)
{
    //
    // Clear the bits that correspond to signal to be disabled.
    //
    EALLOW;
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_ADCSOCOUTSELECT) &= ~adcsocSrc;
    EDIS;
}

/**
 * @brief   Turn on EPWM TBCLK Sync
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_TBCLKSYNC(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_TBCLKSYNC) = 1<<18;
}

/**
 * 
 * @brief   Turn off EPWM TBCLK Sync
 * @remarks The target register is in AHB
 *
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_TBCLKSYNC(void)
{
    HWREG(PREEPWM_BASE + AHB1_COMM_PARA_TBCLKSYNC) = 0;

}

/**
 * @brief SMP  Digital sampling is enable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_SMP_EN(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_EN) |= 1;
}

/**
 * @brief SMP  Digital sampling is disable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_SMP_EN(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_EN) = 0;
}

/**
 * @brief  SMP Digital sampling mode is enable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_SMP_MODE(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_MODE) |= 1;
}

/**
 * @brief SMP  Digital sampling mode is disable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_SMP_MODE(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_MODE) = 0;
}

/**
 * @brief SMP  Digital sampling interval is enable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_SMP_GAP(uint32_t value)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_GAP) |= 0xff&&value;
}

/**
 * @brief   Digital sampling length is disable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_SMP_GAP(uint32_t value)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_GAP) &= ~0xff;
}

/**
 * @brief SMP  Digital sampling length is enable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_SMP_LEN(uint32_t value)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_LEN) |= 0x3fff&&value;
}

/**
 * @brief   Digital sampling length is disable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_SMP_LEN(uint32_t value)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_LEN) &= ~0x3fff;
}

#elif IS_GS32F00xx(0x30)
/*
 * Set the value of syncselsyncout for the output synchronization signal
 */
__STATIC_INLINE void SysCtl_setSyncOutputConfig(SysCtl_SyncOutputSource syncSrc)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SYNCSELSYNCOUT) = syncSrc;
}


/**
 * 
 *  Enables ePWM SOC signals to drive an external (off-chip) ADCSOC signal.
 * 
 *  @param [in] adcsocSrc is a bit field of the selected signals to be enabled
 * 
 *  This function configures which ePWM SOC signals are enabled as a source for
 *  either ADCSOCAO or ADCSOCBO. The  adcsocSrc parameter takes a logical OR
 *  of  SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
 *  signals.
 * 
 *  @return None.
 * 
 */
static inline void
SysCtl_enableExtADCSOCSource(uint32_t adcsocSrc)
{
    //
    // Set the bits that correspond to signal to be enabled.
    //
    EALLOW;
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_ADCSOCOUTSELECT) |= adcsocSrc;
    EDIS;
}

/**
 * 
 *  Disables ePWM SOC signals from driving an external ADCSOC signal.
 * 
 *  @param [in] adcsocSrc is a bit field of the selected signals to be disabled
 * 
 *  This function configures which ePWM SOC signals are disabled as a source
 *  for either ADCSOCAO or ADCSOCBO. The  adcsocSrc parameter takes a logical
 *  OR of  SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
 *  signals.
 * 
 *  @return None.
 * 
 */
static inline void
SysCtl_disableExtADCSOCSource(uint32_t adcsocSrc)
{
    //
    // Clear the bits that correspond to signal to be disabled.
    //
    EALLOW;
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_ADCSOCOUTSELECT) &= ~adcsocSrc;
    EDIS;
}

/**
 * @brief   Turn on EPWM TBCLK Sync
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_TBCLKSYNC(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_TBCLKSYNC) = 1<<18;
}

/**
 * @brief   Turn off EPWM TBCLK Sync
 * @remarks The target register is in AHB
 *
 *
 *
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_TBCLKSYNC(void)
{
    HWREG(PREEPWM_BASE + AHB1_COMM_PARA_TBCLKSYNC) = 0;

}



/**
 * @brief ADC sampling mode selection
 * @remarks The target register is in AHB
 *
 * @return  None
 */
__STATIC_INLINE void SysCtl_ADC_Sample_Mode_Slect(uint32_t value)
{

    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_ADC_SAMPLE_SEL) = value;

}

/**
 * @brief SMP  Digital sampling is enable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_SMP_EN(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_EN) |= 1;
}

/**
 * @brief SMP  Digital sampling is disable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_SMP_EN(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_EN) = 0;
}

/**
 * @brief  SMP Digital sampling mode is enable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_SMP_MODE(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_MODE) |= 1;
}

/**
 * @brief SMP  Digital sampling mode is disable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_SMP_MODE(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_MODE) = 0;
}

/**
 * @brief SMP  Digital sampling interval is enable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_SMP_GAP(uint32_t value)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_GAP) |= 0xff&&value;
}

/**
 * @brief   Digital sampling length is disable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_SMP_GAP(uint32_t value)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_GAP) &= ~0xff;
}

/**
 * @brief SMP  Digital sampling length is enable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_SMP_LEN(uint32_t value)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_LEN) |= 0x3fff&value;
}

/**
 * @brief   Digital sampling length is disable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_SMP_LEN(uint32_t value)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_LEN) &= ~0x3fff;
}
#endif
#if IS_GS32F3xx()

#if IS_GS32F3xx(0x22)


/*
 * Set the value of syncselsyncout for the output synchronization signal
 */
__STATIC_INLINE void SysCtl_setSyncOutputConfig(SysCtl_SyncOutputSource syncSrc)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SYNCSELSYNCOUT) = ((uint32_t)syncSrc<<24);
}


/**
 * 
 *  Enables ePWM SOC signals to drive an external (off-chip) ADCSOC signal.
 * 
 *  @param [in] adcsocSrc is a bit field of the selected signals to be enabled
 * 
 *  This function configures which ePWM SOC signals are enabled as a source for
 *  either ADCSOCAO or ADCSOCBO. The  adcsocSrc parameter takes a logical OR
 *  of  SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
 *  signals.
 * 
 *  @return None.
 * 
 */
static inline void
SysCtl_enableExtADCSOCSource(uint32_t adcsocSrc)
{
    //
    // Set the bits that correspond to signal to be enabled.
    //
    EALLOW;
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_ADCSOCOUTSELECT) |= adcsocSrc;
    EDIS;
}

/**
 * 
 *  Disables ePWM SOC signals from driving an external ADCSOC signal.
 * 
 *  @param [in] adcsocSrc is a bit field of the selected signals to be disabled
 * 
 *  This function configures which ePWM SOC signals are disabled as a source
 *  for either ADCSOCAO or ADCSOCBO. The  adcsocSrc parameter takes a logical
 *  OR of  SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
 *  signals.
 * 
 *  @return None.
 * 
 */
static inline void
SysCtl_disableExtADCSOCSource(uint32_t adcsocSrc)
{
    //
    // Clear the bits that correspond to signal to be disabled.
    //
    EALLOW;
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_ADCSOCOUTSELECT) &= ~adcsocSrc;
    EDIS;
}

/**
 * 
 *  Enables ePWM SOC signals to drive an external (off-chip) ADCSOC signal.
 * 
 *  @param [in] adcsocSrc is a bit field of the selected signals to be enabled
 * 
 *  This function configures which ePWM SOC signals are enabled as a source for
 *  either ADCSOCAO or ADCSOCBO. The  adcsocSrc parameter takes a logical OR
 *  of  SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
 *  signals.
 * 
 *  @return None.
 * 
 */
static inline void
SysCtl_enableExtADCSOCSource_1(uint32_t adcsocSrc)
{
    //
    // Set the bits that correspond to signal to be enabled.
    //
    EALLOW;
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_ADCSOCOUTSELECT_1) |= adcsocSrc;
    EDIS;
}

/**
 * 
 *  Disables ePWM SOC signals from driving an external ADCSOC signal.
 * 
 *  @param [in] adcsocSrc is a bit field of the selected signals to be disabled
 * 
 *  This function configures which ePWM SOC signals are disabled as a source
 *  for either ADCSOCAO or ADCSOCBO. The  adcsocSrc parameter takes a logical
 *  OR of  SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
 *  signals.
 * 
 *  @return None.
 * 
 */
static inline void
SysCtl_disableExtADCSOCSource_1(uint32_t adcsocSrc)
{
    //
    // Clear the bits that correspond to signal to be disabled.
    //
    EALLOW;
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_ADCSOCOUTSELECT_1) &= ~adcsocSrc;
    EDIS;
}
#endif
/**
 * @brief   Turn on EPWM TBCLK Sync
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_TBCLKSYNC(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_TBCLKSYNC) = 1<<18;
}

/**
 * @brief   Turn off EPWM TBCLK Sync
 * @remarks The target register is in AHB
 *
 *
 *
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_TBCLKSYNC(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_TBCLKSYNC) = 0;

}

/**
 * @brief SMP  Digital sampling is enable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_SMP_EN(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_EN) |= 1;
}

/**
 * @brief SMP  Digital sampling is disable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_SMP_EN(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_EN) = 0;
}

/**
 * @brief  SMP Digital sampling mode is enable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_SMP_MODE(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_MODE) |= 1;
}

/**
 * @brief SMP  Digital sampling mode is disable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_SMP_MODE(void)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_MODE) = 0;
}

/**
 * @brief SMP  Digital sampling interval is enable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_SMP_GAP(uint32_t value)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_GAP) |= 0xff&&value;
}

/**
 * @brief   Digital sampling length is disable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_SMP_GAP(uint32_t value)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_GAP) &= ~0xff;
}

/**
 * @brief SMP  Digital sampling length is enable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral_SMP_LEN(uint32_t value)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_LEN) |= 0x3fff&&value;
}

/**
 * @brief   Digital sampling length is disable
 * @remarks The target register is in AHB
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral_SMP_LEN(uint32_t value)
{
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_SMP_LEN) &= ~0x3fff;
}
#endif



#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_SYSCTL_AHB_H_ */
