/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *   @file    sysctl_io_v30.h
 *   @brief   do NOT include this file directly, include "sysctl.h" instead
 *
 */
/**
 * commit history
 * 20241218,Hezhiyuan,Modify comments
 */

#ifndef DEVICE_DRIVERLIB_SYSCTL_IO_V30_H_
#define DEVICE_DRIVERLIB_SYSCTL_IO_V30_H_

#include "gs32_version.h"

#if IS_GS32F00xx(0x30)

#ifdef __cplusplus
extern "C" {
#endif

#include "debug.h"
#include "inc/hw_nmi.h"
#include "inc/hw_memmap.h"
#include "inc/hw_dsp_apb_cfg.h"
#include "inc/hw_sysctl_io_v30.h"
#include "inc/hw_sysctl_dig_v30.h"



#define IO_LOCK_ENKEY            (0x5A5A5A5A)
#define GPIO_CTRL_REGS_STEP      ((GPIO_O_GPBCTRL - GPIO_O_GPACTRL))
#define GPIO_EXT_REGS_STEP       ((GPIO_O_GPBPDD - GPIO_O_GPAPDD))
#define GPIO_DATA_REGS_STEP      (GPIO_O_GPBDAT - GPIO_O_GPADAT)
#define GPIO_DATA_READ_REGS_STEP (GPIO_O_GPBDAT_R - GPIO_O_GPADAT_R)

#define GPIO_INT_REGS_STEP       (GPIO_O_GPB_INTTYPE - GPIO_O_GPA_INTTYPE)

#define GPIO_GPxCTRL_INDEX       ((GPIO_O_GPACTRL - GPIO_O_GPACTRL) / 4U)
#define GPIO_GPxQSEL_INDEX       ((GPIO_O_GPAQSEL1 - GPIO_O_GPACTRL) / 4U)
#define GPIO_GPxMUX_INDEX        ((GPIO_O_GPAMUX1 - GPIO_O_GPACTRL) / 4U)
#define GPIO_GPxDIR_INDEX        ((GPIO_O_GPADIR - GPIO_O_GPACTRL) / 4U)
#define GPIO_GPxPUD_INDEX        ((GPIO_O_GPAPUD - GPIO_O_GPACTRL) / 4U)
#define GPIO_GPxINV_INDEX        ((GPIO_O_GPAINV - GPIO_O_GPACTRL) / 4U)
#define GPIO_GPxAMSEL_INDEX      ((GPIO_O_GPAAMSEL - GPIO_O_GPACTRL) / 4U)
#define GPIO_GPxGMUX_INDEX       ((GPIO_O_GPAGMUX1 - GPIO_O_GPACTRL) / 4U)

#define GPIO_GPxPDD_INDEX        ((GPIO_O_GPAPDD - GPIO_O_GPAPDD) / 4U)
#define GPIO_GPxDS_INDEX         ((GPIO_O_GPADS1 - GPIO_O_GPAPDD) / 4U)
#define GPIO_GPxIS_INDEX         ((GPIO_O_GPAIS - GPIO_O_GPAPDD) / 4U)
#define GPIO_GPxOUTINV_INDEX     ((GPIO_O_GPAOUTINV - GPIO_O_GPAPDD) / 4U)
#define GPIO_GPxLOOPEN_INDEX     ((GPIO_O_GPALOOPEN - GPIO_O_GPAPDD) / 4U)
#define GPIO_GPxPI2C_MODE_INDEX  ((GPIO_O_GPAPI2C_MODE - GPIO_O_GPAPDD) / 4U)
#define GPIO_GPxPOUT_RPT_INDEX   ((GPIO_O_GPA_POUT_RPT - GPIO_O_GPAPDD) / 4U)

#define GPIO_GPxDAT_INDEX        (0U)
#define GPIO_GPxSET_INDEX        ((GPIO_O_GPASET - GPIO_O_GPADAT) / 4)
#define GPIO_GPxCLEAR_INDEX      ((GPIO_O_GPACLEAR - GPIO_O_GPADAT) / 4)
#define GPIO_GPxTOGGLE_INDEX     ((GPIO_O_GPATOGGLE - GPIO_O_GPADAT) / 4)

#define GPIO_GPxDAT_R_INDEX      (0U)

#define GPIO_INTTYPE_INDEX       (0U)
#define GPIO_INTPOL_INDEX        ((GPIO_O_GPA_INTPOL - GPIO_O_GPA_INTTYPE) / 4)
#define GPIO_INTMASK_INDEX       ((GPIO_O_GPA_INTMASK - GPIO_O_GPA_INTTYPE) / 4)
#define GPIO_INTCLR_INDEX        ((GPIO_O_GPA_INTCLR - GPIO_O_GPA_INTTYPE) / 4)
#define GPIO_INTRAW_INDEX        ((GPIO_O_GPA_INTRAW - GPIO_O_GPA_INTTYPE) / 4)
#define GPIO_INTSTATUS_INDEX     ((GPIO_O_GPA_INTSTATUS - GPIO_O_GPA_INTTYPE) / 4)

#define GPIO_MUX_TO_GMUX         (GPIO_O_GPAGMUX1 - GPIO_O_GPAMUX1)

/*******************************************************************************
 *
 * Values that can be passed to GPIO_setPadConfig() as the pinType parameter
 * and returned by GPIO_getPadConfig().
 *
 *******************************************************************************/
#if (GS32_PART_NUM==0x035 || GS32_PART_NUM==0x027)

#define GPIO_PIN_TYPE_STD        		0x0000U /* Push-pull output or floating input */
#define GPIO_PIN_TYPE_PULLUP     		0x0001U /* Pull-up enable for input */
#define GPIO_PIN_TYPE_PULLDOWN   		0x0008U /* Pull-Down Input */
#define GPIO_PIN_TYPE_OUTPUT_INVERT     0x0010U /* Inverted Push-Pull Output*/

#else
#define GPIO_PIN_TYPE_STD        		0x0000U /* Push-pull output or floating input */
#define GPIO_PIN_TYPE_PULLUP     		0x0001U /* Pull-up enable for input */
#define GPIO_PIN_TYPE_INVERT     		0x0002U /* Invert polarity on input */

#define GPIO_PIN_TYPE_PULLDOWN   		0x0008U /* Pull-Down Input */
#define GPIO_PIN_TYPE_OUTPUT_INVERT     0x0010U /* Inverted Push-Pull Output */

#endif

/*******************************************************************************
 *
 * @brief GPIO Pull-up/Pull-down Resistor Configuration State Enumeration
 *
 * Defines the enable state of the internal pull-up/pull-down resistors for GPIO,
 * used to configure control registers like GPAPUD, etc.
 *
 *******************************************************************************/
typedef enum {
    GPIO_PULL_ENABLE  = 0,   // Enable internal pull-up/pull-down resistors
    GPIO_PULL_DISABLE = 1  // Disable internal pull-up/pull-down resistors
} GpioPullUp_Regs;

/*******************************************************************************
 *
 * Values that can be passed to GPIO_setInterruptPin(),
 * GPIO_setInterruptType(), GPIO_getInterruptType(), GPIO_enableInterrupt(),
 * GPIO_disableInterrupt(), as the extIntNum parameter.
 *
 *******************************************************************************/

typedef enum {
    GPIO_INT_XINT1 = 1, // External Interrupt 1
    GPIO_INT_XINT2,     // External Interrupt 2
    GPIO_INT_XINT3,     // External Interrupt 3
    GPIO_INT_XINT4,     // External Interrupt 4
    GPIO_INT_XINT5,     // External Interrupt 5
    GPIO_INT_XINT6,     // External Interrupt 6
    GPIO_INT_XINT7,     // External Interrupt 7
    GPIO_INT_XINT8,     // External Interrupt 8
    GPIO_INT_XINT9,     // External Interrupt 9
    GPIO_INT_XINT10,    // External Interrupt 10
    GPIO_INT_XINT11,    // External Interrupt 11
    GPIO_INT_XINT12,    // External Interrupt 12
    GPIO_INT_XINT13,    // External Interrupt 13
    GPIO_INT_XINT14,    // External Interrupt 14
    GPIO_INT_XINT15,    // External Interrupt 15
    GPIO_INT_XINT16     // External Interrupt 16
} GPIO_ExternalIntNum;

/*******************************************************************************
 *
 * Values that can be passed to GPIO_setInterruptType() as the intType
 * parameter and returned from GPIO_getInterruptType().
 *
 *******************************************************************************/
typedef enum {
    GPIO_INT_TYPE_RISING_EDGE  = 0x04, // Interrupt on rising edge
    GPIO_INT_TYPE_BOTH_EDGES   = 0x0C  // Interrupt on both edges
} GPIO_IntType;

/*******************************************************************************
 *
 * Values that can be passed to GPIO_readPortData(), GPIO_setPortPins(),
 * GPIO_clearPortPins(), and GPIO_togglePortPins() as the port parameter.
 *
 *******************************************************************************/
typedef enum {
    GPIO_PORT_A = 0, // GPIO port A
    GPIO_PORT_B = 1, // GPIO port B
    GPIO_PORT_G = 6, // GPIO port G
    GPIO_PORT_H = 7  // GPIO port H
} GPIO_Port;

/*******************************************************************************
 *
 * Values that can be passed to GPIO_setDirectionMode() as the pinIO
 * parameter and returned from GPIO_getDirectionMode().
 *
 *******************************************************************************/

typedef enum {
    GPIO_DIR_MODE_IN    = 0, // Pin is a GPIO input
    GPIO_DIR_MODE_OUT   = 1, // Pin is a GPIO output
    AIO_IE_MODE_IN		= 2, // Pin is a AIO input
    AIO_IE_MODE_NO_IN 	= 3  // Pin is not a AIO input
} GPIO_Direction;

/*******************************************************************************
 *
 * Values that can be passed to GPIO_setQualificationMode() as the
 * qualification parameter and returned by GPIO_getQualificationMode().
 *
 *******************************************************************************/

typedef enum {
    GPIO_QUAL_SYNC,    // Synchronization to SYSCLK
    GPIO_QUAL_3SAMPLE, // Qualified with 3 samples
    GPIO_QUAL_6SAMPLE, // Qualified with 6 samples
    GPIO_QUAL_ASYNC    // No synchronization
} GPIO_QualificationMode;

/*******************************************************************************
 *
 * Values that can be passed to GPIO_setAnalogMode() as the mode parameter.
 *
 *******************************************************************************/
typedef enum {
    GPIO_ANALOG_DISABLED, // Pin is in digital mode
    GPIO_ANALOG_ENABLED   // Pin is in analog mode
} GPIO_AnalogMode;

/*******************************************************************************
 *
 * Values that can be returned by `check_pin_type()` to indicate the pin type.
 *
 *******************************************************************************/
typedef enum {
    HW_PIN_TYPE_AIO     = 0,
    HW_PIN_TYPE_AGPIO   = 1,
    HW_PIN_TYPE_GPIO    = 2,
    HW_PIN_TYPE_UNKNOWN = 3
} HwPinType;

/*******************************************************************************
 *
 *  Values that can be passed to GPIO_setStrength() as the  mode parameter.
 *
 *******************************************************************************/
typedef enum {
    GPIO_DRV_2MA  = 0,
    GPIO_DRV_4MA  = 1,
    GPIO_DRV_8MA  = 2,
    GPIO_DRV_12MA = 3,
    GPIO_DRV_20MA = 4,
} GPIO_DriveStrength;

typedef enum {
    GPIO_TYPE_PI2C = 0, // I2C GPIO : 4MA/20MA
    GPIO_TYPE_GPIO = 1  // Standard GPIO : 2MA/4MA/8MA/12MA
} GPIO_Type;

// GPIO configuration structure
typedef struct
{
    uint8_t   pin_number; // Pin number
    uint8_t   pin_S;      // Pin offset
    GPIO_Type type;       // Pin type (Standard/I2C)
} gpio_config_t;

/*******************************************************************************
 *
 *  Values that can be passed to GPIO_readPortOutReport() as the  port parameter.
 *
 *******************************************************************************/
typedef enum {
    GPIO_PORTA = 0, //< GPIO port A, 32 bits
    GPIO_PORTB = 1, //< GPIO port B, 32 bits
    GPIO_PORTG = 6, //< GPIO port G, 32 bits
    GPIO_PORTH = 7  //< GPIO port H, 32 bits
} GPIO_Port_32BIT;

/*******************************************************************************
 *
 * Values that can be passed to GPIO_setInterruptType_NO_XInt() as the
 * intTypePolarity parameter and returned by GPIO_getInterruptType_NO_XInt().
 *
 *******************************************************************************/
typedef enum {
    GPIO_INTERRUPT_TYPE_LEVEL_LOW  = 0, // 0: level (low)
    GPIO_INTERRUPT_TYPE_LEVEL_HIGH = 1, // 1: level (high)
    GPIO_INTERRUPT_TYPE_EDGE_RISE  = 2, // 2: edge (rise)
    GPIO_INTERRUPT_TYPE_EDGE_FALL  = 3  // 3: edge (fall)
} GPIO_InterruptType_NO_XInt;

/*******************************************************************************
 *
 * Values that can be passed to GPIO_setClkRef() as the
 * mode parameter and returned by GPIO_getClkRef().
 *
 *******************************************************************************/
typedef enum {
    GPIO_ClkRef_CRYSTAL        = 0 << 1 | 1 << 0, // HSE type crystal
    GPIO_ClkRef_OSCILLATOR     = 1 << 1 | 0 << 0, // HSE type oscillator
    GPIO_ClkRef_1MHz_TO_4MHz   = 0 << 2,          // 1MHz < freq <= 4MHz
    GPIO_ClkRef_4MHz_TO_12MHz  = 2 << 2,          // 4MHz < freq <= 12MHz
    GPIO_ClkRef_12MHz_TO_24MHz = 1 << 2,          // 12MHz < freq <= 24MHz
    GPIO_ClkRef_24MHz_TO_48MHz = 3 << 2,          // 24MHz < freq <= 48MHz
    GPIO_ClkRef_R_500K         = 0 << 4,          // 500KOm feedback resistor
    GPIO_ClkRef_R_100K         = 1 << 4,          // 100KOm feedback resistor
} GPIO_ClkRefMode;
/*******************************************************************************
 *
 *  Values that can be passed to GPIO_CLB_HRPWM_EX_EN() as the CLB_HRPWM_EX_EN_T parameter.
 *
 *******************************************************************************/
typedef enum {
    CLB_HRPWM_EX_EN_EPWM1_A = 0, // epwm1_a: 0 = normal mode, 1 = exchange mode
    CLB_HRPWM_EX_EN_EPWM1_B = 1, // epwm1_b: 0 = normal mode, 1 = exchange mode
    CLB_HRPWM_EX_EN_EPWM2_A = 2, // epwm2_a: 0 = normal mode, 1 = exchange mode
    CLB_HRPWM_EX_EN_EPWM2_B = 3, // epwm2_b: 0 = normal mode, 1 = exchange mode
    CLB_HRPWM_EX_EN_EPWM3_A = 4, // epwm3_a: 0 = normal mode, 1 = exchange mode
    CLB_HRPWM_EX_EN_EPWM3_B = 5, // epwm3_b: 0 = normal mode, 1 = exchange mode
    CLB_HRPWM_EX_EN_EPWM4_A = 6, // epwm4_a: 0 = normal mode, 1 = exchange mode
    CLB_HRPWM_EX_EN_EPWM4_B = 7  // epwm4_b: 0 = normal mode, 1 = exchange mode
} CLB_HRPWM_EX_EN_T;

/*******************************************************************************
 *
 * Enum for AIO Mux 1 register configurations.
 * 
 * The two possible configurations for each AIO pin:
 * - AIO_MUX1_IO_ENABLED (0): Enables the GPIO function for the corresponding pin (00 or 01).
 * - AIO_MUX1_IO_DISABLED (3): Disables the GPIO function for the corresponding pin (10 or 11).
 *
 *******************************************************************************/
typedef enum {
    AIO_MUX1_IO_ENABLED  = 0,  // Enables GPIO function for the corresponding pin (00 or 01)
    AIO_MUX1_IO_DISABLED = 3   // Disables GPIO function for the corresponding pin (10 or 11)
} AIOMux1_Regs;


/*******************************************************************************
 *
 * Enum for GPIO interrupt types.
 *
 * This register configures the type of interrupt for GPIO pins.
 * There are two interrupt types:
 * - Level-triggered interrupt: The interrupt is triggered when the GPIO signal is either high or low.
 * - Edge-triggered interrupt: The interrupt is triggered when the GPIO signal changes, either from low to high (rising edge) or from high to low (falling edge).
 *
 *******************************************************************************/
typedef enum {
    GPIO_INTTYPE_LEVEL  = 0,  // 0 : Level-triggered interrupt (high or low)
    GPIO_INTTYPE_EDGE   = 1   // 1 : Edge-triggered interrupt (rising or falling edge)
} GPIOIntType_Regs;


/*******************************************************************************
 *
 * Enum for GPIO interrupt polarity.
 *
 * This register configures the polarity of the interrupt for GPIO pins .
 * - 0 indicates a low level or falling edge trigger for the interrupt.
 * - 1 indicates a high level or rising edge trigger for the interrupt.
 *
 *******************************************************************************/
typedef enum {
    GPIO_INTPOL_NEG_EDGE  = 0,  // 0 : Low level or falling edge
    GPIO_INTPOL_POS_EDGE  = 1   // 1 : High level or rising edge
} GPIOIntPol_Regs;


/*******************************************************************************
 *
 * Enum for GPIO interrupt masking.
 *
 * This register configures the mask for GPIO interrupt.
 * - 0 means the interrupt is masked (disabled).
 * - 1 means the interrupt is unmasked (enabled).
 *
 *******************************************************************************/
typedef enum {
    GPIO_INTMASK_MASK   = 0,  // 0 : Mask the interrupt (disable interrupt)
    GPIO_INTMASK_UNMASK = 1   // 1 : Unmask the interrupt (enable interrupt)
} GPIOIntMask_Regs;

#if (GS32_PART_NUM==0x035 || GS32_PART_NUM==0x027)

/*******************************************************************************
 *
 * Values that can be passed to GPIO_setAuxClkInSel() as the AuxClkInIO
 * parameter and returned from GPIO_getAuxClkInSel().
 *
 *******************************************************************************/
typedef enum {
	AUXCLKIN_GPIO38 = 0,
	AUXCLKIN_GPIO19 = 1
}AuxClkInSel_Regs;

/*******************************************************************************
 *
 * Values that can be passed to GPIO_setAuxClkInEn() as the value
 * parameter and returned from GPIO_getAuxClkInEn().
 *
 *******************************************************************************/
typedef enum {
	AUXCLKIN_DISABLE = 0,
	AUXCLKIN_ENABLE  = 1
}AuxClkInEn_Regs;

#endif

/*******************************************************************************
 *
 * Prototypes for the APIs.
 *
 *******************************************************************************/
/**
 * @brief    This function enables writing all registers
 *
 * @param    None
 *
 * @retval   None
 */
__STATIC_INLINE void IO_unlockAllRegisters(void)
{
    HWREG(IO_CFG_BASE + IO_CFG_PARA_O_IO_CFG_LOCK) = IO_LOCK_ENKEY;
}

/**
 * @brief   This function disables writing all registers
 *
 * @param   None
 *
 * @retval  None
 */
__STATIC_INLINE void IO_lockAllRegisters(void)
{
    HWREG(IO_CFG_BASE + IO_CFG_PARA_O_IO_CFG_LOCK) = (~IO_LOCK_ENKEY);
}

/**
 * @brief   Sets the interrupt type for the specified pin.
 *
 * @param [in] extIntNum Specifies the external interrupt. The following
 *                       defines can be used:
 *                       - GPIO_INT_XINT1
 *                       - GPIO_INT_XINT2
 *                       - GPIO_INT_XINT3
 *                       - ...
 *                       - GPIO_INT_XINT16
 *
 * @param [in] intType   Specifies the type of interrupt trigger mechanism.
 *                       One of the following flags can be used:
 *                       - GPIO_INT_TYPE_FALLING_EDGE: Sets detection to edge
 *                         and trigger to falling.
 *                       - GPIO_INT_TYPE_RISING_EDGE: Sets detection to edge
 *                         and trigger to rising.
 *                       - GPIO_INT_TYPE_BOTH_EDGES: Sets detection to both edges.
 *
 * @note    This function sets up the various interrupt trigger mechanisms for
 *          the specified pin on the selected GPIO port.
 *
 * @retval  None
 */

__STATIC_INLINE void GPIO_setInterruptType(GPIO_ExternalIntNum extIntNum, GPIO_IntType intType)
{
    //
    // Write the selected polarity to the appropriate register.
    //
    HWREGH(NMI_BASE + NMI_O_XINT_CFG(extIntNum)) =
        (HWREGH(NMI_BASE + NMI_O_XINT_CFG(extIntNum)) & ~NMI_XINT_POLARITY_M) |
        intType;
}

/**
 * @brief   Gets the interrupt type for a pin.
 *
 * @param [in] extIntNum Specifies the external interrupt. The following
 *                       defines can be used:
 *                       - GPIO_INT_XINT1
 *                       - GPIO_INT_XINT2
 *                       - GPIO_INT_XINT3
 *                       - ...
 *                       - GPIO_INT_XINT16
 *
 * @note    This function gets the interrupt type for an interrupt. The interrupt
 *          can be configured as a falling-edge, rising-edge, or both-edges detected
 *          interrupt.
 *
 * @retval  Returns one of the flags described for GPIO_setInterruptType().
 */

__STATIC_INLINE GPIO_IntType
GPIO_getInterruptType(GPIO_ExternalIntNum extIntNum)
{
    //
    // Read the selected polarity from the appropriate register.
    //
    return ((GPIO_IntType)((uint16_t)(HWREGH(NMI_BASE + NMI_O_XINT_CFG(extIntNum)) &
                                    NMI_XINT_POLARITY_M)));
}

/**
 * @brief   Enables the specified external interrupt.
 *
 * @param [in] extIntNum Specifies the external interrupt. The following
 *                       defines can be used:
 *                       - GPIO_INT_XINT1
 *                       - GPIO_INT_XINT2
 *                       - GPIO_INT_XINT3
 *                       - ...
 *                       - GPIO_INT_XINT16
 *
 * @note    This function enables the indicated external interrupt sources. Only
 *          the sources that are enabled can be reflected to the processor interrupt.
 *          Disabled sources have no effect on the processor.
 *
 * @retval  None
 */

__STATIC_INLINE void
GPIO_enableInterrupt(GPIO_ExternalIntNum extIntNum)
{
    //
    // Set the enable bit for the specified interrupt.
    //
    HWREGH(NMI_BASE + NMI_O_XINT_CFG(extIntNum)) |= NMI_XINT_ENABLE;
}

/**
 * @brief   Disables the specified external interrupt.
 *
 * @param [in] extIntNum Specifies the external interrupt. The following
 *                       defines can be used:
 *                       - GPIO_INT_XINT1
 *                       - GPIO_INT_XINT2
 *                       - GPIO_INT_XINT3
 *                       - ...
 *                       - GPIO_INT_XINT16
 *
 * @note    This function disables the indicated external interrupt sources. Only
 *          the sources that are enabled can be reflected to the processor interrupt.
 *          Disabled sources have no effect on the processor.
 *
 * @retval  None
 */

__STATIC_INLINE void
GPIO_disableInterrupt(GPIO_ExternalIntNum extIntNum)
{
    //
    // Clear the enable bit for the specified interrupt
    //
    HWREGH(NMI_BASE + NMI_O_XINT_CFG(extIntNum)) &= ~NMI_XINT_ENABLE;
}

/**
 * @brief   Gets the value of the external interrupt counter.
 *
 * @param [in] extIntNum Specifies the external interrupt. The following
 *                       defines can be used:
 *                       - GPIO_INT_XINT1
 *                       - GPIO_INT_XINT2
 *                       - GPIO_INT_XINT3
 *
 * @note    The counter is clocked at the SYSCLKOUT rate.
 *
 * @retval  Returns the external interrupt counter value.
 */
__STATIC_INLINE uint16_t
GPIO_getInterruptCounter(GPIO_ExternalIntNum extIntNum)
{
    ASSERT(extIntNum <= GPIO_INT_XINT3);

    //
    // Read the counter value from the appropriate register.
    //
    return (HWREGH(NMI_BASE + NMI_O_XINT_COUNTER(extIntNum)));
}

#if (GS32_PART_NUM==0x5000 || GS32_PART_NUM==0x4000)

/**
 * @brief 	Enable IO function for MT5000 or MT4000 models 48,49,56,57
 *
 * @param [in] None
 *
 * @retval  None
 */
__STATIC_INLINE void
SYSCTL_EnableIO_MT48_49_56_57(void)
{
	/* Enable IO function for MT5000 or MT4000 models 48,49,56,57 */
	HWREG(SYSCTL_BASE + MT5000_100PIN_ENABLE) |= 0xF;
}

#endif

#if !(GS32_PART_NUM==0x035 || GS32_PART_NUM==0x027)

/**
 * @brief   Checks that a pin number is valid for a device.
 *
 * @note    This function reflects the highest possible GPIO number of a
 *          device on its biggest package. Check the datasheet to see what
 *          the actual range of valid pin numbers is for a specific package.
 *
 * @retval  None
 */

#ifdef DEBUG
__STATIC_INLINE bool
GPIO_isPinValid(uint32_t pin)
{
    return ((pin <= 63U) || (pin == 200U) || ((pin >= 224U) && (pin <= 255U)));
}
#endif

/**
 * @brief   Reads the value present on the specified pin.
 *
 * @param [in] pin Specifies the identifying GPIO number of the pin. For example,
 *                 GPIO34 is specified by passing 34 as pin.
 *
 * @note    The value at the specified pin is read, as specified by pin. The value
 *          is returned for both input and output pins.
 *
 * @retval  Returns the value in the data register for the specified pin.
 */
__STATIC_INLINE uint32_t
GPIO_readPin(uint32_t pin)
{

#if (GS32_PART_NUM==0x5000 || GS32_PART_NUM==0x4000)

    switch (pin) {
    case 56:
        return (HWREG(IO_CFG_BASE + GPIO_O_AIODAT_RE) & 0x1000) ? 1U : 0U;
    case 57:
        return (HWREG(IO_CFG_BASE + GPIO_O_AIODAT_RE) & 0x4000) ? 1U : 0U;
    default:
        break;
    }

#endif

    volatile uint32_t *gpioDataReg;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));
    gpioDataReg = (uint32_t *)(GPIODATA_BASE + (pin / 32U) * GPIO_DATA_REGS_STEP);

    if (pin != 200) {
        return ((gpioDataReg[GPIO_GPxDAT_INDEX] >> (pin % 32U)) & (uint32_t)0x1U);
    } else {
        return (gpioDataReg[GPIO_GPxDAT_INDEX] & (uint32_t)0x1U);
    }
}

/**
 * @brief   Reads the data register value for the specified pin.
 *
 * @param [in] pin Specifies the identifying GPIO number of the pin. For example,
 *                 GPIO34 is specified by passing 34 as pin.
 *
 * @note    The value available at the data register for the specified pin is read,
 *          as specified by pin. The value is returned for both input and output pins.
 *
 * @ref     GPIO_readPin()
 *
 * @retval  Returns the value in the data register for the specified pin.
 */
__STATIC_INLINE uint32_t
GPIO_readPinDataRegister(uint32_t pin)
{
    volatile uint32_t *gpioDataReg;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));
    gpioDataReg = (uint32_t *)(GPIODATAREAD_BASE + (pin / 32U) * GPIO_DATA_READ_REGS_STEP);

    if (pin != 200) {
        return ((gpioDataReg[GPIO_GPxDAT_R_INDEX] >> (pin % 32U)) & (uint32_t)0x1U);
    } else {
        return (gpioDataReg[GPIO_GPxDAT_R_INDEX] & (uint32_t)0x1U);
    }
}

/**
 * @brief   Writes a value to the specified pin.
 *
 * @param [in] pin    Specifies the identifying GPIO number of the pin. For example,
 *                    GPIO34 is specified by passing 34 as  pin.
 * @param [in] outVal Specifies the value to write to the pin.
 *
 * @note    Writes the corresponding bit values to the output pin specified by
 *           pin. Writing to a pin configured as an input pin has no effect.
 *
 * @retval  None
 */

__STATIC_INLINE void
GPIO_writePin(uint32_t pin, uint32_t outVal)
{
    volatile uint32_t *gpioDataReg;
    uint32_t           pinMask;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

#if (GS32_PART_NUM==0x5000 || GS32_PART_NUM==0x4000)
    switch (pin) {
    case 56:
        if (outVal == 0U) {
            HWREG(DSP_APB_CFG_BASE + DIG_SYS_SPARE0) &= 0xFFFFFFFE;
        } else {
            HWREG(DSP_APB_CFG_BASE + DIG_SYS_SPARE0) |= 0x1;
        }
        return;
    case 57:
        if (outVal == 0U) {
            HWREG(DSP_APB_CFG_BASE + DIG_SYS_SPARE0) &= 0xFFFFFFFD;
        } else {
            HWREG(DSP_APB_CFG_BASE + DIG_SYS_SPARE0) |= 0x2;
        }
        return;
    default:
        break;
    }
#endif // (GS32_PART_NUM==0x5000 || GS32_PART_NUM==0x4000)


    gpioDataReg = (uint32_t *)(GPIODATA_BASE +
                            (pin / 32U) * GPIO_DATA_REGS_STEP);

    if (pin != 200) {
        pinMask = (uint32_t)1U << (pin % 32U);
    } else {
        pinMask = (uint32_t)1U << (0U);
    }

    if (outVal == 0U) {
        gpioDataReg[GPIO_GPxCLEAR_INDEX] = pinMask;
    } else {
        gpioDataReg[GPIO_GPxSET_INDEX] = pinMask;
    }
}

/**
 * @brief   Toggles the specified pin.
 *
 * @param [in] pin Specifies the identifying GPIO number of the pin. For example,
 *                 GPIO34 is specified by passing 34 as  pin.
 *
 * @note    Writes the corresponding bit values to the output pin specified by
 *           pin. Writing to a pin configured as an input pin has no effect.
 *
 * @retval  None
 */
__STATIC_INLINE void
GPIO_togglePin(uint32_t pin)
{
    volatile uint32_t *gpioDataReg;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));
#if (GS32_PART_NUM==0x5000 || GS32_PART_NUM==0x4000)
    switch (pin) {
    case 56:
        HWREG(DSP_APB_CFG_BASE + DIG_SYS_SPARE0) ^= 0x1;
        return;
    case 57:
        HWREG(DSP_APB_CFG_BASE + DIG_SYS_SPARE0) ^= 0x2;
        return;
    default:
        break;
    }
#endif //(GS32_PART_NUM==0x5000 || GS32_PART_NUM==0x4000)
    gpioDataReg = (uint32_t *)(GPIODATA_BASE +
                            (pin / 32U) * GPIO_DATA_REGS_STEP);
    if (pin != 200) {
        gpioDataReg[GPIO_GPxTOGGLE_INDEX] = (uint32_t)1U << (pin % 32U);
    } else {
        gpioDataReg[GPIO_GPxTOGGLE_INDEX] = (uint32_t)1U;
    }
}

/**
 * @brief   Reads the data on the specified port.
 *
 * @param [in] port Specifies the GPIO port being accessed in the form of
 *                   GPIO_PORT_X, where X is the port letter.
 *
 * @retval  Returns the value available on the pins for the specified port. Each
 *          bit of the return value represents a pin on the port, where bit 0
 *          represents GPIO port pin 0, bit 1 represents GPIO port pin 1, and so on.
 */
__STATIC_INLINE uint32_t
GPIO_readPortData(GPIO_Port port)
{
    volatile uint32_t *gpioDataReg;

    //
    // Get the starting address of the port's registers and return DATA.
    //
    gpioDataReg = (uint32_t *)(GPIODATA_BASE + ((uint32_t)port * GPIO_DATA_REGS_STEP));

    return (gpioDataReg[GPIO_GPxDAT_INDEX]);
}

/**
 * @brief   Reads the data written in the GPIO Data Register.
 *
 * @param [in] port Specifies the GPIO port being accessed in the form of
 *                   GPIO_PORT_X, where X is the port letter.
 *
 * @note    Reads the data written in the GPIO Data Register for the specified port.
 *          In previous devices, reading the GPIO data registers resulted in reading
 *          the corresponding pins. The function  GPIO_readPortData() returns the
 *          value on the pins.
 *
 * @see     GPIO_readPortData()
 *
 * @retval  Returns the value in the data register for the specified port. Each
 *          bit of the return value represents a pin on the port, where bit 0
 *          represents GPIO port pin 0, bit 1 represents GPIO port pin 1, and so on.
 */
__STATIC_INLINE uint32_t
GPIO_readPortDataRegister(GPIO_Port port)
{
    volatile uint32_t *gpioDataReg;

    //
    // Get the starting address of the port's registers and return DATA.
    //
    gpioDataReg = (uint32_t *)(GPIODATAREAD_BASE + ((uint32_t)port * GPIO_DATA_READ_REGS_STEP));

    return (gpioDataReg[GPIO_GPxDAT_R_INDEX]);
}

/**
 * @brief   Writes a value to the specified port.
 *
 * @param [in] port   Specifies the GPIO port being accessed. The parameter takes
 *                    a value in the form of  GPIO_PORT_X, where X is the port letter.
 *                    For example, use  GPIO_PORT_A to affect port A (GPIOs 0-31).
 * @param [in] outVal Specifies the value to write to the port. This is a bit-packed
 *                    value, where each bit represents a pin on the GPIO port.
 *                    Bit 0 represents GPIO port pin 0, bit 1 represents GPIO port pin 1,
 *                    and so on.
 *
 * @retval  None
 */
__STATIC_INLINE void
GPIO_writePortData(GPIO_Port port, uint32_t outVal)
{
    volatile uint32_t *gpioDataReg;

    //
    // Get the starting address of the port's registers and write to DATA.
    //
    gpioDataReg                    = (uint32_t *)(GPIODATA_BASE +
                            ((uint32_t)port * GPIO_DATA_REGS_STEP));
    gpioDataReg[GPIO_GPxDAT_INDEX] = outVal;
}

/**
 * @brief   Sets all of the specified pins on the specified port.
 *
 * @param [in] port    Specifies the GPIO port being accessed. The parameter takes
 *                     a value in the form of GPIO_PORT_X, where X is the port letter.
 *                     For example, use GPIO_PORT_A to affect port A (GPIOs 0-31).
 * @param [in] pinMask Specifies a mask of which of the 32 pins on the port are affected.
 *                     This is a bit-packed value, where each bit that is set identifies
 *                     the pin to be set. Bit 0 represents GPIO port pin 0, bit 1 represents
 *                     GPIO port pin 1, and so on.
 *
 * @retval  None
 */
__STATIC_INLINE void
GPIO_setPortPins(GPIO_Port port, uint32_t pinMask)
{
    volatile uint32_t *gpioDataReg;

    //
    // Get the starting address of the port's registers and write to SET.
    //
    gpioDataReg = (uint32_t *)(GPIODATA_BASE +
                            ((uint32_t)port * GPIO_DATA_REGS_STEP));

    gpioDataReg[GPIO_GPxSET_INDEX] = pinMask;
}

/**
 * @brief   Clears all of the specified pins on the specified port.
 *
 * @param [in] port    Specifies the GPIO port being accessed. The parameter takes
 *                     a value in the form of GPIO_PORT_X, where X is the port letter.
 *                     For example, use GPIO_PORT_A to affect port A (GPIOs 0-31).
 * @param [in] pinMask Specifies a mask of which of the 32 pins on the port are affected.
 *                     This is a bit-packed value, where each bit that is set identifies
 *                     the pin to be cleared. Bit 0 represents GPIO port pin 0, bit 1 represents
 *                     GPIO port pin 1, and so on.
 *
 * @retval  None
 */
__STATIC_INLINE void
GPIO_clearPortPins(GPIO_Port port, uint32_t pinMask)
{
    volatile uint32_t *gpioDataReg;

    //
    // Get the starting address of the port's registers and write to CLEAR.
    //
    gpioDataReg = (uint32_t *)(GPIODATA_BASE +
                            ((uint32_t)port * GPIO_DATA_REGS_STEP));

    gpioDataReg[GPIO_GPxCLEAR_INDEX] = pinMask;
}

/**
 * @brief   Toggles all of the specified pins on the specified port.
 *
 * @param [in] port    Specifies the GPIO port being accessed. The parameter takes
 *                     a value in the form of GPIO_PORT_X, where X is the port letter.
 *                     For example, use GPIO_PORT_A to affect port A (GPIOs 0-31).
 * @param [in] pinMask Specifies a mask of which of the 32 pins on the port are affected.
 *                     This is a bit-packed value, where each bit that is set identifies
 *                     the pin to be toggled. Bit 0 represents GPIO port pin 0, bit 1 represents
 *                     GPIO port pin 1, and so on.
 *
 * @retval  None
 */
__STATIC_INLINE void
GPIO_togglePortPins(GPIO_Port port, uint32_t pinMask)
{
    volatile uint32_t *gpioDataReg;

    //
    // Get the starting address of the port's registers and write to TOGGLE.
    //
    gpioDataReg = (uint32_t *)(GPIODATA_BASE +
                            ((uint32_t)port * GPIO_DATA_REGS_STEP));

    gpioDataReg[GPIO_GPxTOGGLE_INDEX] = pinMask;
}

/**
 * @brief   Determine whether the pin type is AIO or AGPIO
 * @param pin_num Pin number
 * @return  HW_PIN_TYPE_AIO: AIO pin
 *          HW_PIN_TYPE_AGPIO: AGPIO pin
 *          HW_PIN_TYPE_GPIO: GPIO pin
 *          HW_PIN_TYPE_UNKNOWN: Invalid pin
 */
extern HwPinType
check_pin_type(int pin_num);

/**
 * @brief   Sets the direction and mode of the specified pin.
 *
 * @param [in] pin   Specifies the identifying GPIO number of the pin. For example,
 *                   GPIO34 is specified by passing 34 as pin.
 * @param [in] pinIO Specifies the pin direction mode. This parameter is an enumerated
 *                   data type that can take one of the following values:
 *                   - GPIO_DIR_MODE_IN: Configures the pin as an input.
 *                   - GPIO_DIR_MODE_OUT: Configures the pin as an output.
 *
 * @note    This function configures the specified pin on the selected GPIO port
 *          as either input or output.
 *
 * @retval  None
 */
extern void
GPIO_setDirectionMode(uint32_t pin, GPIO_Direction pinIO);

__STATIC_INLINE void
GPIO_disableWritePin(uint32_t pin)
{
    GPIO_setDirectionMode(pin, GPIO_DIR_MODE_IN);
}

__STATIC_INLINE void
GPIO_enableWritePin(uint32_t pin)
{
    GPIO_setDirectionMode(pin, GPIO_DIR_MODE_OUT);
}

/**
 * @brief   Gets the direction mode of a specified pin.
 *
 * @param [in] pin Specifies the identifying GPIO number of the pin.
 *
 * @note    This function retrieves the direction mode of the specified pin, which
 *          can be configured as either an input or output. The direction type is
 *          returned as an enumerated data type.
 *
 * @retval  Returns one of the enumerated data types described for GPIO_setDirectionMode().
 */
extern GPIO_Direction
GPIO_getDirectionMode(uint32_t pin);

/**
 * @brief   Sets the pin for the specified external interrupt.
 *
 * @param [in] pin       Specifies the identifying GPIO number of the pin. For example,
 *                       GPIO34 is specified by passing 34 as pin.
 * @param [in] extIntNum Specifies the external interrupt to be triggered by the pin.
 *                       The following defines can be used to specify this parameter:
 *                       - GPIO_INT_XINT1
 *                       - GPIO_INT_XINT2
 *                       - GPIO_INT_XINT3
 *                       - ...
 *                       - GPIO_INT_XINT16
 *
 * @note    This function sets which pin triggers the selected external interrupt.
 *
 * @ref      XBAR_setInputPin()
 *
 * @retval  None
 */
extern void
GPIO_setInterruptPin(uint32_t pin, GPIO_ExternalIntNum extIntNum);

/**
 * @brief   Sets the pad configuration for the specified pin.
 *
 * @param [in] pin     Specifies the identifying GPIO number of the pin. For example,
 *                     GPIO34 is specified by passing 34 as  pin.
 * @param [in] pinType Specifies the pin type configuration. The  pinType parameter
 *                     can take the following values:
 *                     -  GPIO_PIN_TYPE_STD: Configures the pin as a push-pull output
 *                       or a floating input.
 *                     -  GPIO_PIN_TYPE_PULLUP: Enables the pull-up resistor for an input.
 *                     -  GPIO_PIN_TYPE_INVERT: Enables inverted polarity on an input.
 *                     -  GPIO_PIN_TYPE_PULLDOWN  Pull-Down Input
 *                     -  GPIO_PIN_TYPE_SCHMITT  Inverted Push-Pull Output
 *
 * @retval  None
 */
extern void
GPIO_setPadConfig(uint32_t pin, uint32_t pinType);

/**
 * @brief   Gets the pad configuration for a specified pin.
 *
 * @param [in] pin Specifies the identifying GPIO number of the pin.
 *
 * @note    This function returns the pin type configuration for the specified pin.
 *          The returned value corresponds to the values used in GPIO_setPadConfig().
 *
 * @retval  Returns a bit field containing one or more of the following values:
 *          - GPIO_PIN_TYPE_STD
 *          - GPIO_PIN_TYPE_PULLUP
 *          - GPIO_PIN_TYPE_INVERT
 *          - GPIO_PIN_TYPE_PULLDOWN
 *          - GPIO_PIN_TYPE_SCHMITT
 */
extern uint32_t
GPIO_getPadConfig(uint32_t pin);

/**
 * @brief   Sets the qualification mode for the specified pin.
 *
 * @param [in] pin            Specifies the identifying GPIO number of the pin.
 * @param [in] qualification  Specifies the qualification mode for the pin. This
 *                            parameter can take one of the following values:
 *                            -  GPIO_QUAL_SYNC: Synchronous mode.
 *                            -  GPIO_QUAL_3SAMPLE: 3-sample qualification mode.
 *                            -  GPIO_QUAL_6SAMPLE: 6-sample qualification mode.
 *                            -  GPIO_QUAL_ASYNC: Asynchronous mode.
 *
 * @note    To configure the qualification sampling period, use
 *           GPIO_setQualificationPeriod().
 *
 * @retval  None
 */
extern void
GPIO_setQualificationMode(uint32_t pin, GPIO_QualificationMode qualification);

/**
 * @brief   Gets the qualification type for the specified pin.
 *
 * @param [in] pin Specifies the identifying GPIO number of the pin.
 *
 * @retval  Returns the qualification mode as one of the following values:
 *          -  GPIO_QUAL_SYNC: Synchronous mode.
 *          -  GPIO_QUAL_3SAMPLE: 3-sample qualification mode.
 *          -  GPIO_QUAL_6SAMPLE: 6-sample qualification mode.
 *          -  GPIO_QUAL_ASYNC: Asynchronous mode.
 */
extern GPIO_QualificationMode
GPIO_getQualificationMode(uint32_t pin);

/**
 * @brief   Sets the qualification period for a group of pins.
 *
 * @param [in] pin      Specifies the identifying GPIO number of the pin. This sets the
 *                      qualification period for a group of 8 pins. For example:
 *                      - Passing 3 as pin configures GPIO0 through GPIO7.
 *                      - Passing 98 as pin configures GPIO96 through GPIO103.
 *                      This is because the register field that configures the divider
 *                      is shared among 8 pins.
 *
 * @param [in] divider  Specifies the value by which the frequency of SYSCLKOUT is divided.
 *                      The value can be:
 *                      - 1
 *                      - Any even value between 2 and 510 (inclusive).
 *
 * @note    The range of pins configured is determined by the equation:
 *           GPIO (pin & ~(7)) through GPIO ((pin & ~(7)) + 7).
 *
 * @retval  None
 */
extern void
GPIO_setQualificationPeriod(uint32_t pin, uint32_t divider);

/**
 * @brief   Sets the analog mode of the specified pin.
 *
 * @param [in] pin  Specifies the identifying GPIO number of the pin. For example,
 *                  GPIO34 is specified by passing 34 as pin.
 * @param [in] mode Specifies the selected analog mode for the pin. This parameter
 *                  can take one of the following values:
 *                  - GPIO_ANALOG_DISABLED: Configures the pin in digital mode.
 *                  - GPIO_ANALOG_ENABLED: Configures the pin in analog mode.
 *
 * @note    Not all GPIO pins support switching to analog mode. Refer to the
 *          technical reference manual for specific details. This configuration
 *          acts as an additional level of muxing.
 *
 * @note    The pin parameter applies to both AIO and GPIO pins because the
 *          GPAxMSEL.GPIOy register configures both types.
 *
 * @retval  None
 */
extern void
GPIO_setAnalogMode(uint32_t pin, GPIO_AnalogMode mode);

/**
 * @brief   Configures the alternate function of a GPIO pin.
 *
 * @param [in] pinConfig Specifies the pin configuration value. This parameter
 *                       must be one of the GPIO_#_???? values defined in
 *                       <tt>pinmap.h</tt>.
 *
 * @note    This function configures the pin mux to select the peripheral function
 *          associated with the specified GPIO pin. Only one peripheral function
 *          can be associated with a GPIO pin at a time. Additionally, each peripheral
 *          function should be assigned to a single GPIO pin, even though many
 *          functions can be associated with multiple GPIO pins.
 *
 * @retval  None
 */
extern void
GPIO_setPinConfig(uint32_t pinConfig);

/**
 * @brief   Configures the output drive strength for the specified GPIO pin.
 *
 * @param [in] pin           Specifies the identifying GPIO number of the pin.
 * @param [in] StrengthLevel Specifies the desired drive strength level. The
 *                           available levels depend on the type of GPIO:
 *                           - For I2C GPIO:
 *                             - GPIO_DRV_4MA
 *                             - GPIO_DRV_20MA
 *                           - For standard GPIO:
 *                             - GPIO_DRV_2MA
 *                             - GPIO_DRV_4MA
 *                             - GPIO_DRV_8MA
 *                             - GPIO_DRV_12MA
 *
 * @note    Ensure the drive strength level is suitable for the intended application
 *          and complies with the device's technical reference manual.
 *
 * @retval  None
 */
extern void
GPIO_setStrength(uint32_t pin, GPIO_DriveStrength StrengthLevel);

/**
 * @brief   Configures the Schmitt trigger for the specified GPIO pin.
 *
 * @param [in] pin      Specifies the identifying GPIO number of the pin.
 * @param [in] pinType  Specifies whether the Schmitt trigger is enabled or disabled.
 *                      This parameter can be:
 *                      - true: Enable Schmitt trigger.
 *                      - false: Disable Schmitt trigger.
 *
 * @note    The Schmitt trigger improves noise immunity for input signals. Refer to
 *          the device's technical reference manual for additional details.
 *
 * @retval  None
 */
extern void
GPIO_setSchmittTrigger(uint32_t pin, bool pinType);

/**
 * @brief   Configures the output polarity inversion for the specified GPIO pin.
 *
 * @param [in] pin      Specifies the identifying GPIO number of the pin.
 * @param [in] pinType  Specifies whether to enable or disable output polarity inversion.
 *                      This parameter can be:
 *                      - true: Enable output polarity inversion (inverted logic).
 *                      - false: Disable output polarity inversion (normal logic).
 *
 * @note    Enabling output polarity inversion causes the output signal to be inverted.
 *          Refer to the device's technical reference manual for implementation details.
 *
 * @retval  None
 */
extern void
GPIO_setOutPolarityInv(uint32_t pin, bool pinType);

/**
 * @brief   Configures the loopback mode for the specified GPIO pin.
 *
 * @param [in] pin      Specifies the identifying GPIO number of the pin.
 * @param [in] pinType  Specifies whether to enable or disable loopback mode.
 *                      This parameter can be:
 *                      - true: Enable loopback mode.
 *                      - false: Disable loopback mode.
 *
 * @note    Loopback mode allows the output signal of the pin to be internally
 *          connected to its input for testing or diagnostic purposes. Refer to
 *          the device's technical reference manual for further details.
 *
 * @retval  None
 */
extern void
GPIO_setLoopBack(uint32_t pin, bool pinType);

/**
 * @brief   Configures the I2C mode for the specified GPIO pin.
 *
 * @param [in] pin      Specifies the identifying GPIO number of the pin.
 * @param [in] pinType  Specifies whether to enable or disable I2C mode.
 *                      This parameter can be:
 *                      - true: Enable I2C mode.
 *                      - false: Disable I2C mode.
 *
 * @note    Enabling I2C mode configures the pin for I2C communication. Ensure the
 *          pin supports I2C functionality and that other related configurations
 *          are set appropriately. Refer to the device's technical reference manual
 *          for details.
 *
 * @retval  None
 */
extern void
GPIO_setI2CMode(uint32_t pin, bool pinType);

/**
 * @brief   Reads the output state of the specified GPIO pin.
 *
 * @param [in] pin  Specifies the identifying GPIO number of the pin.
 *
 * @return  The current output state of the pin:
 *          - 0: The pin output is low.
 *          - 1: The pin output is high.
 *
 * @note    This function retrieves the logical output state of the pin, not the
 *          physical level. Ensure the pin is configured as an output before
 *          using this function.
 */
extern uint32_t
GPIO_readPinOutReport(uint32_t pin);

/**
 * @brief   Reads the output state of all pins in the specified GPIO port.
 *
 * @param [in] port  Specifies the 32-bit GPIO port identifier.
 *
 * @return  A 32-bit value representing the output states of all pins in the port:
 *          - Each bit corresponds to a pin in the port.
 *          - 0: The pin output is low.
 *          - 1: The pin output is high.
 *
 * @note    This function retrieves the logical output states of the port's pins,
 *          not their physical levels. Ensure the pins in the port are configured
 *          as outputs before using this function.
 */
extern uint32_t
GPIO_readPortOutReport(GPIO_Port_32BIT port);

/**
 * @brief   Configures the interrupt type and polarity for the specified GPIO pin.
 *
 * @param [in] pin               Specifies the identifying GPIO number of the pin
 * @param [in] intTypePolarity   Specifies the interrupt type and polarity for the GPIO pin.
 *                               This parameter can be one of the following:
 *                               - GPIO_INTERRUPT_TYPE_LEVEL_LOW (0): Level-sensitive interrupt (low level).
 *                               - GPIO_INTERRUPT_TYPE_LEVEL_HIGH (1): Level-sensitive interrupt (high level).
 *                               - GPIO_INTERRUPT_TYPE_EDGE_RISE (2): Edge-sensitive interrupt (rising edge).
 *                               - GPIO_INTERRUPT_TYPE_EDGE_FALL (3): Edge-sensitive interrupt (falling edge).
 *
 * @note    The interrupt configuration depends on both the type (level or edge) and the polarity
 *          (high or low level, rising or falling edge). For example:
 *          - Level-sensitive interrupt: Determines the state of the pin (high or low) when the interrupt is triggered.
 *          - Edge-sensitive interrupt: Triggered by a transition from low to high (rising edge) or high to low (falling edge).
 *          The corresponding interrupt polarity is also determined by the `intTypePolarity` setting.
 *
 * @retval  None
 */
extern void
GPIO_setInterruptType_NO_XInt(uint32_t pin, GPIO_InterruptType_NO_XInt intTypePolarity);

/**
 * @brief   Retrieves the interrupt type and polarity configuration for the specified GPIO pin.
 *
 * @param [in] pin  Specifies the identifying GPIO number of the pin
 *
 * @return  The current interrupt type and polarity configuration for the pin, which can be one of the following:
 *          - GPIO_INTERRUPT_TYPE_LEVEL_LOW (0): Level-sensitive interrupt (low level).
 *          - GPIO_INTERRUPT_TYPE_LEVEL_HIGH (1): Level-sensitive interrupt (high level).
 *          - GPIO_INTERRUPT_TYPE_EDGE_RISE (2): Edge-sensitive interrupt (rising edge).
 *          - GPIO_INTERRUPT_TYPE_EDGE_FALL (3): Edge-sensitive interrupt (falling edge).
 *
 * @note    This function allows you to query the interrupt configuration for a specific pin,
 *          including whether it is level or edge-sensitive and its polarity (low/high level or rise/fall edge).
 */
extern GPIO_InterruptType_NO_XInt
GPIO_getInterruptType_NO_XInt(uint32_t pin);

/**
 * @brief   Enables the interrupt for the specified GPIO pin.
 *
 * @param [in] pin  Specifies the identifying GPIO number of the pin
 *
 * @note    This function unmasks the interrupt for the specified GPIO pin, allowing the interrupt to trigger
 *          when the configured conditions (type and polarity) are met. The interrupt will be handled according
 *          to the settings defined for the pin using `GPIO_setInterruptType_NO_XInt`.
 *
 * @retval  None
 */
extern void
GPIO_enableInterrupt_NO_XInt(uint32_t pin);

/**
 * @brief   Disables the interrupt for the specified GPIO pin.
 *
 * @param [in] pin  Specifies the identifying GPIO number of the pin
 *
 * @note    This function masks the interrupt for the specified GPIO pin, preventing the interrupt from triggering
 *          even if the configured interrupt conditions (type and polarity) are met.
 *
 * @retval  None
 */
extern void
GPIO_disableInterrupt_NO_XInt(uint32_t pin);

/**
 * @brief   Retrieves the interrupt mask state for the specified GPIO pin.
 *
 * @param [in] pin  Specifies the identifying GPIO number of the pin
 *
 * @return  A boolean value indicating the interrupt mask state of the specified pin:
 *          - true: The interrupt for the pin is unmasked (enabled).
 *          - false: The interrupt for the pin is masked (disabled).
 *
 * @note    This function reads the corresponding bit in the `GPG_INTMASK` register to determine
 *          the interrupt mask state for the specified GPIO pin.
 *
 * @note    Ensure that the pin value corresponds to the valid range for GPIO pins
 *          associated with the `GPG_INTMASK` register
 */
extern bool
GPIO_getInterruptState_NO_XInt(uint32_t pin);

/**
 * @brief   Clears the interrupt state for a specified GPIO pin
 *
 * @details This function clears the interrupt for a specific GPIO pin by writing a `1` to the
 *          corresponding bit in the `GPB_INTCLR` register. The `GPB_INTCLR` register is a
 *          Write-1-to-Set-Reset (W1SR0) type, meaning only bits written with `1` will be affected.
 *
 * @param [in] pin  Specifies the GPIO pin number to clear the interrupt for.
 *
 * @return  This function does not return a value.
 */
extern void
GPIO_clearInterrupt_NO_XInt(uint32_t pin);

/**
 * @brief   Reads the raw interrupt status for a specified GPIO pin
 *
 * @details This function reads the `GPB_INTRAW` register to determine if a raw interrupt
 *          has been generated for a specific GPIO pin. The `GPB_INTRAW` register is read-only
 *          and indicates the raw interrupt signal before it is masked or processed.
 *
 * @note    A value of `1` indicates that an interrupt has been generated for the specified pin.
 *          A value of `0` indicates no interrupt has been generated. Interrupts are level signals
 *          and are cleared using the `GPB_INTCLR` register.
 *
 * @param [in] pin  Specifies the GPIO pin number to check the raw interrupt status for.
 *
 * @return  Returns `true` if an interrupt has been generated for the specified pin, `false` otherwise.
 *
 */
extern bool
GPIO_readInterruptRaw_NO_XInt(uint32_t pin);

/**
 * @brief   Reads the interrupt status for a specified GPIO pin
 *
 * @details This function reads the `GPB_INTSTATUS` register to determine if an interrupt
 *          has been generated for a specific GPIO pin. The `GPB_INTSTATUS` register reflects
 *          the masked interrupt status after processing the raw interrupt signal.
 *
 * @note    A value of `1` indicates that an interrupt has been generated for the specified pin.
 *          A value of `0` indicates no interrupt has been generated. Interrupts are level signals
 *          and are cleared using the `GPB_INTCLR` register.
 *
 * @param [in] pin  Specifies the GPIO pin number to check the interrupt status for.
 *
 * @return  Returns `true` if an interrupt has been generated for the specified pin, `false` otherwise.
 */
extern bool
GPIO_readInterruptStatus_NO_XInt(uint32_t pin);

#else

/**
 * @brief   Configures the alternate function of a GPIO pin.
 *
 * @param [in] pin
 *
 * @param [in] pinConfig Specifies the pin configuration value. This parameter
 *                       must be one of the GPIO_#_???? values defined in
 *                       <tt>pinmap.h</tt>.
 *
 * @note    This function configures the pin mux to select the peripheral function
 *          associated with the specified GPIO pin. Only one peripheral function
 *          can be associated with a GPIO pin at a time. Additionally, each peripheral
 *          function should be assigned to a single GPIO pin, even though many
 *          functions can be associated with multiple GPIO pins.
 *
 * @retval  None
 */
extern void
GPIO_setPinConfig(uint32_t pinConfig);

/**
 * @brief   Sets the pad configuration for the specified pin.
 *
 * @param [in] pin     Specifies the identifying GPIO number of the pin. For example,
 *                     GPIO34 is specified by passing 34 as  pin.
 * @param [in] pinType Specifies the pin type configuration. The  pinType parameter
 *                     can take the following values:
 *                     -  GPIO_PIN_TYPE_STD: Configures the pin as a push-pull output
 *                       or a floating input.
 *                     -  GPIO_PIN_TYPE_PULLUP: Enables the pull-up resistor for an input.
 *                     -  GPIO_PIN_TYPE_PULLDOWN  Pull-Down Input
 *                     -  GPIO_PIN_TYPE_OUTPUT_INVERT  Inverted Push-Pull Output
 *
 * @retval  None
 */
extern void
GPIO_setPadConfig(uint32_t pin, uint32_t pinType);

/**
 * @brief   Gets the pad configuration for a specified pin.
 *
 * @param [in] pin Specifies the identifying GPIO number of the pin.
 *
 * @note    This function returns the pin type configuration for the specified pin.
 *          The returned value corresponds to the values used in GPIO_setPadConfig().
 *
 * @retval  Returns a bit field containing one or more of the following values:
 *          - GPIO_PIN_TYPE_STD
 *          - GPIO_PIN_TYPE_PULLUP
 *          - GPIO_PIN_TYPE_PULLDOWN
 *          - GPIO_PIN_TYPE_OUTPUT_INVERT
 */
extern uint32_t
GPIO_getPadConfig(uint32_t pin);

/**
 * @brief Set the auxiliary clock input signal source (GPIO pin selection).
 *
 * This function configures the signal source for the auxiliary clock input (auxclkin)
 * by selecting between GPIO38 and GPIO19. It is applicable for model 035 (where GPIO38/19
 * are used) or model 027. The selection is made by writing to the AUXCLKIN_SEL register.
 *
 * @param [in] AuxClkInIO The GPIO pin to select as the auxclkin signal source.
 *                        Use AUXCLKIN_GPIO38 for GPIO38 or AUXCLKIN_GPIO19 for GPIO19.
 *
 * @retval None
 */
__STATIC_INLINE void
GPIO_setAuxClkInSel(AuxClkInSel_Regs AuxClkInIO)
{
	HWREG(IO_CFG_BASE + GPIO_O_AUXCLKIN_SEL) = AuxClkInIO;
}

/**
 * @brief Get the current auxiliary clock input signal source (GPIO pin selection).
 *
 * This function retrieves the currently configured signal source for the auxiliary
 * clock input (auxclkin), indicating whether GPIO38 or GPIO19 is selected. It is
 * applicable for model 035 (where GPIO38/19 are used) or model 027. The value is read
 * from the AUXCLKIN_SEL register.
 *
 * @param [in] None
 *
 * @retval AuxClkInSel_Regs The current auxclkin signal source.
 *                          Returns AUXCLKIN_GPIO38 if GPIO38 is selected,
 *                          or AUXCLKIN_GPIO19 if GPIO19 is selected.
 */
__STATIC_INLINE AuxClkInSel_Regs
GPIO_getAuxClkInSel(void)
{
	return(HWREG(IO_CFG_BASE + GPIO_O_AUXCLKIN_SEL));
}

/**
 * @brief Enable or disable the auxiliary clock input (auxclkin).
 *
 * This function enables or disables the auxiliary clock input (auxclkin) by writing
 * to the AUXCLKIN_EN register. This functionality is applicable for model 035; for
 * other models, this setting has no effect.
 *
 * @param [in] value The enable/disable state for auxclkin.
 *                   Use AUXCLKIN_ENABLE to enable auxclkin, or AUXCLKIN_DISABLE to disable it.
 *
 * @retval None
 */
__STATIC_INLINE void
GPIO_setAuxClkInEn(AuxClkInEn_Regs value)
{
	HWREG(IO_CFG_BASE + GPIO_O_AUXCLKIN_EN) = value;
}

/**
 * @brief Get the current enable/disable state of the auxiliary clock input (auxclkin).
 *
 * This function retrieves the current enable/disable state of the auxiliary clock input
 * (auxclkin) by reading from the AUXCLKIN_EN register. This functionality is applicable
 * for model 035; for other models, this setting has no effect.
 *
 * @param [in] None
 *
 * @retval AuxClkInEn_Regs The current enable/disable state of auxclkin.
 *                         Returns AUXCLKIN_ENABLE if auxclkin is enabled,
 *                         or AUXCLKIN_DISABLE if auxclkin is disabled.
 */
__STATIC_INLINE AuxClkInEn_Regs
GPIO_getAuxClkInEn(void)
{
	return(HWREG(IO_CFG_BASE + GPIO_O_AUXCLKIN_EN));
}

/**
 * @brief   Configures the output drive strength for the specified GPIO pin.
 *
 * @param [in] pin           Specifies the identifying GPIO number of the pin.
 * @param [in] StrengthLevel Specifies the desired drive strength level. The
 *                           available levels depend on the type of GPIO:
 *                           - For I2C GPIO:
 *                             - GPIO_DRV_4MA
 *                             - GPIO_DRV_20MA
 *                           - For standard GPIO:
 *                             - GPIO_DRV_2MA
 *                             - GPIO_DRV_4MA
 *                             - GPIO_DRV_8MA
 *                             - GPIO_DRV_12MA
 *
 * @note    Ensure the drive strength level is suitable for the intended application
 *          and complies with the device's technical reference manual.
 *
 * @retval  None
 */
extern void
GPIO_setStrength(uint32_t pin, GPIO_DriveStrength StrengthLevel);

/**
 * @brief   Configures the Schmitt trigger for the specified GPIO pin.
 *
 * @param [in] pin      Specifies the identifying GPIO number of the pin.
 * @param [in] pinType  Specifies whether the Schmitt trigger is enabled or disabled.
 *                      This parameter can be:
 *                      - true: Enable Schmitt trigger.
 *                      - false: Disable Schmitt trigger.
 *
 * @note    The Schmitt trigger improves noise immunity for input signals. Refer to
 *          the device's technical reference manual for additional details.
 *
 * @retval  None
 */
extern void
GPIO_setSchmittTrigger(uint32_t pin, bool pinType);

/**
 * @brief   Configures the output polarity inversion for the specified GPIO pin.
 *
 * @param [in] pin      Specifies the identifying GPIO number of the pin.
 * @param [in] pinType  Specifies whether to enable or disable output polarity inversion.
 *                      This parameter can be:
 *                      - true: Enable output polarity inversion (inverted logic).
 *                      - false: Disable output polarity inversion (normal logic).
 *
 * @note    Enabling output polarity inversion causes the output signal to be inverted.
 *          Refer to the device's technical reference manual for implementation details.
 *
 * @retval  None
 */
extern void
GPIO_setOutPolarityInv(uint32_t pin, bool pinType);

/**
 * @brief   Configures the loopback mode for the specified GPIO pin.
 *
 * @param [in] pin      Specifies the identifying GPIO number of the pin.
 * @param [in] pinType  Specifies whether to enable or disable loopback mode.
 *                      This parameter can be:
 *                      - true: Enable loopback mode.
 *                      - false: Disable loopback mode.
 *
 * @note    Loopback mode allows the output signal of the pin to be internally
 *          connected to its input for testing or diagnostic purposes. Refer to
 *          the device's technical reference manual for further details.
 *
 * @retval  None
 */
extern void
GPIO_setLoopBack(uint32_t pin, bool pinType);

/**
 * @brief   Configures the I2C mode for the specified GPIO pin.
 *
 * @param [in] pin      Specifies the identifying GPIO number of the pin.
 * @param [in] pinType  Specifies whether to enable or disable I2C mode.
 *                      This parameter can be:
 *                      - true: Enable I2C mode.
 *                      - false: Disable I2C mode.
 *
 * @note    Enabling I2C mode configures the pin for I2C communication. Ensure the
 *          pin supports I2C functionality and that other related configurations
 *          are set appropriately. Refer to the device's technical reference manual
 *          for details.
 *
 * @retval  None
 */
extern void
GPIO_setI2CMode(uint32_t pin, bool pinType);

/**
 * @brief   Reads the output state of the specified GPIO pin.
 *
 * @param [in] pin  Specifies the identifying GPIO number of the pin.
 *
 * @return  The current output state of the pin:
 *          - 0: The pin output is low.
 *          - 1: The pin output is high.
 *
 * @note    This function retrieves the logical output state of the pin, not the
 *          physical level. Ensure the pin is configured as an output before
 *          using this function.
 */
extern uint32_t
GPIO_readPinOutReport(uint32_t pin);

/**
 * @brief   Reads the output state of all pins in the specified GPIO port.
 *
 * @param [in] port  Specifies the 32-bit GPIO port identifier.
 *
 * @return  A 32-bit value representing the output states of all pins in the port:
 *          - Each bit corresponds to a pin in the port.
 *          - 0: The pin output is low.
 *          - 1: The pin output is high.
 *
 * @note    This function retrieves the logical output states of the port's pins,
 *          not their physical levels. Ensure the pins in the port are configured
 *          as outputs before using this function.
 */
extern uint32_t
GPIO_readPortOutReport(GPIO_Port_32BIT port);


#endif /* !(GS32_PART_NUM==0x035 || GS32_PART_NUM==0x027) */



/**
 * @brief   Configures the reference clock mode for the GPIO.
 *
 * @details This function sets the reference clock mode by writing to the `CLK_REF_PIN_CTRL` register.
 *          The `CLK_REF_PIN_CTRL` register controls the crystal or oscillator type, frequency range,
 *          and feedback resistor selection for the reference clock input.
 *
 * @param [in] mode  Specifies the reference clock configuration. It should be one of the values from
 *                   the `GPIO_ClkRefMode` enumeration.
 *
 * @note    The `GPIO_ClkRefMode` enumeration provides options to configure:
 *          - Crystal or oscillator selection.
 *          - Frequency range (1MHz to 48MHz).
 *          - Feedback resistor value (500K? or 100K?).
 *
 * @enum GPIO_ClkRefMode
 * | Mode                      | Description                                        |
 * |---------------------------|----------------------------------------------------|
 * | GPIO_ClkRef_CRYSTAL       | HSE type crystal.                                  |
 * | GPIO_ClkRef_OSCILLATOR    | HSE type oscillator.                               |
 * | GPIO_ClkRef_1MHz_TO_4MHz  | Frequency range: 1MHz < freq <= 4MHz.              |
 * | GPIO_ClkRef_4MHz_TO_12MHz | Frequency range: 4MHz < freq <= 12MHz.             |
 * | GPIO_ClkRef_12MHz_TO_24MHz| Frequency range: 12MHz < freq <= 24MHz.            |
 * | GPIO_ClkRef_24MHz_TO_48MHz| Frequency range: 24MHz < freq <= 48MHz.            |
 * | GPIO_ClkRef_R_500K        | 500K? feedback resistor.                           |
 * | GPIO_ClkRef_R_100K        | 100K? feedback resistor.                           |
 *
 * @return  None.
 *
 */
__STATIC_INLINE void
GPIO_setClkRef(GPIO_ClkRefMode mode)
{
    HWREG(IO_CFG_BASE + GPIO_O_CLK_REF_PIN_CTRL) = mode;
}

/**
 * @brief   Retrieves the current reference clock configuration.
 *
 * @details This function reads the current configuration from the `CLK_REF_PIN_CTRL` register
 *          and returns it as a 32-bit value. The returned value reflects the active reference
 *          clock mode, including the crystal/oscillator type, frequency range, and feedback resistor.
 *
 * @return  Returns a 32-bit value representing the current reference clock configuration.
 *          The value corresponds to the bits set in the `CLK_REF_PIN_CTRL` register.
 *
 * @note    The returned value can be used to check the current reference clock settings
 *          and compare them with the desired settings.
 *
 */
__STATIC_INLINE uint32_t
GPIO_getClkRef(void)
{
    return (HWREG(IO_CFG_BASE + GPIO_O_CLK_REF_PIN_CTRL));
}

/**
 * @brief   Configures the number of OSCCLK clock cycles to qualify inputs during wake-up from standby mode.
 *
 * @details This function writes a value to the `LPMCR_QUALSTDBY` register to determine the number of OSCCLK clock cycles
 *          required to qualify the selected inputs when waking up the system from STANDBY mode. The valid range is
 *          from 0 (2 OSCCLK cycles) to 63 (65 OSCCLK cycles).
 *
 * @param   time  Number of OSCCLK cycles to qualify the wake-up inputs. Valid range:
 *                - `0x00` (2 OSCCLK cycles)
 *                - `0x01` (3 OSCCLK cycles)
 *                - ...
 *                - `0x3F` (65 OSCCLK cycles)
 *
 * @note    The value of `time` must not exceed `0x3F`. If an invalid value is passed, the behavior is undefined.
 *
 */
__STATIC_INLINE void
GPIO_setLPMCR_QUALSTDBY(uint32_t time)
{
    HWREG(IO_CFG_BASE + GPIO_O_LPMCR_QUALSTDBY) = (time - 2) << 2;
}

/**
 * @brief   Connects the specified GPIO pin to the low-power mode (LPM) circuit.
 *
 * @details This function establishes the connection between the given GPIO pin and the low-power mode circuit,
 *          allowing it to operate in low-power configurations.
 *
 * @param   pin  The GPIO pin number to be connected to the low-power mode circuit.
 *
 */
__STATIC_INLINE void
GPIOLPM_Set(uint8_t pin)
{
    if (pin < 32) {
        HWREG(IO_CFG_BASE + GPIO_O_GPIOLPMSEL0) |= (1 << pin); // Set the corresponding bit of GPOLPMSEL0
    } else if (pin < 64) {
        HWREG(IO_CFG_BASE + GPIO_O_GPIOLPMSEL1) |= (1 << (pin - 32)); // Set the corresponding bit of GPOLPMSEL1
    }
}

/**
 * @brief   Disconnects the specified GPIO pin from the low-power mode (LPM) circuit.
 *
 * @details This function removes the connection between the given GPIO pin and the low-power mode circuit,
 *          disabling its low-power configuration.
 *
 * @param   pin  The GPIO pin number to be disconnected from the low-power mode circuit.
 *
 */
__STATIC_INLINE void
GPIOLPM_Clear(uint8_t pin)
{
    if (pin < 32) {
        HWREG(IO_CFG_BASE + GPIO_O_GPIOLPMSEL0) &= ~(1 << pin); // Clear the corresponding bit of GPIOLPMSEL0
    } else if (pin < 64) {
        HWREG(IO_CFG_BASE + GPIO_O_GPIOLPMSEL1) &= ~(1 << (pin - 32)); // Clear the corresponding bit of GPIOLPMSEL1
    }
}
/**
 * @brief   Checks if the specified GPIO pin is connected to the low-power mode (LPM) circuit.
 *
 * @details This function returns the connection status of a given GPIO pin with the low-power mode circuit.
 *
 * @param   pin  The GPIO pin number to check.
 *
 * @return  `1` if the pin is connected to the LPM circuit, otherwise `0`.
 *
 */
__STATIC_INLINE uint8_t
GPIOLPM_IsConnected(uint8_t pin)
{
    if (pin < 32) {
        return (HWREG(IO_CFG_BASE + GPIO_O_GPIOLPMSEL0) & (1 << pin)) ? 1 : 0; // Check the corresponding bit of GPIOLPMSEL0
    } else if (pin < 64) {
        return (HWREG(IO_CFG_BASE + GPIO_O_GPIOLPMSEL1) & (1 << (pin - 32))) ? 1 : 0; // Check the corresponding bit of GPIOLPMSEL1
    }
    return 3; // Invalid GPIO pin
}

/**
 * @brief Configures the exchange mode for a specified ePWM channel.
 *
 * @details This function sets the exchange mode for a specific ePWM channel based on the
 *          provided enum value. The exchange mode enables interaction between the CLB and HRPWM circuits.
 *
 * @param[in] exchange_mode  The ePWM channel and mode to be configured, as defined in `CLB_HRPWM_EX_EN_T`.
 *
 * @return None
 *
 */
__STATIC_INLINE void
GPIO_CLB_HRPWM_EX_EN(CLB_HRPWM_EX_EN_T CLB_HRPWM_EX_EN_T)
{
    uint32_t reg_value = HWREG(IO_CFG_BASE + GPIO_O_CLB_HRPWM_EX_EN);
    reg_value |= (1 << CLB_HRPWM_EX_EN_T);
    HWREG(IO_CFG_BASE + GPIO_O_CLB_HRPWM_EX_EN) = reg_value;
}
/**
 * @brief Disables the exchange mode for a specified ePWM channel.
 *
 * @details This function clears the exchange mode for a specific ePWM channel,
 *          returning it to normal operation mode. The exchange mode enables
 *          interaction between the CLB and HRPWM circuits, and this function
 *          disables it for the selected channel.
 *
 * @param[in] exchange_mode  The ePWM channel to be disabled, as defined in `CLB_HRPWM_EX_EN_T`.
 *
 * @return None
 *
 */
__STATIC_INLINE void
GPIO_CLB_HRPWM_EX_DIS(CLB_HRPWM_EX_EN_T CLB_HRPWM_EX_EN_T)
{
    uint32_t reg_value = HWREG(IO_CFG_BASE + GPIO_O_CLB_HRPWM_EX_EN);
    reg_value &= ~(1 << CLB_HRPWM_EX_EN_T);
    HWREG(IO_CFG_BASE + GPIO_O_CLB_HRPWM_EX_EN) = reg_value;
}

#ifdef __cplusplus
}
#endif

#endif /*IS_GS32F00xx(0x30)*/

#endif /* DEVICE_DRIVERLIB_SYSCTL_APB_PARA_H_ */
