/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *   @file    sysctl_nmi_wd.h
 *   @brief   do NOT include this file directly, include "sysctl.h" instead
 *
 */
/**
 * commit history
 * 20241218,Hezhiyuan,Modify comments
 */
#ifndef DEVICE_DRIVERLIB_SYSCTL_NMI_WD_H_
#define DEVICE_DRIVERLIB_SYSCTL_NMI_WD_H_

#include "gs32_version.h"

#ifdef __cplusplus
extern "C" {
#endif

#include "core_feature_base.h"
#include "inc/hw_memmap.h"
#include "inc/hw_nmi.h"
#include "inc/hw_types.h"

/* The following are key for the register NMI_WDG_LOCK   */
#define NMI_WDG_LOCK_ENKEY 0x5A5A5A5AU

#if IS_GS32F00xx(0x12)
/**
 *
 * Values that can be passed to SysCtl_clearNMIStatus(),
 * SysCtl_forceNMIFlags(), SysCtl_isNMIFlagSet(), and
 * SysCtl_isNMIShadowFlagSet() as the nmiFlags parameter and returned by
 * SysCtl_getNMIFlagStatus() and SysCtl_getNMIShadowFlagStatus().
 *
 */
// NMI Interrupt Flag Definitions
#define SYSCTL_NMI_NMIINT        0x1U    //  DSP NMI Interrupt Flag
#define SYSCTL_NMI_DLM_ECC       0x2U    //  DLM ECC Error Flag
#define SYSCTL_NMI_ILM_ECC       0x4U    //  ILM ECC Error Flag
#define SYSCTL_NMI_DCACHE_ECC    0x8U    //  DCACHE ECC Error Flag
#define SYSCTL_NMI_ICACHE_ECC    0x10U   //  ICACHE ECC Error Flag
#define SYSCTL_NMI_FLH_CACHE_ECC 0x20U   //  FLH CACHE ECC Error Flag
#define SYSCTL_NMI_FLH_MACRO_ECC 0x40U   //  FLH MACRO ECC Error Flag
#define SYSCTL_NMI_CCM_ECC       0x80U   //  CCM ECC Error Flag
#define SYSCTL_NMI_ERAD_NMI      0x100U  //  ERAD NMI Flag
#define SYSCTL_NMI_SRAM1_PARITY  0x200U  //  SRAM1 Parity Error Flag
#define SYSCTL_NMI_GS_RAM_PARITY 0x400U  //  GS RAM Parity Error Flag
#define SYSCTL_NMI_DSP_EXCP      0x800U  //  DSP Exception Flag
#define SYSCTL_NMI_CLOCKFAIL     0x2000U //  Clock Fail Interrupt Flag

#elif IS_GS32F00xx(0x30)

#define SYSCTL_NMI_NMIINT        0x1U    //  DSP NMI Interrupt Flag
#define SYSCTL_NMI_DLM_ECC       0x2U    //  DLM ECC Error Flag
#define SYSCTL_NMI_ILM_ECC       0x4U    //  ILM ECC Error Flag
#define SYSCTL_NMI_DCACHE_ECC    0x8U    //  DCACHE ECC Error Flag
#define SYSCTL_NMI_ICACHE_ECC    0x10U   //  ICACHE ECC Error Flag
#define SYSCTL_NMI_FLH_ECC       0x20U   //  FLH ECC Error Flag
#define SYSCTL_NMI_CRC_FALL      0x40U   //  CRC_FAIL Error Flag
#define SYSCTL_NMI_CLB_NMI       0x80U   //  CLB NMI Error Flag
#define SYSCTL_NMI_ERAD_NMI      0x100U  //  ERAD NMI Flag
#define SYSCTL_NMI_SWAP0_PARITY  0x200U  //  SWAP0 Parity Error Flag
#define SYSCTL_NMI_SWAP1_PARITY  0x400U  //  SWAP1 Parity Error Flag
#define SYSCTL_NMI_GS_RAM_PARITY 0x800U  //  GS RAM Parity Error Flag

#define SYSCTL_NMI_CLOCKFAIL     0x2000U //  Clock Fail Interrupt Flag
#define SYSCTL_NMI_DSP_WWDG      0x4000U //	 dsp_wwdg Flag
#define SYSCTL_NMI_DSP_GWDG      0x8000U //	 dsp_gwdg Flag

#define SYSCTL_NMI_FLAG_MASK     0xFFFEU //	 Flag MASK

#elif IS_GS32F3xx(0x22)
#define SYSCTL_NMI_NMIINT           0x01U // CPU1 or CPU2 Total Interrupt Flag (Global for CPU1 or CPU2)
#define SYSCTL_NMI_CPU2_ECC         0x2U  // CPU2 ECC Error Flag
#define SYSCTL_NMI_CPU1_ECC         0x4U  // CPU1 ECC Error Flag
#define SYSCTL_NMI_EFC_ECC          0x10U // EFC ECC Error Flag

#define SYSCTL_NMI_CPU2_NMI_WDT     0x20U // CPU2 NMI Watchdog Timeout Reset Flag (Only CPU1 receives signal)
#define SYSCTL_NMI_CPU2_WINDOW_WDT  0x40U // CPU2 Window Watchdog Timeout Reset Flag (Only CPU1 receives signal)
#define SYSCTL_NMI_CPU2_WDT_TIMEOUT 0x80U // CPU2 Watchdog Timeout Reset Flag (Only CPU1 receives signal)

#define SYSCTL_NMI_CLOCKFAIL        0x1000U //  Clock Fail Interrupt Flag Flag

#endif

#if IS_GS32F00xx() || IS_GS32F3xx(0x22) 

/**
 * @brief    This function enables writing all registers
 *
 * @param [in]   None
 *
 * @retval   None
 */
__STATIC_INLINE void NMI_WDG_unlockAllRegisters(void)
{
    HWREG(NMI_BASE + NMI_O_NMI_WDG_LOCK) = NMI_WDG_LOCK_ENKEY;
}

/**
 * @brief   This function disables writing all registers
 *
 * @param [in]   None
 *
 * @retval  None
 */
__STATIC_INLINE void NMI_WDG_lockAllRegisters(void)
{
    HWREG(NMI_BASE + NMI_O_NMI_WDG_LOCK) = (~NMI_WDG_LOCK_ENKEY);
}

/**
 *
 * @brief Enable the NMI Global interrupt bit
 *
 * @remarks
 * - This bit should be set after the device security related
 *   initialization is complete.
 *
 * @return None.
 *
 */
__STATIC_INLINE void
SysCtl_enableNMIGlobalInterrupt(void)
{
    NMI_WDG_unlockAllRegisters();
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    HWREG(NMI_BASE + NMI_O_DSP_NMI_CFG) |= NMI_DSP_NMI_CFG_DSP_NMI_EN;
#elif IS_GS32F3xx(0x22)
    if (__get_hart_id())
        HWREG(NMI_BASE + NMI_O_CPU2_NMI_CFG) |= NMI_CPU2_NMI_CFG_CPU2_NMI_EN;
    else
        HWREG(NMI_BASE + NMI_O_CPU1_NMI_CFG) |= NMI_CPU1_NMI_CFG_CPU1_NMI_EN;
#endif
    NMI_WDG_lockAllRegisters();
}

/**
 *
 *  @brief Read NMI interrupts.
 *
 *  Read the current state of NMI interrupt.
 *
 *  @return  true if NMI interrupt is triggered,  false if not.
 *
 */
__STATIC_INLINE Bool
SysCtl_getNMIStatus(void)
{
    //
    // Read and return the current value of the NMI flag register, masking out
    // all but the NMI bit.
    //
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    return ((HWREGH(NMI_BASE + NMI_O_DSP_NMI_FLG) & NMI_DSP_NMI_FLG_NMIINT) != 0U);

#elif IS_GS32F3xx(0x22)
    if (__get_hart_id())
        return ((HWREGH(NMI_BASE + NMI_O_CPU2_NMI_FLG) & NMI_CPU2_NMI_FLG_NMIINT) != 0U);
    else
        return ((HWREGH(NMI_BASE + NMI_O_CPU1_NMI_FLG) & NMI_CPU1_NMI_FLG_NMIINT) != 0U);
#endif
}

/**
 *
 *  @brief Read NMI Flags.
 *
 *  Read the current state of individual NMI interrupts
 *
 *  @return Value of NMIFLG register. These defines are provided to decode
 *  the value:
 *
 * IS_GS32F00xx(0x12)
 *  -  SYSCTL_NMI_NMIINT            -  bit:0    DSP NMI Interrupt Flag
 *  -  SYSCTL_NMI_DLM_ECC           -  bit:1    DLM ECC Error Flag
 *  -  SYSCTL_NMI_ILM_ECC           -  bit:2    ILM ECC Error Flag
 *  -  SYSCTL_NMI_DCACHE_ECC        -  bit:3    DCACHE ECC Error Flag
 *  -  SYSCTL_NMI_ICACHE_ECC        -  bit:4    ICACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_CACHE_ECC     -  bit:5    FLH CACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_MACRO_ECC     -  bit:6    FLH MACRO ECC Error Flag
 *  -  SYSCTL_NMI_CCM_ECC           -  bit:7    CCM ECC Error Flag
 *  -  SYSCTL_NMI_ERAD_NMI          -  bit:9    ERAD NMI Flag
 *  -  SYSCTL_NMI_SRAM1_PARITY      -  bit:10   SRAM1 Parity Error Flag
 *  -  SYSCTL_NMI_GS_RAM_PARITY     -  bit:11   GS RAM Parity Error Flag
 *  -  SYSCTL_NMI_DSP_EXCP          -  bit:12   DSP Exception Flag
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:13   Clock Fail Interrupt Flag
 *
 * IS_GS32F3xx(0x22)
 *  -  SYSCTL_NMI_NMIINT            -  bit:0    CPU1 or CPU2 Total Interrupt Flag
 *  -  SYSCTL_NMI_CPU2_ECC          -  bit:1    CPU2 ECC Error Flag
 *  -  SYSCTL_NMI_CPU1_ECC          -  bit:2    CPU1 ECC Error Flag
 *  -  SYSCTL_NMI_EFC_ECC           -  bit:4    EFC ECC Error Flag
 *  -  SYSCTL_NMI_CPU2_NMI_WDT      -  bit:5    CPU1 NMI Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WINDOW_WDT   -  bit:6    CPU1 Window Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WDT_TIMEOUT  -  bit:7    CPU1 Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:12    Clock Fail Interrupt Flag Flag
 *
 *
 */
__STATIC_INLINE uint16_t
SysCtl_getNMIFlagStatus(void)
{
    //
    // Read and return the current value of the NMI flag register.
    //
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    return ((uint16_t)(HWREGH(NMI_BASE + NMI_O_DSP_NMI_FLG) & 0xFFFF));

#elif IS_GS32F3xx(0x22)
    if (__get_hart_id())
        return ((uint16_t)(HWREGH(NMI_BASE + NMI_O_CPU2_NMI_FLG) & NMI_CPU2_NMI_FLG_CPU2_NMI_FLG_M));
    else
        return ((uint16_t)(HWREGH(NMI_BASE + NMI_O_CPU1_NMI_FLG) & NMI_CPU1_NMI_FLG_CPU1_NMI_FLG_M));
#endif
}

/**
 *
 *  @brief Check if the individual NMI interrupts are set.
 *
 *  @param [in] nmiFlags Bit mask of the NMI interrupts that user wants to clear.
 *  The bit format of this parameter is same as of the NMIFLG register. These
 *  defines are provided:
 *
 * IS_GS32F00xx(0x12)
 *  -  SYSCTL_NMI_NMIINT            -  bit:0    DSP NMI Interrupt Flag
 *  -  SYSCTL_NMI_DLM_ECC           -  bit:1    DLM ECC Error Flag
 *  -  SYSCTL_NMI_ILM_ECC           -  bit:2    ILM ECC Error Flag
 *  -  SYSCTL_NMI_DCACHE_ECC        -  bit:3    DCACHE ECC Error Flag
 *  -  SYSCTL_NMI_ICACHE_ECC        -  bit:4    ICACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_CACHE_ECC     -  bit:5    FLH CACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_MACRO_ECC     -  bit:6    FLH MACRO ECC Error Flag
 *  -  SYSCTL_NMI_CCM_ECC           -  bit:7    CCM ECC Error Flag
 *  -  SYSCTL_NMI_ERAD_NMI          -  bit:9    ERAD NMI Flag
 *  -  SYSCTL_NMI_SRAM1_PARITY      -  bit:10   SRAM1 Parity Error Flag
 *  -  SYSCTL_NMI_GS_RAM_PARITY     -  bit:11   GS RAM Parity Error Flag
 *  -  SYSCTL_NMI_DSP_EXCP          -  bit:12   DSP Exception Flag
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:13   Clock Fail Interrupt Flag
 *
 * IS_GS32F3xx(0x22)
 *  -  SYSCTL_NMI_NMIINT            -  bit:0    CPU1 or CPU2 Total Interrupt Flag
 *  -  SYSCTL_NMI_CPU2_ECC          -  bit:1    CPU2 ECC Error Flag
 *  -  SYSCTL_NMI_CPU1_ECC          -  bit:2    CPU1 ECC Error Flag
 *  -  SYSCTL_NMI_EFC_ECC           -  bit:4    EFC ECC Error Flag
 *  -  SYSCTL_NMI_CPU2_NMI_WDT      -  bit:5    CPU1 NMI Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WINDOW_WDT   -  bit:6    CPU1 Window Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WDT_TIMEOUT  -  bit:7    CPU1 Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:12    Clock Fail Interrupt Flag Flag
 *
 *
 *  Check if interrupt flags corresponding to the passed in bit mask are
 *  asserted.
 *
 *  @return  true if any of the NMI asked for in the parameter bit mask
 *  is set.  false if none of the NMI requested in the parameter bit mask are
 *  set.
 *
 */
__STATIC_INLINE Bool
SysCtl_isNMIFlagSet(uint16_t nmiFlags)
{

#if IS_GS32F00xx(0x12)

    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
                           SYSCTL_NMI_NMIINT |
                           SYSCTL_NMI_DLM_ECC |
                           SYSCTL_NMI_ILM_ECC |
                           SYSCTL_NMI_DCACHE_ECC |
                           SYSCTL_NMI_ICACHE_ECC |
                           SYSCTL_NMI_FLH_CACHE_ECC |
                           SYSCTL_NMI_FLH_MACRO_ECC |
                           SYSCTL_NMI_CCM_ECC |
                           SYSCTL_NMI_ERAD_NMI |
                           SYSCTL_NMI_SRAM1_PARITY |
                           SYSCTL_NMI_GS_RAM_PARITY |
                           SYSCTL_NMI_DSP_EXCP |
                           SYSCTL_NMI_CLOCKFAIL)) == 0U);

    //
    // Read the flag register and return true if any of them are set.
    //
    return ((HWREGH(NMI_BASE + NMI_O_DSP_NMI_FLG) & nmiFlags) != 0U);

#elif IS_GS32F00xx(0x30)

    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
							SYSCTL_NMI_NMIINT |
							SYSCTL_NMI_DLM_ECC |
							SYSCTL_NMI_ILM_ECC |
							SYSCTL_NMI_DCACHE_ECC |
							SYSCTL_NMI_ICACHE_ECC |
							SYSCTL_NMI_FLH_ECC |
							SYSCTL_NMI_CRC_FALL |
							SYSCTL_NMI_CLB_NMI |
							SYSCTL_NMI_ERAD_NMI |
							SYSCTL_NMI_SWAP0_PARITY |
							SYSCTL_NMI_SWAP1_PARITY |
							SYSCTL_NMI_GS_RAM_PARITY |
							SYSCTL_NMI_CLOCKFAIL |
							SYSCTL_NMI_DSP_WWDG|
							SYSCTL_NMI_DSP_GWDG)) == 0U);


    //
    // Read the flag register and return true if any of them are set.
    //
    return ((HWREGH(NMI_BASE + NMI_O_DSP_NMI_FLG) & nmiFlags) != 0U);


#elif IS_GS32F3xx(0x22)

    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
                           SYSCTL_NMI_NMIINT |
                           SYSCTL_NMI_CPU2_ECC |
                           SYSCTL_NMI_CPU1_ECC |
                           SYSCTL_NMI_EFC_ECC |
                           SYSCTL_NMI_CPU2_NMI_WDT |
                           SYSCTL_NMI_CPU2_WINDOW_WDT |
                           SYSCTL_NMI_CPU2_WDT_TIMEOUT |
                           SYSCTL_NMI_CLOCKFAIL)) == 0U);

    //
    // Read the flag register and return true if any of them are set.
    //

    if (__get_hart_id())
        return ((HWREGH(NMI_BASE + NMI_O_CPU2_NMI_FLG) & nmiFlags) != 0U);
    else
        return ((HWREGH(NMI_BASE + NMI_O_CPU1_NMI_FLG) & nmiFlags) != 0U);
#endif
}

/**
 *
 *  @brief Function to clear individual NMI interrupts.
 *
 *  @param [in] nmiFlags Bit mask of the NMI interrupts that user wants to clear.
 *  The bit format of this parameter is same as of the NMIFLG register. These
 *  defines are provided:
 *
 * IS_GS32F00xx(0x12)
 *  -  SYSCTL_NMI_NMIINT            -  bit:0    DSP NMI Interrupt Flag
 *  -  SYSCTL_NMI_DLM_ECC           -  bit:1    DLM ECC Error Flag
 *  -  SYSCTL_NMI_ILM_ECC           -  bit:2    ILM ECC Error Flag
 *  -  SYSCTL_NMI_DCACHE_ECC        -  bit:3    DCACHE ECC Error Flag
 *  -  SYSCTL_NMI_ICACHE_ECC        -  bit:4    ICACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_CACHE_ECC     -  bit:5    FLH CACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_MACRO_ECC     -  bit:6    FLH MACRO ECC Error Flag
 *  -  SYSCTL_NMI_CCM_ECC           -  bit:7    CCM ECC Error Flag
 *  -  SYSCTL_NMI_ERAD_NMI          -  bit:9    ERAD NMI Flag
 *  -  SYSCTL_NMI_SRAM1_PARITY      -  bit:10   SRAM1 Parity Error Flag
 *  -  SYSCTL_NMI_GS_RAM_PARITY     -  bit:11   GS RAM Parity Error Flag
 *  -  SYSCTL_NMI_DSP_EXCP          -  bit:12   DSP Exception Flag
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:13   Clock Fail Interrupt Flag
 *
 * IS_GS32F3xx(0x22)
 *  -  SYSCTL_NMI_NMIINT            -  bit:0    CPU1 or CPU2 Total Interrupt Flag
 *  -  SYSCTL_NMI_CPU2_ECC          -  bit:1    CPU2 ECC Error Flag
 *  -  SYSCTL_NMI_CPU1_ECC          -  bit:2    CPU1 ECC Error Flag
 *  -  SYSCTL_NMI_EFC_ECC           -  bit:4    EFC ECC Error Flag
 *  -  SYSCTL_NMI_CPU2_NMI_WDT      -  bit:5    CPU1 NMI Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WINDOW_WDT   -  bit:6    CPU1 Window Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WDT_TIMEOUT  -  bit:7    CPU1 Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:12    Clock Fail Interrupt Flag Flag
 *
 *  Clear NMI interrupt flags that correspond with the passed in bit mask.
 *
 * @remarks
 * - The NMI Interrupt flag is always cleared by default and
 *   therefore doesn't have to be included in the bit mask.
 *
 *  @return None.
 *
 */
__STATIC_INLINE void
SysCtl_clearNMIStatus(uint16_t nmiFlags)
{
    NMI_WDG_unlockAllRegisters();
#if IS_GS32F00xx(0x12)

    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
                           SYSCTL_NMI_NMIINT |
                           SYSCTL_NMI_DLM_ECC |
                           SYSCTL_NMI_ILM_ECC |
                           SYSCTL_NMI_DCACHE_ECC |
                           SYSCTL_NMI_ICACHE_ECC |
                           SYSCTL_NMI_FLH_CACHE_ECC |
                           SYSCTL_NMI_FLH_MACRO_ECC |
                           SYSCTL_NMI_CCM_ECC |
                           SYSCTL_NMI_ERAD_NMI |
                           SYSCTL_NMI_SRAM1_PARITY |
                           SYSCTL_NMI_GS_RAM_PARITY |
                           SYSCTL_NMI_DSP_EXCP |
                           SYSCTL_NMI_CLOCKFAIL)) == 0U);
    //
    // Clear the individual flags as well as NMI Interrupt flag
    //
    HWREGH(NMI_BASE + NMI_O_DSP_NMI_FLG_CLR) = nmiFlags;
#elif IS_GS32F00xx(0x30)

    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
							SYSCTL_NMI_NMIINT |
							SYSCTL_NMI_DLM_ECC |
							SYSCTL_NMI_ILM_ECC |
							SYSCTL_NMI_DCACHE_ECC |
							SYSCTL_NMI_ICACHE_ECC |
							SYSCTL_NMI_FLH_ECC |
							SYSCTL_NMI_CRC_FALL |
							SYSCTL_NMI_CLB_NMI |
							SYSCTL_NMI_ERAD_NMI |
							SYSCTL_NMI_SWAP0_PARITY |
							SYSCTL_NMI_SWAP1_PARITY |
							SYSCTL_NMI_GS_RAM_PARITY |
							SYSCTL_NMI_CLOCKFAIL |
							SYSCTL_NMI_DSP_WWDG|
							SYSCTL_NMI_DSP_GWDG)) == 0U);

    //
    // Clear the individual flags as well as NMI Interrupt flag
    //
    HWREGH(NMI_BASE + NMI_O_DSP_NMI_FLG_CLR) = nmiFlags;


#elif IS_GS32F3xx(0x22)

    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
                           SYSCTL_NMI_NMIINT |
                           SYSCTL_NMI_CPU2_ECC |
                           SYSCTL_NMI_CPU1_ECC |
                           SYSCTL_NMI_EFC_ECC |
                           SYSCTL_NMI_CPU2_NMI_WDT |
                           SYSCTL_NMI_CPU2_WINDOW_WDT |
                           SYSCTL_NMI_CPU2_WDT_TIMEOUT |
                           SYSCTL_NMI_CLOCKFAIL)) == 0U);
    //
    // Clear the individual flags as well as NMI Interrupt flag
    //
    if (__get_hart_id()) {
        HWREGH(NMI_BASE + NMI_O_CPU2_NMI_FLG_CLR) = nmiFlags;
    } else {
        HWREGH(NMI_BASE + NMI_O_CPU1_NMI_FLG_CLR) = nmiFlags;
    }

#endif
    NMI_WDG_lockAllRegisters();
}

/**
 *
 *  @brief Clear all the NMI Flags that are currently set.
 *
 *  @return None.
 *
 */
__STATIC_INLINE void
SysCtl_clearAllNMIFlags(void)
{
    NMI_WDG_unlockAllRegisters();
    uint16_t nmiFlags;

    //
    // Read the flag status register and then write to the clear register,
    // clearing all the flags that were returned plus the NMI flag.
    //
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    nmiFlags                                 = SysCtl_getNMIFlagStatus();
    HWREGH(NMI_BASE + NMI_O_DSP_NMI_FLG_CLR) = nmiFlags;
    HWREGH(NMI_BASE + NMI_O_DSP_NMI_FLG_CLR) = NMI_DSP_NMI_FLG_NMIINT;

#elif IS_GS32F3xx(0x22)
    if (__get_hart_id()) {
        nmiFlags                                  = SysCtl_getNMIFlagStatus();
        HWREGH(NMI_BASE + NMI_O_CPU2_NMI_FLG_CLR) = nmiFlags;
        HWREGH(NMI_BASE + NMI_O_CPU2_NMI_FLG_CLR) = NMI_CPU2_NMI_CFG_CPU2_NMI_EN;
    } else {
        nmiFlags                                  = SysCtl_getNMIFlagStatus();
        HWREGH(NMI_BASE + NMI_O_CPU1_NMI_FLG_CLR) = nmiFlags;
        HWREGH(NMI_BASE + NMI_O_CPU2_NMI_FLG_CLR) = NMI_CPU1_NMI_CFG_CPU1_NMI_EN;
    }
#endif
    NMI_WDG_lockAllRegisters();
}

/**
 *
 *  @brief Function to force individual NMI interrupt fail flags
 *
 *  @param [in] nmiFlags Bit mask of the NMI interrupts that user wants to clear.
 *  The bit format of this parameter is same as of the NMIFLG register. These
 *  defines are provided:
 *
 * IS_GS32F00xx(0x12)
 *  -  SYSCTL_NMI_DLM_ECC           -  bit:1    DLM ECC Error Flag
 *  -  SYSCTL_NMI_ILM_ECC           -  bit:2    ILM ECC Error Flag
 *  -  SYSCTL_NMI_DCACHE_ECC        -  bit:3    DCACHE ECC Error Flag
 *  -  SYSCTL_NMI_ICACHE_ECC        -  bit:4    ICACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_CACHE_ECC     -  bit:5    FLH CACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_MACRO_ECC     -  bit:6    FLH MACRO ECC Error Flag
 *  -  SYSCTL_NMI_CCM_ECC           -  bit:7    CCM ECC Error Flag
 *  -  SYSCTL_NMI_ERAD_NMI          -  bit:9    ERAD NMI Flag
 *  -  SYSCTL_NMI_SRAM1_PARITY      -  bit:10   SRAM1 Parity Error Flag
 *  -  SYSCTL_NMI_GS_RAM_PARITY     -  bit:11   GS RAM Parity Error Flag
 *  -  SYSCTL_NMI_DSP_EXCP          -  bit:12   DSP Exception Flag
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:13   Clock Fail Interrupt Flag
 *
 * IS_GS32F3xx(0x22)
 *  -  SYSCTL_NMI_CPU2_ECC          -  bit:1    CPU2 ECC Error Flag
 *  -  SYSCTL_NMI_CPU1_ECC          -  bit:2    CPU1 ECC Error Flag
 *  -  SYSCTL_NMI_EFC_ECC           -  bit:4    EFC ECC Error Flag
 *  -  SYSCTL_NMI_CPU2_NMI_WDT      -  bit:5    CPU1 NMI Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WINDOW_WDT   -  bit:6    CPU1 Window Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WDT_TIMEOUT  -  bit:7    CPU1 Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:12    Clock Fail Interrupt Flag Flag
 *
 *  @return None.
 *
 */
__STATIC_INLINE void
SysCtl_forceNMIFlags(uint16_t nmiFlags)
{
    NMI_WDG_unlockAllRegisters();
#if IS_GS32F00xx(0x12)

    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
                           SYSCTL_NMI_DLM_ECC |
                           SYSCTL_NMI_ILM_ECC |
                           SYSCTL_NMI_DCACHE_ECC |
                           SYSCTL_NMI_ICACHE_ECC |
                           SYSCTL_NMI_FLH_CACHE_ECC |
                           SYSCTL_NMI_FLH_MACRO_ECC |
                           SYSCTL_NMI_CCM_ECC |
                           SYSCTL_NMI_ERAD_NMI |
                           SYSCTL_NMI_SRAM1_PARITY |
                           SYSCTL_NMI_GS_RAM_PARITY |
                           SYSCTL_NMI_DSP_EXCP |
                           SYSCTL_NMI_CLOCKFAIL)) == 0U);
    //
    // Set the Flags for the individual interrupts in the NMI flag
    // force register
    //
    HWREGH(NMI_BASE + NMI_O_DSP_NMI_FLG_FRC) |= nmiFlags;

#elif IS_GS32F00xx(0x30)

    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
							SYSCTL_NMI_NMIINT |
							SYSCTL_NMI_DLM_ECC |
							SYSCTL_NMI_ILM_ECC |
							SYSCTL_NMI_DCACHE_ECC |
							SYSCTL_NMI_ICACHE_ECC |
							SYSCTL_NMI_FLH_ECC |
							SYSCTL_NMI_CRC_FALL |
							SYSCTL_NMI_CLB_NMI |
							SYSCTL_NMI_ERAD_NMI |
							SYSCTL_NMI_SWAP0_PARITY |
							SYSCTL_NMI_SWAP1_PARITY |
							SYSCTL_NMI_GS_RAM_PARITY |
							SYSCTL_NMI_CLOCKFAIL |
							SYSCTL_NMI_DSP_WWDG|
							SYSCTL_NMI_DSP_GWDG)) == 0U);

    //
    // Set the Flags for the individual interrupts in the NMI flag
    // force register
    //
    HWREGH(NMI_BASE + NMI_O_DSP_NMI_FLG_FRC) |= nmiFlags;


#elif IS_GS32F3xx(0x22)
    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
                           SYSCTL_NMI_CPU2_ECC |
                           SYSCTL_NMI_CPU1_ECC |
                           SYSCTL_NMI_EFC_ECC |
                           SYSCTL_NMI_CPU2_NMI_WDT |
                           SYSCTL_NMI_CPU2_WINDOW_WDT |
                           SYSCTL_NMI_CPU2_WDT_TIMEOUT |
                           SYSCTL_NMI_CLOCKFAIL)) == 0U);
    //
    // Set the Flags for the individual interrupts in the NMI flag
    // force register
    //
    if (__get_hart_id()) {
        HWREGH(NMI_BASE + NMI_O_CPU2_NMI_FLG_FRC) |= nmiFlags;
    } else {
        HWREGH(NMI_BASE + NMI_O_CPU1_NMI_FLG_FRC) |= nmiFlags;
    }

#endif
    NMI_WDG_lockAllRegisters();
}

/**
 *
 *  @brief Gets the NMI watchdog counter value.
 *
 *  @remarks
 * - The counter is clocked at the SYSCLKOUT rate.
 *
 *  @return Returns the NMI watchdog counter register's current value.
 *
 */
__STATIC_INLINE uint16_t
SysCtl_getNMIWatchdogCounter(void)
{
    //
    // Read and return the NMI watchdog counter register's value.
    //
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    return ((uint16_t)(HWREGH(NMI_BASE + NMI_O_DSP_WD_CNT)));

#elif IS_GS32F3xx(0x22)
    if (__get_hart_id())
        return ((uint16_t)(HWREGH(NMI_BASE + NMI_O_CPU2_WD_CNT)));
    else
        return ((uint16_t)(HWREGH(NMI_BASE + NMI_O_CPU1_WD_CNT)));
#endif
}

/**
 *
 *  @brief Sets the NMI watchdog period value.
 *
 *  @param [in] wdPeriod is the 16-bit value at which a reset is generated.
 *
 *  This function writes to the NMI watchdog period register that holds the
 *  value to which the NMI watchdog counter is compared. When the two registers
 *  match, a reset is generated. By default, the period is 0xFFFF.
 *
 *  @remarks
 * - If a value smaller than the current counter value is passed into the
 *   wdPeriod parameter, a NMIRSn will be forced.
 *
 *  @return None.
 *
 */
__STATIC_INLINE void
SysCtl_setNMIWatchdogPeriod(uint16_t wdPeriod)
{
    NMI_WDG_unlockAllRegisters();
    //
    // Write to the period register.
    //
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    HWREGH(NMI_BASE + NMI_O_DSP_WD_PRD) = wdPeriod;

#elif IS_GS32F3xx(0x22)
    if (__get_hart_id())
        HWREGH(NMI_BASE + NMI_O_CPU2_WD_PRD) = wdPeriod;
    else
        HWREGH(NMI_BASE + NMI_O_CPU1_WD_PRD) = wdPeriod;
#endif
    NMI_WDG_lockAllRegisters();
}

/**
 *
 *  @brief Gets the NMI watchdog period value.
 *
 *  @return Returns the NMI watchdog period register's current value.
 *
 */
__STATIC_INLINE uint16_t
SysCtl_getNMIWatchdogPeriod(void)
{
    //
    // Read and return the NMI watchdog period register's value.
    //
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    return ((uint16_t)(HWREGH(NMI_BASE + NMI_O_DSP_WD_PRD)));

#elif IS_GS32F3xx(0x22)
    if (__get_hart_id())
        return ((uint16_t)(HWREGH(NMI_BASE + NMI_O_CPU2_WD_PRD)));
    else
        return ((uint16_t)(HWREGH(NMI_BASE + NMI_O_CPU1_WD_PRD)));
#endif
}

/**
 *
 *  @brief Read NMI Shadow Flags.
 *
 *  Read the current state of individual NMI interrupts
 *
 *  @return Value of NMISHDFLG register. These defines are provided to decode
 *  the value:
 *
 * IS_GS32F00xx(0x12)
 *  -  SYSCTL_NMI_DLM_ECC           -  bit:1    DLM ECC Error Flag
 *  -  SYSCTL_NMI_ILM_ECC           -  bit:2    ILM ECC Error Flag
 *  -  SYSCTL_NMI_DCACHE_ECC        -  bit:3    DCACHE ECC Error Flag
 *  -  SYSCTL_NMI_ICACHE_ECC        -  bit:4    ICACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_CACHE_ECC     -  bit:5    FLH CACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_MACRO_ECC     -  bit:6    FLH MACRO ECC Error Flag
 *  -  SYSCTL_NMI_CCM_ECC           -  bit:7    CCM ECC Error Flag
 *  -  SYSCTL_NMI_ERAD_NMI          -  bit:9    ERAD NMI Flag
 *  -  SYSCTL_NMI_SRAM1_PARITY      -  bit:10   SRAM1 Parity Error Flag
 *  -  SYSCTL_NMI_GS_RAM_PARITY     -  bit:11   GS RAM Parity Error Flag
 *  -  SYSCTL_NMI_DSP_EXCP          -  bit:12   DSP Exception Flag
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:13   Clock Fail Interrupt Flag
 *
 * IS_GS32F3xx(0x22)
 *  -  SYSCTL_NMI_CPU2_ECC          -  bit:1    CPU2 ECC Error Flag
 *  -  SYSCTL_NMI_CPU1_ECC          -  bit:2    CPU1 ECC Error Flag
 *  -  SYSCTL_NMI_EFC_ECC           -  bit:4    EFC ECC Error Flag
 *  -  SYSCTL_NMI_CPU2_NMI_WDT      -  bit:5    CPU1 NMI Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WINDOW_WDT   -  bit:6    CPU1 Window Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WDT_TIMEOUT  -  bit:7    CPU1 Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:12    Clock Fail Interrupt Flag Flag
 *
 */
__STATIC_INLINE uint32_t
SysCtl_getNMIShadowFlagStatus(void)
{
    //
    // Read and return the current value of the NMI shadow flag register.
    //
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    return ((uint32_t)(HWREGH(NMI_BASE + NMI_O_DSP_NMI_SHD_FLG)));

#elif IS_GS32F3xx(0x22)
    if (__get_hart_id())
        return ((uint32_t)(HWREGH(NMI_BASE + NMI_O_CPU2_NMI_SHD_FLG)));
    else
        return ((uint32_t)(HWREGH(NMI_BASE + NMI_O_CPU1_NMI_SHD_FLG)));
#endif
}

/**
 *
 *  @brief Check if the individual NMI shadow flags are set.
 *
 *  @param [in] nmiFlags Bit mask of the NMI interrupts that user wants  to clear.
 *  The bit format of this parameter is same as of the NMIFLG register. These
 *  defines are provided:
 *
 * IS_GS32F00xx(0x12)
 *  -  SYSCTL_NMI_DLM_ECC           -  bit:1    DLM ECC Error Flag
 *  -  SYSCTL_NMI_ILM_ECC           -  bit:2    ILM ECC Error Flag
 *  -  SYSCTL_NMI_DCACHE_ECC        -  bit:3    DCACHE ECC Error Flag
 *  -  SYSCTL_NMI_ICACHE_ECC        -  bit:4    ICACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_CACHE_ECC     -  bit:5    FLH CACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_MACRO_ECC     -  bit:6    FLH MACRO ECC Error Flag
 *  -  SYSCTL_NMI_CCM_ECC           -  bit:7    CCM ECC Error Flag
 *  -  SYSCTL_NMI_ERAD_NMI          -  bit:9    ERAD NMI Flag
 *  -  SYSCTL_NMI_SRAM1_PARITY      -  bit:10   SRAM1 Parity Error Flag
 *  -  SYSCTL_NMI_GS_RAM_PARITY     -  bit:11   GS RAM Parity Error Flag
 *  -  SYSCTL_NMI_DSP_EXCP          -  bit:12   DSP Exception Flag
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:13   Clock Fail Interrupt Flag
 *
 * IS_GS32F3xx(0x22)
 *  -  SYSCTL_NMI_CPU2_ECC          -  bit:1    CPU2 ECC Error Flag
 *  -  SYSCTL_NMI_CPU1_ECC          -  bit:2    CPU1 ECC Error Flag
 *  -  SYSCTL_NMI_EFC_ECC           -  bit:4    EFC ECC Error Flag
 *  -  SYSCTL_NMI_CPU2_NMI_WDT      -  bit:5    CPU1 NMI Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WINDOW_WDT   -  bit:6    CPU1 Window Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WDT_TIMEOUT  -  bit:7    CPU1 Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:12    Clock Fail Interrupt Flag Flag
 *
 *  Check if interrupt flags corresponding to the passed in bit mask are
 *  asserted.
 *
 *  @return  true if any of the NMI asked for in the parameter bit mask
 *  is set.  false if none of the NMI requested in the parameter bit mask are
 *  set.
 *
 */
__STATIC_INLINE bool
SysCtl_isNMIShadowFlagSet(uint16_t nmiFlags)
{

#if IS_GS32F00xx(0x12)
    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
                           SYSCTL_NMI_DLM_ECC |
                           SYSCTL_NMI_ILM_ECC |
                           SYSCTL_NMI_DCACHE_ECC |
                           SYSCTL_NMI_ICACHE_ECC |
                           SYSCTL_NMI_FLH_CACHE_ECC |
                           SYSCTL_NMI_FLH_MACRO_ECC |
                           SYSCTL_NMI_CCM_ECC |
                           SYSCTL_NMI_ERAD_NMI |
                           SYSCTL_NMI_SRAM1_PARITY |
                           SYSCTL_NMI_GS_RAM_PARITY |
                           SYSCTL_NMI_DSP_EXCP |
                           SYSCTL_NMI_CLOCKFAIL)) == 0U);
    //
    // Read the flag register and return true if any of them are set.
    //
    return ((HWREGH(NMI_BASE + NMI_O_DSP_NMI_SHD_FLG) & nmiFlags) != 0U);
#elif IS_GS32F00xx(0x30)
    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
							SYSCTL_NMI_NMIINT |
							SYSCTL_NMI_DLM_ECC |
							SYSCTL_NMI_ILM_ECC |
							SYSCTL_NMI_DCACHE_ECC |
							SYSCTL_NMI_ICACHE_ECC |
							SYSCTL_NMI_FLH_ECC |
							SYSCTL_NMI_CRC_FALL |
							SYSCTL_NMI_CLB_NMI |
							SYSCTL_NMI_ERAD_NMI |
							SYSCTL_NMI_SWAP0_PARITY |
							SYSCTL_NMI_SWAP1_PARITY |
							SYSCTL_NMI_GS_RAM_PARITY |
							SYSCTL_NMI_CLOCKFAIL |
							SYSCTL_NMI_DSP_WWDG|
							SYSCTL_NMI_DSP_GWDG)) == 0U);
    //
    // Read the flag register and return true if any of them are set.
    //
    return ((HWREGH(NMI_BASE + NMI_O_DSP_NMI_SHD_FLG) & nmiFlags) != 0U);

#elif IS_GS32F3xx(0x22)

    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
                           SYSCTL_NMI_CPU2_ECC |
                           SYSCTL_NMI_CPU1_ECC |
                           SYSCTL_NMI_EFC_ECC |
                           SYSCTL_NMI_CPU2_NMI_WDT |
                           SYSCTL_NMI_CPU2_WINDOW_WDT |
                           SYSCTL_NMI_CPU2_WDT_TIMEOUT |
                           SYSCTL_NMI_CLOCKFAIL)) == 0U);

    //
    // Read the flag register and return true if any of them are set.
    //
    if (__get_hart_id())
        return ((HWREGH(NMI_BASE + NMI_O_CPU2_NMI_SHD_FLG) & nmiFlags) != 0U);
    else
        return ((HWREGH(NMI_BASE + NMI_O_CPU1_NMI_SHD_FLG) & nmiFlags) != 0U);
#endif
}

/**
 *
 *  @brief Function to mask individual NMI interrupts.
 *
 *  @param [in] nmiFlags Bit mask of the NMI interrupts that user wants to mask.
 *  The bit format of this parameter is the same as that of the NMIFLG register.
 *  The following defines are provided for the corresponding flags:
 *
 * IS_GS32F00xx(0x12)
 *  -  SYSCTL_NMI_DLM_ECC           -  bit:1    DLM ECC Error Flag
 *  -  SYSCTL_NMI_ILM_ECC           -  bit:2    ILM ECC Error Flag
 *  -  SYSCTL_NMI_DCACHE_ECC        -  bit:3    DCACHE ECC Error Flag
 *  -  SYSCTL_NMI_ICACHE_ECC        -  bit:4    ICACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_CACHE_ECC     -  bit:5    FLH CACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_MACRO_ECC     -  bit:6    FLH MACRO ECC Error Flag
 *  -  SYSCTL_NMI_CCM_ECC           -  bit:7    CCM ECC Error Flag
 *  -  SYSCTL_NMI_ERAD_NMI          -  bit:9    ERAD NMI Flag
 *  -  SYSCTL_NMI_SRAM1_PARITY      -  bit:10   SRAM1 Parity Error Flag
 *  -  SYSCTL_NMI_GS_RAM_PARITY     -  bit:11   GS RAM Parity Error Flag
 *  -  SYSCTL_NMI_DSP_EXCP          -  bit:12   DSP Exception Flag
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:13   Clock Fail Interrupt Flag
 *
 * IS_GS32F3xx(0x22)
 *  -  SYSCTL_NMI_CPU2_ECC          -  bit:1    CPU2 ECC Error Flag
 *  -  SYSCTL_NMI_CPU1_ECC          -  bit:2    CPU1 ECC Error Flag
 *  -  SYSCTL_NMI_EFC_ECC           -  bit:4    EFC ECC Error Flag
 *  -  SYSCTL_NMI_CPU2_NMI_WDT      -  bit:5    CPU1 NMI Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WINDOW_WDT   -  bit:6    CPU1 Window Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WDT_TIMEOUT  -  bit:7    CPU1 Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:12    Clock Fail Interrupt Flag Flag
 *
 *  Masks (disables) the NMI interrupt flags that correspond to the bits set in the input bitmask.
 *
 *  @remarks
 * - The NMI Interrupt flag is always cleared by default and therefore doesn't need to be included in the bitmask.
 *
 *  @return None.
 *
 */
__STATIC_INLINE void
SysCtl_maskNMIStatus(uint16_t nmiFlags)
{
    NMI_WDG_unlockAllRegisters();
#if IS_GS32F00xx(0x12)
    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
                           SYSCTL_NMI_NMIINT |
                           SYSCTL_NMI_DLM_ECC |
                           SYSCTL_NMI_ILM_ECC |
                           SYSCTL_NMI_DCACHE_ECC |
                           SYSCTL_NMI_ICACHE_ECC |
                           SYSCTL_NMI_FLH_CACHE_ECC |
                           SYSCTL_NMI_FLH_MACRO_ECC |
                           SYSCTL_NMI_CCM_ECC |
                           SYSCTL_NMI_ERAD_NMI |
                           SYSCTL_NMI_SRAM1_PARITY |
                           SYSCTL_NMI_GS_RAM_PARITY |
                           SYSCTL_NMI_DSP_EXCP |
                           SYSCTL_NMI_CLOCKFAIL)) == 0U);
    HWREG(NMI_BASE + NMI_O_DSP_NMI_CFG) |= ((nmiFlags >> 1) << 16);

#elif IS_GS32F00xx(0x30)
    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
							SYSCTL_NMI_NMIINT |
							SYSCTL_NMI_DLM_ECC |
							SYSCTL_NMI_ILM_ECC |
							SYSCTL_NMI_DCACHE_ECC |
							SYSCTL_NMI_ICACHE_ECC |
							SYSCTL_NMI_FLH_ECC |
							SYSCTL_NMI_CRC_FALL |
							SYSCTL_NMI_CLB_NMI |
							SYSCTL_NMI_ERAD_NMI |
							SYSCTL_NMI_SWAP0_PARITY |
							SYSCTL_NMI_SWAP1_PARITY |
							SYSCTL_NMI_GS_RAM_PARITY |
							SYSCTL_NMI_CLOCKFAIL |
							SYSCTL_NMI_DSP_WWDG|
							SYSCTL_NMI_DSP_GWDG)) == 0U);

    HWREG(NMI_BASE + NMI_O_DSP_NMI_CFG) |= (nmiFlags << 16);

#elif IS_GS32F3xx(0x22)
    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
                           SYSCTL_NMI_CPU2_ECC |
                           SYSCTL_NMI_CPU1_ECC |
                           SYSCTL_NMI_EFC_ECC |
                           SYSCTL_NMI_CPU2_NMI_WDT |
                           SYSCTL_NMI_CPU2_WINDOW_WDT |
                           SYSCTL_NMI_CPU2_WDT_TIMEOUT |
                           SYSCTL_NMI_CLOCKFAIL)) == 0U);

    //
    // Masks (disables) the individual flags as well as NMI Interrupt flag
    //
    if (__get_hart_id())
    	HWREG(NMI_BASE + NMI_O_CPU2_NMI_CFG) |= ((nmiFlags >> 1) << 16);
    else
    	HWREG(NMI_BASE + NMI_O_CPU1_NMI_CFG) |= ((nmiFlags >> 1) << 16);
#endif
    NMI_WDG_lockAllRegisters();
}

/**
 *
 *  @brief Function to unmask (enable) individual NMI interrupts.
 *
 *  @param [in] nmiFlags Bit mask of the NMI interrupts that user wants to unmask.
 *  The bit format of this parameter is the same as that of the NMIFLG register.
 *  The following defines are provided for the corresponding flags:
 *
 * IS_GS32F00xx(0x12)
 *  -  SYSCTL_NMI_DLM_ECC           -  bit:1    DLM ECC Error Flag
 *  -  SYSCTL_NMI_ILM_ECC           -  bit:2    ILM ECC Error Flag
 *  -  SYSCTL_NMI_DCACHE_ECC        -  bit:3    DCACHE ECC Error Flag
 *  -  SYSCTL_NMI_ICACHE_ECC        -  bit:4    ICACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_CACHE_ECC     -  bit:5    FLH CACHE ECC Error Flag
 *  -  SYSCTL_NMI_FLH_MACRO_ECC     -  bit:6    FLH MACRO ECC Error Flag
 *  -  SYSCTL_NMI_CCM_ECC           -  bit:7    CCM ECC Error Flag
 *  -  SYSCTL_NMI_ERAD_NMI          -  bit:9    ERAD NMI Flag
 *  -  SYSCTL_NMI_SRAM1_PARITY      -  bit:10   SRAM1 Parity Error Flag
 *  -  SYSCTL_NMI_GS_RAM_PARITY     -  bit:11   GS RAM Parity Error Flag
 *  -  SYSCTL_NMI_DSP_EXCP          -  bit:12   DSP Exception Flag
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:13   Clock Fail Interrupt Flag
 *
 * IS_GS32F3xx(0x22)
 *  -  SYSCTL_NMI_CPU2_ECC          -  bit:1    CPU2 ECC Error Flag
 *  -  SYSCTL_NMI_CPU1_ECC          -  bit:2    CPU1 ECC Error Flag
 *  -  SYSCTL_NMI_EFC_ECC           -  bit:4    EFC ECC Error Flag
 *  -  SYSCTL_NMI_CPU2_NMI_WDT      -  bit:5    CPU1 NMI Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WINDOW_WDT   -  bit:6    CPU1 Window Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CPU2_WDT_TIMEOUT  -  bit:7    CPU1 Watchdog Timeout Reset Flag (Only CPU1 receives signal)
 *  -  SYSCTL_NMI_CLOCKFAIL         -  bit:12    Clock Fail Interrupt Flag Flag
 *
 *  Unmasks (enables) the NMI interrupt flags that correspond to the bits set in the input bitmask.
 *
 *  @remarks
 * - The NMI Interrupt flag is always cleared by default and therefore doesn't need to be included in the bitmask.
 *
 *  @return None.
 *
 */
__STATIC_INLINE void
SysCtl_unmaskNMIStatus(uint16_t nmiFlags)
{
    NMI_WDG_unlockAllRegisters();
#if IS_GS32F00xx(0x12)
    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
                           SYSCTL_NMI_NMIINT |
                           SYSCTL_NMI_DLM_ECC |
                           SYSCTL_NMI_ILM_ECC |
                           SYSCTL_NMI_DCACHE_ECC |
                           SYSCTL_NMI_ICACHE_ECC |
                           SYSCTL_NMI_FLH_CACHE_ECC |
                           SYSCTL_NMI_FLH_MACRO_ECC |
                           SYSCTL_NMI_CCM_ECC |
                           SYSCTL_NMI_ERAD_NMI |
                           SYSCTL_NMI_SRAM1_PARITY |
                           SYSCTL_NMI_GS_RAM_PARITY |
                           SYSCTL_NMI_DSP_EXCP |
                           SYSCTL_NMI_CLOCKFAIL)) == 0U);
    HWREG(NMI_BASE + NMI_O_DSP_NMI_CFG) &= ~((nmiFlags >> 1) << 16);

#elif IS_GS32F00xx(0x30)
    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
							SYSCTL_NMI_NMIINT |
							SYSCTL_NMI_DLM_ECC |
							SYSCTL_NMI_ILM_ECC |
							SYSCTL_NMI_DCACHE_ECC |
							SYSCTL_NMI_ICACHE_ECC |
							SYSCTL_NMI_FLH_ECC |
							SYSCTL_NMI_CRC_FALL |
							SYSCTL_NMI_CLB_NMI |
							SYSCTL_NMI_ERAD_NMI |
							SYSCTL_NMI_SWAP0_PARITY |
							SYSCTL_NMI_SWAP1_PARITY |
							SYSCTL_NMI_GS_RAM_PARITY |
							SYSCTL_NMI_CLOCKFAIL |
							SYSCTL_NMI_DSP_WWDG|
							SYSCTL_NMI_DSP_GWDG)) == 0U);

    HWREG(NMI_BASE + NMI_O_DSP_NMI_CFG) &= ~((nmiFlags) << 16);

#elif IS_GS32F3xx(0x22)
    //
    // Check the arguments.
    // Make sure if reserved bits are not set in nmiFlags.
    //
    ASSERT((nmiFlags & ~(
                           SYSCTL_NMI_CPU2_ECC |
                           SYSCTL_NMI_CPU1_ECC |
                           SYSCTL_NMI_EFC_ECC |
                           SYSCTL_NMI_CPU2_NMI_WDT |
                           SYSCTL_NMI_CPU2_WINDOW_WDT |
                           SYSCTL_NMI_CPU2_WDT_TIMEOUT |
                           SYSCTL_NMI_CLOCKFAIL)) == 0U);

    //
    // Unmasks (enables) the individual flags as well as NMI Interrupt flag
    //
    if (__get_hart_id())
    	HWREG(NMI_BASE + NMI_O_CPU2_NMI_CFG) &= ~((nmiFlags >> 1) << 16);
    else
    	HWREG(NMI_BASE + NMI_O_CPU1_NMI_CFG) &= ~((nmiFlags >> 1) << 16);
#endif
    NMI_WDG_lockAllRegisters();
}

/**
 *
 *  @brief Enable the NMI Reset Request bit
 *
 *  @remarks
 * - This bit should be set after the device security related
 *   initialization is complete.
 *
 *  @return None.
 *
 */
__STATIC_INLINE void
SysCtl_enableNMIResetRequest(void)
{
    NMI_WDG_unlockAllRegisters();
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    HWREG(NMI_BASE + NMI_O_DSP_NMI_CFG) |= NMI_DSP_NMI_CFG_DSP_NMI_RSTREQ_EN;
#elif IS_GS32F3xx(0x22)
    if (__get_hart_id())
        HWREG(NMI_BASE + NMI_O_CPU2_NMI_CFG) |= NMI_CPU2_NMI_CFG_CPU2_NMI_RSTREQ_EN;
    else
        HWREG(NMI_BASE + NMI_O_CPU1_NMI_CFG) |= NMI_CPU1_NMI_CFG_CPU1_NMI_RSTREQ_EN;
#endif
    NMI_WDG_lockAllRegisters();
}

/**
 *
 *  @brief Disable the NMI Reset Request bit
 *
 *  @remarks
 * - This bit should be set after the device security related
 *   initialization is complete.
 *
 *  @return None.
 *
 */
__STATIC_INLINE void
SysCtl_disableNMIResetRequest(void)
{
    NMI_WDG_unlockAllRegisters();
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    HWREG(NMI_BASE + NMI_O_DSP_NMI_CFG) &= (~NMI_DSP_NMI_CFG_DSP_NMI_RSTREQ_EN);
#elif IS_GS32F3xx(0x22)
    if (__get_hart_id())
        HWREG(NMI_BASE + NMI_O_CPU2_NMI_CFG) &= (~NMI_CPU2_NMI_CFG_CPU2_NMI_RSTREQ_EN);
    else
        HWREG(NMI_BASE + NMI_O_CPU1_NMI_CFG) &= (~NMI_CPU1_NMI_CFG_CPU1_NMI_RSTREQ_EN);
#endif
    NMI_WDG_lockAllRegisters();
}

/**
 *
 *  @brief when cpu debug, wdg count hold
 *
 *  @remarks
 * - This bit should be set after the device security related
 *   initialization is complete.
 *
 *  @return None.
 *
 */
__STATIC_INLINE void
SysCtl_enableNMIWDGhold(void)
{
    NMI_WDG_unlockAllRegisters();
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    HWREG(NMI_BASE + NMI_O_DSP_NMI_CFG) |= NMI_DSP_NMI_CFG_DSP_DBG_WDG_HOLD_EN;
#elif IS_GS32F3xx(0x22)
    if (__get_hart_id())
        HWREG(NMI_BASE + NMI_O_CPU2_NMI_CFG) |= NMI_CPU2_NMI_CFG_CPU2_DBG_WDG_HOLD_EN;
    else
        HWREG(NMI_BASE + NMI_O_CPU1_NMI_CFG) |= NMI_CPU1_NMI_CFG_CPU1_DBG_WDG_HOLD_EN;
#endif
    NMI_WDG_lockAllRegisters();
}

/**
 *
 *  @brief when cpu debug, wdg count not hold
 *
 *  @remarks
 * - This bit should be set after the device security related
 *   initialization is complete.
 *
 *  @return None.
 *
 */
__STATIC_INLINE void
SysCtl_disableNMIWDGhold(void)
{
    NMI_WDG_unlockAllRegisters();
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    HWREG(NMI_BASE + NMI_O_DSP_NMI_CFG) &= (~NMI_DSP_NMI_CFG_DSP_DBG_WDG_HOLD_EN);
#elif IS_GS32F3xx(0x22)
    if (__get_hart_id())
        HWREG(NMI_BASE + NMI_O_CPU2_NMI_CFG) &= (~NMI_CPU2_NMI_CFG_CPU2_DBG_WDG_HOLD_EN);
    else
        HWREG(NMI_BASE + NMI_O_CPU1_NMI_CFG) &= (~NMI_CPU1_NMI_CFG_CPU1_DBG_WDG_HOLD_EN);
#endif
    NMI_WDG_lockAllRegisters();
}

/**
 *
 *  @brief Trigger a software interrupt.
 *
 *  @param [in] id The ID of the software interrupt to trigger. The valid range is from 1 to 12.
 *  Each ID corresponds to a specific software interrupt in the system.
 *
 *  @return None.
 *
 */
__STATIC_INLINE void SysCtl_triggerSoftInt(uint32_t id)
{
    NMI_WDG_unlockAllRegisters();
    ASSERT(id <= 12 && id >= 1);
    HWREG(NMI_BASE + NMI_O_DSP_SOFT_INT + ((id - 1) * 4U)) = 1;
    NMI_WDG_lockAllRegisters();
}
/**
 *
 *  @brief Clear a triggered software interrupt.
 *
 *  @param [in] id The ID of the software interrupt to clear. The valid range is from 1 to 12.
 *  Each ID corresponds to a specific software interrupt in the system.
 *
 *  @return None.
 *
 */
__STATIC_INLINE void SysCtl_clearSoftInt(uint32_t id)
{
    NMI_WDG_unlockAllRegisters();
    ASSERT(id <= 12 && id >= 1);
    HWREG(NMI_BASE + NMI_O_DSP_SOFT_INT + ((id - 1) * 4U)) = 0;
    __RWMB();
    NMI_WDG_lockAllRegisters();
}

#endif // 1.2 or 2.2

#ifdef __cplusplus
}
#endif

#endif /* DEVICE_DRIVERLIB_SYSCTL_H_ */
