/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   \file    UART.h
*   \brief
*
*/

#ifndef DEVICE_DRIVERLIB_UART_H_
#define DEVICE_DRIVERLIB_UART_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "gs32_version.h"

#include "gs32_version.h"
#include "inc/hw_types.h"
#include "inc/hw_uart.h"
#include "debug.h"
#include "stdbool.h"

#if IS_GS32F00xx(0x30) || IS_GS32F3xx(0x23)

#include "uart_v30.h"

#endif

#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)

/**
 * @brief Mask for extracting word length
 */
#define SCI_CONFIG_WLEN_MASK    0x0003U

/**
 * @brief Number of bits configured for valid data
 */
#define SCI_CONFIG_WLEN_8       0x0003U
#define SCI_CONFIG_WLEN_7       0x0002U
#define SCI_CONFIG_WLEN_6       0x0001U
#define SCI_CONFIG_WLEN_5       0x0000U

/**
 * @brief Number of bits configured for stop
 */
#define SCI_CONFIG_STOP_MASK    0x0004U
#define SCI_CONFIG_STOP_ONE     0x0000U
#define SCI_CONFIG_STOP_TWO     0x0004U

/**
 * @brief Configure parity check mode
 */
#define SCI_CONFIG_PAR_MASK     0x0018U
#define SCI_NO_PARITY           0x0000U
#define SCI_PARITY_EVEN         0x0018U
#define SCI_PARITY_ODD          0x0008U

/**
 * @brief The SCI reads data with a length of the least significant 8 bits
 */
#define SCI_RXBUF_SAR_M        0xFFU

/**
 * @brief Receive watermark and transmit watermark mask and bit field
 */
#define SCI_FFRX_RXFFIL_M      0xC0U
#define SCI_FFTX_TXFFIL_M      0x30U
#define SCI_FFRX_RT_SHIFT       (6)
#define SCI_FFTX_TET_SHIFT      (4)

/**
 * @brief SCI interrupt status
 * 
 * @param SCI_RX_DATA_AVAILABLE_INT Receiver data available (non-FIFO mode or FIFOs disabled) or RCVR FIFO trigger level reached (FIFO mode and FIFOs enabled)
 * @param SCI_TX_EMPTY_INT Enable Transmit Holding Register Empty Interrupt.
 * @param SCI_LINE_STATUS_INT Overrun/parity/ framing errors, break interrupt
 * @param SCI_MODEN_STATUS_INT Enable Modem Status Interrupt
 * @param SCI_LSR_CLEAR_METHOD Method for clearing the status in the LSR
 * @param SCI_THRE_INT_MODE Programmable THRE Interrupt Mode Enable
 */
#define SCI_RX_DATA_AVAILABLE_INT   0x01U
#define SCI_TX_EMPTY_INT            0x02U
#define SCI_LINE_STATUS_INT         0x04U
#define SCI_MODEN_STATUS_INT        0x08U
#define SCI_LSR_CLEAR_METHOD        0x10U
#define SCI_THRE_INT_MODE           0x80U

/**
 * @brief SCI status
 * 
 * @param SCI_UART_STATUS_BUSY This indicates that a serial transfer is in progress
 * @param SCI_UART_STATUS_TF_NOT_FULL This is used to indicate that the transmit FIFO is not full
 * @param SCI_UART_STATUS_TF_EMPTY This is used to indicate that the transmit FIFO is completely empty
 * @param SCI_UART_STATUS_RF_NOT_EMPTY This is used to indicate that the receive FIFO contains one or more entries
 * @param SCI_UART_SYATUS_RF_FULL This is used to indicate that the receive FIFO is completely full.
 */
#define SCI_UART_STATUS_BUSY           0x01
#define SCI_UART_STATUS_TF_NOT_FULL    0x02
#define SCI_UART_STATUS_TF_EMPTY       0x04
#define SCI_UART_STATUS_RF_NOT_EMPTY   0x08
#define SCI_UART_SYATUS_RF_FULL        0x10

/**
 * @brief SCI config parity check for error detection
 */
typedef enum
{
    SCI_CONFIG_PAR_NONE = 0x0000U,
    SCI_CONFIG_PAR_EVEN = 0x0018U,
    SCI_CONFIG_PAR_ODD  = 0x0008U
} SCI_ParityType;

/**
 * @brief SCI adopts 1-bit or 2-bit stop bit configuration.
 * 
 * @param SCI_STOP_1_5BIT_OR_2BIT 1.5 stop bits when DLS (LCR[1:0]) is zero, else 2 stop bit
 */
typedef enum
{
	SCI_STOP_1BIT = 0X00,
	SCI_STOP_1_5BIT_OR_2BIT = 0X04,
} SCI_StopBit;

/**
 * @brief Data bit length configuration in SCI
 */
typedef enum
{
	SCI_CHAR_5BITS = 0X00,
	SCI_CHAR_6BITS = 0X01,
	SCI_CHAR_7BITS = 0X02,
	SCI_CHAR_8BITS = 0X03,
} SCI_DataLenSelect;

/**
 * @brief This indicates the highest priority pending interrupt which can be one of the following types specified in values.
 */
typedef enum
{
	SCI_MODEM_STATUS         = 0x00,
	SCI_NO_INTERRUPT_PENDING = 0x01,
	SCI_THR_EMPTY            = 0x02,
	SCI_RX_DATA_AVAILABLE    = 0x04,
	SCI_RX_LINE_STATUS       = 0x06,
	SCI_BUSY_DETECT          = 0x07,
	SCI_CHARACTER_TIMEOUT    = 0x0c,
} SCI_INT_STATUS;

/**
 * @brief Configure transmit watermark to trigger FIFO mode transmission watermark interrupt
 */
typedef enum
{
    SCI_FIFO_TX0  = 0x0,

    SCI_FIFO_TX2  = 0x1,

    SCI_FIFO_TX4  = 0x2,

    SCI_FIFO_TX8  = 0x3,
} SCI_TxFIFOLevel;

/**
 * @brief Configure receive watermark to trigger FIFO mode receive watermark interrupt
 */
typedef enum
{
    SCI_FIFO_RX1  = 0x0,
    SCI_FIFO_RX4  = 0x1, 
    SCI_FIFO_RX8  = 0x2,
    SCI_FIFO_RX14 = 0x3,
} SCI_RxFIFOLevel;

/**
 * @brief This indicates the number of data entries in the receive FIFO.
 */
typedef enum
{
	SCI_RX_FIFO_0 = 0,
	SCI_RX_FIFO_1,
	SCI_RX_FIFO_2,
	SCI_RX_FIFO_3,
	SCI_RX_FIFO_4,
	SCI_RX_FIFO_5,
	SCI_RX_FIFO_6,
	SCI_RX_FIFO_7,
	SCI_RX_FIFO_8,
	SCI_RX_FIFO_9,
	SCI_RX_FIFO_10,
	SCI_RX_FIFO_11,
	SCI_RX_FIFO_12,
	SCI_RX_FIFO_13,
	SCI_RX_FIFO_14,
	SCI_RX_FIFO_15,
	SCI_RX_FIFO_16,
} SCI_RxFIFOStatus;


/**
 * @brief This indicates the number of data entries in the transmission FIFO.
 */
typedef enum
{
	SCI_TX_FIFO_0 = 0,
	SCI_TX_FIFO_1,
	SCI_TX_FIFO_2,
	SCI_TX_FIFO_3,
	SCI_TX_FIFO_4,
	SCI_TX_FIFO_5,
	SCI_TX_FIFO_6,
	SCI_TX_FIFO_7,
	SCI_TX_FIFO_8,
	SCI_TX_FIFO_9,
	SCI_TX_FIFO_10,
	SCI_TX_FIFO_11,
	SCI_TX_FIFO_12,
	SCI_TX_FIFO_13,
	SCI_TX_FIFO_14,
	SCI_TX_FIFO_15,
	SCI_TX_FIFO_16,
} SCI_TxFIFOStatus;

/**
 * @brief This bit is used to control the type of transmit mode during 9-bit data transfers.
 * @param TX_9BitMode_0 9-bit wide,software wr 9th bit
 * @param TX_9BitMode_1 8-bit wide(SCI_O_TAR)+ SEND_ADDR=1
 */
typedef enum{
    TX_9BitMode_0 = 0,
    TX_9BitMode_1,
}SCI_Mode9bit_Tx;

/**
 * @brief This bit is used to enable the address match feature during receive.
 * @param Rx_9BitMode_0 Normal mode
 * @param Rx_9BitMode_1 Address match mode
 */
typedef enum{
    Rx_9BitMode_0 = 0,
    Rx_9BitMode_1,
}SCI_Mode9bit_Rx;

/**
 * @brief This determines the DMA signalling mode used for the dma_tx_req_n and dma_rx_req_n output signals
 * @param DMAM_0 supports single DMA data transfers at a time
 * @param DMAM_1 supports multi-DMA data transfers untile FIFO filled
 */
typedef enum{
    DMAM_0 = 0,
    DMAM_1 = 1,
}SCI_DmaMode;

/**
 * @brief Get the sci status
 * @param SCI_RXSTATUS_READY This is used to indicate that the receiver contains at least one character in the RBR or the receiver FIFO
 * @param SCI_RXSTATUS_OVERRUN This is used to indicate the occurrence of an overrun error.
 * @param SCI_RXSTATUS_PARITY This is used to indicate the occurrence of a parity error in the receiver.
 * @param SCI_RXSTATUS_FRAMING This is used to indicate the occurrence of a framing error in the receiver.
 * @param SCI_RXSTATUS_BREAK This is used to indicate the detection of a break sequence on the serial input data.
 * @param SCI_TXREG_EMPTY this bit indicates that the THR or TX FIFO is empty.
 * @param SCI_TXSTATUS_EMPTY this bit is indicates the Transmitter Shift Register and the FIFO are both empty.
 * @param SCI_RXFIFO_ERROR This is used to indicate if there is at least one parity error, framing error, or break indication in the FIFO.
 */
typedef enum{
	SCI_RXSTATUS_READY   = 0x0001U,
	SCI_RXSTATUS_OVERRUN = 0x0002U,
	SCI_RXSTATUS_PARITY  = 0x0004U,
	SCI_RXSTATUS_FRAMING = 0x0008U,
	SCI_RXSTATUS_BREAK   = 0x0010U,
	SCI_TXREG_EMPTY      = 0x0020U,
	SCI_TXSTATUS_EMPTY   = 0x0040U,
	SCI_RXFIFO_ERROR     = 0x0080U
} SCI_Line_Status;


#ifdef DEBUG
__STATIC_INLINE boolean SCI_isBaseValid(uint32_t base)
{
    return(
           (base == SCIA_BASE) || (base == SCIB_BASE)
#if defined(SCIC_BASE)
           || (base == SCIC_BASE)
#endif
#if defined(SCID_BASE)
           || (base == SCID_BASE)
#endif
          );
}
#endif


/**
 * @brief Gets current line status flags. This function get the line status by reading LSR register.
 * @param [in] base is the base address of the SCI port.
 *
 * @return  This function returns the current line status flags:
 * 			SCI_RXSTATUS_READY, SCI_RXSTATUS_OVERRUN, SCI_RXSTATUS_PARITY
 * 			SCI_RXSTATUS_FRAMING, SCI_RXSTATUS_BREAK, SCI_TXREG_EMPTY
 * 			SCI_TXSTATUS_EMPTY, SCI_RXFIFO_ERROR
 */
__STATIC_INLINE SCI_Line_Status
SCI_getLineStatus(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Return the current value of the receive status register. */
    return (SCI_Line_Status)(HWREG(base + SCI_O_LSR));
}

/**
 * @brief This function get the status about FIFO status.
 * @param [in] base is the base address of the SCI port.
 *
 * @return the value of USR register
 *	       value & SCI_UART_STATUS_BUSY ?  BUSY : IDLE
 * 		   value & SCI_UART_STATUS_TF_NOT_FULL ? NOT FULL : FULL
 * 		   value & SCI_UART_STATUS_TF_EMPTY ? EMPTY : NOT EMPTY
 * 		   value & SCI_UART_STATUS_RF_NOT_EMPTY ? NOT_EMPTY : EMPTY
 * 		   value & SCI_UART_SYATUS_RF_FULL ? receive fifo full : not full
 */
__STATIC_INLINE uint32_t
SCI_getSCIStatusRegister(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    return(HWREG(base + SCI_O_USR));
}

/**
 * @brief Judgment the transmit fifo is or not full
 * @param [in] base is the base address of SCI port
 *
 * @return true the fifo is full
 * 		   false the fifo is not full
 */
__STATIC_INLINE boolean
SCI_isTxFifoFull(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    return ((HWREG(base + SCI_O_USR) & SCI_USR_TFNF) == 0);
}

/**
 * @brief Sets the type of parity.
 * @param [in] base is the base address of the SCI port.
 * @param [in] SCI_ParityType:
 * 				SCI_CONFIG_PAR_NONE
 * 				SCI_CONFIG_PAR_EVEN
 * 				SCI_CONFIG_PAR_ODD
 *
 * @return None
 */
__STATIC_INLINE void
SCI_setParityMode(uint32_t base, SCI_ParityType parity)
{

    ASSERT(SCI_isBaseValid(base));
    ASSERT((parity == SCI_CONFIG_PAR_NONE) ||
    	   (parity == SCI_CONFIG_PAR_EVEN) ||
		   (parity == SCI_CONFIG_PAR_ODD));

    /* Set the parity mode. */
    HWREG(base + SCI_O_LCR) = ((HWREG(base + SCI_O_LCR) &
                                 ~(SCI_CONFIG_PAR_MASK)) | (uint16_t)parity);
}

/**
 * @brief gets the type of parity.
 * @param [in] base is the base address of the SCI port.
 *
 * @return SCI_ParityType:
 *          SCI_ParityType:
 *          SCI_CONFIG_PAR_NONE
 *          SCI_CONFIG_PAR_EVEN
 *          SCI_CONFIG_PAR_ODD
 *
 */
__STATIC_INLINE SCI_ParityType
SCI_getParityMode(uint32_t base)
{
    uint16_t parity;

    ASSERT(SCI_isBaseValid(base));

    /* Return the current parity setting. */
    parity = (HWREG(base + SCI_O_LCR) & (SCI_CONFIG_PAR_MASK));

    return((SCI_ParityType)parity);
}

/**
 * @brief set multiprocess mode.
 * @param [in] base is the base address of the sci port.
 *
 * @return None.
 */
__STATIC_INLINE void
SCI_setAddrMultiProcessorMode(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Enable the address-bit mode protocol, 9bit mode enable */
    HWREG(base + SCI_O_LCR_EXT) |= SCI_LCR_EXT_DLS_E;
}

/**
 * @brief Sets the FIFO interrupt level at which interrupts are generated.
 * @param [in] base is the base address of the SCI port.
 * @param [in] SCI_TxFIFOLevel SCI_FIFO_TX0/SCI_FIFO_TX2/SCI_FIFO_TX4/SCI_FIFO_TX8
 * @param [in] SCI_RxFIFOLevel SCI_FIFO_RX1/SCI_FIFO_RX4/SCI_FIFO_RX8/SCI_FIFO_RX14
 *
 * @return None
 */
__STATIC_INLINE void
SCI_setFIFOInterruptLevel(uint32_t base, SCI_TxFIFOLevel txLevel,
                          SCI_RxFIFOLevel rxLevel)
{

    ASSERT(SCI_isBaseValid(base));

    ASSERT((txLevel == SCI_FIFO_TX0)||
    	   (txLevel == SCI_FIFO_TX2)||
		   (txLevel == SCI_FIFO_TX4)||
		   (txLevel == SCI_FIFO_TX8));

    ASSERT((rxLevel == SCI_FIFO_RX1)||
    	   (rxLevel == SCI_FIFO_RX4)||
		   (rxLevel == SCI_FIFO_RX8)||
		   (rxLevel == SCI_FIFO_RX14));

    uint32_t FCR_value = HWREG(base + SCI_O_SCR);

    FCR_value &= ~(0xf << SCI_FFTX_TET_SHIFT);

    FCR_value |= (((txLevel << SCI_FFTX_TET_SHIFT) | (rxLevel << SCI_FFRX_RT_SHIFT) | (SCI_FCR_FIFOE)));

    HWREG(base + SCI_O_IER_DLH) |= SCI_IER_PTIME;

    HWREG(base + SCI_O_IIR_FCR) = FCR_value;

    HWREG(base + SCI_O_SCR) = FCR_value;
}


/**
 * @brief Gets the FIFO interrupt level at which interrupts are generated.
 * @param [in] base the base address of the SCI port.
 * @param [out] txLevel point to the transmit FIFO interrupt level
 * @param [out]	rxLevel point to the receive FIFO interrupt level
 *
 * @return None
 */
__STATIC_INLINE void
SCI_getFIFOInterruptLevel(uint32_t base, SCI_TxFIFOLevel *txLevel,
                          SCI_RxFIFOLevel *rxLevel)
{
    ASSERT(SCI_isBaseValid(base));

    uint32_t FCR_value = HWREG(base + SCI_O_SCR);

    /* Extract the transmit and receive FIFO levels. */
    *txLevel = (SCI_TxFIFOLevel)((FCR_value & SCI_FFTX_TXFFIL_M) >> SCI_FFTX_TET_SHIFT);
    *rxLevel = (SCI_RxFIFOLevel)((FCR_value & SCI_FFRX_RXFFIL_M) >> SCI_FFRX_RT_SHIFT);
}

/**
 * @brief Gets the current configuration of a SCI.
 * @param [in] base the base address of the SCI port.
 * @param [in] lspclkHz the rate of the clock supplied to the SCI module.This is the LSPCLK.
 * @param [out] baud is a pointer to storage for the baud rate.
 * @param [out]	config is a pointer to storage for the data format.
 *
 * @return None
 */
__STATIC_INLINE void
SCI_getConfig(uint32_t base, uint32_t lspclkHz, uint32_t *baud,
              uint32_t *config)
{

    ASSERT(SCI_isBaseValid(base));

    HWREG(base + SCI_O_LCR) |= SCI_LCR_DLAB;
    *baud = lspclkHz /
            ((uint32_t)((HWREG(base + SCI_O_IER_DLH) << 8U) | HWREG(base + SCI_O_RBR_THR_DLL)) * 16U);
    HWREG(base + SCI_O_LCR) &= ~SCI_LCR_DLAB;

     /* Get the parity, data length, and number of stop bits. */
    *config = HWREG(base + SCI_O_LCR) & (SCI_CONFIG_PAR_MASK |
                                          SCI_CONFIG_STOP_MASK |
                                          SCI_CONFIG_WLEN_MASK);
}

/**
 * @brief Sets the configuration of a SCI.
 * @param [in] base is the base address of the SCI port.
 * @param [in] parity the type of parity.
 * @param [in] stop_bit the stop bit number.
 * @param [in] data_len the number of bits of valid data.
 * 	-	  SCI_ParityType is the parity type
 * 		  		SCI_CONFIG_PAR_NONE/SCI_CONFIG_PAR_EVEN/SCI_CONFIG_PAR_ODD
 * 	-	  SCI_DataLenSelect is the number of data bits
 * 		  	    SCI_CHAR_5BITS/SCI_CHAR_6BITS/SCI_CHAR_7BITS/SCI_CHAR_8BITS
 *
 * @return None.
 */
__STATIC_INLINE void
SCI_setInitConfig(uint32_t base, SCI_ParityType parity, SCI_StopBit stop_bit, SCI_DataLenSelect data_len)
{
	uint32_t LCR_value = 0;

	ASSERT(SCI_isBaseValid(base));

	ASSERT((parity == SCI_CONFIG_PAR_NONE) ||
		   (parity == SCI_CONFIG_PAR_EVEN) ||
		   (parity == SCI_CONFIG_PAR_ODD));

	ASSERT((stop_bit == SCI_STOP_1BIT) ||
		   (stop_bit == SCI_STOP_1_5BIT_OR_2BIT));

	ASSERT((data_len == SCI_CHAR_5BITS) ||
		   (data_len == SCI_CHAR_6BITS) ||
		   (data_len == SCI_CHAR_7BITS) ||
		   (data_len == SCI_CHAR_8BITS));

	LCR_value = (HWREG(base + SCI_O_LCR) & (~(SCI_CONFIG_PAR_MASK | SCI_CONFIG_STOP_MASK | SCI_CONFIG_WLEN_MASK)));

	LCR_value |= (parity | stop_bit | data_len);

	HWREG(base + SCI_O_LCR) = LCR_value;
}

/**
 * @brief Check Whether FIFO Function Is Enabled.
 * @param [in] base is the base address of the SCI port.
 *
 * @return TRUE if the FIFO enhancement is enabled.
 * 		   FALSE if the FIFO enhancement is disabled.
 */
__STATIC_INLINE boolean
SCI_isFIFOEnabled(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Return TRUE if the FIFO is enabled and FALSE if it is disabled. */
    return(((HWREG(base + SCI_O_IIR_FCR) & SCI_IIR_RO_FIFOSE_M) ==
            SCI_IIR_RO_FIFOSE_M) ? TRUE : FALSE);
}

/**
 * @brief Enables the transmit and receive FIFOs.
 *        This function must be called firstly, before the relevant fifo functions are configured.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
__STATIC_INLINE void
SCI_enableFIFO(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    if (SCI_isFIFOEnabled(base) == FALSE)
    {
    	uint32_t FCR_value = HWREG(base + SCI_O_SCR);

    	FCR_value |= SCI_FCR_FIFOE;
    	HWREG(base + SCI_O_IIR_FCR) = FCR_value;

    	HWREG(base + SCI_O_SCR) = FCR_value;
    }
}

/**
 * @brief Get the receive error status.
 * @param [in] base is the base address of SCI port.
 * 
 * @return Return the logical or result of SCI error flags.
 *          Receive correctly or error
 *          (bit(4): break interrupt bit)
 *          (bit(3): framing error)
 *          (bit(2): parity error bit)
 *          (bit(1): overrun error bit)
 */
__STATIC_INLINE uint32_t
SCI_getRxError(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    return (HWREG(base + SCI_O_LSR) & (SCI_LSR_FE | SCI_LSR_PE | SCI_LSR_BI | SCI_LSR_OE));
}

/**
 * @brief disable the transmit and receive fifo.
 * @param [in] base is the base address of the SCI poart.
 *
 * @return None
 */
__STATIC_INLINE void
SCI_disableFIFO(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    uint32_t FCR_value = HWREG(base + SCI_O_SCR);

    /* Disable the FIFO. */
    FCR_value &= ~SCI_FCR_FIFOE;
    HWREG(base + SCI_O_IIR_FCR) = FCR_value;

    HWREG(base + SCI_O_SCR) = FCR_value;
}


/**
 * @brief Resets the receive FIFO. Clear the receive fifo and shift register.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None.
 */
__STATIC_INLINE void
SCI_resetRxFIFO(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    uint32_t FCR_value = HWREG(base + SCI_O_SCR);

    /* Reset the specified FIFO.*/
    HWREG(base + SCI_O_IIR_FCR) = FCR_value | SCI_FCR_RFIFOR;
}

/**
 * @brief Resets the transmit FIFO. Clear the transmit fifo and shift register.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None.
 */
__STATIC_INLINE void
SCI_resetTxFIFO(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    uint32_t FCR_value = HWREG(base + SCI_O_SCR);

    /* Reset the specified FIFO. */
    HWREG(base + SCI_O_IIR_FCR) = FCR_value | SCI_FCR_XFIFOR;
}

/**
 * @brief Resets the SCI Transmit and Receive Channels.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None.
 */
__STATIC_INLINE void
SCI_resetChannels(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    uint32_t FCR_value = HWREG(base + SCI_O_SCR);

    /*Reset the Tx and Rx Channels*/
    HWREG(base + SCI_O_IIR_FCR) = FCR_value | (SCI_FCR_RFIFOR | SCI_FCR_XFIFOR);
}

/**
 * @brief Determines if there are any characters in the receive buffer when the FIFO enhancement is not enabled.
 * @param [in] base is the base address of the SCI port.
 *
 * @return TRUE if there is data in the receive buffer.
 *  	   FALSE if there is no data in the receive buffer.
 */
__STATIC_INLINE boolean
SCI_isDataAvailable(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Return the availability of characters with FIFO disabled. */
    return(((SCI_getLineStatus(base) & SCI_LSR_DR) ==
            SCI_LSR_DR) ? TRUE : FALSE);
}

/**
 * @brief Indicate that the receiver contains at least one character.
 * @param [in] base is the base address of the SCI port.
 *
 * @return 0x0 data not ready
 *         0x1 data ready
 */
__STATIC_INLINE uint32_t
SCI_rxDataReady(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    return (SCI_getLineStatus(base) & SCI_LSR_DR);
}

/**
 * @brief read the data byte received on serial input port
 * @param [in] base is the base address of the SCI port.
 *
 * @return uint32_t
 */
__STATIC_INLINE uint32_t
SCI_readData(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Return the availability of characters with FIFO disabled. */
    return ((HWREG(base + SCI_O_RBR_THR_DLL) & 0xff));
}

/**
 * @brief In FIFO mode, determine if the Transmitter Shift Register and the FIFO are both available.
 * 		  In nonFIFO mode, determint Transmitter buffer and Transmitter Shift Register are both available.
 * @param [in] base is the base address of the SCI port.
 *
 * @return TRUE if there is space available in the transmit buffer.
 *         FALSE if there is no space available in the transmit buffer.
 */
__STATIC_INLINE boolean
SCI_isSpaceAvailableNonFIFO(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

     /* Return the availability of space. */
    return(((SCI_getLineStatus(base) & SCI_LSR_TEMT) ==
            SCI_LSR_TEMT) ? TRUE : FALSE);
}

/**
 * @brief Determines the transmitter is empty
 * @param [in] base is the base address of the SCI port.
 *
 * @return  true transmitter is empty
 *          false transmitter is not empty
 */
__STATIC_INLINE boolean
SCI_isTxRegEmpty(uint32_t base)
{

    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    /* Return the availability of space. */
    return(((SCI_getLineStatus(base) & SCI_LSR_THRE) ==
    		SCI_LSR_THRE) ? TRUE : FALSE);
}

/**
 * @brief In FIFO mode, determine if the Transmitter Shift Register and the FIFO are both empty.
 * 		  In nonFIFO mode, determint Transmitter buffer and Transmitter Shift Register are both empty.
 *
 * @param [in] base Base address of the SCI port.
 *
 * @return boolean
 *   - true	: Transmitter Holding Register is empty (ready for new data).
 *   - false: Transmitter Holding Register is not empty (still occupied).
 */
__STATIC_INLINE boolean
SCI_isTxEmpty(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Return the availability of space. */
    return(((SCI_getLineStatus(base) & SCI_LSR_TEMT) ==
            SCI_LSR_TEMT) ? TRUE : FALSE);
}

/**
 * @brief Get the number of data entries in the transmit FIFO.
 * @param [in] base is the base address of the SCI port.
 *
 * @return uint32_t
 * - the current number of data entry in the transmit FIFO:
 *   SCI_FIFO_TX0, SCI_FIFO_TX1, SCI_FIFO_TX2, SCI_FIFO_TX3, .....,
 *   SCI_FIFO_TX16
 */
__STATIC_INLINE uint32_t
SCI_getTxFIFOStatus(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Get the number of data entries in the transmit FIFO. */
    return (SCI_TxFIFOStatus)(HWREG(base + SCI_O_TFL) & SCI_TFL_MSK);
}

/**
 * @brief Get the number of data entries in the receive FIFO.
 * @param [in] base is the base address of sci port
 *
 * @return uint32_t Returns the current number0~16 of words in the receive FIFO
 */
__STATIC_INLINE uint32_t
SCI_getRxFIFOStatus(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Get the current FIFO status */
    return (SCI_RxFIFOStatus)(HWREG(base + SCI_O_RFL) & SCI_RFL_MSK);
}

/**
 * @brief Determines whether the SCI transmitter is busy or not.
 * @param [in] base is the base address of the SCI port.
 *
 * @return  TRUE the SCI is transmitting
 *          FALSE transmissions are complete.
 */
__STATIC_INLINE boolean
SCI_isTransmitterBusy(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Check if FIFO enhancement is enabled. */
    if(SCI_isFIFOEnabled(base))
    {
    	/* With FIFO enhancement, determine if the SCI is busy. */
        return(((HWREGH(base + SCI_O_TFL) & SCI_TFL_MSK) != 0) ? TRUE : FALSE);
    }
    else
    {
        /* Without FIFO enhancement, determine if the SCI is busy.
           Check if the transmit buffer and shift register empty. */
        return(((SCI_getLineStatus(base) & SCI_LSR_TEMT) != SCI_LSR_TEMT) ? TRUE : FALSE);
    }
}

/**
 * @brief Waits to send a character from the specified port when the FIFO enhancement is enabled
 * @param [in] base is the base address of the SCI port.
 * @param [in] data is the character to be transmitted.
 *
 * @return None
 */
__STATIC_INLINE void
UART_writeCharBlockingFIFO(uint32_t base, uint8_t data)
{

     /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    /* Wait until space is available in the transmit FIFO. */
    while(SCI_isTxFifoFull(base))
    {
    }

    /* Send a char. */
    HWREG(base + SCI_O_RBR_THR_DLL) = data;
}

/**
 * @brief Waits to send a character from the specified port when the FIFO enhancement is enabled.
 * @param [in] base is the base address of the SCI port.
 * @param [in] data is the character to be transmitted.
 *
 * @return None
 * @remarks
 * - only the low 8 bit can be transmitted
 */
__STATIC_INLINE void
SCI_writeCharBlockingFIFO(uint32_t base, uint16_t data)
{

     /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    /* Wait until space is available in the transmit FIFO. */
    while(SCI_isTxFifoFull(base))
    {
    }

    /* Send a char. */
    HWREG(base + SCI_O_RBR_THR_DLL) = data;
}

/**
 * @brief Waits the avaliable space to send a character from the specified port when the FIFO enhancement is disabled
 * @param [in] base is the base address of the SCI port.
 * @param [in] data is the character to be transmitted.
 * 
 * @return None
 */
__STATIC_INLINE void
SCI_writeCharBlockingNonFIFO(uint32_t base, uint16_t data)
{

    ASSERT(SCI_isBaseValid(base));

    /*Wait until space is available in the transmit buffer. */
    while(!SCI_isSpaceAvailableNonFIFO(base))
    {
    }

    /* Send a char. */
    HWREG(base + SCI_O_RBR_THR_DLL) = data;
}

/**
 * @brief Waits the avaliable space to send a character from the specified port when the FIFO enhancement is disabled
 * @param [in] base is the base address of the SCI port.
 * @param [in] data is the character to be transmitted.
 *
 * @return None
 */
__STATIC_INLINE void
UART_writeCharBlockingNonFIFO(uint32_t base, uint8_t data)
{

    ASSERT(SCI_isBaseValid(base));

    /* Wait until space is available in the transmit buffer. */
    while(!SCI_isSpaceAvailableNonFIFO(base))
    {
    }

    /* Send a char. */
    HWREG(base + SCI_O_RBR_THR_DLL) = data;
}

/**
 * @brief Sends a character to the specified port.This function does not block and only writes to the transmit buffer.
 * @param [in] base is the base address of the SCI port.
 * @param [in] data is the character to be transmitted.
 * 
 * @return None
 */
__STATIC_INLINE void
SCI_writeCharNonBlocking(uint32_t base, uint16_t data)
{

    ASSERT(SCI_isBaseValid(base));

    /* Send a char. */
    HWREG(base + SCI_O_RBR_THR_DLL) = data;
}


/**
 * @brief Sends a character to the specified port.This function does not block and only writes to the transmit buffer.
 * @param [in] base is the base address of the SCI port.
 * @param [in] data is the character to be transmitted.
 *
 * @return None
 */
__STATIC_INLINE void
UART_writeCharNonBlocking(uint32_t base, uint8_t data)
{

    ASSERT(SCI_isBaseValid(base));

    /* Send a char.*/
    HWREG(base + SCI_O_RBR_THR_DLL) = data;
}

/**
 * @brief Waits for a character from the specified port when the FIFO enhancement is enabled.
 * @param [in] base is the base address of the SCI port.
 *
 * @return Returns uint16_t data
 * @remarks
 *  the character read from the specified port as uint16_t
 *  or 0x0 in case of Error. The application must use SCI_getLineStatus()
 *  API to check if some error occurred before consuming the data
 */
__STATIC_INLINE uint16_t
SCI_readCharBlockingFIFO(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Wait until a character is available in the receive FIFO. */
    while(SCI_getRxFIFOStatus(base) == SCI_RX_FIFO_0)
    {

    	/* If there is any error return */
        if((SCI_getLineStatus(base) & (SCI_LSR_FE | SCI_LSR_PE | SCI_LSR_OE)) != 0U)
        {
            return(0U);
        }
    }

    /* Return the character from the receive buffer. */
    return((uint16_t)(HWREG(base + SCI_O_RBR_THR_DLL) & SCI_RBR_SAR_M));
}

/**
 * @brief Waite reading a character from the specified port when the FIFO enhancement is not enabled.
 * @param [in] base is the base address of the SCI port.
 *
 * @return uint16_t
 * @remarks
 *  character read from the specified port as uint16_t.
 */
__STATIC_INLINE uint16_t
SCI_readCharBlockingNonFIFO(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Wait until a character is available in the receive FIFO. */
    while(!SCI_isDataAvailable(base))
    {
    }

     /* Return the character from the receive buffer. */
    return((uint16_t)(HWREG(base + SCI_O_RBR_THR_DLL) & SCI_RBR_SAR_M));
}

/**
 * @brief Gets a character from the receive buffer for the specified port.
 * 		  This function does not block and only reads the receive buffer.
 * @param [in] base is the base address of the SCI port.
 *
 * @return uin16_t
 */
__STATIC_INLINE uint16_t
SCI_readCharNonBlocking(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Return the character from the receive buffer. */
    return((uint16_t)(HWREG(base + SCI_O_RBR_THR_DLL) & SCI_RBR_SAR_M));
}

/**
 * @brief Performs a software reset of the SCI and Clears all reported receiver status flags.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
__STATIC_INLINE void
SCI_performSoftwareReset(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    SCI_getLineStatus(base);
}

/**
 * @brief Get UART Line Status Register (LSR) value.
 * @param [in] base is the base address of the SCI port.
 *
 * @return LSR register value containing:
 *         - Receiver status (bits 0-4): 
 *             DR, OE, PE, FE, BI
 *         - Transmitter status (bits 5-6):
 *             THRE, TEMT
.
 */
__STATIC_INLINE uint32_t
SCI_getRxStatus(uint32_t base)
{
    ASSERT(SCI_isBaseValid(base));

    /* Return the current value of the receive status register. */
    return(HWREG(base + SCI_O_LSR));
}

/**
 * @brief Enables Loop Back Test Mode
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
__STATIC_INLINE void
SCI_enableLoopback(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Set the loop back mode. */
    HWREG(base + SCI_O_MCR) |= SCI_MCR_LOOPBACK;
}

/**
 * @brief disables Loop Back Test Mode.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
__STATIC_INLINE void
SCI_disableLoopback(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Clear the loop back mode. */
    HWREG(base + SCI_O_MCR) &= ~SCI_MCR_LOOPBACK;
}


/**
 * @brief Get the receive FIFO Overflow flag status
 * @param [in] base is the base address of the SCI port.
 *
 * @return  TRUE if overflow has occurred.
 *          FALSE an overflow hasn't occurred.
 */
__STATIC_INLINE boolean
SCI_getOverflowStatus(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Return the current FIFO overflow status */
    return((SCI_getLineStatus(base) & SCI_LSR_OE) == SCI_LSR_OE);
}

/**
 * @brief Clear the receive FIFO Overflow flag status
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
__STATIC_INLINE void
SCI_clearOverflowStatus(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Clear the current FIFO overflow status */
    SCI_getLineStatus(base);
}

/**
 * @brief Enables individual SCI interrupt sources.
 * @param [in] base is the base address of the SCI port.
 * @param [in] intFlags is the bit mask of the interrupt sources to be enabled.
 * 	          intFlags:
 * 	             SCI_RX_DATA_AVAILABLE_INT       - Enable Received Data Available Interrupt
 * 		         SCI_TX_EMPTY_INT                - Enable Transmit Holding Register Empty Interrupt.
 * 		         SCI_LINE_STATUS_INT             - Enable Receiver Line Status Interrupt
 * 		         SCI_MODEN_STATUS_INT            - Enable Modem Status Interrupt
 * 		         SCI_LSR_CLEAR_METHOD            - method for clearing the status in the LSR
 * 		         SCI_THRE_INT_MODE               - Programmable THRE Interrupt Mode Enable
 *
 * @return None
 */
__STATIC_INLINE void
SCI_enableInterrupt(uint32_t base, uint32_t intFlags)
{

    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    ASSERT((intFlags == SCI_RX_DATA_AVAILABLE_INT) ||
    	   (intFlags == SCI_TX_EMPTY_INT) ||
		   (intFlags == SCI_LINE_STATUS_INT) ||
		   (intFlags == SCI_LSR_CLEAR_METHOD) ||
		   (intFlags == SCI_THRE_INT_MODE));

    /* Enable the specified interrupts. */
    HWREG(base + SCI_O_IER_DLH) |= intFlags;
}


/**
 * @brief disables individual SCI interrupt sources.
 * @param [in] base is the base address of the SCI port.
 * @param [in] intFlags is the bit mask of the interrupt sources to be disabled.
 *      	   intFlags:
 * 					SCI_RX_DATA_AVAILABLE_INT       - Enable Received Data Available Interrupt
 * 					SCI_TX_EMPTY_INT                - Enable Transmit Holding Register Empty Interrupt.
 * 					SCI_LINE_STATUS_INT             - Enable Receiver Line Status Interrupt
 * 					SCI_MODEN_STATUS_INT            - Enable Modem Status Interrupt
 * 					SCI_LSR_CLEAR_METHOD            - method for clearing the status in the LSR
 * 					SCI_THRE_INT_MODE               - Programmable THRE Interrupt Mode Enable
 *
 * @return None
 */
__STATIC_INLINE void
SCI_disableInterrupt(uint32_t base, uint32_t intFlags)
{

    ASSERT(SCI_isBaseValid(base));

    ASSERT((intFlags == SCI_RX_DATA_AVAILABLE_INT) ||
    	   (intFlags == SCI_TX_EMPTY_INT) ||
		   (intFlags == SCI_LINE_STATUS_INT) ||
		   (intFlags == SCI_LSR_CLEAR_METHOD) ||
		   (intFlags == SCI_THRE_INT_MODE));

    /* Disable the specified interrupts. */
    HWREG(base + SCI_O_IER_DLH) &= (~intFlags);
}

/**
 * @brief Gets the current interrupt status.
 * @param [in] base is the base address of the SCI port.
 *
 * @return SCI_INT_STATUS Returns the current interrupt status,
 *         enumerated as a bit field of
 *         SCI_MODEM_STATUS         = 0x00,
 *		   SCI_NO_INTERRUPT_PENDING = 0x01,
 *	       SCI_THR_EMPTY            = 0x02,
 *	       SCI_RX_DATA_AVAILABLE    = 0x04,
 *	       SCI_RX_LINE_STATUS       = 0x06,
 *	       SCI_BUSY_DETECT          = 0x07,
 *	       SCI_CHARACTER_TIMEOUT    = 0x0c,
 */
__STATIC_INLINE SCI_INT_STATUS
SCI_getInterruptStatus(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Return the interrupt status. */
    return (SCI_INT_STATUS)(HWREG(base + SCI_O_IIR_FCR) & 0x0f);
}

/**
 * @brief Gets the current interrupt source.
 * @param [in] base is the base address of the SCI port.
 *
 * @return Returns the current interrupt status,
 * - enumerated as a bit field of
 *         SCI_MODEM_STATUS         = 0x00,
 *		   SCI_NO_INTERRUPT_PENDING = 0x01,
 *	       SCI_THR_EMPTY            = 0x02,
 *	       SCI_RX_DATA_AVAILABLE    = 0x04,
 *	       SCI_RX_LINE_STATUS       = 0x06,
 *	       SCI_BUSY_DETECT          = 0x07,
 *	       SCI_CHARACTER_TIMEOUT    = 0x0c,
 */
__STATIC_INLINE SCI_INT_STATUS
SCI_getInterruptCause(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Return the interrupt status. */
    return (SCI_INT_STATUS)(HWREG(base + SCI_O_IIR_FCR) & 0x0f);
}

__STATIC_INLINE void
SCI_clearInterruptStatus(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Clear the requested interrupt sources. */
    SCI_getLineStatus(base);
}

/**
 * @brief Clears SCI interrupt sources.
 * @param [in] base is the base address of the SCI port.
 * @param [in] SCI_INT_STATUS:
 *  SCI_INT_STATUS:
 *  	SCI_MODEM_STATUS         = 0x00,
 *		SCI_NO_INTERRUPT_PENDING = 0x01,
 *		SCI_THR_EMPTY            = 0x02,
 *		SCI_RX_DATA_AVAILABLE    = 0x04,
 *		SCI_RX_LINE_STATUS       = 0x06,
 *		SCI_BUSY_DETECT          = 0x07,
 *		SCI_CHARACTER_TIMEOUT    = 0x0c,
 *
 * @return None
 */
extern void
SCI_clearInterruptFlag(uint32_t base, SCI_INT_STATUS intFlags);

/**
 * @brief Sets SCI Baud rate.
 * @param [in] base is the base address of the SCI port.
 * @param [in] lspclkHz is the rate of the clock supplied to the SCI module. This is the LSPCLK
 * @param [in] baud is the desired baud rate.
 *
 * @return None
 */
extern void
SCI_setBaud(uint32_t base, uint32_t lspclkHz, uint32_t baud);

extern void
SCI_setFractionalBaudRate(uint32_t base, uint32_t lspclkHz, uint32_t baud);

/**
 * @brief Send address control bit.
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
__STATIC_INLINE void
SCI_setSendAddrFlag(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    /* Set the TX send_addr flag bit to indicate
     that the next frame is an address frame. */
    HWREG(base + SCI_O_LCR_EXT) |= SCI_LCR_EXT_SEND_ADDR;
}

/**
 * @brief disable 9-bit data for transmit and receive transfers
 * @param [in] base is the base address of the SCI port.
 *
 * @return None
 */
__STATIC_INLINE void
SCI_disableMode9bit(uint32_t base)
{

    ASSERT(SCI_isBaseValid(base));

    HWREG(base + SCI_O_LCR_EXT) = 0;
}


/**
 * @brief set 9bit mode config
 * @param [in] base is the base address of the SCI port.
 * @param [in] txMode
 * @param [in] rxMode
 * - txMode
 * 		TX_9BitMode_0
 * 		TX_9BitMode_1
 * - rxMode:
 * 	 	RX_9BitMode_0
 * 	    RX_9BitMode_1
 *
 * @return None
 */
__STATIC_INLINE void
SCI_setConfigMode9bit(uint32_t base, SCI_Mode9bit_Tx txMode, SCI_Mode9bit_Rx rxMode)
{
    uint32_t mode = 0;

    ASSERT(SCI_isBaseValid(base));

     /* set mode by config */
    if(txMode == TX_9BitMode_1)
        mode |= SCI_LCR_EXT_TRANSMIT_MODE;
    if(rxMode == Rx_9BitMode_1)
        mode |= SCI_LCR_EXT_ADDR_MATCH;
    mode |= SCI_LCR_EXT_DLS_E;

    HWREG(base + SCI_O_LCR_EXT) = mode;
}

/**
 * @brief base is the base address of SCI port.
 * @param [in] SCI_DmaMode:
 * @details:
 *  SCI_DmaMode:
 * 		DMAM_0
 * 		DMAM_1
 *
 * @return  None
 */
__STATIC_INLINE void
SCI_setDmaMode(uint32_t base, SCI_DmaMode mode)
{
	uint32_t FCR_value = HWREG(base + SCI_O_SCR);

    if(mode == DMAM_0)
    {
        FCR_value &= ~SCI_FCR_DMAM;
    }
    else
    {
        FCR_value |= SCI_FCR_DMAM;
    }

    HWREG(base + SCI_O_IIR_FCR) = FCR_value;

    HWREG(base + SCI_O_SCR) = FCR_value;
}

/**
 * @brief stop external data transmission and receiving.
 * @param [in] base is the base address of SCI port.
 * @details Called before uart_init is initially set (to turn on the loopback),
 * 			and SCI_ClearReset() must be called after initialization (turn off the loopback).
 *
 * @return  None
 * @ref SCI_ClearReset(base)
 */
__STATIC_INLINE void
SCI_reset(uint32_t base)
{
	ASSERT(SCI_isBaseValid(base));

	SCI_enableLoopback(base);
	SCI_disableFIFO(base);

	while (SCI_rxDataReady(base))
	{
		SCI_readData(base);
	}

	SCI_resetChannels(base);

	SCI_clearInterruptStatus(base);
}

/**
 * @brief disable loopback, fifo transmit and receive enable
 * @param [in] base is the base address of SCI port.
 *
 * @return  None
 */

__STATIC_INLINE void
SCI_ClearReset(uint32_t base)
{
	ASSERT(SCI_isBaseValid(base));

	SCI_disableLoopback(base);
}

/**
 * @brief clear the flag:
 *        TX / RX FIFO
 *        Receive Error.
 * @param [in] base is the base address of SCI port.
 *
 * @return  None
 */
__STATIC_INLINE void
SCI_SoftReset(uint32_t base)
{
	ASSERT(SCI_isBaseValid(base));

	SCI_enableLoopback(base);
	while (SCI_rxDataReady(base))
	{
		SCI_readData(base);
	}

	SCI_resetChannels(base);
	SCI_resetRxFIFO(base);
	SCI_clearInterruptStatus(base);
	SCI_disableLoopback(base);
}

/**
 * @brief Sets the configuration of a SCI.
 * @param [in] base is the base address of the SCI port.
 * @param [in] lspclkHz is the rate of the clock supplied to the SCI module
 * @param [in] baud is the desired baud rate.
 * @param [in] config is the data format for the port (number of data bits, number of stop bits, and parity).
 *                  number of data bits:
 *                      SCI_CONFIG_WLEN_8, SCI_CONFIG_WLEN_7, ...., SCI_CONFIG_WLEN_1
 *                      number of stop bits:
 *                      SCI_CONFIG_STOP_ONE
 *                      SCI_CONFIG_STOP_TWO
 *                  parity:
 *        	            SCI_CONFIG_PAR_NONE
 *        	            SCI_CONFIG_PAR_EVEN
 *        	            SCI_CONFIG_PAR_ODD
 *
 * @return None
 */
extern void
SCI_setConfig(uint32_t base, uint32_t lspclkHz, uint32_t baud,
              uint32_t config);

/**
 * @brief send an array of characters from the specified port.
 * @param [in] base is the base address of the SCI port.
 * @param [in] array is the address of the array of characters to be transmitted.
 * @param [in] length is the number of characters in the array to be transmitted.
 *
 * @return None
 */
extern void
SCI_writeCharArray(uint32_t base, const uint16_t * const array,
                   uint32_t length);
extern void
UART_writeCharArray(uint32_t base, const uint8_t * const array,
                   uint32_t length);

/**
 * @brief read an array of characters from the specified port.
 * @param [in] base is the base address of the SCI port.
 * @param [in] array is the address of the array of characters to be readed.
 * @param [in] length is the number of characters in the array to be readed.
 *
 * @return None
 */
extern void
UART_readCharArray(uint32_t base, uint8_t * const array, uint32_t length);
extern void
SCI_readCharArray(uint32_t base, uint16_t * const array, uint32_t length);

/**
 * @brief Send Slave Address
 * @param [in] base is the base address of the SCI port.
 * @param [in] addr is the address of slave.asm
 * 
 * @return None
 */
extern void 
SCI_Mode9bit_SendAddr(uint32_t base, char addr);

/**
 * @brief Receive Address Sent by Remote Host
 * @param [in] base is the base address of the SCI port.
 * @param [out] addr pointer to the storage location of the received address
 * 
 * @return None
 */
extern int 
SCI_Mode9bit_RecvAddr(uint32_t base, char *addr);
extern void 
SCI_Mode9bit_writeCharArray(uint32_t base, const uint8_t * const array,uint16_t length);
extern void 
SCI_Mode9bit_readCharArray(uint32_t base, uint8_t * const array, uint16_t length);

#endif /* #if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22) */

#ifdef __cplusplus
}
#endif 

#endif	/* DEVICE_DRIVERLIB_SCI_H_ */
