/*
 * uart_V30.c
 *
 *  Created on: 2025 May 9
 *      Author: hpec
 */
#ifdef __cplusplus
extern "C"{
#endif

#include "gs32_version.h"
#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "uart.h"

#if IS_GS32F00xx(0x30) || IS_GS32F3xx(0x23)
void SCI_setConfig(uint32_t base, uint32_t lspclkHz, uint32_t baud, uint32_t config)
{
    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));
    ASSERT(baud != 0U);
    ASSERT((baud * 16U) <= lspclkHz);

    /* Stop the SCI. */
    SCI_disableModule(base);

    /* Set baud. */
    SCI_setBaud(base, lspclkHz, baud);

    /* Set parity, data length, and number of stop bits. */
    HWREGH(base + SCI_O_CCR) = ((HWREGH(base + SCI_O_CCR) &
                                 ~(SCI_CONFIG_PAR_MASK |
                                   SCI_CONFIG_STOP_MASK |
                                   SCI_CONFIG_WLEN_MASK)) | config);

    /* Start the SCI. */
    SCI_enableModule(base);
}

void UART_writeCharArray(uint32_t base, const uint8_t * const array,
                        uint32_t length)
{
    uint32_t i;
    char *pArray = (char *)array;

    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    /* Check if FIFO enhancement is enabled. */
    if(SCI_isFIFOEnabled(base))
    {

        /* FIFO is enabled.
        For loop to write (Blocking) 'length' number of characters */
        for(i = 0U; i < length; i++)
        {
           /* Wait until space is available in the transmit FIFO. */
            while(SCI_getTxFIFOStatus(base) == SCI_FIFO_TX16)
            {
            }

             /* Send a char. */
            HWREGH(base + SCI_O_TXBUF) = array[i];
        }
    }
    else
    {

        /* FIFO is not enabled.
        For loop to write (Blocking) 'length' number of characters */
        for(i = 0U; i < length; i++)
        {

            /* Wait until space is available in the transmit buffer. */
            while(!SCI_isSpaceAvailableNonFIFO(base))
            {
            }


            /* Send a char. */
            HWREGH(base + SCI_O_TXBUF) = array[i];
        }
    }
}

void SCI_writeCharArray(uint32_t base, const uint16_t * const array,
                        uint16_t length)
{
    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    uint16_t i;

    /* Check if FIFO enhancement is enabled. */
    if(SCI_isFIFOEnabled(base))
    {

        /* FIFO is enabled.
         For loop to write (Blocking) 'length' number of characters */
        for(i = 0U; i < length; i++)
        {

            /* Wait until space is available in the transmit FIFO. */
            while(SCI_getTxFIFOStatus(base) == SCI_FIFO_TX16)
            {
            }

            /* Send a char. */
            HWREGH(base + SCI_O_TXBUF) = array[i];
        }
    }
    else
    {

        /* FIFO is not enabled.
         For loop to write (Blocking) 'length' number of characters */
        for(i = 0U; i < length; i++)
        {

            /* Wait until space is available in the transmit buffer. */
            while(!SCI_isSpaceAvailableNonFIFO(base))
            {
            }


            /* Send a char. */
            HWREGH(base + SCI_O_TXBUF) = array[i];
        }
    }
}

void UART_readCharArray(uint32_t base, uint8_t * const array, uint32_t length)
{
    uint16_t i;
    char *pArray = (char *)array;

    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    /* Check if FIFO enhancement is enabled. */
    if(SCI_isFIFOEnabled(base))
    {

        /* FIFO is enabled.
         For loop to read (Blocking) 'length' number of characters */
        for(i = 0U; i < length; i++)
        {

            /* Wait until a character is available in the receive FIFO. */
#if IS_GS32F00xx(0x12, 0x30) || IS_GS32F3xx(0x22)
        	while(SCI_getRxFIFOStatus(base) == SCI_FIFO_RX0)
#else
            while(!SCI_isDataAvailableNonFIFO(base))
#endif
            {
            }


            /* Return the character from the receive buffer. */
            pArray[i] = (HWREGH(base + SCI_O_RXBUF) & SCI_RXBUF_SAR_M);
        }
    }
    else
    {

         /* FIFO is not enabled.
         For loop to read (Blocking) 'length' number of characters */
        for(i = 0U; i < length; i++)
        {

             /* Wait until a character is available in the receive buffer. */
            while(!SCI_isDataAvailableNonFIFO(base))
            {
            }


            /* Return the character from the receive buffer. */
            pArray[i] = (HWREGH(base + SCI_O_RXBUF) & SCI_RXBUF_SAR_M);
        }
    }
}

void SCI_readCharArray(uint32_t base, uint16_t * const array, uint16_t length)
{

    ASSERT(SCI_isBaseValid(base));

    uint16_t i;

    /* Check the arguments. */
    if(SCI_isFIFOEnabled(base))
    {
        /* FIFO is enabled.
         For loop to read (Blocking) 'length' number of characters */
        for(i = 0U; i < length; i++)
        {

            /* Wait until a character is available in the receive FIFO. */
            while(SCI_getRxFIFOStatus(base) == SCI_FIFO_RX0)
            {
            }


            /* Return the character from the receive buffer. */
            array[i] = (uint16_t)
                       (HWREGH(base + SCI_O_RXBUF) & SCI_RXBUF_SAR_M);
        }
    }
    else
    {
        /*FIFO is not enabled.
        For loop to read (Blocking) 'length' number of characters*/
        for(i = 0U; i < length; i++)
        {

            /* Wait until a character is available in the receive buffer. */
            while(!SCI_isDataAvailableNonFIFO(base))
            {
            }

            /* Return the character from the receive buffer.*/
            array[i] = (uint16_t)
                       (HWREGH(base + SCI_O_RXBUF) & SCI_RXBUF_SAR_M);
        }
    }
}


void SCI_enableInterrupt(uint32_t base, uint32_t intFlags)
{

    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    /* Enable the specified interrupts. */
    if((intFlags & SCI_INT_RXERR) == SCI_INT_RXERR)
    {
        HWREGH(base + SCI_O_CTL1) |= SCI_CTL1_RXERRINTENA;
    }
    if((intFlags & SCI_INT_RXRDY_BRKDT) == SCI_INT_RXRDY_BRKDT)
    {
        HWREGH(base + SCI_O_CTL2) |= SCI_CTL2_RXBKINTENA;
    }
    if((intFlags & SCI_INT_TXRDY) == SCI_INT_TXRDY)
    {
        HWREGH(base + SCI_O_CTL2) |= SCI_CTL2_TXINTENA;
    }
    if((intFlags & SCI_INT_TXFF) == SCI_INT_TXFF)
    {
        HWREGH(base + SCI_O_FFTX) |= SCI_FFTX_TXFFIENA;
    }
    if((intFlags & SCI_INT_RXFF) == SCI_INT_RXFF)
    {
        HWREGH(base + SCI_O_FFRX) |= SCI_FFRX_RXFFIENA;
    }
    if((intFlags & SCI_INT_RXTIMEOUT) == SCI_INT_RXTIMEOUT)
    {
        HWREGH(base+SCI_O_RXTIMEOUTINT) |= SCI_RXTIMEOUTINT_ENA;
    }
}

void SCI_disableInterrupt(uint32_t base, uint32_t intFlags)
{
    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    /* Disable the specified interrupts.*/
    if((intFlags & SCI_INT_RXERR) == SCI_INT_RXERR)
    {
        HWREGH(base + SCI_O_CTL1) &= ~SCI_CTL1_RXERRINTENA;
    }
    if((intFlags & SCI_INT_RXRDY_BRKDT) == SCI_INT_RXRDY_BRKDT)
    {
        HWREGH(base + SCI_O_CTL2) &= ~SCI_CTL2_RXBKINTENA;
    }
    if((intFlags & SCI_INT_TXRDY) == SCI_INT_TXRDY)
    {
        HWREGH(base + SCI_O_CTL2) &= ~SCI_CTL2_TXINTENA;
    }
    if((intFlags & SCI_INT_TXFF) == SCI_INT_TXFF)
    {
        HWREGH(base + SCI_O_FFTX) &= ~SCI_FFTX_TXFFIENA;
    }
    if((intFlags & SCI_INT_RXFF) == SCI_INT_RXFF)
    {
        HWREGH(base + SCI_O_FFRX) &= ~SCI_FFRX_RXFFIENA;
    }
    if((intFlags & SCI_INT_RXTIMEOUT) == SCI_INT_RXTIMEOUT)
    {
        HWREGH(base+SCI_O_RXTIMEOUTINT) &= ~SCI_RXTIMEOUTINT_ENA;
    }
}


uint32_t SCI_getInterruptStatus(uint32_t base)
{
    uint32_t interruptStatus = 0;

    ASSERT(SCI_isBaseValid(base));

    if((HWREGH(base + SCI_O_CTL2) & SCI_CTL2_TXRDY) == SCI_CTL2_TXRDY)
    {
        interruptStatus |= SCI_INT_TXRDY;
    }
    if((HWREGH(base + SCI_O_RXST) & SCI_RXST_RXERROR) == SCI_RXST_RXERROR)
    {
        interruptStatus |= SCI_INT_RXERR;
    }
    if(((HWREGH(base + SCI_O_RXST) & SCI_RXST_RXRDY) == SCI_RXST_RXRDY)  ||
       ((HWREGH(base + SCI_O_RXST) & SCI_RXST_BRKDT) ==  SCI_RXST_BRKDT))
    {
        interruptStatus |= SCI_INT_RXRDY_BRKDT;
    }
    if((HWREGH(base + SCI_O_FFTX) & SCI_FFTX_TXFFINT) == SCI_FFTX_TXFFINT)
    {
        interruptStatus |= SCI_INT_TXFF;
    }
    if((HWREGH(base + SCI_O_FFRX) & SCI_FFRX_RXFFINT) == SCI_FFRX_RXFFINT)
    {
        interruptStatus |= SCI_INT_RXFF;
    }
    if((HWREGH(base + SCI_O_RXST) & SCI_RXST_FE) == SCI_RXST_FE)
    {
        interruptStatus |= SCI_INT_FE;
    }
    if((HWREGH(base + SCI_O_RXST) & SCI_RXST_OE) == SCI_RXST_OE)
    {
        interruptStatus |= SCI_INT_OE;
    }
    if((HWREGH(base + SCI_O_RXST) & SCI_RXST_PE) == SCI_RXST_PE)
    {
        interruptStatus |= SCI_INT_PE;
    }
    if((HWREGH(base + SCI_O_RXTIMEOUTINT) & SCI_RXTIMEOUTINT_FLG) == SCI_RXTIMEOUTINT_FLG)
    {
        interruptStatus |= SCI_INT_RXTIMEOUT;
    }

    return(interruptStatus);
}


void SCI_clearInterruptStatus(uint32_t base, uint32_t intFlags)
{

    ASSERT(SCI_isBaseValid(base));

    /* Clear the requested interrupt sources. */
    if(((intFlags & SCI_INT_RXERR) == SCI_INT_RXERR) ||
       ((intFlags & SCI_INT_RXRDY_BRKDT) == SCI_INT_RXRDY_BRKDT) ||
       ((intFlags & SCI_INT_FE) == SCI_INT_FE) ||
       ((intFlags & SCI_INT_OE) == SCI_INT_OE) ||
       ((intFlags & SCI_INT_PE) == SCI_INT_PE))
    {
        SCI_performSoftwareReset(base);
    }
    if((intFlags & SCI_INT_TXFF) == SCI_INT_TXFF)
    {
        HWREGH(base + SCI_O_FFTX) |= SCI_FFTX_TXFFINTCLR;
    }
    if((intFlags & SCI_INT_RXFF) == SCI_INT_RXFF)
    {
        HWREGH(base + SCI_O_FFRX) |= SCI_FFRX_RXFFINTCLR;
    }
    if((intFlags & SCI_INT_RXTIMEOUT) == SCI_INT_RXTIMEOUT)
    {
        HWREGH(base + SCI_O_RXTIMEOUTINT) |= SCI_RXTIMEOUTINT_CLR;
    }
}


void SCI_setBaud(uint32_t base, uint32_t lspclkHz, uint32_t baud)
{
    uint32_t divider;


    /* Compute the baud rate divider. */
   	divider = (lspclkHz*4U  / baud );
   	uint32_t intDiv = divider >> 6;
   	uint32_t fractionDiv = divider & 0x3F;
   	uint32_t intDivbaud =  ((intDiv *2U)-1);

     /* Set the baud rate. */
   	HWREGH(base + SCI_O_HBAUD) = (intDivbaud & 0xFF00U) >> 8U;
    HWREGH(base + SCI_O_LBAUD) = intDivbaud & 0x00FFU;

   	HWREGH(base + SCI_O_DLF) = fractionDiv;
}


void SCI_setWakeFlag(uint32_t base)
{

    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    /* Set the TX wake flag bit to indicate
     that the next frame is an address frame. */
    HWREGH(base + SCI_O_CTL1) |= SCI_CTL1_TXWAKE;
}

#endif /* IS_GS32F00xx(0x30) || IS_GS32F3xx(0x23) */

#ifdef __cplusplus
}
#endif
