/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    xdma.c
*   @brief   
*   @details
*
*/
/*
 * commit history
 * 20240308, LYF, verify for chip 2.0
 * 20240319, LYF, Organize XDMA and DMAMUX driver version management
 * 20240612, LYF, move the HkSelect and HardInf configuration inside the driver by force 
 * 20240612, LYF, add XDMA_setRWLen for special application
 */


#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "stdio.h"
#include "xdma.h"

#if (XDMAC_VERSION == 0x20 || XDMAC_VERSION == 0x22)
#define XDMA_CFG_STEP        			0X04
/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */
/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

/**
 * @brief Configures the DMA channel
 * This function configures the source and destination addresses of a DMA
 * channel. The parameters are pointers to the data to be transferred.
 *
 * @param base is the base address of the DMA channel control registers.
 * @param *destAddr is the destination address.
 * @param *srcAddr is a source address.
 * @return None.
 */
GS32_DRIVER_DMA_FUNC_T void XDMA_configAddresses(uint32_t base, uint32_t destAddr, uint32_t srcAddr)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /* Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /* Set up SOURCE address*/
    hwChannel->SAR_L.bit.SAR = srcAddr;
    hwChannel->SAR_H.bit.SAR = 0x00;    

    /* Set up DESTINATION address*/
    hwChannel->DAR_L.bit.DAR = destAddr;
    hwChannel->DAR_H.bit.DAR = 0x00;


}

/**
 * @brief Configures the DMA channel's burst settings.
 * This function configures the size of each burst and the address step size.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @param size size is the number of words transferred per burst.
 * @param srcStep srcStep is the amount to increment or decrement the source address
 * after each word of a burst.
 * @param destStep destStep is the amount to increment or decrement the destination
 * address after each word of a burst.
 * @return None.
 */
GS32_DRIVER_DMA_FUNC_T void XDMA_configBurst(uint32_t base, uint16_t size, int16_t srcStep,
                     int16_t destStep)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    ASSERT((size >= 1U) && (size <= 32U));

    /* Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /* Set up BURST registers*/
    hwChannel->CTL_L.bit.SRC_MSIZE = size;
    hwChannel->CTL_L.bit.DST_MSIZE = size;
}

/**
 * @brief Configures the DMA channel's transfer settings.
 * This function configures the size of each burst and the address step size.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @param size size is the number of words transferred per burst.
 * @param srcStep srcStep srcStep is the amount to increment or decrement the source address
 * after each burst of a transfer unless a wrap occurs.
 * @param destStep destStep is the amount to increment or decrement the destination
 * address after each burst of a transfer unless a wrap occurs.
 * @return None.
 */
GS32_DRIVER_DMA_FUNC_T void XDMA_configTransfer(uint32_t base, uint32_t transferSize, int16_t srcStep,
                        int16_t destStep)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;
    uint32_t burst_size;
    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /* Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /* Set up TRANSFER registers*/
    burst_size =  hwChannel->CTL_L.bit.SRC_MSIZE;

    /* Check the total size.(Block Transfer Size.)*/
    ASSERT((transferSize * burst_size) <= 0x3FFFFFU);

    /* Set Block Transfer Size*/
    hwChannel->BLOCK_L.bit.BLOCK_TS = transferSize * burst_size;    
}

//*****************************************************************************
//
// DMA_configWrap
//
//*****************************************************************************
GS32_DRIVER_DMA_FUNC_T void XDMA_configWrap(uint32_t base, uint32_t srcWrapSize, int16_t srcStep,
                    uint32_t destWrapSize, int16_t destStep)
{
    ASSERT(FALSE);
}

/**
 * @brief Configures the DMA channel trigger and mode.
 * This function configures the DMA channel's trigger and mode.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @param trigger trigger is the interrupt source that triggers a DMA transfer.
 * @param config config is a bit field of several configuration selections.
 * @return None.
 */
GS32_DRIVER_DMA_FUNC_T void XDMA_configMode(uint32_t base, DMAMUX_TrigId_Type trigger, uint32_t config)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;
    uint8_t data_size;
    uint32_t channel = 0U;
    uint32_t dmamuxBase = 0U;

    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /* Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /* Covert the channel base to channel number, dmac and dmamux base address*/
    channel = XDMA_convertChnBase2ChnNum(base) ;
    dmamuxBase = XDMA_convertChnbase2DmamuxBase(base);

    /* Set up trigger selection in the CMA/CLA trigger source selection
     * registers. These are considered part of system control.
     */
    DMAMUX_setDmaMuxReqId(dmamuxBase, channel, trigger);

    /* ONESHOT MODE
    * do not support to config
	*/

    /*  Set MultBLK Type:
     * case 0: XDMA_CONTIGOUS;
     * case 1: XDMA_RELOAD
     * case 2: XDMA_SHADOW_REGISTER
     * case 3: XDMA_LINKED_LIST
     */
    ASSERT (((config & XDMA_CFG_MULTBLK_TYPE_M) >> XDMA_CFG_MULTBLK_TYPE_S) <= XDMA_LINKED_LIST);
    hwChannel->CFG_L.bit.DST_MULTBLK_TYPE = (config & XDMA_CFG_MULTBLK_TYPE_M) >> XDMA_CFG_MULTBLK_TYPE_S;
    hwChannel->CFG_L.bit.SRC_MULTBLK_TYPE = (config & XDMA_CFG_MULTBLK_TYPE_M) >> XDMA_CFG_MULTBLK_TYPE_S;

    /* Set SRC/DST length.
     * case 0: XDMA_BTL_1
     * case 1: XDMA_BTL_4
     * case 2: XDMA_BTL_8
     * case 3: XDMA_BTL_16
     * case 4: XDMA_BTL_32
     * case 5: XDMA_BTL_64
     * case 6: XDMA_BTL_128
     * case 7: XDMA_BTL_256
     * case 8: XDMA_BTL_512
     * case 9: XDMA_BTL_1024
     */

    ASSERT(((config & XDMA_CFG_DATA_SIZE_TYPE_M) >> XDMA_CFG_DATA_SIZE_TYPE_S) <= XDMA_BTL_1024);
    hwChannel->CTL_L.bit.SRC_MSIZE = (config & XDMA_CFG_DATA_SIZE_TYPE_M) >> XDMA_CFG_DATA_SIZE_TYPE_S;
    hwChannel->CTL_L.bit.DST_MSIZE = (config & XDMA_CFG_DATA_SIZE_TYPE_M) >> XDMA_CFG_DATA_SIZE_TYPE_S;

    /* Set SRC/DST Width.
     * case 0: XDMA_TR_WIDTH_BYTE_1
     * case 1: XDMA_TR_WIDTH_BYTE_2
     * case 2: XDMA_TR_WIDTH_BYTE_4
     * case 3: XDMA_TR_WIDTH_BYTE_8
     * case 4: XDMA_TR_WIDTH_BYTE_16
     * case 5: XDMA_TR_WIDTH_BYTE_32
     * case 6: XDMA_TR_WIDTH_BYTE_64
     * case 7: XDMA_TR_WIDTH_BYTE_128
     */
    ASSERT(((config & XDMA_CFG_DATA_WIDTH_TYPE_M) >> XDMA_CFG_DATA_WIDTH_TYPE_S) <=  XDMA_TR_WIDTH_BYTE_128);
    hwChannel->CTL_L.bit.DST_TR_WIDTH = 
        (config & XDMA_CFG_DATA_WIDTH_TYPE_M) >> XDMA_CFG_DATA_WIDTH_TYPE_S;
    hwChannel->CTL_L.bit.SRC_TR_WIDTH = 
        (config & XDMA_CFG_DATA_WIDTH_TYPE_M) >> XDMA_CFG_DATA_WIDTH_TYPE_S;
    
}

//*****************************************************************************
//
// XDMA_configChannel
//
//*****************************************************************************
GS32_DRIVER_DMA_FUNC_T void XDMA_configChannel(uint32_t base, const XDMA_ConfigParams *transfParams)
{
	XDMA_Channel_TypeDef *hwChannel = NULL;
    uint32_t dmamuxBase = 0U;
    uint32_t channel = 0U;
    XDMA_HandshakingSelect srcHkSelect = XDMA_HKS_SOFTWARE;
    XDMA_HandshakingSelect destHkSelect = XDMA_HKS_SOFTWARE;
    XDMA_HK_HARDWARE_INF   srcHardInf = XDMA_HKS_HARD_INF_14; 
    XDMA_HK_HARDWARE_INF   destHardInf= XDMA_HKS_HARD_INF_15;
    uint32_t regValue;
    
    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
#if !(DMAMUX_RGCR_P2P_SUPPORT)
    /* Not support DMAMUX P2P trigger case*/
    ASSERT((transfParams->dmaSrcReqId < DMAMUX_ReqId_max) ||
            (transfParams->ttfc == XDMA_TT_FC_2_P2M_DMAC) ||
            (transfParams->ttfc == XDMA_TT_FC_4_P2M_P) ||
            (transfParams->ttfc == XDMA_TT_FC_1_M2P_DMAC) ||
            (transfParams->ttfc == XDMA_TT_FC_5_P2P_SP));
#endif 

    /* Covert the channel base to the channel number and the dmamux base address..*/
    channel = XDMA_convertChnBase2ChnNum(base);    
    dmamuxBase = XDMA_convertChnbase2DmamuxBase(base);


    /* Point to the Channel register*/
    hwChannel = XDMA_Channel(base);  

    /*Handle speical configuration for XDMA in detial device.
     * These special configurations include, but are not limited to:
     * a: AXI bus layer
     * b: RAM type
     * 		RAM support D cache
     * 		RAM not support D cache
     * C: DMA can handle RAM cache by manual or Auto.
     */
    XDMA_doExtraConfOfChips(base, transfParams); 

    /* Forced distribution of the handshake signals by channel.*/
    switch (transfParams->ttfc)
    {
        case XDMA_TT_FC_2_P2M_DMAC:
        case XDMA_TT_FC_4_P2M_P:
            srcHkSelect  = XDMA_HKS_HARDWARE;    
            destHkSelect = XDMA_HKS_SOFTWARE;
            break;
        
        case XDMA_TT_FC_1_M2P_DMAC:
        case XDMA_TT_FC_6_M2P_P:
            srcHkSelect  = XDMA_HKS_SOFTWARE;    
            destHkSelect = XDMA_HKS_HARDWARE;
            break;
        
        case XDMA_TT_FC_3_P2P_DMAC:
        case XDMA_TT_FC_5_P2P_SP:
        case XDMA_TT_FC_7_P2P_DP:
            srcHkSelect  = XDMA_HKS_HARDWARE;    
            destHkSelect = XDMA_HKS_HARDWARE;

            break;
        
        default:
            srcHkSelect  = XDMA_HKS_SOFTWARE;    
            destHkSelect = XDMA_HKS_SOFTWARE;        
            break;
    }

    /* XDMA_HKS_HARD_INF_14 used as source handshaking signal of channel7
     * XDMA_HKS_HARD_INF_15 used as dest handshaking signal of channel7
     * XDMA_HKS_HARD_INF_0 used as source handshaking signal of channel0
     * XDMA_HKS_HARD_INF_1 used as dest handshaking signal of channel0
     * XDMA_HKS_HARD_INF_2 used as source handshaking signal of channel1
     * XDMA_HKS_HARD_INF_3 used as dest handshaking signal of channel1
     */
    srcHardInf   = 2* channel;
    destHardInf  = 2* channel + 1U;

    /*Configure DMA Channel source address and destination address*/
    XDMA_configAddresses(base, transfParams->destAddr, transfParams->srcAddr); 

    /* hwChannel->CFG_H.bit.TT_FC = transfParams->ttfc;
     * Transfer Type and Flow Control.
     * The following transfer types are supported.
     * 0=m2m, 1=m2p, 2=p2m, 3=p2p, 4=p2m_fc_p, 5=p2p_fc_sp, ......
     */
    regValue=HWREG(base+XDMA_O_XCTL);
    regValue=(regValue&(~XDMA_CTL_L_M))         \
    		 |(transfParams->srcBtl<<XDMA_SRC_MSIZE_S)
			 |(transfParams->destBtl<<XDMA_DST_MSIZE_S)
			 |(transfParams->srcAddrDirect<<XDMA_SINC_S)
			 |(transfParams->destAddrDirect<<XDMA_DINC_S)
			 |(transfParams->srcTrWidthBytes<<XDMA_SRC_TR_WIDTH_S)
			 |(transfParams->destTrWidthBytes<<XDMA_DST_TR_WIDTH_S);
    HWREG(base+XDMA_O_XCTL)=regValue;
    /* Source Burst Transaction Length, 1/4/8/.../1024cnt */
//    hwChannel->CTL_L.bit.SRC_MSIZE = transfParams->srcBtl;

      /*Destination Burst Transaction Length, 1/4/8/.../1024cnt  */
//    hwChannel->CTL_L.bit.DST_MSIZE = transfParams->destBtl;

      /*Source Address Increment, 0=increment src addr, 1=no change.*/
//    hwChannel->CTL_L.bit.SINC = transfParams->srcAddrDirect;

      /* Destination Address Increment,  0=increment src addr, 1=no change.*/
//    hwChannel->CTL_L.bit.DINC = transfParams->destAddrDirect;

      /* Source Transfer Width. 8/16/32/.../1024bit.*/
//    hwChannel->CTL_L.bit.SRC_TR_WIDTH = transfParams->srcTrWidthBytes;

      /*Destination Transfer Width, 8/16/32/.../1024bit.*/
//    hwChannel->CTL_L.bit.DST_TR_WIDTH = transfParams->destTrWidthBytes;


    /* Setting for ARLEN and AWLEN FIFO size*/
    XDMA_setRWLen(base, transfParams);

    /*Interrupt enable, it's used to mask all interrupts related to this channel */
    if (transfParams->enableInterrupt == TRUE)
    {
    	/*Enable the globally interrupt generation*/
        XDMA_enableInterrupt(base);
        /* Enable all the interrupt signal generation for current channel.*/
        hwChannel->ENINTSIGL_H.all = XDMA_CHANNEL_INT_SIGL_H_ENABLE;
        hwChannel->ENINTSIGL_L.all = XDMA_CHANNEL_INT_SIGL_L_ENABLE;
    }
    else
    {
    	/* Enable the globally interrupt generation*/
        XDMA_disableInterrupt(base);
        /* Enable all the interrupt signal generation for current channel.*/
        hwChannel->ENINTSIGL_H.all = XDMA_CHANNEL_INT_SIGL_H_DISABLE;
        hwChannel->ENINTSIGL_L.all = XDMA_CHANNEL_INT_SIGL_L_DISABLE;
    }

    /* ! The number programmed into BLOCK_TS field indicates the total number of data of width
     * ! CHx_CTL.SRC_TR_WIDTH to be transferred in a DMA block transfer.
     * ! Block Transfer Size = BLOCK_TS + 1U
     */
    hwChannel->BLOCK_L.bit.BLOCK_TS = (transfParams->blockTS - 1U) & 0x3FFFFF;




    regValue=HWREG(base+XDMA_O_XCFG+XDMA_CFG_STEP);

    regValue=(regValue&(~XDNA_CFG_H_M))      \
    		 |(transfParams->ttfc<<XDMA_TT_FC_S)
			 |(transfParams->chPriority<<XDMA_CH_PRIOR_S)
			 |(srcHkSelect<<XDMA_HS_SEL_SRC_S)
			 |(destHkSelect<<XDMA_HS_SEL_DST_S)
			 |(srcHardInf<<XDMA_SRC_PER_S)
			 |(destHardInf<<XDMA_DST_PER_S);
    HWREG(base+XDMA_O_XCFG+XDMA_CFG_STEP)=regValue;

    regValue=HWREG(base+XDMA_O_XCFG);
    regValue=(regValue&(~XDNA_CFG_L_M))        \
    		 |(transfParams->reloadSrc<<XDMA_SRC_MULTBLK_TYPE_S)
			 |(transfParams->reloadDst<<XDMA_DST_MULTBLK_TYPE_S);
    HWREG(base+XDMA_O_XCFG)=regValue;

    /* Channel Priority, 0 is the lowest one, while 7(DMAX_NUM_CHANNELS-1) is the highest priority*/
//    hwChannel->CFG_H.bit.CH_PRIOR = transfParams->chPriority;

    /* Source Software or Hardware Handshaking Select.*/
//    hwChannel->CFG_H.bit.HS_SEL_SRC = srcHkSelect;
//
    /* Destination Software or Hardware Handshaking Select.*/
//    hwChannel->CFG_H.bit.HS_SEL_DST = destHkSelect;
//
    /*  Source Multi Block Transfer Type.
     * 0: Contiguous, 1:Reload, 2:Shadow Register, 3: Linked List
     */
//    hwChannel->CFG_L.bit.SRC_MULTBLK_TYPE = transfParams->reloadSrc;
//
    /* Destination Multi Block Transfer Type.
     * 0: Contiguous, 1:Reload, 2:Shadow Register, 3: Linked List
     */
//    hwChannel->CFG_L.bit.DST_MULTBLK_TYPE = transfParams->reloadDst;
//
    /* SRC_PER select handshaking signal from 0 to 15*/
//    hwChannel->CFG_H.bit.SRC_PER = srcHardInf;
//
    /* DST_PER select handshaking signal from 0 to 15*/
//    hwChannel->CFG_H.bit.DST_PER = destHardInf;

    /* Config for DMAMUX when use hardware handshaking signal. */
    if (transfParams->ttfc == XDMA_TT_FC_2_P2M_DMAC || transfParams->ttfc == XDMA_TT_FC_3_P2P_DMAC ||
        transfParams->ttfc == XDMA_TT_FC_4_P2M_P || transfParams->ttfc == XDMA_TT_FC_5_P2P_SP || transfParams->ttfc == XDMA_TT_FC_7_P2P_DP)
        DMAMUX_configDmaMux(dmamuxBase, srcHardInf, transfParams->dmaSrcReqId);
    if (transfParams->ttfc == XDMA_TT_FC_1_M2P_DMAC || transfParams->ttfc == XDMA_TT_FC_3_P2P_DMAC ||
        transfParams->ttfc == XDMA_TT_FC_5_P2P_SP || transfParams->ttfc == XDMA_TT_FC_6_M2P_P || transfParams->ttfc == XDMA_TT_FC_7_P2P_DP)
        DMAMUX_configDmaMux(dmamuxBase, destHardInf, transfParams->dmaDstReqId);
}

//*****************************************************************************
//
// Clear XDMA_configChannel
//
//*****************************************************************************
GS32_DRIVER_DMA_FUNC_T void XDMA_DeConfChannel(uint32_t base, const XDMA_ConfigParams *transfParams)
{
    uint32_t dmamuxBase = 0U;
    uint32_t channel = 0U;    
    XDMA_HK_HARDWARE_INF srcHardInf = XDMA_HKS_HARD_INF_14; 
    XDMA_HK_HARDWARE_INF destHardInf = XDMA_HKS_HARD_INF_15;

    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /* Covert the channel base to the channel number and the dmamux base address..*/
    channel = XDMA_convertChnBase2ChnNum(base);
    dmamuxBase = XDMA_convertChnbase2DmamuxBase(base);

    /* DMA_HKS_HARD_INF_0 used as source handshaking signal of channel0
     * DMA_HKS_HARD_INF_1 used as dest handshaking signal of channel0
     * DMA_HKS_HARD_INF_2 used as source handshaking signal of channel1
     * DMA_HKS_HARD_INF_3 used as dest handshaking signal of channel1
     * ..............................................................
     * ..............................................................
     * DMA_HKS_HARD_INF_14 used as source handshaking signal of channel7
     * DMA_HKS_HARD_INF_15 used as dest handshaking signal of channel7
     */
    srcHardInf   = 2* channel;
    destHardInf  = 2* channel + 1U;

    if (transfParams->ttfc == XDMA_TT_FC_2_P2M_DMAC || transfParams->ttfc == XDMA_TT_FC_3_P2P_DMAC ||
        transfParams->ttfc == XDMA_TT_FC_4_P2M_P || transfParams->ttfc == XDMA_TT_FC_5_P2P_SP || transfParams->ttfc == XDMA_TT_FC_7_P2P_DP)
        DMAMUX_deConfDmaMux(dmamuxBase, srcHardInf, transfParams->dmaSrcReqId);
    if (transfParams->ttfc == XDMA_TT_FC_1_M2P_DMAC || transfParams->ttfc == XDMA_TT_FC_3_P2P_DMAC ||
        transfParams->ttfc == XDMA_TT_FC_5_P2P_SP || transfParams->ttfc == XDMA_TT_FC_6_M2P_P || transfParams->ttfc == XDMA_TT_FC_7_P2P_DP)
        DMAMUX_deConfDmaMux(dmamuxBase, destHardInf, transfParams->dmaDstReqId);
}


//*****************************************************************************
//
// XDMA_configChannel
//
//*****************************************************************************
GS32_DRIVER_DMA_FUNC_T void XDmaChan_config(uint32_t base, const XDmaCh_Parameters *chParams)
{
    uint32_t ttfc = chParams->cfg.cfg_h.bit.TT_FC;
    uint32_t srcHkSelect = chParams->cfg.cfg_h.bit.HS_SEL_SRC;
    uint32_t destHkSelect = chParams->cfg.cfg_h.bit.HS_SEL_DST;
    uint32_t srcHardInf = chParams->cfg.cfg_h.bit.SRC_PER;
    uint32_t destHardInf = chParams->cfg.cfg_h.bit.DST_PER;
    uint32_t srcReqId = chParams->srcCcr.bit.dmareq_id;
    uint32_t destReqId = chParams->destCcr.bit.dmareq_id;
    uint32_t dmamuxBase = 0U;

    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /* Covert the channel base to the dmamux base address*/
    dmamuxBase = XDMA_convertChnbase2DmamuxBase(base);


    /* config ctl register*/
    XDMA_Ch_setCtl(base, (XDmaCh_Ctl *)&(chParams->ctl));

    /* config cfg register*/
    XDMA_Ch_setCfg(base, (XDmaCh_Cfg *)&(chParams->cfg));

    if (srcHkSelect == XDMA_HKS_HARDWARE)
    {
    	/* source is Peripheral*/
        if (ttfc == XDMA_TT_FC_2_P2M_DMAC || ttfc == XDMA_TT_FC_3_P2P_DMAC || ttfc == XDMA_TT_FC_4_P2M_P ||
            ttfc == XDMA_TT_FC_5_P2P_SP || ttfc == XDMA_TT_FC_7_P2P_DP)
        {
        	/* config dmamux ccr*/
            DMAMUX_setDmaMuxCcr(dmamuxBase, srcHardInf, chParams->srcCcr);

            /*config dmamux rgcr*/
            if (srcReqId >= 0 && srcReqId <= 7)
                DMAMUX_setDmaMuxRgcr(dmamuxBase, srcReqId, chParams->srcRgcr);
        }
    }

    if (destHkSelect == XDMA_HKS_HARDWARE)
    {
    	/* dest is Peripheral*/
        if (ttfc == XDMA_TT_FC_1_M2P_DMAC || ttfc == XDMA_TT_FC_3_P2P_DMAC || ttfc == XDMA_TT_FC_5_P2P_SP ||
            ttfc == XDMA_TT_FC_6_M2P_P || ttfc == XDMA_TT_FC_7_P2P_DP)
        {
        	/* config dmamux ccr*/
            DMAMUX_setDmaMuxCcr(dmamuxBase, destHardInf, chParams->destCcr);

            /* config dmamux rgcr*/
            if (destReqId >= 0 && destReqId <= 7)
                DMAMUX_setDmaMuxRgcr(dmamuxBase, destReqId, chParams->destRgcr);
        }
    }
}

#ifdef __cplusplus
}
#endif

#endif /* #if (XDMAC_VERSION == 0x20 || XDMAC_VERSION == 0x22) */

