/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/**
*
* @file     gj320_math.h
*
******************************************************************************/
/*
 * commit history
 * 20240315, Jason, change __fmin and __fmax to use none GCC lib function.
 * 20240317, Jason, add MACRO define for sin cos sqrt for IQmathLib Float_Math type.
 * 20240318, Zhao Lei, add 3.0 support.
 * 20240330, Jason, update __sin and __atan2 define for 2.0ES DSP,
 *                  add __fmin and __fmax for DSP.
 * 20240407, ralf, update __fsqsum, delete __fmin&__fmax, which are not supported
 * 20240410, Jason, add "__iqn2float" define to remove warnings for 2.0ES
 * 20240413, Jason, add "__iqn2float" and "__float2iqn" static inline functiond define.
 * 20240507, ralf, add __fmin,__fmax, __mmaxf32, __mminf32 instructions for dsp;
 * 20240514, Jason, replace #if defined(GJ320F00xx) with #if IS_GJ320F00xx(0x11)
 *                  add 0x22 version into support group of __gs_xxxx instructions
 * 20240514, Jason, split this file into gj320_dsp_math.h and gj320_arm_math.h
 * 20240523, ralf, change the location of argument dst to instructions as gcc version:_vmulsum3_f3_r1, vmulsum3_f3_r1_neg1, vmulsum3_f3_r1_neg2,
 *                 vmulsum3_f3_r1_neg3, vmulsum3_f3_r1_neg12, vmulsum3_f3_r1_neg13, vmulsum3_f3_r1_neg23, vmulsum3_f3_r1_neg123,
 *                 vmulsum_mv3_r0_v3_m1, vmulsum_f3_r1_v3, vmulsum_f3_r1_v3_m1, vec_3p2l_oc2, vec_3p2l_abc, zi_3p2l_uz2, s24_3p3l_t.
 * 20240627, ralf, updated nice instructions with compiler, add the aux and ext instructions.
 * 20241218, ralf, merge gs32_dsp_math.h into gs32_math.h, and delete dsp instructions declaration;
 * 20250102, LiLu, add "__tan" "__asin" and "__acos" static inline function define.
 * 20250314, ralf, add "__expf32" "__logf32" and "__powf32" static inline function declaration;
 * 20250605, LiLu, add "__mminf32" "__mmaxf32" "__einvf32" and "__eisqrtf32" static inline function;
 */
#ifndef GS32_MATH_LIB_H
#define GS32_MATH_LIB_H

/******************************************************************************
* Includes
******************************************************************************/
#include "gs32_version.h"
#include "inc/hw_types.h"
#include <math.h>
//#include <stdlib.h>

/******************************************************************************
* MACRO Constants Defines
******************************************************************************/
#define MATH_2PI_INV   0.1591549431f
#define MATH_PI_INV    0.3183098862f
#define MATH_PI_DIV4   0.7853981634f
#define MATH_PI_DIV2   1.5707963268f
#define MATH_PI        3.1415926536f
#define MATH_2PI       6.2831853072f

#define MATH_MINUS_2PI_INV   -0.1591549431f
#define MATH_MINUS_PI_INV    -0.3183098862f
#define MATH_MINUS_PI_DIV4   -0.7853981634f
#define MATH_MINUS_PI_DIV2   -1.5707963268f
#define MATH_MINUS_PI        -3.1415926536f
#define MATH_MINUS_2PI       -6.2831853072f

#define sqrt         __sqrt
#define sqrtf        __sqrt
#define sin          __sin
#define sinf         __sin
#define cos          __cos
#define cosf         __cos
#define fabs         __fabs
#define fabsf        __fabs
#define atan2        __atan2
#define atan2f       __atan2

/******************************************************************************
* Prototypes for the APIs.
******************************************************************************/

/*******************GS32 custom instructions**********************************
 ****************************************************************************/
/*
 * @brief		   		float divider
 * @param[in] y
 * @param[in] x
 *
 * @return          	y/x
 */
__EXTERN_INLINE_MANUAL float __divf32(float y, float x)
{
    return __gs_dsp_fast_fdiv(y, x);
}

/*
 * @brief		   		float square add
 * @param[in] src1
 * @param[in] src2
 *
 * @return          	src1*src1 + src2*src2
 */
__EXTERN_INLINE_MANUAL float __sqsumf(float src1, float src2)
{
    return __gs_dsp_sqsum(src1, src2);
}

/*
 * @brief		   		float square root
 * @param[in] src
 *
 * @return          	sqrt(src)
 */
__EXTERN_INLINE_MANUAL float __sqrt(float src)
{
    return __gs_dsp_fast_sqrt(src);
}
__EXTERN_INLINE_MANUAL float __sqrtf(float src)
{
    return __gs_dsp_fast_sqrt(src);
}

/*
 * @brief		   		Compute 1/x
 * @param[in] 			x
 *
 * @return          	1/x
 */
__EXTERN_INLINE_MANUAL float __einvf32(float x)
{
    return __divf32(1, x);
}

/*
 * @brief		   		the square root of 1/x
 * @param[in] 			x
 *
 * @return          	1/sqrt(x)
 */
__EXTERN_INLINE_MANUAL float __eisqrtf32(float x)
{
    return __divf32(1, __sqrt(x));
}

/*
 * @brief		   		float sine value of src per unit
 * @param[in] src
 *
 * @return          	sin_pu(src)
 */
__EXTERN_INLINE_MANUAL float __sinpuf32(float src)
{
    return __gs_dsp_fsin_pu(src);
}

/*
 * @brief		   		float cosine value of src per unit
 * @param[in] src
 *
 * @return          	cos_pu(src)
 */
__EXTERN_INLINE_MANUAL float __cospuf32(float src)
{
    return __gs_dsp_fcos_pu(src);
}

/*
 * @brief		   		float actange value of src per unit
 * @param[in] src
 *
 * @return          	atan_pu(src)
 */
__EXTERN_INLINE_MANUAL float __atanpuf32(float src)
{
    return __gs_dsp_fatan_pu(src);
}

/*
 * @brief		   		float sine value of src
 * @param[in] src
 *
 * @return          	sin(src)
 */
__EXTERN_INLINE_MANUAL float __sin(float src)
{
    return __gs_dsp_fsin(src);
}

/*
 * @brief		   		float cosine value of src
 * @param[in] src
 *
 * @return          	cos(src)
 */
__EXTERN_INLINE_MANUAL float __cos(float src)
{
    return __gs_dsp_fcos(src);
}

/*
 * @brief		   		float tangent value of src
 * @param[in] src
 *
 * @return          	tan(src)
 */
__EXTERN_INLINE_MANUAL float __tan(float src)
{
    return __divf32(  __sin(src), __cos(src));
}



/*
 * @brief		   		float actange value of src
 * @param[in] src
 *
 * @return          	atan(src)
 */
__EXTERN_INLINE_MANUAL float __atan(float src)
{
    return __gs_dsp_fatan(src);
}

//__EXTERN_INLINE_MANUAL float __quadrant(float y, float x) // div == y/x
//{
//    return __gs_dsp_quadrant(y, x);
//}
//
//__EXTERN_INLINE_MANUAL float __quad_rat(float ydx, float xdy)
//{
//    return __gs_dsp_quad_rat(ydx, xdy);
//}

/*
 * @brief		   		float atan2_pu
 * @param[in] y
 * @param[in] x
 *
 * @return          	atan2_pu(src2, src1)
 */
__EXTERN_INLINE_MANUAL float __atan2puf32(float y, float x)
{
#if IS_GS32F00xx(0x30)
    return __gs_dsp_quad_atan2_pu(y, x);
#elif IS_GS32F00xx(0x12)||IS_GS32F3xx(0x22)
    return __gs_dsp_quad_atan2_pu_1p(y, x);
#else
	#error "__atan2puf32 instruction is not supported!"
#endif
}

/*
 * @brief		   		float atan2
 * @param[in] y
 * @param[in] x
 *
 * @return          	atan2(src2, src1)
 */
__EXTERN_INLINE_MANUAL float __atan2(float y, float x)
{
#if IS_GS32F00xx(0x30)
    return __gs_dsp_quad_atan2(y, x);
#elif IS_GS32F00xx(0x12)||IS_GS32F3xx(0x22)
    return __gs_dsp_quad_atan2_1p(y, x);
#else
	#error "__atan2 instruction is not supported!"
#endif
}


/*
 * @brief		   		float asin
 * @param[in] x
 *
 * @return          	asin(src)
 */
__EXTERN_INLINE_MANUAL float __asin(float x)
{
    return __atan2(x, __sqrtf(1-x*x));
}

//__EXTERN_INLINE_MANUAL float __asin(float x)
//{
//    return __atan(__divf32(x, __sqrtf(1-x*x))); //Domain is not wide enough.
//}


/*
 * @brief		   		float acos
 * @param[in] x
 *
 * @return          	acos(src)
 */
__EXTERN_INLINE_MANUAL float __acos(float x)
{
    return __atan2( __sqrtf(1-x*x) , x );
}



/*
 * @brief		   			converter an iqn value to float
 * @param[in] src			32bit IQN int
 * @param[in] imm5u_const 	q number of src
 *
 * @return          		float value of src
 */
#define __iqn2float(src, imm5u_const)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_iqn2float (src, imm5u_const);	\
    __res;                      \
  })

/*
 * @brief		   			converter an float value to iqn
 * @param[in] src			float
 * @param[in] imm5u_const 	q number of return
 *
 * @return          		iqn value of src
 */
//#define __float2iqn(src, imm5u_const)    \
//  __extension__                 \
//  ({                            \
//    int32 __res = __gs_dsp_float2iqn (src, imm5u_const);	\
//    __res;                      \
//  })
__EXTERN_INLINE_MANUAL int32_t __float2iqn(float src, const uint32_t imm5u_const)
{
	uint32_t u32tmp = *(uint32_t *)(&src);
	if(u32tmp == 0x80000000)
		return 0;
	else {
		return __gs_dsp_float2iqn (src, imm5u_const);
	}
}

/*
 * @brief					get the integer partition of float
 * @param[in] src			float
 *
 * @return          		integer of src
 */
__EXTERN_INLINE_MANUAL float __ftrunc(float src)
{
    return __gs_dsp_ftrunc(src);
}

/*
 * @brief					get the fraction partition of float
 * @param[in] src			float
 *
 * @return          		fractional of src
 */
__EXTERN_INLINE_MANUAL float __ffract(float src)
{
    return __gs_dsp_ffract(src);
}

/*
 * @brief		   			upper&lower limitation to src
 * @param[in] src			float
 * @param[in] max, 			upper limit value
 * @param[in] min, 			lower limit value
 *
 * @return
 */
__EXTERN_INLINE_MANUAL float __fsat(float src, float max, float min)
{
    return __gs_dsp_fsat(src, max, min);
}

/*
 * @brief           	move src1, src2, src3 from rx registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r0(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r0(src1, src2, src3);
}

/*
 * @brief           	move -src1, src2, src3 from rx registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r0_neg1(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r0_neg1(src1, src2, src3);
}

/*
 * @brief           	move src1, -src2, src3 from rx registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r0_neg2(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r0_neg2(src1, src2, src3);
}

/*
 * @brief           	move src1, src2, -src3 from rx registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r0_neg3(uint32 src1, uint32 src2, uint32 src3)
{
    __gs_dsp_mv3_x2mpr_r0_neg3(src1, src2, src3);
}

/*
 * @brief           	move -src1, -src2, src3 from rx registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r0_neg12(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r0_neg12(src1, src2, src3);
}

/*
 * @brief           	move -src1, src2, -src3 from rx registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r0_neg13(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r0_neg13(src1, src2, src3);
}

/*
 * @brief           	move src1, -src2, -src3 from rx registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r0_neg23(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r0_neg23(src1, src2, src3);
}

/*
 * @brief           	move -src1, -src2, -src3 from rx registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r0_neg123(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r0_neg123(src1, src2, src3);
}

/*
 * @brief           	move src1, src2, src3 from rx registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r1(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r1       (src1, src2, src3);
}

/*
 * @brief           	move -src1, src2, src3 from rx registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r1_neg1(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r1_neg1(src1, src2, src3);
}

/*
 * @brief           	move src1, -src2, src3 from rx registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r1_neg2(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r1_neg2(src1, src2, src3);
}

/*
 * @brief           	move src1, src2, -src3 from rx registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r1_neg3(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r1_neg3(src1, src2, src3);
}

/*
 * @brief           	move -src1, -src2, src3 from rx registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r1_neg12(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r1_neg12(src1, src2, src3);
}

/*
 * @brief           	move -src1, src2, -src3 from rx registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r1_neg13(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r1_neg13(src1, src2, src3);
}

/*
 * @brief           	move src1, -src2, -src3 from rx registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r1_neg23(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r1_neg23(src1, src2, src3);
}

/*
 * @brief           	move -src1, -src2, -src3 from rx registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_x2mpr_r1_neg123(int32 src1, int32 src2, int32 src3)
{
    __gs_dsp_mv3_x2mpr_r1_neg123(src1, src2, src3);
}

/*
 * @brief           	move src1, src2, src3 from float registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r0(float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r0(src1, src2, src3);
}

/*
 * @brief           	move -src1, src2, src3 from float registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r0_neg1(float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r0_neg1(src1, src2, src3);
}

/*
 * @brief           	move src1, -src2, src3 from float registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r0_neg2(float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r0_neg2(src1, src2, src3);
}

/*
 * @brief           	move src1, src2, -src3 from float registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r0_neg3(float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r0_neg3(src1, src2, src3);
}

/*
 * @brief           	move -src1, -src2, src3 from float registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r0_neg12(float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r0_neg12(src1, src2, src3);
}

/*
 * @brief           	move -src1, src2, -src3 from float registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r0_neg13 (float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r0_neg13 (src1, src2, src3);
}

/*
 * @brief           	move src1, -src2, -src3 from float registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r0_neg23 (float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r0_neg23 (src1, src2, src3);
}

/*
 * @brief           	move -src1, -src2, -src3 from float registers to mpr row0
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r0_neg123(float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r0_neg123(src1, src2, src3);
}

/*
 * @brief           	move src1, src2, src3 from float registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r1(float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r1(src1, src2, src3);
}

/*
 * @brief           	move -src1, src2, src3 from float registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r1_neg1(float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r1_neg1(src1, src2, src3);
}

/*
 * @brief           	move src1, -src2, src3 from float registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r1_neg2(float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r1_neg2(src1, src2, src3);
}

/*
 * @brief           	move src1, src2, -src3 from float registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r1_neg3(float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r1_neg3(src1, src2, src3);
}

/*
 * @brief           	move -src1, -src2, src3 from float registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r1_neg12(float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r1_neg12(src1, src2, src3);
}

/*
 * @brief           	move -src1, src2, -src3 from float registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r1_neg13(float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r1_neg13(src1, src2, src3);
}

/*
 * @brief           	move src1, -src2, -src3 from float registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r1_neg23(float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r1_neg23(src1, src2, src3);
}

/*
 * @brief           	move -src1, -src2, -src3 from float registers to mpr row1
 * @param[in] src1
 * @param[in] src2
 * @param[in] src3
 *
 * @return          	None
 */
__EXTERN_INLINE_MANUAL void __mv3_f2mpr_r1_neg123(float src1, float src2, float src3)
{
    __gs_dsp_mv3_f2mpr_r1_neg123(src1, src2, src3);
}

/*
 * @brief           		move src1, src2, from rx registers to
 * 							mpr row1&loc1, rol2&loc2
 * @param[in] src1			32-bit int
 * @param[in] src2			32-bit int
 * @param[in] imm1u_row1	target mpr register row number to src1
 * @param[in] imm2u_loc1	target mpr register loc number to src1
 * @param[in] imm1u_row2	target mpr register row number to src2
 * @param[in] imm2u_loc2	target mpr register loc number to src2
 *
 * @return          		None
 */
#define __mv2_x2mpr(imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2)    \
  __extension__                 \
  ({                            \
    __gs_dsp_mv2_x2mpr      (imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2);	\
  })

/*
 * @brief           		move -src1, src2, from rx registers to
 * 							mpr row1&loc1, rol2&loc2
 * @param[in] src1			32-bit int
 * @param[in] src2			32-bit int
 * @param[in] imm1u_row1	target mpr register row number to src1
 * @param[in] imm2u_loc1	target mpr register loc number to src1
 * @param[in] imm1u_row2	target mpr register row number to src2
 * @param[in] imm2u_loc2	target mpr register loc number to src2
 *
 * @return          		None
 */
#define __mv2_x2mpr_neg1(imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2)    \
  __extension__                 \
  ({                            \
    __gs_dsp_mv2_x2mpr_neg1 (imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2);	\
  })

/*
 * @brief           		move src1, -src2, from rx registers to
 * 							mpr row1&loc1, rol2&loc2
 * @param[in] src1			32-bit int
 * @param[in] src2			32-bit int
 * @param[in] imm1u_row1	target mpr register row number to src1
 * @param[in] imm2u_loc1	target mpr register loc number to src1
 * @param[in] imm1u_row2	target mpr register row number to src2
 * @param[in] imm2u_loc2	target mpr register loc number to src2
 *
 * @return          		None
 */
#define __mv2_x2mpr_neg2(imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2)    \
  __extension__                 \
  ({                            \
    __gs_dsp_mv2_x2mpr_neg2 (imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2);	\
  })

/*
 * @brief           		move -src1, -src2, from rx registers to
 * 							mpr row1&loc1, rol2&loc2
 * @param[in] src1			32-bit int
 * @param[in] src2			32-bit int
 * @param[in] imm1u_row1	target mpr register row number to src1
 * @param[in] imm2u_loc1	target mpr register loc number to src1
 * @param[in] imm1u_row2	target mpr register row number to src2
 * @param[in] imm2u_loc2	target mpr register loc number to src2
 *
 * @return          		None
 */
#define __mv2_x2mpr_neg12(imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2)    \
  __extension__                 \
  ({                            \
    __gs_dsp_mv2_x2mpr_neg12(imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2);	\
  })

/*
 * @brief           		move src1, src2, from float registers to
 * 							mpr row1&loc1, rol2&loc2
 * @param[in] src1			float
 * @param[in] src2			float
 * @param[in] imm1u_row1	target mpr register row number to src1
 * @param[in] imm2u_loc1	target mpr register loc number to src1
 * @param[in] imm1u_row2	target mpr register row number to src2
 * @param[in] imm2u_loc2	target mpr register loc number to src2
 *
 * @return          		None
 */
#define __mv2_f2mpr(imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2)    \
  __extension__                 \
  ({                            \
    __gs_dsp_mv2_f2mpr      (imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2);	\
  })

/*
 * @brief           		move -src1, src2 from float registers to
 * 							mpr row1&loc1, rol2&loc2
 * @param[in] src1			float
 * @param[in] src2			float
 * @param[in] imm1u_row1	target mpr register row number to src1
 * @param[in] imm2u_loc1	target mpr register loc number to src1
 * @param[in] imm1u_row2	target mpr register row number to src2
 * @param[in] imm2u_loc2	target mpr register loc number to src2
 *
 * @return          		None
 */
#define __mv2_f2mpr_neg1(imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2)    \
  __extension__                 \
  ({                            \
    __gs_dsp_mv2_f2mpr_neg1 (imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2);	\
  })

/*
 * @brief           		move src1, -src2 from float registers to
 * 							mpr row1&loc1, rol2&loc2
 * @param[in] src1			float
 * @param[in] src2			float
 * @param[in] imm1u_row1	target mpr register row number to src1
 * @param[in] imm2u_loc1	target mpr register loc number to src1
 * @param[in] imm1u_row2	target mpr register row number to src2
 * @param[in] imm2u_loc2	target mpr register loc number to src2
 *
 * @return          		None
 */
#define __mv2_f2mpr_neg2(imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2)    \
  __extension__                 \
  ({                            \
    __gs_dsp_mv2_f2mpr_neg2 (imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2);	\
  })

/*
 * @brief           		move -src1, -src2 from float registers to
 * 							mpr row1&loc1, rol2&loc2
 * @param[in] src1			float
 * @param[in] src2			float
 * @param[in] imm1u_row1	target mpr register row number to src1
 * @param[in] imm2u_loc1	target mpr register loc number to src1
 * @param[in] imm1u_row2	target mpr register row number to src2
 * @param[in] imm2u_loc2	target mpr register loc number to src2
 *
 * @return          		None
 */
#define __mv2_f2mpr_neg12(imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2)    \
  __extension__                 \
  ({                            \
    __gs_dsp_mv2_f2mpr_neg12(imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2, src1, src2);	\
  })

/*
 * @brief           		move src1 from rx registers to
 * 							mpr row1&loc1
 * @param[in] src1			32-bit int
 * @param[in] imm1u_row1	target mpr register row number to src1
 * @param[in] imm2u_loc1	target mpr register loc number to src1
 *
 * @return          		None
 */
#define __mv1_x2mpr(imm1u_row, imm2u_loc, src1)    \
  __extension__                 \
  ({                            \
    __gs_dsp_mv1_x2mpr(imm1u_row, imm2u_loc, src1);	\
  })

/*
 * @brief           		move -src1 from rx registers to
 * 							mpr row1&loc1
 * @param[in] src1			32-bit int
 * @param[in] imm1u_row1	target mpr register row number to src1
 * @param[in] imm2u_loc1	target mpr register loc number to src1
 *
 * @return          		None
 */
#define __mv1_x2mpr_neg(imm1u_row, imm2u_loc, src1)    \
  __extension__                 \
  ({                            \
    __gs_dsp_mv1_x2mpr_neg(imm1u_row, imm2u_loc, src1);	\
  })

/*
 * @brief           		move src1 from float registers to
 * 							mpr row1&loc1
 * @param[in] src1			float
 * @param[in] imm1u_row1	target mpr register row number to src1
 * @param[in] imm2u_loc1	target mpr register loc number to src1
 *
 * @return          		None
 */
#define __mv1_f2mpr(imm1u_row, imm2u_loc, src1)    \
  __extension__                 \
  ({                            \
    __gs_dsp_mv1_f2mpr(imm1u_row, imm2u_loc, src1);	\
  })

/*
 * @brief           		move -src1 from float registers to
 * 							mpr row1&loc1
 * @param[in] src1			float
 * @param[in] imm1u_row1	target mpr register row number to src1
 * @param[in] imm2u_loc1	target mpr register loc number to src1
 *
 * @return          		None
 */
#define __mv1_f2mpr_neg(imm1u_row, imm2u_loc, src1)    \
  __extension__                 \
  ({                            \
    __gs_dsp_mv1_f2mpr_neg(imm1u_row, imm2u_loc, src1);	\
  })

/*
 * @brief           		move float value from mpr row1&loc1 to
 * 							float registers
 *
 * @param[in] imm1u_row	 	mpr register row number
 * @param[in] imm2u_loc	 	mpr register loc number
 *
 * @return          		float value
 */
#define __mv1_mpr2f(imm1u_row, imm2u_loc)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_mv1_mpr2f(imm1u_row, imm2u_loc);	\
    __res;                      \
  })

/*
 * @brief           		move int value from mpr row1&loc1 to
 * 							rx registers
 *
 * @param[in] imm1u_row	 	mpr register row number
 * @param[in] imm2u_loc	 	mpr register loc number
 *
 * @return          		int value
 */
#define __mv1_mpr2x(imm1u_row, imm2u_loc)    \
  __extension__                 \
  ({                            \
    int32 __res = __gs_dsp_mv1_mpr2x(imm1u_row, imm2u_loc);	\
    __res;                      \
  })

/*
 * @brief           		move two int value from mpr row1&loc1, row2&loc2 to
 * 							rx registers
 *
 * @param[in] imm1u_row1	mpr register row number of int1
 * @param[in] imm2u_loc1	mpr register loc number of int1
 * @param[in] imm1u_row2	mpr register row number of int2
 * @param[in] imm2u_loc2	mpr register loc number of int2
 *
 * @return          		return int64 type value, value1 located low 32bits,
 * 							and value2 located higher 32bits
 */
#define __mv2_mpr2x(imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2)    \
  __extension__                 \
  ({                            \
    int64 __res = __gs_dsp_mv2_mpr2x(imm1u_row1, imm2u_loc1, imm1u_row2, imm2u_loc2);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[v1_1, v1_2] in mpr row0 and vector2[v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*v2_1 + v1_2*v2_2
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2(void)
{
    return __gs_dsp_vmulsum_v2();
}

 /*
  * @brief					dot product of vector1[-v1_1, v1_2] in mpr row0 and vector2[v2_1, v2_2]
  * 						in mpr row1, value number in vector1 and vector2 is 2. Before this
  * 						instruction, vector1&vector2 must move into mpr register.
  *
  * @return					-v1_1*v2_1 + v1_2*v2_2
  */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0neg1 (void)
{
    return __gs_dsp_vmulsum_v2_r0neg1();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2] in mpr row0 and vector2[v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*-v2_1 + v1_2*v2_2
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0neg2 (void)
{
    return __gs_dsp_vmulsum_v2_r0neg2();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2] in mpr row0 and vector2[v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*-v2_1 + v1_2*v2_2
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0neg12(void)
{
    return __gs_dsp_vmulsum_v2_r0neg12();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2] in mpr row0 and vector2[-v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*v2_1 + (-v1_2*v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1neg1 (void)
{
    return __gs_dsp_vmulsum_v2_r1neg1();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2] in mpr row0 and vector2[v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*v2_1 + (v1_2*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1neg2 (void)
{
    return __gs_dsp_vmulsum_v2_r1neg2();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2] in mpr row0 and vector2[-v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*v2_1 + (-v1_2*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1neg12(void)
{
    return __gs_dsp_vmulsum_v2_r1neg12();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2] in mpr row0 and vector2[-v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*v2_1 + (-v1_2*v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0neg1_r1neg1  (void)
{
    return __gs_dsp_vmulsum_v2_r0neg1_r1neg1();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2] in mpr row0 and vector2[v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*v2_1 + (v1_2*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0neg1_r1neg2  (void)
{
    return __gs_dsp_vmulsum_v2_r0neg1_r1neg2();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2] in mpr row0 and vector2[-v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*v2_1 + (-v1_2*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0neg1_r1neg12 (void)
{
    return __gs_dsp_vmulsum_v2_r0neg1_r1neg12();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2] in mpr row0 and vector2[-v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*-v2_1 + (-v1_2*v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0neg2_r1neg1  (void)
{
    return __gs_dsp_vmulsum_v2_r0neg2_r1neg1();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2] in mpr row0 and vector2[v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*-v2_1 + (v1_2*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0neg2_r1neg2  (void)
{
    return __gs_dsp_vmulsum_v2_r0neg2_r1neg2();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2] in mpr row0 and vector2[-v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*-v2_1 + (-v1_2*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0neg2_r1neg12 (void)
{
    return __gs_dsp_vmulsum_v2_r0neg2_r1neg12();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2] in mpr row0 and vector2[-v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*-v2_1 + (-v1_2*v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0neg12_r1neg1 (void)
{
    return __gs_dsp_vmulsum_v2_r0neg12_r1neg1();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2] in mpr row0 and vector2[v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*-v2_1 + (v1_2*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0neg12_r1neg2 (void)
{
    return __gs_dsp_vmulsum_v2_r0neg12_r1neg2();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2] in mpr row0 and vector2[-v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*-v2_1 + (-v1_2*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0neg12_r1neg12(void)
{
    return __gs_dsp_vmulsum_v2_r0neg12_r1neg12();
}

/*
 * @brief					dot product of vector1[v1_2, v1_1] in mpr row0 and vector2[v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*v2_1 + (v1_1*v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap        (void)
{
    return __gs_dsp_vmulsum_v2_r0swap();
}

/*
 * @brief					dot product of vector1[-v1_2, v1_1] in mpr row0 and vector2[v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*v2_1 + (v1_1*v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r0neg1 (void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r0neg1();
}

/*
 * @brief					dot product of vector1[v1_2, -v1_1] in mpr row0 and vector2[v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*v2_1 + (-v1_1*v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r0neg2 (void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r0neg2();
}

/*
 * @brief					dot product of vector1[-v1_2, -v1_1] in mpr row0 and vector2[v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*v2_1 + (-v1_1*v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r0neg12(void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r0neg12();
}

/*
 * @brief					dot product of vector1[v1_2, v1_1] in mpr row0 and vector2[-v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*-v2_1 + (v1_1*v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r1neg1 (void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r1neg1();
}

/*
 * @brief					dot product of vector1[v1_2, v1_1] in mpr row0 and vector2[v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*v2_1 + (v1_1*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r1neg2 (void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r1neg2();
}

/*
 * @brief					dot product of vector1[v1_2, v1_1] in mpr row0 and vector2[-v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*-v2_1 + (v1_1*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r1neg12(void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r1neg12();
}

/*
 * @brief					dot product of vector1[-v1_2, v1_1] in mpr row0 and vector2[-v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*-v2_1 + (v1_1*v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r0neg1_r1neg1  (void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r0neg1_r1neg1();
}

/*
 * @brief					dot product of vector1[-v1_2, v1_1] in mpr row0 and vector2[v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*v2_1 + (v1_1*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r0neg1_r1neg2  (void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r0neg1_r1neg2();
}

/*
 * @brief					dot product of vector1[-v1_2, v1_1] in mpr row0 and vector2[-v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*-v2_1 + (v1_1*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r0neg1_r1neg12 (void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r0neg1_r1neg12();
}

/*
 * @brief					dot product of vector1[v1_2, -v1_1] in mpr row0 and vector2[-v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*-v2_1 + (-v1_1*v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r0neg2_r1neg1  (void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r0neg2_r1neg1();
}

/*
 * @brief					dot product of vector1[v1_2, -v1_1] in mpr row0 and vector2[v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*v2_1 + (-v1_1*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r0neg2_r1neg2  (void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r0neg2_r1neg2();
}

/*
 * @brief					dot product of vector1[v1_2, -v1_1] in mpr row0 and vector2[-v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*-v2_1 + (-v1_1*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r0neg2_r1neg12 (void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r0neg2_r1neg12();
}

/*
 * @brief					dot product of vector1[-v1_2, -v1_1] in mpr row0 and vector2[-v2_1, v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*-v2_1 + (-v1_1*v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r0neg12_r1neg1 (void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r0neg12_r1neg1();
}

/*
 * @brief					dot product of vector1[-v1_2, -v1_1] in mpr row0 and vector2[v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*v2_1 + (-v1_1*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r0neg12_r1neg2 (void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r0neg12_r1neg2();
}

/*
 * @brief					dot product of vector1[-v1_2, -v1_1] in mpr row0 and vector2[-v2_1, -v2_2]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*-v2_1 + (-v1_1*-v2_2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r0swap_r0neg12_r1neg12(void)
{
    return __gs_dsp_vmulsum_v2_r0swap_r0neg12_r1neg12();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2] in mpr row0 and vector2[v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*v2_2 + (v1_2*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap        (void)
{
    return __gs_dsp_vmulsum_v2_r1swap();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2] in mpr row0 and vector2[v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*v2_2 + (v1_2*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r0neg1 (void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r0neg1();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2] in mpr row0 and vector2[v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*v2_2 + (-v1_2*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r0neg2 (void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r0neg2();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2] in mpr row0 and vector2[v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*v2_2 + (-v1_2*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r0neg12(void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r0neg12();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2] in mpr row0 and vector2[-v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*-v2_2 + (v1_2*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r1neg1 (void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r1neg1();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2] in mpr row0 and vector2[v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*v2_2 + (v1_2*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r1neg2 (void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r1neg2();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2] in mpr row0 and vector2[-v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*-v2_2 + (v1_2*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r1neg12(void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r1neg12();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2] in mpr row0 and vector2[-v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*-v2_2 + (v1_2*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r0neg1_r1neg1  (void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r0neg1_r1neg1();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2] in mpr row0 and vector2[v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*v2_2 + (v1_2*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r0neg1_r1neg2  (void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r0neg1_r1neg2();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2] in mpr row0 and vector2[-v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*-v2_2 + (v1_2*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r0neg1_r1neg12 (void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r0neg1_r1neg12();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2] in mpr row0 and vector2[-v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*-v2_2 + (-v1_2*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r0neg2_r1neg1  (void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r0neg2_r1neg1();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2] in mpr row0 and vector2[v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*v2_2 + (-v1_2*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r0neg2_r1neg2  (void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r0neg2_r1neg2();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2] in mpr row0 and vector2[-v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_1*-v2_2 + (-v1_2*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r0neg2_r1neg12 (void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r0neg2_r1neg12();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2] in mpr row0 and vector2[-v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*-v2_2 + (-v1_2*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r0neg12_r1neg1 (void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r0neg12_r1neg1();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2] in mpr row0 and vector2[v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*v2_2 + (-v1_2*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r0neg12_r1neg2 (void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r0neg12_r1neg2();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2] in mpr row0 and vector2[-v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_1*-v2_2 + (-v1_2*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r1swap_r0neg12_r1neg12(void)
{
    return __gs_dsp_vmulsum_v2_r1swap_r0neg12_r1neg12();
}

/*
 * @brief					dot product of vector1[v1_2, v1_1] in mpr row0 and vector2[v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*v2_2 + (v1_1*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap(void)
{
    return __gs_dsp_vmulsum_v2_r01swap();
}

/*
 * @brief					dot product of vector1[-v1_2, v1_1] in mpr row0 and vector2[v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*v2_2 + (v1_1*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r0neg1(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r0neg1();
}

/*
 * @brief					dot product of vector1[v1_2, -v1_1] in mpr row0 and vector2[v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*v2_2 + (-v1_1*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r0neg2(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r0neg2();
}

/*
 * @brief					dot product of vector1[-v1_2, -v1_1] in mpr row0 and vector2[v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*v2_2 + (-v1_1*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r0neg12(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r0neg12();
}

/*
 * @brief					dot product of vector1[v1_2, v1_1] in mpr row0 and vector2[-v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*-v2_2 + (v1_1*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r1neg1(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r1neg1();
}

/*
 * @brief					dot product of vector1[v1_2, v1_1] in mpr row0 and vector2[v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*v2_2 + (v1_1*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r1neg2(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r1neg2();
}

/*
 * @brief					dot product of vector1[v1_2, v1_1] in mpr row0 and vector2[-v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*-v2_2 + (v1_1*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r1neg12(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r1neg12();
}

/*
 * @brief					dot product of vector1[-v1_2, v1_1] in mpr row0 and vector2[-v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*-v2_2 + (v1_1*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r0neg1_r1neg1(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r0neg1_r1neg1();
}

/*
 * @brief					dot product of vector1[-v1_2, v1_1] in mpr row0 and vector2[v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*v2_2 + (v1_1*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r0neg1_r1neg2(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r0neg1_r1neg2();
}

/*
 * @brief					dot product of vector1[-v1_2, v1_1] in mpr row0 and vector2[-v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*-v2_2 + (v1_1*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r0neg1_r1neg12(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r0neg1_r1neg12();
}

/*
 * @brief					dot product of vector1[v1_2, -v1_1] in mpr row0 and vector2[-v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*-v2_2 + (-v1_1*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r0neg2_r1neg1(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r0neg2_r1neg1();
}

/*
 * @brief					dot product of vector1[v1_2, -v1_1] in mpr row0 and vector2[v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*v2_2 + (-v1_1*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r0neg2_r1neg2(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r0neg2_r1neg2();
}

/*
 * @brief					dot product of vector1[v1_2, -v1_1] in mpr row0 and vector2[-v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					v1_2*-v2_2 + (-v1_1*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r0neg2_r1neg12(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r0neg2_r1neg12();
}

/*
 * @brief					dot product of vector1[-v1_2, -v1_1] in mpr row0 and vector2[-v2_2, v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*-v2_2 + (-v1_1*v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r0neg12_r1neg1(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r0neg12_r1neg1();
}

/*
 * @brief					dot product of vector1[-v1_2, -v1_1] in mpr row0 and vector2[v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*v2_2 + (-v1_1*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r0neg12_r1neg2(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r0neg12_r1neg2();
}

/*
 * @brief					dot product of vector1[-v1_2, -v1_1] in mpr row0 and vector2[-v2_2, -v2_1]
 * 							in mpr row1, value number in vector1 and vector2 is 2. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					-v1_2*-v2_2 + (-v1_1*-v2_1)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v2_r01swap_r0neg12_r1neg12(void)
{
    return __gs_dsp_vmulsum_v2_r01swap_r0neg12_r1neg12();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, v1_3] in mpr row0 and vector2[v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*v2_1) + (v1_2*v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3(void)
{
    return __gs_dsp_vmulsum_v3();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, v1_3] in mpr row0 and vector2[v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*v2_1) + (v1_2*v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg1(void)
{
    return __gs_dsp_vmulsum_v3_r0neg1();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, v1_3] in mpr row0 and vector2[v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*v2_1) + (-v1_2*v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg2(void)
{
    return __gs_dsp_vmulsum_v3_r0neg2();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, -v1_3] in mpr row0 and vector2[v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*v2_1) + (v1_2*v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg3(void)
{
    return __gs_dsp_vmulsum_v3_r0neg3();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, v1_3] in mpr row0 and vector2[v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*v2_1) + (-v1_2*v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg12 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg12 ();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, -v1_3] in mpr row0 and vector2[v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*v2_1) + (v1_2*v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg13 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg13 ();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, -v1_3] in mpr row0 and vector2[v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*v2_1) + (-v1_2*v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg23 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg23 ();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, -v1_3] in mpr row0 and vector2[v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*v2_1) + (-v1_2*v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg123(void)
{
    return __gs_dsp_vmulsum_v3_r0neg123();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, v1_3] in mpr row0 and vector2[-v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (v1_2*v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r1neg1  (void)
{
    return __gs_dsp_vmulsum_v3_r1neg1  ();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, v1_3] in mpr row0 and vector2[v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*v2_1) + (v1_2*-v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r1neg2  (void)
{
    return __gs_dsp_vmulsum_v3_r1neg2  ();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, v1_3] in mpr row0 and vector2[v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*v2_1) + (v1_2*v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r1neg3  (void)
{
    return __gs_dsp_vmulsum_v3_r1neg3  ();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, v1_3] in mpr row0 and vector2[-v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (v1_2*-v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r1neg12 (void)
{
    return __gs_dsp_vmulsum_v3_r1neg12 ();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, v1_3] in mpr row0 and vector2[-v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (v1_2*v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r1neg13 (void)
{
    return __gs_dsp_vmulsum_v3_r1neg13 ();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, v1_3] in mpr row0 and vector2[v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*v2_1) + (v1_2*-v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r1neg23 (void)
{
    return __gs_dsp_vmulsum_v3_r1neg23 ();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, v1_3] in mpr row0 and vector2[-v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (v1_2*-v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r1neg123(void)
{
    return __gs_dsp_vmulsum_v3_r1neg123();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, v1_3] in mpr row0 and vector2[-v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
* @return					(-v1_1*-v2_1) + (v1_2*v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg1_r1neg1  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg1_r1neg1  ();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, v1_3] in mpr row0 and vector2[v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
* @return					(-v1_1*v2_1) + (v1_2*-v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg1_r1neg2  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg1_r1neg2  ();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, v1_3] in mpr row0 and vector2[v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
* @return					(-v1_1*v2_1) + (v1_2*v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg1_r1neg3  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg1_r1neg3  ();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, v1_3] in mpr row0 and vector2[-v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
* @return					(-v1_1*-v2_1) + (v1_2*-v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg1_r1neg12 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg1_r1neg12 ();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, v1_3] in mpr row0 and vector2[-v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
* @return					(-v1_1*-v2_1) + (v1_2*v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg1_r1neg13 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg1_r1neg13 ();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, v1_3] in mpr row0 and vector2[v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
* @return					(-v1_1*v2_1) + (v1_2*-v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg1_r1neg23 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg1_r1neg23 ();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, v1_3] in mpr row0 and vector2[-v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
* @return					(-v1_1*-v2_1) + (v1_2*-v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg1_r1neg123(void)
{
    return __gs_dsp_vmulsum_v3_r0neg1_r1neg123();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, v1_3] in mpr row0 and vector2[-v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
* @return					(v1_1*-v2_1) + (-v1_2*v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg2_r1neg1  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg2_r1neg1  ();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, v1_3] in mpr row0 and vector2[v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
* @return					(v1_1*v2_1) + (-v1_2*-v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg2_r1neg2  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg2_r1neg2  ();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, v1_3] in mpr row0 and vector2[v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
* @return					(v1_1*v2_1) + (-v1_2*v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg2_r1neg3  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg2_r1neg3  ();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, v1_3] in mpr row0 and vector2[-v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (-v1_2*-v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg2_r1neg12 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg2_r1neg12 ();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, v1_3] in mpr row0 and vector2[-v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (-v1_2*v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg2_r1neg13 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg2_r1neg13 ();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, v1_3] in mpr row0 and vector2[v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*v2_1) + (-v1_2*-v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg2_r1neg23 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg2_r1neg23 ();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, v1_3] in mpr row0 and vector2[-v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (-v1_2*-v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg2_r1neg123(void)
{
    return __gs_dsp_vmulsum_v3_r0neg2_r1neg123();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, -v1_3] in mpr row0 and vector2[-v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (v1_2*v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg3_r1neg1  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg3_r1neg1  ();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, -v1_3] in mpr row0 and vector2[v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*v2_1) + (v1_2*-v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg3_r1neg2  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg3_r1neg2  ();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, -v1_3] in mpr row0 and vector2[v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*v2_1) + (v1_2*v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg3_r1neg3  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg3_r1neg3  ();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, -v1_3] in mpr row0 and vector2[-v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (v1_2*-v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg3_r1neg12 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg3_r1neg12 ();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, -v1_3] in mpr row0 and vector2[-v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (v1_2*v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg3_r1neg13 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg3_r1neg13 ();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, -v1_3] in mpr row0 and vector2[v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*v2_1) + (v1_2*-v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg3_r1neg23 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg3_r1neg23 ();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2, -v1_3] in mpr row0 and vector2[-v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (v1_2*-v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg3_r1neg123(void)
{
    return __gs_dsp_vmulsum_v3_r0neg3_r1neg123();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, v1_3] in mpr row0 and vector2[-v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*-v2_1) + (-v1_2*v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg12_r1neg1  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg12_r1neg1  ();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, v1_3] in mpr row0 and vector2[v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*v2_1) + (-v1_2*-v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg12_r1neg2  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg12_r1neg2  ();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, v1_3] in mpr row0 and vector2[v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*v2_1) + (-v1_2*v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg12_r1neg3  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg12_r1neg3  ();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, v1_3] in mpr row0 and vector2[-v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*-v2_1) + (-v1_2*-v2_2) + (v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg12_r1neg12 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg12_r1neg12 ();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, v1_3] in mpr row0 and vector2[-v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*-v2_1) + (-v1_2*v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg12_r1neg13 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg12_r1neg13 ();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, v1_3] in mpr row0 and vector2[v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*v2_1) + (-v1_2*-v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg12_r1neg23 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg12_r1neg23 ();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, v1_3] in mpr row0 and vector2[-v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*-v2_1) + (-v1_2*-v2_2) + (v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg12_r1neg123(void)
{
    return __gs_dsp_vmulsum_v3_r0neg12_r1neg123();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, -v1_3] in mpr row0 and vector2[-v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*-v2_1) + (v1_2*v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg13_r1neg1  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg13_r1neg1  ();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, -v1_3] in mpr row0 and vector2[v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*v2_1) + (v1_2*-v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg13_r1neg2  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg13_r1neg2  ();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, -v1_3] in mpr row0 and vector2[v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*v2_1) + (v1_2*v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg13_r1neg3  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg13_r1neg3  ();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, -v1_3] in mpr row0 and vector2[-v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*-v2_1) + (v1_2*-v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg13_r1neg12 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg13_r1neg12 ();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, -v1_3] in mpr row0 and vector2[-v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*-v2_1) + (v1_2*v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg13_r1neg13 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg13_r1neg13 ();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, -v1_3] in mpr row0 and vector2[v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*v2_1) + (v1_2*-v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg13_r1neg23 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg13_r1neg23 ();
}

/*
 * @brief					dot product of vector1[-v1_1, v1_2, -v1_3] in mpr row0 and vector2[-v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*-v2_1) + (v1_2*-v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg13_r1neg123(void)
{
    return __gs_dsp_vmulsum_v3_r0neg13_r1neg123();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, -v1_3] in mpr row0 and vector2[-v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (-v1_2*v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg23_r1neg1  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg23_r1neg1  ();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, -v1_3] in mpr row0 and vector2[v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*v2_1) + (-v1_2*-v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg23_r1neg2  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg23_r1neg2  ();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, -v1_3] in mpr row0 and vector2[v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*v2_1) + (-v1_2*v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg23_r1neg3  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg23_r1neg3  ();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, -v1_3] in mpr row0 and vector2[-v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (-v1_2*-v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg23_r1neg12 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg23_r1neg12 ();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, -v1_3] in mpr row0 and vector2[-v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (-v1_2*v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg23_r1neg13 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg23_r1neg13 ();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, -v1_3] in mpr row0 and vector2[v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*v2_1) + (-v1_2*-v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg23_r1neg23 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg23_r1neg23 ();
}

/*
 * @brief					dot product of vector1[v1_1, -v1_2, -v1_3] in mpr row0 and vector2[-v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(v1_1*-v2_1) + (-v1_2*-v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg23_r1neg123(void)
{
    return __gs_dsp_vmulsum_v3_r0neg23_r1neg123();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, -v1_3] in mpr row0 and vector2[-v2_1, v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*-v2_1) + (-v1_2*v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg123_r1neg1  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg123_r1neg1  ();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, -v1_3] in mpr row0 and vector2[v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*v2_1) + (-v1_2*-v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg123_r1neg2  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg123_r1neg2  ();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, -v1_3] in mpr row0 and vector2[v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*v2_1) + (-v1_2*v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg123_r1neg3  (void)
{
    return __gs_dsp_vmulsum_v3_r0neg123_r1neg3  ();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, -v1_3] in mpr row0 and vector2[-v2_1, -v2_2, v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*-v2_1) + (-v1_2*-v2_2) + (-v1_3*v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg123_r1neg12 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg123_r1neg12 ();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, -v1_3] in mpr row0 and vector2[-v2_1, v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*-v2_1) + (-v1_2*v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg123_r1neg13 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg123_r1neg13 ();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, -v1_3] in mpr row0 and vector2[v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*v2_1) + (-v1_2*-v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg123_r1neg23 (void)
{
    return __gs_dsp_vmulsum_v3_r0neg123_r1neg23 ();
}

/*
 * @brief					dot product of vector1[-v1_1, -v1_2, -v1_3] in mpr row0 and vector2[-v2_1, -v2_2, -v2_3]
 * 							in mpr row1, value number in vector1 and vector2 is 3. Before this
 * 							instruction, vector1&vector2 must move into mpr register.
 *
 * @return					(-v1_1*-v2_1) + (-v1_2*-v2_2) + (-v1_3*-v2_3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum_v3_r0neg123_r1neg123(void)
{
    return __gs_dsp_vmulsum_v3_r0neg123_r1neg123();
}

/*
 * @brief					dot product of vector1[v1_1, v1_2] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0(-1/2, sqrt3/2)
 * 							1(-1/2, -sqrt3/2)
 *
 * @return					v1_1*-1/2 + v1_2*sqrt3/2
 * 							v1_1*-1/2 + v1_2*-sqrt3/2
 */
#define __vmulsum_v2_const(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v2_const(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[-v1_1, v1_2] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0(-1/2, sqrt3/2)
 * 							1(-1/2, -sqrt3/2)
 *
 * @return					-v1_1*-1/2 + v1_2*sqrt3/2
 * 							-v1_1*-1/2 + v1_2*-sqrt3/2
 */
#define __vmulsum_v2_const_r0neg1(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v2_const_r0neg1(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[v1_1, -v1_2] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0(-1/2, sqrt3/2)
 * 							1(-1/2, -sqrt3/2)
 *
 * @return					v1_1*-1/2 + (-v1_2*sqrt3/2)
 * 							v1_1*-1/2 + (-v1_2*-sqrt3/2)
 */
#define __vmulsum_v2_const_r0neg2(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v2_const_r0neg2(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[-v1_1, -v1_2] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0(-1/2, sqrt3/2)
 * 							1(-1/2, -sqrt3/2)
 *
 * @return					-v1_1*-1/2 + (-v1_2*sqrt3/2)
 * 							-v1_1*-1/2 + (-v1_2*-sqrt3/2)
 */
#define __vmulsum_v2_const_r0neg12(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v2_const_r0neg12(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[v1_2, v1_1] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0(-1/2, sqrt3/2)
 * 							1(-1/2, -sqrt3/2)
 *
 * @return					v1_2*-1/2 + (v1_1*sqrt3/2)
 * 							v1_2*-1/2 + (v1_1*-sqrt3/2)
 */
#define __vmulsum_v2_const_r0swap(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v2_const_r0swap(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[-v1_2, v1_1] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0(-1/2, sqrt3/2)
 * 							1(-1/2, -sqrt3/2)
 *
 * @return					-v1_2*-1/2 + (v1_1*sqrt3/2)
 * 							-v1_2*-1/2 + (v1_1*-sqrt3/2)
 */
#define __vmulsum_v2_const_r0swap_r0neg1(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v2_const_r0swap_r0neg1(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[v1_2, -v1_1] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0(-1/2, sqrt3/2)
 * 							1(-1/2, -sqrt3/2)
 *
 * @return					v1_2*-1/2 + (-v1_1*sqrt3/2)
 * 							v1_2*-1/2 + (-v1_1*-sqrt3/2)
 */
#define __vmulsum_v2_const_r0swap_r0neg2(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v2_const_r0swap_r0neg2(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[-v1_2, -v1_1] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0(-1/2, sqrt3/2)
 * 							1(-1/2, -sqrt3/2)
 *
 * @return					-v1_2*-1/2 + (-v1_1*sqrt3/2)
 * 							-v1_2*-1/2 + (-v1_1*-sqrt3/2)
 */
#define __vmulsum_v2_const_r0swap_r0neg12(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v2_const_r0swap_r0neg12(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[v1, v2, v3] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0{2/3, -1/3, -1/3}
 * 							1{0, sqrt3/3, -sqrt3/3}
 *
 * @return					v1*2/3 + v2*-1/3 + v3*-1/3
 * 							v1*0 + v2*sqrt3/3 + v3*-sqrt3/3
 */
#define __vmulsum_v3_const(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v3_const(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[-v1, v2, v3] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0{2/3, -1/3, -1/3}
 * 							1{0, sqrt3/3, -sqrt3/3}
 *
 * @return					-v1*2/3 + v2*-1/3 + v3*-1/3
 * 							-v1*0 + v2*sqrt3/3 + v3*-sqrt3/3
 */
#define __vmulsum_v3_const_r0neg1(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v3_const_r0neg1(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[v1, -v2, v3] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0{2/3, -1/3, -1/3}
 * 							1{0, sqrt3/3, -sqrt3/3}
 *
 * @return					v1*2/3 + -v2*-1/3 + v3*-1/3
 * 							v1*0 + -v2*sqrt3/3 + v3*-sqrt3/3
 */
#define __vmulsum_v3_const_r0neg2(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v3_const_r0neg2(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[v1, v2, -v3] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0{2/3, -1/3, -1/3}
 * 							1{0, sqrt3/3, -sqrt3/3}
 *
 * @return					v1*2/3 + (v2*-1/3) + (-v3*-1/3)
 * 							v1*0 + (v2*sqrt3/3) + (-v3*-sqrt3/3)
 */
#define __vmulsum_v3_const_r0neg3(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v3_const_r0neg3(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[-v1, -v2, v3] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0{2/3, -1/3, -1/3}
 * 							1{0, sqrt3/3, -sqrt3/3}
 *
 * @return					-v1*2/3 + (-v2*-1/3) + (v3*-1/3)
 * 							-v1*0 + (-v2*sqrt3/3) + (v3*-sqrt3/3)
 */
#define __vmulsum_v3_const_r0neg12(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v3_const_r0neg12(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[-v1, v2, -v3] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0{2/3, -1/3, -1/3}
 * 							1{0, sqrt3/3, -sqrt3/3}
 *
 * @return					-v1*2/3 + (v2*-1/3) + (-v3*-1/3)
 * 							-v1*0 + (v2*sqrt3/3) + (-v3*-sqrt3/3)
 */
#define __vmulsum_v3_const_r0neg13(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v3_const_r0neg13(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[v1, -v2, -v3] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0{2/3, -1/3, -1/3}
 * 							1{0, sqrt3/3, -sqrt3/3}
 *
 * @return					v1*2/3 + (-v2*-1/3) + (-v3*-1/3)
 * 							v1*0 + (-v2*sqrt3/3) + (-v3*-sqrt3/3)
 */
#define __vmulsum_v3_const_r0neg23(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v3_const_r0neg23(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product of vector1[-v1, -v2, -v3] in mpr row0 and const vector2
 * 							determined by imm2u_const
 * @param[in] imm2u_const
 * 							0{2/3, -1/3, -1/3}
 * 							1{0, sqrt3/3, -sqrt3/3}
 *
 * @return					-v1*2/3 + (-v2*-1/3) + (-v3*-1/3)
 * 							-v1*0 + (-v2*sqrt3/3) + (-v3*-sqrt3/3)
 */
#define __vmulsum_v3_const_r0neg123(imm2u_mode)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_vmulsum_v3_const_r0neg123(imm2u_mode);	\
    __res;                      \
  })

/*
 * @brief					dot product vector[src1, src2] with vector1[v1, v2] in mpr row0.
 * 							Before this instruction, vector1 must be moved into mpr row0 register.
 * @param[in] src1
 * @param[in] src2
 *
 * @return					v1*src1 + (v2*src2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum2_f2_r1(float src1, float src2)
{
    return __gs_dsp_vmulsum2_f2_r1(src1, src2);
}

/*
 * @brief					dot product vector[-src1, src2] with vector1[v1, v2] in mpr row0.
 * 							Before this instruction, vector1 must be moved into mpr row0 register.
 * @param[in] src1
 * @param[in] src2
 *
 * @return					v1*-src1 + (v2*src2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum2_f2_r1_neg1(float src1, float src2)
{
    return __gs_dsp_vmulsum2_f2_r1_neg1 (src1, src2);
}

/*
 * @brief					dot product vector[src1, -src2] with vector1[v1, v2] in mpr row0.
 * 							Before this instruction, vector1 must be moved into mpr row0 register.
 * @param[in] src1
 * @param[in] src2
 *
 * @return					v1*src1 + (v2*-src2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum2_f2_r1_neg2(float src1, float src2)
{
    return __gs_dsp_vmulsum2_f2_r1_neg2 (src1, src2);
}

/*
 * @brief					dot product vector[-src1, -src2] with vector1[v1, v2] in mpr row0.
 * 							Before this instruction, vector1 must be moved into mpr row0 register.
 * @param[in] src1
 * @param[in] src2
 *
 * @return					v1*-src1 + (v2*-src2)
 */
__EXTERN_INLINE_MANUAL float __vmulsum2_f2_r1_neg12(float src1, float src2)
{
    return __gs_dsp_vmulsum2_f2_r1_neg12(src1, src2);
}

/*
 * @brief					float dot product vector[src1, src2, src3] with vector1[v1, v2, v3] in mpr row0.
 * 							Before this instruction, vector1 must be moved into mpr row0 register.
 * @param[in] src1
 * @param[in] src2
 * @param[in] src2
 *
 * @return					v1*src1 + (v2*src2) + (v3*src3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum3_f3_r1(float src1, float src2, float dst)
{
    return __gs_dsp_vmulsum3_f3_r1(src1, src2, dst);
}

/*
 * @brief					float dot product vector[-src1, src2, src3] with vector1[v1, v2, v3] in mpr row0.
 * 							Before this instruction, vector1 must be moved into mpr row0 register.
 * @param[in] src1
 * @param[in] src2
 * @param[in] src2
 *
 * @return					v1*-src1 + (v2*src2) + (v3*src3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum3_f3_r1_neg1(float src1, float src2, float dst)
{
    return __gs_dsp_vmulsum3_f3_r1_neg1(src1, src2, dst);
}

/*
 * @brief					float dot product vector[src1, -src2, src3] with vector1[v1, v2, v3] in mpr row0.
 * 							Before this instruction, vector1 must be moved into mpr row0 register.
 * @param[in] src1
 * @param[in] src2
 * @param[in] src2
 *
 * @return					v1*src1 + (v2*-src2) + (v3*src3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum3_f3_r1_neg2(float src1, float src2, float dst)
{
    return __gs_dsp_vmulsum3_f3_r1_neg2(src1, src2, dst);
}

/*
 * @brief					float dot product vector[src1, src2, -src3] with vector1[v1, v2, v3] in mpr row0.
 * 							Before this instruction, vector1 must be moved into mpr row0 register.
 * @param[in] src1
 * @param[in] src2
 * @param[in] src2
 *
 * @return					v1*src1 + (v2*src2) + (v3*-src3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum3_f3_r1_neg3(float src1, float src2, float dst)
{
    return __gs_dsp_vmulsum3_f3_r1_neg3(src1, src2, dst);
}

/*
 * @brief					float dot product vector[-src1, -src2, src3] with vector1[v1, v2, v3] in mpr row0.
 * 							Before this instruction, vector1 must be moved into mpr row0 register.
 * @param[in] src1
 * @param[in] src2
 * @param[in] src2
 *
 * @return					v1*-src1 + (v2*-src2) + (v3*src3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum3_f3_r1_neg12 (float src1, float src2, float dst)
{
    return __gs_dsp_vmulsum3_f3_r1_neg12 (src1, src2, dst);
}

/*
 * @brief					float dot product vector[-src1, src2, -src3] with vector1[v1, v2, v3] in mpr row0.
 * 							Before this instruction, vector1 must be moved into mpr row0 register.
 * @param[in] src1
 * @param[in] src2
 * @param[in] src2
 *
 * @return					v1*-src1 + (v2*src2) + (v3*-src3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum3_f3_r1_neg13 (float src1, float src2, float dst)
{
    return __gs_dsp_vmulsum3_f3_r1_neg13 (src1, src2, dst);
}

/*
 * @brief					float dot product vector[src1, -src2, -src3] with vector1[v1, v2, v3] in mpr row0.
 * 							Before this instruction, vector1 must be moved into mpr row0 register.
 * @param[in] src1
 * @param[in] src2
 * @param[in] src2
 *
 * @return					v1*src1 + (v2*-src2) + (v3*-src3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum3_f3_r1_neg23 (float src1, float src2, float dst)
{
    return __gs_dsp_vmulsum3_f3_r1_neg23 (src1, src2, dst);
}

/*
 * @brief					float dot product vector[-src1, -src2, -src3] with vector1[v1, v2, v3] in mpr row0.
 * 							Before this instruction, vector1 must be moved into mpr row0 register.
 * @param[in] src1
 * @param[in] src2
 * @param[in] src2
 *
 * @return					v1*-src1 + (v2*-src2) + (v3*-src3)
 */
__EXTERN_INLINE_MANUAL float __vmulsum3_f3_r1_neg123(float src1, float src2, float dst)
{
    return __gs_dsp_vmulsum3_f3_r1_neg123(src1, src2, dst);
}

/*
 * @brief			Multiple nop operation
 * @param[in]       imm8u_cnt, immediate number
 *
 * @return       	None
 */
#define __nop_cnt(imm8u_cnt)    \
  __extension__                 \
  ({                            \
    __gs_nop_cnt (imm8u_cnt);	\
  })

/*
 * @brief			Software breakpoint
 *
 * @return       	None
 */
__EXTERN_INLINE_MANUAL void __breakpoint(void)
{
    __gs_breakpoint();
}

/*
 * @brief			Software trap
 *
 * @return       	None
 */
__EXTERN_INLINE_MANUAL void __trap(void)
{
    __gs_trap();
}

/*
 * @brief			32-bit unsigned int byte reverse
 * @param[in]       src
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __bswap32(uint32 src)
{
    return __gs_bswap32(src);
}

/*
 * @brief			16-bit unsigned short byte reverse
 * @param[in]       src
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint16 __bswap16(uint16 src)
{
    return __gs_bswap16(src);
}

/*
 * @brief			32-bit unsigned int bit reverse
 * @param[in]       32-bit unsigned int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __rbit(uint32 src)
{
    return __gs_rbit(src);
}

/*
 * @brief			32-bit unsigned int bit reverse
 * @param[in]       32-bit unsigned int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __flip32(uint32 src)
{
    return __gs_flip32(src);
}

/*
 * @brief		counter number of the leading zero bits
 * @param[in]	32-bit unsigned int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int32 __norm(uint32 src)
{
    return __gs_norm(src);
}


/*
 * @brief		counter number of the leading zero bits
 * @param[in]	32-bit unsigned int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int32 __clz(uint32 src)
{
    return __gs_clz(src);
}

/*
 * @brief		counter number of the trailing zero bits
 * @param[in]	32-bit unsigned int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int32 __ctz(uint32 src)
{
    return __gs_ctz(src);
}

/*
 * @brief		counter number of the redundant sign bits
 * @param[in]	32-bit signed int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int32 __clrsb(int32 src)
{
    return __gs_clrsb(src);
}

/*
 * @brief				rotate shift right
 * @param[in] src1		32-bit unsigned int
 * @param[in] amount	bit shift number, 1 <= amount <= 31
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __ror(uint32 src1, uint32 amount)
{
    return __gs_ror(src1, amount);
}

/*
 * @brief				rotate shift left
 * @param[in] src1		32-bit unsigned int
 * @param[in] amount	bit shift number, 1 <= amount <= 31
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __rol(uint32 src1, uint32 amount)
{
    return __gs_rol(src1, amount);
}

/*
 * @brief			float absolute value
 * @param[in] src	float
 *
 * @return
 */
__EXTERN_INLINE_MANUAL float __fabsf(float src)
{
    return __gs_fabs_s(src);
}
__EXTERN_INLINE_MANUAL float __fabs(float src)
{
    return __gs_fabs_s(src);
}

/*
 * @brief				float multiply add, float op1 multiply
 * 						with op2, add to addend
 * @param[in] op1
 * @param[in] op2
 * @param[in] addend
 *
 * @return
 */
__EXTERN_INLINE_MANUAL float __fmaf(float op1, float op2, float addend)
{
    return __gs_fma_s(op1, op2, addend);
}

/*
 * @brief					float multiply with a constant
 * @param[in] src			float
 * @param[in] imm4u_const
 * 				0- 2pi
 * 				1- 1/2pi
 * 				2- sqrt3/3
 * 				3- sqrt3/2
 * 				4- sqrt3
 * 				5- sqrt3/6
 * 				6- 1/2
 * 				7- 1/3
 * 				8- 2/3
 * 				9- 2/sqrt3
 * 				10- 1/4
 * 				11- 2
 *
 * @return
 */
#define __fmul_const(src, imm4u_const)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_fmul_const (src, imm4u_const);	\
    __res;                      \
  })

/*
 * @brief					float multiply with a negative constant
 * @param[in] src			float
 * @param[in] imm4u_const
 * 				0- -2pi
 * 				1- -1/2pi
 * 				2- -sqrt3/3
 * 				3- -sqrt3/2
 * 				4- -sqrt3
 * 				5- -sqrt3/6
 * 				6- -1/2
 * 				7- -1/3
 * 				8- -2/3
 * 				9- -2/sqrt3
 * 				10- -1/4
 * 				11- -2
 *
 * @return
 */
#define __fmul_const_neg(src, imm4u_const)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_fmul_const_neg (src, imm4u_const);	\
    __res;                      \
  })

/*
 * @brief					float add with a constant
 * @param[in] src			float
 * @param[in] imm4u_const
 * 				0- sqrt3
 * 				1- pi/3
 * 				2- 1
 *
 * @return
 */
#define __fadd_const(src, imm4u_const)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_fadd_const (src, imm4u_const);	\
    __res;                      \
  })

/*
 * @brief					float subtract a constant
 * @param[in] src			float
 * @param[in] imm4u_const
 * 				0- sqrt3
 * 				1- pi/3
 * 				2- 1
 *
 * @return
 */
#define __fsub_const(src, imm4u_const)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_fsub_const (src, imm4u_const);	\
    __res;                      \
  })

/*
 * @brief					negative float, then add with a constant
 * @param[in] src			float
 * @param[in] imm4u_const
 * 				0- sqrt3
 * 				1- pi/3
 * 				2- 1
 *
 * @return
 */
#define __neg1_fadd_const(src, imm4u_const)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_neg1_fadd_const (src, imm4u_const);	\
    __res;                      \
  })

/*
 * @brief					negative float, then subtract a constant
 * @param[in] src			float
 * @param[in] imm4u_const
 * 				0- sqrt3
 * 				1- pi/3
 * 				2- 1
 *
 * @return
 */
#define __neg1_fsub_const(src, imm4u_const)    \
  __extension__                 \
  ({                            \
    float __res = __gs_dsp_neg1_fsub_const (src, imm4u_const);	\
    __res;                      \
  })

/*
 * @brief				float add one
 * @param[in] src
 *
 * @return          	src + 1
 */
__EXTERN_INLINE_MANUAL float __fadd_one(float src)
{
    return __gs_dsp_fadd_one(src);
}

/*
 * @brief				float subtract one
 * @param[in] src
 *
 * @return          	src - 1
 */
__EXTERN_INLINE_MANUAL float __fsub_one(float src)
{
    return __gs_dsp_fsub_one(src);
}

/*
 * @brief				float subtract constant sqrt3
 * @param[in] src
 *
 * @return          	src - sqrt3
 */
__EXTERN_INLINE_MANUAL float __fsub_3sqrt(float src)
{
    return __gs_dsp_fsub_3sqrt(src);
}

/*
 * @brief				negative float, add constant sqrt3
 * @param[in] src
 *
 * @return          	-src + sqrt3
 */
__EXTERN_INLINE_MANUAL float __neg1_fadd_3sqrt(float src)
{
    return __gs_dsp_neg1_fadd_3sqrt(src);
}

/*
 * @brief				float multiply with constant 2pi
 * @param[in] src
 *
 * @return				src*2pi
 */
__EXTERN_INLINE_MANUAL float __mpy2pif32(float src)
{
    return __gs_dsp_fmul2pi(src);
}

/*
 * @brief				float divided with constant 2pi
 * @param[in] src
 *
 * @return       		src/2pi
 */
__EXTERN_INLINE_MANUAL float __div2pif32(float src)
{
    return __gs_dsp_fdiv2pi(src);
}

/*
 * @brief				float multiply with constant -2pi
 * @param[in] src
 *
 * @return          	src * (-2PI)
 */
__EXTERN_INLINE_MANUAL float __fmul2pi_neg(float src)
{
    return __gs_dsp_fmul2pi_neg(src);
}

/*
 * @brief				float divided with constant -2pi
 * @param[in] src
 *
 * @return          	src / (-2PI)
 */
__EXTERN_INLINE_MANUAL float __fdiv2pi_neg(float src)
{
    return __gs_dsp_fdiv2pi_neg(src);
}

/*
 * @brief				float multiply with constant 2
 * @param[in] src
 *
 * @return          	src * 2
 */
__EXTERN_INLINE_MANUAL float __fmul2(float src)
{
    return __gs_dsp_fmul_2(src);
}

/*
 * @brief		   		float divided constant 2
 * @param[in] src
 *
 * @return          	src/2
 */
__EXTERN_INLINE_MANUAL float __fdiv2(float src)
{
    return __gs_dsp_fdiv_2(src);
}

/*
 * @brief		   		float divided constant 3
 * @param[in] src
 *
 * @return          	src/3
 */
__EXTERN_INLINE_MANUAL float __fdiv3(float src)
{
    return __gs_dsp_fdiv_3(src);
}

/*
 * @brief		   		float divided constant 4
 * @param[in] srcc
 *
 * @return          	src/4
 */
__EXTERN_INLINE_MANUAL float __fdiv4(float src)
{
    return __gs_dsp_fdiv_4(src);
}

/*
 * @brief				float multiply with constant -2
 * @param[in] src
 *
 * @return          	src * -2
 */
__EXTERN_INLINE_MANUAL float __fmul2_neg(float src)
{
    return __gs_dsp_fmul_2_neg(src);
}

/*
 * @brief		   		float divided constant -2
 * @param[in] src
 *
 * @return          	src/-2
 */
__EXTERN_INLINE_MANUAL float __fdiv2_neg(float src)
{
    return __gs_dsp_fdiv_2_neg(src);
}

/*
 * @brief		   		float divided constant -3
 * @param[in] src
 *
 * @return          	src/-3
 */
__EXTERN_INLINE_MANUAL float __fdiv3_neg(float src)
{
    return __gs_dsp_fdiv_3_neg(src);
}

/*
 * @brief		   		float divided constant -4
 * @param[in] src
 *
 * @return          	src/-4
 */
__EXTERN_INLINE_MANUAL float __fdiv4_neg(float src)
{
    return __gs_dsp_fdiv_4_neg(src);
}

/*
 * @brief		   		float multiply with constant sqrt3
 * @param[in] src
 *
 * @return          	src*sqrt(3)
 */
__EXTERN_INLINE_MANUAL float __fmul3sqrt(float src)
{
    return __gs_dsp_fmul3sqrt(src);
}

/*
 * @brief				float divided with constant sqrt3
 * @param[in] src
 *
 * @return       		src/sqrt3
 */
__EXTERN_INLINE_MANUAL float __fdiv3sqrt(float src)
{
    return __gs_dsp_fdiv3sqrt(src);
}

/*
 * @brief		   		float multiply with constant -sqrt3
 * @param[in] src
 *
 * @return          	src*-sqrt(3)
 */
__EXTERN_INLINE_MANUAL float __fmul3sqrt_neg(float src)
{
    return __gs_dsp_fmul3sqrt_neg(src);
}

/*
 * @brief				float divided with constant -sqrt3
 * @param[in] src
 *
 * @return       		src/-sqrt3
 */
__EXTERN_INLINE_MANUAL float __fdiv3sqrt_neg(float src)
{
    return __gs_dsp_fdiv3sqrt_neg(src);
}

/*
 * @brief				float multiply constant 2/3
 * @param[in] src
 *
 * @return          	src * (2/3)
 */
__EXTERN_INLINE_MANUAL float __fmul2_div3(float src)
{
    return __gs_dsp_fmul2_div3(src);
}

/*
 * @brief				float multiply constant -2/3
 * @param[in] src
 *
 * @return          	src * (-2/3)
 */
__EXTERN_INLINE_MANUAL float __fmul2_div3_neg(float src)
{
    return __gs_dsp_fmul2_div3_neg(src);
}

/*
 * @brief				float multiply constant sqrt3/2
 * @param[in] src
 *
 * @return          	src * sqrt3/2
 */
__EXTERN_INLINE_MANUAL float __fmul3sqrt_div2(float src)
{
    return __gs_dsp_fmul3sqrt_div2(src);
}

/*
 * @brief				float multiply constant sqrt3/6
 * @param[in] src
 *
 * @return          	src * sqrt3/6
 */
__EXTERN_INLINE_MANUAL float __fmul3sqrt_div6(float src)
{
    return __gs_dsp_fmul3sqrt_div6(src);
}

/*
 * @brief				float multiply constant 2/sqrt3
 * @param[in] src
 *
 * @return          	src * 2/sqrt3
 */
__EXTERN_INLINE_MANUAL float __fdiv3sqrt_mul2(float src)
{
    return __gs_dsp_fdiv3sqrt_mul2(src);
}

/*
 * @brief				float multiply constant -sqrt3/2
 * @param[in] src
 *
 * @return          	src * -sqrt3/2
 */
__EXTERN_INLINE_MANUAL float __fmul3sqrt_div2_neg(float src)
{
    return __gs_dsp_fmul3sqrt_div2_neg(src);
}

/*
 * @brief				float multiply constant -sqrt3/6
 * @param[in] src
 *
 * @return          	src * -sqrt3/6
 */
__EXTERN_INLINE_MANUAL float __fmul3sqrt_div6_neg(float src)
{
    return __gs_dsp_fmul3sqrt_div6_neg(src);
}

/*
 * @brief				float multiply constant -2/sqrt3
 * @param[in] src
 *
 * @return          	src * -2/sqrt3
 */
__EXTERN_INLINE_MANUAL float __fdiv3sqrt_mul2_neg(float src)
{
    return __gs_dsp_fdiv3sqrt_mul2_neg(src);
}




/*
 * @brief				negative float, then add constant 1
 *
 * @param[in] src
 *
 * @return          	-src + 1
 */
__EXTERN_INLINE_MANUAL float __neg1_fadd_one(float src)
{
    return __gs_dsp_neg1_fadd_one(src);
}

/*
 * @brief				negative float, then subtract constant 1
 *
 * @param[in] src
 *
 * @return          	-src - 1
 */
__EXTERN_INLINE_MANUAL float __neg1_fsub_one(float src)
{
    return __gs_dsp_neg1_fsub_one(src);
}

/*
 * @brief				float add constant sqrt3
 *
 * @param[in] src
 *
 * @return          	src + sqrt3
 */
__EXTERN_INLINE_MANUAL float __fadd_3sqrt(float src)
{
    return __gs_dsp_fadd_3sqrt(src);
}

/*
 * @brief				negative float, then subtract constant sqrt3
 *
 * @param[in] src
 *
 * @return          	-src - sqrt3
 */
__EXTERN_INLINE_MANUAL float __neg1_fsub_3sqrt(float src)
{
    return __gs_dsp_neg1_fsub_3sqrt(src);
}

/*
 * @brief				float add constant PI/3
 *
 * @param[in] src
 *
 * @return          	src + PI/3
 */
__EXTERN_INLINE_MANUAL float __fadd_pi3rd(float src)
{
    return __gs_dsp_fadd_pi3rd(src);
}

/*
 * @brief				float sub constant PI/3
 *
 * @param[in] src
 *
 * @return          	src - PI/3
 */
__EXTERN_INLINE_MANUAL float __fsub_pi3rd(float src)
{
    return __gs_dsp_fsub_pi3rd(src);
}

/*
 * @brief				negative float, then add constant PI/3
 *
 * @param[in] src
 *
 * @return          	-src + PI/3
 */
__EXTERN_INLINE_MANUAL float __neg1_fadd_pi3rd(float src)
{
    return __gs_dsp_neg1_fadd_pi3rd(src);
}

/*
 * @brief				negative float, then sub constant PI/3
 *
 * @param[in] src
 *
 * @return          	-src - PI/3
 */
__EXTERN_INLINE_MANUAL float __neg1_fsub_pi3rd(float src)
{
    return __gs_dsp_neg1_fsub_pi3rd(src);
}

/*
 * @brief				count number of bit value is 1
 * @param[in] src		32-bit unsigned int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int32 __popcount(uint32 src)
{
    return __gs_dsp_popcount(src);
}

/*
 * @brief				get minimal value of src1, src2
 * @param[in] src1		32-bit int
 * @param[in] src2		32-bit int
 *
 * @return				min(src1, src2)
 */
__EXTERN_INLINE_MANUAL int32 __smin(int32 src1, int32 src2)
{
    return __gs_dsp_smin(src1, src2);
}

/*
 * @brief				get maximal value of src1, src2
 * @param[in] src1		32-bit int
 * @param[in] src2		32-bit int
 *
 * @return				max(src1, src2)
 */
__EXTERN_INLINE_MANUAL int32 __smax(int32 src1, int32 src2)
{
    return __gs_dsp_smax(src1, src2);
}

/*
 * @brief				get minimal value of src1, src2
 * @param[in] src1		32-bit unsigned int
 * @param[in] src2		32-bit unsigned int
 *
 * @return				min(src1, src2)
 */
__EXTERN_INLINE_MANUAL uint32 __umin(uint32 src1, uint32 src2)
{
    return __gs_dsp_umin(src1, src2);
}

/*
 * @brief				get maximal value of src1, src2
 * @param[in] src1		32-bit unsigned int
 * @param[in] src2		32-bit unsigned int
 *
 * @return				max(src1, src2)
 */
__EXTERN_INLINE_MANUAL uint32 __umax(uint32 src1, uint32 src2)
{
    return __gs_dsp_umax(src1, src2);
}

/*
 * @brief				get high word of multiply product src1, src2
 * @param[in] src1		32-bit int
 * @param[in] src2		32-bit int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int32 __smul_high(int32 src1, int32 src2)
{
    return __gs_dsp_smul_high(src1, src2);
}

/*
 * @brief				get high word of multiply product src1, src2
 * @param[in] src1		32-bit unsigned int
 * @param[in] src2		32-bit unsigned int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __umul_high(uint32 src1, uint32 src2)
{
    return __gs_dsp_umul_high(src1, src2);
}

/*
 * @brief				get high word of multiply product src1, src2
 * @param[in] src1		32-bit unsigned int
 * @param[in] src2		32-bit int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int32 __usmul_high(uint32 src1, int32 src2)
{
    return __gs_dsp_usmul_high(src1, src2);
}

/*
 * @brief				multiply product src1, src2
 * @param[in] src1		32-bit unsigned int
 * @param[in] src2		32-bit int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int64 __usmul64w(uint32 src1, int32 src2)
{
    return __gs_dsp_usmul64w(src1, src2);
}

/*
 * @brief				bit and not of src1, src2
 * @param[in] src1		32-bit unsigned int
 * @param[in] src2		32-bit unsigned int
 *
 * @return				!(src1&src2)
 */
__EXTERN_INLINE_MANUAL uint32 __andn(uint32 src1, uint32 src2)
{
    return __gs_dsp_andn(src1, src2);
}

/*
 * @brief				bit or not of src1, src2
 * @param[in] src1		32-bit unsigned int
 * @param[in] src2		32-bit unsigned int
 *
 * @return				!(src1|src2)
 */
__EXTERN_INLINE_MANUAL uint32 __orn(uint32 src1, uint32 src2)
{
    return __gs_dsp_orn(src1, src2);
}

/*
 * @brief				bit not xor of src1, src2
 * @param[in] src1		32-bit unsigned int
 * @param[in] src2		32-bit unsigned int
 *
 * @return				((!src1)^(!src2))
 */
__EXTERN_INLINE_MANUAL uint32 __nxor(uint32 src1, uint32 src2)
{
    return __gs_dsp_nxor(src1, src2);
}

/*
 * @brief				float minimal value of src1, src2
 * @param[in] src1
 * @param[in] src2
 *
 * @return
 */
__EXTERN_INLINE_MANUAL float __fminf(float src1, float src2)
{
    return __gs_dsp_fmin_s(src1, src2);
}
__EXTERN_INLINE_MANUAL float __fmin(float src1, float src2)
{
    return __gs_dsp_fmin_s(src1, src2);
}
__EXTERN_INLINE_MANUAL float __mminf32(float src1, float src2)
{
    return __gs_dsp_fmin_s(src1, src2);
}

/*
 * @brief				float maximal value of src1, src2
 * @param[in] src1
 * @param[in] src2
 *
 * @return
 */
__EXTERN_INLINE_MANUAL float __fmaxf(float src1, float src2)
{
    return __gs_dsp_fmax_s(src1, src2);
}
__EXTERN_INLINE_MANUAL float __fmax(float src1, float src2)
{
    return __gs_dsp_fmax_s(src1, src2);
}
__EXTERN_INLINE_MANUAL float __mmaxf32(float src1, float src2)
{
    return __gs_dsp_fmax_s(src1, src2);
}

/*
 * @brief				float classification
 * @param[in] src
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __fclassf(float src)
{
    return __gs_dsp_fclass_s(src);
}

/*
 * @brief				double classification
 * @param[in] src
 *
 * @return
 */
//__EXTERN_INLINE_MANUAL uint32 __fclass(double src)
//{
//    return __gs_dsp_fclass_d(src);
//}

/*
 * @brief				float multiply add, float op1 multiply with
 * 						op2, then add addend
 * @param[in] op1
 * @param[in] op2
 * @param[in] addend
 *
 * @return				op1*op2 + addend
 */
__EXTERN_INLINE_MANUAL float __fmadd3f(float op1, float op2, float addend)
{
    return __gs_dsp_fmadd3_s(op1, op2, addend);
}

/*
 * @brief				float multiply sub, float op1 multiply with
 * 						op2, then sub addend
 * @param[in] op1
 * @param[in] op2
 * @param[in] addend
 *
 * @return				op1*op2 - addend
 */
__EXTERN_INLINE_MANUAL float __fmsub3f(float op1, float op2, float addend)
{
    return __gs_dsp_fmsub3_s(op1, op2, addend);
}

/*
 * @brief				negative float multiply add, float op1 multiply with
 * 						op2, then add addend
 * @param[in] op1
 * @param[in] op2
 * @param[in] addend
 *
 * @return				-(op1*op2 + addend)
 */
__EXTERN_INLINE_MANUAL float __neg_fmadd3f(float op1, float op2, float addend)
{
    return __gs_dsp_neg_fmadd3_s(op1, op2, addend);
}

/*
 * @brief				negative float multiply sub, float op1 multiply with
 * 						op2, then sub addend
 * @param[in] op1
 * @param[in] op2
 * @param[in] addend
 *
 * @return				-(op1*op2 - addend)
 */
__EXTERN_INLINE_MANUAL float __neg_fmsub3f(float op1, float op2, float addend)
{
    return __gs_dsp_neg_fmsub3_s(op1, op2, addend);
}

/*
 * @brief				double multiply add, double op1 multiply with op2, add to addend
 * @param[in] op1
 * @param[in] op2
 * @param[in] addend
 *
 * @return				(op1*op2 + addend)
 */
__EXTERN_INLINE_MANUAL double __fma(double op1, double op2, double addend)
{
    return __gs_dsp_fma_d(op1, op2, addend);
}

/*
 * @brief		   		double square root
 * @param[in] src
 *
 * @return          	sqrt(src)
 */
__EXTERN_INLINE_MANUAL double __sqrt_d(double src)
{
    return __gs_dsp_sqrt_d(src);
}

/*
 * @brief				double absolute value
 * @param[in] src		double
 *
 * @return
 */
__EXTERN_INLINE_MANUAL double __fabs_d(double src)
{
    return __gs_dsp_fabs_d(src);
}

/*
 * @brief				get double minimal value of src1, src2
 * @param[in] src1
 * @param[in] src2
 *
 * @return
 */
__EXTERN_INLINE_MANUAL double __fmin_d(double src1, double src2)
{
    return __gs_dsp_fmin_d(src1, src2);
}

/*
 * @brief				get double maximal value of src1, src2
 * @param[in] src1
 * @param[in] src2
 *
 * @return
 */
__EXTERN_INLINE_MANUAL double __fmax_d(double src1, double src2)
{
    return __gs_dsp_fmax_d(src1, src2);
}

#if IS_GS32F3xx(0x22)
/*
 * @brief				double multiply add, double op1 multiply with
 * 						op2, then add addend
 * @param[in] op1
 * @param[in] op2
 * @param[in] addend
 *
 * @return				op1*op2 + addend
 */
extern double __gs_dsp_fmadd3_d(double op1, double op2, double addend);
__EXTERN_INLINE_MANUAL double __fmadd3(double op1, double op2, double addend)
{
    return __gs_dsp_fmadd3_d(op1, op2, addend);
}

/*
 * @brief				double multiply sub, double op1 multiply with
 * 						op2, then sub addend
 * @param[in] op1
 * @param[in] op2
 * @param[in] addend
 *
 * @return				op1*op2 - addend
 */
extern double __gs_dsp_fmsub3_d(double op1, double op2, double addend);
__EXTERN_INLINE_MANUAL double __fmsub3(double op1, double op2, double addend)
{
    return __gs_dsp_fmsub3_d(op1, op2, addend);
}

/*
 * @brief				negative double multiply add, double op1 multiply with
 * 						op2, then add addend
 * @param[in] op1
 * @param[in] op2
 * @param[in] addend
 *
 * @return				-(op1*op2 + addend)
 */
extern double __gs_dsp_neg_fmadd3_d(double op1, double op2, double addend);
__EXTERN_INLINE_MANUAL double __neg_fmadd3(double op1, double op2, double addend)
{
    return __gs_dsp_neg_fmadd3_d(op1, op2, addend);
}

/*
 * @brief				negative double multiply sub, double op1 multiply with
 * 						op2, then sub addend
 * @param[in] op1
 * @param[in] op2
 * @param[in] addend
 *
 * @return				-(op1*op2 - addend)
 */
extern double __gs_dsp_neg_fmsub3_d(double op1, double op2, double addend);
__EXTERN_INLINE_MANUAL double __neg_fmsub3(double op1, double op2, double addend)
{
    return __gs_dsp_neg_fmsub3_d(op1, op2, addend);
}
#endif

/*
 * @brief				4 pieces of byte add
 * @param[in] src1		32-bit int
 * @param[in] src2		32-bit int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int32 __sadd8_4p(int32 src1, int32 src2)
{
    return __gs_dsp_sadd8_4p(src1, src2);
}

/*
 * @brief				4 pieces of byte add
 * @param[in] src1		32-bit unsigned int
 * @param[in] src2		32-bit unsigned int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __uadd8_4p(uint32 src1, uint32 src2)
{
    return __gs_dsp_uadd8_4p(src1, src2);
}

/*
 * @brief				4 pieces of byte sub
 * @param[in] src1		32-bit int
 * @param[in] src2		32-bit int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int32 __ssub8_4p(int32 src1, int32 src2)
{
    return __gs_dsp_ssub8_4p(src1, src2);
}

/*
 * @brief				4 pieces of byte sub
 * @param[in] src1		32-bit unsigned int
 * @param[in] src2		32-bit unsigned int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __usub8_4p(uint32 src1, uint32 src2)
{
    return __gs_dsp_usub8_4p(src1, src2);
}

/*
 * @brief				pack half-word from bottoms of src1, src2
 * @param[in] src1		32-bit unsigned int
 * @param[in] src2		32-bit unsigned int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __pkh_bb(uint32 src1, uint32 src2)
{
    return __gs_dsp_pkh_bb(src1, src2);
}

/*
 * @brief				pack half-word from bottom of src1, top of src2
 * @param[in] src1		32-bit unsigned int
 * @param[in] src2		32-bit unsigned int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __pkh_bt(uint32 src1, uint32 src2)
{
    return __gs_dsp_pkh_bt(src1, src2);
}

/*
 * @brief				pack half-word from top of src1, bottom of src2
 * @param[in] src1		32-bit unsigned int
 * @param[in] src2		32-bit unsigned int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __pkh_tb(uint32 src1, uint32 src2)
{
    return __gs_dsp_pkh_tb(src1, src2);
}

/*
 * @brief				pack half-word from top of src1, top of src2
 * @param[in] src1		32-bit unsigned int
 * @param[in] src2		32-bit unsigned int
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __pkh_tt(uint32 src1, uint32 src2)
{
    return __gs_dsp_pkh_tt(src1, src2);
}

/*
 * @brief				Saturating Signed Multiplication Add,
 * 						and return the high word
 * @param[in] dst
 * @param[in] src1
 * @param[in] src2
 *
 * @return				SAT(dst + (rs1*rs2)[63:32])
 */
__EXTERN_INLINE_MANUAL int32 __ssadd_mulh(int32 dst, int32 src1, int32 src2)
{
    return __gs_dsp_ssadd_mulh(dst, src1, src2);
}

/*
 * @brief				Saturating Signed Multiplication and Subtraction
 * 						and return the high word
 * @param[in] dst
 * @param[in] src1
 * @param[in] src2
 *
 * @return				SAT(rd - (rs1*rs2)[63:32])
 */
__EXTERN_INLINE_MANUAL int32 __sssub_mulh(int32 dst, int32 src1, int32 src2)
{
    return __gs_dsp_sssub_mulh(dst, src1, src2);
}

/*
 * @brief				Saturating Signed Multiplication and Double,
 *                  	return the high word
 * @param[in] src1
 * @param[in] src2
 *
 * @return				SAT((rs1*rs2 <<1)[63:32])
 */
__EXTERN_INLINE_MANUAL int32 __ss2mul_high(int32 src1, int32 src2)
{
    return __gs_dsp_ss2mul_high(src1, src2);
}

/*
 * @brief				Saturating 16bit-Signed Clip Value
 * @param[in] src
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int32 __sat16(int32 src)
{
    return __gs_dsp_sat16(src);
}

/*
 * @brief				Saturating 16bit-unsigned Clip Value
 * @param[in] src
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __sat16u(int32 src)
{
    return __gs_dsp_sat16u(src);
}

/*
 * @brief				Saturating 32bit-Unsigned Clip Value
 * @param[in] src
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __sat32u(int32 src)
{
    return __gs_dsp_sat32u(src);
}

/*
 * @brief					Saturating specified bit-signed Clip Value
 * @param[in] src
 * @param[in] imm5u_valbits
 *
 * @return
 */
#define __sati(src, imm5u_valbits)    \
  __extension__                 \
  ({                            \
    int32 __res = __gs_dsp_sati (src, imm5u_valbits);	\
    __res;                      \
  })

/*
 * @brief					Saturating specified bit-Unsigned Clip Value
 * @param[in] src
 * @param[in] imm5u_valbits
 *
 * @return
 */
#define __satiu(src, imm5u_valbits)    \
  __extension__                 \
  ({                            \
    uint32 __res = __gs_dsp_satiu (src, imm5u_valbits);	\
    __res;                      \
  })

/*
 * @brief					Count numbers of leading 1
 * @param[in] src
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int32 __clo(uint32 src)
{
    return __gs_dsp_clo(src);
}

/*
 * @brief					64-bit Signed Halving Addition
 * @param[in] src1
 * @param[in] src2
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int64 __shadd64(int64 src1, int64 src2)
{
    return __gs_dsp_shadd64(src1, src2);
}

/*
 * @brief					64-bit Unsigned Halving Addition
 * @param[in] src1
 * @param[in] src2
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint64 __uhadd64(uint64 src1, uint64 src2)
{
    return __gs_dsp_uhadd64(src1, src2);
}

/*
 * @brief					64-bit Signed Halving Subtraction
 * @param[in] src1
 * @param[in] src2
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int64 __shsub64(int64 src1, int64 src2)
{
    return __gs_dsp_shsub64(src1, src2);
}

/*
 * @brief					64-bit Unsigned Halving Subtraction
 * @param[in] src1
 * @param[in] src2
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint64 __uhsub64(uint64 src1, uint64 src2)
{
    return __gs_dsp_uhsub64(src1, src2);
}

/*
 * @brief					64-bit Signed Saturation Halving Addition
 * @param[in] src1
 * @param[in] src2
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int64 __ssadd64(int64 src1, int64 src2)
{
    return __gs_dsp_ssadd64(src1, src2);
}

/*
 * @brief					64-bit Unsigned Saturation Halving Addition
 * @param[in] src1
 * @param[in] src2
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint64 __usadd64(uint64 src1, uint64 src2)
{
    return __gs_dsp_usadd64(src1, src2);
}

/*
 * @brief					64-bit Signed Saturation Halving Subtraction
 * @param[in] src1
 * @param[in] src2
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int64 __sssub64(int64 src1, int64 src2)
{
    return __gs_dsp_sssub64(src1, src2);
}

/*
 * @brief					64-bit Unsigned Saturation Halving Subtraction
 * @param[in] src1
 * @param[in] src2
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint64 __ussub64(uint64 src1, uint64 src2)
{
    return __gs_dsp_ussub64(src1, src2);
}

/*
 * @brief					Signed Multiply and Add to 64-Bit Data
 * @param[in] dst
 * @param[in] src1
 * @param[in] src2
 *
 * @return					dst + src1*src2
 */
__EXTERN_INLINE_MANUAL int64 __smadd64w(int64 dst, int32 src1, int32 src2)
{
    return __gs_dsp_smadd64w(dst, src1, src2);
}

/*
 * @brief					Signed Multiply and Sub to 64-Bit Data
 * @param[in] dst
 * @param[in] src1
 * @param[in] src2
 *
 * @return					dst - src1*src2
 */
__EXTERN_INLINE_MANUAL int64 __smsub64w(int64 dst, int32 src1, int32 src2)
{
    return __gs_dsp_smsub64w(dst, src1, src2);
}

/*
 * @brief					Unsigned Multiply and Add to 64-Bit Data
 * @param[in] dst
 * @param[in] src1
 * @param[in] src2
 *
 * @return					dst + src1*src2
 */
__EXTERN_INLINE_MANUAL uint64 __umadd64w(uint64 dst, uint32 src1, uint32 src2)
{
    return __gs_dsp_umadd64w(dst, src1, src2);
}

/*
 * @brief					Unsigned Multiply and Subtract from 64-Bit Data
 * @param[in] dst
 * @param[in] src1
 * @param[in] src2
 *
 * @return					dst - src1*src2
 */
__EXTERN_INLINE_MANUAL uint64 __umsub64w(uint64 dst, uint32 src1, uint32 src2)
{
    return __gs_dsp_umsub64w(dst, src1, src2);
}

/*
 * @brief					Signed Multiply and Saturating Add to 64-Bit Data
 * @param[in] dst
 * @param[in] src1
 * @param[in] src2
 *
 * @return					SAT(dst + src1*src2)
 */
__EXTERN_INLINE_MANUAL int64 __ssmadd64w(int64 dst, int32 src1, int32 src2)
{
    return __gs_dsp_ssmadd64w(dst, src1, src2);
}

/*
 * @brief					Signed Multiply and Saturating Sub to 64-Bit Data
 * @param[in] dst
 * @param[in] src1
 * @param[in] src2
 *
 * @return					SAT(dst - src1*src2)
 */
__EXTERN_INLINE_MANUAL int64 __ssmsub64w(int64 dst, int32 src1, int32 src2)
{
    return __gs_dsp_ssmsub64w(dst, src1, src2);
}

/*
 * @brief					Unsigned Multiply and Saturating Add to 64-Bit Data
 * @param[in] dst
 * @param[in] src1
 * @param[in] src2
 *
 * @return					SAT(dst + src1*src2)
 */
__EXTERN_INLINE_MANUAL uint64 __usmadd64w(uint64 dst, uint32 src1, uint32 src2)
{
    return __gs_dsp_usmadd64w(dst, src1, src2);
}

/*
 * @brief					Unsigned Multiply and Saturating Subtract from 64-Bit Data
 * @param[in] dst
 * @param[in] src1
 * @param[in] src2
 *
 * @return					SAT(dst - src1*src2)
 */
__EXTERN_INLINE_MANUAL uint64 __usmsub64w(uint64 dst, uint32 src1, uint32 src2)
{
    return __gs_dsp_usmsub64w(dst, src1, src2);
}

/*
 * @brief					Saturation Signed Addition
 * @param[in] src1
 * @param[in] src2
 *
 * @return					SAT(src1+src2)
 */
__EXTERN_INLINE_MANUAL int32 __ssadd(int32 src1, int32 src2)
{
    return __gs_dsp_ssadd(src1, src2);
}

/*
 * @brief					Saturation Unsigned Addition
 * @param[in] src1
 * @param[in] src2
 *
 * @return					SAT(src1+src2)
 */
__EXTERN_INLINE_MANUAL uint32 __usadd(uint32 src1, uint32 src2)
{
    return __gs_dsp_usadd(src1, src2);
}

/*
 * @brief					Saturation Signed Subtraction
 * @param[in] src1
 * @param[in] src2
 *
 * @return					SAT(src1-src2)
 */
__EXTERN_INLINE_MANUAL int32 __sssub(int32 src1, int32 src2)
{
    return __gs_dsp_sssub(src1, src2);
}

/*
 * @brief					Saturation Unsigned Subtraction
 * @param[in] src1
 * @param[in] src2
 *
 * @return					SAT(src1-src2)
 */
__EXTERN_INLINE_MANUAL uint32 __ussub(uint32 src1, uint32 src2)
{
    return __gs_dsp_ussub(src1, src2);
}

/*
 * @brief					Shift Left with Saturation or Shift Right Arithmetic
 * @param[in] src
 * @param[in] amount		amount>0, shift right
 * 							amount<0, shift left
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int32 __sls_sra(int32 src, int32 amount)
{
    return __gs_dsp_sls_sra(src, amount);
}

/*
 * @brief					Shift Left Saturation
 * @param[in] src
 * @param[in] amount		shift bit number
 *
 * @return					SAT(src<<amount)
 */
__EXTERN_INLINE_MANUAL int32 __sls(int32 src, uint32 amount)
{
    return __gs_dsp_sls(src, amount);
}

/*
 * @brief					Shift Left Saturating Immediate
 * @param[in] src
 * @param[in] imm5u_shamt	shift bit number
 *
 * @return					SAT(src<<amount)
 */
#define __slsi(src, imm5u_shamt)    \
  __extension__                 \
  ({                            \
    int32 __res = __gs_dsp_slsi (src, imm5u_shamt);	\
    __res;                      \
  })

/*
 * @brief					Absolute Value with Saturation
 * @param[in] src
 *
 * @return					SAT(Abs(src))
 */
__EXTERN_INLINE_MANUAL uint32 __abs_sat(int32 src)
{
    return __gs_dsp_abs_sat(src);
}

/*
 * @brief					32-bit Signed Halving Addition
 * @param[in] src1
 * @param[in] src2
 *
 * @return					(src1+src2)>>1
 */
__EXTERN_INLINE_MANUAL int32 __shadd(int32 src1, int32 src2)
{
    return __gs_dsp_shadd(src1, src2);
}

/*
 * @brief					32-bit UnSigned Halving Addition
 * @param[in] src1
 * @param[in] src2
 *
 * @return					(src1+src2)>>1
 */
__EXTERN_INLINE_MANUAL uint32 __uhadd(uint32 src1, uint32 src2)
{
    return __gs_dsp_uhadd(src1, src2);
}

/*
 * @brief					32-bit Signed Halving Subtraction
 * @param[in] src1
 * @param[in] src2
 *
 * @return					(src1-src2)>>1
 */
__EXTERN_INLINE_MANUAL int32 __shsub(int32 src1, int32 src2)
{
    return __gs_dsp_shsub(src1, src2);
}

/*
 * @brief					32-bit UnSigned Halving Subtraction
 * @param[in] src1
 * @param[in] src2
 *
 * @return					(src1-src2)>>1
 */
__EXTERN_INLINE_MANUAL uint32 __uhsub(uint32 src1, uint32 src2)
{
    return __gs_dsp_uhsub(src1, src2);
}

/*
 * @brief					Unsigned Multiply to 64-bit data
 * @param[in] src1
 * @param[in] src2
 *
 * @return					(src1*src2)
 */
__EXTERN_INLINE_MANUAL uint64 __umul64w(uint32 src1, uint32 src2)
{
    return __gs_dsp_umul64w(src1, src2);
}

/*
 * @brief					Signed Multiply to 64-bit data
 * @param[in] src1
 * @param[in] src2
 *
 * @return					(src1*src2)
 */
__EXTERN_INLINE_MANUAL int64 __smul64w(int32 src1, int32 src2)
{
    return __gs_dsp_smul64w(src1, src2);
}

/*
 * @brief		Read Saturation OV flag
 *
 * @return      None
 */
__EXTERN_INLINE_MANUAL uint32 __read_ov(void)
{
    return __gs_dsp_read_ov();
}

/*
 * @brief		Read Saturation OV flag and Clear
 *
 * @return      None
 */
__EXTERN_INLINE_MANUAL uint32 __clr_ov_rd(void)
{
    return __gs_dsp_read_clr_ov();
}

/*
 * @brief		Clear Saturation OV flag
 *
 * @return      None
 */
__EXTERN_INLINE_MANUAL void __clr_ov(void)
{
    __gs_dsp_clr_ov();
}

/*
 * @brief					Average with Rounding
 * @param[in] src1
 * @param[in] src2
 *
 * @return					round((src1+src2)/2)
 */
__EXTERN_INLINE_MANUAL int32 __rnd_ave(int32 src1, int32 src2)
{
    return __gs_dsp_rnd_ave(src1, src2);
}

//__EXTERN_INLINE_MANUAL uint32 __rev_msb(uint32 src, uint32 msb)
//{
//    return __gs_dsp_rev_msb(src, msb);
//}

/*
 * @brief					imm5u_msb-Bit Reverse
 * @param[in] src
 * @param[in] imm5u_msb
 *
 * @return
 */
#define __rbit_msbi(src, imm5u_msb)    \
  __extension__                 \
  ({                            \
    uint32 __res = __gs_dsp_rbit_msbi (src, imm5u_msb);	\
    __res;                      \
  })

/*
 * @brief				Extract specified Word from 64-bit
 * @param[in] src
 * @param[in] pos
 *
 * @return				src[(31+pos[4:0]):pos[4:0]]
 */
__EXTERN_INLINE_MANUAL int32 __w_64ext(int64 src, uint32 pos)
{
    return __gs_dsp_w_64ext(src, pos);
}

/*
 * @brief				Extract specified Word from 64-bit
 * @param[in] src
 * @param[in] imm5u_pos
 *
 * @return				src[(31+imm5u_pos):imm5u_pos]
 */
#define __w_64exti(src, imm5u_pos)    \
  __extension__                 \
  ({                            \
    int32 __res = __gs_dsp_w_64exti (src, imm5u_pos);	\
    __res;                      \
  })

/*
 * @brief				Bits-Mask Pick
 * @param[in] src1
 * @param[in] src2
 * @param[in] bitmask
 *
 * @return				rd[i] = bitmask[i]? rs1[i] : rs2[i]
 */
__EXTERN_INLINE_MANUAL uint32 __bm_pick(uint32 src1, uint32 src2, uint32 bitmask)
{
    return __gs_dsp_bm_pick(src1, src2, bitmask);
}

/*
 * @brief				Multiply and Add to Singed 32-Bit Word
 * @param[in] src1
 * @param[in] src2
 * @param[in] dst
 *
 * @return				dst + src1*src2
 */
__EXTERN_INLINE_MANUAL int32 __smadd32(int32 dst, int32 src1, int32 src2)
{
    return __gs_dsp_smadd32(dst, src1, src2);
}

/*
 * @brief				Multiply and Add to UnSinged 32-Bit Word
 * @param[in] src1
 * @param[in] src2
 * @param[in] dst
 *
 * @return				dst + src1*src2
 */
__EXTERN_INLINE_MANUAL uint32 __umadd32(uint32 dst, uint32 src1, uint32 src2)
{
    return __gs_dsp_umadd32(dst, src1, src2);
}

/*
 * @brief					signed extract bits-field
 * @param[in] src
 * @param[in] imm5u_len		>=1
 * @param[in] imm5u_pos 	>=0
 *
 * @return					Sext(src[min(len+pos-1, 31):pos5u])
 */
#define __bf_exti(src, imm5u_len, imm5u_pos)    \
  __extension__                 \
  ({                            \
    int32 __res = __gs_dsp_bf_exti (src, imm5u_len, imm5u_pos);	\
    __res;                      \
  })

/*
 * @brief					Unsigned extract bits-field
 * @param[in] src
 * @param[in] imm5u_len		>=1
 * @param[in] imm5u_pos 	>=0
 *
 * @return					Sext(src[min(len+pos-1, 31):pos5u])
 */
#define __bf_extui(src, imm5u_len, imm5u_pos)    \
  __extension__                 \
  ({                            \
    uint32 __res = __gs_dsp_bf_extui (src, imm5u_len, imm5u_pos);	\
    __res;                      \
  })

/*
 * @brief					signed extract bits-field
 * @param[in] src
 * @param[in] len5u_pos5u	bit[4:0]-base position,
 * 							bit[9:5]-top position
 *
 * @return					Sext(src[min(len5u_pos5u[9:5]+len5u_pos5u[4:0]-1, 31):len5u_pos5u[4:0])
 */
__EXTERN_INLINE_MANUAL int32 __bf_ext(uint32 src, uint32 len5u_pos5u)
{
    return __gs_dsp_bf_ext(src, len5u_pos5u);
}

/*
 * @brief					Unsigned extract bits-field
 * @param[in] src
 * @param[in] len5u_pos5u	bit[4:0]-base position,
 * 							bit[9:5]-top position
 *
 * @return					Sext(src[min(len5u_pos5u[9:5]+len5u_pos5u[4:0]-1, 31):len5u_pos5u[4:0])
 */
__EXTERN_INLINE_MANUAL uint32 __bf_extu(uint32 src, uint32 len5u_pos5u)
{
    return __gs_dsp_bf_extu(src, len5u_pos5u);
}

/*
 * @brief					Insert-Modify bits-field
 * @param[in] src
 * @param[in] imm5u_inc
 * @param[in] imm5u_pos
 *
 * @return					rd[min(inc5u+pos5u, 31):pos5u] = src[inc5u-(max(inc5u+pos5u,32)-32):0])
 */
#define __bf_insi(src, imm5u_inc, imm5u_pos)    \
  __extension__                 \
  ({                            \
    uint32 __res = __gs_dsp_bf_insi (src, imm5u_inc, imm5u_pos);	\
    __res;                      \
  })

/*
 * @brief					move and set all one for bits-field
 * @param[in] src
 * @param[in] imm5u_inc
 * @param[in] imm5u_pos
 *
 * @return					rd = src | (((1 << (inc5u+1)) - 1) << pos5u)
 */
#define __bf_seti(src, imm5u_inc, imm5u_pos)    \
  __extension__                 \
  ({                            \
    uint32 __res = __gs_dsp_bf_seti (src, imm5u_inc, imm5u_pos);	\
    __res;                      \
  })

/*
 * @brief					set bits-mask
 * @param[in] src
 * @param[in] inc5u_pos5u
 *
 * @return					rd = rs1 | (((1 << (inc5u_pos5u[9:5]+1)) -1) << inc5u_pos5u[4:0])
 */
__EXTERN_INLINE_MANUAL uint32 __bf_set(uint32 src, uint32 inc5u_pos5u)
{
    return __gs_dsp_bf_set(src, inc5u_pos5u);
}

/*
 * @brief					move and clr all zero for bits-field
 * @param[in] src
 * @param[in] imm5u_inc
 * @param[in] imm5u_pos
 *
 * @return					rd = src & (~(((1 << (inc5u+1))-1) << pos5u))
 */
#define __bf_clri(src, imm5u_inc, imm5u_pos)    \
  __extension__                 \
  ({                            \
    uint32 __res = __gs_dsp_bf_clri (src, imm5u_inc, imm5u_pos);	\
    __res;                      \
  })

/*
 * @brief					clr bits-mask
 * @param[in] src
 * @param[in] inc5u_pos5u
 *
 * @return					rd = rs1 & ( ~(((1 << (rs2[9:5]+1))-1) << rs2[4:0]))
 */
__EXTERN_INLINE_MANUAL uint32 __bf_clr(uint32 src, uint32 inc5u_pos5u)
{
    return __gs_dsp_bf_clr(src, inc5u_pos5u);
}

/*
 * @brief					Count the leading bit (0 or 1), if src is 0, 0 is returned.
 * @param[in] src
 *
 * @return
 */
__EXTERN_INLINE_MANUAL int32 __clbit(uint32 src)
{
    return __gs_dsp_clbit(src);
}

/*
 * @brief					first non-zero byte, otherwise return zero, src byte number is 0,1,2,3
 * @param[in] src
 *
 * @return
 */
__EXTERN_INLINE_MANUAL uint32 __ffnz(uint32 src)
{
    return __gs_dsp_ffnz(src);
}

__EXTERN_INLINE_MANUAL int16_t __byte(uint32_t* baseAdd, uint32 offset)
{
	return (int16_t)(*((uint8_t *)baseAdd + offset));
}

extern float __gs_dsp_exp(float x);

extern float __gs_dsp_log(float x);

extern float __gs_dsp_powf(float x, float y);

/*
 * @brief					Base-e exponential of x
 *
 * @return					e^x
 */
__EXTERN_INLINE_MANUAL float __expf32  (float x)
{
    return __gs_dsp_exp(x);
}

/*
 * @brief					Base-e logarithm of x
 *
 * @return					ln x
 */
__EXTERN_INLINE_MANUAL float __logf32  (float x)
{
    return __gs_dsp_log(x);
}

/*
 * @brief					x raised to the power of y
 *
 * @return					x^y
 */
__EXTERN_INLINE_MANUAL float __powf32  (float x, float y)
{
    return __gs_dsp_powf(x,y);
}


#endif /* GS32_MATH_LIB_H */
