/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File adc_ex02_soc_epwm_bitfield.c
 *
 * @Title   ADC ePWM Triggering
 *
 * This example sets up ePWM1 to periodically trigger a
 * conversion on ADCA.
 * The time between samples is determined based on the period of the ePWM timer.
 *
 * @External Connections:
 * - A0 should be connected to signals to convert.
 *
 * @Watch Variables:
 * - myADC0Results - A sequence of analog-to-digital
 * 					 conversion samples from pin A0.
 *
 */
#ifdef __cplusplus
extern "C"
{
#endif

/*
 * Include Files
 */
#include "adc_ex02_board.h"

/*
 * Macros & Typedefs
 */
#define RESULTS_BUFFER_SIZE 256

/*
 * Global Variables
 */
/* Buffer for results */
uint16_t myADC0Results[RESULTS_BUFFER_SIZE];
/* Index into result buffer */
uint16_t myindex;
/* Flag to indicate buffer is full */
volatile uint16_t bufferFull;

/*
 * Local Function Prototypes
 */
void initEPWM(void);

/* Main */
void main(void)
{
    /* Initialize device clock and peripherals */
    Device_init();

    /*
     * Board Initialization
     * - Set up the ADC and initialize the SOC.
     * - Enable ADC interrupt.
     */
    Board_init();

    /* Set up the ePWM */
    initEPWM();

    /* Initialize results buffer */
    for(myindex = 0; myindex < RESULTS_BUFFER_SIZE; myindex++)
    {
        myADC0Results[myindex] = 0;
    }
    myindex = 0;
    bufferFull = 0;

    /* Enable Global Interrupt (INTM) and realtime interrupt (DBGM) */
    EINT;

    /* Loop indefinitely */
    while(1)
    {
    	/* Start ePWM1, enabling SOCA and putting the counter in up-count mode */
        EPWM_enableADCTrigger(EPWM1_BASE, EPWM_SOC_A);
        EPWM_setTimeBaseCounterMode(EPWM1_BASE, EPWM_COUNTER_MODE_UP);
        SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

        /*
         * Wait while ePWM1 causes ADC conversions which then cause interrupts.
         * When the results buffer is filled, the bufferFull flag will be set.
         */
        while(bufferFull == 0)
        {
        }
        /* Clear the buffer full flag */
        bufferFull = 0;

        /* Stop ePWM1, disabling SOCA and freezing the counter */
        EPWM_disableADCTrigger(EPWM1_BASE, EPWM_SOC_A);
        EPWM_setTimeBaseCounterMode(EPWM1_BASE, EPWM_COUNTER_MODE_STOP_FREEZE);

    }
}

/*
 * Local Function Definitions
 */
/* Function to configure ePWM1 to generate the SOC. */
void initEPWM(void)
{
    /* Disable SOCA */
    EPWM_disableADCTrigger(EPWM1_BASE, EPWM_SOC_A);

    /* Configure the SOC to occur on the first up-count event */
    EPWM_setADCTriggerSource(EPWM1_BASE, EPWM_SOC_A, EPWM_SOC_TBCTR_U_CMPA);
    EPWM_setADCTriggerEventPrescale(EPWM1_BASE, EPWM_SOC_A, 1);

    /*
     * Set the compare A value to 1000 and the period to 1999
     * Assuming ePWM clock is 100MHz, this would give 50kHz sampling
     * 50MHz ePWM clock would give 25kHz sampling, etc.
     * The sample rate can also be modulated by changing the ePWM period
     * directly (ensure that the compare A value is less than the period).
     */
    EPWM_setCounterCompareValue(EPWM1_BASE, EPWM_COUNTER_COMPARE_A, 1000);
    EPWM_setTimeBasePeriod(EPWM1_BASE, 1999);

    /* Set the local ePWM module clock divider to /1 */
    EPWM_setClockPrescaler(EPWM1_BASE,
                           EPWM_CLOCK_DIVIDER_1,
                           EPWM_HSCLOCK_DIVIDER_1);

    /* Freeze the counter */
    EPWM_setTimeBaseCounterMode(EPWM1_BASE, EPWM_COUNTER_MODE_STOP_FREEZE);
}

/* adcA1ISR - ADC A Interrupt 1 ISR */
void adcA1ISR(void)
{
	/* Add the latest result to the buffer */
	myADC0Results[myindex++] = AdcaResultRegs.ADCRESULT0;

	/* Set the bufferFull flag if the buffer is full */
    if(RESULTS_BUFFER_SIZE <= myindex)
    {
        myindex = 0;
        bufferFull = 1;
    }

    /* Clear the interrupt flag */
    AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;

    /* Check if overflow has occurred */
    if(true ==  AdcaRegs.ADCINTOVF.bit.ADCINT1)
    {
    	AdcaRegs.ADCINTOVFCLR.bit.ADCINT1 = 1;
    	AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;
    }
}

#ifdef __cplusplus
}
#endif
