/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef __cplusplus
extern "C"
{
#endif

/*
 * Include Files
 */
#include "adc_ex07_board.h"
#include "cpu.h"

/*
 * Macros & Typedefs
 */
#define EPWM_MODE_SELECT EPWM_MODE_SPWM_0

/*
 * @brief Board Configurations
 *
 * Initializes the rest of the modules.
 * Call this function in your application if you wish to do all module
 * initialization.
 * If you wish to not use some of the initializations, instead of the
 * Board_init use the individual Module_inits.
 */
void Board_init()
{
	myADC0_init();
	myADC1_init();
}
void myADC0_init()
{
	/*
	 * @brief ADC Initialization:
	 * 		  Write ADC configurations and power up the ADC.
	 */
	/* Select ADC Reference mode and Reference voltage */
	ADC_setVREF(myADC0_BASE, ADC_REFERENCE_INTERNAL, ADC_REFERENCE_3_3V);

	/* Set trim */
	ADC_setINLTrim(myADC0_BASE);

	/* Configures the analog-to-digital converter module prescaler. */
	ADC_setPrescaler(myADC0_BASE, ADC_CLK_DIV_4_0);

	/* Sets the timing of the end-of-conversion pulse */
	ADC_setInterruptPulseMode(myADC0_BASE, ADC_PULSE_END_OF_CONV);

	/* Powers up the ADCA core. */
	ADC_enableConverter(myADC0_BASE);

	/* Delay for 1ms to allow ADCA time to power up */
	DEVICE_DELAY_US(5000);

	/*
	 * @brief SOC Configuration:
	 * 		  Setup ADC EPWM channel and trigger settings.
	 */
	/* Disables SOC burst mode. */
	ADC_disableBurstMode(myADC0_BASE);

	/* Sets the priority mode of the SOCs. */
	ADC_setSOCPriority(myADC0_BASE, ADC_PRI_ALL_ROUND_ROBIN);

	/*
	 * @brief Start of Conversion 0 Configuration.
	 *
	 * Configures a SOC in the ADC and its interrupt SOC trigger.
	 * 			SOC number		: 0
	 * 			Trigger			: ADC_TRIGGER_SW_ONLY
	 * 			Channel			: ADC_CH_ADCIN2
	 * 			Sample Window	: 50 SYSCLK cycles
	 * 			Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	 */
	ADC_setupSOC(myADC0_BASE, ADC_SOC_NUMBER0, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN2, 50U);
	ADC_setInterruptSOCTrigger(myADC0_BASE, ADC_SOC_NUMBER0, ADC_INT_SOC_TRIGGER_NONE);

	/*
	 * @brief Start of Conversion 1 Configuration.
	 *
	 * Configures a SOC in the ADC and its interrupt SOC trigger.
	 * 			SOC number		: 1
	 * 			Trigger			: ADC_TRIGGER_SW_ONLY
	 * 			Channel			: ADC_CH_ADCIN2
	 * 			Sample Window	: 50 SYSCLK cycles
	 * 			Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	 */
	ADC_setupSOC(myADC0_BASE, ADC_SOC_NUMBER1, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN2, 50U);
	ADC_setInterruptSOCTrigger(myADC0_BASE, ADC_SOC_NUMBER1, ADC_INT_SOC_TRIGGER_NONE);

	/*
	 * @brief ADC Interrupt Initialization:
	 *
	 * ADC Interrupt 1 Configuration
	 * 		Source	: ADC_INT_TRIGGER_EOC1
	 * 		Interrupt Source: enabled
	 * 		Continuous Mode	: disabled
	 */
	ADC_setInterruptSource(myADC0_BASE, ADC_INT_NUMBER1, ADC_INT_TRIGGER_EOC1);
	ADC_clearInterruptStatus(myADC0_BASE, ADC_INT_NUMBER1);
	ADC_disableContinuousMode(myADC0_BASE, ADC_INT_NUMBER1);
	ADC_enableInterrupt(myADC0_BASE, ADC_INT_NUMBER1);

	/*
	 * PPB Configuration: Configure high and low limits detection for ADCPPB
	 *
	 * Post Processing Block 1 Configuration
	 * 		Configures a post-processing block (PPB) in the ADC.
	 * 		PPB Number				: 1
	 * 		SOC/EOC number			: 1
	 * 		Calibration Offset		: 100
	 * 		Reference Offset		: 0
	 * 		Two's Complement		: Disabled
	 * 		Trip High Limit			: 0
	 * 		Trip Low Limit			: 0
	 * 		Clear PPB Event Flags	: Disabled
	 */
	ADC_setupPPB(myADC0_BASE, ADC_PPB_NUMBER1, ADC_SOC_NUMBER1);
	ADC_disablePPBEvent(myADC0_BASE, ADC_PPB_NUMBER1, (ADC_EVT_TRIPHI | ADC_EVT_TRIPLO | ADC_EVT_ZERO));
	ADC_disablePPBEventInterrupt(myADC0_BASE, ADC_PPB_NUMBER1, (ADC_EVT_TRIPHI | ADC_EVT_TRIPLO | ADC_EVT_ZERO));
	ADC_setPPBCalibrationOffset(myADC0_BASE, ADC_PPB_NUMBER1, 100);
	ADC_setPPBReferenceOffset(myADC0_BASE, ADC_PPB_NUMBER1, 0);
	ADC_disablePPBTwosComplement(myADC0_BASE, ADC_PPB_NUMBER1);
	ADC_setPPBTripLimits(myADC0_BASE, ADC_PPB_NUMBER1, 0, 0);
	ADC_disablePPBEventCBCClear(myADC0_BASE, ADC_PPB_NUMBER1);
}

void myADC1_init()
{
	/*
	 * @brief ADC Initialization:
	 * 		  Write ADC configurations and power up the ADC.
	 */
	/* Select ADC Reference mode and Reference voltage */
	ADC_setVREF(myADC1_BASE, ADC_REFERENCE_INTERNAL, ADC_REFERENCE_3_3V);

	/* Set trim */
	ADC_setINLTrim(myADC1_BASE);

	/* Configures the analog-to-digital converter module prescaler. */
	ADC_setPrescaler(myADC1_BASE, ADC_CLK_DIV_4_0);

	/* Sets the timing of the end-of-conversion pulse */
	ADC_setInterruptPulseMode(myADC1_BASE, ADC_PULSE_END_OF_CONV);

	/* Powers up the ADCA core. */
	ADC_enableConverter(myADC1_BASE);

	/* Delay for 1ms to allow ADCA time to power up */
	DEVICE_DELAY_US(5000);

	/*
	 * @brief SOC Configuration:
	 * 		  Setup ADC EPWM channel and trigger settings.
	 */
	/* Disables SOC burst mode. */
	ADC_disableBurstMode(myADC1_BASE);

	/* Sets the priority mode of the SOCs. */
	ADC_setSOCPriority(myADC1_BASE, ADC_PRI_ALL_ROUND_ROBIN);

	/*
	 * @brief Start of Conversion 0 Configuration.
	 *
	 * Configures a SOC in the ADC and its interrupt SOC trigger.
	 * 			SOC number		: 0
	 * 			Trigger			: ADC_TRIGGER_SW_ONLY
	 * 			Channel			: ADC_CH_ADCIN2
	 * 			Sample Window	: 50 SYSCLK cycles
	 * 			Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	 */
	ADC_setupSOC(myADC1_BASE, ADC_SOC_NUMBER0, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN2, 50U);
	ADC_setInterruptSOCTrigger(myADC1_BASE, ADC_SOC_NUMBER0, ADC_INT_SOC_TRIGGER_NONE);

	/*
	 * @brief Start of Conversion 1 Configuration.
	 *
	 * Configures a SOC in the ADC and its interrupt SOC trigger.
	 * 			SOC number		: 1
	 * 			Trigger			: ADC_TRIGGER_SW_ONLY
	 * 			Channel			: ADC_CH_ADCIN2
	 * 			Sample Window	: 50 SYSCLK cycles
	 * 			Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	 */
	ADC_setupSOC(myADC1_BASE, ADC_SOC_NUMBER1, ADC_TRIGGER_SW_ONLY, ADC_CH_ADCIN2, 9U);
	ADC_setInterruptSOCTrigger(myADC1_BASE, ADC_SOC_NUMBER1, ADC_INT_SOC_TRIGGER_NONE);

	/*
	 * @brief ADC Interrupt Initialization:
	 *
	 * ADC Interrupt 1 Configuration
	 * 		Source	: ADC_INT_TRIGGER_EOC1
	 * 		Interrupt Source: enabled
	 * 		Continuous Mode	: disabled
	 */
	ADC_setInterruptSource(myADC1_BASE, ADC_INT_NUMBER1, ADC_INT_TRIGGER_EOC1);
	ADC_clearInterruptStatus(myADC1_BASE, ADC_INT_NUMBER1);
	ADC_disableContinuousMode(myADC1_BASE, ADC_INT_NUMBER1);
	ADC_enableInterrupt(myADC1_BASE, ADC_INT_NUMBER1);

	/*
	 * PPB Configuration: Configure high and low limits detection for ADCPPB
	 *
	 * Post Processing Block 1 Configuration
	 * 		Configures a post-processing block (PPB) in the ADC.
	 * 		PPB Number				: 1
	 * 		SOC/EOC number			: 1
	 * 		Calibration Offset		: -100
	 * 		Reference Offset		: 0
	 * 		Two's Complement		: Disabled
	 * 		Trip High Limit			: 0
	 * 		Trip Low Limit			: 0
	 * 		Clear PPB Event Flags	: Disabled
	 */
	ADC_setupPPB(myADC1_BASE, ADC_PPB_NUMBER1, ADC_SOC_NUMBER1);
	ADC_disablePPBEvent(myADC1_BASE, ADC_PPB_NUMBER1, (ADC_EVT_TRIPHI | ADC_EVT_TRIPLO | ADC_EVT_ZERO));
	ADC_disablePPBEventInterrupt(myADC1_BASE, ADC_PPB_NUMBER1, (ADC_EVT_TRIPHI | ADC_EVT_TRIPLO | ADC_EVT_ZERO));
	ADC_setPPBCalibrationOffset(myADC1_BASE, ADC_PPB_NUMBER1, -100);
	ADC_setPPBReferenceOffset(myADC1_BASE, ADC_PPB_NUMBER1, 0);
	ADC_disablePPBTwosComplement(myADC1_BASE, ADC_PPB_NUMBER1);
	ADC_setPPBTripLimits(myADC1_BASE, ADC_PPB_NUMBER1, 0, 0);
	ADC_disablePPBEventCBCClear(myADC1_BASE, ADC_PPB_NUMBER1);
}

#ifdef __cplusplus

#endif
