/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File adc_ex13_soc_oversampling_bitfield.c
 *
 * @Title ADC Oversampling Using Multiple SOCs
 *
 * This example sets up ePWM1 to periodically trigger a set of conversions on
 * ADCA including multiple SOCs that all convert A2 to achieve oversampling on
 * A2.
 *
 * @External Connections:
 * - A0, A1, A2 should be connected to signals to be converted.
 *
 * @Watch Variables
 * - adcAResult0 - Digital representation of the voltage on pin A0.
 * - adcAResult1 - Digital representation of the voltage on pin A1.
 * - adcAResult2 - Digital representation of the voltage on pin A2
 *
 */

/*
 * Include Files
 */
#include "adc_ex13_board.h"

/*
 * Global Variables
 */
uint16_t adcAResult0;
uint16_t adcAResult1;
uint16_t adcAResult2;

/*
 * Local Function Prototypes
 */
void initEPWM();

/*
 * ISR Function
 */
void adcA1ISR(void);

/* Main */
void main(void)
{
	/* Initialize device clock and peripherals */
    Device_init();

    /*
     * Set up ADCs:
     *
     * Configure SOC0-1 in round robin mode to sample A0 and A1 channel
     * respectively and SOC2-SOC5 oversamples channel A2 on EPWM1SOCA
     * trigger.
     *
     * Register and enable the interrupts:
     * SOC0 and SOC1 results are read after each PWM trigger for
     * channel 0 and 1 outputs while SOC2-SOC5 results will be
     * averaged together for oversampled channel A2 result
     */
    Board_init();

    /* Initialize PWM */
    initEPWM();

    /* Enable Global Interrupt (INTM) */
    EINT;

    /* Start ePWM1, enabling SOCA and putting the counter in up-count mode */
    EPWM_enableADCTrigger(EPWM1_BASE, EPWM_SOC_A);
    EPWM_setTimeBaseCounterMode(EPWM1_BASE, EPWM_COUNTER_MODE_UP);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    /* Take conversions indefinitely in loop */
    do
    {
        /*
         * Wait while ePWM causes ADC conversions.
         * ADCA1 ISR processes each new set of conversions.
         */
    }
    while(1);
}

/* Function to configure ePWM1 to generate the SOC. */
void initEPWM(void)
{
    /* Disable SOCA */
    EPWM_disableADCTrigger(EPWM1_BASE, EPWM_SOC_A);

    /* Configure the SOC to occur on the first up-count event */
    EPWM_setADCTriggerSource(EPWM1_BASE, EPWM_SOC_A, EPWM_SOC_TBCTR_U_CMPA);
    EPWM_setADCTriggerEventPrescale(EPWM1_BASE, EPWM_SOC_A, 1);

    /*
     * Set the compare A value to 1000 and the period to 1999
     * Assuming ePWM clock is 100MHz, this would give 50kHz sampling
     * 50MHz ePWM clock would give 25kHz sampling, etc.
     * The sample rate can also be modulated by changing the ePWM period
     * directly (ensure that the compare A value is less than the period).
     */
    EPWM_setCounterCompareValue(EPWM1_BASE, EPWM_COUNTER_COMPARE_A, 1000);
    EPWM_setTimeBasePeriod(EPWM1_BASE, 1999);

    /* Set the local ePWM module clock divider to /1 */
    EPWM_setClockPrescaler(EPWM1_BASE,
                           EPWM_CLOCK_DIVIDER_1,
                           EPWM_HSCLOCK_DIVIDER_1);

    /* Freeze the counter */
    EPWM_setTimeBaseCounterMode(EPWM1_BASE, EPWM_COUNTER_MODE_STOP_FREEZE);
}

/* adcA1ISR - ADC A Interrupt 1 ISR */
void adcA1ISR(void)
{
	/* Store results for A0 and A1 */
	adcAResult0 = AdcaResultRegs.ADCRESULT0;
	adcAResult1 = AdcaResultRegs.ADCRESULT1;

    /* Average the 4 oversampled A2 results together */
	adcAResult2 = (AdcaResultRegs.ADCRESULT2 + AdcaResultRegs.ADCRESULT3 + AdcaResultRegs.ADCRESULT4 + AdcaResultRegs.ADCRESULT5) >> 2;

    /* Clear the interrupt flag */
	AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;

    /* Check if overflow has occurred */
	if(true == AdcaRegs.ADCINTOVF.bit.ADCINT1)
    {
		AdcaRegs.ADCINTOVFCLR.bit.ADCINT1 = 1;
		AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;
    }

}

/* End of file */
