/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *   @file    HPEC_NICE.h
 *   @brief
 *
 */

#ifndef GS32_PWM_H_
#define GS32_PWM_H_

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "gs32_pwm_clu.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */


/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */
/**
 * \brief   PWM Generator Function Definition
 *
 */
static inline void
PWM_vect_3P2L_run_ref(PWM_VECT_3P2L_Obj *v, PWM_VECT_3P2L_MODE mode) {
    float32_t t_delta, t_total;
    volatile uint16 CMPx, CMPy, CMPz;
    volatile float32_t Ts;
    uint16 CMP_delta;
    OVER_MOD_STATE over_mod_e = LINEAR;

    /* calculate Ts = 2*PRD*/
    Ts = 2 * v->PRD_pwm;

    /* calculate X, Y, Z*/
    v->X = (SQRT3 * Ts / v->Udc) * v->Ubeta; // X=sqrt3*Ts*Ubeta/Udc
    v->Y = (SQRT3 * Ts / v->Udc) * (0.8660254f * v->Ualpha + 0.5f * v->Ubeta);
    v->Z = (SQRT3 * Ts / v->Udc) * (-0.8660254f * v->Ualpha + 0.5f * v->Ubeta);

    /* calculate the sector Nr E(S)*/
    //1-3-2-6-5-4
    //
      v->Sector = 0; // initialize sector to 0 in each calculation
      v->Sector = (v->X > 0) ? (v->Sector + 1) : v->Sector;
      v->Sector = (v->Y < 0) ? (v->Sector + 4) : v->Sector;
      v->Sector = (v->Z < 0) ? (v->Sector + 2) : v->Sector;

    switch (v->Sector) {
        case 1:  //sector_S = 2
        	v->t1 = v->Z;
        	v->t2 = v->Y;
            break;
        case 2: //sector_S = 6
        	v->t1 = v->Y;
        	v->t2 = -v->X;
            break;
        case 3:  //sector_S = 1
        	v->t1 = -v->Z;
        	v->t2 = v->X;
            break;
        case 4:  //sector_S = 4
        	v->t1 = -v->X;
        	v->t2 = v->Z;
            break;
        case 5:  //sector_S = 3
        	v->t1 = v->X;
        	v->t2 = -v->Y;
            break;
        case 6:  //sector_S = 5
        	v->t1 = -v->Y;
        	v->t2 = -v->Z;
            break;
        default:  //sector_S = 0
        	v->t1 = 0;
        	v->t2 = 0;
    }

    /* calculate k in different DPWM mode*/
    switch (mode) {
        case SVPWM:
            v->k = 0.5f;
            break;
        case VECT_DPWM0:
            if ((v->Sector == 3) || (v->Sector == 5) || (v->Sector == 6))
                v->k = 0;
            else
                v->k = 1;
            break;
        case VECT_DPWM1:
            if ((v->Sector == 1) || (v->Sector == 6)) {
                if (v->Ualpha >= 0)
                    v->k = 0;
                else
                    v->k = 1;
            } else if ((v->Sector == 3) || (v->Sector == 4))
                if (v->Ualpha >= SQRT3 * v->Ubeta)
                    v->k = 1;
                else
                    v->k = 0;
            else if (-v->Ualpha >= SQRT3 * v->Ubeta)
                v->k = 0;
            else
                v->k = 1;
            break;
        case VECT_DPWM2:
            if ((v->Sector == 3) || (v->Sector == 5) || (v->Sector == 6))
                v->k = 1;
            else
                v->k = 0;
            break;
        case VECT_DPWM3:
            if ((v->Sector == 1) || (v->Sector == 6)) {
                if (v->Ualpha >= 0)
                    v->k = 1;
                else
                    v->k = 0;
            } else if ((v->Sector == 3) || (v->Sector == 4))
                if (v->Ualpha >= SQRT3 * v->Ubeta)
                    v->k = 0;
                else
                    v->k = 1;
            else if (-v->Ualpha >= SQRT3 * v->Ubeta)
                v->k = 1;
            else
                v->k = 0;
            break;
        case VECT_DPWMMAX:
            v->k = 1;
            break;
        case VECT_DPWMMIN:
            v->k = 0;
            break;
    }

	/* over modulation control */
	// Ualpha��Ubeta�ķ�ֵ��Ҫ����Udc/1.732���Բ��Թ�����
	//
	#if (OVERMOD_EN == 1)
    t_total = v->t1 + v->t2;
    if (t_total > (1.1547f * Ts)) {
        over_mod_e = OVER_MOD2; // over modulation in area 2
        t_delta = t_total - (1.1547f * Ts);
        if(v->t1 > v->t2) {
            if (v->t2 < t_delta) {
            	v->t2 = 0;
            	v->t1 = Ts;
            }
            else {
            	v->t1 = v->t1 + t_delta;
            	v->t2 = v->t2 - t_delta;
            }
        }
        else{
            if(v->t1 < t_delta) {
            	v->t1 = 0;
            	v->t2 = Ts;
            }
            else{
            	v->t1 = v->t1 - t_delta;
            	v->t2 = v->t2 + t_delta;
            }
        }
    }
    else if((t_total > Ts) && (t_total <= (1.1547f * Ts))) {
        over_mod_e = OVER_MOD1; // over modulation in area 1
        v->t1 = (v->t1 * Ts) / t_total;
        v->t2 = (v->t2 * Ts) / t_total;
    }
	else {
	over_mod_e = LINEAR;
	//	t1 = t1;
	//	t2 = t2;
	}
	#endif

    /* calculate ta, tb, tc*/
    v->Ta = 0.5f * (1-v->k) * (Ts - v->t1 - v->t2);
    v->Tb = v->Ta + 0.5f * v->t1;
    v->Tc = v->Tb + 0.5f * v->t2;

    /* CMPx, CMPy, CMPZ remap */
    CMPx = (uint16)(v->Ta);
    CMPy = (uint16)(v->Tb);
    CMPz = (uint16)(v->Tc);

    /* remap CMPx, CMPy, CMPz to CAMPa, CMPb, CMPc */
    switch (v->Sector) {
        case 1: // sector N=2
            v->CMPa = CMPy;
            v->CMPb = CMPx;
            v->CMPc = CMPz;
            break;
        case 2: // sector N=6
            v->CMPa = CMPx;
            v->CMPb = CMPz;
            v->CMPc = CMPy;
            break;
        case 3: // sector N=1
            v->CMPa = CMPx;
            v->CMPb = CMPy;
            v->CMPc = CMPz;
            break;
        case 4: // sector N=4
            v->CMPa = CMPz;
            v->CMPb = CMPy;
            v->CMPc = CMPx;
            break;
        case 5: // sector N=3
            v->CMPa = CMPz;
            v->CMPb = CMPx;
            v->CMPc = CMPy;
            break;
        case 6: // sector N=5
            v->CMPa = CMPy;
            v->CMPb = CMPz;
            v->CMPc = CMPx;
            break;
        default: //50% pwm
            v->CMPa = (uint16)(v->PRD_pwm * 0.5f);
            v->CMPb = (uint16)(v->PRD_pwm * 0.5f);
            v->CMPc = (uint16)(v->PRD_pwm * 0.5f);
    }

        /* inverse phase process */
	#if (PHASE_INVER_EN == 1)
    v->CMPa = v->PRD_pwm - v->CMPa;
    v->CMPb = v->PRD_pwm - v->CMPb;
    v->CMPc = v->PRD_pwm - v->CMPc;
	#endif

    /* narrow PWM */
	#if (NARROR_PWM_EN == 1)
    CMP_delta = (uint16)(v->PRD_pwm * NARROR_PWM_VALUE);

    if (v->CMPa < CMP_delta)
        v->CMPa = 0;
    else if (v->CMPa > (v->PRD_pwm - CMP_delta))
        v->CMPa = v->PRD_pwm;

    if (v->CMPb < CMP_delta)
        v->CMPb = 0;
    else if (v->CMPb > (v->PRD_pwm - CMP_delta))
        v->CMPb = v->PRD_pwm;

    if (v->CMPc < CMP_delta)
        v->CMPc = 0;
    else if (v->CMPc > (v->PRD_pwm - CMP_delta))
        v->CMPc = v->PRD_pwm;
	#endif
}

/**
 * \brief   PWM Generator Function Definition
 *
 */
static inline void
PWM_zero_run_ref(PWM_ZERO_Obj *v, PWM_ZERO_MODE mode) {
    float32_t Ua, Ub, Uc;
    float32_t Vax, Vbx, Vcx;
    float32_t max, min, Vmax, Vmin;
    uint16 CMP_delta;
    volatile float32_t Ts;

    /* calculate Ts = PRD */
    Ts = v->PRD_pwm;

    /* iClark transform */
    Ua = v->Ualpha;
    Ub = 0.8660254f * v->Ubeta - 0.5f * v->Ualpha;
    Uc = -0.8660254f * v->Ubeta - 0.5f * v->Ualpha;

    /* calculate the Uzero*/
    Vax = Ua - Ub;
    Vbx = Ub - Uc;
    Vcx = Uc - Ua;

    /* find the max and min value to X, Y , Z*/
    max = fmaxf(fmaxf(Ua, Ub), Uc);
    min = fminf(fminf(Ua, Ub), Uc);

    /* find the max value to max, Vax, Vbx, Vcx*/
    Vmax = fmaxf(fmaxf(Vax, Vbx), Vcx);

    /* find the min value to min, Vax, Vbx, Vcx*/
    Vmin = fminf(fminf(Vax, Vbx), Vcx);

    switch (mode) {
        case SPWM:
            v->Uzero = 0;
            break;

        case ZERO_DPWMMAX:
        	v->Uzero = (0.5773502f * v->Udc) - max;
            break;

        case ZERO_DPWMMIN:
        	v->Uzero = -(0.5773502f * v->Udc) - min;
            break;

        case ZERO_DPWM0:
            if (v->Speed >= 0) {
                if (fabsf(Vmax) >= fabsf(Vmin))
                	v->Uzero = (0.5773502f * v->Udc) - max;
                else
                	v->Uzero = -(0.5773502f * v->Udc) - min;
            }
            else {
                if (fabsf(Vmax) > fabsf(Vmin))
                	v->Uzero = -(0.5773502f * v->Udc) - min;
                else
                	v->Uzero = (0.5773502f * v->Udc) - max;
            }
            break;

        case ZERO_DPWM1:
            if (fabsf(max) >= fabsf(min))
            	v->Uzero = (0.5773502f * v->Udc) - max;
            else
            	v->Uzero = -(0.5773502f * v->Udc) - min;
            break;

        case ZERO_DPWM2:
            if (v->Speed > 0) {
                if (fabsf(Vmax) >= fabsf(Vmin))
                	v->Uzero = -(0.5773502f * v->Udc) - min;
                else
                	v->Uzero = (0.5773502f * v->Udc) - max;
            }
            else {
                if (fabsf(Vmax) >= fabsf(Vmin))
                	v->Uzero = (0.5773502f * v->Udc) - max;
                else
                	v->Uzero = -(0.5773502f * v->Udc) - min;
            }
            break;

        case ZERO_DPWM3:
            if (fabsf(max) >= fabsf(min))
            	v->Uzero = -(0.5773502f * v->Udc) - min;
            else
            	v->Uzero = (0.5773502f * v->Udc) - max;
            break;

        case SYPWM:
        	v->Uzero = -0.5f * (0.5773502f * v->Udc) * (min + max);
            break;
    }

    /* calculate Ta, Tb, Tc */
    v->Ta = Ts * (0.5f - (Ua + v->Uzero) / v->Udc);
    v->Tb = Ts * (0.5f - (Ub + v->Uzero) / v->Udc);
    v->Tc = Ts * (0.5f - (Uc + v->Uzero) / v->Udc);

    /* Clamp the Ta, Tb, Tc */
    v->Ta = fmaxf(fminf(v->Ta, Ts), 0);
    v->Tb = fmaxf(fminf(v->Tb, Ts), 0);
    v->Tc = fmaxf(fminf(v->Tc, Ts), 0);

    /* CMP remap, PRD/Ts is the slope factor for Ta&Tb&Tc converter to CMP*/
    v->CMPa = (uint16)(v->Ta);
    v->CMPb = (uint16)(v->Tb);
    v->CMPc = (uint16)(v->Tc);

	/* phase converted*/
	//
	#if (PHASE_INVER_EN == 1)
		v->CMPa = v->PRD_pwm - v->CMPa;
		v->CMPb = v->PRD_pwm - v->CMPb;
		v->CMPc = v->PRD_pwm - v->CMPc;
	#endif

	/* narrow PWM */
	#if (NARROR_PWM_EN == 1)
		CMP_delta = (uint16)(v->PRD_pwm * NARROR_PWM_VALUE);

		if (v->CMPa < CMP_delta)
			v->CMPa = 0;
		else if (v->CMPa > (v->PRD_pwm - CMP_delta))
			v->CMPa = v->PRD_pwm;

		if (v->CMPb < CMP_delta)
			v->CMPb = 0;
		else if (v->CMPb > (v->PRD_pwm - CMP_delta))
			v->CMPb = v->PRD_pwm;

		if (v->CMPc < CMP_delta)
			v->CMPc = 0;
		else if (v->CMPc > (v->PRD_pwm - CMP_delta))
			v->CMPc = v->PRD_pwm;
#endif
}


/**
 * \brief   PWM 3p3l Vector Generator Function Definition
 */
static inline
void SVPWM_3P3L_run_ref(SVPWM_3P3L_Obj *v, NVOLT_CONTROL_TYPE type) {
    float32_t X, Y, Z;
    float32_t tx, ty, tz;
    float32_t tx_c, ty_c, tz_c;
    float32_t vref, theta;
//    float32_t X1, X11, Y1, Y11, Z1, Z11;
//    float32_t X2, Y2, Z2;
    uint16 sector_E = 0, K;
    uint16 CMPx, CMPy, CMPz;
    uint16 CMPa, CMPb, CMPc, CMP_delta;
    float32_t nuo, neutr_volt_e, nuo_Kp, nuo_Ki, nuo_Ki_last;
    register float32_t Ts;
    float32_t neutr_volt_t;
    OVER_MOD_STATE over_mod_e = LINEAR;

    /*
     * vBus calculation
     */
    v->Udc = v->Udc1 + v->Udc2;

    /*
     * vBUS Over-Voltage Protection
     */
    if(fabsf(v->Udc_ref - v->Udc)>(0.1f * v->Udc_ref))
    {
        v->CMPa1 = (uint16)(v->PRD_pwm * 0.5f);
        v->CMPa2 = (uint16)(v->PRD_pwm * 0.5f);
        v->CMPb1 = (uint16)(v->PRD_pwm * 0.5f);
        v->CMPb2 = (uint16)(v->PRD_pwm * 0.5f);
        v->CMPc1 = (uint16)(v->PRD_pwm * 0.5f);
        v->CMPc2 = (uint16)(v->PRD_pwm * 0.5f);
    }
    else
    {
        Ts = 2 * v->PRD_pwm;

    /* calculate X, Y, Z*/
    //
    X = v->Ubeta;
    Y = v->Ualpha - 0.5773502f * v->Ubeta;
    Z = -v->Ualpha - 0.5773502f * v->Ubeta;

    /* calculate the sector Nr E(S)*/
    //
    sector_E = 0;
    sector_E = (X > 0) ? (sector_E + 1) : sector_E;
    sector_E = (Y > 0) ? (sector_E + 2) : sector_E;
    sector_E = (Z > 0) ? (sector_E + 4) : sector_E;

    /* calculate the X1, X11, Y1, Y11, Z1, Z11 for small sector n detection*/
    //
    v->X1 = 0.33333333f * v->Udc - (v->Ualpha + 0.57735027f * v->Ubeta);
    v->X11 = 0.33333333f * v->Udc + (v->Ualpha + 0.57735027f * v->Ubeta);
    v->Y1 = -0.33333333f * v->Udc + (v->Ualpha - 0.57735027f * v->Ubeta);
    v->Y11 = -0.33333333f * v->Udc - (v->Ualpha - 0.57735027f * v->Ubeta);
    v->Z1 = -0.28867513f * v->Udc + v->Ubeta;
    v->Z11 = -0.28867513f * v->Udc - v->Ubeta;

    /*
     * calculate small sector n to 6*big sector N
     */
    v->Sector_n = 0;
    switch (sector_E) {
        case 1:
            v->Sector_N = 2;
            if(v->Z1 < 0)
                v->Sector_n = 1;
            else if(v->X1 < 0)
                v->Sector_n = 2;
            else if(v->Y1 >= 0)
                v->Sector_n = 4;
            else
                v->Sector_n = 3;

            break;
        case 2:
            v->Sector_N = 6;
            if(v->Y1 < 0)
                v->Sector_n = 1;
            else if(v->Z11 >= 0)
                v->Sector_n = 2;
            else if(v->X1 < 0)
                v->Sector_n = 4;
            else
                v->Sector_n = 3;

            break;
        case 3:
            v->Sector_N = 1;
            if(v->X1 >= 0)
                v->Sector_n = 1;
            else if(v->Y1 >= 0)
                v->Sector_n = 2;
            else if(v->Z1 >= 0)
                v->Sector_n = 4;
            else
                v->Sector_n = 3;

            break;
        case 4:
            v->Sector_N = 4;
            if(v->Y11 >= 0)
                v->Sector_n = 2;
            else if(v->X11 >= 0 )
                v->Sector_n = 1;
            else if(v->Z11 >= 0)
                v->Sector_n = 4;
            else
                v->Sector_n = 3;
            break;
        case 5:
            v->Sector_N = 3;
            if(v->Y11 < 0)
                v->Sector_n = 1;
            else if(v->X11  < 0 )
                v->Sector_n = 4;
            else if(v->Z1 >= 0)
                v->Sector_n = 2;
            else
                v->Sector_n = 3;
            break;
        case 6:
            v->Sector_N = 5;
            if(v->Z11 < 0)
                v->Sector_n = 1;
            else if(v->X11  < 0 )
                v->Sector_n = 2;
            else if(v->Y11 >= 0)
                v->Sector_n = 4;
            else
                v->Sector_n = 3;
            break;
        default:
            v->Sector_N = 0;
            v->Sector_n = 0;
            break;
    }

    /* calculate X2, Y2, Z2 */
    v->X2 = (3.4641f * Ts * v->Ubeta) / v->Udc;
    v->Y2 = ((1.73205f * Ts * v->Ubeta) + (3 * Ts * v->Ualpha)) / v->Udc;
    v->Z2 = (-(1.73205f * Ts * v->Ubeta) + (3 * Ts * v->Ualpha)) / v->Udc;

    /* calculate t1, t2 */
    switch (v->Sector_N) {
        case 1:
            if (v->Sector_n == 1) {
            	v->t1 = v->X2;
            	v->t2 = Ts - v->Y2;
            } else if (v->Sector_n == 2) {
            	v->t1 = v->Z2 - Ts;
            	v->t2 = v->X2;
            } else if (v->Sector_n == 3) {
            	v->t1 = Ts - v->Z2;
            	v->t2 = v->Y2 - Ts;
            } else if (v->Sector_n == 4) {
            	v->t1 = v->Z2;
            	v->t2 = v->X2 - Ts;
            }
            break;
        case 2:
            if (v->Sector_n == 1) {
            	v->t1 = Ts - v->X2;
            	v->t2 = -v->Z2;
            } else if (v->Sector_n == 2) {
            	v->t1 = -v->Z2;
            	v->t2 = -Ts + v->Y2;
            } else if (v->Sector_n == 3) {
            	v->t1 = -Ts + v->X2;
            	v->t2 = Ts - v->Y2;
            } else if (v->Sector_n == 4) {
            	v->t1 = -Ts - v->Z2;
            	v->t2 = v->Y2;
            }
            break;
        case 3:
            if (v->Sector_n == 1) {
            	v->t1 = -v->Y2;
            	v->t2 = Ts + v->Z2;
            } else if (v->Sector_n == 2) {
            	v->t1 = v->X2 - Ts;
            	v->t2 = -v->Y2;
            } else if (v->Sector_n == 3) {
            	v->t1 = Ts - v->X2;
            	v->t2 = -v->Z2 - Ts;
            } else if (v->Sector_n == 4) {
            	v->t1 = v->X2;
            	v->t2 = -v->Y2 - Ts;
            }
            break;
        case 4:
            if (v->Sector_n == 1) {
            	v->t1 = Ts + v->Y2;
            	v->t2 = -v->X2;
            } else if (v->Sector_n == 2) {
            	v->t1 = -v->X2;
            	v->t2 = -v->Z2 - Ts;
            } else if (v->Sector_n == 3) {
            	v->t1 = -Ts - v->Y2;
            	v->t2 = v->Z2 + Ts;
            } else if (v->Sector_n == 4) {
            	v->t1 = -Ts - v->X2;
            	v->t2 = -v->Z2;
            }
            break;
        case 5:
            if (v->Sector_n == 1) {
            	v->t1 = v->Z2;
            	v->t2 = Ts + v->X2;
            } else if (v->Sector_n == 2) {
            	v->t1 = -v->Y2 - Ts;
            	v->t2 = v->Z2;
            } else if (v->Sector_n == 3) {
            	v->t1 = Ts + v->Y2;
            	v->t2 = -v->X2 - Ts;
            } else if (v->Sector_n == 4) {
            	v->t1 = -v->Y2;
            	v->t2 = v->Z2 - Ts;
            }
            break;
        case 6:
            if (v->Sector_n == 1) {
            	v->t1 = Ts - v->Z2;
            	v->t2 = v->Y2;
            } else if (v->Sector_n == 2) {
            	v->t1 = v->Y2;
            	v->t2 = -Ts - v->X2;
            } else if (v->Sector_n == 3) {
            	v->t1 = -Ts + v->Z2;
            	v->t2 = Ts + v->X2;
            } else if (v->Sector_n == 4) {
            	v->t1 = -Ts + v->Y2;
            	v->t2 = -v->X2;
            }
            break;
        default:
        	v->t1 = Ts * 0.5f;
        	v->t2 = Ts * 0.5f;
            break;
    }

	/*
	 * over-modulation control
	 */
	#if(OVERMOD_EN == 1)
		/*over modulation */
		vref = fminf(((0.5f * v->Udc) / ONE_PI), sqrtf(v->Ualpha * v->Ualpha + v->Ubeta * v->Ubeta));
		theta = atan2f(v->Ubeta, v->Ualpha);

		if(theta <0) {
			theta = theta + 2 * ONE_PI;
		}

		while(theta > (ONE_PI / 3)) {
			theta = theta - ONE_PI/3;
		}

		//
		/* over modulation with no compensaton*/
		//
		if((v->t1 + v->t2) < Ts) {
			over_mod_e = LINEAR;
//			v->t1 = v->t1;
//			v->t2 = v->t2;
		}
		else {
			over_mod_e = OVER_MOD1;

			if(vref > (0.6666666f * v->Udc)) {
				if((theta < (ONE_PI / 6)) && (theta > 0)) {
					v->t1 = Ts;
					v->t2 = 0;
				}
				else {
					v->t1 = 0;
					v->t2 = Ts;
				}
			}
			else {
				v->t1 = ((SQRT3 - tanf(theta)) * Ts) / (SQRT3 + tanf(theta));
				v->t2 = Ts - v->t1;
			}
		}

	#endif

    /* calculate tx, ty, tz*/
    v->Ta = 0.25f * (Ts - v->t1 - v->t2);
    v->Tb = v->Ta + 0.5f * v->t1;
    v->Tc = v->Tb + 0.5f * v->t2;

    /* neutral voltage balance control*/
    //
    // hysterisis delay control
    //
    if (type == P_CLOOP) {
        neutr_volt_e = v->Udc2 - 0.5f * v->Udc;

        if (fabsf(neutr_volt_e)<(neutr_volt_e * 0.01f))
            nuo = 0;
        else
            nuo = v->coef;

       //偏置计算 并加限幅
        neutr_volt_t  = nuo * neutr_volt_e;
        if (neutr_volt_t > 0.25f)
        neutr_volt_t = 0.25f;
        else if (neutr_volt_t <- 0.25f)
        neutr_volt_t = -0.25f;

        // correct tx, ty, tz
        v->Ta = v->Ta * (1 + neutr_volt_t);
        v->Tb = v->Ta + 0.5f * v->t1;
        v->Tc = v->Tb + 0.5f * v->t2;
    }

    // PI close loop control
    //
    else if (type == PI_CLOOP) {
        neutr_volt_e = v->Udc2 - 0.5f * v->Udc;

        if (fabsf(neutr_volt_e)<(neutr_volt_e * 0.01f)) {
            nuo_Ki = 0;
        }

        nuo_Kp = v->coef * neutr_volt_e; //计算P控制部分
        nuo_Ki = nuo_Ki_last + (v->coef * neutr_volt_e * 0.1f); //计算I控制部分
        nuo = nuo_Kp + nuo_Ki;
        nuo_Ki_last = nuo_Ki;

        //PI控制I部分限幅
        //
        if (nuo_Ki_last > 0.25f * KI_FACT)
            nuo_Ki_last = 0.25f * KI_FACT;
        else if (nuo_Ki_last < -0.25f * KI_FACT)
            nuo_Ki_last = -0.25f * KI_FACT;

        //限制PI控制幅度
        //
        if (nuo > 0.25f)
            nuo = 0.25f;
        else if (nuo <- 0.25f)
            nuo = -0.25f;

        v->Ta = v->Ta * (1 + nuo);
        v->Tb = v->Ta + 0.5f * v->t1;
        v->Tc = v->Tb + 0.5f * v->t2;
    }

    /* CMPx, CMPy, CMPZ remap */
    //
    CMPx = (uint16)(v->Ta);
    CMPy = (uint16)(v->Tb);
    CMPz = (uint16)(v->Tc);

    /* remap CMPx, CMPy, CMPz to CAMPa, CMPb, CMPc */
    //
    if (((v->Sector_N == 1) && (v->Sector_n == 1)) || ((v->Sector_N == 2) && (v->Sector_n == 3)) ||
        ((v->Sector_N == 3) && (v->Sector_n == 2)) || ((v->Sector_N == 3) && (v->Sector_n == 4))) {
        CMPa = CMPz;
        CMPb = CMPx;
        CMPc = CMPy;
    } else if (((v->Sector_N == 1) && (v->Sector_n == 2)) || ((v->Sector_N == 1) && (v->Sector_n == 4)) ||
               ((v->Sector_N == 5) && (v->Sector_n == 1)) || ((v->Sector_N == 6) && (v->Sector_n == 3))) {
        CMPa = CMPx;
        CMPb = CMPy;
        CMPc = CMPz;
    } else if (((v->Sector_N == 1) && (v->Sector_n == 3)) || ((v->Sector_N == 2) && (v->Sector_n == 2)) ||
               ((v->Sector_N == 2) && (v->Sector_n == 4)) || ((v->Sector_N == 6) && (v->Sector_n == 1))) {
        CMPa = CMPy;
        CMPb = CMPx;
        CMPc = CMPz;
    } else if (((v->Sector_N == 2) && (v->Sector_n == 1)) || ((v->Sector_N == 3) && (v->Sector_n == 3)) ||
               ((v->Sector_N == 4) && (v->Sector_n == 2)) || ((v->Sector_N == 4) && (v->Sector_n == 4))) {
        CMPa = CMPz;
        CMPb = CMPy;
        CMPc = CMPx;
    } else if (((v->Sector_N == 3) && (v->Sector_n == 1)) || ((v->Sector_N == 4) && (v->Sector_n == 3)) ||
               ((v->Sector_N == 5) && (v->Sector_n == 2)) || ((v->Sector_N == 5) && (v->Sector_n == 4))) {
        CMPa = CMPy;
        CMPb = CMPz;
        CMPc = CMPx;
    } else if (((v->Sector_N == 4) && (v->Sector_n == 1)) || ((v->Sector_N == 5) && (v->Sector_n == 3)) ||
               ((v->Sector_N == 6) && (v->Sector_n == 2)) || ((v->Sector_N == 6) && (v->Sector_n == 4))) {
        CMPa = CMPx;
        CMPb = CMPz;
        CMPc = CMPy;
    }
    else {
        CMPa = v->PRD_pwm / 2;
        CMPb = v->PRD_pwm / 2;
        CMPc = v->PRD_pwm / 2;
    }

    /* inverse phase process */
	#if (PHASE_INVER_EN == 1)
		CMPa = v->PRD_pwm - CMPa;
		CMPb = v->PRD_pwm - CMPb;
		CMPc = v->PRD_pwm - CMPc;
	#endif

    /* narrow PWM */
	#if (NARROR_PWM_EN == 1)
		CMP_delta = (uint16)(v->PRD_pwm * NARROR_PWM_VALUE);

		if (CMPa < CMP_delta)
			CMPa = 0;
		else if (CMPa > (v->PRD_pwm - CMP_delta))
			CMPa = v->PRD_pwm;

		if (CMPb < CMP_delta)
			CMPb = 0;
		else if (CMPb > (v->PRD_pwm - CMP_delta))
			CMPb = v->PRD_pwm;

		if (CMPc < CMP_delta)
			CMPc = 0;
		else if (CMPc > (v->PRD_pwm - CMP_delta))
			CMPc = v->PRD_pwm;
	#endif

	//
	//
	K = 4 * (v->Sector_N - 1) + v->Sector_n;
	if ((K >= 8)&&(K <= 19)) {
		v->CMPa1 = v->PRD_pwm;
		v->CMPa2 = CMPa;
	}
	else {
		v->CMPa1 = CMPa;
		v->CMPa2 = 0;
	}

	if ((K >= 4)&&(K <= 15)) {
		v->CMPb1 = CMPb;
		v->CMPb2 = 0;
	}
	else {
		v->CMPb1 =  v->PRD_pwm;
		v->CMPb2 =  CMPb;
	}

	if ( (K >= 12)&&(K <= 23)) {
		v->CMPc1 = CMPc;
		v->CMPc2 = 0;
	}
	else {
		v->CMPc1 = v->PRD_pwm;
		v->CMPc2 = CMPc;
	}

    }
}

#ifdef __cplusplus
}
#endif

#endif /* GJ320_PWM_H_*/
