/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *   @file    test.h
 *   @brief
 *
 */

#ifndef DSP_NICE_TEST_H_
#define DSP_NICE_TEST_H_

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "gs32_transform.h"
#include "abc_dq0_neg.h"
#include "abc_dq0_pos.h"
#include "dq0_abc.h"
#include "driverlib.h"
#include "rampgen.h"
#include "stdint.h"
#include <math.h>
#include <float.h>
#include "IQMathLib.h"
#include "log.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define TEST_ERROR 0.01
#define ARRAY_SIZE 4096

enum {FLASH_LOC, RAM_LOC, ILM_LOC};

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */
typedef struct {
    float32_t d[ARRAY_SIZE];
    float32_t q[ARRAY_SIZE];
    float32_t z[ARRAY_SIZE];
} ABC_DQ_Error;

typedef struct {
    float32_t a[ARRAY_SIZE];
    float32_t b[ARRAY_SIZE];
    float32_t c[ARRAY_SIZE];
} DQ_ABC_Error;

#define ABC_DQ_Error_Default \
    { 0, 0, 0 }
#define DQ_ABC_Error_Default \
    { 0, 0, 0 }


/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

static uint32_t cost_time_error = 4;

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */

int CalibrateMyTimer(){
	const unsigned int n = 100;
	uint32_t cost_time_total = 0;

	for(int i = 0; i < n; i++){
		volatile uint32_t begin_time = CPU_getCycleCnt();
		volatile uint32_t end_time = CPU_getCycleCnt();

		cost_time_total += end_time - begin_time;
	}
	cost_time_error = cost_time_total / n;

	return cost_time_error;
}
/**
 *ABC_DQ_POS test, z reference, w/o sine&cosine
 */
static inline void
ABC_DQ_POS_dsp_test(void) {
    register uint32_t Counter_Start = 0;
    register uint32_t Counter_Stop = 0;
    register uint32_t Counter_cycle = 0;
    register uint32_t Counter_Total = 0;
    register uint32_t Counter_Ave = 0;
    register uint32_t Counter_Error = 0;
    vuint32_t error_ctr = 0;
    float32_t angle_radians = 0;
    register float32_t input_sine_A, input_sine_B, input_sine_C;
    register float32_t sin_theta, cos_theta;
    volatile float32_t error_max_d, error_max_q, error_max_z;

    RAMPGEN input_rgen;
    ABC_DQ_POS dq0_pos;
    ABC_DQ0_POS dq0_pos_ref;
    static ABC_DQ_Error ABC_DQ_POS_Error = ABC_DQ_Error_Default;

    RAMPGEN_reset(&input_rgen);
    RAMPGEN_config(&input_rgen, 500, ARRAY_SIZE);
    ABC_DQ_POS_reset(&dq0_pos);
    ABC_DQ0_POS_reset(&dq0_pos_ref);

    for (uint32_t i = 0; i < ARRAY_SIZE; i++) {
        // Բ
        RAMPGEN_run(&input_rgen);
        angle_radians = input_rgen.out;
        input_sine_A = 600 * __sinpuf32(angle_radians) * 0.95f; // imbalance with factor 0.95
        sin_theta = __sinpuf32(angle_radians);
        cos_theta = __cospuf32(angle_radians);
        input_sine_B = 600 * __sinpuf32(angle_radians - 2.0f / 3.0f) * 0.93f; // imbalance with factor 0.93
        input_sine_C = 600 * __sinpuf32(angle_radians - 4.0f / 3.0f) * 1.02f; // imbalance with factor 1.02

        /*
         * calculate the cycles
         */
        Counter_Start = CPU_getCycleCnt();
        ABC_DQ_POS_run(&dq0_pos,
                             input_sine_A, input_sine_B, input_sine_C,
                             sin_theta, cos_theta);
        Counter_Stop = CPU_getCycleCnt();
        Counter_cycle = Counter_Stop- Counter_Start - cost_time_error;
        Counter_Total += Counter_cycle;

        /*
         * calculate the ref with C code
         */
        ABC_DQ0_POS_run(&dq0_pos_ref,
                        input_sine_A, input_sine_B, input_sine_C,
                        sin_theta, cos_theta);

        ABC_DQ_POS_Error.d[i] = __fabs(dq0_pos_ref.d - dq0_pos.d);
        ABC_DQ_POS_Error.q[i] = __fabs(dq0_pos_ref.q - dq0_pos.q);
        ABC_DQ_POS_Error.z[i] = __fabs(dq0_pos_ref.z - dq0_pos.z);
        error_max_d = (error_max_d < ABC_DQ_POS_Error.d[i]) ? ABC_DQ_POS_Error.d[i] : error_max_d;
        error_max_q = (error_max_q < ABC_DQ_POS_Error.q[i]) ? ABC_DQ_POS_Error.q[i] : error_max_q;
        error_max_z = (error_max_z < ABC_DQ_POS_Error.z[i]) ? ABC_DQ_POS_Error.z[i] : error_max_z;
        error_ctr = ((ABC_DQ_POS_Error.d[i] > TEST_ERROR) || (ABC_DQ_POS_Error.q[i] > TEST_ERROR) ||  \
                     (ABC_DQ_POS_Error.z[i] > TEST_ERROR)) ? (error_ctr + 1) : error_ctr;
    }
    Counter_Ave = Counter_Total / ARRAY_SIZE;

    if (error_ctr == 0) {
        log_info("\r\nABC_DQ_POS_dsp test Pass! \n error_max_d is : %.8f \n error_max_q is : %.8f \n error_max_z is : %.8f \n "
               "Counter_Ave %d \n", error_max_d, error_max_q, error_max_z, Counter_Ave);
    } else
    	log_info("\r\nABC_DQ_POS_dsp test Fail! \n error_max_d is : %.8f \n error_max_q is : %.8f \n error_max_z is : %.8f \n "
    	               "Counter_Ave %d \n", error_max_d, error_max_q, error_max_z, Counter_Ave);
}


/**
 *ABC_DQ1_POS_run test, z reference, with sine&cosine
 */

static inline void
ABC_DQ1_POS_dsp_test(void) {
    register uint32_t Counter_Start = 0;
    register uint32_t Counter_Stop = 0;
    register uint32_t Counter_cycle = 0;
    register uint32_t Counter_Total = 0;
    register uint32_t Counter_Ave = 0;
    register uint32_t Counter_Error = 0;
    vuint32_t error_ctr = 0;
    float32_t angle_radians = 0;
    register float32_t input_sine_A, input_sine_B, input_sine_C;
    register float32_t sin_theta, cos_theta;
    float32_t error_max_d, error_max_q, error_max_z;

    RAMPGEN input_rgen;
    ABC_DQ_POS dq0_pos;
    ABC_DQ0_POS dq0_pos_ref;
    static ABC_DQ_Error ABC_DQ_POS_Error = ABC_DQ_Error_Default;

    RAMPGEN_reset(&input_rgen);
    RAMPGEN_config(&input_rgen, 500, ARRAY_SIZE);
    ABC_DQ_POS_reset(&dq0_pos);
    ABC_DQ0_POS_reset(&dq0_pos_ref);

    for (uint32_t i = 0; i < ARRAY_SIZE; i++) {
        // Բ
        RAMPGEN_run(&input_rgen);
        angle_radians = input_rgen.out;
        input_sine_A = 600 * __sinpuf32(angle_radians) * 0.95f; // imbalance with factor 0.95
        sin_theta = __sinpuf32(angle_radians);
        cos_theta = __cospuf32(angle_radians);
        input_sine_B = 600 * __sinpuf32(angle_radians - 2.0f / 3.0f) * 0.93f; // imbalance with factor 0.93
        input_sine_C = 600 * __sinpuf32(angle_radians - 4.0f / 3.0f) * 1.02f;// imbalance with factor 1.02

        /*
         * calculate the cycles
         */
        Counter_Start = CPU_getCycleCnt();
        ABC_DQ1_POS_run(&dq0_pos,
                             input_sine_A, input_sine_B, input_sine_C,
							 input_rgen.out);
        Counter_Stop = CPU_getCycleCnt();
        Counter_cycle = Counter_Stop- Counter_Start - cost_time_error;
        Counter_Total += Counter_cycle;

        /*
         * calculate the ref with C code
         */
        ABC_DQ0_POS_run(&dq0_pos_ref,
                        input_sine_A, input_sine_B, input_sine_C,
                        sin_theta, cos_theta);

        ABC_DQ_POS_Error.d[i] = __fabs(dq0_pos_ref.d - dq0_pos.d);
        ABC_DQ_POS_Error.q[i] = __fabs(dq0_pos_ref.q - dq0_pos.q);
        ABC_DQ_POS_Error.z[i] = __fabs(dq0_pos_ref.z - dq0_pos.z);
        error_max_d = (error_max_d < ABC_DQ_POS_Error.d[i]) ? ABC_DQ_POS_Error.d[i] : error_max_d;
        error_max_q = (error_max_q < ABC_DQ_POS_Error.q[i]) ? ABC_DQ_POS_Error.q[i] : error_max_q;
        error_max_z = (error_max_z < ABC_DQ_POS_Error.z[i]) ? ABC_DQ_POS_Error.z[i] : error_max_z;
        error_ctr = ((ABC_DQ_POS_Error.d[i] > TEST_ERROR) || (ABC_DQ_POS_Error.q[i] > TEST_ERROR) ||  \
                     (ABC_DQ_POS_Error.z[i] > TEST_ERROR)) ? (error_ctr + 1) : error_ctr;
    }
    Counter_Ave = Counter_Total / ARRAY_SIZE;

    if (error_ctr == 0) {
        log_info("\r\nABC_DQ1_POS_dsp test Pass! \n error_max_d is : %.8f \n error_max_q is : %.8f \n error_max_z is : %.8f \n "
               "Counter_Ave %d \n", error_max_d, error_max_q, error_max_z, Counter_Ave);
    } else
    	log_info("\r\nABC_DQ1_POS_dsp test Fail! \n error_max_d is : %.8f \n error_max_q is : %.8f \n error_max_z is : %.8f \n "
    	               "Counter_Ave %d \n", error_max_d, error_max_q, error_max_z, Counter_Ave);
}

/**
 *ABC_DQ0_run test, z=0
 */
static inline void
ABC_DQ0_dsp_test(void) {
    register uint32_t Counter_Start = 0;
    register uint32_t Counter_Stop = 0;
    register uint32_t Counter_cycle = 0;
    register uint32_t Counter_Total = 0;
    register uint32_t Counter_Ave = 0;
    register uint32_t Counter_Error = 0;
    vuint32_t error_ctr = 0;
    float32_t angle_radians = 0;
    register float32_t input_sine_A, input_sine_B, input_sine_C;
    register float32_t sin_theta, cos_theta;
    float32_t error_max_d, error_max_q, error_max_z;

    RAMPGEN input_rgen;
    ABC_DQ_POS dq0_pos;
    ABC_DQ0_POS dq0_pos_ref;
    static ABC_DQ_Error ABC_DQ_POS_Error = ABC_DQ_Error_Default;

    RAMPGEN_reset(&input_rgen);
    RAMPGEN_config(&input_rgen, 500, ARRAY_SIZE);
    ABC_DQ_POS_reset(&dq0_pos);
    ABC_DQ0_POS_reset(&dq0_pos_ref);

    for (uint32_t i = 0; i < ARRAY_SIZE; i++) {
        // Բ
        RAMPGEN_run(&input_rgen);

        angle_radians = input_rgen.out;
        input_sine_A = 600 * __sinpuf32(angle_radians); // imbalance with factor 0.95
        sin_theta = __sinpuf32(angle_radians);
        cos_theta = __cospuf32(angle_radians);
        input_sine_B = 600 * __sinpuf32(angle_radians - 2.0f / 3.0f); // imbalance with factor 0.93
//        input_sine_C = 600 * __sinpuf32(angle_radians - 4.0f / 3.0f); // imbalance with factor 1.02
        input_sine_C = -(input_sine_A+input_sine_B);
        /*
         * calculate the cycles
         */
        Counter_Start = CPU_getCycleCnt();
        ABC_DQ0_run(&dq0_pos,
                             input_sine_A, input_sine_B, input_sine_C,
							 sin_theta, cos_theta);
        Counter_Stop = CPU_getCycleCnt();
        Counter_cycle = Counter_Stop- Counter_Start - cost_time_error;
        Counter_Total += Counter_cycle;

        /*
         * calculate the ref with C code
         */
        ABC_DQ0_POS_run(&dq0_pos_ref,
                        input_sine_A, input_sine_B, input_sine_C,
                        sin_theta, cos_theta);

        ABC_DQ_POS_Error.d[i] = __fabs(dq0_pos_ref.d - dq0_pos.d);
        ABC_DQ_POS_Error.q[i] = __fabs(dq0_pos_ref.q - dq0_pos.q);
        ABC_DQ_POS_Error.z[i] = __fabs(dq0_pos_ref.z - dq0_pos.z);
        error_max_d = (error_max_d < ABC_DQ_POS_Error.d[i]) ? ABC_DQ_POS_Error.d[i] : error_max_d;
        error_max_q = (error_max_q < ABC_DQ_POS_Error.q[i]) ? ABC_DQ_POS_Error.q[i] : error_max_q;
        error_max_z = (error_max_z < ABC_DQ_POS_Error.z[i]) ? ABC_DQ_POS_Error.z[i] : error_max_z;
        error_ctr = ((ABC_DQ_POS_Error.d[i] > TEST_ERROR) || (ABC_DQ_POS_Error.q[i] > TEST_ERROR) ||  \
                     (ABC_DQ_POS_Error.z[i] > TEST_ERROR)) ? (error_ctr + 1) : error_ctr;
    }
    Counter_Ave = Counter_Total / ARRAY_SIZE;

    if (error_ctr == 0) {
        log_info("\r\nABC_DQ0_dsp test Pass! \n error_max_d is : %.8f \n error_max_q is : %.8f \n error_max_z is : %.8f \n "
               "Counter_Ave %d \n", error_max_d, error_max_q, error_max_z, Counter_Ave);
    } else
    	log_info("\r\nABC_DQ0_dsp test Fail! \n error_max_d is : %.8f \n error_max_q is : %.8f \n error_max_z is : %.8f \n "
    	               "Counter_Ave %d \n", error_max_d, error_max_q, error_max_z, Counter_Ave);
}

/**
 *ABC_DQ1_run test, z=0, with sine&cosine
 */
static inline void
ABC_DQ1_dsp_test(void) {
    register uint32_t Counter_Start = 0;
    register uint32_t Counter_Stop = 0;
    register uint32_t Counter_cycle = 0;
    register uint32_t Counter_Total = 0;
    register uint32_t Counter_Ave = 0;
    register uint32_t Counter_Error = 0;
    vuint32_t error_ctr = 0;
    float32_t angle_radians = 0;
    register float32_t input_sine_A, input_sine_B, input_sine_C;
    register float32_t sin_theta, cos_theta;
    float32_t error_max_d, error_max_q, error_max_z;

    RAMPGEN input_rgen;
    ABC_DQ_POS dq0_pos;
    ABC_DQ0_POS dq0_pos_ref;
    static ABC_DQ_Error ABC_DQ_POS_Error = ABC_DQ_Error_Default;

    RAMPGEN_reset(&input_rgen);
    RAMPGEN_config(&input_rgen, 500, ARRAY_SIZE);
    ABC_DQ_POS_reset(&dq0_pos);
    ABC_DQ0_POS_reset(&dq0_pos_ref);

    for (uint32_t i = 0; i < ARRAY_SIZE; i++) {
        // Բ
        RAMPGEN_run(&input_rgen);

        angle_radians = input_rgen.out;
        input_sine_A = 600 * __sinpuf32(angle_radians); // imbalance with factor 0.95
        sin_theta = __sinpuf32(angle_radians);
        cos_theta = __cospuf32(angle_radians);
        input_sine_B = 600 * __sinpuf32(angle_radians - 2.0f / 3.0f); // imbalance with factor 0.93
//        input_sine_C = 600 * __sinpuf32(angle_radians - 4.0f / 3.0f); // imbalance with factor 1.02
        input_sine_C = -(input_sine_A+input_sine_B);
        /*
         * calculate the cycles
         */
        Counter_Start = CPU_getCycleCnt();
        ABC_DQ1_run(&dq0_pos,
                             input_sine_A, input_sine_B, input_sine_C,
							 input_rgen.out);
        Counter_Stop = CPU_getCycleCnt();
        Counter_cycle = Counter_Stop- Counter_Start - cost_time_error;
        Counter_Total += Counter_cycle;

        /*
         * calculate the ref with C code
         */
        ABC_DQ0_POS_run(&dq0_pos_ref,
                        input_sine_A, input_sine_B, input_sine_C,
                        sin_theta, cos_theta);

        ABC_DQ_POS_Error.d[i] = __fabs(dq0_pos_ref.d - dq0_pos.d);
        ABC_DQ_POS_Error.q[i] = __fabs(dq0_pos_ref.q - dq0_pos.q);
        ABC_DQ_POS_Error.z[i] = __fabs(dq0_pos_ref.z - dq0_pos.z);
        error_max_d = (error_max_d < ABC_DQ_POS_Error.d[i]) ? ABC_DQ_POS_Error.d[i] : error_max_d;
        error_max_q = (error_max_q < ABC_DQ_POS_Error.q[i]) ? ABC_DQ_POS_Error.q[i] : error_max_q;
        error_max_z = (error_max_z < ABC_DQ_POS_Error.z[i]) ? ABC_DQ_POS_Error.z[i] : error_max_z;
        error_ctr = ((ABC_DQ_POS_Error.d[i] > TEST_ERROR) || (ABC_DQ_POS_Error.q[i] > TEST_ERROR) ||  \
                     (ABC_DQ_POS_Error.z[i] > TEST_ERROR)) ? (error_ctr + 1) : error_ctr;
    }
    Counter_Ave = Counter_Total / ARRAY_SIZE;

    if (error_ctr == 0) {
        log_info("\r\nABC_DQ1_dsp test Pass! \n error_max_d is : %.8f \n error_max_q is : %.8f \n error_max_z is : %.8f \n "
               "Counter_Ave %d \n", error_max_d, error_max_q, error_max_z, Counter_Ave);
    } else
    	log_info("\r\nABC_DQ1_dsp test Fail! \n error_max_d is : %.8f \n error_max_q is : %.8f \n error_max_z is : %.8f \n "
    	               "Counter_Ave %d \n", error_max_d, error_max_q, error_max_z, Counter_Ave);
}

/**
 *ABC_DQ_NEG, with z reference, w/o sine&cosine
 */
static inline void
ABC_DQ_NEG_dsp_test(void) {
    register uint32_t Counter_Start = 0;
    register uint32_t Counter_Stop = 0;
    register uint32_t Counter_cycle = 0;
    register uint32_t Counter_Total = 0;
    register uint32_t Counter_Ave = 0;
    register uint32_t Counter_Error = 0;
    vuint32_t error_ctr = 0;
    register float32_t angle_radians = 0;
    register float32_t input_sine_A, input_sine_B, input_sine_C;
    register float32_t sin_theta, cos_theta;
    float32_t error_max_d, error_max_q, error_max_z;

    RAMPGEN input_rgen;
    ABC_DQ_NEG dq0_neg;
    ABC_DQ0_NEG dq0_neg_ref;
    static ABC_DQ_Error ABC_DQ_NEG_Error = ABC_DQ_Error_Default;

    RAMPGEN_reset(&input_rgen);
    RAMPGEN_config(&input_rgen, 500, ARRAY_SIZE);
    ABC_DQ_NEG_reset(&dq0_neg);
    ABC_DQ0_NEG_reset(&dq0_neg_ref);

    for (uint32_t i = 0; i < ARRAY_SIZE; i++) {
        // Բ
        RAMPGEN_run(&input_rgen);

        angle_radians = input_rgen.out;
        input_sine_A = 600 * __sinpuf32(angle_radians) * 0.95f; // imbalance with factor 0.95
        sin_theta = __sinpuf32(angle_radians);
        cos_theta = __cospuf32(angle_radians);
        input_sine_B = 600 * __sinpuf32(angle_radians - 2.0f / 3.0f) * 0.93f; // imbalance with factor 0.93
        input_sine_C = 600 * __sinpuf32(angle_radians - 4.0f / 3.0f) * 1.02f; // imbalance with factor 1.02

        /*
         * calculate the cycles
         */
        Counter_Start = CPU_getCycleCnt();
        ABC_DQ_NEG_run(&dq0_neg,
                             input_sine_A, input_sine_B, input_sine_C,
                             sin_theta, cos_theta);
        Counter_Stop = CPU_getCycleCnt();
        Counter_cycle = Counter_Stop- Counter_Start - cost_time_error;
        Counter_Total += Counter_cycle;

        /*
         * calculate the ref with C code
         */
        ABC_DQ0_NEG_run(&dq0_neg_ref,
                        input_sine_A, input_sine_B, input_sine_C,
                        sin_theta, cos_theta);

        ABC_DQ_NEG_Error.d[i] = __fabs(dq0_neg_ref.d - dq0_neg.d);
        ABC_DQ_NEG_Error.q[i] = __fabs(dq0_neg_ref.q - dq0_neg.q);
        ABC_DQ_NEG_Error.z[i] = __fabs(dq0_neg_ref.z - dq0_neg.z);
        error_max_d = (error_max_d < ABC_DQ_NEG_Error.d[i]) ? ABC_DQ_NEG_Error.d[i] : error_max_d;
        error_max_q = (error_max_q < ABC_DQ_NEG_Error.q[i]) ? ABC_DQ_NEG_Error.q[i] : error_max_q;
        error_max_z = (error_max_z < ABC_DQ_NEG_Error.z[i]) ? ABC_DQ_NEG_Error.z[i] : error_max_z;
        error_ctr = ((ABC_DQ_NEG_Error.d[i] > TEST_ERROR) || (ABC_DQ_NEG_Error.q[i] > TEST_ERROR) ||   \
                     (ABC_DQ_NEG_Error.z[i] > TEST_ERROR))                                             \
                        ? (error_ctr + 1) : error_ctr;
    }
    Counter_Ave = Counter_Total / ARRAY_SIZE;

    if (error_ctr == 0) {
    	log_info("\r\nABC_DQ_NEG_dsp test Pass!\n, error_max_d is : %.8f\n error_max_q is : %.8f\n error_max_z is : %.8f\n "
               "Counter_Ave %d\n",error_max_d, error_max_q, error_max_z, Counter_Ave);
    } else
    	log_info("\r\nABC_DQ_NEG_dsp test Fail!\n error_max_d is : %.8f\n error_max_q is : %.8f\n error_max_z is : %.8f\n "
    	               "Counter_Ave %d\n",error_max_d, error_max_q, error_max_z, Counter_Ave);
}

/**
 *ABC_DQ1_NEG, with z reference, w/ sine&cosine
 */
static inline void
ABC_DQ1_NEG_dsp_test(void) {
    register uint32_t Counter_Start = 0;
    register uint32_t Counter_Stop = 0;
    register uint32_t Counter_cycle = 0;
    register uint32_t Counter_Total = 0;
    register uint32_t Counter_Ave = 0;
    register uint32_t Counter_Error = 0;
    vuint32_t error_ctr = 0;
    register float32_t angle_radians = 0;
    register float32_t input_sine_A, input_sine_B, input_sine_C;
    register float32_t sin_theta, cos_theta;
    float32_t error_max_d, error_max_q, error_max_z;

    RAMPGEN input_rgen;
    ABC_DQ_NEG dq0_neg;
    ABC_DQ0_NEG dq0_neg_ref;
    static ABC_DQ_Error ABC_DQ_NEG_Error = ABC_DQ_Error_Default;

    RAMPGEN_reset(&input_rgen);
    RAMPGEN_config(&input_rgen, 500, ARRAY_SIZE);
    ABC_DQ_NEG_reset(&dq0_neg);
    ABC_DQ0_NEG_reset(&dq0_neg_ref);

    for (uint32_t i = 0; i < ARRAY_SIZE; i++) {
        // Բ
        RAMPGEN_run(&input_rgen);

        angle_radians = input_rgen.out;
        input_sine_A = 600 * __sinpuf32(angle_radians) * 0.95f; // imbalance with factor 0.95
        sin_theta = __sinpuf32(angle_radians);
        cos_theta = __cospuf32(angle_radians);
        input_sine_B = 600 * __sinpuf32(angle_radians - 2.0f / 3.0f) * 0.93f; // imbalance with factor 0.93
        input_sine_C = 600 * __sinpuf32(angle_radians - 4.0f / 3.0f) * 1.02f; // imbalance with factor 1.02

        /*
         * calculate the cycles
         */
        Counter_Start = CPU_getCycleCnt();
        ABC_DQ1_NEG_run(&dq0_neg,
                             input_sine_A, input_sine_B, input_sine_C,
							 input_rgen.out);
        Counter_Stop = CPU_getCycleCnt();
        Counter_cycle = Counter_Stop- Counter_Start - cost_time_error;
        Counter_Total += Counter_cycle;

        /*
         * calculate the ref with C code
         */
        ABC_DQ0_NEG_run(&dq0_neg_ref,
                        input_sine_A, input_sine_B, input_sine_C,
                        sin_theta, cos_theta);

        ABC_DQ_NEG_Error.d[i] = __fabs(dq0_neg_ref.d - dq0_neg.d);
        ABC_DQ_NEG_Error.q[i] = __fabs(dq0_neg_ref.q - dq0_neg.q);
        ABC_DQ_NEG_Error.z[i] = __fabs(dq0_neg_ref.z - dq0_neg.z);
        error_max_d = (error_max_d < ABC_DQ_NEG_Error.d[i]) ? ABC_DQ_NEG_Error.d[i] : error_max_d;
        error_max_q = (error_max_q < ABC_DQ_NEG_Error.q[i]) ? ABC_DQ_NEG_Error.q[i] : error_max_q;
        error_max_z = (error_max_z < ABC_DQ_NEG_Error.z[i]) ? ABC_DQ_NEG_Error.z[i] : error_max_z;
        error_ctr = ((ABC_DQ_NEG_Error.d[i] > TEST_ERROR) || (ABC_DQ_NEG_Error.q[i] > TEST_ERROR) ||   \
                     (ABC_DQ_NEG_Error.z[i] > TEST_ERROR))                                             \
                        ? (error_ctr + 1) : error_ctr;
    }
    Counter_Ave = Counter_Total / ARRAY_SIZE;

    if (error_ctr == 0) {
    	log_info("\r\nABC_DQ1_NEG_dsp test Pass!\n, error_max_d is : %.8f\n error_max_q is : %.8f\n error_max_z is : %.8f\n "
               "Counter_Ave %d\n",error_max_d, error_max_q, error_max_z, Counter_Ave);
    } else
    	log_info("\r\nABC_DQ1_NEG_dsp test Fail!\n error_max_d is : %.8f\n error_max_q is : %.8f\n error_max_z is : %.8f\n "
    	               "Counter_Ave %d\n",error_max_d, error_max_q, error_max_z, Counter_Ave);
}

/**
 *test dq_abc with z reference w/o sine&cosine
 *
 */
static inline void
DQ_ABC_dsp_Test() {
    register uint32_t Counter_Start = 0;
    register uint32_t Counter_Stop = 0;
    register uint32_t Counter_cycle = 0;
    register uint32_t Counter_Total = 0;
    register uint32_t Counter_Ave = 0;
    register uint32_t Counter_Error = 0;
    vuint32_t error_ctr = 0;
    register float32_t angle_radians = 0;
    register float32_t input_sine_A, input_sine_B, input_sine_C;
    register float32_t sin_theta, cos_theta;
    float32_t error_max_a, error_max_b, error_max_c;

    RAMPGEN input_rgen;
    ABC_DQ0_POS dq0_pos_input;

    DQ_ABC dq_abc_dsp;
    DQ0_ABC dq_abc;
    static DQ_ABC_Error DQ_ABC_Error = DQ_ABC_Error_Default;

    RAMPGEN_reset(&input_rgen);
    RAMPGEN_config(&input_rgen, 500, ARRAY_SIZE);
    ABC_DQ0_POS_reset(&dq0_pos_input);
    DQ0_ABC_reset(&dq_abc);
    DQ_ABC_reset(&dq_abc_dsp);

    for (uint32_t i = 0; i < ARRAY_SIZE; i++) {
        /*
         * generate the d&q&z for input
         */
        RAMPGEN_run(&input_rgen);
        angle_radians = input_rgen.out;
                input_sine_A = 600 * __sinpuf32(angle_radians) * 0.95f; // imbalance with factor 0.95
                sin_theta = __sinpuf32(angle_radians);
                cos_theta = __cospuf32(angle_radians);
                input_sine_B = 600 * __sinpuf32(angle_radians - 2.0f / 3.0f) * 0.93f; // imbalance with factor 0.93
                input_sine_C = 600 * __sinpuf32(angle_radians - 4.0f / 3.0f) * 1.02f; //

        ABC_DQ0_POS_run(&dq0_pos_input,
                        input_sine_A, input_sine_B, input_sine_C,
                        sin_theta, cos_theta);
        /*
         * calculate the cycles
         */
        Counter_Start = CPU_getCycleCnt();
        DQ_ABC_run(&dq_abc_dsp,
                         dq0_pos_input.d, dq0_pos_input.q, dq0_pos_input.z,
                         sin_theta, cos_theta);
        Counter_Stop = CPU_getCycleCnt();
        Counter_cycle = Counter_Stop- Counter_Start - cost_time_error;
        Counter_Total += Counter_cycle;

        /*
         * calculate the ref with C code
         */
        DQ0_ABC_run_ref(&dq_abc,
                    dq0_pos_input.d, dq0_pos_input.q, dq0_pos_input.z,
                    sin_theta, cos_theta);

        DQ_ABC_Error.a[i] = __fabs(dq_abc.a - dq_abc_dsp.a);
        DQ_ABC_Error.b[i] = __fabs(dq_abc.b - dq_abc_dsp.b);
        DQ_ABC_Error.c[i] = __fabs(dq_abc.c - dq_abc_dsp.c);
        error_max_a = (error_max_a < DQ_ABC_Error.a[i]) ? DQ_ABC_Error.a[i] : error_max_a;
        error_max_b = (error_max_b < DQ_ABC_Error.b[i]) ? DQ_ABC_Error.b[i] : error_max_b;
        error_max_c = (error_max_c < DQ_ABC_Error.c[i]) ? DQ_ABC_Error.c[i] : error_max_c;
        error_ctr = ((DQ_ABC_Error.a[i] > TEST_ERROR) || (DQ_ABC_Error.b[i] > TEST_ERROR) || \
                     (DQ_ABC_Error.c[i] > TEST_ERROR)) ? (error_ctr + 1) : error_ctr;
    }
    Counter_Ave = Counter_Total / ARRAY_SIZE;

    if (error_ctr == 0) {
    	log_info("\r\nDQ_ABC_run_dsp test Pass! \n error_max_a is : %.8f \n error_max_b is : %.8f \n error_max_c is : %.8f \n "
               "Counter_Ave %d \n",
               error_max_a, error_max_b, error_max_c, Counter_Ave);
    } else
    	log_info("\r\nDQ_ABC_run_dsp test Fail!\n error_max_a is : %.8f\n error_max_b is : %.8f\n error_max_c is : %.8f\n "
               "Counter_Ave %d \n",
			   error_max_a, error_max_b, error_max_c, Counter_Ave);
}

/**
 *test dq1_abc with z reference with sine&cosine
 *
 */
static inline void
DQ1_ABC_dsp_Test() {
    register uint32_t Counter_Start = 0;
    register uint32_t Counter_Stop = 0;
    register uint32_t Counter_cycle = 0;
    register uint32_t Counter_Total = 0;
    register uint32_t Counter_Ave = 0;
    register uint32_t Counter_Error = 0;
    vuint32_t error_ctr = 0;
    register float32_t angle_radians = 0;
    register float32_t input_sine_A, input_sine_B, input_sine_C;
    register float32_t sin_theta, cos_theta;
    float32_t error_max_a, error_max_b, error_max_c;

    RAMPGEN input_rgen;
    ABC_DQ0_POS dq0_pos_input;

    DQ_ABC dq_abc_dsp;
    DQ0_ABC dq_abc;
    static DQ_ABC_Error DQ_ABC_Error = DQ_ABC_Error_Default;

    RAMPGEN_reset(&input_rgen);
    RAMPGEN_config(&input_rgen, 500, ARRAY_SIZE);
    ABC_DQ0_POS_reset(&dq0_pos_input);
    DQ0_ABC_reset(&dq_abc);
    DQ_ABC_reset(&dq_abc_dsp);

    for (uint32_t i = 0; i < ARRAY_SIZE; i++) {
        /*
         * generate the d&q&z for input
         */
        RAMPGEN_run(&input_rgen);
        angle_radians = input_rgen.out;
		input_sine_A = 600 * __sinpuf32(angle_radians) * 0.95f; // imbalance with factor 0.95
		sin_theta = __sinpuf32(angle_radians);
		cos_theta = __cospuf32(angle_radians);
		input_sine_B = 600 * __sinpuf32(angle_radians - 2.0f / 3.0f) * 0.93f; // imbalance with factor 0.93
		input_sine_C = 600 * __sinpuf32(angle_radians - 4.0f / 3.0f) * 1.02f; //

        ABC_DQ0_POS_run(&dq0_pos_input,
                        input_sine_A, input_sine_B, input_sine_C,
                        sin_theta, cos_theta);
        /*
         * calculate the cycles
         */
        Counter_Start = CPU_getCycleCnt();
        DQ1_ABC_run(&dq_abc_dsp,
                         dq0_pos_input.d, dq0_pos_input.q, dq0_pos_input.z,
						 input_rgen.out);
        Counter_Stop = CPU_getCycleCnt();
        Counter_cycle = Counter_Stop- Counter_Start - cost_time_error;
        Counter_Total += Counter_cycle;

        /*
         * calculate the ref with C code
         */
        DQ0_ABC_run_ref(&dq_abc,
                    dq0_pos_input.d, dq0_pos_input.q, dq0_pos_input.z,
                    sin_theta, cos_theta);

        DQ_ABC_Error.a[i] = __fabs(dq_abc.a - dq_abc_dsp.a);
        DQ_ABC_Error.b[i] = __fabs(dq_abc.b - dq_abc_dsp.b);
        DQ_ABC_Error.c[i] = __fabs(dq_abc.c - dq_abc_dsp.c);
        error_max_a = (error_max_a < DQ_ABC_Error.a[i]) ? DQ_ABC_Error.a[i] : error_max_a;
        error_max_b = (error_max_b < DQ_ABC_Error.b[i]) ? DQ_ABC_Error.b[i] : error_max_b;
        error_max_c = (error_max_c < DQ_ABC_Error.c[i]) ? DQ_ABC_Error.c[i] : error_max_c;
        error_ctr = ((DQ_ABC_Error.a[i] > TEST_ERROR) || (DQ_ABC_Error.b[i] > TEST_ERROR) || \
                     (DQ_ABC_Error.c[i] > TEST_ERROR)) ? (error_ctr + 1) : error_ctr;
    }
    Counter_Ave = Counter_Total / ARRAY_SIZE;

    if (error_ctr == 0) {
    	log_info("\r\nDQ1_ABC_run_dsp test Pass! \n error_max_a is : %.8f \n error_max_b is : %.8f \n error_max_c is : %.8f \n "
               "Counter_Ave %d \n",
               error_max_a, error_max_b, error_max_c, Counter_Ave);
    } else
    	log_info("\r\nDQ1_ABC_run_dsp test Fail!\n error_max_a is : %.8f\n error_max_b is : %.8f\n error_max_c is : %.8f\n "
               "Counter_Ave %d \n",
			   error_max_a, error_max_b, error_max_c, Counter_Ave);
}

/**
 *test dq0_abc with z reference with sine&cosine
 *
 */
static inline void
DQ0_ABC_dsp_Test() {
    register uint32_t Counter_Start = 0;
    register uint32_t Counter_Stop = 0;
    register uint32_t Counter_cycle = 0;
    register uint32_t Counter_Total = 0;
    register uint32_t Counter_Ave = 0;
    register uint32_t Counter_Error = 0;
    vuint32_t error_ctr = 0;
    register float32_t angle_radians = 0;
    register float32_t input_sine_A, input_sine_B, input_sine_C;
    register float32_t sin_theta, cos_theta;
    float32_t error_max_a, error_max_b, error_max_c;

    RAMPGEN input_rgen;
    ABC_DQ0_POS dq0_pos_input;

    DQ_ABC dq_abc_dsp;
    DQ0_ABC dq_abc;
    static DQ_ABC_Error DQ_ABC_Error = DQ_ABC_Error_Default;

    RAMPGEN_reset(&input_rgen);
    RAMPGEN_config(&input_rgen, 500, ARRAY_SIZE);
    ABC_DQ0_POS_reset(&dq0_pos_input);
    DQ0_ABC_reset(&dq_abc);
    DQ_ABC_reset(&dq_abc_dsp);

    for (uint32_t i = 0; i < ARRAY_SIZE; i++) {
        /*
         * generate the d&q&z for input
         */
        RAMPGEN_run(&input_rgen);
        angle_radians = input_rgen.out;
		input_sine_A = 600 * __sinpuf32(angle_radians); // imbalance with factor 0.95
		sin_theta = __sinpuf32(angle_radians);
		cos_theta = __cospuf32(angle_radians);
		input_sine_B = 600 * __sinpuf32(angle_radians - 2.0f / 3.0f); // imbalance with factor 0.93
//		input_sine_C = 600 * __sinpuf32(angle_radians - 4.0f / 3.0f); //
		input_sine_C = -(input_sine_A+input_sine_B);

        ABC_DQ0_POS_run(&dq0_pos_input,
                        input_sine_A, input_sine_B, input_sine_C,
                        sin_theta, cos_theta);
        /*
         * calculate the cycles
         */
        Counter_Start = CPU_getCycleCnt();
        DQ0_ABC_run(&dq_abc_dsp,
                         dq0_pos_input.d, dq0_pos_input.q,
						 sin_theta, cos_theta);
        Counter_Stop = CPU_getCycleCnt();
        Counter_cycle = Counter_Stop- Counter_Start - cost_time_error;
        Counter_Total += Counter_cycle;

        /*
         * calculate the ref with C code
         */
        DQ0_ABC_run_ref(&dq_abc,
                    dq0_pos_input.d, dq0_pos_input.q, dq0_pos_input.z,
                    sin_theta, cos_theta);

        DQ_ABC_Error.a[i] = __fabs(dq_abc.a - dq_abc_dsp.a);
        DQ_ABC_Error.b[i] = __fabs(dq_abc.b - dq_abc_dsp.b);
        DQ_ABC_Error.c[i] = __fabs(dq_abc.c - dq_abc_dsp.c);
        error_max_a = (error_max_a < DQ_ABC_Error.a[i]) ? DQ_ABC_Error.a[i] : error_max_a;
        error_max_b = (error_max_b < DQ_ABC_Error.b[i]) ? DQ_ABC_Error.b[i] : error_max_b;
        error_max_c = (error_max_c < DQ_ABC_Error.c[i]) ? DQ_ABC_Error.c[i] : error_max_c;
        error_ctr = ((DQ_ABC_Error.a[i] > TEST_ERROR) || (DQ_ABC_Error.b[i] > TEST_ERROR) || \
                     (DQ_ABC_Error.c[i] > TEST_ERROR)) ? (error_ctr + 1) : error_ctr;
    }
    Counter_Ave = Counter_Total / ARRAY_SIZE;

    if (error_ctr == 0) {
    	log_info("\r\nDQ0_ABC_run_dsp test Pass! \n error_max_a is : %.8f \n error_max_b is : %.8f \n error_max_c is : %.8f \n "
               "Counter_Ave %d \n",
               error_max_a, error_max_b, error_max_c, Counter_Ave);
    } else
    	log_info("\r\nDQ0_ABC_run_dsp test Fail!\n error_max_a is : %.8f\n error_max_b is : %.8f\n error_max_c is : %.8f\n "
               "Counter_Ave %d \n",
			   error_max_a, error_max_b, error_max_c, Counter_Ave);
}

/**
 *test dq2_abc with z reference with sine&cosine
 *
 */
static inline void
DQ2_ABC_dsp_Test() {
    register uint32_t Counter_Start = 0;
    register uint32_t Counter_Stop = 0;
    register uint32_t Counter_cycle = 0;
    register uint32_t Counter_Total = 0;
    register uint32_t Counter_Ave = 0;
    register uint32_t Counter_Error = 0;
    vuint32_t error_ctr = 0;
    register float32_t angle_radians = 0;
    register float32_t input_sine_A, input_sine_B, input_sine_C;
    register float32_t sin_theta, cos_theta;
    float32_t error_max_a, error_max_b, error_max_c;

    RAMPGEN input_rgen;
    ABC_DQ0_POS dq0_pos_input;

    DQ_ABC dq_abc_dsp;
    DQ0_ABC dq_abc;
    volatile DQ_ABC_Error DQ_ABC_Error = DQ_ABC_Error_Default;

    RAMPGEN_reset(&input_rgen);
    RAMPGEN_config(&input_rgen, 500, ARRAY_SIZE);
    ABC_DQ0_POS_reset(&dq0_pos_input);
    DQ0_ABC_reset(&dq_abc);
    DQ_ABC_reset(&dq_abc_dsp);

    for (uint32_t i = 0; i < ARRAY_SIZE; i++) {
        /*
         * generate the d&q&z for input
         */
        RAMPGEN_run(&input_rgen);
        angle_radians = input_rgen.out;
		input_sine_A = 600 * __sinpuf32(angle_radians); // imbalance with factor 0.95
		sin_theta = __sinpuf32(angle_radians);
		cos_theta = __cospuf32(angle_radians);
		input_sine_B = 600 * __sinpuf32(angle_radians - 2.0f / 3.0f); // imbalance with factor 0.93
//		input_sine_C = 600 * __sinpuf32(angle_radians - 4.0f / 3.0f); //
		input_sine_C = -(input_sine_A+input_sine_B);

        ABC_DQ0_POS_run(&dq0_pos_input,
                        input_sine_A, input_sine_B, input_sine_C,
                        sin_theta, cos_theta);
        /*
         * calculate the cycles
         */
        Counter_Start = CPU_getCycleCnt();
        DQ2_ABC_run(&dq_abc_dsp,
                         dq0_pos_input.d, dq0_pos_input.q,
						 input_rgen.out);
        Counter_Stop = CPU_getCycleCnt();
        Counter_cycle = Counter_Stop- Counter_Start - cost_time_error;
        Counter_Total += Counter_cycle;

        /*
         * calculate the ref with C code
         */
        DQ0_ABC_run_ref(&dq_abc,
                    dq0_pos_input.d, dq0_pos_input.q, 0,
                    sin_theta, cos_theta);

        DQ_ABC_Error.a[i] = __fabs(dq_abc.a - dq_abc_dsp.a);
        DQ_ABC_Error.b[i] = __fabs(dq_abc.b - dq_abc_dsp.b);
        DQ_ABC_Error.c[i] = __fabs(dq_abc.c - dq_abc_dsp.c);
        error_max_a = (error_max_a < DQ_ABC_Error.a[i]) ? DQ_ABC_Error.a[i] : error_max_a;
        error_max_b = (error_max_b < DQ_ABC_Error.b[i]) ? DQ_ABC_Error.b[i] : error_max_b;
        error_max_c = (error_max_c < DQ_ABC_Error.c[i]) ? DQ_ABC_Error.c[i] : error_max_c;
        error_ctr = ((DQ_ABC_Error.a[i] > TEST_ERROR) || (DQ_ABC_Error.b[i] > TEST_ERROR) || \
                     (DQ_ABC_Error.c[i] > TEST_ERROR)) ? (error_ctr + 1) : error_ctr;
    }
    Counter_Ave = Counter_Total / ARRAY_SIZE;

    if (error_ctr == 0) {
    	log_info("\r\nDQ2_ABC_run_dsp test Pass! \n error_max_a is : %.8f \n error_max_b is : %.8f \n error_max_c is : %.8f \n "
               "Counter_Ave %d \n",
               error_max_a, error_max_b, error_max_c, Counter_Ave);
    } else
    	log_info("\r\nDQ2_ABC_run_dsp test Fail!\n error_max_a is : %.8f\n error_max_b is : %.8f\n error_max_c is : %.8f\n "
               "Counter_Ave %d \n",
			   error_max_a, error_max_b, error_max_c, Counter_Ave);
}

/**
 *ABC_alphabeta_DQ0_run test with sine&cosine, z=0
 */
static inline void
ABC_alphabeta_DQ0_dsp_test(void) {
    register uint32_t Counter_Start = 0;
    register uint32_t Counter_Stop = 0;
    register uint32_t Counter_cycle = 0;
    register uint32_t Counter_Total = 0;
    register uint32_t Counter_Ave = 0;
    register uint32_t Counter_Error = 0;
    vuint32_t error_ctr = 0;
    float32_t angle_radians = 0;
    register float32_t input_sine_A, input_sine_B, input_sine_C;
    register float32_t sin_theta, cos_theta;
    float32_t error_max_d, error_max_q, error_max_z;

    RAMPGEN input_rgen;
    ABC_DQ_POS dq0_pos;
    ABC_DQ0_POS dq0_pos_ref;
    volatile ABC_DQ_Error ABC_DQ_POS_Error = ABC_DQ_Error_Default;

    RAMPGEN_reset(&input_rgen);
    RAMPGEN_config(&input_rgen, 500, ARRAY_SIZE);
    ABC_DQ_POS_reset(&dq0_pos);
    ABC_DQ0_POS_reset(&dq0_pos_ref);

    for (uint32_t i = 0; i < ARRAY_SIZE; i++) {
        // Բ
        RAMPGEN_run(&input_rgen);

        angle_radians = input_rgen.out;
        input_sine_A = 600 * __sinpuf32(angle_radians); // imbalance with factor 0.95
        sin_theta = __sinpuf32(angle_radians);
        cos_theta = __cospuf32(angle_radians);
        input_sine_B = 600 * __sinpuf32(angle_radians - 2.0f / 3.0f); // imbalance with factor 0.93
//        input_sine_C = 600 * __sinpuf32(angle_radians - 4.0f / 3.0f); // imbalance with factor 1.02
        input_sine_C = -(input_sine_A+input_sine_B);
        /*
         * calculate the cycles
         */
        Counter_Start = CPU_getCycleCnt();
        ABC_alphabeta_CLARKE_run(&dq0_pos,
                					input_sine_A, input_sine_B, input_sine_C);
        alphabeta_DQ0_PARK_run(&dq0_pos,
				 	 	 	 	 	 sin_theta, cos_theta);
        Counter_Stop = CPU_getCycleCnt();
        Counter_cycle = Counter_Stop- Counter_Start - cost_time_error;
        Counter_Total += Counter_cycle;

        /*
         * calculate the ref with C code
         */
        ABC_DQ0_POS_run(&dq0_pos_ref,
                        input_sine_A, input_sine_B, input_sine_C,
                        sin_theta, cos_theta);

        ABC_DQ_POS_Error.d[i] = __fabs(dq0_pos_ref.d - dq0_pos.d);
        ABC_DQ_POS_Error.q[i] = __fabs(dq0_pos_ref.q - dq0_pos.q);
        ABC_DQ_POS_Error.z[i] = __fabs(dq0_pos_ref.z - dq0_pos.z);
        error_max_d = (error_max_d < ABC_DQ_POS_Error.d[i]) ? ABC_DQ_POS_Error.d[i] : error_max_d;
        error_max_q = (error_max_q < ABC_DQ_POS_Error.q[i]) ? ABC_DQ_POS_Error.q[i] : error_max_q;
        error_max_z = (error_max_z < ABC_DQ_POS_Error.z[i]) ? ABC_DQ_POS_Error.z[i] : error_max_z;
        error_ctr = ((ABC_DQ_POS_Error.d[i] > TEST_ERROR) || (ABC_DQ_POS_Error.q[i] > TEST_ERROR) ||  \
                     (ABC_DQ_POS_Error.z[i] > TEST_ERROR)) ? (error_ctr + 1) : error_ctr;
    }
    Counter_Ave = Counter_Total / ARRAY_SIZE;

    if (error_ctr == 0) {
        log_info("\r\nABC_alphabeta_DQ0_dsp test Pass! \n error_max_d is : %.8f \n error_max_q is : %.8f \n error_max_z is : %.8f \n "
               "Counter_Ave %d \n", error_max_d, error_max_q, error_max_z, Counter_Ave);
    } else
    	log_info("\r\nABC_alphabeta_DQ0_dsp test Fail! \n error_max_d is : %.8f \n error_max_q is : %.8f \n error_max_z is : %.8f \n "
    	               "Counter_Ave %d \n", error_max_d, error_max_q, error_max_z, Counter_Ave);
}

/**
 *ABC_alphabeta_DQ1_run test, z=0
 */
static inline void
ABC_alphabeta_DQ1_dsp_test(void) {
    register uint32_t Counter_Start = 0;
    register uint32_t Counter_Stop = 0;
    register uint32_t Counter_cycle = 0;
    register uint32_t Counter_Total = 0;
    register uint32_t Counter_Ave = 0;
    register uint32_t Counter_Error = 0;
    vuint32_t error_ctr = 0;
    float32_t angle_radians = 0;
    register float32_t input_sine_A, input_sine_B, input_sine_C;
    register float32_t sin_theta, cos_theta;
    float32_t error_max_d, error_max_q, error_max_z;

    RAMPGEN input_rgen;
    ABC_DQ_POS dq0_pos;
    ABC_DQ0_POS dq0_pos_ref;
    volatile ABC_DQ_Error ABC_DQ_POS_Error = ABC_DQ_Error_Default;

    RAMPGEN_reset(&input_rgen);
    RAMPGEN_config(&input_rgen, 500, ARRAY_SIZE);
    ABC_DQ_POS_reset(&dq0_pos);
    ABC_DQ0_POS_reset(&dq0_pos_ref);

    for (uint32_t i = 0; i < ARRAY_SIZE; i++) {
        // Բ
        RAMPGEN_run(&input_rgen);

        angle_radians = input_rgen.out;
        input_sine_A = 600 * __sinpuf32(angle_radians); // imbalance with factor 0.95
        sin_theta = __sinpuf32(angle_radians);
        cos_theta = __cospuf32(angle_radians);
        input_sine_B = 600 * __sinpuf32(angle_radians - 2.0f / 3.0f); // imbalance with factor 0.93
//        input_sine_C = 600 * __sinpuf32(angle_radians - 4.0f / 3.0f); // imbalance with factor 1.02
        input_sine_C = -(input_sine_A+input_sine_B);
        /*
         * calculate the cycles
         */
        Counter_Start = CPU_getCycleCnt();
        ABC_alphabeta_CLARKE_run(&dq0_pos,
                					input_sine_A, input_sine_B, input_sine_C);
        alphabeta_DQ1_PARK_run(&dq0_pos, input_rgen.out);
        Counter_Stop = CPU_getCycleCnt();
        Counter_cycle = Counter_Stop- Counter_Start - cost_time_error;
        Counter_Total += Counter_cycle;

        /*
         * calculate the ref with C code
         */
        ABC_DQ0_POS_run(&dq0_pos_ref,
                        input_sine_A, input_sine_B, input_sine_C,
                        sin_theta, cos_theta);

        ABC_DQ_POS_Error.d[i] = __fabs(dq0_pos_ref.d - dq0_pos.d);
        ABC_DQ_POS_Error.q[i] = __fabs(dq0_pos_ref.q - dq0_pos.q);
        ABC_DQ_POS_Error.z[i] = __fabs(dq0_pos_ref.z - dq0_pos.z);
        error_max_d = (error_max_d < ABC_DQ_POS_Error.d[i]) ? ABC_DQ_POS_Error.d[i] : error_max_d;
        error_max_q = (error_max_q < ABC_DQ_POS_Error.q[i]) ? ABC_DQ_POS_Error.q[i] : error_max_q;
        error_max_z = (error_max_z < ABC_DQ_POS_Error.z[i]) ? ABC_DQ_POS_Error.z[i] : error_max_z;
        error_ctr = ((ABC_DQ_POS_Error.d[i] > TEST_ERROR) || (ABC_DQ_POS_Error.q[i] > TEST_ERROR) ||  \
                     (ABC_DQ_POS_Error.z[i] > TEST_ERROR)) ? (error_ctr + 1) : error_ctr;
    }
    Counter_Ave = Counter_Total / ARRAY_SIZE;

    if (error_ctr == 0) {
        log_info("\r\nABC_alphabeta_DQ1_dsp test Pass! \n error_max_d is : %.8f \n error_max_q is : %.8f \n error_max_z is : %.8f \n "
               "Counter_Ave %d \n", error_max_d, error_max_q, error_max_z, Counter_Ave);
    } else
    	log_info("\r\nABC_alphabeta_DQ1_dsp test Fail! \n error_max_d is : %.8f \n error_max_q is : %.8f \n error_max_z is : %.8f \n "
    	               "Counter_Ave %d \n", error_max_d, error_max_q, error_max_z, Counter_Ave);
}

/**
 *test dq0_abc with z reference with sine&cosine
 *
 */
static inline void
DQ0_alphabeta_ABC_dsp_Test() {
    register uint32_t Counter_Start = 0;
    register uint32_t Counter_Stop = 0;
    register uint32_t Counter_cycle = 0;
    register uint32_t Counter_Total = 0;
    register uint32_t Counter_Ave = 0;
    register uint32_t Counter_Error = 0;
    vuint32_t error_ctr = 0;
    register float32_t angle_radians = 0;
    register float32_t input_sine_A, input_sine_B, input_sine_C;
    register float32_t sin_theta, cos_theta;
    float32_t error_max_a, error_max_b, error_max_c;

    RAMPGEN input_rgen;
    ABC_DQ0_POS dq0_pos_input;

    DQ_ABC dq_abc_dsp;
    DQ0_ABC dq_abc;
    volatile DQ_ABC_Error DQ_ABC_Error = DQ_ABC_Error_Default;

    RAMPGEN_reset(&input_rgen);
    RAMPGEN_config(&input_rgen, 500, ARRAY_SIZE);
    ABC_DQ0_POS_reset(&dq0_pos_input);
    DQ0_ABC_reset(&dq_abc);
    DQ_ABC_reset(&dq_abc_dsp);

    for (uint32_t i = 0; i < ARRAY_SIZE; i++) {
        /*
         * generate the d&q&z for input
         */
        RAMPGEN_run(&input_rgen);
        angle_radians = input_rgen.out;
		input_sine_A = 600 * __sinpuf32(angle_radians); // imbalance with factor 0.95
		sin_theta = __sinpuf32(angle_radians);
		cos_theta = __cospuf32(angle_radians);
		input_sine_B = 600 * __sinpuf32(angle_radians - 2.0f / 3.0f); // imbalance with factor 0.93
//		input_sine_C = 600 * __sinpuf32(angle_radians - 4.0f / 3.0f); //
		input_sine_C = -(input_sine_A+input_sine_B);

        ABC_DQ0_POS_run(&dq0_pos_input,
                        input_sine_A, input_sine_B, input_sine_C,
                        sin_theta, cos_theta);
        /*
         * calculate the cycles
         */
        Counter_Start = CPU_getCycleCnt();
        DQ0_alphabeta_IPARK_run(&dq_abc_dsp,
                				dq0_pos_input.d, dq0_pos_input.q,
								sin_theta, cos_theta);
        alphabeta_ABC_ICLARKE_run( &dq_abc_dsp );
        Counter_Stop = CPU_getCycleCnt();
        Counter_cycle = Counter_Stop- Counter_Start - cost_time_error;
        Counter_Total += Counter_cycle;

        /*
         * calculate the ref with C code
         */
        DQ0_ABC_run_ref(&dq_abc,
                    dq0_pos_input.d, dq0_pos_input.q, dq0_pos_input.z,
                    sin_theta, cos_theta);

        DQ_ABC_Error.a[i] = __fabs(dq_abc.a - dq_abc_dsp.a);
        DQ_ABC_Error.b[i] = __fabs(dq_abc.b - dq_abc_dsp.b);
        DQ_ABC_Error.c[i] = __fabs(dq_abc.c - dq_abc_dsp.c);
        error_max_a = (error_max_a < DQ_ABC_Error.a[i]) ? DQ_ABC_Error.a[i] : error_max_a;
        error_max_b = (error_max_b < DQ_ABC_Error.b[i]) ? DQ_ABC_Error.b[i] : error_max_b;
        error_max_c = (error_max_c < DQ_ABC_Error.c[i]) ? DQ_ABC_Error.c[i] : error_max_c;
        error_ctr = ((DQ_ABC_Error.a[i] > TEST_ERROR) || (DQ_ABC_Error.b[i] > TEST_ERROR) || \
                     (DQ_ABC_Error.c[i] > TEST_ERROR)) ? (error_ctr + 1) : error_ctr;
    }
    Counter_Ave = Counter_Total / ARRAY_SIZE;

    if (error_ctr == 0) {
    	log_info("\r\nDQ0_alphabeta_ABC_run_dsp test Pass! \n error_max_a is : %.8f \n error_max_b is : %.8f \n error_max_c is : %.8f \n "
               "Counter_Ave %d \n",
               error_max_a, error_max_b, error_max_c, Counter_Ave);
    } else
    	log_info("\r\nDQ0_alphabeta_ABC_run_dsp test Fail!\n error_max_a is : %.8f\n error_max_b is : %.8f\n error_max_c is : %.8f\n "
               "Counter_Ave %d \n",
			   error_max_a, error_max_b, error_max_c, Counter_Ave);
}

/**
 *test dq2_abc with z reference with sine&cosine
 *
 */
static inline void
DQ1_alphabeta_ABC_dsp_Test() {
    register uint32_t Counter_Start = 0;
    register uint32_t Counter_Stop = 0;
    register uint32_t Counter_cycle = 0;
    register uint32_t Counter_Total = 0;
    register uint32_t Counter_Ave = 0;
    register uint32_t Counter_Error = 0;
    vuint32_t error_ctr = 0;
    register float32_t angle_radians = 0;
    register float32_t input_sine_A, input_sine_B, input_sine_C;
    register float32_t sin_theta, cos_theta;
    float32_t error_max_a, error_max_b, error_max_c;

    RAMPGEN input_rgen;
    ABC_DQ0_POS dq0_pos_input;

    DQ_ABC dq_abc_dsp;
    DQ0_ABC dq_abc;
    volatile DQ_ABC_Error DQ_ABC_Error = DQ_ABC_Error_Default;

    RAMPGEN_reset(&input_rgen);
    RAMPGEN_config(&input_rgen, 500, ARRAY_SIZE);
    ABC_DQ0_POS_reset(&dq0_pos_input);
    DQ0_ABC_reset(&dq_abc);
    DQ_ABC_reset(&dq_abc_dsp);

    for (uint32_t i = 0; i < ARRAY_SIZE; i++) {
        /*
         * generate the d&q&z for input
         */
        RAMPGEN_run(&input_rgen);
        angle_radians = input_rgen.out;
		input_sine_A = 600 * __sinpuf32(angle_radians); // imbalance with factor 0.95
		sin_theta = __sinpuf32(angle_radians);
		cos_theta = __cospuf32(angle_radians);
		input_sine_B = 600 * __sinpuf32(angle_radians - 2.0f / 3.0f); // imbalance with factor 0.93
//		input_sine_C = 600 * __sinpuf32(angle_radians - 4.0f / 3.0f); //
		input_sine_C = -(input_sine_A+input_sine_B);

        ABC_DQ0_POS_run(&dq0_pos_input,
                        input_sine_A, input_sine_B, input_sine_C,
                        sin_theta, cos_theta);
        /*
         * calculate the cycles
         */
        Counter_Start = CPU_getCycleCnt();
        DQ1_alphabeta_IPARK_run(&dq_abc_dsp,
                         dq0_pos_input.d, dq0_pos_input.q,
						 input_rgen.out);
        alphabeta_ABC_ICLARKE_run(&dq_abc_dsp);
        Counter_Stop = CPU_getCycleCnt();
        Counter_cycle = Counter_Stop- Counter_Start - cost_time_error;
        Counter_Total += Counter_cycle;

        /*
         * calculate the ref with C code
         */
        DQ0_ABC_run_ref(&dq_abc,
                    dq0_pos_input.d, dq0_pos_input.q, 0,
                    sin_theta, cos_theta);

        DQ_ABC_Error.a[i] = __fabs(dq_abc.a - dq_abc_dsp.a);
        DQ_ABC_Error.b[i] = __fabs(dq_abc.b - dq_abc_dsp.b);
        DQ_ABC_Error.c[i] = __fabs(dq_abc.c - dq_abc_dsp.c);
        error_max_a = (error_max_a < DQ_ABC_Error.a[i]) ? DQ_ABC_Error.a[i] : error_max_a;
        error_max_b = (error_max_b < DQ_ABC_Error.b[i]) ? DQ_ABC_Error.b[i] : error_max_b;
        error_max_c = (error_max_c < DQ_ABC_Error.c[i]) ? DQ_ABC_Error.c[i] : error_max_c;
        error_ctr = ((DQ_ABC_Error.a[i] > TEST_ERROR) || (DQ_ABC_Error.b[i] > TEST_ERROR) || \
                     (DQ_ABC_Error.c[i] > TEST_ERROR)) ? (error_ctr + 1) : error_ctr;
    }
    Counter_Ave = Counter_Total / ARRAY_SIZE;

    if (error_ctr == 0) {
    	log_info("\r\nDQ1_alphabeta_ABC_run_dsp test Pass! \n error_max_a is : %.8f \n error_max_b is : %.8f \n error_max_c is : %.8f \n "
               "Counter_Ave %d \n",
               error_max_a, error_max_b, error_max_c, Counter_Ave);
    } else
    	log_info("\r\nDQ1_alphabeta_ABC_run_dsp test Fail!\n error_max_a is : %.8f\n error_max_b is : %.8f\n error_max_c is : %.8f\n "
               "Counter_Ave %d \n",
			   error_max_a, error_max_b, error_max_c, Counter_Ave);
}

#endif
