/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <string.h>

#include "device.h"
#include "can_epwm_synchronous_phaser.h"

CAN_RxMessage_t RxMsgBuf = {0};
CAN_TxMessage_t TxMsgBuf = {0};
volatile uint32_t status;
volatile uint16_t tbctr=0;
volatile uint16_t tbctrB=0;
volatile uint64_t timestamp;
volatile int sign=0;
volatile int compensation=50000;
void can_interrupt_irq_handler(void)
{
	status = CAN_getIntrStatus(CAN_BASE);
	sign++;
	if (status & CAN_RTIF_RIF_M) {
		CAN_receiveMsg(CAN_BASE, &RxMsgBuf);
		if(RxMsgBuf.id==0xC0){
			timestamp=RxMsgBuf.timestamp;
		}
		else{
			timestamp=RxMsgBuf.timestamp-timestamp;
			tbctr=RxMsgBuf.data[1];
			tbctr=tbctr*0xff+RxMsgBuf.data[0];

			tbctr+=188*timestamp/100+215;
			tbctr=tbctr%3750;
			tbctrB=EPWM_getTimeBaseCounterValue(myEPWM1_BASE);

			compensation=tbctr-tbctrB;
		}
	}
	if (CAN_getRxBufState(CAN_BASE) == CAN_RXBUFFER_EMPTY)
		CAN_clearIntrStatus(CAN_BASE, CAN_RTIF_RIF_M);
	if (status & CAN_RTIF_TSIF_M) {
		if(master){
		tbctr=EPWM_getTimeBaseCounterValue(myEPWM1_BASE);
		TxMsgBuf.data[0]=tbctr%0xff;
		TxMsgBuf.data[1]=tbctr/0xff%0xff;
		TxMsgBuf.id = 0xC1;

		while (!CAN_transmitHighPriorityMsg(CAN_BASE, &TxMsgBuf));
		}

		CAN_clearIntrStatus(CAN_BASE, CAN_RTIF_TSIF_M);
	}
}
uint32_t gs_epwmIsrCnt = 0;
void epwm1ISR(void)
{
	gs_epwmIsrCnt++;
	if(compensation>compe_cnt){
		EPWM_setTimeBaseCounter(myEPWM1_BASE, EPWM_getTimeBaseCounterValue(myEPWM1_BASE)+compe_cnt);
		compensation=compensation-compe_cnt;
	}
	else if(compensation<-compe_cnt){
		EPWM_setTimeBaseCounter(myEPWM1_BASE, EPWM_getTimeBaseCounterValue(myEPWM1_BASE)-compe_cnt);
		compensation=compensation+compe_cnt;
	}
    EPWM_clearEventTriggerInterruptFlag(myEPWM1_BASE);

}
void epwm_init()
{
    GPIO_setPinConfig(myEPWM1_EPWMA_PIN_CONFIG);
    GPIO_setPadConfig(myEPWM1_EPWMA_GPIO, GPIO_PIN_TYPE_STD);
    GPIO_setQualificationMode(myEPWM1_EPWMA_GPIO, GPIO_QUAL_SYNC);

    EPWM_setClockPrescaler(myEPWM1_BASE, EPWM_CLOCK_DIVIDER_1, EPWM_HSCLOCK_DIVIDER_1);
    //	Set TBPRD = 12000
    EPWM_setTimeBasePeriod(myEPWM1_BASE, 3750);
    //	Set TBCTR = 0
    EPWM_setTimeBaseCounter(myEPWM1_BASE, 0);
    //	Counting mode: UP_DOWN
    EPWM_setTimeBaseCounterMode(myEPWM1_BASE, EPWM_COUNTER_MODE_UP);
    //	Disable phase shift load
    EPWM_disablePhaseShiftLoad(myEPWM1_BASE);
    //	Set TBPHS = 0
    EPWM_setPhaseShift(myEPWM1_BASE, 0);
    //	Set CMPA = 6000
    EPWM_setCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_A, 1875);
    //	Load the new CMPA when TBCTR = 0
    EPWM_setCounterCompareShadowLoadMode(myEPWM1_BASE, EPWM_COUNTER_COMPARE_A, EPWM_COMP_LOAD_ON_CNTR_ZERO);
    //AQ
    //	TBCTR = UP_CMPA -> ePWM2A:HIGH
    EPWM_setActionQualifierAction(myEPWM1_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
    //	TBCTR = DOWN_CMPA -> ePWM2A:LOW
    EPWM_setActionQualifierAction(myEPWM1_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);

    Interrupt_register(INT_EPWM1, &epwm1ISR);
    EPWM_enableInterrupt(myEPWM1_BASE);
    //ePWM1 interrupt Source is TBCTR = ZERO
    EPWM_setInterruptSource(myEPWM1_BASE, EPWM_INT_TBCTR_ZERO);
    //Sets the ePWM1 interrupt event counts to 3
    EPWM_setInterruptEventCount(myEPWM1_BASE, 1);
    if(master==0)
    Interrupt_enable(INT_EPWM1);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

}

void can_config(void)
{
	CAN_InitParams_t initParams = {0};
	CAN_FilterElement_t stdFiltelem = {0};

	initParams.fdMode = false;
	initParams.txbmode = CAN_TS_MODE_PRIORITY;
	initParams.timestamp = CAN_TIMESTAMP_EOF;

	/* accept all CAN ID frames. */
	stdFiltelem.acode = 0xc0;
	stdFiltelem.amsk = 0x1fffffff;
	stdFiltelem.filerFrameType = CAN_IDTYPE_BOTH_STD_EXT;

	/* Initial CAN module. */
	if (!CAN_initModule(CAN_BASE))
		while (1);

	/* Configuration CAN basic function. */
	if (!CAN_initConfig(CAN_BASE, &initParams))
		while (1);

	/* Set the arbitration segment bit rate based on sampling points. */
	if (!CAN_setBitRateSamplePoint(CAN_BASE, 40000000, 1U * 1000 * 1000, 0.8))
		while (1);

	/* Set CAN ID acceptance filter 0. */
	if (!CAN_configMsgFilter(CAN_BASE, 0, &stdFiltelem))
		while (1);

	/* Enable CAN ID acceptance filter 0. */
	if (!CAN_configMsgFilterEnable(CAN_BASE, 0))
		while (1);

	/* Start running CAN module. */
	if (!CAN_startModule(CAN_BASE))
		while (1);

	/* Enable auto retransmission. */
	if (!CAN_setAutoRetransmission(CAN_BASE, true))
		while (1);
}

void can_ctrl_init(void)
{
	SysCtl_setCanClkSrcSel(CAN_CLK_TYPE_PLL);

	/* CAN clock source = 40MHz, PLL_CLK = 2* SYS_CLK */
	SysCtl_setCanClkDiv((DEVICE_SYSCLK_FREQ * 2 / 40000000));

	GPIO_setPinConfig(CAN_RX_PIN);
	GPIO_setPinConfig(CAN_TX_PIN);

	can_config();

	Interrupt_SetPriority(CAN_IRQ_NUM, 0, 0);
	Interrupt_register(CAN_IRQ_NUM, &can_interrupt_irq_handler);
	Interrupt_enable(CAN_IRQ_NUM);
	CAN_enableInterrupt(CAN_BASE, CAN_RTIE_RIE_M|CAN_RTIE_TSIE_M);
}

void mcan_send()
{
	uint32_t i = 0;
	TxMsgBuf.dlc = CAN_DATA_LENGTH_8;
	TxMsgBuf.id = 0xC0;
	TxMsgBuf.ide = 0;
	TxMsgBuf.rtr = 0;
	for (i = 0; i < CAN_getMessageLength(TxMsgBuf.dlc); i++)
	{
		TxMsgBuf.data[i] = i;
	}
	while (!CAN_transmitMsg(CAN_BASE, &TxMsgBuf));
}
