/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    cputimer_ex1_irq.c
*   @brief   CPUTimer0 is configured as periodic mode
*            CPUTimer1 is configured as one-shot mode
*            CPUTimer2 is configured as free run mode
*   @details view the variables: dlog_timer_int_cnt
*                                dlog_timer0_timestamp
*                                dlog_isr_entry_delay
*                                dlog_timer2_ts
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "device.h"
#include "board_cfg.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* CPUTimer clock frequency is "DEVICE_APBCLK_FREQ", the CPU timer0 IRQ frequency is 10Hz */
/* CPUTimer0 period is 100mS */
#define CPUTIMER0_PERIOD (DEVICE_APBCLK_FREQ/10)
/* CPUTimer1 delay is 50mS */
#define CPUTIMER1_DELAY (DEVICE_APBCLK_FREQ/20)
#define MY_GPIO_DATA_REG_ADDR(pin)  (GPIODATA_BASE + (GPIO_MASKLOWBYTE) + 0x400*((pin%16)/8) + (pin/16U) * (GPIO_BASE_ADDR_STEP) + 4*(0x01<<(pin%8)))

/*
 * TEST_MODE = 1,CPUTIMER0 runs at a fixed period and generates an interrupt, and CPUTIMER1 runs in one-shot mode and generates an interrupt
 * TEST_MODE = 2,CPUTIMER0 runs on a fixed period and generates interrupts, and CPUTIMER1 runs in free-run mode
 *
 * If it is a 1.2 or 2.2 series chip, CPUTIMER1 can be changed to CPUTIMER2 when cputime_mode = 2
 * */
#define TEST_MODE  2
/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

uint32_t dlog_timer_int_cnt[2];
uint32_t dlog_timer0_timestamp;
uint32_t dlog_isr_entry_delay;
uint32_t dlog_timer2_ts[4];

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/*
 * @brief    Configure the CPUTimer as periodic mode
 */
void CPUTimer_initPeriodMode(uint32_t base, uint32_t nPeriod)
{
	CPUTimer_stopTimer(base);
	 /* Set CPUTimer clock prescaler = 1 */
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
    CPUTimer_setPreScaler(base, CPUTIMER_CLOCK_PRESCALER_1);
#else
    CPUTimer_setPreScaler(base, 1);
#endif
	/* Set the timer width to 32-bit or 16 bit */
    CPUTimer_setTimerSize(base, CPUTIMER_TIMERSIZE_32BIT);
	/* Set the timer mode to cycle timing */
	CPUTimer_setTimerMode(base, CPUTIMER_TIMERMODE_PERIODIC);
    CPUTimer_setEmulationMode(base, CPUTIMER_EMULATIONMODE_STOPAFTERNEXTDECREMENT);
	/* Set timer cycle */
	CPUTimer_setPeriod(base, nPeriod);
	/* Enable timer interrupt */
	CPUTimer_enableInterrupt(base);
	/* clear interrupt flag */
	CPUTimer_clearOverflowFlag(base);
}

#if(TEST_MODE == 1)
/*
 * @brief    Configure the CPUTimer as one-shot mode
 */
void CPUTimer_initOneShotMode(uint32_t base, uint32_t nDelay)
{
	CPUTimer_stopTimer(base);
	 /* Set CPUTimer clock prescaler = 1 */
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
    CPUTimer_setPreScaler(base, CPUTIMER_CLOCK_PRESCALER_1);
#else
    CPUTimer_setPreScaler(base, 1);
#endif
	/* Set the timer width to 32-bit or 16 bit */
    CPUTimer_setTimerSize(base, CPUTIMER_TIMERSIZE_32BIT);
	/* Set the timer mode to cycle timing */
	CPUTimer_setTimerMode(base, CPUTIMER_TIMERMODE_ONESHOT);
	/* Set timer cycle */
	CPUTimer_setPeriod(base, nDelay);
	/* Enable timer interrupt */
	CPUTimer_enableInterrupt(base);
	/* clear interrupt flag */
	CPUTimer_clearOverflowFlag(base);
}

#elif(TEST_MODE == 2)
/*
 * @brief    Configure the CPUTimer as free-run mode
 */
void CPUTimer_initFreeRunMode(uint32_t base)
{
	CPUTimer_stopTimer(base);
	 /* Set CPUTimer clock prescaler = 1 */
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)    
    CPUTimer_setPreScaler(base, CPUTIMER_CLOCK_PRESCALER_1);
#else
    CPUTimer_setPreScaler(base, 1);
#endif
	/* Set the timer width to 32-bit or 16 bit */
    CPUTimer_setTimerSize(base, CPUTIMER_TIMERSIZE_32BIT);
	/* Set the timer mode to cycle timing */
	CPUTimer_setTimerMode(base, CPUTIMER_TIMERMODE_FREERUNNING);
	//CPUTimer_setEmulationMode(base, CPUTIMER_EMULATIONMODE_STOPAFTERNEXTDECREMENT);
	/* Set timer cycle */
	CPUTimer_setPeriod(base, 0xFFFFFFFF);
}
#endif

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

/**
 * @brief   CPU Timer0 IRQ handler
 * @param   none
 * @retval  None
 */
__INTERRUPT void Cputimer0_IRQHandler(void)
{
#if(TEST_MODE == 2)
	/* calculate the ISR entry delay */
	dlog_timer0_timestamp = CPUTimer_getTimerCount(CPUTIMER0_BASE);
	dlog_isr_entry_delay = CPUTIMER0_PERIOD - dlog_timer0_timestamp;

	/* Calculate the interval of this ISR occurrence. */
	dlog_timer2_ts[1] = CPUTimer_getTimerCount(CPUTIMER1_BASE);
	dlog_timer2_ts[2] = dlog_timer2_ts[0] - dlog_timer2_ts[1];
	dlog_timer2_ts[0] = dlog_timer2_ts[1];
#endif
	/* Clear CPU Timer0 interrupt flag bit */
    CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
    /* Flip the GPIO outputCorresponds to the LED on the development board */
    GPIO_togglePin(GPIO_PIN_LED1);
#if(TEST_MODE == 1)
    dlog_timer_int_cnt[0]++;
    /* If 10 interrupts occur, restart timer 1 */
    if((dlog_timer_int_cnt[0]%10) == 0)
    {
        CPUTimer_startTimer(CPUTIMER1_BASE);
    }
#endif
}

#if(TEST_MODE == 1)
/**
 * @brief   CPU Timer1 IRQ handler
 * @param   none
 * @retval  None
 */
__INTERRUPT void Cputimer1_IRQHandler(void)
{
	/* Flip the GPIO outputCorresponds to the LED on the development board */
    GPIO_togglePin(GPIO_PIN_LED2);
    dlog_timer_int_cnt[1]++;
    /* Clear CPU Timer1 interrupt flag bit */
    CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);
    /* Need call this function if IntFlag is cleared at the end of ISR, its purpose is to wait till this flag is really cleared */
    __DSB();
}
#endif

/**
  * @brief  Main program
  * @param  None
  * @retval None
  */
int main(void)
{
	//SysCtl_setObsSigSel()
    Device_init();
    /* Configure a GPIO as output mode to driver LED0 */
    GPIO_setPinConfig(GPIO_CFG_LED1);
    GPIO_setDirectionMode(GPIO_PIN_LED1, GPIO_DIR_MODE_OUT);
#if (GS32_PART_NUM==0x0039)
    GPIO_setAnalogMode(GPIO_PIN_LED1, GPIO_ANALOG_DISABLED);
#endif

    /* Configure a GPIO as output mode to driver LED1 */
    GPIO_setPinConfig(GPIO_CFG_LED2);
    GPIO_setDirectionMode(GPIO_PIN_LED2, GPIO_DIR_MODE_OUT);


    /* Initialize CpuTimer0 to run on a fixed period and generate an interrupt*/
    CPUTimer_initPeriodMode(CPUTIMER0_BASE, CPUTIMER0_PERIOD);  /* CPUTimer_init API function includes starting the timer. It's not used here */
    /* Register the CPUTimer interrupt handler*/
    Interrupt_register(INT_TIMER0, Cputimer0_IRQHandler);
    /* Enable CpuTimer0 interrupt in the system interrupt controller */
    Interrupt_enable(INT_TIMER0);
    /* Start CPUTimer0 */
    CPUTimer_startTimer(CPUTIMER0_BASE);

#if(TEST_MODE == 1)
    /* Initializes CpuTimer1 to run in one-shot mode and generates an interrupt */
    CPUTimer_initOneShotMode(CPUTIMER1_BASE, CPUTIMER1_DELAY);
    /* Register the CPUTimer interrupt handler*/
    Interrupt_register(INT_TIMER1, Cputimer1_IRQHandler);
    /* Enable CpuTimer1 interrupt in the system interrupt controller */
    Interrupt_enable(INT_TIMER1);

#elif(TEST_MODE == 2)
    /* Initializes CpuTimer1 to run in free-run mode */
    CPUTimer_initFreeRunMode(CPUTIMER1_BASE);
    CPUTimer_startTimer(CPUTIMER1_BASE);
#endif
    /* global interrupt enable */
    __enable_irq();
    for(;;);
}

#ifdef __cplusplus
}
#endif

