/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    cputimer_ex3_timed_task.c
*   @brief   CPUTimer0 is configured as 1mS periodic interrupt mode
*            a simple task scheduler is created based on the 1mS timer tick.
*            The timed tasks flags are set in CPUTimer ISR, and the task is
*            executed in the background while loop.
*   @details view variable Task_Timers.
*
*/

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "log.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
/* Set CPUTimer0 period 1mS */
#define CPUTIMER0_PERIOD (DEVICE_APBCLK_FREQ/1000)

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

typedef volatile struct
{
    uint16_t Flag_1ms;
    uint16_t Flag_10ms;
    uint16_t Flag_100ms;

    uint16_t Cnt_1ms;
    uint16_t Cnt_10ms;
    uint16_t Cnt_100ms;

    uint32_t Tick_1ms;
    uint32_t Tick_10ms;
    uint32_t Tick_100ms;
} TASK_TIMER_Type;

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

volatile TASK_TIMER_Type Task_Timers;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

//volatile struct EPWM_REGS * const ptrEPwm1Regs=(struct EPWM_REGS *)EPWM1_BASE;

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/*
 * @brief  Cputimer0 ISR
 */
__INTERRUPT void Cputimer0_IRQHandler(void)
{
	/* Clear CPUTimer Int Flag */
	CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);

	/* Increment the timer ticks and set corresponding flags */
	Task_Timers.Cnt_1ms++;
	Task_Timers.Tick_1ms++;
	Task_Timers.Flag_1ms = 1;
	if(Task_Timers.Cnt_1ms >= 10)
	{
		Task_Timers.Cnt_1ms = 0;
		Task_Timers.Flag_10ms = 1;
		Task_Timers.Cnt_10ms++;
		Task_Timers.Tick_10ms++;
		if(Task_Timers.Cnt_10ms >= 10)
		{
			Task_Timers.Cnt_10ms = 0;
			Task_Timers.Flag_100ms = 1;
			Task_Timers.Cnt_100ms++;
			Task_Timers.Tick_100ms++;
			if(Task_Timers.Cnt_100ms >= 10)
			{
				Task_Timers.Cnt_100ms = 0;
			}
		}
	}
}

/*
 * @brief  Task run with 1mS interval
 */
void Task_runThread1ms(void)
{
    if(Task_Timers.Flag_1ms)
    {
    	//Calling 1ms task functions starts here
    	//......
    	//GPIO_togglePin(0);
    	//Calling 1ms task functions ends here

    	Task_Timers.Flag_1ms = 0;
    }
}

/*
 * @brief  Task run with 10mS interval
 */
void Task_runThread10ms(void)
{
    if(Task_Timers.Flag_10ms)
    {
    	//Calling 10ms task functions starts here
    	//......
    	//GPIO_togglePin(1);
    	//Calling 10ms task functions ends here

    	Task_Timers.Flag_10ms = 0;
    }
}

/*
 * @brief  Task run with 100mS interval
 */
void Task_runThread100ms(void)
{
    if(Task_Timers.Flag_100ms)
    {
    	GPIO_togglePin(GPIO_PIN_LED1);
    	//Calling 100ms task functions starts here
    	//......
    	if((Task_Timers.Tick_100ms % 10) == 0)
    	{
    		GPIO_togglePin(GPIO_PIN_LED2);
    	}

    	//Calling 100ms task functions ends here

    	Task_Timers.Flag_100ms = 0;
    }
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/*
 * @brief  main function.
 */
int main(void)
{
    Device_init();
	/* Configure GPIO for LED1 */
    GPIO_setPinConfig(GPIO_CFG_LED1);
	GPIO_setDirectionMode(GPIO_PIN_LED1, GPIO_DIR_MODE_OUT);
#if GS32_PART_NUM==0x0039
	GPIO_setAnalogMode(GPIO_PIN_LED1, GPIO_ANALOG_DISABLED);
#endif

    /* Configure GPIO for LED2 */
    GPIO_setPinConfig(GPIO_CFG_LED2);
	GPIO_setDirectionMode(GPIO_PIN_LED2, GPIO_DIR_MODE_OUT);

    /* ʼCpuTimer0Թ̶вж */
    CPUTimer_init(CPUTIMER0_BASE, CPUTIMER0_PERIOD);
    /* עCPUTimer0жϴ */
    Interrupt_register(INT_TIMER0, Cputimer0_IRQHandler);
    /* ϵͳжϿʹCpuTimer0ж */
    Interrupt_enable(INT_TIMER0);

    /* global interrupt enable */
    __enable_irq();

    while (1) {
    	__NOP();
    	Task_runThread1ms();
    	Task_runThread10ms();
    	Task_runThread100ms();
    }

    return 0;
}

